document.addEventListener('DOMContentLoaded', () => {
    // Get DOM elements
    const htmlInput = document.getElementById('html-input');
    const cssInput = document.getElementById('css-input');
    const jsInput = document.getElementById('js-input');
    // const fileStructure = document.getElementById('file-structure'); // Removed
    const tabButtons = document.querySelectorAll('.tab-btn');
    const inputSections = document.querySelectorAll('.input-section');
    const previewBtn = document.getElementById('preview-btn');
    const previewNewTabBtn = document.getElementById('preview-new-tab-btn');
    const generateBtn = document.getElementById('generate-zip');
    const clearBtn = document.getElementById('clear-btn');
    const previewSection = document.getElementById('preview-section');
    const previewContainer = document.getElementById('preview-container');
    const resultSection = document.getElementById('result-section');
    const filenameDisplay = document.getElementById('filename-display');
    const downloadLink = document.getElementById('download-link');
    const generateAIzipperBtn = document.getElementById('generate-aizipper-zip-btn'); // New button

    // ZIP import UI
    const importZipBtn = document.getElementById('import-zip-btn');
    const zipFileInput = document.getElementById('zip-input');
    const includeGaCheckbox = document.getElementById('include-ga');
    const includeAdsenseCheckbox = document.getElementById('include-adsense');
    const footerUrlInput = document.getElementById('footer-url');
    const footerAuthorInput = document.getElementById('footer-author');
    const footerFreeTextInput = document.getElementById('footer-free-text');
    const toggleAddonsBtn = document.getElementById('toggle-addons-btn');
    const addonsSection = document.querySelector('section.addons');
    const GA_ID = 'G-S9EWRY1CPJ';
    const ADSENSE_CLIENT = 'ca-pub-0121577198857509';

    // State for imported ZIP contents
    let importedFiles = {}; // path -> { text?: string, base64?: string }
    let importedMainPaths = { html: null, css: null, js: null };
    let createdBlobUrls = []; // object URLs created for preview; revoked on next preview

    if (importZipBtn && zipFileInput) {
        importZipBtn.addEventListener('click', () => zipFileInput.click());
        zipFileInput.addEventListener('change', async (e) => {
            const file = e.target.files && e.target.files[0];
            if (!file) return;
            try {
                await importProjectZip(file);
            } catch (err) {
                console.error('Failed to import ZIP:', err);
                alert('Failed to import ZIP file. Please ensure it is a valid .zip.');
            }
        });
    }

    // Toggle Optional add-ons section visibility
    if (toggleAddonsBtn && addonsSection) {
        toggleAddonsBtn.addEventListener('click', () => {
            const isHidden = window.getComputedStyle(addonsSection).display === 'none';
            if (isHidden) {
                addonsSection.style.display = '';
                toggleAddonsBtn.textContent = 'Hide Optional add-ons';
                // Default to ON when shown
                if (includeGaCheckbox) includeGaCheckbox.checked = true;
                if (includeAdsenseCheckbox) includeAdsenseCheckbox.checked = true;
            } else {
                addonsSection.style.display = 'none';
                toggleAddonsBtn.textContent = 'Show Optional add-ons';
            }
        });
    }

    // Drag and drop functionality
    function setupDragAndDrop(textareaElement) {
        textareaElement.addEventListener('dragover', (event) => {
            event.stopPropagation();
            event.preventDefault();
            event.dataTransfer.dropEffect = 'copy'; // Explicitly show this is a copy.
            textareaElement.classList.add('dragover'); // Optional: add a class for visual feedback
        });

        textareaElement.addEventListener('dragleave', (event) => {
            event.stopPropagation();
            event.preventDefault();
            textareaElement.classList.remove('dragover'); // Optional: remove visual feedback class
        });

        textareaElement.addEventListener('drop', (event) => {
            event.stopPropagation();
            event.preventDefault();
            textareaElement.classList.remove('dragover'); // Optional: remove visual feedback class

            const files = event.dataTransfer.files;
            if (files.length > 0) {
                const file = files[0];
                const lowerName = (file.name || '').toLowerCase();

                // If a ZIP is dropped into the HTML area, import the project ZIP
                const isZip = lowerName.endsWith('.zip') || (file.type && file.type.includes('zip'));
                if (isZip) {
                    if (textareaElement === htmlInput) {
                        // Use the existing ZIP import flow
                        importProjectZip(file);
                    } else {
                        alert('To import a project ZIP, drop it into the HTML area.');
                    }
                    return;
                }

                // Accept common text/code files
                if ((file.type && file.type.match('text.*')) || lowerName.endsWith('.html') || lowerName.endsWith('.css') || lowerName.endsWith('.js') || lowerName.endsWith('.txt')) {
                    const reader = new FileReader();
                    reader.onload = (e) => {
                        textareaElement.value = e.target.result;
                    };
                    reader.readAsText(file);
                } else {
                    alert('Please drop a text file (e.g., .html, .css, .js, .txt) or drop a .zip into the HTML area to import a project.');
                }
            }
        });
    }

    setupDragAndDrop(htmlInput);
    setupDragAndDrop(cssInput);
    setupDragAndDrop(jsInput);
    
    // Tab switching functionality
    tabButtons.forEach(button => {
        button.addEventListener('click', () => {
            // Remove active class from all buttons and sections
            tabButtons.forEach(btn => btn.classList.remove('active'));
            inputSections.forEach(section => section.classList.remove('active'));
            
            // Add active class to clicked button and corresponding section
            const tabName = button.getAttribute('data-tab');
            button.classList.add('active');
            document.getElementById(`${tabName}-section`).classList.add('active');
        });
    });

    // Preview HTML when preview button is clicked
    previewBtn.addEventListener('click', () => {
        const htmlCode = htmlInput.value; // Keep original for textarea
        const cssCode = cssInput.value.trim();
        const jsCode = jsInput.value.trim();

        if (!htmlCode.trim()) {
            alert('Please paste some HTML code first!');
            return;
        }

        // Cleanup any previous object URLs to avoid leaks
        revokeCreatedBlobUrls();

        previewContainer.innerHTML = '';
        const iframe = document.createElement('iframe');
        iframe.style.width = '100%';
        // iframe.style.height = '400px'; // Removed fixed height
        iframe.style.border = '1px solid #ddd';
        previewContainer.appendChild(iframe);

        const iframeDoc = iframe.contentDocument || iframe.contentWindow.document;
        iframeDoc.open();

        let contentForPreview = htmlCode.trim();

        // Apply optional add-ons (GA, AdSense, footer credit)
        contentForPreview = applyAddOns(contentForPreview);

        // Preprocess CSS if present (rewrite url() using imported assets if available)
        let cssForPreview = cssCode;
        if (cssForPreview !== '' && Object.keys(importedFiles).length) {
            const cssBase = importedMainPaths.css || (importedMainPaths.html || 'index.html');
            cssForPreview = replaceCssUrls(cssForPreview, cssBase);
        }

        // Embed CSS if provided
        if (cssForPreview !== '') {
            if (contentForPreview.includes('</head>')) {
                contentForPreview = contentForPreview.replace('</head>', `<style>\n${cssForPreview}\n</style>\n</head>`);
            } else {
                contentForPreview = `<style>\n${cssForPreview}\n</style>\n` + contentForPreview;
            }
        }

        // Embed JS if provided
        if (jsCode !== '') {
            if (contentForPreview.includes('</body>')) {
                contentForPreview = contentForPreview.replace('</body>', `<script>\n${jsCode}\n</script>\n</body>`);
            } else {
                contentForPreview += `\n<script>\n${jsCode}\n</script>`;
            }
        }

        // Rewrite asset paths in HTML to blob URLs if an imported ZIP is available
        if (Object.keys(importedFiles).length && importedMainPaths.html) {
            contentForPreview = replaceAssetPathsInHtml(contentForPreview, importedMainPaths.html);
        }
        
        iframeDoc.write(contentForPreview);
        iframeDoc.close();

        // Adjust iframe height to content
        iframe.onload = () => {
            try {
                const body = iframe.contentWindow.document.body;
                const html = iframe.contentWindow.document.documentElement;
                const height = Math.max( body.scrollHeight, body.offsetHeight, 
                                       html.clientHeight, html.scrollHeight, html.offsetHeight );
                iframe.style.height = height + 'px';
            } catch (e) {
                console.warn("Could not resize iframe to content, likely due to cross-origin restrictions if previewing external content directly.", e);
                iframe.style.height = '400px'; // Fallback height
            }
        };
        // For browsers that might not fire onload for srcdoc/write, try to set height after write
        // This is a bit of a race condition, onload is preferred.
        try {
            const body = iframe.contentWindow.document.body;
            const html = iframe.contentWindow.document.documentElement;
            // Timeout to allow rendering
            setTimeout(() => {
                 const height = Math.max( body.scrollHeight, body.offsetHeight, 
                                       html.clientHeight, html.scrollHeight, html.offsetHeight );
                iframe.style.height = height + 'px';
            }, 100); // Adjust timeout if needed
        } catch (e) {
            // Fallback if immediate access fails (e.g. about:blank not ready)
             if (!iframe.onload) { // if onload didn't get set or fire
                iframe.style.height = '400px'; // Fallback height
             }
        }


        previewSection.classList.remove('hidden');
        previewSection.scrollIntoView({ behavior: 'smooth' });
    });

    // Preview in new tab when "Preview in New Tab" button is clicked
    previewNewTabBtn.addEventListener('click', () => {
        const htmlCode = htmlInput.value;
        const cssCode = cssInput.value.trim();
        const jsCode = jsInput.value.trim();

        if (!htmlCode.trim()) {
            alert('Please paste some HTML code first!');
            return;
        }

        let contentForPreview = htmlCode.trim();

        // Apply optional add-ons (GA, AdSense, footer credit)
        contentForPreview = applyAddOns(contentForPreview);

        // Preprocess CSS if present (rewrite url() using imported assets if available)
        let cssForPreview = cssCode;
        if (cssForPreview !== '' && Object.keys(importedFiles).length) {
            const cssBase = importedMainPaths.css || (importedMainPaths.html || 'index.html');
            cssForPreview = replaceCssUrls(cssForPreview, cssBase);
        }

        // Embed CSS if provided
        if (cssForPreview !== '') {
            if (contentForPreview.includes('</head>')) {
                contentForPreview = contentForPreview.replace('</head>', `<style>\n${cssForPreview}\n</style>\n</head>`);
            } else {
                contentForPreview = `<style>\n${cssForPreview}\n</style>\n` + contentForPreview;
            }
        }

        // Embed JS if provided
        if (jsCode !== '') {
            if (contentForPreview.includes('</body>')) {
                contentForPreview = contentForPreview.replace('</body>', `<script>\n${jsCode}\n</script>\n</body>`);
            } else {
                contentForPreview += `\n<script>\n${jsCode}\n</script>`;
            }
        }

        // Rewrite asset paths in HTML to blob URLs if an imported ZIP is available
        if (Object.keys(importedFiles).length && importedMainPaths.html) {
            contentForPreview = replaceAssetPathsInHtml(contentForPreview, importedMainPaths.html);
        }

        const blob = new Blob([contentForPreview], { type: 'text/html' });
        const url = URL.createObjectURL(blob);
        window.open(url, '_blank');
    });

    // Generate ZIP file when button is clicked
    generateBtn.addEventListener('click', async () => {
        const htmlCode = htmlInput.value; // Keep original for textarea
        const cssCode = cssInput.value.trim();
        const jsCode = jsInput.value.trim();

        if (!htmlCode.trim()) {
            alert('Please paste some HTML code first!');
            return;
        }
        
        // Show loading state
        generateBtn.disabled = true;
        generateBtn.textContent = 'Processing...';
        
        try {
            await generateZip(htmlCode.trim(), cssCode, jsCode);
        } finally {
            // Reset button state
            generateBtn.disabled = false;
            generateBtn.textContent = 'Generate ZIP';
        }
    });
    
    // Clear the input field
    clearBtn.addEventListener('click', () => {
        htmlInput.value = '';
        cssInput.value = '';
        jsInput.value = '';
        previewSection.classList.add('hidden');
        resultSection.classList.add('hidden');

        // Reset imported ZIP state
        importedFiles = {};
        importedMainPaths = { html: null, css: null, js: null };
        if (zipFileInput) zipFileInput.value = '';
    });

    // Generate AIzipper_YYYYMMDD.zip when new button is clicked
    if (generateAIzipperBtn) {
        generateAIzipperBtn.addEventListener('click', () => {
            const htmlCode = htmlInput.value;
            const cssCode = cssInput.value.trim();
            const jsCode = jsInput.value.trim();

            if (!htmlCode.trim()) {
                alert('Please provide HTML code in the HTML textarea to generate the ZIP.');
                return;
            }
            generateAIzipperFile(htmlCode.trim(), cssCode, jsCode);
        });
    }
    
    // Import a project ZIP and populate editors
    async function importProjectZip(file) {
        const zip = await JSZip.loadAsync(file);

        importedFiles = {};
        importedMainPaths = { html: null, css: null, js: null };

        const textExts = ['.html', '.htm', '.css', '.js', '.json', '.txt', '.md', '.svg'];
        const keys = Object.keys(zip.files);

        for (const path of keys) {
            const entry = zip.files[path];
            if (entry.dir) continue;

            const lower = path.toLowerCase();
            const dot = lower.lastIndexOf('.');
            const ext = dot >= 0 ? lower.slice(dot) : '';

            if (textExts.includes(ext)) {
                const content = await entry.async('string');
                importedFiles[path] = { text: content };

                // Prefer well-known names; otherwise first encountered of each type
                if (!importedMainPaths.html && (lower.endsWith('index.html') || ext === '.html' || ext === '.htm')) {
                    importedMainPaths.html = path;
                }
                if (!importedMainPaths.css && (lower.endsWith('styles.css') || ext === '.css')) {
                    importedMainPaths.css = path;
                }
                if (!importedMainPaths.js && (lower.endsWith('script.js') || ext === '.js')) {
                    importedMainPaths.js = path;
                }
            } else {
                // Binary asset (images, fonts, etc.) – keep base64 so we can re-zip unchanged
                const base64 = await entry.async('base64');
                importedFiles[path] = { base64 };
            }
        }

        // Fallbacks if not found
        if (!importedMainPaths.html) {
            importedMainPaths.html = keys.find(k => k.toLowerCase().endsWith('.html') || k.toLowerCase().endsWith('.htm')) || null;
        }
        if (!importedMainPaths.css) {
            importedMainPaths.css = keys.find(k => k.toLowerCase().endsWith('.css')) || null;
        }
        if (!importedMainPaths.js) {
            importedMainPaths.js = keys.find(k => k.toLowerCase().endsWith('.js')) || null;
        }

        // Populate editors
        htmlInput.value = (importedMainPaths.html && importedFiles[importedMainPaths.html] && importedFiles[importedMainPaths.html].text) ? importedFiles[importedMainPaths.html].text : '';
        cssInput.value = (importedMainPaths.css && importedFiles[importedMainPaths.css] && importedFiles[importedMainPaths.css].text) ? importedFiles[importedMainPaths.css].text : '';
        jsInput.value = (importedMainPaths.js && importedFiles[importedMainPaths.js] && importedFiles[importedMainPaths.js].text) ? importedFiles[importedMainPaths.js].text : '';

        // Notify user
        const assetCount = Object.keys(importedFiles).length;
        const mainsCount = [importedMainPaths.html, importedMainPaths.css, importedMainPaths.js].filter(Boolean).length;
        const others = Math.max(assetCount - mainsCount, 0);
        const mainEl = document.querySelector('main');
        if (mainEl) {
            const notice = document.createElement('div');
            notice.innerHTML = `
                <div style="background:#fff8e1;border:1px solid #fbc02d;border-radius:4px;padding:12px;margin:10px 0;">
                    <strong>ZIP imported:</strong> ${assetCount} file(s) detected.
                    ${others > 0 ? others + ' additional asset(s) will be preserved in the generated ZIP.' : 'No additional assets detected.'}
                </div>
            `;
            mainEl.insertBefore(notice, resultSection);
            setTimeout(() => notice.remove(), 8000);
        }
    }

    // Preview helpers for imported assets
    function normalizePath(p) {
        return p.replace(/\\/g, '/');
    }
    function dirname(p) {
        p = normalizePath(p);
        const idx = p.lastIndexOf('/');
        return idx >= 0 ? p.slice(0, idx) : '';
    }
    function resolvePath(basePath, rel) {
        if (/^https?:\/\//i.test(rel) || rel.startsWith('data:') || rel.startsWith('blob:') || rel.startsWith('#') || rel.startsWith('mailto:') || rel.startsWith('tel:')) {
            return rel; // external or special URLs
        }
        let base = dirname(basePath);
        let segs = (base ? base + '/' + rel : rel).split('/');
        const out = [];
        for (const s of segs) {
            if (!s || s === '.') continue;
            if (s === '..') out.pop();
            else out.push(s);
        }
        return out.join('/');
    }
    function guessMimeFromExt(path) {
        const ext = path.toLowerCase().split('.').pop();
        switch (ext) {
            case 'png': return 'image/png';
            case 'jpg':
            case 'jpeg': return 'image/jpeg';
            case 'gif': return 'image/gif';
            case 'webp': return 'image/webp';
            case 'svg': return 'image/svg+xml';
            case 'ico': return 'image/x-icon';
            case 'bmp': return 'image/bmp';
            case 'mp3': return 'audio/mpeg';
            case 'wav': return 'audio/wav';
            case 'ogg': return 'audio/ogg';
            case 'mp4': return 'video/mp4';
            case 'webm': return 'video/webm';
            case 'woff2': return 'font/woff2';
            case 'woff': return 'font/woff';
            case 'ttf': return 'font/ttf';
            case 'css': return 'text/css';
            case 'js': return 'text/javascript';
            case 'json': return 'application/json';
            case 'txt': return 'text/plain';
            default: return 'application/octet-stream';
        }
    }
    function getBlobUrlForImportedPath(path) {
        const file = importedFiles[path];
        if (!file) return null;
        try {
            let blob;
            if (file.base64) {
                const mime = guessMimeFromExt(path);
                const byteCharacters = atob(file.base64);
                const byteNumbers = new Array(byteCharacters.length);
                for (let i = 0; i < byteCharacters.length; i++) {
                    byteNumbers[i] = byteCharacters.charCodeAt(i);
                }
                const byteArray = new Uint8Array(byteNumbers);
                blob = new Blob([byteArray], { type: mime });
            } else if (typeof file.text === 'string') {
                const mime = guessMimeFromExt(path);
                blob = new Blob([file.text], { type: mime + ';charset=utf-8' });
            } else {
                return null;
            }
            const url = URL.createObjectURL(blob);
            createdBlobUrls.push(url);
            return url;
        } catch (e) {
            console.warn('Failed to create blob URL for', path, e);
            return null;
        }
    }
    function replaceCssUrls(css, basePath) {
        if (!css) return css;
        return css.replace(/url\((['"]?)([^'")]+)\1\)/gi, (m, q, p) => {
            const resolved = resolvePath(basePath, p);
            const file = importedFiles[resolved];
            if (file) {
                const url = getBlobUrlForImportedPath(resolved);
                return `url(${url})`;
            }
            return m;
        });
    }
    function replaceAssetPathsInHtml(html, basePath) {
        // Replace common asset attributes
        html = html.replace(/\b(src|href|poster)=(['"])([^'"]+)\2/gi, (m, attr, q, p) => {
            const resolved = resolvePath(basePath, p);
            const file = importedFiles[resolved];
            if (file) {
                const url = getBlobUrlForImportedPath(resolved);
                return `${attr}=${q}${url}${q}`;
            }
            return m;
        });
        return html;
    }
    function revokeCreatedBlobUrls() {
        createdBlobUrls.forEach(u => URL.revokeObjectURL(u));
        createdBlobUrls = [];
    }

    // Add-on injection helpers (GA, AdSense, footer credit)
    function applyAddOns(html) {
        try {
            let out = html || '';
            const visible = addonsSection && window.getComputedStyle(addonsSection).display !== 'none';
            if (!visible) {
                // Hidden by default means add-ons are OFF
                return out;
            }
            const includeGA = !!(includeGaCheckbox && includeGaCheckbox.checked);
            const includeAds = !!(includeAdsenseCheckbox && includeAdsenseCheckbox.checked);
            const author = (footerAuthorInput && footerAuthorInput.value || 'lookang').trim();
            const creditUrl = (footerUrlInput && footerUrlInput.value || '').trim();
            const freeText = (footerFreeTextInput && footerFreeTextInput.value || '').trim();

            if (includeGA) out = injectGA(out, GA_ID);
            if (includeAds) out = injectAdSense(out, ADSENSE_CLIENT);

            // Prefer free-text credit if provided; otherwise use author/URL format
            if (freeText) {
                out = injectFooterCreditFreeText(out, freeText);
            } else if (author || creditUrl) {
                out = injectFooterCredit(out, author, creditUrl);
            }
            return out;
        } catch (e) {
            console.warn('applyAddOns failed:', e);
            return html;
        }
    }

    function injectGA(html, trackingId) {
        if (!trackingId) return html;
        // Avoid duplicate injection
        if (/\bgoogletagmanager\.com\/gtag\/js\b/i.test(html) || /gtag\(\s*['"]config['"]\s*,/i.test(html)) {
            return html;
        }
        const loader = `<script async="true" src="https://www.googletagmanager.com/gtag/js?id=${trackingId}"></script>`;
        const inline = `<script>
  window.dataLayer = window.dataLayer || [];
  function gtag(){dataLayer.push(arguments);}
  gtag('js', new Date());
  gtag('config', '${trackingId}');
</script>`;
        if (html.includes('</head>')) {
            return html.replace('</head>', `${loader}\n${inline}\n</head>`);
        }
        return `${loader}\n${inline}\n` + html;
    }

    function injectAdSense(html, clientId) {
        if (!clientId) return html;
        // Avoid duplicate injection
        if (/\bpagead2\.googlesyndication\.com\/pagead\/js\/adsbygoogle\.js\b/i.test(html)) {
            return html;
        }
        const tag = `<script data-ad-client="${clientId}" async="true" src="https://pagead2.googlesyndication.com/pagead/js/adsbygoogle.js"></script>`;
        if (html.includes('</head>')) {
            return html.replace('</head>', `${tag}\n</head>`);
        }
        return `${tag}\n` + html;
    }

    // Safely escape HTML before linkifying
    function escapeHtml(text) {
        return text
            .replace(/&/g, '&')
            .replace(/</g, '<')
            .replace(/>/g, '>')
            .replace(/"/g, '"')
            .replace(/'/g, '&#39;');
    }

    // Convert URLs in plain text to clickable anchors
    function linkifyUrls(text) {
        const urlRegex = /(https?:\/\/[^\s<>"']+)/g;
        return text.replace(urlRegex, (m) => `<a href="${m}" target="_blank" rel="noopener noreferrer">${m}</a>`);
    }

    function injectFooterCreditFreeText(html, text) {
        // Avoid duplicate if we've already injected our marker
        if (/id="generated-footer-credit"/i.test(html)) return html;
        const safe = escapeHtml(text);
        const linked = linkifyUrls(safe);
        const creditHtml = `<p id="generated-footer-credit">${linked}</p>`;
        if (/<footer[^>]*>/i.test(html)) {
            return html.replace(/<\/footer>/i, `${creditHtml}\n</footer>`);
        }
        const wrapper = `<footer>\n${creditHtml}\n</footer>`;
        if (html.includes('</body>')) {
            return html.replace('</body>', `${wrapper}\n</body>`);
        }
        return html + `\n${wrapper}`;
    }

    function injectFooterCredit(html, author, url) {
        const name = author && author.trim() ? author.trim() : 'lookang';
        const safeUrl = (url || '').replace(/"/g, '"');
        const linkPart = safeUrl ? ` For more resources: <a href="${safeUrl}" target="_blank" rel="noopener noreferrer">${safeUrl}</a>` : '';
        const creditHtml = `<p>Made by <strong>${name}</strong>, using <strong>Gemini 2.5 Pro</strong>.${linkPart}</p>`;
        // Avoid duplicate
        if (/Made by\s*/i.test(html)) return html;

        if (/<footer[^>]*>/i.test(html)) {
            return html.replace(/<\/footer>/i, `${creditHtml}\n</footer>`);
        }
        const wrapper = `<footer>\n${creditHtml}\n</footer>`;
        if (html.includes('</body>')) {
            return html.replace('</body>', `${wrapper}\n</body>`);
        }
        return html + `\n${wrapper}`;
    }
    
    // Function to detect external URLs in code
    function detectExternalUrls(htmlCode, cssCode, jsCode) {
        const externalUrls = new Set();
        
        // Detect external URLs in HTML
        // Script tags with src
        const scriptMatches = htmlCode.matchAll(/<script[^>]+src\s*=\s*['"]([^'"]+)['"][^>]*>/gi);
        for (const match of scriptMatches) {
            const url = match[1];
            if (isExternalUrl(url)) {
                externalUrls.add(url);
            }
        }
        
        // Link tags with href (CSS)
        const linkMatches = htmlCode.matchAll(/<link[^>]+href\s*=\s*['"]([^'"]+)['"][^>]*>/gi);
        for (const match of linkMatches) {
            const url = match[1];
            if (isExternalUrl(url)) {
                externalUrls.add(url);
            }
        }
        
        // Import maps - detect URLs in JSON import maps
        const importMapMatches = htmlCode.matchAll(/<script[^>]*type\s*=\s*['"]importmap['"][^>]*>([\s\S]*?)<\/script>/gi);
        for (const match of importMapMatches) {
            try {
                const importMapContent = match[1];
                const jsonMatch = importMapContent.match(/\{[\s\S]*\}/);
                if (jsonMatch) {
                    const importMap = JSON.parse(jsonMatch[0]);
                    if (importMap.imports) {
                        for (const [key, value] of Object.entries(importMap.imports)) {
                            if (isExternalUrl(value)) {
                                externalUrls.add(value);
                            }
                            // Handle wildcard imports like "three/addons/"
                            if (typeof value === 'string' && value.endsWith('/') && isExternalUrl(value)) {
                                // For wildcard imports, we'll need to handle them specially
                                externalUrls.add(value);
                            }
                        }
                    }
                }
            } catch (e) {
                console.warn('Failed to parse import map:', e);
            }
        }
        
        // ES6 module imports in script tags
        const moduleScriptMatches = htmlCode.matchAll(/<script[^>]*type\s*=\s*['"]module['"][^>]*>([\s\S]*?)<\/script>/gi);
        for (const match of moduleScriptMatches) {
            const scriptContent = match[1];
            // Find import statements
            const importMatches = scriptContent.matchAll(/import\s+.*?from\s*['"]([^'"]+)['"]/gi);
            for (const importMatch of importMatches) {
                const url = importMatch[1];
                if (isExternalUrl(url)) {
                    externalUrls.add(url);
                }
            }
        }
        
        // Detect external URLs in CSS
        // @import statements
        const importMatches = cssCode.matchAll(/@import\s+(?:url\()?['"]?([^'")\s]+)['"]?\)?/gi);
        for (const match of importMatches) {
            const url = match[1];
            if (isExternalUrl(url)) {
                externalUrls.add(url);
            }
        }
        
        // url() functions in CSS
        const urlMatches = cssCode.matchAll(/url\(['"]?([^'")\s]+)['"]?\)/gi);
        for (const match of urlMatches) {
            const url = match[1];
            if (isExternalUrl(url)) {
                externalUrls.add(url);
            }
        }
        
        // Detect external URLs in JavaScript
        // ES6 import statements
        const jsImportMatches = jsCode.matchAll(/import\s+.*?from\s*['"]([^'"]+)['"]/gi);
        for (const match of jsImportMatches) {
            const url = match[1];
            if (isExternalUrl(url)) {
                externalUrls.add(url);
            }
        }
        
        // Dynamic imports
        const dynamicImportMatches = jsCode.matchAll(/import\s*\(\s*['"]([^'"]+)['"]\s*\)/gi);
        for (const match of dynamicImportMatches) {
            const url = match[1];
            if (isExternalUrl(url)) {
                externalUrls.add(url);
            }
        }
        
        // Common patterns for loading external resources
        const jsUrlPatterns = [
            /['"]https?:\/\/[^'"]+['"]/gi,
            /src\s*[:=]\s*['"]https?:\/\/[^'"]+['"]/gi,
            /href\s*[:=]\s*['"]https?:\/\/[^'"]+['"]/gi,
            /url\s*[:=]\s*['"]https?:\/\/[^'"]+['"]/gi
        ];
        
        for (const pattern of jsUrlPatterns) {
            const matches = jsCode.matchAll(pattern);
            for (const match of matches) {
                const url = match[0].replace(/['"]/g, '').replace(/^(src|href|url)\s*[:=]\s*/, '');
                if (isExternalUrl(url)) {
                    externalUrls.add(url);
                }
            }
        }
        
        return Array.from(externalUrls);
    }
    
    // Function to check if URL is external
    function isExternalUrl(url) {
        return /^https?:\/\//.test(url) && !url.includes('localhost') && !url.includes('127.0.0.1');
    }
    
    // Function to download external file
    async function downloadExternalFile(url) {
        try {
            const response = await fetch(url, {
                mode: 'cors',
                headers: {
                    'Accept': '*/*'
                }
            });
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const content = await response.text();
            return content;
        } catch (error) {
            console.warn(`Failed to download ${url}:`, error);
            // Return a comment indicating the file couldn't be downloaded
            const fileType = url.split('.').pop().toLowerCase();
            if (fileType === 'js') {
                return `// Failed to download: ${url}\n// Error: ${error.message}\n// Please manually include this file or check the URL.`;
            } else if (fileType === 'css') {
                return `/* Failed to download: ${url} */\n/* Error: ${error.message} */\n/* Please manually include this file or check the URL. */`;
            }
            return `<!-- Failed to download: ${url} -->\n<!-- Error: ${error.message} -->\n<!-- Please manually include this file or check the URL. -->`;
        }
    }
    
    // Function to generate local filename from URL
    function generateLocalFilename(url, index = 0) {
        try {
            const urlObj = new URL(url);
            let filename = urlObj.pathname.split('/').pop();
            
            // If no filename or extension, generate one
            if (!filename || !filename.includes('.')) {
                const extension = url.includes('.css') ? 'css' : 
                                url.includes('.js') ? 'js' : 
                                url.includes('font') ? 'woff2' : 'txt';
                filename = `external_${index}.${extension}`;
            }
            
            // Ensure unique filename and proper extension for modules
            if (index > 0) {
                const parts = filename.split('.');
                const ext = parts.pop();
                const name = parts.join('.');
                filename = `${name}_${index}.${ext}`;
            }
            
            // Ensure .js files that are modules have proper extension
            if (url.includes('module.js') || url.includes('/jsm/')) {
                if (!filename.endsWith('.js')) {
                    filename = filename.replace(/\.[^.]*$/, '.js');
                }
            }
            
            return filename;
        } catch (error) {
            const extension = url.includes('.css') ? 'css' : 
                            url.includes('.js') ? 'js' : 'txt';
            return `external_${index}.${extension}`;
        }
    }
    
    // Function to update code with local file references
    function updateCodeWithLocalFiles(htmlCode, cssCode, jsCode, urlMappings) {
        let updatedHtml = htmlCode;
        let updatedCss = cssCode;
        let updatedJs = jsCode;
        
        // Update HTML
        for (const [originalUrl, localFilename] of Object.entries(urlMappings)) {
            // Update script src attributes
            updatedHtml = updatedHtml.replace(
                new RegExp(`(<script[^>]+src\\s*=\\s*['"])${escapeRegExp(originalUrl)}(['"][^>]*>)`, 'gi'),
                `$1${localFilename}$2`
            );
            
            // Update link href attributes
            updatedHtml = updatedHtml.replace(
                new RegExp(`(<link[^>]+href\\s*=\\s*['"])${escapeRegExp(originalUrl)}(['"][^>]*>)`, 'gi'),
                `$1${localFilename}$2`
            );
            
            // Update import maps
            updatedHtml = updatedHtml.replace(
                new RegExp(`(['"])${escapeRegExp(originalUrl)}(['"])`, 'gi'),
                `$1${localFilename}$2`
            );
            
            // Update ES6 imports in inline script tags
            updatedHtml = updatedHtml.replace(
                new RegExp(`(import\\s+.*?from\\s*['"])${escapeRegExp(originalUrl)}(['"])`, 'gi'),
                `$1${localFilename}$2`
            );
            
            // Update dynamic imports in inline script tags
            updatedHtml = updatedHtml.replace(
                new RegExp(`(import\\s*\\(\\s*['"])${escapeRegExp(originalUrl)}(['"]\\s*\\))`, 'gi'),
                `$1${localFilename}$2`
            );
        }
        
        // Update CSS
        for (const [originalUrl, localFilename] of Object.entries(urlMappings)) {
            // Update @import statements
            updatedCss = updatedCss.replace(
                new RegExp(`(@import\\s+(?:url\\()?['"]?)${escapeRegExp(originalUrl)}(['"]?\\)?[^;]*;?)`, 'gi'),
                `$1${localFilename}$2`
            );
            
            // Update url() functions
            updatedCss = updatedCss.replace(
                new RegExp(`(url\\(['"]?)${escapeRegExp(originalUrl)}(['"]?\\))`, 'gi'),
                `$1${localFilename}$2`
            );
        }
        
        // Update JavaScript
        for (const [originalUrl, localFilename] of Object.entries(urlMappings)) {
            // Update ES6 import statements
            updatedJs = updatedJs.replace(
                new RegExp(`(import\\s+.*?from\\s*['"])${escapeRegExp(originalUrl)}(['"])`, 'gi'),
                `$1${localFilename}$2`
            );
            
            // Update dynamic imports
            updatedJs = updatedJs.replace(
                new RegExp(`(import\\s*\\(\\s*['"])${escapeRegExp(originalUrl)}(['"]\\s*\\))`, 'gi'),
                `$1${localFilename}$2`
            );
            
            // Update general URL references
            updatedJs = updatedJs.replace(
                new RegExp(`(['"])${escapeRegExp(originalUrl)}(['"])`, 'gi'),
                `$1${localFilename}$2`
            );
        }
        
        return { updatedHtml, updatedCss, updatedJs };
    }
    
    // Helper function to escape special regex characters
    function escapeRegExp(string) {
        return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
    }
    
    // Function to convert ES6 modules to SLS-compatible format
    function convertToSlsCompatible(htmlCode, urlMappings) {
        let updatedHtml = htmlCode;
        
        // Check if the HTML contains ES6 modules or import maps
        const hasImportMap = /<script[^>]*type\s*=\s*['"]importmap['"][^>]*>/i.test(updatedHtml);
        const hasModuleScript = /<script[^>]*type\s*=\s*['"]module['"][^>]*>/i.test(updatedHtml);
        
        if (!hasImportMap && !hasModuleScript) {
            return updatedHtml; // No ES6 modules to convert
        }
        
        // Create SLS-compatible version
        let slsCompatibleHtml = updatedHtml;
        
        // Extract import map information before removing it
        let importMapData = {};
        const importMapMatches = slsCompatibleHtml.matchAll(/<script[^>]*type\s*=\s*['"]importmap['"][^>]*>([\s\S]*?)<\/script>/gi);
        for (const match of importMapMatches) {
            try {
                const importMapContent = match[1];
                const jsonMatch = importMapContent.match(/\{[\s\S]*\}/);
                if (jsonMatch) {
                    const importMap = JSON.parse(jsonMatch[0]);
                    if (importMap.imports) {
                        importMapData = { ...importMapData, ...importMap.imports };
                    }
                }
            } catch (e) {
                console.warn('Failed to parse import map for conversion:', e);
            }
        }
        
        // Remove import maps entirely (they don't work in SLS)
        slsCompatibleHtml = slsCompatibleHtml.replace(/<script[^>]*type\s*=\s*['"]importmap['"][^>]*>[\s\S]*?<\/script>/gi, 
            '<!-- Import map removed for SLS compatibility -->');
        
        // Add traditional script tags for libraries that were in the import map
        let traditionalScripts = '';
        for (const [key, value] of Object.entries(importMapData)) {
            if (urlMappings[value]) {
                // Find the local filename for this URL
                const localFile = urlMappings[value];
                if (key === 'three') {
                    traditionalScripts += `<script src="${localFile}"></script>\n`;
                }
            }
        }
        
        // Insert traditional script tags before the first module script
        if (traditionalScripts) {
            const firstModuleScript = slsCompatibleHtml.search(/<script[^>]*type\s*=\s*['"]module['"][^>]*>/i);
            if (firstModuleScript !== -1) {
                slsCompatibleHtml = slsCompatibleHtml.slice(0, firstModuleScript) + 
                                  traditionalScripts + 
                                  slsCompatibleHtml.slice(firstModuleScript);
            } else {
                // If no module script found, add before closing head
                if (slsCompatibleHtml.includes('</head>')) {
                    slsCompatibleHtml = slsCompatibleHtml.replace('</head>', traditionalScripts + '</head>');
                }
            }
        }
        
        // Convert module scripts to regular scripts
        slsCompatibleHtml = slsCompatibleHtml.replace(/<script([^>]*)\s+type\s*=\s*['"]module['"]([^>]*)>/gi, 
            '<script$1$2>');
        
        // Convert ES6 import statements to global variable access
        // Handle different import patterns
        slsCompatibleHtml = slsCompatibleHtml.replace(/import\s+\*\s+as\s+(\w+)\s+from\s+['"]three['"];?\s*/gi, 
            '// Using global THREE instead of ES6 import\n        const $1 = window.THREE || THREE;\n        ');
        
        slsCompatibleHtml = slsCompatibleHtml.replace(/import\s+\{\s*([^}]+)\s*\}\s+from\s+['"]three\/addons\/([^'"]*?)['"];?\s*/gi, 
            '// Three.js addons - converted for SLS compatibility\n        // Original: import { $1 } from "three/addons/$2"\n        // Note: Addons may need manual integration\n        ');
        
        // Handle other common import patterns
        slsCompatibleHtml = slsCompatibleHtml.replace(/import\s+([^{][^from]*?)\s+from\s+['"]three['"];?\s*/gi, 
            '// Using global THREE instead of ES6 import\n        const $1 = window.THREE || THREE;\n        ');
        
        // Add a comment explaining the conversion
        const conversionNote = `
    <!-- 
    ⚠️ SLS COMPATIBILITY MODE ACTIVATED ⚠️
    
    This HTML has been automatically converted for Singapore Learning Space (SLS) compatibility:
    - Import maps have been removed (not supported in SLS)
    - ES6 module scripts converted to regular scripts
    - Import statements converted to global variable access
    - External libraries loaded as traditional script tags
    
    Note: Some advanced features may not work in SLS due to security restrictions.
    For full functionality, use the included local server files.
    -->
    `;
        
        // Add the conversion note after the opening <head> tag
        if (slsCompatibleHtml.includes('<head>')) {
            slsCompatibleHtml = slsCompatibleHtml.replace('<head>', '<head>' + conversionNote);
        } else {
            slsCompatibleHtml = conversionNote + slsCompatibleHtml;
        }
        
        return slsCompatibleHtml;
    }
    
    // Function to show SLS compatibility warnings
    function showSlsWarnings(externalUrls) {
        if (externalUrls.length === 0) return '';
        
        const warnings = [
            '⚠️ SLS (Singapore Learning Space) Compatibility Warnings:',
            '',
            'The following external dependencies were detected and downloaded:',
            ...externalUrls.map(url => `• ${url}`),
            '',
            '🚨 IMPORTANT: ES6 Modules and Import Maps Limitation:',
            'If your code uses ES6 modules (import/export) or import maps, it will NOT work',
            'when opened directly as a file (file://) due to CORS restrictions.',
            '',
            '✅ SOLUTION: Use the included local server',
            '• Windows: Double-click "start_server.bat"',
            '• Mac/Linux: Run "bash start_server.sh" or "python3 server.py"',
            '• Then open http://localhost:8000 in your browser',
            '',
            'Note: Some features may not work properly in SLS due to:',
            '• Restricted internet access in SLS environment',
            '• Content Security Policy limitations',
            '• JavaScript execution restrictions',
            '• ES6 module loading restrictions',
            '',
            'All external files have been downloaded and included in this ZIP for offline use.',
            'If you encounter issues in SLS, consider:',
            '• Using simpler, vanilla JavaScript instead of complex libraries',
            '• Avoiding external API calls',
            '• Converting ES6 modules to regular script tags',
            '• Testing thoroughly in the SLS environment',
            ''
        ];
        
        return warnings.join('\n');
    }

    // Function to generate ZIP file
    async function generateZip(htmlCode, cssCode, jsCode) {
        try {
            const zip = new JSZip();
            let updatedHtmlCode = htmlCode;
            let updatedCssCode = cssCode;
            let updatedJsCode = jsCode;

            const preferredCssPath = importedMainPaths.css || 'styles.css';
            const preferredJsPath = importedMainPaths.js || 'script.js';
            
            // Detect external URLs
            const externalUrls = detectExternalUrls(htmlCode, cssCode, jsCode);
            
            if (externalUrls.length > 0) {
                // Show progress for downloading external files
                const progressDiv = document.createElement('div');
                progressDiv.innerHTML = `
                    <div style="background: #e3f2fd; border: 1px solid #2196f3; border-radius: 4px; padding: 15px; margin: 10px 0;">
                        <h4 style="margin: 0 0 10px 0; color: #1976d2;">📥 Downloading External Dependencies</h4>
                        <div id="download-progress">Found ${externalUrls.length} external file(s) to download...</div>
                    </div>
                `;
                
                // Insert progress div before result section
                const main = document.querySelector('main');
                main.insertBefore(progressDiv, resultSection);
                
                const progressElement = document.getElementById('download-progress');
                const urlMappings = {};
                
                // Download external files
                for (let i = 0; i < externalUrls.length; i++) {
                    const url = externalUrls[i];
                    progressElement.textContent = `Downloading ${i + 1}/${externalUrls.length}: ${url}`;
                    
                    const content = await downloadExternalFile(url);
                    const localFilename = generateLocalFilename(url, i);
                    
                    // Add to ZIP
                    zip.file(localFilename, content);
                    urlMappings[url] = localFilename;
                }
                
                // Update code with local file references
                const updated = updateCodeWithLocalFiles(htmlCode, cssCode, jsCode, urlMappings);
                updatedHtmlCode = updated.updatedHtml;
                updatedCssCode = updated.updatedCss;
                updatedJsCode = updated.updatedJs;
                
                // Convert ES6 modules to SLS-compatible format
                updatedHtmlCode = convertToSlsCompatible(updatedHtmlCode, urlMappings);
                
                // Add SLS warnings as a README file
                const slsWarnings = showSlsWarnings(externalUrls);
                if (slsWarnings) {
                    zip.file('README_SLS_COMPATIBILITY.txt', slsWarnings);
                }
                
                // Add a simple HTTP server file for local testing
                const serverScript = `#!/usr/bin/env python3
# Simple HTTP Server for testing ES6 modules locally
# Usage: python3 server.py
# Then open http://localhost:8000 in your browser

import http.server
import socketserver
import os

PORT = 8000

class MyHTTPRequestHandler(http.server.SimpleHTTPRequestHandler):
    def end_headers(self):
        self.send_header('Cross-Origin-Embedder-Policy', 'require-corp')
        self.send_header('Cross-Origin-Opener-Policy', 'same-origin')
        super().end_headers()

os.chdir(os.path.dirname(os.path.abspath(__file__)))

with socketserver.TCPServer(("", PORT), MyHTTPRequestHandler) as httpd:
    print(f"Server running at http://localhost:{PORT}/")
    print("Press Ctrl+C to stop the server")
    try:
        httpd.serve_forever()
    except KeyboardInterrupt:
        print("\\nServer stopped.")
`;
                zip.file('server.py', serverScript);
                
                // Add a batch file for Windows users
                const batchScript = `@echo off
echo Starting local HTTP server...
echo Open http://localhost:8000 in your browser
echo Press Ctrl+C to stop the server
python server.py
pause
`;
                zip.file('start_server.bat', batchScript);
                
                // Add a shell script for Mac/Linux users
                const shellScript = `#!/bin/bash
echo "Starting local HTTP server..."
echo "Open http://localhost:8000 in your browser"
echo "Press Ctrl+C to stop the server"
python3 server.py
`;
                zip.file('start_server.sh', shellScript);
                
                // Remove progress div
                progressDiv.remove();
            }

            // Add stylesheet if CSS code is provided and update HTML links
            if (updatedCssCode !== '') {
                zip.file(preferredCssPath, updatedCssCode);
                // Update existing link hrefs to point to preferredCssPath (unless external placeholder)
                updatedHtmlCode = updatedHtmlCode.replace(/<link\s+[^>]*href\s*=\s*['"]([^'"]*\.(?:css|scss|less))['"][^>]*>/gi, (match, oldHref) => {
                    if (oldHref.toLowerCase() !== preferredCssPath.toLowerCase() && !oldHref.startsWith('external_')) {
                        return match.replace(oldHref, preferredCssPath);
                    }
                    return match;
                });
                // If no link tag for preferredCssPath exists after replacement and no inline <style> tag, add one.
                if (!updatedHtmlCode.match(new RegExp(`<link\\s+[^>]*href\\s*=\\s*['"]${escapeRegExp(preferredCssPath)}['"][^>]*>`, 'gi')) && !updatedHtmlCode.includes('<style>')) {
                    if (updatedHtmlCode.includes('</head>')) {
                        updatedHtmlCode = updatedHtmlCode.replace('</head>', `<link rel="stylesheet" href="${preferredCssPath}">\n</head>`);
                    } else {
                        // Fallback if no </head> tag, prepend to the document (less ideal)
                        updatedHtmlCode = `<link rel="stylesheet" href="${preferredCssPath}">\n` + updatedHtmlCode;
                    }
                }
            }
            
            // Add script file if JavaScript code is provided and update HTML links
            if (updatedJsCode !== '') {
                zip.file(preferredJsPath, updatedJsCode);
                // Update existing script src to point to preferredJsPath (unless external placeholder)
                updatedHtmlCode = updatedHtmlCode.replace(/<script\s+[^>]*src\s*=\s*['"]([^'"]*\.js)['"][^>]*>\s*<\/script>/gi, (match, oldSrc) => {
                    if (oldSrc.toLowerCase() !== preferredJsPath.toLowerCase() && !oldSrc.startsWith('external_')) {
                        return match.replace(oldSrc, preferredJsPath);
                    }
                    return match;
                });
                // If no script tag for preferredJsPath exists after replacement and no inline <script> (without src), add one.
                if (!updatedHtmlCode.match(new RegExp(`<script\\s+[^>]*src\\s*=\\s*['"]${escapeRegExp(preferredJsPath)}['"][^>]*>\\s*<\\/script>`, 'gi')) && !updatedHtmlCode.match(/<script\s*(?!src)[^>]*>[\s\S]*?<\/script>/gi)) {
                    if (updatedHtmlCode.includes('</body>')) {
                        updatedHtmlCode = updatedHtmlCode.replace('</body>', `<script src="${preferredJsPath}"></script>\n</body>`);
                    } else {
                        // Fallback if no </body> tag, append to the document (less ideal)
                        updatedHtmlCode += `\n<script src="${preferredJsPath}"></script>`;
                    }
                }
            }

            // Include any additional imported files (excluding the main HTML/CSS/JS which are added below)
            if (Object.keys(importedFiles).length) {
                Object.entries(importedFiles).forEach(([path, file]) => {
                    const lower = path.toLowerCase();
                    const isMain =
                        (importedMainPaths.html && lower === importedMainPaths.html.toLowerCase()) ||
                        (importedMainPaths.css && lower === importedMainPaths.css.toLowerCase()) ||
                        (importedMainPaths.js && lower === importedMainPaths.js.toLowerCase());
                    if (isMain) return; // main files handled below

                    try {
                        if (file && file.base64) {
                            zip.file(path, file.base64, { base64: true });
                        } else if (file && typeof file.text === 'string') {
                            zip.file(path, file.text);
                        }
                    } catch (e) {
                        console.warn('Failed to add imported file to ZIP:', path, e);
                    }
                });
            }

            // Apply optional add-ons (GA, AdSense, footer credit) before saving HTML
            updatedHtmlCode = applyAddOns(updatedHtmlCode);

            // Always add index.html (potentially updated)
            zip.file('index.html', updatedHtmlCode);
            
            // Generate the ZIP file
            const zipBlob = await zip.generateAsync({ type: 'blob' });
            
            // Extract title from HTML content
            let title = 'gemini_code'; // Default title
            const titleMatch = updatedHtmlCode.match(/<title>([^<]+)<\/title>/i);
            if (titleMatch && titleMatch[1]) {
                title = titleMatch[1].trim().replace(/\s+/g, '_'); // Replace spaces with underscores
            }
            
            // Create a timestamp for the filename (YYYYMMDD format)
            const now = new Date();
            const year = now.getFullYear();
            const month = String(now.getMonth() + 1).padStart(2, '0');
            const day = String(now.getDate()).padStart(2, '0');
            const timestamp = `${year}${month}${day}`;
            
            // Create a meaningful filename with title and timestamp
            const filename = `${title}_${timestamp}.zip`;
            
            // Create a download URL for the ZIP file
            const downloadUrl = URL.createObjectURL(zipBlob);
            
            // Update the UI
            filenameDisplay.textContent = filename;
            downloadLink.href = downloadUrl;
            downloadLink.download = filename;
            resultSection.classList.remove('hidden');
            
            // Show success message with external files info
            if (externalUrls.length > 0) {
                const successDiv = document.createElement('div');
                successDiv.innerHTML = `
                    <div style="background: #e8f5e8; border: 1px solid #4caf50; border-radius: 4px; padding: 15px; margin: 10px 0;">
                        <h4 style="margin: 0 0 10px 0; color: #2e7d32;">✅ Self-Contained ZIP Created Successfully!</h4>
                        <p style="margin: 0;">Downloaded and included ${externalUrls.length} external file(s). Check README_SLS_COMPATIBILITY.txt for important notes about SLS usage.</p>
                    </div>
                `;
                resultSection.insertBefore(successDiv, resultSection.firstChild);
                
                // Remove success message after 10 seconds
                setTimeout(() => {
                    if (successDiv.parentNode) {
                        successDiv.remove();
                    }
                }, 10000);
            }
            
            // Scroll to the result section
            resultSection.scrollIntoView({ behavior: 'smooth' });
        } catch (error) {
            console.error('Error generating ZIP file:', error);
            alert('An error occurred while generating the ZIP file. Please try again.');
        }
    }

    // Function to generate AIzipper_YYYYMMDD.zip
    async function generateAIzipperFile(htmlCode, cssCode, jsCode) {
        try {
            const zip = new JSZip();
            
            zip.file('index.html', htmlCode); 
            
            if (cssCode.trim() !== '') {
                zip.file('styles.css', cssCode.trim());
            }
            
            if (jsCode.trim() !== '') {
                zip.file('script.js', jsCode.trim());
            }
            
            const zipBlob = await zip.generateAsync({ type: 'blob' });
            
            const now = new Date();
            const year = now.getFullYear();
            const month = String(now.getMonth() + 1).padStart(2, '0');
            const day = String(now.getDate()).padStart(2, '0');
            const timestamp = `${year}${month}${day}`;
            
            const filename = `AIzipper_${timestamp}.zip`;
            
            const downloadUrl = URL.createObjectURL(zipBlob);
            const tempLink = document.createElement('a');
            tempLink.href = downloadUrl;
            tempLink.download = filename;
            document.body.appendChild(tempLink);
            tempLink.click();
            document.body.removeChild(tempLink);
            URL.revokeObjectURL(downloadUrl);

            resultSection.classList.add('hidden');
            alert(`Downloading ${filename}...`); 
        } catch (error) {
            console.error('Error generating AIzipper ZIP file:', error);
            alert('An error occurred while generating the AIzipper ZIP file. Please try again.');
        }
    }
});
