// ===== DETECT IF RUNNING IN IFRAME =====
// Adjust container height based on whether we're in an iframe or standalone
function detectIframe() {
    if (window.self !== window.top) {
        document.body.classList.add('in-iframe');
    }
}
detectIframe();

// ===== FLOWER PARTS DATA =====
// Updated with exact terminology from user instructions
const flowerParts = {
    anther: {
        name: 'Anther',
        type: 'male',
        description: 'The anther contains pollen grains.',
        details: [
            'Contains pollen grains',
            'Pollen grains contain the male reproductive cells',
            'Part of the male reproductive structures'
        ]
    },
    filament: {
        name: 'Filament',
        type: 'male',
        description: 'The filament holds up the anther.',
        details: [
            'Holds up the anther',
            'Part of the male reproductive structures',
            'Supports the anther for pollination'
        ]
    },
    stigma: {
        name: 'Stigma',
        type: 'female',
        description: 'The stigma receives pollen grains.',
        details: [
            'Receives pollen grains',
            'In some flowering plants, the stigma is sticky so that pollen grains can be collected easily',
            'Part of the female reproductive structures'
        ]
    },
    style: {
        name: 'Style',
        type: 'female',
        description: 'The style connects the stigma to the ovary.',
        details: [
            'Connects the stigma to the ovary',
            'Part of the female reproductive structures',
            'Acts as a pathway between stigma and ovary'
        ]
    },
    ovary: {
        name: 'Ovary',
        type: 'female',
        description: 'The ovary contains ovules.',
        details: [
            'Contains ovules',
            'Part of the female reproductive structures',
            'Located at the base of the flower'
        ]
    },
    ovule: {
        name: 'Ovule',
        type: 'female',
        description: 'The ovules contain female reproductive cells, also known as egg cells.',
        details: [
            'Contains female reproductive cells (egg cells)',
            'Found inside the ovary',
            'Part of the female reproductive structures',
            'Essential for reproduction'
        ]
    }
};

// ===== GUIDED LEARNING STEPS =====
// Updated with exact terminology from user instructions
const guidedSteps = [
    {
        title: 'Welcome to Flower Reproduction! 🌸',
        content: 'Flowers have special parts that help them reproduce. Let\'s learn about the male reproductive structures first, then the female reproductive structures.',
        highlight: []
    },
    {
        title: 'Male Reproductive Structure 1: Anther',
        content: 'The <strong>anther</strong> contains pollen grains. Pollen grains contain the male reproductive cells.',
        highlight: ['anther']
    },
    {
        title: 'Male Reproductive Structure 2: Filament',
        content: 'The <strong>filament</strong> holds up the anther. Together, the anther and filament are the male reproductive structures.',
        highlight: ['filament', 'anther']
    },
    {
        title: 'Female Reproductive Structure 1: Stigma',
        content: 'The <strong>stigma</strong> receives pollen grains. In some flowering plants, the stigma is sticky so that pollen grains can be collected easily.',
        highlight: ['stigma']
    },
    {
        title: 'Female Reproductive Structure 2: Style',
        content: 'The <strong>style</strong> connects the stigma to the ovary. It acts as a pathway for pollen.',
        highlight: ['style', 'stigma']
    },
    {
        title: 'Female Reproductive Structure 3: Ovary',
        content: 'The <strong>ovary</strong> contains ovules. It is located at the base of the flower.',
        highlight: ['ovary']
    },
    {
        title: 'Female Reproductive Structure 4: Ovule',
        content: 'The <strong>ovules</strong> contain female reproductive cells, also known as egg cells. The ovules are found inside the ovary.',
        highlight: ['ovule', 'ovary']
    }
];

// ===== QUIZ QUESTIONS =====
// Updated with exact terminology from user instructions
const quizQuestions = [
    {
        type: 'multiple-choice',
        question: 'Which part contains pollen grains?',
        choices: ['Stigma', 'Anther', 'Ovary', 'Style'],
        correct: 1,
        feedback: {
            correct: 'Excellent! The anther contains pollen grains. ✓',
            incorrect: 'Not quite. Think about which male reproductive structure contains pollen grains.'
        }
    },
    {
        type: 'fill-blank',
        question: 'The ______ contains ovules.',
        answer: 'ovary',
        feedback: {
            correct: 'Well done! The ovary contains ovules. ✓',
            incorrect: 'Try again. Which female reproductive structure contains ovules?'
        }
    },
    {
        type: 'multiple-choice',
        question: 'What does the filament do?',
        choices: ['Contains pollen grains', 'Holds up the anther', 'Receives pollen grains', 'Contains egg cells'],
        correct: 1,
        feedback: {
            correct: 'Correct! The filament holds up the anther. ✓',
            incorrect: 'Think about what the filament does for the anther.'
        }
    },
    {
        type: 'fill-blank',
        question: 'The ______ receives pollen grains.',
        answer: 'stigma',
        feedback: {
            correct: 'Perfect! The stigma receives pollen grains. ✓',
            incorrect: 'Hint: This part may be sticky in some flowering plants.'
        }
    },
    {
        type: 'multiple-choice',
        question: 'What do ovules contain?',
        choices: ['Male reproductive cells', 'Pollen grains', 'Female reproductive cells (egg cells)', 'Petals'],
        correct: 2,
        feedback: {
            correct: 'Excellent! The ovules contain female reproductive cells, also known as egg cells. ✓',
            incorrect: 'Remember, ovules are part of the female reproductive structures.'
        }
    },
    {
        type: 'multiple-choice',
        question: 'The style connects which two parts?',
        choices: ['Anther and filament', 'Stigma and ovary', 'Ovary and ovule', 'Petal and sepal'],
        correct: 1,
        feedback: {
            correct: 'Great work! The style connects the stigma to the ovary. ✓',
            incorrect: 'Think about which female reproductive structures the style connects.'
        }
    }
];

// ===== STATE MANAGEMENT =====
let currentMode = 'explore';
let currentGuidedStep = 0;
let currentQuizQuestion = 0;
let quizScore = 0;
let quizAnswered = [];

// ===== MODE SWITCHING =====
// Handle switching between Explore, Guided, and Quiz modes
const modeButtons = document.querySelectorAll('.mode-btn');
const modeContents = document.querySelectorAll('.mode-content');

modeButtons.forEach(btn => {
    btn.addEventListener('click', () => {
        const mode = btn.dataset.mode;
        switchMode(mode);
    });
});

function switchMode(mode) {
    currentMode = mode;
    
    // Update button states
    modeButtons.forEach(btn => {
        btn.classList.toggle('active', btn.dataset.mode === mode);
    });
    
    // Update content visibility
    modeContents.forEach(content => {
        content.classList.remove('active');
    });
    
    // Show selected mode
    if (mode === 'explore') {
        document.getElementById('explore-mode').classList.add('active');
        resetExploreMode();
    } else if (mode === 'guided') {
        document.getElementById('guided-mode').classList.add('active');
        currentGuidedStep = 0;
        showGuidedStep(0);
    } else if (mode === 'quiz') {
        document.getElementById('quiz-mode').classList.add('active');
        startQuiz();
    }
}

// ===== EXPLORE MODE =====
// Interactive exploration of flower parts
function resetExploreMode() {
    const infoDisplay = document.getElementById('info-display');
    infoDisplay.innerHTML = `
        <div class="welcome-message">
            <h2>Welcome! 🌸</h2>
            <p>Click on any part of the flower to learn about its reproductive structures.</p>
            <div class="legend">
                <div class="legend-item">
                    <span class="legend-color male"></span>
                    <span>Male parts</span>
                </div>
                <div class="legend-item">
                    <span class="legend-color female"></span>
                    <span>Female parts</span>
                </div>
            </div>
        </div>
    `;
    
    // Remove all highlights
    document.querySelectorAll('.flower-part').forEach(part => {
        part.classList.remove('highlighted');
    });
}

// Handle clicks on flower parts in explore mode
document.querySelectorAll('#flower-diagram .flower-part').forEach(part => {
    part.addEventListener('click', (e) => {
        if (currentMode !== 'explore') return;
        
        const partName = e.target.dataset.part;
        showPartInfo(partName);
        
        // Highlight clicked part
        document.querySelectorAll('.flower-part').forEach(p => {
            p.classList.remove('highlighted');
        });
        
        // Highlight all parts of the same type
        document.querySelectorAll(`[data-part="${partName}"]`).forEach(p => {
            p.classList.add('highlighted');
        });
    });
});

// Display information about a flower part
function showPartInfo(partName) {
    const part = flowerParts[partName];
    if (!part) return;
    
    const infoDisplay = document.getElementById('info-display');
    
    const detailsList = part.details.map(detail => `<li>${detail}</li>`).join('');
    
    infoDisplay.innerHTML = `
        <div class="part-info">
            <h2>${part.name}</h2>
            <span class="part-type ${part.type}">${part.type.toUpperCase()} PART</span>
            <p><strong>${part.description}</strong></p>
            <ul>
                ${detailsList}
            </ul>
        </div>
    `;
}

// ===== GUIDED MODE =====
// Step-by-step learning experience
function showGuidedStep(stepIndex) {
    if (stepIndex < 0 || stepIndex >= guidedSteps.length) return;
    
    currentGuidedStep = stepIndex;
    const step = guidedSteps[stepIndex];
    
    // Update text content
    const guidedText = document.querySelector('.guided-text');
    guidedText.innerHTML = `
        <h3>${step.title}</h3>
        <p>${step.content}</p>
    `;
    
    // Update step indicator
    document.getElementById('step-indicator').textContent = `Step ${stepIndex + 1} of ${guidedSteps.length}`;
    
    // Update navigation buttons
    document.getElementById('prev-step').disabled = stepIndex === 0;
    document.getElementById('next-step').disabled = stepIndex === guidedSteps.length - 1;
    
    // Highlight relevant parts in diagram
    const guidedParts = document.querySelectorAll('.guided-part');
    guidedParts.forEach(part => {
        part.classList.remove('active');
        const partName = part.dataset.part;
        if (step.highlight.includes(partName)) {
            part.classList.add('active');
        }
    });
}

// Navigation buttons for guided mode
document.getElementById('prev-step').addEventListener('click', () => {
    showGuidedStep(currentGuidedStep - 1);
});

document.getElementById('next-step').addEventListener('click', () => {
    showGuidedStep(currentGuidedStep + 1);
});

// ===== QUIZ MODE =====
// Assessment with immediate feedback
function startQuiz() {
    currentQuizQuestion = 0;
    quizScore = 0;
    quizAnswered = new Array(quizQuestions.length).fill(false);
    
    document.getElementById('question-area').style.display = 'block';
    document.getElementById('results-area').style.display = 'none';
    
    showQuestion(0);
}

function showQuestion(questionIndex) {
    if (questionIndex < 0 || questionIndex >= quizQuestions.length) {
        showResults();
        return;
    }
    
    currentQuizQuestion = questionIndex;
    const question = quizQuestions[questionIndex];
    
    // Update question number
    document.getElementById('question-number').textContent = `Question ${questionIndex + 1} of ${quizQuestions.length}`;
    
    // Update question text
    document.getElementById('question-text').textContent = question.question;
    
    // Clear previous answer and feedback
    const answerArea = document.getElementById('answer-area');
    const feedbackArea = document.getElementById('feedback-area');
    feedbackArea.style.display = 'none';
    feedbackArea.className = '';
    document.getElementById('next-question').style.display = 'none';
    
    // Render answer input based on question type
    if (question.type === 'multiple-choice') {
        answerArea.innerHTML = question.choices.map((choice, index) => 
            `<button class="choice-btn" data-index="${index}">${choice}</button>`
        ).join('');
        
        // Add click handlers to choice buttons
        document.querySelectorAll('.choice-btn').forEach(btn => {
            btn.addEventListener('click', (e) => {
                checkMultipleChoice(parseInt(e.target.dataset.index));
            });
        });
    } else if (question.type === 'fill-blank') {
        answerArea.innerHTML = `
            <input type="text" class="blank-input" id="blank-answer" placeholder="Type your answer here">
            <br><br>
            <button class="submit-answer" id="submit-blank">Submit Answer</button>
        `;
        
        // Add submit handler
        document.getElementById('submit-blank').addEventListener('click', checkFillBlank);
        
        // Allow Enter key to submit
        document.getElementById('blank-answer').addEventListener('keypress', (e) => {
            if (e.key === 'Enter') {
                checkFillBlank();
            }
        });
    }
}

// Check multiple choice answer
function checkMultipleChoice(selectedIndex) {
    const question = quizQuestions[currentQuizQuestion];
    const isCorrect = selectedIndex === question.correct;
    
    // Mark answer as attempted
    quizAnswered[currentQuizQuestion] = true;
    
    // Update score
    if (isCorrect) {
        quizScore++;
    }
    
    // Disable all buttons
    document.querySelectorAll('.choice-btn').forEach(btn => {
        btn.disabled = true;
        const index = parseInt(btn.dataset.index);
        if (index === question.correct) {
            btn.classList.add('correct');
        } else if (index === selectedIndex && !isCorrect) {
            btn.classList.add('incorrect');
        }
    });
    
    // Show feedback
    showFeedback(isCorrect, question.feedback);
}

// Check fill-in-the-blank answer
function checkFillBlank() {
    const question = quizQuestions[currentQuizQuestion];
    const userAnswer = document.getElementById('blank-answer').value.trim().toLowerCase();
    const correctAnswer = question.answer.toLowerCase();
    
    const isCorrect = userAnswer === correctAnswer;
    
    // Mark answer as attempted
    quizAnswered[currentQuizQuestion] = true;
    
    // Update score
    if (isCorrect) {
        quizScore++;
    }
    
    // Disable input and button
    document.getElementById('blank-answer').disabled = true;
    document.getElementById('submit-blank').disabled = true;
    
    // Show feedback
    showFeedback(isCorrect, question.feedback);
}

// Display feedback for answer
function showFeedback(isCorrect, feedback) {
    const feedbackArea = document.getElementById('feedback-area');
    feedbackArea.textContent = isCorrect ? feedback.correct : feedback.incorrect;
    feedbackArea.className = isCorrect ? 'correct' : 'incorrect';
    feedbackArea.style.display = 'block';
    
    // Show next question button
    document.getElementById('next-question').style.display = 'inline-block';
}

// Next question button handler
document.getElementById('next-question').addEventListener('click', () => {
    showQuestion(currentQuizQuestion + 1);
});

// Show quiz results
function showResults() {
    document.getElementById('question-area').style.display = 'none';
    document.getElementById('results-area').style.display = 'block';
    
    const percentage = Math.round((quizScore / quizQuestions.length) * 100);
    
    let message = '';
    if (percentage === 100) {
        message = 'Perfect score! You know your flower reproductive structures very well! 🌟';
    } else if (percentage >= 70) {
        message = 'Great job! You have a good understanding of flower reproductive structures! 🌸';
    } else {
        message = 'Good effort! Review the Explore and Guided modes to learn more. 🌱';
    }
    
    document.getElementById('score-display').innerHTML = `
        <p>You scored <strong>${quizScore}</strong> out of <strong>${quizQuestions.length}</strong></p>
        <p>${message}</p>
    `;
}

// Retry quiz button
document.getElementById('retry-quiz').addEventListener('click', startQuiz);

// ===== TOOLTIP FUNCTIONALITY =====
// Show tooltips on hover for mode buttons
const tooltip = document.getElementById('tooltip');

modeButtons.forEach(btn => {
    btn.addEventListener('mouseenter', (e) => {
        const title = e.target.getAttribute('title');
        if (title) {
            tooltip.textContent = title;
            tooltip.classList.add('visible');
            updateTooltipPosition(e);
        }
    });
    
    btn.addEventListener('mousemove', updateTooltipPosition);
    
    btn.addEventListener('mouseleave', () => {
        tooltip.classList.remove('visible');
    });
});

function updateTooltipPosition(e) {
    const x = e.clientX;
    const y = e.clientY;
    tooltip.style.left = x + 10 + 'px';
    tooltip.style.top = y + 10 + 'px';
}

// ===== INITIALIZATION =====
// Set up initial state when page loads
document.addEventListener('DOMContentLoaded', () => {
    switchMode('explore');
});