/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * AnalyticCurve
 * @class AnalyticCurve 
 * @constructor  
 */
EJSS_DRAWING2D.AnalyticCurve = {	
	
	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy
  	
		dest.setNumPoints(source.getNumPoints());
		dest.setMinimun(source.getMinimun());
		dest.setMaximun(source.getMaximun());
		dest.setVariable(source.getVariable());
		dest.setFunctionX(source.getFunctionX());
		dest.setFunctionY(source.getFunctionY());  	
  	},
  
	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class

		controller.registerProperty("NumPoints", element.setNumPoints, element.getNumPoints);
		controller.registerProperty("Minimum", element.setMinimun, element.getMinimun);
		controller.registerProperty("Maximum", element.setMaximun, element.getMaximun);
		controller.registerProperty("Variable", element.setVariable, element.getVariable);
		controller.registerProperty("FunctionX", element.setFunctionX, element.getFunctionX);
		controller.registerProperty("FunctionY", element.setFunctionY, element.getFunctionY);		
    
        controller.registerAction("OnError");

	}
			
};

/**
 * Creates a 2D AnalyticCurve
 * @method analyticCurve
 */
EJSS_DRAWING2D.analyticCurve = function(name) {
	var self = EJSS_DRAWING2D.element(name);
 
 	var mMinimun; 	
 	var mMaximun;
 	var mVariable = "t";
 	var mFunctionX = "t";
 	var mFunctionY = "0";
	var mParameters = {};
 	var mNumPoints = screen.width * 5;

	self.getClass = function() {
		return "ElementAnalyticCurve";
	}
	 
	/**
	 * Sets parameters for the evaluation of the function
	 * @method setParameters
	 * @param parameters Object { "p1" : value1, "p2" : value2, ...}
	 * @return void
	 */
	self.setParameters = function(parameters) {
	  for (var param in parameters) { 
          mParameters[param] = parameters[param];
	    }
	  self.setChanged(true);
    }

	/**
	 * Gets parameters for the evaluation of the function
	 * @method getParameters
	 * @return Object { "p1" : value1, "p2" : value2, ...}
	 */
	self.getParameters = function() {
	  return mParameters;
    }
	 
	/**
	 * Returns bounds for an element
	 * @method getBounds
	 * @return Object{left,rigth,top,bottom}
	 */
	self.getBounds = function(element) {
	    var x = self.getX(), y = self.getY();
	    var sx = self.getSizeX(), sy = self.getSizeY();
	  	var mx = sx/2, my = sy/2;  	
	  	var d = self.getRelativePositionOffset(sx,sy);
	   	// calculate points
	   	var parser = EJSS_DRAWING2D.functionsParser();
	   	var exprfx;
	   	var exprfy;
	   	var mustReturn = false;
	   	try {
	   	  exprfx = parser.parse(mFunctionX);
	   	}
	   	catch (errorfx) {
  	   	  console.log ("Analytic curve error parsing FunctionX: "+mFunctionX);
	   	  mustReturn = true;
	   	}
	   	if (!mustReturn) {
	   	  try {
	   	    exprfy = parser.parse(mFunctionY);
	   	  }
	   	  catch (errorfy) {
  	   	    console.log ("Analytic curve error parsing FunctionY: "+mFunctionY);
	   	    mustReturn = true;
	   	  }
	   	}
	   	if (mustReturn) {
	   	  self.getController().invokeAction("OnError");
	   	  return {
	   	    left: ((x+d[0])-mx)-sx,
			right: ((x+d[0])-mx)+sx,
			top: ((y+d[1])-my)+sy,
			bottom: ((y+d[1])-my)-sy
	   	  };
	   	}  	

		var min = ( (typeof mMinimun == "undefined" || mMinimun === null) ?  self.getPanel().getRealWorldXMin() : mMinimun);
//		console.log ("Minimumm = "+mMinimun+" : min = "+min);
		var max = ( (typeof mMaximun == "undefined" || mMaximun === null) ?  self.getPanel().getRealWorldXMax() : mMaximun);

	   	var step = (max-min)/mNumPoints;
		var points = [];	   	
	    var vblevalue = {};
	    for (var param in mParameters) { 
          vblevalue[param] = mParameters[param];
	    }
	    try {
	   	  for(var j=0, i=min; i<max; i+=step) {
	   		vblevalue[mVariable] = i;
	   		var fxvalue = exprfx.evaluate(vblevalue);
	   		var fyvalue = exprfy.evaluate(vblevalue);
	   		  if(!isNaN(fxvalue) && !isNaN(fyvalue)) {
		   		points[j] = [];		   		   		
		    	points[j][0] = fxvalue;	
				points[j++][1] = fyvalue;
			}
		  }
		}
		catch (errorEvaluate) {
	      self.getController().invokeAction("OnError");
		  return {
	   	    left: ((x+d[0])-mx)-sx,
			right: ((x+d[0])-mx)+sx,
			top: ((y+d[1])-my)+sy,
			bottom: ((y+d[1])-my)-sy
	   	  };
		} 
	
		// calculate max and min
		var maxfx = 0;
		var maxfy = 0;
		var minfx = 0;
		var minfy = 0;
		if(points.length > 0) {
			maxfx = points[0][0], minfx = points[0][0];
			maxfy = points[0][1], minfy = points[0][1];
		  	for(var k=0; k<points.length; k++) {
		    	if(points[k][0] > maxfx) maxfx = points[k][0];
		    	if(points[k][0] < minfx) minfx = points[k][0];
		    	if(points[k][1] > maxfy) maxfy = points[k][1];
		    	if(points[k][1] < minfy) minfy = points[k][1];
			}
		} 
		
		return {
			left: ((x+d[0])-mx)+minfx*sx,
			right: ((x+d[0])-mx)+maxfx*sx,
			top: ((y+d[1])-my)+maxfy*sy,
			bottom: ((y+d[1])-my)+minfy*sy
		}
	};
  	 		 
	/** 
	 * @method setNumPoint
	 * @param numpoint
	 */
	self.setNumPoints = function (numpoints) {
	  	if(mNumPoints != numpoints) {
	  		mNumPoints = numpoints;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getNumPoint
	 * @return
	 */
	self.getNumPoints = function() { 
		return mNumPoints; 
	}

	/** 
	 * @method setMinimun
	 * @param minimun
	 */
	self.setMinimun = function (minimun) {
	  	if(mMinimun != minimun) {
	  		mMinimun = minimun;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getMinimun
	 * @return
	 */
	self.getMinimun = function() { 
		return mMinimun; 
	}

	/** 
	 * @method setMaximun
	 * @param maximun
	 */
	self.setMaximun = function (maximun) {
	  	if(mMaximun != maximun) {
	  		mMaximun = maximun;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getMaximun
	 * @return
	 */
	self.getMaximun = function() { 
		return mMaximun; 
	}

	/** 
	 * @method setVariable
	 * @param variable
	 */
	self.setVariable = function (variable) {
	  	if(mVariable != variable) {
	  		mVariable = variable;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getVariable
	 * @return
	 */
	self.getVariable = function() { 
		return mVariable; 
	}

	/** 
	 * @method setFunctionX
	 * @param functionx
	 */
	self.setFunctionX = function (functionx) {
	  	if(mFunctionX != functionx) {
	  		mFunctionX = functionx;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getFunctionX
	 * @return
	 */
	self.getFunctionX = function() { 
		return mFunctionX; 
	}

	/** 
	 * @method setFunctionY
	 * @param functiony
	 */
	self.setFunctionY = function (functiony) {
	  	if(mFunctionY != functiony) {
	  		mFunctionY = functiony;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getFunctionY
	 * @return
	 */
	self.getFunctionY = function() { 
		return mFunctionY; 
	}


	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.AnalyticCurve.registerProperties(self, controller);
	};
  
	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	self.setSize([1,1]);
    self.setRelativePosition("SOUTH_WEST");
		
	return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * Arrow is a class to display vectors
 * @class EJSS_DRAWING2D.Arrow 
 * @parent EJSS_DRAWING2D.Element
 * @constructor  
 */
EJSS_DRAWING2D.Arrow = {
    NONE : 0,
    ANGLE : 1,
    CIRCLE : 2,
    RECTANGLE : 3,
    LINE : 4,
    CURVE : 5,
    TRIANGLE: 6,
    DIAMOND: 7,     
    WEDGE: 8,
	POINTED: 9,
	INVANGLE: 10,
    INVTRIANGLE: 11,
    
    SPLINE_NONE : 0,
    SPLINE_CUBIC : 1,
    SPLINE_QUADRATIC : 2,
    SPLINE_CUBIC_WITH_MIDDLE_POINT : 3,
    SPLINE_QUADRATIC_WITH_MIDDLE_POINT : 4,
    
	    	
    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy
  	
      dest.setRelativePosition(source.getRelativePosition());
      dest.setMarkEnd(source.getMarkEnd());
      dest.setMarkEndWidth(source.getMarkEndWidth());
      dest.setMarkEndHeight(source.getMarkEndHeight());
      dest.setMarkEndColor(source.getMarkEndColor());
      dest.setMarkEndStroke(source.getMarkEndStroke());
      dest.setMarkEndOrient(source.getMarkEndOrient());
      dest.setMarkStart(source.getMarkStart());
      dest.setMarkStartWidth(source.getMarkStartWidth());
      dest.setMarkStartHeight(source.getMarkStartHeight());
	  dest.setMarkStartColor(source.getMarkStartColor());
	  dest.setMarkStartStroke(source.getMarkStartStroke());
      dest.setMarkStartOrient(source.getMarkStartOrient());

      dest.setMarkMiddle(source.getMarkMiddle());
      dest.setMarkMiddleWidth(source.getMarkMiddleWidth());
      dest.setMarkMiddleHeight(source.getMarkMiddleHeight());
      dest.setMarkMiddleColor(source.getMarkMiddleColor());
      dest.setMarkMiddleColorSnd(source.getMarkMiddleColorSnd());
      dest.setMarkMiddleStroke(source.getMarkMiddleStroke());
      dest.setMarkMiddleOrient(source.getMarkMiddleOrient());
      dest.setMarkProportion(source.getMarkProportion());

      dest.setSplineType(source.getSplineType());

  	},


    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class
      
      controller.registerProperty("SplineType",element.setSplineType);

      controller.registerProperty("InteractionPosition", element.setInteractionPosition);
      controller.registerProperty("Offset", element.setRelativePosition, element.getRelativePosition);
      controller.registerProperty("MarkEnd",element.setMarkEnd);
      controller.registerProperty("MarkEndWidth",element.setMarkEndWidth);
      controller.registerProperty("MarkEndHeight",element.setMarkEndHeight);
      controller.registerProperty("MarkEndDiameter",element.setMarkEndDiameter);
      controller.registerProperty("MarkEndColor",element.setMarkEndColor);     
      controller.registerProperty("MarkEndStroke",element.setMarkEndStroke);
      /*** 
	  * Orientation of the marker. Values are "auto" or the angle to rotate the marker 
	  * @property MarkEndOrient 
	  * @type int|String
	  * @default "auto"
	  */ 
      controller.registerProperty("MarkEndOrient",element.setMarkEndOrient);
      
      controller.registerProperty("MarkStart",element.setMarkStart);
      controller.registerProperty("MarkStartWidth",element.setMarkStartWidth);
      controller.registerProperty("MarkStartHeight",element.setMarkStartHeight);
      controller.registerProperty("MarkStartDiameter",element.setMarkStartDiameter);
	  controller.registerProperty("MarkStartColor",element.setMarkStartColor);
	  controller.registerProperty("MarkStartStroke",element.setMarkStartStroke);
      /*** 
	  * Orientation of the marker. Values are "auto" or the angle to rotate the marker 
	  * @property MarkStartOrient 
	  * @type int|String
	  * @default "auto"
	  */ 
	  controller.registerProperty("MarkStartOrient",element.setMarkStartOrient);

      controller.registerProperty("MarkMiddle",element.setMarkMiddle);
      controller.registerProperty("MarkMiddleWidth",element.setMarkMiddleWidth);
      controller.registerProperty("MarkMiddleHeight",element.setMarkMiddleHeight);
      controller.registerProperty("MarkMiddleDiameter",element.setMarkMiddleDiameter);
      controller.registerProperty("MarkMiddleColor",element.setMarkMiddleColor);
      controller.registerProperty("MarkMiddleColorSnd",element.setMarkMiddleColorSnd);
      controller.registerProperty("MarkMiddleStroke",element.setMarkMiddleStroke);
      /*** 
	  * Orientation of the marker. Values are "auto" or the angle to rotate the marker 
	  * @property MarkMiddleOrient 
	  * @type int|String
	  * @default "auto"
	  */ 
      controller.registerProperty("MarkMiddleOrient",element.setMarkMiddleOrient);
      
      controller.registerProperty("MarkProportion",element.setMarkProportion);
    },  
};

/**
 * Creates a 2D Arrow
 * @method arrow
 */
EJSS_DRAWING2D.arrow = function (name) {
  var self = EJSS_DRAWING2D.element(name);

  // Implementation variables
  var mMarkProportion = 0.5;
  var mMarkStart = EJSS_DRAWING2D.Arrow.NONE;
  var mMarkStartColor = "none";
  var mMarkStartWidth = 12;
  var mMarkStartHeight = 10;
  var mMarkStartStroke = -1;
  var mMarkStartOrient = "auto";
  var mMarkEnd = EJSS_DRAWING2D.Arrow.ANGLE;
  var mMarkEndColor = "none";
  var mMarkEndWidth = 12;
  var mMarkEndHeight = 12;
  var mMarkEndStroke = -1;
  var mMarkEndOrient = "auto";
  var mMarkMiddle = EJSS_DRAWING2D.Arrow.NONE;
  var mMarkMiddleColor = "none";  
  var mMarkMiddleColorSnd = "none";  
  var mMarkMiddleWidth = 12;
  var mMarkMiddleHeight = 10;
  var mMarkMiddleStroke = -1;
  var mMarkMiddleOrient = "auto";
  var mSplineType = EJSS_DRAWING2D.Arrow.SPLINE_NONE;

  self.getClass = function() {
  	return "ElementArrow";
  }

  self.setInteractionPosition = function(position) {
    if (typeof position == "string") position = EJSS_DRAWING2D.Element[position.toUpperCase()];
    var inter = self.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION);
    if (inter.getPositionOffset() != position) {
      inter.setPositionOffset(position);
      self.setChanged(true);
    }	
  }

  self.getInteractionPosition = function() {
    var inter = self.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION);
    return inter.getPositionOffset();
  }
  
  self.setMarkProportion = function(prop) {
    if (mMarkProportion != prop) {
      mMarkProportion = prop;
      self.setChanged(true);
    }
  };

  self.getMarkProportion = function() {
  	return mMarkProportion;
  }	

  self.setMarkStart = function(mark) {
    if (typeof mark === 'string')
      mMarkStart = EJSS_DRAWING2D.Arrow[mark.toUpperCase()];
    else 
      mMarkStart = mark;
  };

  self.getMarkStart = function() {
  	return mMarkStart;
  }	

  self.setMarkStartColor = function(color) { 
    if (typeof color !== "string") color = EJSS_TOOLS.DisplayColors.getLineColor(color);
    if (color!=mMarkStartColor) {
      mMarkStartColor = color; 
      self.setChanged(true);
    }
    return self;
  };
    
  self.getMarkStartColor = function() { 
    return mMarkStartColor; 
  };

  self.setMarkStartStroke = function(width) {
    if (mMarkStartStroke != width) {
      mMarkStartStroke = width;
      self.setChanged(true);
    }
  };

  self.getMarkStartStroke = function() {
  	return mMarkStartStroke;
  }	

  self.setMarkStartWidth = function(width) {
    if (mMarkStartWidth != width) {
      mMarkStartWidth = width;
      self.setChanged(true);
    }
  };

  self.getMarkStartWidth = function() {
  	return mMarkStartWidth;
  }	

  self.setMarkStartHeight = function(height) {
    if (mMarkStartHeight != height) {
      mMarkStartHeight = height;
      self.setChanged(true);
    }
  };

  self.getMarkStartHeight = function() {
  	return mMarkStartHeight;
  }	

  self.setMarkStartDiameter = function(diameter) {
    if ((mMarkStartHeight != diameter) || (mMarkStartWidth != diameter)) {
      mMarkStartHeight = diameter;
      mMarkStartWidth = diameter;
      self.setChanged(true);
    }
  };

  self.getMarkStartOrient = function() {
  	return mMarkStartOrient;
  }	

  self.setMarkStartOrient = function(orient) {
    if (mMarkStartOrient != orient) {
      mMarkStartOrient = orient;
      self.setChanged(true);
    }
  };

  self.setMarkEnd = function(mark) {
    if (typeof mark === 'string')
      mMarkEnd = EJSS_DRAWING2D.Arrow[mark.toUpperCase()];
    else 
      mMarkEnd = mark;
  };

  self.getMarkEnd = function() {
  	return mMarkEnd;
  }	

  self.setMarkEndColor = function(color) { 
    if (typeof color !== "string") color = EJSS_TOOLS.DisplayColors.getLineColor(color);
    if (color!=mMarkEndColor) {
      mMarkEndColor = color; 
      self.setChanged(true);
    }
    return self;
  };
    
  self.getMarkEndColor = function() { 
    return mMarkEndColor; 
  };
  
  self.setMarkEndStroke = function(width) {
    if (mMarkEndStroke != width) {
      mMarkEndStroke = width;
      self.setChanged(true);
    }
  };

  self.getMarkEndStroke = function() {
  	return mMarkEndStroke;
  }	

  self.setMarkEndWidth = function(width) {
    if (mMarkEndWidth != width) {
      mMarkEndWidth = width;
      self.setChanged(true);
    }
  };

  self.getMarkEndWidth = function() {
  	return mMarkEndWidth;
  }	

  self.setMarkEndHeight = function(height) {
    if (mMarkEndHeight != height) {
      mMarkEndHeight = height;
      self.setChanged(true);
    }
  };

  self.getMarkEndHeight = function() {
  	return mMarkEndHeight;
  }	

  self.setMarkEndDiameter = function(diameter) {
    if ((mMarkEndHeight != diameter) || (mMarkEndWidth != diameter)) {
      mMarkEndHeight = diameter;
      mMarkEndWidth = diameter;
      self.setChanged(true);
    }
  };

  self.getMarkEndOrient = function() {
  	return mMarkEndOrient;
  }	

  self.setMarkEndOrient = function(orient) {
    if (mMarkEndOrient != orient) {
      mMarkEndOrient = orient;
      self.setChanged(true);
    }
  };

  self.setMarkMiddle = function(mark) {
    if (typeof mark === 'string')
      mMarkMiddle = EJSS_DRAWING2D.Arrow[mark.toUpperCase()];
    else 
      mMarkMiddle = mark;
  };

  self.getMarkMiddle = function() {
  	return mMarkMiddle;
  }	

  self.setMarkMiddleColor = function(color) { 
    if (typeof color !== "string") color = EJSS_TOOLS.DisplayColors.getLineColor(color);
    if (color!=mMarkMiddleColor) {
      mMarkMiddleColor = color; 
      self.setChanged(true);
    }
    return self;
  };
    
  self.getMarkMiddleColor = function() { 
    return mMarkMiddleColor; 
  };

  self.setMarkMiddleStroke = function(width) {
    if (mMarkMiddleStroke != width) {
      mMarkMiddleStroke = width;
      self.setChanged(true);
    }
  };

  self.getMarkMiddleStroke = function() {
  	return mMarkMiddleStroke;
  }	

  self.setMarkMiddleColorSnd = function(color) { 
    if (typeof color !== "string") color = EJSS_TOOLS.DisplayColors.getLineColor(color);
    if (color!=mMarkMiddleColorSnd) {
      mMarkMiddleColorSnd = color; 
      self.setChanged(true);
    }
    return self;
  };
    
  self.getMarkMiddleColorSnd = function() { 
    return mMarkMiddleColorSnd; 
  };

  self.setMarkMiddleWidth = function(width) {
    if (mMarkMiddleWidth != width) {
      mMarkMiddleWidth = width;
      self.setChanged(true);
    }
  };

  self.getMarkMiddleWidth = function() {
  	return mMarkMiddleWidth;
  }	

  self.setMarkMiddleHeight = function(height) {
    if (mMarkMiddleHeight != height) {
      mMarkMiddleHeight = height;
      self.setChanged(true);
    }
  };

  self.getMarkMiddleHeight = function() {
  	return mMarkMiddleHeight;
  }	

  self.setMarkMiddleDiameter = function(diameter) {
    if ((mMarkMiddleHeight != diameter) || (mMarkMiddleWidth != diameter)) {
      mMarkMiddleHeight = diameter;
      mMarkMiddleWidth = diameter;
      self.setChanged(true);
    }
  };

  self.getMarkMiddleOrient = function() {
  	return mMarkMiddleOrient;
  }	

  self.setMarkMiddleOrient = function(orient) {
    if (mMarkMiddleOrient != orient) {
      mMarkMiddleOrient = orient;
      self.setChanged(true);
    }
  };

  // ----------------------------------------------------
  // SVG Splines
  // ----------------------------------------------------

  
  /**
   * 
   */
  self.setSplineType = function(spline) {
    if (mSplineType != spline) {
      mSplineType = spline;
      self.setChanged(true);
    }
  };

  self.getSplineType = function() {
    return mSplineType;
  } 

  // ----------------------------------------------------
  // Properties and copy
  // ----------------------------------------------------

  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.Arrow.registerProperties(self,controller);
  };
  
  self.copyTo = function(element) {
    EJSS_DRAWING2D.Arrow.copyTo(self,element);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([0.1,0.1]);
  self.setRelativePosition("SOUTH_WEST");
  // self.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setPositionOffset("SOUTH_WEST");

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//ArrowSet
//---------------------------------

/***
 * ArrowSet is a set of Arrows
 * @class EJSS_DRAWING2D.ArrowSet 
 * @constructor  
 */
EJSS_DRAWING2D.ArrowSet = {

    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING2D.ElementSet;
      
      ElementSet.registerProperties(set,controller);
      controller.registerProperty("Offset", 
          function(v) { set.setToEach(function(element,value) { element.setRelativePosition(value); }, v); }
      );
      controller.registerProperty("InteractionPosition", 
          function(v) { set.setToEach(function(element,value) { element.setInteractionPosition(value); }, v); }
      );      

      controller.registerProperty("MarkEnd", 
          function(v) { set.setToEach(function(element,value) { element.setMarkEnd(value); }, v); }
      );
      controller.registerProperty("MarkEndWidth", 
          function(v) { set.setToEach(function(element,value) { element.setMarkEndWidth(value); }, v); }
      );
      controller.registerProperty("MarkEndHeight", 
          function(v) { set.setToEach(function(element,value) { element.setMarkEndHeight(value); }, v); }
      );
      controller.registerProperty("MarkEndDiameter", 
          function(v) { set.setToEach(function(element,value) { element.setMarkEndDiameter(value); }, v); }
      );
      controller.registerProperty("MarkEndColor", 
          function(v) { set.setToEach(function(element,value) { element.setMarkEndColor(value); }, v); }
      );
      controller.registerProperty("MarkEndStroke", 
          function(v) { set.setToEach(function(element,value) { element.setMarkEndStroke(value); }, v); }
      );
      controller.registerProperty("MarkEndRotate", 
          function(v) { set.setToEach(function(element,value) { element.setMarkEndRotate(value); }, v); }
      );
      /*** 
	  * Orientation of the marker. Values are "auto" or the angle to rotate the marker 
	  * @property MarkEndOrient 
	  * @type int|String
	  * @default "auto"
	  */ 
      controller.registerProperty("MarkEndOrient", 
          function(v) { set.setToEach(function(element,value) { element.setMarkEndOrient(value); }, v); }
      );
      

      controller.registerProperty("MarkStart", 
          function(v) { set.setToEach(function(element,value) { element.setMarkStart(value); }, v); }
      );
      controller.registerProperty("MarkStartWidth", 
          function(v) { set.setToEach(function(element,value) { element.setMarkStartWidth(value); }, v); }
      );
      controller.registerProperty("MarkStartHeight", 
          function(v) { set.setToEach(function(element,value) { element.setMarkStartHeight(value); }, v); }
      );
      controller.registerProperty("MarkStartDiameter", 
          function(v) { set.setToEach(function(element,value) { element.setMarkStartDiameter(value); }, v); }
      );
      controller.registerProperty("MarkStartColor", 
          function(v) { set.setToEach(function(element,value) { element.setMarkStartColor(value); }, v); }
      );
      controller.registerProperty("MarkStartStroke", 
          function(v) { set.setToEach(function(element,value) { element.setMarkStartStroke(value); }, v); }
      );
      controller.registerProperty("MarkStartRotate", 
          function(v) { set.setToEach(function(element,value) { element.setMarkStartRotate(value); }, v); }
      );
      /*** 
	  * Orientation of the marker. Values are "auto" or the angle to rotate the marker 
	  * @property MarkStartOrient 
	  * @type int|String
	  * @default "auto"
	  */ 
      controller.registerProperty("MarkStartOrient", 
          function(v) { set.setToEach(function(element,value) { element.setMarkStartOrient(value); }, v); }
      );


      controller.registerProperty("MarkMiddle", 
          function(v) { set.setToEach(function(element,value) { element.setMarkMiddle(value); }, v); }
      );
      controller.registerProperty("MarkMiddleWidth", 
          function(v) { set.setToEach(function(element,value) { element.setMarkMiddleWidth(value); }, v); }
      );
      controller.registerProperty("MarkMiddleHeight", 
          function(v) { set.setToEach(function(element,value) { element.setMarkMiddleHeight(value); }, v); }
      );
      controller.registerProperty("MarkMiddleDiameter", 
          function(v) { set.setToEach(function(element,value) { element.setMarkMiddleDiameter(value); }, v); }
      );
      controller.registerProperty("MarkMiddleColor", 
          function(v) { set.setToEach(function(element,value) { element.setMarkMiddleColor(value); }, v); }
      );
      controller.registerProperty("MarkMiddleColorSnd", 
          function(v) { set.setToEach(function(element,value) { element.setMarkMiddleColorSnd(value); }, v); }
      );
      controller.registerProperty("MarkMiddleStroke", 
          function(v) { set.setToEach(function(element,value) { element.setMarkMiddleStroke(value); }, v); }
      );
      controller.registerProperty("MarkMiddleRotate", 
          function(v) { set.setToEach(function(element,value) { element.setMarkMiddleRotate(value); }, v); }
      );
      controller.registerProperty("MarkProportion", 
          function(v) { set.setToEach(function(element,value) { element.setMarkProportion(value); }, v); }
      );
      /*** 
	  * Orientation of the marker. Values are "auto" or the angle to rotate the marker 
	  * @property MarkMiddleOrient 
	  * @type int|String
	  * @default "auto"
	  */ 
      controller.registerProperty("MarkMiddleOrient", 
          function(v) { set.setToEach(function(element,value) { element.setMarkMiddleOrient(value); }, v); }
      );


    }

};


/**
 * Creates a set of Segments
 * @method arrowSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING2D.arrowSet = function (mName) {
  var self = EJSS_DRAWING2D.elementSet(EJSS_DRAWING2D.arrow, mName);

  // Static references
  var ArrowSet = EJSS_DRAWING2D.ArrowSet;		// reference for ArrowSet
  
  self.registerProperties = function(controller) {
    ArrowSet.registerProperties(self,controller);
  };


  return self;
};/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * An Axis is the 2D drawable used by PlottingPanels as decoration  
 * @class EJSS_DRAWING2D.Axis 
 * @parent EJSS_DRAWING2D.Segment
 * @constructor  
 */
EJSS_DRAWING2D.Axis = {
	// axis orientation 
	AXIS_VERTICAL   : 0,
	AXIS_HORIZONTAL : 1,
	// ticks position (reference for text)
	TICKS_UP   : 2,
	TICKS_DOWN : 3,
	// ticks increment
	SCALE_NUM	: 0,	// decimal
	SCALE_LOG	: 1,	// logarithmic

	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class

		// ticks
		controller.registerProperty("Step", element.setStep, element.getStep);
		controller.registerProperty("TickStep", element.setTickStep, element.getTickStep);
		controller.registerProperty("Ticks", element.setTicks, element.getTicks);
		controller.registerProperty("AutoTicks", element.setAutoTicks, element.getAutoTicks);				
		controller.registerProperty("FixedTick", element.setFixedTick, element.getFixedTick);				

		// axis properties
		controller.registerProperty("Orient", element.setOrient, element.getOrient);
		controller.registerProperty("Scale", element.setScale, element.getScale);
		
		// ticks properties
		controller.registerProperty("TicksMode", element.setTicksMode, element.getTicksMode);
		controller.registerProperty("TicksSize", element.setTicksSize, element.getTicksSize);
	
		// auto ticks		
		controller.registerProperty("AutoStepMin", element.setAutoStepMin, element.getAutoStepMin);				
		controller.registerProperty("AutoTicksRange", element.setAutoTicksRange, element.getAutoTicksRange);				
		
		// ticks text properties
		controller.registerProperty("TextPosition", element.setTextPosition, element.getTextPosition);
		controller.registerProperty("ScalePrecision", element.setScalePrecision, element.getScalePrecision);		
		controller.registerProperty("Font", element.getFont().setFont);
		controller.registerProperty("FontFamily", element.getFont().setFontFamily);
		controller.registerProperty("FontSize", element.getFont().setFontSize);
		controller.registerProperty("LetterSpacing", element.getFont().setLetterSpacing);
		controller.registerProperty("FontOutlineColor", element.getFont().setOutlineColor);
		controller.registerProperty("FontWeight", element.getFont().setFontWeight);
		controller.registerProperty("FontFillColor", element.getFont().setFillColor);		
		
		// show
		controller.registerProperty("Show", element.setShow, element.getShow);
	}
			
};

/**
 * Creates a 2D Axis
 * @method axis
 */
EJSS_DRAWING2D.axis = function(mName) {
	var self = EJSS_DRAWING2D.segment(mName);
 
 	// drawing priority: mAutoTicks - mTicks - mStep
 	var mAutoTicks = true;								// auto-ticks
 	var mTicks = 0;										// number of ticks
 	var mStep = 20; 									// step between ticks in pixels
 	var mTickStep = 0; 									// step between ticks in real coordinates
	var mFixedTick = Number.NaN;								// ticks fixed in axis	

	// axis properties	
	var mOrient = EJSS_DRAWING2D.Axis.AXIS_VERTICAL;	// axis orientation
	var mScale = [-1,1];									// axis scale
	var mInvertedScaleY = false;						// whether inverted scale in Y

	// ticks properties
 	var mTicksMode = EJSS_DRAWING2D.Axis.SCALE_NUM;		// axis scale
 	var mTextPosition = EJSS_DRAWING2D.Axis.TICKS_UP;	// tick orientation
	var mTicksSize = 10;								// tick pixel size

	// ticks text properties
	var mScalePrecision = 1;							// number of decimals
	var mFont = EJSS_DRAWING2D.font(mName);				// font for text
 
	// auto ticks 	
	var mAutoStepMin = 40;					// step minimun in pixels
	var mAutoTicksRange = [5,10,20];		// ticks range: 
											//   is the step minimun possible in mAutoTicksRange[length-1]?
											//   and in mAutoTicksRange[length-2]? ... 
											//   then mAutoTicksRange[length-2] is the number of ticks
	// show
	var mShow = true;
	   
	self.getClass = function() {
		return "ElementAxis";
	}

	self.setShow = function(show) {
		if(mShow != show) {
			mShow = show;
			self.setChanged(true);
		}
	}

	self.getShow = function() {
		return mShow;
	}

	self.getFont = function() {
		return mFont;
	}
		
	/** 
	 * @method setAutoTicksRangeX
	 * @param range
	 */
	self.setAutoTicksRange = function (range) {
	  	if(mAutoTicksRange != range) {
	  		mAutoTicksRange = range;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getAutoTicksRange
	 * @return
	 */
	self.getAutoTicksRange = function() { 
		return mAutoTicksRange; 
	}

	/** 
	 * @method setAutoStepMin
	 * @param min
	 */
	self.setAutoStepMin = function (min) {
	  	if(mAutoStepMin != min) {
	  		mAutoStepMin = min;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getAutoStepMin
	 * @return
	 */
	self.getAutoStepMin = function() { 
		return mAutoStepMin; 
	}

	/** 
	 * @method setFixedTick
	 * @param fixed
	 */
	self.setFixedTick = function (fixed) {
	  	if(mFixedTick != fixed) {
	  		mFixedTick = fixed;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getFixedTick
	 * @return
	 */
	self.getFixedTick = function() { 
		return mFixedTick; 
	}
			
	/***
	 * @method setAutoTicks
	 * @param auto
	 */
	self.setAutoTicks = function (auto) {
	  	if(mAutoTicks != auto) {
	  		mAutoTicks = auto;
	  		self.setChanged(true);
	  	}
	}
	  
	/***
	 * @method getAutoTicks
	 * @return
	 */
	self.getAutoTicks = function() { 
		return mAutoTicks; 
	}
			 
	/** 
	 * @method setStep
	 * @param stet
	 */
	self.setStep = function (step) {
		if(mStep != step) {
	  		mStep = step;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getStep
	 * @return
	 */
	self.getStep = function() { 
		return mStep; 
	}

	/** 
	 * @method setTickStep
	 * @param stet
	 */
	self.setTickStep = function (TickStep) {
		if(mTickStep != TickStep) {
	  		mTickStep = TickStep;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getTickStep
	 * @return
	 */
	self.getTickStep = function() { 
		return mTickStep; 
	}
		 
	/** 
	 * @method setTicks
	 * @param ticks
	 */
	self.setTicks = function (ticks) {
	  	if(mTicks != ticks) {
	  		mTicks = ticks;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getTicks
	 * @return
	 */
	self.getTicks = function() { 
		return mTicks; 
	}

	/** 
	 * @method setTicksMode
	 * @param ticksMode
	 */
	self.setTicksMode = function (ticksMode) {
    	if (typeof ticksMode == "string") ticksMode = EJSS_DRAWING2D.Grid[ticksMode.toUpperCase()];
    	if(mTicksMode != ticksMode) {
    		mTicksMode = ticksMode;
    		self.setChanged(true);
    	}	
	}
	  
	/**
	 * @method getTicksMode
	 * @return
	 */
	self.getTicksMode = function() { 
		return mTicksMode; 
	}

	/** 
	 * @method setTextPosition
	 * @param textPosition
	 */
	self.setTextPosition = function (textPosition) {
    	if (typeof textPosition == "string") textPosition = EJSS_DRAWING2D.Axis[textPosition.toUpperCase()];
    	if(mTextPosition != textPosition) {
    		mTextPosition = textPosition;
    		self.setChanged(true);
    	}	
	}
	  
	/**
	 * @method getTextPosition
	 * @return
	 */
	self.getTextPosition = function() { 
		return mTextPosition; 
	}

	/** 
	 * @method setOrient
	 * @param orient
	 */
	self.setOrient = function (orient) {
    	if (typeof orient == "string") orient = EJSS_DRAWING2D.Axis[orient.toUpperCase()];
    	if (mOrient != orient) {
    		mOrient = orient;
    		self.setChanged(true);	
    	}	
	}
	  
	/**
	 * @method getOrient
	 * @return
	 */
	self.getOrient = function() { 
		return mOrient; 
	}

	/** 
	 * @method setScale
	 * @param scale
	 */
	self.setScale = function (scale) {
		if(mScale != scale) {
	  		mScale = scale;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getScale
	 * @return
	 */
	self.getScale = function() { 
		return mScale; 
	}

	/** 
	 * @method setScalePrecision
	 * @param scalePrecision
	 */
	self.setScalePrecision = function (scalePrecision) {
		if(mScalePrecision != scalePrecision) {
	  		mScalePrecision = scalePrecision;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getScalePrecision
	 * @return
	 */
	self.getScalePrecision = function() { 
		return mScalePrecision; 
	}

	/** 
	 * @method setTicksSize
	 * @param ticksSize
	 */
	self.setTicksSize = function (ticksSize) {
		if(mTicksSize != ticksSize) {
	  		mTicksSize = ticksSize;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getTicksSize
	 * @return
	 */
	self.getTicksSize = function() { 
		return mTicksSize; 
	}
	
	/**
	 * Set inverted scale in Y
	 * @method setInvertedScaleY
	 */
	self.setInvertedScaleY = function(invertedscale) {
		if(mInvertedScaleY != invertedscale) {
			mInvertedScaleY = invertedscale;
	  		self.setChanged(true);
		}
	};

	/**
	 * Get inverted scale in Y
	 * @method getInvertedScaleY
	 * @return boolean
	 */
	self.getInvertedScaleY = function() {
		return mInvertedScaleY;
	};
	
	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.Axis.registerProperties(self, controller);
	};
  
	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	mFont.setChangeListener(function (change) { self.setChanged(true); });

	return self;
};

/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * ByteRaster
 * @class ByteRaster 
 * @constructor  
 */
EJSS_DRAWING2D.ByteRaster = {	
	
	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class

		controller.registerProperty("Data", element.setData, element.getData);
		controller.registerProperty("NumColors", element.getColorMapper().setNumberOfColors);
		controller.registerProperty("Colors", element.getColorMapper().setColorPalette);
		controller.registerProperty("Palette", element.getColorMapper().setPaletteType);	
        controller.registerProperty("AutoUpdate", element.setAutoupdate);  
        controller.registerProperty("RGBData", element.setRGBData);  
	}
			
};

/**
 * Creates a 2D ByteRaster
 * @method byteRaster
 */
EJSS_DRAWING2D.byteRaster = function(name) {
	var self = EJSS_DRAWING2D.element(name);
 
 	var mColorMapper = EJSS_DRAWING2D.colorMapper(20, EJSS_DRAWING2D.ColorMapper.REDBLUE_SHADE); 	
	var mData = [];
	var mDataChanged = false;
	var mAutoUpdate=true;
	var mRGBData = false;
 
	self.getClass = function() {
		return "ElementByteRaster";
	}

	/** 
	 * @method setData
	 * @param data
	 */
	self.setData = function (data) {
	  if (mAutoUpdate || mData != data) {
	  		mData = data;
	  		mDataChanged = true;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getData
	 * @return
	 */
	self.getData = function() { 
		return mData; 
	}
	 
  self.setRGBData= function (rgbData) {
    mRGBData = rgbData;
    if (mAutoUpdate) mDataChanged = true;
  }

  self.isRGBData= function () {
    return mRGBData;
  }
	 
  self.setAutoupdate= function (auto) {
    mAutoUpdate = auto;
    if (mAutoUpdate) mDataChanged = true;
  }
	 
	/** 
	 * @method setColorMapper
	 * @param colormapper
	 */
	self.setColorMapper = function (colormapper) {
	  	if(mColorMapper != colormapper) {
	  		mColorMapper = colormapper;
	  		mDataChanged = true;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getColorMapper
	 * @return
	 */
	self.getColorMapper = function() { 
		return mColorMapper; 
	}
	 
	self.setIndexedColor = function (value) {
	  //console.log ("Value = "+value);
	  var color = value.color; 
	  var colorStr = (color[3]===undefined) ? "rgb("+color[0]+","+color[1]+","+color[2]+")" : "rgb("+color[0]+","+color[1]+","+color[2]+","+color[3]+")";
	  console.log(self.getName()+": setIndexedColor("+value.index+","+colorStr+")");
	  var colors = mColorMapper.getColors();
	  colors[value.index] = colorStr;  
	};
	
	self.getDataChanged = function() {
		return mDataChanged;
	}

	self.setDataChanged = function(dataChanged) {
		mDataChanged = dataChanged;
	}

  self.invalidate = function() {
    mDataChanged = true;
  }

	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.ByteRaster.registerProperties(self, controller);
	};
  
	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	mColorMapper.setChangeListener(function (change) { self.setChanged(true); });
	return self;
};

 /*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Canvas
 * @class Canvas 
 * @constructor  
 */
EJSS_DRAWING2D.Canvas = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy
  	
  		dest.setDimensions(source.getDimensions());
  	},


    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class

		controller.registerProperty("Dimensions", element.setDimensions, element.getDimensions);

    },
};

/**
 * Creates a 2D Segment
 * @method Canvas
 */
EJSS_DRAWING2D.canvas = function (name) {
  var self = EJSS_DRAWING2D.element(name);

  var mDrawables = [];
  var mxMin = -10;
  var mxMax = 10;
  var myMin = -10;
  var myMax = 10;

  self.getClass = function() {
  	return "ElementCanvas";
  };

  // xMin, xMax, yMin, yMax
  self.setDimensions = function(dimensions) {
  	xMin = dimensions[0];
  	xMax = dimensions[1];
  	yMin = dimensions[2];
  	yMax = dimensions[3];

  	self.setChanged(true);
  }
  
  self.getDimensions = function() {
  	return [xMin,xMax,yMin,yMax];
  }

  self.addDrawable = function(drawable) {
    mDrawables.push(drawable);
  	self.setChanged(true);
  }

  self.addImageField = function(data, xMin, xMax, xPoints, yMin, yMax, yPoints, autoscale, zMin, zMax) {
  	var imgField = function() {
  	  var shape = {};
  	  shape.imageField = true;
	  shape.data = data || undefined;
	  shape.xMin = xMin === undefined ? -5: xMin;
	  shape.xMax = xMax === undefined ? 5: xMax;
	  shape.nx = xPoints || 128;
	  shape.yMin = yMin === undefined ? -5: yMin;
	  shape.yMax = yMax === undefined ? 5: yMax;
	  shape.ny = yPoints || 128;
	  shape.autoscale = autoscale || true;
	  shape.lower = zMin === undefined ? -1: zMin;
	  shape.upper = zMax === undefined ? 1: zMax;
	  shape.center = (shape.lower+shape.upper)/2;  // usually zero
	  shape.maskRadius=0;  // no circular mask if radius=0

	  function arrayCoordinates(min, max, points) {
	    var coordinates =[];
	    var step = (max - min) /(points -1)
	    var val = min;
	    for (var i = 0; i < points; i++) {
	        //pushes in each of the x values
	        coordinates.push(val);
	        val += step;
	    }
	    return coordinates;
	  }
	 
	  shape.xPos = arrayCoordinates(shape.xMin, shape.xMax, shape.nx);
	  shape.yPos = arrayCoordinates(shape.yMin, shape.yMax, shape.ny); 	  
	 
	  shape.updateData = function (newData) {
	    //updates the contour lines if data are changed
	    shape.data = newData || shape.data;
	    shape.xPos = arrayCoordinates(shape.xMin, shape.xMax, shape.nx);
	    shape.yPos = arrayCoordinates(shape.yMin, shape.yMax, shape.ny);
	    
	    self.setChanged(true);
	  }
	
	  shape.setThreshold = function (lower, upper) {
	    shape.lower = lower === undefined ? shape.lower: lower;
	    shape.upper = upper === undefined ? shape.upper: upper;
	    
	    self.setChanged(true);
	  }
	  
	  // Draw a curcular image
	  shape.setCircularMask = function (radius) {
	    shape.maskRadius = radius;
	    
	    self.setChanged(true);
	  }
 
 	  return shape;
  	}();
  	
  	self.addDrawable(imgField);
  	return imgField;
  }
 
  self.getDrawables = function() {
  	return mDrawables;
  }

  this.clearObjects = function () {
    mDrawables =[];
  	self.setChanged(true);
  }
  
  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.Canvas.registerProperties(self,controller);
  };
  
  self.copyTo = function(element) {
    EJSS_DRAWING2D.Canvas.copyTo(self,element);
  };
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  return self;
};



/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * CellLattice
 * @class CellLattice 
 * @constructor  
 */
EJSS_DRAWING2D.CellLattice = {	
	
	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class

		controller.registerProperty("Data", element.setData, element.getData);
		controller.registerProperty("ShowGrid", element.setShowGrid, element.getShowGrid);
		controller.registerProperty("NumColors", element.getColorMapper().setNumberOfColors);
		controller.registerProperty("Colors", element.getColorMapper().setColorPalette);
		controller.registerProperty("Palette", element.getColorMapper().setPaletteType);
        controller.registerProperty("AutoUpdate", element.setAutoupdate);  
	}
			
};

/**
 * Creates a 2D CellLattice
 * @method cellLattice
 */
EJSS_DRAWING2D.cellLattice = function(name) {
	var self = EJSS_DRAWING2D.element(name);
 
 	var mColorMapper = EJSS_DRAWING2D.colorMapper(20, EJSS_DRAWING2D.ColorMapper.REDBLUE_SHADE); 	
 	var mShowGrid = true;
	var mData = [];
	var mAutoUpdate=true;
 
	self.getClass = function() {
		return "ElementCellLattice";
	}


	/** 
	 * @method setData
	 * @param data
	 */
	self.setData = function (data) {
	  if (mAutoUpdate || mData != data) {
	    mData = data;
		mDataChanged = true;
		self.setChanged(true);
	  }
	}
	  
	/**
	 * @method getData
	 * @return
	 */
	self.getData = function() { 
		return mData; 
	}
	 
	self.setAutoupdate= function (auto) {
	  mAutoUpdate = auto;
      if (mAutoUpdate) mDataChanged = true;
    }
	
	/** 
	 * @method setColorMapper
	 * @param colormapper
	 */
	self.setColorMapper = function (colormapper) {
	  	if(mColorMapper != colormapper) {
	  		mColorMapper = colormapper;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getColorMapper
	 * @return
	 */
	self.getColorMapper = function() { 
		return mColorMapper; 
	}
	 
	/** 
	 * @method setShowGrid
	 * @param showgrid
	 */
	self.setShowGrid = function (showgrid) {
	  	if(mShowGrid != showgrid) {
	  		mShowGrid = showgrid;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getShowGrid
	 * @return
	 */
	self.getShowGrid = function() { 
		return mShowGrid; 
	}


	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.CellLattice.registerProperties(self, controller);
	};
  
	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	return self;
};

 
/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Cursor
 * @class Cursor 
 * @constructor  
 */
EJSS_DRAWING2D.Cursor = {
    HORIZONTAL : 0,
    VERTICAL : 1,
    CROSSHAIR : 2,
    
	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class
		
		controller.registerProperty("CursorType", element.setCursorType, element.getCursorType);
	},
};

/**
 * Creates a 2D Cursor
 * @method cursor
 */
EJSS_DRAWING2D.cursor = function(name) {
	var self = EJSS_DRAWING2D.element(name);	
	var Cursor = EJSS_DRAWING2D.Cursor;
	
	var mCursorType;

	self.getClass = function() {
		return "ElementCursor";
	}
	
	self.setCursorType = function(cursorType) {
	  var InteractionTarget = EJSS_DRAWING2D.InteractionTarget;
	  var target = self.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION);
      if (typeof cursorType === 'string') 
        mCursorType = EJSS_DRAWING2D.Cursor[cursorType.toUpperCase()];
      else mCursorType = cursorType;
      switch (mCursorType) {
        default : target.setSensitivityType(InteractionTarget.SENSITIVITY_ANY);break;
        case Cursor.HORIZONTAL : target.setSensitivityType(InteractionTarget.SENSITIVITY_HORIZONTAL);break;
        case Cursor.VERTICAL : target.setSensitivityType(InteractionTarget.SENSITIVITY_VERTICAL);break;
     }   
	}

	self.getCursorType = function() {
	  return mCursorType;
	}

	self.isChanged = function() {
		return true;
	}

	self.registerProperties = function(controller) {
		Cursor.registerProperties(self, controller);
	};

	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	//self.setSize([0.1,0.1]);
	self.setMeasured(false);
	mCursorType = Cursor.CROSSHAIR;
	self.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setSensitivityType(EJSS_DRAWING2D.InteractionTarget.SENSITIVITY_ANY);

	return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * A Custom element is a 2D drawing element that executes a user function for drawing 
 * @class EJSS_DRAWING2D.Custom
 * @parent EJSS_DRAWING2D.Element
 * @constructor  
 */
EJSS_DRAWING2D.Custom = {

    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class
    },  


};

/**
 * Creates a Custom 2D Element
 * @method custom
 */
EJSS_DRAWING2D.custom = function (name) {
  var self = EJSS_DRAWING2D.element(name);
  var mFunction;
  
  self.getClass = function() {
  	return "ElementCustom";
  };
  
    self.registerProperties = function(controller) {
    EJSS_DRAWING2D.Custom.registerProperties(self,controller);
  };


  /**
  * A function to call when the element must draw.
  * The function will receive the graphic context of the panel in which it draws
  */
  self.setFunction = function(customFunction) {
    mFunction = customFunction;
  };
  
  self.getFunction = function() {
    return mFunction;
  };
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([1,1]);

  return self;
};



/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia and Félix J. García 
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*/

/**
 * Framework for 2D drawing.
 * @module 2Ddrawing
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * A DrawingPanel is a 2D drawing panel with no decoration. 
 * @class EJSS_DRAWING2D.DrawingPanel
 * @see EJSS_DRAWING2D.PlottingPanel
 * @constructor
 */
EJSS_DRAWING2D.DrawingPanel = {
	MOUSE_ENTERED : 0,
	MOUSE_EXITED : 1,
	MOUSE_PRESSED : 2,
	MOUSE_DRAGGED : 3,
	MOUSE_RELEASED : 4,

	GRAPHICS2D_SVG : 0,
	GRAPHICS2D_CANVAS : 1,

	SCALE_NUM : 0,
	SCALE_LOG : 1,
	
	ENABLED_NONE : 0,
	ENABLED_ANY : 1,
	ENABLED_X : 2,
	ENABLED_Y : 3,
	ENABLED_NO_MOVE : 4,

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		// No super class
		// EJSS_INTERFACE.Element.registerProperties(element.getGraphics(), controller);

		element.setController(controller);

	  	controller.registerProperty("AutoScale", element.setAutoScale, element.getAutoScale);
	 /*** 
	  * Whether the panel should autoscale in the X dimension
	  * @property AutoScaleX 
	  * @type boolean
	  * @default "false"
	  */ 
		controller.registerProperty("AutoScaleX", element.setAutoScaleX, element.getAutoScaleX);
	 /*** 
	  * Whether the panel should autoscale in the Y dimension
	  * @property AutoScaleY 
	  * @type boolean
	  * @default "false"
	  */ 
		controller.registerProperty("AutoScaleY", element.setAutoScaleY, element.getAutoScaleY);		
		controller.registerProperty("InvertedScaleY", element.setInvertedScaleY, element.getInvertedScaleY);
	 /*** 
	  * The minimum value for the X coordinates of elements. 
	  * If specified, the minimum is respected even if AutoscaleX is true and there are no elements close to the minimum.   
	  * @property MinimumX 
	  * @type int or double
	  * @default "-1"
	  */ 
		controller.registerProperty("MinimumX", element.setWorldXMin, element.getWorldXMin);
	 /*** 
	  * The maximum value for the X coordinates of elements. 
	  * If specified, the maximum is respected even if AutoscaleX is true and there are no elements close to the maximum.   
	  * @property MaximumX 
	  * @type int or double
	  * @default "+1"
	  */ 
		controller.registerProperty("MaximumX", element.setWorldXMax, element.getWorldXMax);
	 /*** 
	  * The minimum value for the Y coordinates of elements. 
	  * If specified, the minimum is respected even if AutoscaleY is true and there are no elements close to the minimum.   
	  * @property MinimumY 
	  * @type int or double
	  * @default "-1"
	  */ 
		controller.registerProperty("MinimumY", element.setWorldYMin, element.getWorldYMin);
	 /*** 
	  * The maximum value for the Y coordinates of elements. 
	  * If specified, the maximum is respected even if AutoscaleY is true and there are no elements close to the maximum.   
	  * @property MaximumXY
	  * @type int or double
	  * @default "+1"
	  */ 
		controller.registerProperty("MaximumY", element.setWorldYMax, element.getWorldYMax);
		controller.registerProperty("Bounds", element.setWorldCoordinates, element.getWorldCoordinates);
	 /*** 
	  * The type of scale for the X axis 
	  * @property ScaleXType
	  * @type int or String. One of: <ul>
	  *   <li>EJSS_DRAWING2D.DrawingPanel.SCALE_NUM ("SCALE_NUM" for short)</li>
	  *   <li>EJSS_DRAWING2D.DrawingPanel.SCALE_LOG ("SCALE_LOG" for short)</li>
	  * </ul>
	  * @default "SCALE_NUM"
	  */ 
		controller.registerProperty("ScaleXType", function(v) {
			element.setTypeScaleX(v);
			element.scale();
		}, element.getTypeScaleX);
	 /*** 
	  * The type of scale for the Y axis 
	  * @property ScaleYType
	  * @type int or String. One of: <ul>
	  *   <li>EJSS_DRAWING2D.DrawingPanel.SCALE_NUM ("SCALE_NUM" for short)</li>
	  *   <li>EJSS_DRAWING2D.DrawingPanel.SCALE_LOG ("SCALE_LOG" for short)</li>
	  * </ul>
	  * @default "SCALE_NUM"
	  */ 
		controller.registerProperty("ScaleYType", function(v) {
			element.setTypeScaleY(v);
			element.scale();
		}, element.getTypeScaleY);
	 /*** 
	  * When autoscaling the X axis, the percentage of the X range that should be left (on both sides) around the elements 
	  * @property MarginX
	  * @type int or double (in the range [0,100])
	  * @default "0"
	  */ 
		controller.registerProperty("MarginX", element.setMarginX,element.getMarginX);
	 /*** 
	  * When autoscaling the Y axis, the percentage of the Y range that should be left (on both sides) around the elements 
	  * @property MarginY
	  * @type int or double (in the range [0,100])
	  * @default "0"
	  */ 
		controller.registerProperty("MarginY", element.setMarginY,element.getMarginY);

		controller.registerProperty("Parent", element.getGraphics().setParent, element.getGraphics().getParent);
	 /*** 
	  * A classname for the element that can be used in CSS files 
	  * @property ClassName
	  * @type String
	  * @default "undefined"
	  */ 
        controller.registerProperty("ClassName", element.getGraphics().setClassName);

	 /*** 
	  * The width of the HTML5 element.
	  * See <a href="http://www.w3schools.com/cssref/pr_dim_width.asp">this link</a> for possible values.
	  * @property Width
	  * @type int or String 
	  * @default "auto"
	  */ 
		controller.registerProperty("Width", function(v) {
			if (element.getGraphics().setWidth(v))
				element.scale();
		}, element.getGraphics().getBox().width);
	 /*** 
	  * The height of the HTML5 element.
	  * See <a href="http://www.w3schools.com/cssref/pr_dim_width.asp">this link</a> for possible values.
	  * @property Height
	  * @type int or String 
	  * @default "auto"
	  */ 
		controller.registerProperty("Height", function(v) {
			if (element.getGraphics().setHeight(v))
				element.scale();
		}, element.getGraphics().getBox().height);

		//    controller.registerProperty("X",null,function() { return mInteraction.lastPoint[0]; });
		//    controller.registerProperty("Y",null,function() { return mInteraction.lastPoint[1]; });

	 /*** 
	  * Whether the panel should modify its scales to keep a 1:1 aspect ratio. 
	  * @property SquareAspect
	  * @type boolean 
	  * @default "false"
	  */ 
		controller.registerProperty("SquareAspect", element.setSquareAspect);
	 /*** 
	  * The type of graphics implementation of the panel. 
	  * Changing the graphics mode may influence what elements are actually displayed and the interaction response of the panel.
	  * But it may also influence the rendering speed.   
	  * @property GraphicsMode
	  * @type int or String. One of: <ul>
	  *   <li>EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_SVG ("SVG" for short): SVG based drawing, full interaction, may be slower</li> 
	  *   <li>EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_CANVAS ("CANVAS" for short): Canvas-based drawing, reduced interaction, usually faster</li>
	  * </ul>
	  * @default "SCALE_NUM"
	  */ 
		controller.registerProperty("GraphicsMode", element.setGraphicsMode);
	 /*** 
	  * Whether the panel should respond to user interaction.
	  * This is set to "false" by default, since 'listening' to interaction slows down the rendering.  
	  * @property Enabled
	  * @type boolean 
	  * @default "false"
	  */ 
		controller.registerProperty("Enabled", element.getPanelInteraction().setEnabled);
	 /*** 
	  * Whether to enable the move event listener.
	  * This is set to "false" by default and is ignored when Enabled is set to "false".  
	  * @property StopMoveEvents
	  * @type boolean 
	  * @default "false"
	  */ 
		controller.registerProperty("StopMoveEvents", element.getPanelInteraction().setStopMoveEvents);
	 /*** 
	  * Whether to propagate events and gestures to the containing HTML element.
	  * This may be useful, for instance, to prevent ePub readers from reacting to what should be just an interaction with the panel. 
	  * @property StopEventPropagation
	  * @type boolean 
	  * @default "true"
	  */ 
		controller.registerProperty("StopEventPropagation", element.getPanelInteraction().setStopGestures);

	 /*** 
	  * The dimensions of a strip around the main panel drawing area
	  * @property Gutters
	  * @type int[4] representing [left,top,right,bottom], the size of the strip (in pixels) in that part of the panel  
	  * @default "[0,0,0,0] for DrawingPanels, [50,50,50,50] for PlottingPanels"
	  */ 
		controller.registerProperty("Gutters", element.setGutters);
	 /*** 
	  * The stroke color for the outer border of the panel (exterior border of the gutter).
	  * See <a href="http://www.w3schools.com/cssref/css_colornames.asp">this link</a> for possible values.
	  * @property GuttersLineColor 
	  * @type string Any valid CSS color name. "none" for no color
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "Black"
	  */                          
		controller.registerProperty("GuttersLineColor", element.getGuttersStyle().setLineColor);
	 /*** 
	  * The stroke width for the outer border of the panel (exterior border of the gutter).
	  * @property GuttersLineWidth 
	  * @type int width in pixels
	  * @default "1"
	  */                          
		controller.registerProperty("GuttersLineWidth", element.getGuttersStyle().setLineWidth);
	 /*** 
	  * Whether to draw the outer border of the panel (exterior border of the gutter).
	  * @property GuttersDrawLines 
	  * @type boolean
	  * @default "true"
	  */                          
		controller.registerProperty("GuttersDrawLines", element.getGuttersStyle().setDrawLines);
	 /*** 
	  * The fill color for the gutter.
	  * See <a href="http://www.w3schools.com/cssref/css_colornames.asp">this link</a> for possible values.
	  * @property GuttersColor 
	  * @type string Any valid CSS color name. "none" for no color
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "rgb(239,239,255) for DrawingPanels, rgb(211,216,255) for PlottingPanels"
	  */                          
		controller.registerProperty("GuttersColor",     element.getGuttersStyle().setFillColor);
	 /*** 
	  * Whether to fill the gutter.
	  * @property GuttersFill 
	  * @type boolean
	  * @default "true"
	  */                          
		controller.registerProperty("GuttersFill",      element.getGuttersStyle().setDrawFill);
	 /*** 
	  * SVG shape rendering for the gutters
	  * @property GuttersRendering 
	  * @type string
	  * @values "auto","optimizeSpeed","crispEdges","geometricPrecision"  
	  * @default "auto"
	  */                          
		controller.registerProperty("GuttersRendering", element.getStyle().setShapeRendering);

	 /*** 
	  * The fill color for the main drawing area of the panel
	  * See <a href="http://www.w3schools.com/cssref/css_colornames.asp">this link</a> for possible values.
	  * @property Background 
	  * @type string Any valid CSS color name. "none" for no color
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "rgb(239,239,255) for DrawingPanels, White for PlottingPanels"
	  */                          
		controller.registerProperty("Background", element.getStyle().setFillColor);
	 /*** 
	  * The stroke color for the border of the main drawing area of the panel (inner border of the gutters)
	  * See <a href="http://www.w3schools.com/cssref/css_colornames.asp">this link</a> for possible values.
	  * @property Foreground 
	  * @type string Any valid CSS color name. "none" for no color
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "Black"
	  */                          
		controller.registerProperty("Foreground", element.getStyle().setLineColor);
	 /*** 
	  * The stroke color for the border of the main drawing area of the panel (inner border of the gutters).
	  * Same as "Foreground".
	  * See <a href="http://www.w3schools.com/cssref/css_colornames.asp">this link</a> for possible values.
	  * @property LineColor 
	  * @type string Any valid CSS color name. "none" for no color
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "Black"
	  */                          
		controller.registerProperty("LineColor", element.getStyle().setLineColor);
	 /***
	  * The stroke width for the border of the main drawing area of the panel (inner border of the gutters).
	  * @property LineWidth 
	  * @type int width in pixels
	  * @default "1"
	  */                          
		controller.registerProperty("LineWidth", element.getStyle().setLineWidth);
	 /*** 
	  * Whether to draw the inner border of the panel (inside border of the gutter).
	  * @property DrawLines 
	  * @type boolean
	  * @default "true"
	  */                          
		controller.registerProperty("DrawLines", element.getStyle().setDrawLines);
	 /*** 
	  * The fill color for the main drawing area of the panel.
	  * Same as "Background"
	  * See <a href="http://www.w3schools.com/cssref/css_colornames.asp">this link</a> for possible values.
	  * @property FillColor 
	  * @type string Any valid CSS color name. "none" for no color
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "rgb(239,239,255) for DrawingPanels, White for PlottingPanels"
	  */                          
		controller.registerProperty("FillColor", element.getStyle().setFillColor);
	 /*** 
	  * Whether to fill the main drawing area of the panel.
	  * @property DrawFill 
	  * @type boolean
	  * @default "true"
	  */                          
		controller.registerProperty("DrawFill", element.getStyle().setDrawFill);
	 /*** 
	  * SVG shape rendering for the gutters
	  * @property GuttersRendering 
	  * @type string
	  * @values "auto","optimizeSpeed","crispEdges","geometricPrecision"  
	  * @default "auto"
	  */                          
		controller.registerProperty("ShapeRendering", element.getStyle().setShapeRendering);

	 /*** 
	  * Whether the element is visible
	  * @property Visibility 
	  * @type boolean
	  * @default true
	  */                          
	    controller.registerProperty("Visibility", element.getGraphics().getStyle().setVisibility);      
	 /*** 
	  * The CSS display property of the element
	  * See <a href="http://www.w3schools.com/cssref/pr_class_display.asp">this link</a> for possible values.
	  * @property Display 
	  * @type string
	  * @default "inline"
	  */                          
        controller.registerProperty("Display", element.getGraphics().getStyle().setDisplay); 
	 /*** 
	  * An object with one or more CSS properties.
	  * Example { "background-color" : "red", "float" : "right" }
	  * @property CSS 
	  * @type Object 
	  * @default "inline"
	  */                          
		controller.registerProperty("CSS", element.getGraphics().getStyle().setCSS);

		controller.registerProperty("TRMessage", element.getMessageDecoration("TR").setText);
		controller.registerProperty("TRMessageFont", element.getMessageDecoration("TR").getFont().setFont);
		controller.registerProperty("TRMessageColor", element.getMessageDecoration("TR").getFont().setFillColor);
		controller.registerProperty("TRMessageFillColor", element.getMessageDecoration("TR").getStyle().setFillColor);
		controller.registerProperty("TRMessageLineColor", element.getMessageDecoration("TR").getStyle().setLineColor);		
		
		controller.registerProperty("TLMessage", element.getMessageDecoration("TL").setText);
		controller.registerProperty("TLMessageFont", element.getMessageDecoration("TL").getFont().setFont);
		controller.registerProperty("TLMessageColor", element.getMessageDecoration("TL").getFont().setFillColor);
		controller.registerProperty("TLMessageFillColor", element.getMessageDecoration("TL").getStyle().setFillColor);
		controller.registerProperty("TLMessageLineColor", element.getMessageDecoration("TL").getStyle().setLineColor);		

		controller.registerProperty("BRMessage", element.getMessageDecoration("BR").setText);
		controller.registerProperty("BRMessageFont", element.getMessageDecoration("BR").getFont().setFont);
		controller.registerProperty("BRMessageColor", element.getMessageDecoration("BR").getFont().setFillColor);
		controller.registerProperty("BRMessageFillColor", element.getMessageDecoration("BR").getStyle().setFillColor);
		controller.registerProperty("BRMessageLineColor", element.getMessageDecoration("BR").getStyle().setLineColor);		

		controller.registerProperty("BLMessage", element.getMessageDecoration("BL").setText);
		controller.registerProperty("BLMessageFont", element.getMessageDecoration("BL").getFont().setFont);
		controller.registerProperty("BLMessageColor", element.getMessageDecoration("BL").getFont().setFillColor);
		controller.registerProperty("BLMessageFillColor", element.getMessageDecoration("BL").getStyle().setFillColor);
		controller.registerProperty("BLMessageLineColor", element.getMessageDecoration("BL").getStyle().setLineColor);		

		controller.registerProperty("CoordinatesFormat", element.getCoordinates().setFormat);
		controller.registerProperty("ShowCoordinates", element.setShowCoordinates);
		controller.registerProperty("ShowAreaRectangle", element.setShowAreaRectangle);
		controller.registerProperty("EnabledZooming", element.setEnabledZooming);
		controller.registerProperty("EnabledDragging", element.setEnabledDragging);

	 /*** 
	  * The type of cursor when an element is moved.
	  * See <a href="http://www.w3schools.com/cssref/pr_class_cursor.asp">this link</a> for possible values.
	  * @property CursorTypeForMove 
	  * @type String
	  * @default "move"
	  */ 
		controller.registerProperty("CursorTypeForMove", element.getInteraction().setCursorTypeForMove);

        controller.registerAction("OnDoubleClick", element.getInteraction().getInteractionPoint);      
		controller.registerAction("OnMove", element.getInteraction().getInteractionPoint, element.getOnMoveHandler);
		controller.registerAction("OnPress", element.getInteraction().getInteractionPoint, element.getOnPressHandler);
		controller.registerAction("OnDrag", element.getInteraction().getInteractionPoint, element.getOnDragHandler);
		controller.registerAction("OnRelease", element.getInteraction().getInteractionBounds, element.getOnReleaseHandler);
		controller.registerAction("OnResize", element.getInteraction().getInteractionBounds, element.scale);
		controller.registerAction("OnOrientationChange", element.getInteraction().getOrientation, element.scale);
		controller.registerAction("OnZoom", element.getInteraction().getPinchDistance, element.getOnZoomHandler);
	}
};

/**
 * Constructor for DrawingPanel
 * @method drawingPanel
 * @param mName string
 * @returns An abstract 2D drawing panel
 */
EJSS_DRAWING2D.drawingPanel = function(mName,mGraphicsMode) {
	var self = {}; // reference returned	

	// Graphics implementation
	var mGraphics;

	// Instance variables
	var mStyle = EJSS_DRAWING2D.style(mName);	// style for panel
	var mBottomDecorations = [];				// decorations list for panel
	var mElements = [];							// elements list for panel
	var mElementsChanged = false;				// whether elements list has changed
	var mAutoScaleX = false;					// whether auto-scale with measure elements in X
	var mAutoScaleY = false;					// whether auto-scale with measure elements in Y
	var mInvertedScaleY = false;				// whether inverted scale in Y
	var mTypeScaleX = 0;						// type of scale in X
	var mTypeScaleY = 0;						// type of scale in Y
	var mTopDecorations = [];					// top decorations list for panel
    var mShowCoordinates = true;				// whether coordinates in panel
    var mShowAreaRectangle = true;				// whether area rectangle in panel
	var mEnabledDragging = false;				// whether dragging panel
	var mEnabledZooming = false;				// whether scaling panel
	var mZoomRate = 1.10;						// scaling rate
	var mZoomLimits = [0.1,1000];				// zooming limits (width min and max)
	var mCollectersList = [];		            // Array of all control elements that need a call to dataCollected() after data collection

	// Configuration variables	
	var xmindef = -1, xmaxdef = +1, ymindef = -1, ymaxdef = +1;  // default dimensions
	var mWorld = {
		// preferred dimensions
		xminPreferred : NaN, xmaxPreferred : NaN, yminPreferred : NaN, ymaxPreferred : NaN,
		// measured dimensions
		xminMeasured : NaN, xmaxMeasured : NaN, yminMeasured : NaN, ymaxMeasured : NaN,
		// margin
		xmargin : 0, ymargin : 0,
		// real dimensions after scalation, based on preferred dimensions and added margin
		xmin : -1, xmax : +1, ymin : -1, ymax : +1,
		// keep square aspect
		squareAspect : false,
		// origin in panel
		xorigin : 0, yorigin : 0,
		// pixel per unit for panel
		xscale : 1, yscale : 1		
	};

	// Gutters for panel
	var mGutters = {
		left : 0, right : 0, top : 0, bottom : 0, // sizes
		visible : false, // whether the gutters are visible
	};
	var mGuttersStyle = EJSS_DRAWING2D.style(mName + " gutters")	// style

	// Implementation variables
	var mPanelChanged = true;	// whether panel changed (style, decorations, gutters)
	var mMustScale = true;		// whether panel must scale
	var mEnabledRedering = true;// whether redering is enabled
	
	var mController = {// dummy controller object
		propertiesChanged : function() { },
		invokeAction : function() { }
	};

	// ----------------------------------------
	// Instance functions
	// ----------------------------------------

	/***
	 * Get name for drawing panel
	 * @method getName
	 * @return string
	 */
	self.getName = function() {
		return mName;
	};

	/***
	 * Returns the graphics implementation
	 * @method getGraphics
	 * @return Graphics
	 */
	self.getGraphics = function() {
		return mGraphics;
	};

	/***
	 * Returns the svg image in Base64 format
	 * @method importGraphics
	 * @param callback
	 * @return string 
	 */
    self.importGraphics = function(callback) {
    	if(mGraphics.importSVG)
    		return mGraphics.importSVG(callback);
    	return null;
    }
    
	/***
	 * Return the drawing style of the inner rectangle for panel
	 * @method getStyle
	 * @return Style
	 */
	self.getStyle = function() {
		return mStyle;
	};


	/***
	 * Get the graphics mode
	 * @method getGraphicsMode
	 * @return One of the possible graphics implementation mode: EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_SVG or EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_CANVAS
	 */
	self.getGraphicsMode = function() {
	  return mGraphicsMode;
	};
	
	/***
	 * Get the graphics mode
	 * @method getGraphicsModeName
	 * @return The implementation mode as a string: either "SVG" or "CANVAS"
	 */
	self.getGraphicsModeName = function() {
	  switch (mGraphicsMode) {
	    case EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_CANVAS : return "CANVAS";
	    default:
	    case EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_SVG : return "SVG";
	  }
	};


	/***
	 * Set graphics
	 * @method setGraphicsMode
	 * @param mode One of the possible graphics implementation mode: 
	 *   EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_SVG or EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_CANVAS. Or simply "CANVAS" or "SVG"
	 */
	self.setGraphicsMode = function(mode) {
		// get mode
    	if (typeof mode === 'string') {
    	    if (mode.indexOf("GRAPHICS2D_")!=0) mode = "GRAPHICS2D_" + mode;
      		mode = EJSS_DRAWING2D.DrawingPanel[mode.toUpperCase()] | EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_SVG;
      	} else if(typeof mode === 'undefined') {
			mode = EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_SVG;      		
      	}
    	
      	if (mode == mGraphicsMode) return;
      	mGraphicsMode = mode; 
		var exists = typeof mGraphics !== 'undefined';
		
		if(exists) {
			// get params
			var parent = mGraphics.getParent();
			var width = mGraphics.getWidth();
			var height = mGraphics.getHeight();
			var style = mGraphics.getEventContext().getAttribute("style");
			
			// remove current graphics
			var ele = document.getElementById(mName);
	   		ele.parentNode.removeChild(ele);			
		}
		
		// create new graphics
		if (mode == EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_SVG) {
			mGraphics = EJSS_GRAPHICS.svgGraphics(mName);
		} else if (mode == EJSS_DRAWING2D.DrawingPanel.GRAPHICS2D_CANVAS) {
			mGraphics = EJSS_GRAPHICS.canvasGraphics(mName);
		} else {
			console.log("WARNING: setGraphics() - Graphics not supported");
		}
		
		if(exists) {
			// set params
			mGraphics.setParent(parent);
			mGraphics.setWidth(width);	
			mGraphics.setHeight(height);	
			mGraphics.getEventContext().setAttribute("style",style);	
			mElementsChanged = true;
			
			// interactions
			mInteraction.reload();			
		}
	};
	
	/***
	 * Get auto-scale
	 * @method getAutoScale
	 * @return boolean
	 */
	self.getAutoScale = function() {
		return (mAutoScaleX || mAutoScaleY);
	};

	/***
	 * Set auto-scale
	 * @method setAutoScale
	 * @param auto
	 */
	self.setAutoScale = function(autoscale) {		
		self.setAutoScaleX(autoscale);
		self.setAutoScaleY(autoscale);
	};

	/***
	 * Set auto-scale in X
	 * @method setAutoScaleX
	 * @param auto
	 */
	self.setAutoScaleX = function(autoscale) {
		mAutoScaleX = autoscale;
	};

	/***
	 * Get auto-scale in X
	 * @method getAutoScaleX
	 * @return boolean
	 */
	self.getAutoScaleX = function() {
		return mAutoScaleX;
	};

	/***
	 * Set auto-scale in Y
	 * @method setAutoScaleY
	 * @param auto
	 */
	self.setAutoScaleY = function(autoscale) {
		mAutoScaleY = autoscale;
	};

	/***
	 * Get auto-scale in Y
	 * @method getAutoScaleY
	 * @return boolean
	 */
	self.getAutoScaleY = function() {
		return mAutoScaleY;
	};

	/***
	 * Set inverted scale in Y
	 * @method setInvertedScaleY
	 * @param inverted
	 */
	self.setInvertedScaleY = function(invertedscale) {
		if(mInvertedScaleY != invertedscale) {
			mInvertedScaleY = invertedscale;
			
			// InvertedScaleY ONLY supports mTypeScaleY NUM
			if(mInvertedScaleY) mTypeScaleY = EJSS_DRAWING2D.DrawingPanel.SCALE_NUM;
			
			// report decorations
			var changed = self.reportDecorations("bounds");
			// decorations may request for a repaint
			if (changed) mPanelChanged = true;			
		}
	};

	/***
	 * Get inverted scale in Y
	 * @method getInvertedScaleY
	 * @return boolean
	 */
	self.getInvertedScaleY = function() {
		return mInvertedScaleY;
	};

	/***
	 * Set type of scale in X
	 * @method setTypeScaleX
	 * @param type
	 */
	self.setTypeScaleX = function(typescale) {
    	if (typeof typescale == "string") typescale = EJSS_DRAWING2D.DrawingPanel[typescale.toUpperCase()];
		mTypeScaleX = typescale;
	};

	/***
	 * Get type of scale in X
	 * @method getTypeScaleX
	 * @return int
	 */
	self.getTypeScaleX = function() {
		return mTypeScaleX;
	};

	/***
	 * Set type of scale in Y
	 * @method setTypeScaleY
	 * @param type
	 */
	self.setTypeScaleY = function(typescale) {
		if (typeof typescale == "string") typescale = EJSS_DRAWING2D.DrawingPanel[typescale.toUpperCase()];
		mTypeScaleY = typescale;
		
		// InvertedScaleY ONLY supports mTypeScaleY NUM
		if (mTypeScaleY == EJSS_DRAWING2D.DrawingPanel.SCALE_LOG) mInvertedScaleY = false;	
	};

	/***
	 * Get type of scale in Y
	 * @method getTypeScaleY
	 * @return int
	 */
	self.getTypeScaleY = function() {
		return mTypeScaleY;
	};

	// ----------------------------------------
	// World coordinates
	// ----------------------------------------

	/***
	 * Sets the preferred minimum X coordinate for the panel
	 * @method setWorldXMin
	 * @param xmin
	 */
	self.setWorldXMin = function(xmin) {
		if (xmin !== mWorld.xminPreferred) {
			mWorld.xminPreferred = xmin;
		}
	};

	/***
	 * Returns the preferred minimum X coordinate for the panel
	 * @method getWorldXMin
	 * @return double
	 */
	self.getWorldXMin = function() {
		return mWorld.xminPreferred;
	};

	/***
	 * Sets the preferred maximum X coordinate for the panel
	 * @method setWorldXMax
	 * @param xmax
	 */
	self.setWorldXMax = function(xmax) {
		if (xmax !== mWorld.xmaxPreferred) {
			mWorld.xmaxPreferred = xmax;
		}
	};

	/***
	 * Returns the preferred maximum X coordinate for the panel
	 * @method getWorldXMax
	 * @return double
	 */
	self.getWorldXMax = function() {
		return mWorld.xmaxPreferred;
	};

	/***
	 * Sets the preferred minimum Y coordinate for the panel
	 * @method setWorldYMin
	 * @param ymin
	 */
	self.setWorldYMin = function(ymin) {
		if (ymin !== mWorld.yminPreferred) {
			mWorld.yminPreferred = ymin;
		}
	};

	/***
	 * Returns the preferred minimum Y coordinate for the panel
	 * @method getWorldYMin
	 * @return double
	 */
	self.getWorldYMin = function() {
		return mWorld.yminPreferred;
	};

	/***
	 * Sets the preferred maximum Y coordinate for the panel
	 * @method setWorldYMax
	 * @param ymax
	 */
	self.setWorldYMax = function(ymax) {
		if (ymax !== mWorld.ymaxPreferred) {
			mWorld.ymaxPreferred = ymax;
		}
	};

	/***
	 * Returns the preferred maximum Y coordinate for the panel
	 * @method getWorldYMax
	 * @return double
	 */
	self.getWorldYMax = function() {
		return mWorld.ymaxPreferred;
	};

	/***
	 * Sets the preferred user coordinates for the panel
	 * @method setWorldCoordinates
	 * @param bounds
	 */
	self.setWorldCoordinates = function(bounds) {
		self.setWorldXMin(bounds[0]);
		self.setWorldXMax(bounds[1]);
		self.setWorldYMin(bounds[2]);
		self.setWorldYMax(bounds[3]);
	};

	/***
	 * Gets the preferred user coordinates for the panel
	 * @method getWorldCoordinates
	 * @return bounds
	 */
	self.getWorldCoordinates = function() {
		return [self.getWorldXMin(), self.getWorldXMax(), self.getWorldYMin(), self.getWorldYMax()];
	};

	/***
	 * Gets the measured coordinates for the panel
	 * @method getMeasuredCoordinates
	 * @return bounds
	 */
	self.getMeasuredCoordinates = function() {
		return [mWorld.xminMeasured, mWorld.xmaxMeasured, mWorld.yminMeasured, mWorld.ymaxMeasured];
	};

	/***
	 * Gets the user coordinates for the panel
	 * @method getRealWorldCoordinates
	 * @return bounds
	 */
	self.getRealWorldCoordinates = function() {
		return [mWorld.xmin, mWorld.xmax, mWorld.ymin, mWorld.ymax];
	};

	/***
	 * Returns the minimum X coordinate for the panel
	 * @method getRealWorldXMin
	 * @return double
	 */
	self.getRealWorldXMin = function() {
		return mWorld.xmin;
	};

	/***
	 * Returns the maximum X coordinate for the panel
	 * @method getRealWorldXMax
	 * @return double
	 */
	self.getRealWorldXMax = function() {
		return mWorld.xmax;
	};

	/***
	 * Returns the minimum Y coordinate for the panel
	 * @method getRealWorldYMin
	 * @return double
	 */
	self.getRealWorldYMin = function() {
		return mWorld.ymin;
	};

	/***
	 * Returns the maximum Y coordinate for the panel
	 * @method getRealWorldYMax
	 * @return double
	 */
	self.getRealWorldYMax = function() {
		return mWorld.ymax;
	};

	/***
	 * Whether the panel should keep a 1:1 aspect ratio between X and Y coordinates
	 * @method setSquareAspect
	 * @param boolean
	 */
	self.setSquareAspect = function(square) {
		if (square !== mWorld.squareAspect) {
			mWorld.squareAspect = square;
			mMustScale = true;
		}
	};

	/***
	 * Set margin X
	 * @method setMarginX
	 * @param margin
	 */
	self.setMarginX = function(margin) {
		if(mWorld.xmargin != margin) {
			mWorld.xmargin = margin;
			mMustScale = true;			
		}		
	};
	
	/***
	 * Get margin X
	 * @method getMarginX
	 * @return margin
	 */	
	self.getMarginX = function() {
		return mWorld.xmargin;
	}

	/***
	 * Set margin Y
	 * @method setMarginY
	 * @param margin
	 */
	self.setMarginY = function(margin) {
		if(mWorld.ymargin != margin) {
			mWorld.ymargin = margin;
			mMustScale = true;			
		}		
	};
	
	/***
	 * Get margin Y
	 * @method getMarginY
	 * @return margin
	 */	
	self.getMarginY = function() {
		return mWorld.ymargin;
	}

	// ----------------------------------------
	// Decorations and elements
	// ----------------------------------------

	/***
	 * Allow to drag panel
	 * @method setEnabledDragging
	 * @parem motion ENABLED_NONE: 0, ENABLED_ANY: 1, ENABLED_X: 2, ENABLED_Y: 3, ENABLED_NO_MOVE: 4
	 */
	self.setEnabledDragging = function(motion) {
		if ( typeof motion == "string") {
			value = EJSS_DRAWING2D.DrawingPanel[motion.toUpperCase()];
			mEnabledDragging = (typeof value === 'undefined')?EJSS_DRAWING2D.DrawingPanel.ENABLED_NONE:value;
		}
	    else mEnabledDragging = motion;
	}

	/***
	 * Type of motion allowed to drag panel
	 * @method getEnabledDragging
	 * @return motion
	 */
	self.getEnabledDragging = function() {
		return mEnabledDragging;
	}

	/***
	 * Allow to scale panel
	 * @method setEnabledZooming
	 * @param allowed
	 */
	self.setEnabledZooming = function(allowed) {
		mEnabledZooming = allowed;
	}

	/***
	 * Whether panal scaling is allowed
	 * @method getEnabledZooming
	 * @return boolean
	 */
	self.getEnabledZooming = function() {
		return mEnabledZooming;
	}

	/***
	 * Set zoom rate
	 * @method setZoomRate
	 * @param rate
	 */
	self.setZoomRate = function(rate) {
		mZoomRate = rate;
	}

	/***
	 * Get zoom rate
	 * @method getZoomRate
	 * @return rate
	 */
	self.getZoomRate = function() {
		return mZoomRate;
	}

	/***
	 * Set zoom limits
	 * @method setZoomLimits
	 * @param limits [min,max]
	 */
	self.setZoomLimits = function(limits) {
		mZoomLimits = limits;
	}

	/***
	 * Get zoom limits
	 * @method getZoomLimits
	 * @return limits [min,max]
	 */
	self.getZoomLimits = function() {
		return mZoomLimits;
	}
	
	/***
	 * Show coordinates in panel
	 * @method setShowCoordinates
	 * @param boolean
	 */
	self.setShowCoordinates = function(show) {
		mShowCoordinates = show;
	}

	/***
	 * Whether coordinates in panel are showed
	 * @method getShowCoordinates
	 * @return boolean
	 */
	self.getShowCoordinates = function() {
		return mShowCoordinates;
	}

	/***
	 * Show area rectangle in panel
	 * @method setShowAreaRectangle
	 * @param boolean
	 */
	self.setShowAreaRectangle = function(show) {
		mShowAreaRectangle = show;
	}

	/***
	 * Show area rectangle in panel
	 * @method getShowAreaRectangle
	 * @param boolean
	 */
	self.getShowAreaRectangle = function() {
		return mShowAreaRectangle;
	}

	/***
	 * Adds a decoration to the panel. Decorations are drawn before any other elements.
	 * @method addDecoration
	 * @param drawable decoration element
	 * @param position integer
	 * @param istop top decoration
	 */
	self.addDecoration = function(drawable, position, istop) {
		if (istop)
			EJSS_TOOLS.addToArray(mTopDecorations, drawable, position);
		else
			EJSS_TOOLS.addToArray(mBottomDecorations, drawable, position);
		if (drawable.setPanel)// set this panel to decoration element
			drawable.setPanel(self);
		return self;
	};

	/***
	 * Removes a decoration
	 * @method removeDecoration
	 * @param drawable decoration element
	 */
	self.removeDecoration = function(drawable) {
		EJSS_TOOLS.removeFromArray(mBottomDecorations, drawable);
		EJSS_TOOLS.removeFromArray(mTopDecorations, drawable);
		if (drawable.setPanel)// remove this panel to decoration element
			drawable.setPanel(null);
		return self;
	};

	/***
	 * Add a element to the panel. Elements are asked to draw themselves
	 * whenever the panel needs to render. For this purpose, they will receive a
	 * calls to draw().
	 * Elements are reported of changes in the world coordinates of the panel, in case
	 * they need to recalculate themselves.
	 * @method addElement
	 * @param element Element
	 * @param position int
	 */
	self.addElement = function(element, sibling) {
		if(typeof sibling === 'undefined') {
			EJSS_TOOLS.addToArray(mElements, element);			
		} else if (typeof sibling === 'object') { // sibling object
			var index = EJSS_TOOLS.arrayObjectIndexOf(mElements,sibling);
			EJSS_TOOLS.addToArray(mElements, element, index);
		} else { // position
			EJSS_TOOLS.addToArray(mElements, element, sibling);
		}
		// set this panel to decoration element
		element.setPanel(self);
		if (element.dataCollected) mCollectersList.push(element);
		// elements list has changed
		mElementsChanged = true;
	};

	/***
	 * Remove a element to the panel.
	 * @method removeElement
	 * @param element Element
	 */
	self.removeElement = function(element) {
		EJSS_TOOLS.removeFromArray(mElements, element);
		mInteraction.clearInteractionElement(element);
		element.setPanel(null);
		if (element.dataCollected) EJSS_TOOLS.removeFromArray(mCollectersList, element);
		// elements list has changed
		mElementsChanged = true;
	};

	/***
	 * Return the array of a elements.
	 * @method getElements
	 * @return Elements
	 */
	self.getElements = function() {
		return mElements;
	};

	/***
	 * Return the position of a element.
	 * @method indexOfElement
	 * @param element Element
	 * @return integer
	 */
	self.indexOfElement = function(element) {
		return mElements.indexOf(element);
	};

	// ----------------------------------------
	// Gutters (empty or decorated area around the drawing area)
	// ----------------------------------------

	/***
	 * Sets the gutters dimensions
	 * @method setGutters
	 * @param rect the number of pixels for the drawing area
	 */
	self.setGutters = function(rect) {
		var left = rect[0], top = rect[1], right = rect[2], bottom = rect[3];

		var changed = false;
		if (left !== mGutters.left) {
			mGutters.left = left;
			changed = true;
		}
		if (top !== mGutters.top) {
			mGutters.top = top;
			changed = true;
		}
		if (right !== mGutters.right) {
			mGutters.right = right;
			changed = true;
		}
		if (bottom !== mGutters.bottom) {
			mGutters.bottom = bottom;
			changed = true;
		}
		if (changed) {
			mGutters.visible = (mGutters.left > 0 || mGutters.top > 0 || mGutters.right > 0 || mGutters.bottom > 0);
			mPanelChanged = true;
			mMustScale = true;
		}		
	};

	/***
	 * Return the bounding gutters
	 * @method getGutters
	 * @return Gutters
	 */
	self.getGutters = function() {
		return mGutters;
	};

	/***
	 * Return the drawing style of the bounding gutters
	 * @method getGuttersStyle
	 * @return Style
	 */
	self.getGuttersStyle = function() {
		return mGuttersStyle;
	};

	// ----------------------------------------
	// Apply transformations and conversions
	// ----------------------------------------

	/***
	 * Converts a Y pixel value so that 0 is at the bottom
	 * @method toPixelAxisY
	 * @param y double
	 */
	self.toPixelAxisY = function(y) {
		return (mWorld.yorigin - y) - (mWorld.yscale * mWorld.ymin);;
	};

	/***
	 * Converts a X pixel value so that 0 is at the left
	 * @method toPixelAxisX
	 * @param x double
	 */
	self.toPixelAxisX = function(x) {
		return (mWorld.xorigin + x) - (mWorld.xscale * mWorld.xmin);
	};

	/***
	 * To be used only after a call to render()!
	 * Projects a point from world coordinates to pixel coordinates.
	 * @method toPixelPosition
	 * @param point double[] The original coordinates
	 * @param scale double[] The type of scale (LOG or NUM)
	 * @return double[] An array with the result
	 */
	self.toPixelPosition = function(point,scale) {
		var scaleX = (typeof scale != "undefined")? scale : mTypeScaleX; // if scale is undefined, it uses the panel scale
		var scaleY = (typeof scale != "undefined")? scale : mTypeScaleY; // if scale is undefined, it uses the panel scale
		var pos = [];
		if(scaleX == EJSS_DRAWING2D.DrawingPanel.SCALE_LOG) // LOG
	    	pos[0] = self.toPixelLogScale(point[0], mWorld.xorigin, mWorld.xmin, mWorld.xmax, mWorld.xscale); 						    		   		
		else // NUM
			pos[0] = mWorld.xorigin + mWorld.xscale * (point[0] - mWorld.xmin);		
		
		if(scaleY == EJSS_DRAWING2D.DrawingPanel.SCALE_LOG) // LOG
			pos[1] = self.toPixelLogScale(point[1], mWorld.yorigin, mWorld.ymin, mWorld.ymax, mWorld.yscale);		
		else { // NUM
			if(!mInvertedScaleY)
				pos[1] = mWorld.yorigin + mWorld.yscale * (point[1] - mWorld.ymin);
			else
				pos[1] = mGutters.top + mWorld.yscale * (mWorld.ymin - point[1]);			
		}
			
		return pos;
	};

	/***
	 * To be used only after a call to render()!
	 * Projects a module from world coordinates to pixel coordinates
	 * @method toPixelMod
	 * @param point double[] The original module
	 * @return double[] The same array once transformed
	 */
	self.toPixelMod = function(mod) {
		var pmod = [];
		pmod[0] = mod[0] * mWorld.xscale;
		pmod[1] = mod[1] * mWorld.yscale;		
		return pmod;
	};

	/***
	 * To be used only after a call to render()!
	 * Projects a point from pixel coordinates to world coordinates
	 * @method toPanelPosition
	 * @param point double[] The original coordinates
	 * @param scale double[] The type of scale (LOG or NUM)
	 * @return double[] The same array once transformed
	 */
	self.toPanelPosition = function(point,scale) {	
		var scaleX = (typeof scale != "undefined")? scale : mTypeScaleX; // if scale is undefined, it uses the panel scale
		var scaleY = (typeof scale != "undefined")? scale : mTypeScaleY; // if scale is undefined, it uses the panel scale
		var pos = [];
		if(scaleX == EJSS_DRAWING2D.DrawingPanel.SCALE_LOG)  // LOG 
	    	pos[0] = self.toPanelLogScale(point[0], mWorld.xorigin, mWorld.xmin, mWorld.xmax, mWorld.xscale); 						
		 else  // NUM 
			pos[0] = mWorld.xmin + (point[0] - mWorld.xorigin) / mWorld.xscale;
		
		if(scaleY == EJSS_DRAWING2D.DrawingPanel.SCALE_LOG)  // LOG
			pos[1] = self.toPanelLogScale(point[1], mWorld.yorigin, mWorld.ymin, mWorld.ymax, mWorld.yscale);
		else { // NUM
			if(!mInvertedScaleY)
				pos[1] = mWorld.ymin + (point[1] - mWorld.yorigin) / mWorld.yscale;
			else
				pos[1] = mWorld.ymin - (point[1] - mGutters.top) / mWorld.yscale;			
		}
		return pos;
	};

	/***
	 * To be used only after a call to render()!
	 * Projects a module from pixel coordinates to world coordinates
	 * @method toPanelMod
	 * @param point double[] The original module
	 * @return double[] The same array once transformed
	 */
	self.toPanelMod = function(mod) {
		var pmod = [];
		pmod[0] = ((mWorld.xscale == 0)? 0 : mod[0]/mWorld.xscale);
		pmod[1] = ((mWorld.yscale == 0)? 0 : mod[1]/mWorld.yscale);		
		return pmod;
	};

	/***
	 * Projects a value in world coordinates from a decimal scale 
	 * (defined by origin, min, max and pixratio) to a log scale in pixel coordinates
	 * @method toPixelLogScale
	 * @param value double The original value
	 * @param origin double The origin in panel (pixels)
	 * @param min double The min value in world coordinates
	 * @param max double The max value in world coordinates
	 * @param pixratio double The pixels per unit for panel
	 * @return double The value transformed
	 */
	self.toPixelLogScale = function(value, origin, min, max, pixratio) {
    	var minscale = (min <= 0)? 1 : Math.log(min)/Math.log(10);		// min log value
    	var maxscale = (max <= 0)? 1 : Math.log(max)/Math.log(10);		// max log value
		var segsize = (max - min) * pixratio;
    	var scalesize = maxscale - minscale;			// size of log scale
    	var sizeratio = segsize / scalesize;			// pixel per log value
			
   		var step = value <= 0? 0 : Math.log(value) / Math.log(10);
   		return origin + sizeratio * (step - minscale);   		
	}

	/***
	 * Projects a value in pixel coordinates from a decimal scale 
	 * (defined by origin, min, max and pixratio) to a log scale in world coordinates
	 * @method toPanelLogScale
	 * @param value double The original value
	 * @param origin double The origin in panel (pixels)
	 * @param min double The min value in world coordinates
	 * @param max double The max value in world coordinates
	 * @param pixratio double The pixels per unit for panel
	 * @return double The value transformed
	 */
	self.toPanelLogScale = function(value, origin, min, max, pixratio) {
    	var minscale = (min < 0)? 0 : Math.log(min)/Math.log(10);		// min log value
    	var maxscale = (max < 0)? 0 : Math.log(max)/Math.log(10);		// max log value
		var segsize = (max - min) * pixratio;
    	var scalesize = maxscale - minscale;			// size of log scale
    	var sizeratio = segsize / scalesize;			// pixel per log value
		
		return Math.pow(10, minscale + (value - origin) / sizeratio); 								
	}

	/***
	 * Get pixel position of the origin
	 * @method getPixelPositionWorldOrigin
	 * @return array pixel position
	 */
	self.getPixelPositionWorldOrigin = function() {
		return [mWorld.xorigin,mWorld.yorigin];
	}

	/***
	 * Return drawing box (excluding gutters)
	 * @method getInnerRect
	 * @return box  
	 */
	self.getInnerRect = function() {
	 	var bounds = self.getRealWorldCoordinates();
	    if(!self.getInvertedScaleY())
			var pos = self.toPixelPosition([bounds[0],bounds[3]]);
		else
			var pos = self.toPixelPosition([bounds[0],bounds[2]]);		    
	    var size = self.toPixelMod([bounds[1]-bounds[0],bounds[3]-bounds[2]]);     
	    
	    return {x: pos[0], width: Math.abs(size[0]), y: pos[1], height: Math.abs(size[1])}
	},

	/***
	 * Force scale again
	 * @method scale
	 */
	self.scale = function() {
		mMustScale = true;
		mElementsChanged = true;
	};
	
	/***
	 * Refresh all elements
	 * @method touch
	 */
	self.touch = function() {
		mMustScale = true;
		mElementsChanged = true;
		for (var i = 0, n = mElements.length; i < n; i++)
			mElements[i].setChanged(true);
	};
	
	/***
	 * Recomputes the scales of the panel.
	 * @method recomputeScales
	 */
	self.recomputeScales = function() {	
		var newWorld = false;
		
		// start with the preferred min-max values plus margin.
		var rangeX = (mWorld.xmaxMeasured-mWorld.xminMeasured)/2;
		var rangeY = (mWorld.ymaxMeasured-mWorld.yminMeasured)/2;
		var xmin = mWorld.xminMeasured - Math.abs(rangeX*mWorld.xmargin/100); 
		var xmax = mWorld.xmaxMeasured + Math.abs(rangeX*mWorld.xmargin/100);
		var ymin = mWorld.yminMeasured - Math.abs(rangeY*mWorld.ymargin/100); 
		var ymax = mWorld.ymaxMeasured + Math.abs(rangeY*mWorld.ymargin/100);

		// get sizes and scale
		var box = mGraphics.getBox();
		var width = box.width - (mGutters.left + mGutters.right); // width in pixels
		var height = box.height - (mGutters.bottom + mGutters.top); // height in pixels
		var xPixPerUnit = width / (xmax - xmin); // the x scale in pixels
		var yPixPerUnit = height / (ymax - ymin); // the y scale in pixels
		var stretch, diff;
		if (mWorld.squareAspect) {// keep square aspect
			stretch = Math.abs(xPixPerUnit / yPixPerUnit);
			// relation between x scale and y scale
			if (stretch >= 1) {// make the x range bigger so that aspect ratio is one
				stretch = Math.min(stretch, width);
				// limit the stretch
				diff = (xmax - xmin) * (stretch - 1) / 2.0;
				xmin -= diff;
				xmax += diff;
				xPixPerUnit = width / (xmax - xmin);
				//Math.max(width-leftGutter-rightGutter, 1)/(xmax-xmin);  // the x scale in pixels per unit
			} else {// make the y range bigger so that aspect ratio is one
				stretch = Math.max(stretch, 1.0 / height);
				// limit the stretch
				diff = (ymax - ymin) * (1.0 / stretch - 1) / 2.0;
				ymin -= diff;
				ymax += diff;
				yPixPerUnit = height / (ymax - ymin);
				//Math.max(height-bottomGutter-topGutter, 1)/(ymax-ymin); // the y scale in pixels per unit
			}
		}
		// new world, because new scale
		newWorld = (mWorld.xscale != xPixPerUnit || mWorld.yscale != -yPixPerUnit);
		
		// centered
		mWorld.xscale = xPixPerUnit;
		mWorld.yscale = -yPixPerUnit;
		mWorld.xorigin = mGutters.left + 0.5;
		mWorld.yorigin = height + mGutters.top + 0.5;
		mMustScale = false;
		
		// Check for actual changes
		if ((mWorld.xmin !== xmin) || (mWorld.xmax !== xmax) || (mWorld.ymin !== ymin) || (mWorld.ymax !== ymax)) {
			mWorld.xmin = xmin;
			mWorld.xmax = xmax;
			mWorld.ymin = ymin;
			mWorld.ymax = ymax;
			// report decorations
			var changed = self.reportDecorations("bounds");
			// decorations may request for a repaint
			if (changed)
				mPanelChanged = true;
			// new World
			newWorld = true;
		}
		
		// if new world, then report elements
		if(newWorld) {			
			for (var i = 0, n = mElements.length; i < n; i++)
				mElements[i].setMustProject(true);
		}
	};

	/***
	 * Report event to decoration elements
	 * @method reportDecorations
	 * @param event
	 */
    self.reportDecorations = function(event) {
		var changed = false, i, n;
		// bottom group
		for (i = 0, n = mBottomDecorations.length; i < n; i++) {
			changed |= (mBottomDecorations[i].panelChangeListener && mBottomDecorations[i].panelChangeListener("bounds"));
		}
		// top group
		for (i = 0, n = mTopDecorations.length; i < n; i++) {
			if (mTopDecorations[i].panelChangeListener)
				mTopDecorations[i].panelChangeListener("bounds");
		}
		return changed;    	
    }

	/***
	 * Recalculate world dimensions using measure elements
	 * @method checkMeasure
	 * @return boolean whether dimesions update
	 */
	self.checkMeasure = function() {
		// init measured dimensions 
		var xminMeasured = Number.MAX_VALUE;
		var xmaxMeasured = -Number.MAX_VALUE;
		var yminMeasured = Number.MAX_VALUE;
		var ymaxMeasured = -Number.MAX_VALUE;

		// if autoscale, get measured dimensions		
		if(self.getAutoScaleY() || self.getAutoScaleX()) {							
			for (var i = 0, n = mElements.length; i < n; i++) {
				// considering measured elements (no groups) 
				if (mElements[i].isMeasured() && !mElements[i].isGroup() && 
					(mElements[i].getGroup() == null || mElements[i].getGroup().isMeasured())) {					
					if(mElements[i].isPixelSize()) { // for PixelSize, only the position
						var pos = mElements[i].getPosition();					 
						xminMeasured = Math.min(xminMeasured, pos[0]);
						xmaxMeasured = Math.max(xmaxMeasured, pos[0]);
						yminMeasured = Math.min(yminMeasured, pos[1]);
						ymaxMeasured = Math.max(ymaxMeasured, pos[1]);							
					} else { // get bounds
						var bounds = mElements[i].getAbsoluteBounds();					 
						xminMeasured = Math.min(xminMeasured, bounds.left);
						xmaxMeasured = Math.max(xmaxMeasured, bounds.right);
						yminMeasured = Math.min(yminMeasured, bounds.bottom);
						ymaxMeasured = Math.max(ymaxMeasured, bounds.top);
					}
				}
			}
		}
				
		// update meausured dimensions
		var updatedX = false;
		if (self.getAutoScaleX())  { // auto-scale X
			// measured dimensions bigger than preferred dimensions  
			if(!isNaN(mWorld.xminPreferred)) xminMeasured = Math.min(xminMeasured,mWorld.xminPreferred);
			if(!isNaN(mWorld.xmaxPreferred)) xmaxMeasured = Math.max(xmaxMeasured,mWorld.xmaxPreferred);
			// measured dimensions always positive  
      		if (xmaxMeasured <= xminMeasured) {
				// measuared dimensions not valid values for X
				updatedX = ((mWorld.xminMeasured != xmindef) || (mWorld.xmaxMeasured != xmaxdef));
				mWorld.xminMeasured = xmindef;
				mWorld.xmaxMeasured = xmaxdef;							
      		} else {
				// update measured dimensions
				updatedX = ((mWorld.xminMeasured != xminMeasured) || (mWorld.xmaxMeasured != xmaxMeasured)); 
				mWorld.xminMeasured = xminMeasured;
				mWorld.xmaxMeasured = xmaxMeasured;
			}			
		} else if (!isNaN(mWorld.xminPreferred) && !isNaN(mWorld.xmaxPreferred)) { // preferred X
			// update measured dimensions 
			updatedX = ((mWorld.xminMeasured != mWorld.xminPreferred) || (mWorld.xmaxMeasured != mWorld.xmaxPreferred));
			mWorld.xminMeasured = mWorld.xminPreferred;
			mWorld.xmaxMeasured = mWorld.xmaxPreferred;			
		} else { // no values for X
			updatedX = ((mWorld.xminMeasured != xmindef) || (mWorld.xmaxMeasured != xmaxdef));
			mWorld.xminMeasured = xmindef;
			mWorld.xmaxMeasured = xmaxdef;													
		}
		
		var updatedY = false;
		if (self.getAutoScaleY())  { // auto-scale Y
			// measured dimensions bigger than preferred dimensions  
			if(!isNaN(mWorld.yminPreferred)) yminMeasured = Math.min(yminMeasured,mWorld.yminPreferred);
			if(!isNaN(mWorld.ymaxPreferred)) ymaxMeasured = Math.max(ymaxMeasured,mWorld.ymaxPreferred);
			// measured dimensions always positive  
      		if (ymaxMeasured <= yminMeasured) {
				// measuared dimensions not valid values for Y
				updatedY = ((mWorld.yminMeasured != ymindef) || (mWorld.ymaxMeasured != ymaxdef));
				mWorld.yminMeasured = ymindef;
				mWorld.ymaxMeasured = ymaxdef;							
      		} else {
				// update measured dimensions
				updatedY = ((mWorld.yminMeasured != yminMeasured) || (mWorld.ymaxMeasured != ymaxMeasured)); 
				mWorld.yminMeasured = yminMeasured;
				mWorld.ymaxMeasured = ymaxMeasured;
			}			
		} else if (!isNaN(mWorld.yminPreferred) && !isNaN(mWorld.ymaxPreferred)) { // preferred Y
			// update measured dimensions 
			updatedY = ((mWorld.yminMeasured != mWorld.yminPreferred) || (mWorld.ymaxMeasured != mWorld.ymaxPreferred));
			mWorld.yminMeasured = mWorld.yminPreferred;
			mWorld.ymaxMeasured = mWorld.ymaxPreferred;			
		} else { // no values for Y
			updatedY = ((mWorld.yminMeasured != ymindef) || (mWorld.ymaxMeasured != ymaxdef));
			mWorld.yminMeasured = ymindef;
			mWorld.ymaxMeasured = ymaxdef;													
		}
		
		return (updatedX || updatedY);
	};
	
	// ----------------------------------------
	// Drawing functions
	// ----------------------------------------

	/***
	 * Reset the scene
	 * @method reset
	 */
	self.reset = function() {
		mGraphics.reset();
	};

	/***
	 * Disable rendering
	 * @method disable
	 */
	self.disable = function() {
		mEnabledRedering = false;
	}
	
	/***
	 * Enable rendering
	 * @method enable
	 */
	self.enable = function() {
		mEnabledRedering = true;
	}	
	
	/***
	 * Render the scene
	 * @method render
	 */
	self.render = function() {
		if(mEnabledRedering) {
			var reseted = false;
			if (mGraphics.setImageData || mElementsChanged) { 
				// whether canvas or elements added or removed, reset the scene
				mGraphics.reset();
				reseted = true;
				mElementsChanged = false;
			}
	
	        // check for data collection
			for (var i = 0, n = mCollectersList.length; i < n; i++)
				mCollectersList[i].dataCollected();
				
			// get measured dimensions
			var measuredWorld = self.checkMeasure();
	
			if (mMustScale || measuredWorld) // recompute scales
				self.recomputeScales();
	
			if (mPanelChanged || reseted) {// whether panel changed or reseted
				// draw self
				mGraphics.drawPanel(self);
			}
	
			// draw the bottom decorations
			mGraphics.draw(mBottomDecorations);
	
			// draw visible elements
			mGraphics.draw(mElements, reseted);
	
			if (mPanelChanged || reseted) {// whether panel changed or reseted
				// draw the gutter region
				mGraphics.drawGutters(self);
			}
	
			// draw the top decorations
			mGraphics.draw(mTopDecorations);
	
			// set changed to false
			mPanelChanged = false;
			for (var i = 0, n = mElements.length; i < n; i++)
				mElements[i].setChanged(false);
		}
	};

	// ----------------------------------------
	// Interaction
	// ----------------------------------------

	/***
	 * Return Panel Interaction
	 * @method getPanelInteraction
	 * @param panel interaction
	 */
	self.getPanelInteraction = function() {
		return mInteraction;
	};

	/***
	 * Returns the controller object
	 * @method getController
	 * @return Controller
	 */
	self.getController = function() {
		return mController;
	};

	/***
	 * Set the controller
	 * @method setController
	 * @param Controller
	 */
	self.setController = function(controller) {
		mController = controller;
	};

	// ----------------------------------------------------
	// Properties
	// ----------------------------------------------------

	self.getOnMoveHandler = null;

	self.getOnZoomHandler = function() {
		if (mEnabledZooming) {
			var delta = mInteraction.getInteractionZoomDelta();
			var bounds = self.getWorldCoordinates();
			// if not world coordinates, then get real world coordinates
			if(isNaN(bounds[0]) || isNaN(bounds[1]) || isNaN(bounds[2]) || isNaN(bounds[3]))
				bounds = self.getMeasuredCoordinates();
			var incw = +((1 - mZoomRate) * (bounds[1]-bounds[0]) * delta / 2);
			var inch = +((1 - mZoomRate) * (bounds[3]-bounds[2]) * delta / 2);
			// check limits
			var neww = Math.abs(bounds[0] - bounds[1]) -  2*incw;
			if ( neww > mZoomLimits[0] && neww < mZoomLimits[1]) {				
				// set new world coordinates
				self.setWorldCoordinates([
					bounds[0] + incw, bounds[1] - incw,
					bounds[2] + inch, bounds[3] - inch			 
				]);			
				self.getController().propertiesChanged("MinimumX","MaximumX","MinimumY","MaximumY","Bounds");
			}
		}
	}

	self.getOnPressHandler = function() {
		// info text decoration
		if (mShowCoordinates) {
			var point = mInteraction.getInteractionPoint();
			var bounds = self.getRealWorldCoordinates();
			if ((bounds[0] < point[0]) && (bounds[1] > point[0]) &&
				(bounds[3] > point[1]) && (bounds[2] < point[1])) { // whether event location over panel
				mCoorDecoration.setText(point);
				mCoorDecoration.setVisible(true);
				mCoorDecoration.getController().propertiesChanged("Visible", "Text");	
			}
		}		
	};

	self.getOnDragHandler = function() {
		// info text decoration
		if (mShowCoordinates) {
			var point = mInteraction.getInteractionPoint();
			var bounds = self.getRealWorldCoordinates();
			if ((bounds[0] < point[0]) && (bounds[1] > point[0]) &&
				(bounds[3] > point[1]) && (bounds[2] < point[1])) { // whether event location over panel
				mCoorDecoration.setText(point);
				mCoorDecoration.setVisible(true);
				mCoorDecoration.getController().propertiesChanged("Visible", "Text");			
			}
		}
		
		// update drag decoration
		if (mShowAreaRectangle) {
			var boundsInter = mInteraction.getInteractionBounds();
			if(boundsInter.length > 0 && mTypeScaleX == EJSS_DRAWING2D.DrawingPanel.SCALE_NUM && 
					mTypeScaleY == EJSS_DRAWING2D.DrawingPanel.SCALE_NUM) {
				mDragDecoration.setBounds(boundsInter);
				mDragDecoration.setVisible(true);
				mDragDecoration.getController().propertiesChanged("Position", "X", "Y", "Size", "SizeX", "SizeY", "Visible");
			} else {
				mDragDecoration.setVisible(false);		
				mDragDecoration.getController().propertiesChanged("Visible");			
			}
		}
		
		// drag drawing
		if (mEnabledDragging != 0 &&  mEnabledDragging != 4) {
			var distance = mInteraction.getInteractionDistance();
			if(distance.length > 0 && mTypeScaleX == EJSS_DRAWING2D.DrawingPanel.SCALE_NUM && 
					mTypeScaleY == EJSS_DRAWING2D.DrawingPanel.SCALE_NUM) {
				var bounds = self.getWorldCoordinates();
				// if not world coordinates, then get measured coordinates
				if(isNaN(bounds[0]) || isNaN(bounds[1]) || isNaN(bounds[2]) || isNaN(bounds[3]))
					bounds = self.getMeasuredCoordinates();				
				if (mEnabledDragging == 1) {
					self.setWorldCoordinates([ bounds[0] + distance[0], bounds[1] + distance[0],
						bounds[2] + distance[1], bounds[3] + distance[1]]); 					
				} else if (mEnabledDragging == 2) {
					self.setWorldCoordinates([bounds[0] + distance[0],bounds[1] + distance[0], bounds[2], bounds[3]]); 
				} else {
					self.setWorldCoordinates([bounds[0], bounds[1], bounds[2] + distance[1], bounds[3] + distance[1]]); 
				}
				self.getController().propertiesChanged("MinimumX","MaximumX","MinimumY","MaximumY","Bounds");
			}					
		}
	};

	self.getOnReleaseHandler = function() {
		// info text decoration
		if (mShowCoordinates) {
			mCoorDecoration.setVisible(false);
			mCoorDecoration.getController().propertiesChanged("Visible");
		}
				
		// hidden drag decoration
		if (mShowAreaRectangle) {
			mDragDecoration.setVisible(false);		
			mDragDecoration.getController().propertiesChanged("Visible");
		}
	};

	self.getMessageDecoration = function(whichOne) {
		switch(whichOne) {
			case "TL" :
				return mTLMessageDecoration;
			case "BR" :
				return mBRMessageDecoration;
			case "BL" :
				return mBLMessageDecoration;
		}
		return mTRMessageDecoration;
	};

	self.getCoordinates = function() {
		return mCoorDecoration;
	};

	self.getInteraction = function() {
		return mInteraction;
	};

	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.DrawingPanel.registerProperties(self, controller);
	};

  	/***
   	* Get JSON object with private variables
   	* @method serialize
   	* @visibility private
   	*/
  	self.serialize = function() {
  	  return { 
		mStyle: mStyle.serialize(),
		
		mTopDecorations: mTopDecorations,
		mBottomDecorations: mBottomDecorations,
		mElements: mElements,
		mCollectersList: mCollectersList,
		
		mAutoScaleX: mAutoScaleX, mAutoScaleY: mAutoScaleY,
		mInvertedScaleY: mInvertedScaleY, mTypeScaleX: mTypeScaleX,
		mTypeScaleY: mTypeScaleY, mShowCoordinates: mShowCoordinates, 
		
		xmindef: xmindef, xmaxdef: xmaxdef, ymindef: ymindef, ymaxdef: ymaxdef,
		
		mWorld: mWorld,		
		mGutters: mGutters,
		mGuttersStyle: mGuttersStyle.serialize(),
				
		// mInteraction: mInteraction.serialize() 
	  };
  	}
  
  	/***
   	* Set JSON object with private variables
   	* @method unserialize
   	* @parem json JSON object
   	* @visibility private
   	*/
  	self.unserialize = function(json) {
		mStyle.unserialize(json.mStyle), 

		// not support references changing
		// mBottomDecorations = json.mBottomDecorations,
		// mTopDecorations = json.mTopDecorations,
		// mElements = json.mElements,
		// mCollectersList = json.mCollectersList,
		
		mAutoScaleX = json.mAutoScaleX, mAutoScaleY = json.mAutoScaleY,
		mInvertedScaleY = json.mInvertedScaleY, mTypeScaleX = json.mTypeScaleX,
		mTypeScaleY = json.mTypeScaleY, mShowCoordinates = json.mShowCoordinates, 
		
		xmindef = json.xmindef, xmaxdef = json.xmaxdef, ymindef = json.ymindef, ymaxdef = json.ymaxdef,
		
		mWorld = json.mWorld,		
		mGutters = json.mGutters,
		mGuttersStyle.unserialize(json.mGuttersStyle),
		
		// mInteraction = json.mInteraction 
	
		mPanelChanged = true;
  	}

	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	// set graphics mode
	self.setGraphicsMode(mGraphicsMode);

	// style
	mStyle.setLineColor('black');
	mStyle.setFillColor('rgb(239,239,255)');
	mStyle.setChangeListener(function(change) {
		mPanelChanged = true;
	});

	mGuttersStyle.setChangeListener(function(reporter) {
		mPanelChanged = true;
	});

	var mDragDecoration = EJSS_DRAWING2D.shape(mName + "__cursorBox__");
	mDragDecoration.setShapeType(EJSS_DRAWING2D.Shape.RECTANGLE);
	mDragDecoration.getStyle().setDrawFill(false);
	mDragDecoration.getStyle().setLineColor("black");
	mDragDecoration.setRelativePosition("CENTER");
	mDragDecoration.setVisible(false);
	self.addDecoration(mDragDecoration, 0, true);

	var mTLMessageDecoration = EJSS_DRAWING2D.text(mName + "__tlmessage__");
	mTLMessageDecoration.setRelativePosition("NORTH_WEST");
	mTLMessageDecoration.getStyle().setDrawLines(true);
	mTLMessageDecoration.getStyle().setDrawFill(true);
	mTLMessageDecoration.getStyle().setFillColor("yellow");
	mTLMessageDecoration.getStyle().setLineColor("black");
	mTLMessageDecoration.getFont().setFontSize(12);
	mTLMessageDecoration.setPosition([mWorld.xmin, mWorld.ymax]);
	mTLMessageDecoration.setFramed(true);
	mTLMessageDecoration.panelChangeListener = function(event) {
		if (event == "bounds") {
			if(self.getInvertedScaleY())
				mTLMessageDecoration.setPosition([mWorld.xmin, mWorld.ymin]);
			else
				mTLMessageDecoration.setPosition([mWorld.xmin, mWorld.ymax]);	
		}			
	};
	self.addDecoration(mTLMessageDecoration, 0, true);

	var mBLMessageDecoration = EJSS_DRAWING2D.text(mName + "__blmessage__");
	mBLMessageDecoration.setRelativePosition("SOUTH_WEST");
	mBLMessageDecoration.getStyle().setDrawLines(true);
	mBLMessageDecoration.getStyle().setDrawFill(true);
	mBLMessageDecoration.getStyle().setFillColor("yellow");
	mBLMessageDecoration.getStyle().setLineColor("black");
	mBLMessageDecoration.getFont().setFontSize(12);
	mBLMessageDecoration.setPosition([mWorld.xmin, mWorld.ymin]);
	mBLMessageDecoration.setFramed(true);
	mBLMessageDecoration.panelChangeListener = function(event) {
		if (event == "bounds")
			if(self.getInvertedScaleY())
				mBLMessageDecoration.setPosition([mWorld.xmin, mWorld.ymax]);
			else		
				mBLMessageDecoration.setPosition([mWorld.xmin, mWorld.ymin]);
	};
	self.addDecoration(mBLMessageDecoration, 0, true);

	var mBRMessageDecoration = EJSS_DRAWING2D.text(mName + "__brmessage__");
	mBRMessageDecoration.setRelativePosition("SOUTH_EAST");
	mBRMessageDecoration.getStyle().setDrawLines(true);
	mBRMessageDecoration.getStyle().setDrawFill(true);
	mBRMessageDecoration.getStyle().setFillColor("yellow");
	mBRMessageDecoration.getStyle().setLineColor("black");
	mBRMessageDecoration.getFont().setFontSize(12);
	mBRMessageDecoration.setPosition([mWorld.xmax, mWorld.ymin]);
	mBRMessageDecoration.setFramed(true);
	mBRMessageDecoration.panelChangeListener = function(event) {
		if (event == "bounds")
			if(self.getInvertedScaleY())
				mBRMessageDecoration.setPosition([mWorld.xmax, mWorld.ymax]);
			else
				mBRMessageDecoration.setPosition([mWorld.xmax, mWorld.ymin]);
	};
	self.addDecoration(mBRMessageDecoration, 0, true);

	var mTRMessageDecoration = EJSS_DRAWING2D.text(mName + "__trmessage__");
	mTRMessageDecoration.setRelativePosition("NORTH_EAST");
	mTRMessageDecoration.getStyle().setDrawLines(true);
	mTRMessageDecoration.getStyle().setDrawFill(true);
	mTRMessageDecoration.getStyle().setFillColor("yellow");
	mTRMessageDecoration.getStyle().setLineColor("black");
	mTRMessageDecoration.getFont().setFontSize(12);
	mTRMessageDecoration.setPosition([mWorld.xmax, mWorld.ymax]);
	mTRMessageDecoration.setFramed(true);
	mTRMessageDecoration.panelChangeListener = function(event) {
		if (event == "bounds")
			if(self.getInvertedScaleY())
				mTRMessageDecoration.setPosition([mWorld.xmax, mWorld.ymin]);
			else
				mTRMessageDecoration.setPosition([mWorld.xmax, mWorld.ymax]);
	};
	self.addDecoration(mTRMessageDecoration, 0, true);

	var mCoorDecoration = EJSS_DRAWING2D.infoText(mName + "__coor__");
	mCoorDecoration.setRelativePosition("SOUTH_WEST");
	mCoorDecoration.getFont().setFontSize(12);
	mCoorDecoration.getStyle().setFillColor("yellow");
	mCoorDecoration.setPosition([mWorld.xmin, mWorld.ymin]);
	mCoorDecoration.setFormat("x:0.##,y:0.##");
	mCoorDecoration.setFramed(true);
	mCoorDecoration.setVisible(false);
	mCoorDecoration.panelChangeListener = function(event) {
		if (event == "bounds")
			if(self.getInvertedScaleY())
				mCoorDecoration.setPosition([mWorld.xmin, mWorld.ymax]);
			else		
				mCoorDecoration.setPosition([mWorld.xmin, mWorld.ymin]);
	};
	self.addDecoration(mCoorDecoration, 0, true);

	self.getGuttersStyle().setLineColor('black');
	self.getGuttersStyle().setFillColor('rgb(239,239,255)');
	self.getGuttersStyle().setShapeRendering("RENDER_CRISPEDGES");

	// interactions handler
	var mInteraction = EJSS_DRAWING2D.panelInteraction(self);	

//	mGraphics.getEventContext().addEventListener("dblclick", function(event) { event.preventDefault(); event.stopPropagation(); return false; } ); 
    mGraphics.getEventContext().addEventListener("dblclick", function(event){ event.preventDefault(); event.stopPropagation(); return false; }, true);
//	window.document.addEventListener("dblclick", function(event) { event.preventDefault(); event.stopPropagation(); return false; } ); 

	return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * Basic Element 2D 
 * @class EJSS_DRAWING2D.Element
 */
EJSS_DRAWING2D.Element = {
	CENTER   : 0,
	NORTH    : 1,
	SOUTH    : 2,
	EAST     : 3,
	WEST     : 4,
	NORTH_EAST  :  5,
	NORTH_WEST  :  6,
	SOUTH_EAST  :  7,
	SOUTH_WEST  :  8,
        
    /**
     * Registers properties in a ControlElement
     */
    registerProperties : function(element,controller) {
      var TARGET_POSITION = EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION;
      var TARGET_SIZE = EJSS_DRAWING2D.PanelInteraction.TARGET_SIZE;
      var style = element.getStyle();
                        
      element.setController(controller);

	 /*** 
	  * Parent of the element
	  * @property Parent 
	  * @type Panel|Group
	  */  
      controller.registerProperty("Parent",element.setParent,element.getParent); 

	 /*** 
	  * Position in X
	  * @property X 
	  * @type double
	  * @default 0
	  */  
      controller.registerProperty("X",element.setX,element.getX);
      
	 /*** 
	  * Position in Y
	  * @property Y 
	  * @type double
	  * @default 0
	  */        
      controller.registerProperty("Y",element.setY,element.getY);
      
	 /*** 
	  * Coordinates X and Y
	  * @property Position 
	  * @type double[2]
	  * @default [0,0]
	  */        
      controller.registerProperty("Position",element.setPosition,element.getPosition);
      
	 /*** 
	  * Whether the position is in pixels
	  * @property PixelPosition 
	  * @type boolean
	  * @default false
	  */              
      controller.registerProperty("PixelPosition",element.setPixelPosition,element.getPixelPosition);
      
	 /*** 
	  * Diameter, i.e. the value of the width and the height 
	  * @property Diameter 
	  * @type double
	  */                    
      controller.registerProperty("Diameter",function(diameter) {
      	element.setSize([diameter,diameter]);
      });
      
	 /*** 
	  * Radius, i.e. the half of the diameter 
	  * @property Radius 
	  * @type double
	  */                    
      controller.registerProperty("Radius",function(radius) {
      	element.setSize([radius*2,radius*2]);
      });
      
	 /*** 
	  * Size along the X axis
	  * @property SizeX 
	  * @type double
	  * @default 1
	  */                          
      controller.registerProperty("SizeX",element.setSizeX,element.getSizeX);

	 /*** 
	  * Size along the Y axis
	  * @property SizeY 
	  * @type double
	  * @default 1
	  */                          
      controller.registerProperty("SizeY",element.setSizeY,element.getSizeY);

	 /*** 
	  * Size along the X and Y axes 
	  * @property Size 
	  * @type double[2]
	  * @default [1,1]
	  */                          
      controller.registerProperty("Size",element.setSize,element.getSize);
            
	 /*** 
	  * Whether the size is in pixels
	  * @property PixelSize 
	  * @type boolean
	  * @default false
	  */                          
      controller.registerProperty("PixelSize",element.setPixelSize,element.getPixelSize);
	  
	 /*** 
	  * Position of the bounding box of the element
	  * @property Bounds 
	  * @type Object{left,rigth,top,bottom}
	  */                          
      controller.registerProperty("Bounds",element.setBounds,element.getBounds);

	 /*** 
	  * Position of the coordinates X and Y relative to the element 
	  * @property RelativePosition 
	  * @type string|int
	  * @values "CENTER":0,"NORTH":1,"SOUTH":2,"EAST":3,"WEST":4,"NORTH_EAST":5,"NORTH_WEST":6,
	  * "SOUTH_EAST":7,"SOUTH_WEST":8 
	  * @default "CENTER"
	  */                          
      controller.registerProperty("RelativePosition",element.setRelativePosition,element.getRelativePosition);

	 /*** 
	  * Whether the element is visible
	  * @property Visibility 
	  * @type boolean
	  * @default true
	  */                          
      controller.registerProperty("Visibility",element.setVisible,element.isVisible);
            
	 /*** 
	  * Measurability of the element
	  * @property Measured 
	  * @type boolean
	  * @default true
	  */                          
      controller.registerProperty("Measured",element.setMeasured,element.isMeasured);

	 /*** 
	  * Internal transformation of the element
	  * @property Transformation 
	  * @type int|float[6]|string
	  * @values int:angle in radians to rotate the element with origin its position, 
	  * float[9]: transformation matrix - [scaleX, skewY, skewX, scaleY, translateX, translateY]
	  * @see http://www.w3.org/TR/SVG-Transforms/
	  * @default 0
	  */                          
      controller.registerProperty("Transformation",element.setTransformation);

	 /*** 
	  * Rotation of the element
	  * @property Rotate 
	  * @type int
	  * @values int:angle in radians to rotate the element with origin its position 
	  * @see http://www.w3.org/TR/SVG-Transforms/
	  * @default 0
	  */                          
      controller.registerProperty("Rotate",element.setRotate);

	 /*** 
	  * Scale transformation of the element in X
	  * @property ScaleX 
	  * @type float
	  * @values float: scale in X 
	  * @see http://www.w3.org/TR/SVG-Transforms/
	  * @default 1
	  */                          
      controller.registerProperty("ScaleX",element.setScaleX);

	 /*** 
	  * Scale transformation of the element in Y
	  * @property ScaleY 
	  * @type float
	  * @values float: scale in Y 
	  * @see http://www.w3.org/TR/SVG-Transforms/
	  * @default 1
	  */                          
      controller.registerProperty("ScaleY",element.setScaleY);

	 /*** 
	  * Skew transformation of the element in X
	  * @property SkewX 
	  * @type float
	  * @values float: skew angle in X 
	  * @see http://www.w3.org/TR/SVG-Transforms/
	  * @default 1
	  */                          
      controller.registerProperty("SkewX",element.setSkewX);

	 /*** 
	  * Skew transformation of the element in Y
	  * @property SkewY 
	  * @type float
	  * @values float: skew angle in Y 
	  * @see http://www.w3.org/TR/SVG-Transforms/
	  * @default 1
	  */                          
      controller.registerProperty("SkewY",element.setSkewY);
  
	 /*** 
	  * Stroke color
	  * @property LineColor 
	  * @type string
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "Black"
	  */                          
      controller.registerProperty("LineColor",style.setLineColor,style.getLineColor);

	 /*** 
	  * Stroke width
	  * @property LineWidth 
	  * @type double
	  * @default 0.5
	  */                          
      controller.registerProperty("LineWidth",style.setLineWidth,style.getLineWidth);
      
	 /*** 
	  * Whether the stroke is drawed
	  * @property DrawLines 
	  * @type boolean
	  * @default true
	  */                          
      controller.registerProperty("DrawLines",style.setDrawLines,style.getDrawLines);

	 /*** 
	  * Fill color
	  * @property FillColor 
	  * @type string
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "none"
	  */                          
      controller.registerProperty("FillColor",style.setFillColor,style.getFillColor);
      
	 /*** 
	  * Whether the fill color is drawed
	  * @property DrawFill 
	  * @type boolean
	  * @default true
	  */                          
      controller.registerProperty("DrawFill",style.setDrawFill,style.getDrawFill);

	 /*** 
	  * SVG shape rendering 
	  * @property ShapeRendering 
	  * @type string
	  * @values "auto","optimizeSpeed","crispEdges","geometricPrecision"  
	  * @default "auto"
	  */                          
      controller.registerProperty("ShapeRendering",style.setShapeRendering,style.getShapeRendering);
      
	 /*** 
	  * Inline SVG attributtes  
	  * @property Attributes 
	  * @type object
	  * @values {nameAttr1:value1,nameAttr2:value2,..}
	  */                                
      controller.registerProperty("Attributes",style.setAttributes,style.getAttributes);

	 /*** 
	  * Whether the user could change the position   
	  * @property EnabledPosition 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("EnabledPosition",function(enabled) {
        element.getInteractionTarget(TARGET_POSITION).setMotionEnabled(enabled);
      });

	 /*** 
	  * Whether the group position also changes when the element position changes    
	  * @property MovesGroup 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("MovesGroup", function(affects) {
        element.getInteractionTarget(TARGET_POSITION).setAffectsGroup(affects);
      });

	 /*** 
	  * Whether the user could change the size   
	  * @property EnabledSize 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("EnabledSize",function(enabled) {
        element.getInteractionTarget(TARGET_SIZE).setMotionEnabled(enabled);
      });

	 /*** 
	  * Whether the group size also changes when the element size changes    
	  * @property ResizesGroup 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("ResizesGroup", function(affects) {
        element.getInteractionTarget(TARGET_SIZE).setAffectsGroup(affects);
      });
      
	 /*** 
	  * Sensitivity in pixels when the user touchs the element    
	  * @property Sensitivity 
	  * @type int
	  * @default 20
	  */                                      
      controller.registerProperty("Sensitivity", function (sense) {
        element.getInteractionTarget(TARGET_POSITION).setSensitivity(sense);
        element.getInteractionTarget(TARGET_SIZE).setSensitivity(sense);
      });

	 /*** 
	  * Event when double click     
	  * @action OnDoubleClick 
	  */                                      
      controller.registerAction("OnDoubleClick", element.getOnDoubleClickInformation);      
	 /*** 
	  * Event when the mouse enters in the element     
	  * @action OnEnter 
	  */                                      
      controller.registerAction("OnEnter",element.getOnEnterInformation);

	 /*** 
	  * Event when the mouse exits the element     
	  * @action OnExit 
	  */                                      
      controller.registerAction("OnExit",element.getOnExitInformation);

	 /*** 
	  * Event when the mouse clicks over the element     
	  * @action OnPress 
	  */                                      
      controller.registerAction("OnPress",element.getOnPressInformation);

	 /*** 
	  * Event when the mouse drags the element     
	  * @action OnDrag 
	  */                                      
      controller.registerAction("OnDrag",element.getOnDragInformation);

	 /*** 
	  * Event when the mouse release the element     
	  * @action OnRelease 
	  */                                      
      controller.registerAction("OnRelease",element.getOnReleaseInformation);
    },
    
    /**
     * Copies one element into another
     */
    copyTo : function(source, dest) {
      var InteractionTarget = EJSS_DRAWING2D.InteractionTarget;
      var TARGET_POSITION = EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION;
      var TARGET_SIZE = EJSS_DRAWING2D.PanelInteraction.TARGET_SIZE;
      
      EJSS_DRAWING2D.Style.copyTo(source.getStyle(),dest.getStyle());
      
      dest.setVisible(source.isVisible());
      dest.setMeasured(source.isMeasured());

      dest.setPosition(source.getPosition());
      dest.setPixelPosition(source.isPixelPosition());
      dest.setSize(source.getSize());
      dest.setPixelSize(source.isPixelSize());
      dest.setTransformation(source.getTransformation());
      dest.setRotate(source.getRotate());

	  dest.setRelativePosition(source.getRelativePosition());

      InteractionTarget.copyTo(source.getInteractionTarget(TARGET_POSITION),dest.getInteractionTarget(TARGET_POSITION));
      InteractionTarget.copyTo(source.getInteractionTarget(TARGET_SIZE),dest.getInteractionTarget(TARGET_SIZE));
      
      dest.setParent(source.getParent(), source);
    },
    
  	/**
   	* Returns offsets of the element with respect to its center.  
   	* @method getRelativePositionOffset
   	* @param position relative position 
   	* @param sx X size
   	* @param sy Y size
   	* @return [x,y]
   	*/
  	getRelativePositionOffset : function(position, sx, sy, invertedy) {
  		var dx = 0;
  		var dy = 0;
	    switch(position) {
	      default : 
	      case EJSS_DRAWING2D.Element.CENTER    : dx = 0; 		  dy = 0; 		break;
	      case EJSS_DRAWING2D.Element.NORTH     : dx = 0; 		  dy = -(sy/2); break;
	      case EJSS_DRAWING2D.Element.SOUTH     : dx = 0; 		  dy = (sy/2);  break;
	      case EJSS_DRAWING2D.Element.EAST      : dx = -(sx/2);    dy = 0; 		break;
	      case EJSS_DRAWING2D.Element.WEST      : dx = (sx/2);     dy = 0; 		break;
	      case EJSS_DRAWING2D.Element.NORTH_EAST: dx = -(sx/2);    dy = -(sy/2); break;
	      case EJSS_DRAWING2D.Element.NORTH_WEST: dx = (sx/2);     dy = -(sy/2); break;
	      case EJSS_DRAWING2D.Element.SOUTH_EAST: dx = -(sx/2);    dy = (sy/2);  break;
	      case EJSS_DRAWING2D.Element.SOUTH_WEST: dx = (sx/2);     dy = (sy/2);  break;
	    }
	    if(typeof invertedy != 'undefined' && invertedy) dy = -dy;
		return [dx,dy]; 
	},

  	/**
   	* Returns offsets of the element with respect to its south-west.  
   	* @method getSWRelativePositionOffset
   	* @param position relative position 
   	* @param sx X size
   	* @param sy Y size
   	* @return [x,y]
   	*/
  	getSWRelativePositionOffset : function(position, sx, sy, invertedy) {
  		var dx = 0;
  		var dy = 0;
	   	switch(position) {
	      default : 
	      case EJSS_DRAWING2D.Element.CENTER    : dx = -(sx/2); 	dy = (sy/2); break;
	      case EJSS_DRAWING2D.Element.NORTH     : dx = -(sx/2); 	dy = sy; 	break;
	      case EJSS_DRAWING2D.Element.SOUTH     : dx = -(sx/2); 	dy = 0;  	break;
	      case EJSS_DRAWING2D.Element.EAST      : dx = -sx;    	dy = (sy/2); break;
	      case EJSS_DRAWING2D.Element.WEST      : dx = 0;     	dy = (sy/2); break;
	      case EJSS_DRAWING2D.Element.NORTH_EAST: dx = -sx;    	dy = sy; 	break;
	      case EJSS_DRAWING2D.Element.NORTH_WEST: dx = 0;     	dy = sy; 	break;
	      case EJSS_DRAWING2D.Element.SOUTH_EAST: dx = -sx;    	dy = 0;  	break;
	      case EJSS_DRAWING2D.Element.SOUTH_WEST: dx = 0;     	dy = 0;  	break;
	    }
	    if(typeof invertedy != 'undefined' && invertedy) dy = -dy;	    
		return [dx,dy]; 
  	}
};

/**
 * Constructor for Element
 * @method element
 * @param mName string
 * @returns An abstract 2D element
 */
EJSS_DRAWING2D.element = function(mName) {
  var self = {};							// reference returned 

  // Static references
  var Element = EJSS_DRAWING2D.Element;		// reference for Element
  var PanelInteraction = EJSS_DRAWING2D.PanelInteraction;	// reference for PanelInteraction

  // Instance variables
  var mStyle = EJSS_DRAWING2D.style(mName);	// style for element 
  var mVisible = true;						// whether visible in drawing
  var mMeasured = true;						// whether measure for element
  var mRelativePosition = Element.CENTER;	// relative position

  // Position and size
  var mX = 0;						// position X
  var mY = 0;						// position Y
  var mPixelPosition = false; 		// whether the position is in pixels
  var mSizeX = 1;					// size X
  var mSizeY = 1;					// size Y
  var mPixelSize = false; 			// whether the size is in pixels
  var mTransformation = [];			// transformation for element
  var mRotate = 0;					// rotation for element

  // Interaction      
  var mInteraction = {
      positionTarget : EJSS_DRAWING2D.interactionTarget(self,PanelInteraction.TARGET_POSITION, Element.CENTER),
      sizeTarget     : EJSS_DRAWING2D.interactionTarget(self,PanelInteraction.TARGET_SIZE, Element.NORTH_EAST)
   };

  // Implementation variables    
  var mPanel = null;				// drawing panel for element
  var mGroup = null;				// group for element
  var mSet = null;				    // The set it belongs to (if any)  
  var mIndexInSet = -1;				// The index of the element in a set (if any)  

  var mProjectedPosition = [];		// projected position for element
  var mProjectedSize = [];			// projected size for element

  var mMustProject = true;			// whether drawing panel changed, then element needs to project
  var mElementChanged = true;		// whether element changed (position, size, or group)

  var mController = { 				// dummy controller object
      propertiesChanged : function() {},
      invokeAction : function() {}
  };

  var mCustomObject = null; // a placeholder to keep and object
  
  // ----------------------------------------
  // Public functions
  // ----------------------------------------
  
  /***
   * Gets name for element
   * @method getName
   * @visibility public
   * @return string
   */
  self.getName = function() {
    return mName;
  };  

  /***
   * Sets the property Parent
   * @method setParent
   * @param parent Panel or Group
   * @visibility public
   */
  self.setParent = function(parent, sibling) {
  	if(parent.render) { // is a panel  		
  		self.setGroup(null);
  		parent.addElement(self);
  		self.setPanel(parent);
  	} else if (parent.getClass() == "ElementGroup") { // is a group
  		self.setGroup(parent);
  		self.getGroupPanel().addElement(self, sibling);
  		self.setPanel(self.getGroupPanel());
  	} else { //
  		console.log("WARNING: setParent() - Parent not valid : "+ parent.getName()); 
  	}
  };

  /***
   * Gets the property Parent
   * @method getParent
   * @return Panel|Group
   * @visibility public
   */
  self.getParent = function() {
  	var parent;
  	if(mGroup !== null) parent = self.getGroup();
  	else parent = self.getPanel();
  	return parent  
  };

  /***
   * Is the element a group  
   * @method isGroup
   * @return boolean
   * @visibility public
   */
  self.isGroup = function() {
  	return false;
  }

  /***
   * Return the style associated to the element
   * @method getStyle
   * @return Style 
   * @visibility public
   */
  self.getStyle = function() { 
    return mStyle; 
  };

  /***
   * Store an object for internal use
   * @method setCustomObject
   * @visibility protected
   */
  self.setCustomObject = function(object) { mCustomObject = object; }

  /***
   * Retrieve an object for internal use
   * @method getCustomObject
   * @return Object
   * @visibility protected
   */
  self.getCustomObject = function() { return mCustomObject; }

  // ----------------------------------------
  // Position of the element
  // ----------------------------------------

  /***
   * Sets the property X
   * @method setX(x)
   * @param x double
   * @visibility public
   */
  self.setX = function(x, dummy) { 
    if (mX!=x) { 
      mX = x; 
      mElementChanged = true; 
    } 
  };

  /***
   * Gets the property X
   * @method getX()
   * @return double
   * @visibility public
   */
  self.getX = function() { 
    return mX; 
  };

  /***
   * Sets the property Y
   * @method setY(y)
   * @param y double
   * @visibility public
   */
  self.setY = function(y) {  
    if (mY!=y) { 
      mY = y; 
      mElementChanged = true; 
    } 
  };

  /***
   * Gets the property Y
   * @method getY()
   * @return double
   * @visibility public
   */
  self.getY = function() { 
    return mY; 
  };

  /***
   * Sets the property Position
   * @method setPosition
   * @param position double[2]
   * @visibility public
   */
  self.setPosition = function(position) {
    self.setX(position[0]);
    self.setY(position[1]);
  };

  /***
   * Gets the property Position
   * @method getPosition
   * @return double[2]
   * @visibility public
   */
  self.getPosition = function() { 
    return [mX, mY]; 
  };
  
  /***
   * Sets the property PixelPosition
   * @method setPixelPosition
   * @param pixel boolean
   * @visibility public
   */
  self.setPixelPosition = function(pixel) {
    if (mPixelPosition!=pixel) { 
      mPixelPosition = pixel;
      mElementChanged = true; 
    }
  };

  /***
   * Returns the property PixelPosition
   * @method isPixelPosition
   * @return boolean
   * @visibility public
   */
  self.isPixelPosition = function() {
    return mPixelPosition;
  };
  
  // ----------------------------------------
  // Size of the element
  // ----------------------------------------

  /***
   * Sets the property SizeX
   * @method setSizeX
   * @param sizeX double
   * @visibility public
   */
  self.setSizeX = function(sizeX) { 
    if (mSizeX!=sizeX) { 
      mSizeX = sizeX; 
      mElementChanged = true; 
    } 
  };

  /***
   * Gets the property SizeX
   * @method getSizeX
   * @return double
   * @visibility public
   */
  self.getSizeX = function() { 
    return mSizeX; 
  };

  /***
   * Sets the property SizeY
   * @method setSizeY
   * @param sizeY double
   * @visibility public
   */
  self.setSizeY = function(sizeY) { 
    if (mSizeY!=sizeY) { 
      mSizeY = sizeY; 
      mElementChanged = true; 
    }
  };

  /***
   * Gets the property SizeY
   * @method getSizeY
   * @return double
   * @visibility public
   */
  self.getSizeY = function() { 
    return mSizeY; 
  };

  /***
   * Sets the property Size
   * @method setSize
   * @param position double[2]
   * @visibility public
   */
  self.setSize = function(size) {
    self.setSizeX(size[0]);
    self.setSizeY(size[1]);
  };

  /***
   * Gets the property Size
   * @method getSize
   * @return double[]
   * @visibility public
   */
  self.getSize = function() {
    return [self.getSizeX(), self.getSizeY()];
  };
  
  /***
   * Sets the property PixelSize
   * @method setPixelSize
   * @param pixel boolean
   * @visibility public
   */
  self.setPixelSize = function(pixel) {
    if (mPixelSize!=pixel) { 
      mPixelSize = pixel;
      mElementChanged = true; 
    }
  };

  /***
   * Gets the property PixelSize
   * @method isPixelSize
   * @return boolean
   * @visibility public
   */
  self.isPixelSize = function() {
    return mPixelSize;
  };
    
  /***
   * Sets bounds for an element
   * @method setBounds
   * @param Object{left,rigth,top,bottom}|[left,rigth,top,bottom]
   * @visibility public
   */
  self.setBounds = function(bounds) {
  	var left,right,top,bottom;
  	
  	if(bounds.left) {
		left = bounds.left;
		right = bounds.right;
		top = bounds.top;
		bottom = bounds.bottom;  		
  	} else {
		left = bounds[0];
		right = bounds[1];
		top = bounds[2];
		bottom = bounds[3];  		  		
  	}
  	
	var sx = right-left;
	var sy = bottom-top;
	  
  	var d = self.getRelativePositionOffset(sx,sy);
  	var mx = sx/2, my = sy/2;  	

	var x = left + mx - d[0];
	var y = top + my - d[1];
	 
    self.setX(x);
    self.setY(y); 
    self.setSizeX(sx);
    self.setSizeY(sy); 
  };
    
  /***
   * Returns bounds for an element
   * @method getBounds
   * @return Object{left,rigth,top,bottom}
   * @visibility public
   */
  self.getBounds = function() {
  	var size = (mPixelSize? mPanel.toPanelMod([mSizeX,-mSizeY]) : [mSizeX,mSizeY]); 
  	
  	var mx = size[0]/2, my = size[1]/2;  	
  	var d = self.getRelativePositionOffset(size[0],size[1]);
	return {
		left: (mX+d[0])-mx,
		right: (mX+d[0])+mx,
		top: (mY+d[1])+my,
		bottom: (mY+d[1])-my
	}
  };

  /***
   * Returns bounds for an element (after applying groups) 
   * @method getAbsoluteBounds
   * @param withTransf whether transformation must be considered, note it will be 
   *  false (or undefinded) when transformations are applied by svg (drawing)
   * @return Object{left,rigth,top,bottom}
   * @visibility public
   */
  self.getAbsoluteBounds = function(withTransf) {
	var bounds = self.getBounds();	
	var p1 = self.toGroupSpace([bounds.left, bounds.top],withTransf);
	var p2 = self.toGroupSpace([bounds.right, bounds.bottom],withTransf);

	return {left: p1[0], top: p1[1], right: p2[0], bottom: p2[1]};
  };

  //---------------------------------
  // relative position
  //---------------------------------

  /**
   * Sets the relative position of the element with respect to its (x,y) coordinates.
   * @method setRelativePosition
   * @param position relative position
   */
  self.setRelativePosition = function(position) {
    if (typeof position == "string") position = Element[position.toUpperCase()];
    if (mRelativePosition != position) {
      mRelativePosition = position;
      mElementChanged = true; 
    }
  };
  
  /**
   * Gets the relative position of the element with respect to its (x,y) coordinates.
   * @method getRelativePosition
   * @return relative position
   */
  self.getRelativePosition = function() { 
    return mRelativePosition;
  };  

  /**
   * Returns offsets of the element with respect to its center.  
   * @method getRelativePositionOffset
   * @param s1 X size or size array
   * @param s2 Y size
   * @return [x,y]
   */
  self.getRelativePositionOffset = function(s1, s2) {
	var sx, sy;
  	if (Array.isArray(s1)) { // is array?
  		sx = s1[0];
  		sy = s1[1];
  	} else {
  		sx = s1;
  		sy = s2;
  	}  	
  	
  	var invertedScale = mPanel.getInvertedScaleY? mPanel.getInvertedScaleY():false;
  	return Element.getRelativePositionOffset(mRelativePosition, sx, sy, invertedScale);
  }

  // -------------------------------------
  // Visible and measure
  // -------------------------------------

  /***
   * Sets the property Visibility
   * @method setVisible
   * @param visible boolean
   * @visibility public
   */
  self.setVisible = function(visible) {
    if (visible!=mVisible) { 
      mVisible = visible; 
      mElementChanged = true;
    } 
  };

  /***
   * Gets the property Visibility
   * @method isVisible
   * @return boolean
   * @visibility public
   */
  self.isVisible = function() { 
    return mVisible; 
  };

  /***
   * Returns the real visibility status of the element, 
   * which will be false if it belongs to an invisible group
   * @method isGroupVisible
   * @return boolean
   * @visibility public
   */
  self.isGroupVisible = function() {
    var el = mGroup;
    while (typeof el != "undefined" && el !== null) {
      if (!el.isVisible()) return false;
      el = el.getGroup();
    }
    return mVisible;
  };

  /***
   * Sets the property Measured
   * @method setMeasured
   * @param measured boolean
   * @visibility public
   */
  self.setMeasured = function(measured) { 
    mMeasured = measured; 
  };

  /***
   * Gets the property Measured
   * @method isMeasured
   * @return boolean
   * @visibility public
   */
  self.isMeasured = function() { 
    return mMeasured; 
  };

  // ----------------------------------------------------
  // Transformations
  // ----------------------------------------------------

  /***
   * Sets the property Transformation
   * @method setTransformation
   * @param trans transformation - [scaleX, skewY, skewX, scaleY, translateX, translateY]
   * @visibility public
   */
  self.setTransformation = function(trans) {
    if (typeof trans == "undefined" || trans === null) { 
    	mTransformation = [];
    } else if (Array.isArray(trans) && trans.length>5) {  // matrix 
	    if (!EJSS_TOOLS.compareArrays(mTransformation,trans)) {
	      mTransformation = trans.slice();
	      mElementChanged = true;
	    }
    } else if (mRotate != trans) {  // rotation
		mRotate = trans;
		mElementChanged = true;    		
    }
  };

  /***
   * Gets the property Transformation
   * @method getTransformation
   * @return Transformation - [scaleX, skewY, skewX, scaleY, translateX, translateY] 
   * @visibility public
   */
  self.getTransformation = function() {
    return mTransformation;
  };

  /***
   * Sets the property Rotate
   * @method setRotate
   * @param Rotate angle 
   * @visibility public
   */
  self.setRotate = function(angle) {
	if (mRotate != angle) {
		mRotate = angle;
		mElementChanged = true;    		
    }
  };

  /***
   * Gets the property Rotate
   * @method getRotate
   * @return Rotate angle 
   * @visibility public
   */
  self.getRotate = function() {
    return mRotate;
  };

  /***
   * Sets the property ScaleX
   * @method setScaleX
   * @param scale rate  
   * @visibility public
   */
  self.setScaleX = function(rate) {
	if (mTransformation.length == 0)
		mTransformation = [1, 0, 0, 1, 0, 0];
	if (mTransformation[0] != rate) {
		mTransformation[0] = rate;
		mElementChanged = true;    		
    }
  };

  /***
   * Gets the property ScaleX
   * @method getScaleX
   * @return scale rate  
   * @visibility public
   */
  self.getScaleX = function() {
	if (mTransformation.length == 0)
		return 1;
	else
		return mTransformation[0];
  };

  /***
   * Sets the property ScaleY
   * @method setScaleY
   * @param scale rate  
   * @visibility public
   */
  self.setScaleY = function(rate) {
	if (mTransformation.length == 0)
		mTransformation = [1, 0, 0, 1, 0, 0];
	if (mTransformation[3] != rate) {
		mTransformation[3] = rate;
		mElementChanged = true;    		
    }
  };

  /***
   * Gets the property ScaleY
   * @method getScaleY
   * @return scale rate  
   * @visibility public
   */
  self.getScaleY = function() {
	if (mTransformation.length == 0)
		return 1;
	else
		return mTransformation[3];
  };

  /***
   * Sets the property SkewX
   * @method setSkewX
   * @param skew angle  
   * @visibility public
   */
  self.setSkewX = function(angle) {
	if (mTransformation.length == 0)
		mTransformation = [1, 0, 0, 1, 0, 0];
	if (mTransformation[2] != angle) {
		mTransformation[2] = angle;
		mElementChanged = true;    		
    }
  };

  /***
   * Gets the property SkewX
   * @method getSkewX
   * @return skew angle  
   * @visibility public
   */
  self.getSkewX = function() {
	if (mTransformation.length == 0)
		return 0;
	else
		return mTransformation[2];
  };

  /***
   * Sets the property SkewY
   * @method setSkewY
   * @param skew angle  
   * @visibility public
   */
  self.setSkewY = function(angle) {
	if (mTransformation.length == 0)
		mTransformation = [1, 0, 0, 1, 0, 0];
	if (mTransformation[1] != angle) {
		mTransformation[1] = angle;
		mElementChanged = true;    		
    }
  };

  /***
   * Gets the property SkewY
   * @method getSkewY
   * @return skew angle  
   * @visibility public
   */
  self.getSkewY = function() {
	if (mTransformation.length == 0)
		return 0;
	else
		return mTransformation[1];
  };

  // ----------------------------------------
  // Private functions
  // ----------------------------------------

  /***
   * Gets information for element
   * @method getInfo
   * @visibility private
   * @return string
   */
  self.getInfo = function() {
  	var info = "x=" + mX.toFixed(2) + " y=" + mY.toFixed(2);
  	  	
    return info;
  };     

  // ----------------------------------------
  // Panel and group 
  // ----------------------------------------

  /***
   * Sets the panel for this element
   * @method setPanel
   * @param panel DrawingPanel
   * @visibility private
   */
  self.setPanel = function(panel) {
    mPanel = panel;
    mMustProject = true;
  };

  /***
   * Gets the panel for this element
   * @method getPanel
   * @return DrawingPanel
   * @visibility private
   */
  self.getPanel = function() { 
    return mPanel;
  };

  /***
   * Returns the DrawingPanel in which it (or its final ancestor group) is displayed
   * @method getGroupPanel
   * @return DrawingPanel
   * @visibility private
   */
  self.getGroupPanel = function() { 
    var el = self;
    while (el.getGroup()) el = el.getGroup();
    return el.getPanel();
  };

  /***
   * Sets the group of this element
   * @method setGroup
   * @param group Group
   * @visibility private
   */
  self.setGroup = function(group) {
//	    if (mGroup) mGroup.removeChild(self);
	    mGroup = group;
//	    if (mGroup) mGroup.addChild(self);
	    mElementChanged = true; 
  };

  /***
   * Get the group of this element, if any
   * @method getGroup
   * @return Group
   * @visibility private
   */
  self.getGroup = function() { 
    return mGroup; 
  };

  // ----------------------------------------
  // Set
  // ----------------------------------------

  /***
   * Sets the index of this element in the set
   * @method setSet
   * @param set ElementSet
   * @param index int
   * @visibility private
   */
  self.setSet = function(set,index) {
    mSet = set;
    mIndexInSet = index;
  };

  /***
   * Gets the set of this element, if any
   * @method getSet
   * @return ElementSet
   * @visibility private
   */
  self.getSet = function() { 
    return mSet; 
  };

  /***
   * Gets the index of this element in a set, if any
   * @method getSetIndex
   * @return int
   * @visibility private
   */
  self.getSetIndex = function() { 
    return mIndexInSet; 
  };

  // ----------------------------------------
  // Changes
  // ----------------------------------------

  /***
   * Whether the element has changed
   * @method isChanged
   * @return boolean
   * @visibility private
   */
  self.isChanged = function() {
    return mElementChanged;
  };

  /***
   * Tells the element that it has changed
   * Typically used by subclasses when they change something.
   * @method setChanged
   * @param changed boolean
   * @visibility private
   */
  self.setChanged = function(changed) {
    mElementChanged = changed;
  };

  /***
   * Returns whether the element group has changed
   * @method isGroupChanged
   * @return boolean
   * @visibility private
   */
  self.isGroupChanged = function() {
    var el = self.getGroup();
    while (typeof el != "undefined" && el !== null) {
      if (el.isChanged()) return true;
      el = el.getGroup();
    }
    return false;
  };

  /***
   * Tells the element whether it should reproject its points because the panel
   * has changed its projection parameters. Or, the other way round, sets it to false
   * if someone (typically methods in subclasses) took care of this already
   * @method setMustProject
   * @param needsIt boolean
   * @visibility private
   */
  self.setMustProject = function(needsIt) {
    mMustProject = needsIt;
  };

  /***
   * Whether the element needs to project. Typically used by the dawing panel 
   * whenever it changes its scales
   * @method isMustProject
   * @return boolean
   * @visibility private
   */
  self.isMustProject = function() { 
    return mMustProject;  
  };

  // ----------------------------------------
  // Conversions
  // ----------------------------------------

  /***
   * Returns pixel coordinates 
   * @method getPixelPosition
   * @param withTransf whether transformation must be considered, note it will be 
   *  false (or undefinded) when transformations are applied by svg (drawing)
   * @return double[2]
   * @visibility private
   */
  self.getPixelPosition = function(withTransf) {
	if (self.isChanged() || self.isGroupChanged() || self.isMustProject() || withTransf) {  	
	  	// get projected position
	  	mProjectedPosition = self.getPixelPositionOf(mX,mY,withTransf);
	  	// get projected size
	  	mProjectedSize = self.getPixelSizeOf(mSizeX,mSizeY);

	    self.setMustProject(withTransf); // must be projected the next time  				  		
	}
	return mProjectedPosition;
  };

  /***
   * Returns pixel sizes 
   * @method getPixelSizes
   * @param withTransf whether transformation must be considered, note it will be 
   *  false (or undefinded) when transformations are applied by svg (drawing)
   * @return double[2]
   * @visibility private
   */
  self.getPixelSizes = function(withTransf) {
	if (self.isChanged() || self.isGroupChanged() || self.isMustProject()) {  	
	  	// get projected position
	  	mProjectedPosition = self.getPixelPositionOf(mX,mY,withTransf);
	  	// get projected size
	  	mProjectedSize = self.getPixelSizeOf(mSizeX,mSizeY);

	    self.setMustProject(withTransf); // must be projected the next time 				  		
	}
	return mProjectedSize;
  };

  /***
   * Returns the absolute world sizes in the panel (after applying groups)  
   * @method getAbsoluteSize
   * @return double[2]
   * @visibility private
   */
  self.getAbsoluteSize = function() {
  	var size = (mPixelSize? mPanel.toPanelMod([mSizeX,-mSizeY]) : [mSizeX,mSizeY]); 
    return self.toGroupSpaceMod(size);
  };

  /***
   * Returns the absolute position in the panel (after applying groups)  
   * @method getAbsolutePosition
   * @param withTransf whether transformation must be considered, note it will be 
   *  false (or undefinded) when transformations are applied by svg (drawing)
   * @return double[2]
   * @visibility private
   */
  self.getAbsolutePosition = function(withTransf) {
  	var pos;
  	if(mPixelPosition) {
  		if(mGroup === null)
  			// pos using the panel scale 
  			pos = mPanel.toPanelPosition([mX,mY])
  		else
  			// with group, only support scale NUM  
  			pos = mPanel.toPanelPosition([mX,mY],EJSS_DRAWING2D.DrawingPanel.SCALE_NUM)
  	} else {
  		pos = [mX,mY]; 
  	}
    return self.toGroupSpace(pos,withTransf);
  };

  /***
   * Sets the absolute position in the panel (after applying groups)  
   * Note: Transformations are not considered and it is not supported when pixel position is used
   * @method setAbsolutePosition
   * @return double[2]
   * @visibility private
   */
  self.setAbsolutePosition = function(position) {
  	if(mPixelPosition) {
  		console.log("setAbsolutePosition not supported!");
  	} else {
  		self.setPosition(self.toElementSpace(position)); 
  	}
  };

  /***
   * Sets the absolute position X in the panel (after applying groups)  
   * Note: Transformations are not considered and it is not supported when pixel position is used
   * @method setAbsoluteX
   * @return double
   * @visibility private
   */
  self.setAbsoluteX = function(x) {
  	if(mPixelPosition) {
  		console.log("setAbsoluteX not supported!");
  	} else {
  		self.setX(self.toElementSpace([x,0])[0]); 
  	}
  };

  /***
   * Sets the absolute position Y in the panel (after applying groups)  
   * Note: Transformations are not considered and it is not supported when pixel position is used
   * @method setAbsoluteY
   * @return double
   * @visibility private
   */
  self.setAbsoluteY = function(y) {
  	if(mPixelPosition) {
  		console.log("setAbsoluteY not supported!");
  	} else {
  		self.setY(self.toElementSpace([0,y])[1]); 
  	}
  };

  /***
   * Returns pixel sizes for the given sizes in the element's world coordinates
   * @method getPixelSizeOf
   * @return double[2]
   * @visibility private
   */
  self.getPixelSizeOf = function(sx,sy) {
	if (mPixelSize) return [sx,-sy];
    else return self.getGroupPanel().toPixelMod(self.toGroupSpaceMod([sx,sy]));
  };
 
  /***
   * Returns pixel coordinates for the given point in the element's world coordinates
   * @method getPixelPositionOf
   * @param withTransf whether transformation must be considered, note it will be 
   *  false (or undefinded) when transformations are applied by svg (drawing)
   * @return double[2]
   * @visibility private
   */
  self.getPixelPositionOf = function(x,y,withTransf) {
  	if(mPixelPosition) {
  	  if (mGroup) {
  	    var groupPos = mGroup.getPixelPosition();
  	    return [groupPos[0]+x,groupPos[1]-y];
  	  }
  	  return [self.getGroupPanel().toPixelAxisX(x), self.getGroupPanel().toPixelAxisY(y)];
  	}
  	else {
  		if(mGroup === null)
  			// pos using the panel scale 
  			return self.getGroupPanel().toPixelPosition(self.toGroupSpace([x,y]));
  			// with group, only support scale NUM  
  		return self.getGroupPanel().toPixelPosition(
  			self.toGroupSpace([x,y],withTransf),
  			EJSS_DRAWING2D.DrawingPanel.SCALE_NUM);
  	}
  };


  /***
   * Transforms a module (longitude or size) in the element's world coordinates to
   * the group's world coordinates
   * @method toGroupSpaceMod
   * @param mod double[] The original module in the body frame
   * @return double[] The same array once transformed
   * @visibility private
   */
  self.toGroupSpaceMod = function(mod) {
    var el = mGroup;
    while (typeof el != "undefined" && el !== null) { // apply group transformations 
      // scale vector considering [sizeX,sizeY]
      mod[0] *= el.getSizeX();
      mod[1] *= el.getSizeY();
      el = el.getGroup(); // next group
    }
    return mod;
  };
  
  /***
   * Transforms a double[] point in the element's world coordinates to
   * the group's world coordinates
   * @method toGroupSpace
   * @param point double[] The original coordinates in the body frame
   * @param withTransf bool Considering group transformations
   * @return double[] The same array once transformed
   * @visibility private
   */
  self.toGroupSpace = function(point, withTransf) {
    var el = mGroup;
    while (typeof el != "undefined" && el !== null) { // apply group transformations
      // scale point considering [sizeX,sizeY]
      point[0] *= el.getSizeX();
      point[1] *= el.getSizeY();
      // translate point
      point[0] += el.getX();
      point[1] += el.getY();
      el = el.getGroup();	// next group
    }

    // transformations
    if(withTransf) {
	   el = mGroup;
	   while (typeof el != "undefined" && el !== null) { // apply group transformations
	      	  // note: ignore transformation matrix (please use Sensitivity = 0) 
		      // rotation point
		      var rot = el.getRotate();
			  if (rot != 0) { // rotation angle 	
			      var pos = el.getAbsolutePosition();
			      point = EJSS_TOOLS.Mathematics.rotate(pos,point,-rot);
			      
			  }	// other transfs not considered      	
	      el = el.getGroup();	// next group
       }
    }

    return point;
  };

  /***
   * Transforms a double[] point in the group's world coordinates to
   * the element's world coordinates
   * Note: Transformations are not considered!
   * @method toElementSpace
   * @param point double[] The original coordinates in the body frame
   * @return double[] The same array once translated
   * @visibility private
   */
  self.toElementSpace = function(point) {
  	var list = [];
    var el = mGroup;
    while (typeof el != "undefined" && el !== null) { 
      list.push(el);
      el = el.getGroup();	// next group
    }
	list = list.reverse();
	
	var listLength = list.length;
	for (var i = 0; i < listLength; i++) {
      var el = list[i];
      // translate point
      point[0] -= el.getX();
      point[1] -= el.getY();
      // scale point considering [sizeX,sizeY]
	  if(el.getSizeX() == 0) point[0] = 0;
      else point[0] /= el.getSizeX();
	  if(el.getSizeY() == 0) point[1] = 0;
      else point[1] /= el.getSizeY();
    }
    return point;
  };

  // ----------------------------------------
  // Interaction
  // ----------------------------------------

  /***
   * Returns the controller object
   * @method getController
   * @return Controller
   * @visibility private
   */
  self.getController = function () {
    return mController;
  };

  /***
   * Sets the controller
   * @method setController
   * @param Controller
   * @visibility private
   */
  self.setController = function (controller) {
    mController = controller;
  };

  /***
   * Returns one of the interaction targets defined by the element
   * @method getInteractionTarget
   * @param target
   * @visibility private
   */
  self.getInteractionTarget = function(target) {
    switch (target) {
      case PanelInteraction.TARGET_POSITION : return mInteraction.positionTarget;
      case PanelInteraction.TARGET_SIZE : return mInteraction.sizeTarget;
    }
    return null;
  };

  /***
   * Returns array of the interaction targets defined by the element
   * @method getInteractionTarget
   * @return targets
   * @visibility private
   */
  self.getInteractionTargets = function() {
  	return [mInteraction.positionTarget, mInteraction.sizeTarget];
  };

  /***
   * Returns array of the interaction targets defined by the element
   * @method getInteractionTarget
   * @return targets
   * @visibility private
   */
  self.getInteractionInformation = function() {
  	return { info : self.getPosition(), element : self, point : self.getGroupPanel().getPanelInteraction().getInteractionPoint() };
  };

  self.getOnDoubleClickInformation = function() {
	var info = self.getInteractionInformation();
	info.action = 'OnDoubleClick';
	return info;
  };

  self.getOnEnterInformation = function() {
	var info = self.getInteractionInformation();
	info.action = 'OnEnter';
	return info;
  };

  self.getOnExitInformation = function() {
	var info = self.getInteractionInformation();
	info.action = 'OnExit';
	return info;
  };
	  
  self.getOnPressInformation = function() {
	var info = self.getInteractionInformation();
	info.action = 'OnPress';
	return info;
  };

  self.getOnDragInformation = function() {
	var info = self.getInteractionInformation();
	info.action = 'OnDrag';
	return info;
  };

  self.getOnReleaseInformation = function() {
	var info = self.getInteractionInformation();
	info.action = 'OnRelease';
	return info;
  };
  
  // ----------------------------------------------------
  // Properties
  // ----------------------------------------------------
  
  /***
   * Registers properties in a ControlElement
   * @method registerProperties
   * @param controller A ControlElement that becomes the element controller
   * @visibility private
   */
  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.Element.registerProperties(self,controller);
  };

  /***
   * Copies itself to another element
   * @method copyTo
   * @param element Element
   * @visibility private
   */
  self.copyTo = function(element) {
    EJSS_DRAWING2D.Element.copyTo(self,element);
  };

  /***
   * Get JSON object with private variables
   * @method serialize
   * @visibility private
   */
  self.serialize = function() {
  	return { 
  		mStyle: mStyle.serialize(), 
  		mVisible: mVisible, 
  		mMeasured: mMeasured, 
  		mRelativePosition: mRelativePosition,
  		
  		mX: mX, mY: mY, mPixelPosition: mPixelPosition,
  		mSizeX: mSizeX, mSizeY: mSizeY, mPixelSize: mPixelSize,
  		mTransformation: mTransformation,
		
		mInteraction: {
			//positionTarget: mInteraction.positionTarget.serialize(),
			//sizeTarget: mInteraction.sizeTarget.serialize()
		},
		
		mPanel: (mPanel?mPanel.getName():mPanel), mGroup: (mGroup?mGroup.getName():mGroup), 
		mSet: (mSet?mSet.getName():mSet), mIndexInSet: mIndexInSet,
		
		mProjectedPosition: mProjectedPosition, mProjectedSize: mProjectedSize,		
		 
  		};
  }
  
  /***
   * Set JSON object with private variables
   * @method unserialize
   * @parem json JSON object
   * @visibility private
   */
  self.unserialize = function(json) {
	mStyle.unserialize(json.mStyle), 
	mVisible = json.mVisible, 
	mMeasured = json.mMeasured, 
	mRelativePosition = json.mRelativePosition,
	
	mX = json.mX, mY = json.mY, mPixelPosition = json.mPixelPosition,
	mSizeX = json.mSizeX, mSizeY = json.mSizeY, mPixelSize = json.mPixelSize,
	mTransformation = json.mTransformation,
	
	// mInteraction.positionTarget = json.mInteraction.positionTarget,
	// mInteraction.sizeTarget = json.mInteraction.sizeTarget,
	
	// not support references changing
	//	mPanel = _view[json.mPanel], mGroup = json.mGroup, mSet = json.mSet, 
	mIndexInSet = json.mIndexInSet,
	
	mProjectedPosition = json.mProjectedPosition, mProjectedSize = json.mProjectedSize,		

    mElementChanged = true; 
  }

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  mStyle.setChangeListener(function (change) { mElementChanged = true; });
  
  return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * ElementSet
 * @class ElementSet 
 * @constructor  
 */
EJSS_DRAWING2D.ElementSet = {
    
    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

    /**
     * Registers properties in a ControlElement
     * @method 
     * @param element The element with the properties
     * @param controller A ControlElement that becomes the element controller
     */
    registerProperties : function(set,controller) {
      var TARGET_POSITION = EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION;
      var TARGET_SIZE = EJSS_DRAWING2D.PanelInteraction.TARGET_SIZE;

      set.setController(controller); // remember it, in case you change the number of elements
      set.setToEach(function(element,value) { element.setController(value); }, controller); // make all elements in the set report to the same controller

      controller.registerProperty("Parent", 
      	  function(panel) {
      	  	set.setParent(panel); 
      		set.setToEach(function(element,value) { element.setParent(value); }, set); 
      	  } );

      controller.registerProperty("NumberOfElements", set.setNumberOfElements);

      controller.registerProperty("ElementInteracted", set.setElementInteracted, set.getElementInteracted);

//      controller.registerProperty("x",function(v) { set.foreach("setX",v); }, function() { return set.getall("getX"); });
      controller.registerProperty("X",
          function(v) { set.setToEach(function(element,value) { element.setX(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getX(); } ); }      
          );
      controller.registerProperty("Y",
          function(v) { set.setToEach(function(element,value) { element.setY(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getY(); } ); }      
      );
      controller.registerProperty("Position",
          function(v) { set.setToEach(function(element,value) { element.setPosition(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getPosition(); } ); }      
      );
      controller.registerProperty("PixelPosition", 
          function(v) { set.setToEach(function(element,value) { element.setPixelPosition(value); }, v); }
      );

      controller.registerProperty("Diameter",
          function(v) { set.setToEach(function(element,value) { element.setSize([value,value]); }, v); }
          );
      controller.registerProperty("Radius",
          function(v) { set.setToEach(function(element,value) { element.setSize([value*2,value*2]); }, v); }
          );
      controller.registerProperty("SizeX",
          function(v) { set.setToEach(function(element,value) { element.setSizeX(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getSizeX(); } ); }      
          );
      controller.registerProperty("SizeY",
          function(v) { set.setToEach(function(element,value) { element.setSizeY(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getSizeY(); } ); }      
      );
      controller.registerProperty("Size",
          function(v) { set.setToEach(function(element,value) { element.setSize(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getSize(); } ); }      
      );
      controller.registerProperty("PixelSize", 
          function(v) { set.setToEach(function(element,value) { element.setPixelSize(value); }, v); }
      );

      controller.registerProperty("Bounds",function(bounds) {
        var setBounds = function(element,bound) {
          element.setX(bound[0]);
          element.setY(bound[2]); 
          element.setSizeX(bound[1]-bound[0]);
          element.setSizeY(bound[3]-bound[2]);
        };
        set.setToEach(setBounds, bounds);
      },
      function() {
        var getBounds = function(element) {
          return [element.getX(),element.getX()+element.getSizeX(),
                  element.getY(),element.getY()+element.getSizeY()]; 
        };
        return set.getFromEach(getBounds);
      });

      controller.registerProperty("Visibility", 
          function(v) { set.setToEach(function(element,value) { element.setVisible(value); }, v); }
      );
      controller.registerProperty("Measured", 
          function(v) { set.setToEach(function(element,value) { element.setMeasured(value); }, v); }
      );

      controller.registerProperty("Transformation",
      	  function(v) { set.setToEach(function(element,value) { element.setTransformation(value); }, v);
      });

      controller.registerProperty("Rotate",
      	  function(v) { set.setToEach(function(element,value) { element.setRotate(value); }, v);
      });

      controller.registerProperty("ScaleX",
      	  function(v) { set.setToEach(function(element,value) { element.setScaleX(value); }, v);
      });

      controller.registerProperty("ScaleY",
      	  function(v) { set.setToEach(function(element,value) { element.setScaleY(value); }, v);
      });

      controller.registerProperty("SkewX",
      	  function(v) { set.setToEach(function(element,value) { element.setSkewX(value); }, v);
      });

      controller.registerProperty("SkewY",
      	  function(v) { set.setToEach(function(element,value) { element.setSkewY(value); }, v);
      });

      controller.registerProperty("RelativePosition", 
          function(v) { set.setToEach(function(element,value) { element.setRelativePosition(value); }, v); }
      );

      controller.registerProperty("LineColor", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setLineColor(value); }, v); }
      );
      controller.registerProperty("LineWidth", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setLineWidth(value); }, v); }
      );
      controller.registerProperty("DrawLines", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setDrawLines(value); }, v); }
      );
      controller.registerProperty("FillColor", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setFillColor(value); }, v); }
      );
      controller.registerProperty("DrawFill", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setDrawFill(value); }, v); }
      );
      controller.registerProperty("Attributes", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setAttributes(value); }, v); }
      );
      controller.registerProperty("ShapeRendering", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setShapeRendering(value); }, v); }
      );

      controller.registerProperty("EnabledPosition", 
          function(v) { set.setToEach(function(element,value) { element.getInteractionTarget(TARGET_POSITION).setMotionEnabled(value); }, v); }
      );
      controller.registerProperty("MovesGroup", 
          function(v) { set.setToEach(function(element,value) { element.getInteractionTarget(TARGET_POSITION).setAffectsGroup(value); }, v); }
      );
      controller.registerProperty("EnabledSize", 
          function(v) { set.setToEach(function(element,value) { element.getInteractionTarget(TARGET_SIZE).setMotionEnabled(value); }, v); }
      );
      controller.registerProperty("ResizesGroup", 
          function(v) { set.setToEach(function(element,value) { element.getInteractionTarget(TARGET_SIZE).setAffectsGroup(value); }, v); }
      );
      controller.registerProperty("Sensitivity", 
          function(v) { set.setToEach(function(element,value) { 
            element.getInteractionTarget(TARGET_POSITION).setSensitivity(value); 
            element.getInteractionTarget(TARGET_SIZE).setSensitivity(value); 
          }, v); }
      );

      var dataFunction = function() { 
        //var index = set.getGroupPanel().getPanelInteraction().getIndexElement();
        //set.setElementInteracted(index);
        //return { index: index, position: set.getElements()[index].getPosition() };
        var panel = set.getGroupPanel();
        var index = panel.getPanelInteraction().getIndexElement();
        var element = panel.getElements()[index];
        var elementIndex = element.getSetIndex(); 
        set.setElementInteracted(elementIndex);
  	    controller.propertiesChanged("ElementInteracted");
	    //controller.reportInteractions();	 
        return { index: elementIndex, position: element.getPosition() };   
      };
      
      // Actions
      controller.registerAction("OnDoubleClick",   dataFunction);
      controller.registerAction("OnEnter",   dataFunction);
      controller.registerAction("OnExit",    dataFunction);
      controller.registerAction("OnPress",   dataFunction);
      controller.registerAction("OnDrag",    dataFunction);
      controller.registerAction("OnRelease", dataFunction);
    }
};

/**
 * Element set
 * Creates a basic abstract ElementSet
 * @method elementSet
 * @param mConstructor the function that creates new elements (will be used as element = mConstructor(name))
 * @returns An abstract 2D element set
 */
EJSS_DRAWING2D.elementSet = function (mConstructor, mName) {  
  var self = EJSS_DRAWING2D.group(mName);

  // Static references
  var ElementSet = EJSS_DRAWING2D.ElementSet;		// reference for ElementSet
  
  // Configuration variables
  var mElementList = []; 
  var mNumberOfElementsSet = false;

  // Implementation variables  
  var mElementInteracted = -1;

  // Last list of removed elements
  var mLastElementList = [];

  // ----------------------------------------
  // Configuration methods
  // ----------------------------------------

  /**
   * Sets the number of element of this set
   * @method setNumberOfElements
   * @param numberOfElements the number of elements, must be >= 1
   */
  self.setNumberOfElements = function(numberOfElements) {
	mNumberOfElementsSet = true;
	adjustNumberOfElements(numberOfElements);
  };

  /*
   * Adjusts the number of element of this set
   * @method adjustNumberOfElements
   * @param numberOfElements the number of elements, must be >= 1
   */
  function adjustNumberOfElements(numberOfElements) {
    // keep original settings for the new elements
    var name = self.getName ? self.getName() : "unnamed";
    numberOfElements = Math.max(1,numberOfElements);
    var diff = mElementList.length-numberOfElements;
    if (diff > 0) {
    	mLastElementList = mElementList.splice(numberOfElements, diff);
		for (var j = 0; j < mLastElementList.length; j++) { 
			var panel = mLastElementList[j].getPanel(); // remove element from panel
			if(panel) panel.removeElement(mLastElementList[j])
		} 
    } else if (diff < 0) {
    	mLastElementList = [];
    	var controller = self.getController();
    	var oldElement = mElementList[mElementList.length-1];
		for (var i = mElementList.length; i < numberOfElements; i++) {
			var element = mConstructor(name+"["+i+"]"); // new element			
			element.setSet(self,i);
			oldElement.copyTo(element);
			element.setController(controller);
		  	mElementList.push(element);	
		} 
    }
  };

  /**
   * Returns the array with all elements in the set
   * @method getElements
   * @return array of Elements
   */
  self.getElements = function() {
    return mElementList;
  };

  /**
   * Returns the element in the set at the given index
   * @method getElement
   * @return Element
   */
  self.getElement = function(index) {
    return mElementList[index];
  };
  
  /**
   * Returns last list of removed elements and reset the value
   * @method getLastElements
   * @return last list of Elements
   */
  self.getLastElements = function() {
  	var ret = mLastElementList.slice();
  	mLastElementList = [];
    return ret;
  };
  
  self.setElementInteracted = function(index) {
    mElementInteracted = index;
  };

  self.getElementInteracted = function() {
    return mElementInteracted;
  };
  
  // ----------------------------------------
  // Relation to its panel
  // ----------------------------------------
  
  var super_setPanel = self.setPanel;

  self.setPanel = function(panel) {
    super_setPanel(panel);
    for (var i=0,n=mElementList.length;i<n;i++) mElementList[i].setPanel(panel);
  };

  // ----------------------------------------------------
  // Properties
  // ----------------------------------------------------

  /**
   * Applies to each element in the set the function f with argument v, or v[i], if it is an array
   * @method setToEach
   * @param f function
   * @param v arguments
   */
  self.setToEach = function(f,v) {
    if (Array.isArray(v)) {
      if (!mNumberOfElementsSet) { 
        if (mElementList.length < v.length) adjustNumberOfElements(v.length);
      }
      for (var i=0,n=Math.min(mElementList.length,v.length);i<n;i++) f(mElementList[i],v[i]);
    }
    else {
      for (var i=0,n=mElementList.length;i<n;i++) f(mElementList[i],v);
    }
  };

  /**
   * Returns an array with the result of applying the function to each element of the set
   * @method getFromEach
   * @param f function
   * @return f function return 
   */
  self.getFromEach = function(f) {
    var value = [];
    for (var i=0, n=mElementList.length;i<n;i++) value[i] = f(mElementList[i]);
    return value;
  };

  /**
   * Registers properties in a ControlElement
   * @method registerProperties
   * @param controller A ControlElement that becomes the element controller
   */
  self.registerProperties = function(controller) {
    ElementSet.registerProperties(self,controller);
  };
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------
	
  var element = mConstructor(mName + "[0]"); // new element			
  element.setController(self.getController());
  element.setSet(self,0);
  mElementList.push(element);
  	
  return self;
};
/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Grid
 * @class Grid 
 * @constructor  
 */
EJSS_DRAWING2D.Grid = {	
	SCALE_NUM	: 0,	// decimal
	SCALE_LOG	: 1,	// logarithmic
	
	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class

		controller.registerProperty("StepX", element.setStepX, element.getStepX);
		controller.registerProperty("StepY", element.setStepY, element.getStepY);
		controller.registerProperty("TicksX", element.setTicksX, element.getTicksX);
		controller.registerProperty("TicksY", element.setTicksY, element.getTicksY);
		controller.registerProperty("TickStepX", element.setTickStepX, element.getTickStepX);
		controller.registerProperty("TickStepY", element.setTickStepY, element.getTickStepY);
		controller.registerProperty("TicksXMode", element.setTicksXMode, element.getTicksXMode);
		controller.registerProperty("TicksYMode", element.setTicksYMode, element.getTicksYMode);

		controller.registerProperty("ShowX", element.setShowX, element.getShowX);
		controller.registerProperty("ShowY", element.setShowY, element.getShowY);

		controller.registerProperty("FixedTickX", element.setFixedTickX, element.getFixedTickX);				
		controller.registerProperty("FixedTickY", element.setFixedTickY, element.getFixedTickY);				
		controller.registerProperty("ScaleX", element.setScaleX, element.getScaleX);				
		controller.registerProperty("ScaleY", element.setScaleY, element.getScaleY);				
		
		controller.registerProperty("ScalePrecisionX", element.setScalePrecisionX, element.getScalePrecisionX);		
		controller.registerProperty("ScalePrecisionY", element.setScalePrecisionY, element.getScalePrecisionY);		
		
		// auto ticks
		controller.registerProperty("AutoTicksX", element.setAutoTicksX, element.getAutoTicksX);				
		controller.registerProperty("AutoTicksY", element.setAutoTicksY, element.getAutoTicksY);				
		controller.registerProperty("AutoStepXMin", element.setAutoStepXMin, element.getAutoStepXMin);				
		controller.registerProperty("AutoStepYMin", element.setAutoStepYMin, element.getAutoStepYMin);				
		controller.registerProperty("AutoTicksXRange", element.setAutoTicksXRange, element.getAutoTicksXRange);				
		controller.registerProperty("AutoTicksYRange", element.setAutoTicksYRange, element.getAutoTicksYRange);
		
		// style
		controller.registerProperty("LineColorX", element.setLineColorX, element.getLineColorX);
		controller.registerProperty("LineWidthX", element.setLineWidthX, element.getLineWidthX);
		controller.registerProperty("ShapeRenderingX", element.setShapeRenderingX, element.getShapeRenderingX);
		controller.registerProperty("LineColorY", element.setLineColorY, element.getLineColorY);
		controller.registerProperty("LineWidthY", element.setLineWidthY, element.getLineWidthY);
		controller.registerProperty("ShapeRenderingY", element.setShapeRenderingY, element.getShapeRenderingY);						
	}
			
};

/**
 * Creates a 2D Grid
 * @method grid
 */
EJSS_DRAWING2D.grid = function(name) {
	var self = EJSS_DRAWING2D.element(name);
 
  	// drawing priority: mAutoTicks - mTicks - mStep
 	var mAutoTicksX = true;			// auto-ticks in X
 	var mAutoTicksY = true;			// auto-ticks in Y
 	var mTicksX = 0;				// number of ticks in X
 	var mTicksY = 0;				// number of ticks in Y
 	var mStepX = 0;					// step between ticks in pixels in X
 	var mStepY = 0;					// step between ticks in pixels in Y
    var mTickStepX = 0;
    var mTickStepY = 0;
	var mShowX = true;				// show X
	var mShowY = true;				// show Y

	var mFixedTickX = Number.NaN;			// ticks fixed in axis X	
	var mFixedTickY = Number.NaN;			// ticks fixed in axis Y	
	var mScaleX = [-1,1];			// X axis scale
	var mScaleY = [-1,1];			// Y axis scale
	var mScalePrecisionX = 1;		// number of decimals X
	var mScalePrecisionY = 1;		// number of decimals Y
	
	// ticks properties
 	var mTicksXMode = EJSS_DRAWING2D.Grid.SCALE_NUM;		// X axis scale 
 	var mTicksYMode = EJSS_DRAWING2D.Grid.SCALE_NUM;		// Y axis scale
 
	// auto ticks 	
	var mAutoStepXMin = 40;		// step x minimun in pixels
	var mAutoStepYMin = 40;		// step y minimun in pixels
	var mAutoTicksXRange = [5,10,20];		// ticks x range
	var mAutoTicksYRange = [5,10,20];		// ticks y range
											//   is the step minimun possible in mAutoTicksRange[length-1]?
											//   and in mAutoTicksRange[length-2]? ... 
											//   then mAutoTicksRange[length-2] is the number of ticks   

	// styles
	var mLineColorX = 'black';
	var mLineWidthX = 0.5;
	var mShapeRenderingX = EJSS_DRAWING2D.Style.RENDER_AUTO;
	var mLineColorY = 'black';
	var mLineWidthY = 0.5;
	var mShapeRenderingY = EJSS_DRAWING2D.Style.RENDER_AUTO;

	self.getClass = function() {
		return "ElementGrid";
	}

	/** 
	 * @method setAutoTicksRangeY
	 * @param range
	 */
	self.setAutoTicksRangeY = function (range) {
	  	if(mAutoTicksYRange != range) {
	  		mAutoTicksYRange = range;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getAutoTicksRangeY
	 * @return
	 */
	self.getAutoTicksRangeY = function() { 
		return mAutoTicksYRange; 
	}

	/** 
	 * @method setAutoTicksRangeX
	 * @param range
	 */
	self.setAutoTicksRangeX = function (range) {
	  	if(mAutoTicksXRange != range) {
	  		mAutoTicksXRange = range;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getAutoTicksRangeX
	 * @return
	 */
	self.getAutoTicksRangeX = function() { 
		return mAutoTicksXRange; 
	}

	/** 
	 * @method setAutoStepYMin
	 * @param min
	 */
	self.setAutoStepYMin = function (min) {
	  	if(mAutoStepYMin != min) {
	  		mAutoStepYMin = min;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getAutoStepYMin
	 * @return
	 */
	self.getAutoStepYMin = function() { 
		return mAutoStepYMin; 
	}

	/** 
	 * @method setAutoStepXMin
	 * @param min
	 */
	self.setAutoStepXMin = function (min) {
	  	if(mAutoStepXMin != min) {
	  		mAutoStepXMin = min;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getAutoStepXMin
	 * @return
	 */
	self.getAutoStepXMin = function() { 
		return mAutoStepXMin; 
	}

	/** 
	 * @method setScaleY
	 * @param scale
	 */
	self.setScaleY = function (scale) {
	  	if(mScaleY != scale) {
	  		mScaleY = scale;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getScaleY
	 * @return
	 */
	self.getScaleY = function() { 
		return mScaleY; 
	}
	
	/** 
	 * @method setScaleX
	 * @param scale
	 */
	self.setScaleX = function (scale) {
	  	if(mScaleX != scale) {
	  		mScaleX = scale;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getScaleX
	 * @return
	 */
	self.getScaleX = function() { 
		return mScaleX; 
	}
	
	/** 
	 * @method setScalePrecisionX
	 * @param scalePrecisionX
	 */
	self.setScalePrecisionX = function (scalePrecisionX) {
		if(mScalePrecisionX != scalePrecisionX) {
	  		mScalePrecisionX = scalePrecisionX;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getScalePrecisionX
	 * @return
	 */
	self.getScalePrecisionX = function() { 
		return mScalePrecisionX; 
	}

	/** 
	 * @method setScalePrecisionY
	 * @param scalePrecisionY
	 */
	self.setScalePrecisionY = function (scalePrecisionY) {
		if(mScalePrecisionY != scalePrecisionY) {
	  		mScalePrecisionY = scalePrecisionY;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getScalePrecisionY
	 * @return
	 */
	self.getScalePrecisionY = function() { 
		return mScalePrecisionY; 
	}

	/** 
	 * @method setFixedTickX
	 * @param fixed
	 */
	self.setFixedTickX = function (fixed) {
	  	if(mFixedTickX != fixed) {
	  		mFixedTickX = fixed;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getFixedTickX
	 * @return
	 */
	self.getFixedTickX = function() { 
		return mFixedTickX; 
	}

	/** 
	 * @method setFixedTickY
	 * @param fixed
	 */
	self.setFixedTickY = function (fixed) {
	  	if(mFixedTickY != fixed) {
	  		mFixedTickY = fixed;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getFixedTickY
	 * @return
	 */
	self.getFixedTickY = function() { 
		return mFixedTickY; 
	}
			
	/** 
	 * @method setAutoTicksX
	 * @param auto
	 */
	self.setAutoTicksX = function (auto) {
	  	if(mAutoTicksX != auto) {
	  		mAutoTicksX = auto;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getAutoTicksX
	 * @return
	 */
	self.getAutoTicksX = function() { 
		return mAutoTicksX; 
	}

	/** 
	 * @method setAutoTicksY
	 * @param auto
	 */
	self.setAutoTicksY = function (auto) {
	  	if(mAutoTicksY != auto) {
	  		mAutoTicksY = auto;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getAutoTicksY
	 * @return
	 */
	self.getAutoTicksY = function() { 
		return mAutoTicksY; 
	}
	 
	/** 
	 * @method setStepX
	 * @param stetX
	 */
	self.setStepX = function (stepX) {
	  	if(mStepX != stepX) {
	  		mStepX = stepX;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getStepX
	 * @return
	 */
	self.getStepX = function() { 
		return mStepX; 
	}

	/** 
	 * @method setStepY
	 * @param stetY
	 */
	self.setStepY = function (stepY) {
		if(mStepY != stepY) {
	  		mStepY = stepY;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getStepY
	 * @return
	 */
	self.getStepY = function() { 
		return mStepY; 
	}

	/** 
	 * @method setTickStepX
	 * @param stetX
	 */
	self.setTickStepX = function (TickStepX) {
	  	if(mTickStepX != TickStepX) {
	  		mTickStepX = TickStepX;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getTickStepX
	 * @return
	 */
	self.getTickStepX = function() { 
		return mTickStepX; 
	}

	/** 
	 * @method setTickStepY
	 * @param stetY
	 */
	self.setTickStepY = function (TickStepY) {
		if(mTickStepY != TickStepY) {
	  		mTickStepY = TickStepY;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getTickStepY
	 * @return
	 */
	self.getTickStepY = function() { 
		return mTickStepY; 
	}
	
	/** 
	 * @method setTicksX
	 * @param stetX
	 */
	self.setTicksX = function (ticksX) {
		if(mTicksX != ticksX) {			
	  		mTicksX = ticksX;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getTicksX
	 * @return
	 */
	self.getTicksX = function() { 
		return mTicksX; 
	}

	/** 
	 * @method setTicksY
	 * @param stetY
	 */
	self.setTicksY = function (ticksY) {
	  	if(mTicksY != ticksY) {
	  		mTicksY = ticksY;
	  		self.setChanged(true);
	  	}
	}
	  	  
	/**
	 * @method getTicksY
	 * @return
	 */
	self.getTicksY = function() { 
		return mTicksY; 
	}

	/** 
	 * @method setTicksXMode
	 * @param ticksXMode
	 */
	self.setTicksXMode = function (ticksXMode) {
    	if (typeof ticksXMode == "string") ticksXMode = EJSS_DRAWING2D.Grid[ticksXMode.toUpperCase()];
    	if(mTicksXMode != ticksXMode) {
    		mTicksXMode = ticksXMode;	
    		self.setChanged(true);
    	}
	}
	  
	/**
	 * @method getTicksXMode
	 * @return
	 */
	self.getTicksXMode = function() { 
		return mTicksXMode; 
	}

	/** 
	 * @method setTicksYMode
	 * @param ticksYMode
	 */
	self.setTicksYMode = function (ticksYMode) {
    	if (typeof ticksYMode == "string") ticksXMode = EJSS_DRAWING2D.Grid[ticksYMode.toUpperCase()];
    	if(mTicksYMode != ticksYMode) {
    		mTicksYMode = ticksYMode;
    		self.setChanged(true);
    	}	
	}

	/** 
	 * @method setShowX
	 * @param show
	 */
	self.setShowX = function (show) {
	  	if(mShowX != show) {
	  		mShowX = show;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getShowX
	 * @return
	 */
	self.getShowX = function() { 
		return mShowX; 
	}

	/** 
	 * @method setShowY
	 * @param show
	 */
	self.setShowY = function (show) {
	  	if(mShowY != show) {
	  		mShowY = show;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getShowY
	 * @return
	 */
	self.getShowY = function() { 
		return mShowY; 
	}
	  
	/**
	 * @method getTicksYMode
	 * @return
	 */
	self.getTicksYMode = function() { 
		return mTicksYMode; 
	}

	/**
	* Set the line color of the element
	* @param color a stroke style
	*/
  	self.setLineColorX = function(color) { 
	    if (typeof color !== "string") color = EJSS_TOOLS.DisplayColors.getLineColor(color);
	    if (color!=mLineColorX) {
	      mLineColorX = color; 
		  self.setChanged(true);
	    }
	    return self;
  	};
    
	/**
	* Get the line color
	*/
	self.getLineColorX = function() { 
    	return mLineColorX; 
  	};

	/**
	* Set the line width of the element
	* @param width a stroke width (may be double, such as 0.5, the default)
	*/
	self.setLineWidthX = function(width) { 
	    if (width!=mLineWidthX) {
	      mLineWidthX = width; 
		  self.setChanged(true);
	    }
	};

	/**
	* Get the line width
	*/
  	self.getLineWidthX = function() { 
  		return mLineWidthX; 
  	};

	/**
	* Sets shape rendering
	*/
	self.setShapeRenderingX = function(rendering) {
	    if (rendering.substring(0,6) == "RENDER") rendering = EJSS_DRAWING2D.Style[rendering.toUpperCase()];
	    if (mShapeRenderingX != rendering) {
	      mShapeRenderingX = rendering;
		  self.setChanged(true);
	    }
	};
  
  	self.getShapeRenderingX = function() { 
    	return mShapeRenderingX;
  	};

	/**
	* Set the line color of the element
	* @param color a stroke style
	*/
  	self.setLineColorY = function(color) { 
	    if (typeof color !== "string") color = EJSS_TOOLS.DisplayColors.getLineColor(color);
	    if (color!=mLineColorY) {
	      mLineColorY = color; 
		  self.setChanged(true);
	    }
	    return self;
  	};
    
	/**
	* Get the line color
	*/
	self.getLineColorY = function() { 
    	return mLineColorY; 
  	};

	/**
	* Set the line width of the element
	* @param width a stroke width (may be double, such as 0.5, the default)
	*/
	self.setLineWidthY = function(width) { 
	    if (width!=mLineWidthY) {
	      mLineWidthY = width; 
		  self.setChanged(true);
	    }
	};

	/**
	* Get the line width
	*/
  	self.getLineWidthY = function() { 
  		return mLineWidthY; 
  	};

	/**
	* Sets shape rendering
	*/
	self.setShapeRenderingY = function(rendering) {
	    if (rendering.substring(0,6) == "RENDER") rendering = EJSS_DRAWING2D.Style[rendering.toUpperCase()];
	    if (mShapeRenderingY != rendering) {
	      mShapeRenderingY = rendering;
		  self.setChanged(true);
	    }
	};
  
  	self.getShapeRenderingY = function() { 
    	return mShapeRenderingY;
  	};

	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.Grid.registerProperties(self, controller);
	};
  
	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Group
 * @class Group 
 * @constructor  
 */
EJSS_DRAWING2D.Group = {

};

/**
 * Creates a group
 * @method group
 */
EJSS_DRAWING2D.group = function (name) {
  var self = EJSS_DRAWING2D.element(name);
  
//  var mChildren = [];

  self.getClass = function() {
  	return "ElementGroup";
  }
  
  self.isGroup = function() {
  	return true;
  }
  
//  self.addChild = function(child) {
//	  EJSS_TOOLS.addToArray(mChildren,child);
//  }
//  
//  self.removeChild = function(child) {
//	  EJSS_TOOLS.removeFromArray(mChildren,child);
//  }
//  
//  self.superSetChanged = self.setChanged;
//  
//  self.setChanged = function(changed) {
//	self.superSetChanged(changed);
//	if (changed) for (var i=0,n=mChildren.length; i<n; i++) mChildren[i].setChanged(changed);
//  }
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------
  
  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//GroupSet
//---------------------------------

/**
 * GroupSet
 * @class GroupSet 
 * @constructor  
 */
EJSS_DRAWING2D.GroupSet = {
    
};


/**
 * Creates a set of Segments
 * @method groupSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING2D.groupSet = function (mName) {
  var self = EJSS_DRAWING2D.elementSet(EJSS_DRAWING2D.group, mName);

  return self;
};
/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * A Histogram displays a diagram of frequencies of data
 * @class EJSS_DRAWING2D.Histogram 
 * @parent EJSS_DRAWING2D.Element
 * @constructor  
 */
EJSS_DRAWING2D.Histogram = {

	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy  	
  		
		dest.setInput(source.getInput());
		dest.setEnabled(source.getEnabled());
		dest.setOccurrences(source.getOccurrences());
		dest.setClearAtInput(source.getClearAtInput());
		dest.setDiscrete(source.getDiscrete());
		dest.setNormalized(source.getNormalized());
		dest.setBinWidth(source.getBinWidth());
  	},
  	
  	
	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class
		
	 /*** 
	  * The input value to append
	  * @property PixelPosition 
	  * @type double or double[]
	  * @default false
	  */      		
	 	controller.registerProperty("Input", element.addInput);

	 /*** 
	  * Whether the histogram actually accepts input values
	  * @property Active 
	  * @type boolean
	  * @default true
	  */      		
	 	controller.registerProperty("Active", element.setActive, element.getActive);

	 /*** 
	  * The number of the occurrences of the input
	  * @property Occurrences 
	  * @type int or double
	  * @default 1
	  */      		
	 	controller.registerProperty("Occurrences", element.setOccurrences, element.getOccurrences);

	 /*** 
	  * Whether to clear old data whenever new data gets in
	  * @property ClearAtInput 
	  * @type boolean
	  * @default false
	  */      		
	 	controller.registerProperty("ClearAtInput", element.setClearAtInput, element.getClearAtInput);

	 /*** 
	  * Whether the bins are discrete or continuous. A discrete bin looks like a thin vertical line, 
	  * a continuous one looks like a bar with the width of the bin
	  * @property Discrete 
	  * @type boolean
	  * @default false
	  */      		
	 	controller.registerProperty("Discrete", element.setDiscrete, element.getDiscrete);

	 /*** 
	  * Whether the data are normalized to one
	  * @property Normalized 
	  * @type boolean
	  * @default false
	  */      		
		controller.registerProperty("Normalized", element.setNormalized, element.getNormalized);

	 /*** 
	  * The width of the bins. This is the maximum distance that makes two inputs contribute 
	  * to the same bin height. As an example, is the with is 0.5, the inputs 1.1 and 1.4 
	  * will add their occurrences to the same bin
	  * @property BinWidth 
	  * @type int or double
	  * @default 1
	  */      		
	 	controller.registerProperty("BinWidth", element.setBinWidth, element.getBinWidth);

	 /*** 
	  * Fix bin in the input 
	  * @property FixBin 
	  * @type double
	  * @default 0
	  */      		
	 	controller.registerProperty("FixBin", element.setFixBin, element.getFixBin);
	 	
	}
};

/**
 * Creates a 2D Histogram
 * @method Histogram
 */
EJSS_DRAWING2D.histogram = function(mName) {
	var self = EJSS_DRAWING2D.element(mName);
 
    // Instance variables
	var mCurrentInput = [];
	var mActive = true;
	var mOccurrences = 1;
	var mClearAtInput = false;
	var mDiscrete = false;
    var mNormalized = false;    
    var mBinWidth = 1;
    var mFixBin = 0;
    var mBars = null;
		 
	self.getClass = function() {
		return "ElementHistogram";
	}

  /**
   * Makes it a collector for the view
   * @method dataCollected()
   * @see EJSS_CORE.view
   */
  self.dataCollected  = function() { }  
  
	self.getInput = function() { 
	  	return mCurrentInput; 
	};

	self.addInput = function(input) {
	    if (!mActive) return;
	  	if (mClearAtInput) self.clear();
	  	for (var i=0; i<mOccurrences; i++) {
			if(input instanceof Array) mCurrentInput = mCurrentInput.concat(input);  		
			else mCurrentInput.push(input);
		}
		mBars = null;
	    self.setChanged(true);
	};

	self.clear = function() {
		mCurrentInput = [];
		mBars = null;
		self.setChanged(true);
	}

	self.getClearAtInput = function() { 
	 	return mClearAtInput; 
	}

	self.setClearAtInput = function(clear) {
		if(mClearAtInput != clear) {
	   		mClearAtInput = clear;
	  		self.setChanged(true);
		}
	}

	self.getActive = function() { 
	 	return mActive; 
	}

	self.setActive = function(active) {
		if(mActive != active) {
	   		mActive = active;
	  		self.setChanged(true);
		}
	}

	self.getOccurrences = function() { 
	 	return mOccurrences; 
	}

	self.setOccurrences = function(occurrences) {
		if(mOccurrences != occurrences) {
	   		mOccurrences = occurrences;
	  		self.setChanged(true);
		}
	}

	self.getDiscrete = function() { 
	 	return mDiscrete; 
	}

	self.setDiscrete = function(discrete) {
		if(mDiscrete != discrete) {
	   		mDiscrete = discrete;
	  		self.setChanged(true);
		}
	}
	  
	self.getNormalized = function() { 
	  	return mNormalized; 
	};

	self.setNormalized = function(normalized) {
	  	if(mNormalized != normalized) {
	  		mNormalized = normalized;
	  		mBars = null;
	  		self.setChanged(true);
	  	}
	};

	self.getBinWidth = function() {
	  	return mBinWidth;
	};

	self.setBinWidth = function(width) {
	  	if(mBinWidth != width) {
	  		mBinWidth = width;
	  		mBars = null;
	  		self.setChanged(true);
	  	}
	};

	self.getFixBin = function() {
	  	return mFixBin;
	};

	self.setFixBin = function(fixbin) {
	  	if(mFixBin != fixbin) {
	  		mFixBin = fixbin;
	  		mBars = null;
	  		self.setChanged(true);
	  	}
	};
		
	/**
	 * Computes the histogram as a sequence of [x,height]
	 */      
    self.getBars = function() {
      if (mBars!=null) return mBars;
      if (mCurrentInput.length==0) {
        mBars = [[0,0]];
        return mBars;
      }
      
	  mBars = [];
	  mCurrentInput = mCurrentInput.sort(function(a, b){return a-b});
      // minimum bin
	  var minbin;
	  var halfBin = mBinWidth/2;
	
	  if (mBinWidth == 0) minbin = mCurrentInput[0];
	  else if (mFixBin-halfBin > mCurrentInput[0]) minbin = mFixBin - (Math.floor((mFixBin-halfBin-mCurrentInput[0])/mBinWidth) + 1) * mBinWidth;
	  else if (mFixBin+halfBin < mCurrentInput[0]) minbin = mFixBin + (Math.floor((mCurrentInput[0]-mFixBin-halfBin)/mBinWidth)    ) * mBinWidth;
	  else minbin = mFixBin;
		
		//console.log ("inputs = "+input);
		//console.log ("minbin = "+minbin);
      // iterator
	  var cbin = minbin;	
	  var count = 0;
	  var nInput = mCurrentInput.length;
	
      for (var j=0; j < nInput; j++) {
	    if (mCurrentInput[j] <= cbin + halfBin) count ++
	    else {
		  if (count > 0) {
			if (mNormalized) count /= nInput;
			mBars.push([cbin,count]);
		  }
		  if (mBinWidth == 0) cbin = mCurrentInput[j];
		  else cbin = cbin + (Math.floor((mCurrentInput[j]-cbin-halfBin)/mBinWidth) + 1) * mBinWidth;
		  count = 1;
	    }
      }
      // last bar
      if (count > 0) {
		if (mNormalized) count /= nInput;
		mBars.push([cbin,count]);
	  }
	  return mBars;
	}
	     
	/**
	 * Returns bounds for an element
	 * @override
	 * @method getBounds
	 * @return Object{left,rigth,top,bottom}
	 */
	self.getBounds = function(element) {
	  	self.getBars();
	  	var nBars = mBars.length;
		var xmin = mBars[0][0]; 
		var xmax = mBars[nBars-1][0];
		var ymin = 0; 
		var ymax = 0;
		for (var j=0; j<nBars; j++) ymax = Math.max(ymax,mBars[j][1]);

	    var x = self.getX(), y = self.getY();
	    var sx = self.getSizeX(), sy = self.getSizeY();
	  	var mx = sx/2, my = sy/2;  	
	  	var d = self.getRelativePositionOffset(sx,sy);
	  	
		return {
			left: ((x+d[0])-mx)+xmin*sx,
			right: ((x+d[0])-mx)+xmax*sx,
			top: ((y+d[1])-my)+ymax*sy,
			bottom: ((y+d[1])-my)+ymin*sy
		}
	};  
	
	/**
	 * Returns bounds for an element
	 * @override
	 * @method getBounds
	 * @return Object{left,rigth,top,bottom}
	 */
	self.getBoundsOld = function(element) {
	  	var xmin, xmax, ymin, ymax;
	  	  	
	    var len = mCurrentInput.length;
	    if(len == 0) {
	    	xmin = xmax = ymin = ymax = 0;
	    }             
	    else {
	    	// order and get min and max
	    	console.log ("mCurrentInput="+mCurrentInput);
	    	var ordered = mCurrentInput.sort(function(a, b){return a-b});
	    	var nInput = ordered.length;
			xmin = ordered[0]; xmax = ordered[nInput-1];
			
			// count elements
			ymax = 0; ymin = 0;    	
			var counts = [];
			ordered.forEach(function(x) { counts[x] = (counts[x] || 0)+1; });
			if (mNormalized) {
			  for(var y in counts) {
			    counts[y] /= nInput;
			  }
			} 
			 
			for(var y in counts) { 
				if(counts[y]>ymax) ymax = counts[y]; 
			}									
		}    
	    var x = self.getX(), y = self.getY();
	    var sx = self.getSizeX(), sy = self.getSizeY();
	  	var mx = sx/2, my = sy/2;  	
	  	var d = self.getRelativePositionOffset(sx,sy);
	  	
		return {
			left: ((x+d[0])-mx)+xmin*sx,
			right: ((x+d[0])-mx)+xmax*sx,
			top: ((y+d[1])-my)+ymax*sy,
			bottom: ((y+d[1])-my)+ymin*sy
		}
	};  
              
	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.Histogram.registerProperties(self, controller);
	};
  
  	self.copyTo = function(element) {
    	EJSS_DRAWING2D.Histogram.copyTo(self,element);
  	};
  
	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	self.getStyle().setFillColor("Blue");
	self.getStyle().setLineColor("Black");
  	self.setRelativePosition("SOUTH_WEST");

	return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Image
 * @class Image 
 * @constructor  
 */
EJSS_DRAWING2D.Image = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy
  	
		dest.setImageUrl(source.getImageUrl());
  	},


    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class

      controller.registerProperty("ImageUrl",element.setImageUrl);
      controller.registerProperty("Encode",element.setEncode);
    },
};

/**
 * Creates a 2D Segment
 * @method image
 */
EJSS_DRAWING2D.image = function (name) {
  var self = EJSS_DRAWING2D.element(name);

  var mUrl = "";		// image url
  var mCode = "";
  var mChangedImage = false;	// indica si la fuente de la imagen ha cambiado 

  self.getClass = function() {
  	return "ElementImage";
  };

  self.setImageUrl = function(url) {
    var set = self.getSet(); 
  	var resPathFunction = (set!=null) ? set.getResourcePath : self.getResourcePath; 
  	if (resPathFunction!=null) {
  	  url = resPathFunction(url);
  	  //alert ("ImageUrl set to = "+url+"\n");
  	}
  	else console.log ("No getResourcePath function for "+self.getName()+". Texture = "+url);
  	if(mUrl != url) {
  		mUrl = url;
  		self.setChanged(true);
  		self.setChangedImage(true);
  	}
  }	;

  self.getImageUrl = function() {
  	return mUrl;
  };

  self.forceImageUrl = function(url) {
    console.log ("Setting image to " +url);
    mUrl = window.URL.createObjectURL(url);
  	self.setChanged(true);
  	self.setChangedImage(true);
  };
  
  self.setChangedImage = function(changed) {
  	mChangedImage = changed;
  };

  self.getChangedImage = function() {
  	return mChangedImage;
  };
  
  self.setEncode = function(code) {
  	if(mCode.length != code.length) { // less cost
  		mCode = code;
  		self.setChanged(true);
  		self.setChangedImage(true);
  	}
  };

  self.getEncode = function() {
  	return mCode;
  };
  
  self.getImageData = function(callback) {
    var size = self.getPixelSizes();     

  	var img = new Image();
  	img.src = mUrl;
  	
  	var canvas = document.createElement('canvas');
	var context = canvas.getContext('2d');
	canvas.width = Math.abs(size[0])+1;
	canvas.height = Math.abs(size[1])+1;
	
	if (callback) {
	  	img.onload = function() {
			context.drawImage(img,  
			  		0, 0, img.width, img.height, 
			        0, 0, canvas.width, canvas.height);
		  	callback(context.getImageData(0, 0, canvas.width, canvas.height));
		}
  	} else {
		context.drawImage(img,  
		  		0, 0, img.width, img.height, 
		        0, 0, canvas.width, canvas.height);
		return context.getImageData(0, 0, canvas.width, canvas.height);  		
  	}
  }

  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.Image.registerProperties(self,controller);
  };
  
  self.copyTo = function(element) {
    EJSS_DRAWING2D.Image.copyTo(self,element);
  };
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//ImageSet
//---------------------------------

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * ImageSet
 * @class ImageSet 
 * @constructor  
 */
EJSS_DRAWING2D.ImageSet = {
    
    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING2D.ElementSet;
      
      ElementSet.registerProperties(set,controller);
      
      controller.registerProperty("ImageUrl", 
          function(v) { set.setToEach(function(element,value) { element.setImageUrl(value); }, v); }
      );
    }    
};


/**
 * Creates a set of Segments
 * @method imageSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING2D.imageSet = function (mName) {
  var self = EJSS_DRAWING2D.elementSet(EJSS_DRAWING2D.image, mName);

  // Static references
  var ImageSet = EJSS_DRAWING2D.ImageSet;		// reference for ImageSet

  /**
   * Registers properties in a ControlElement
   * @method registerProperties
   * @param controller A ControlElement that becomes the element controller
   */
  self.registerProperties = function(controller) {
    ImageSet.registerProperties(self,controller);
  };

  return self;
};/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas analyticCurve
 */
EJSS_CANVASGRAPHICS.analyticCurve = function(mContext, mElement) {  

	// draw analyticCurve
	function drawAnalyticCurve(points, x, y, sx, sy) {
		mContext.beginPath();
		for(var i=0; i<points.length; i++) {
		  var point = points[i];	  
	      var xx = x + point[0]*sx;
	      var yy = y + point[1]*sy;
	      var type = point[2];		
		  if ((i==0) || (type == 0))
		  	mContext.moveTo(xx,yy);		// move 
	      else       	
	      	mContext.lineTo(xx,yy);		// line
		}  	  	
	}  	

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;

	// calculate points	    	 
	var numPoints = mElement.getNumPoints();
	var mMinimun = mElement.getMinimun();
	var mMaximun = mElement.getMaximun();
		var min = ( (typeof mMinimun == "undefined" || mMinimun === null) ?  mElement.getPanel().getRealWorldXMin() : mMinimun);
		var max = ( (typeof mMaximun == "undefined" || mMaximun === null) ?  mElement.getPanel().getRealWorldXMax() : mMaximun);
	
//	var min = (mElement.getMinimun()? mElement.getMinimun() : mElement.getPanel().getRealWorldXMin());
//	var max = (mElement.getMaximun()? mElement.getMaximun() : mElement.getPanel().getRealWorldXMax());
	var vble = mElement.getVariable();
	var fx = mElement.getFunctionX();
	var fy = mElement.getFunctionY();	
   	var parser = EJSS_DRAWING2D.functionsParser();
   	
    var exprfx;
	var exprfy;
	var mustReturn = false;
	try {
	  exprfx = parser.parse(fx);
	}
	catch (errorfx) {
  	  console.log ("Analytic curve error parsing FunctionX: "+fx);
	  mustReturn = true;
	}
	if (!mustReturn) {
	  try {
	    exprfy = parser.parse(fy);
	  }
	  catch (errorfy) {
  	    console.log ("Analytic curve error parsing FunctionY: "+fy);
	   	mustReturn = true;
	  }
	}
	if (mustReturn) {
	  mElement.getController().invokeAction("OnError");
	  return;
	}
	
   	var step = (max-min)/(numPoints-1);
   	
	var points = [];
    var vblevalue = {};
    var parameters = mElement.getParameters();
    for (var param in parameters) { 
      vblevalue[param] = parameters[param];
	 }
    
   	try {
   	  for(var j=0, i=min; i<=max; i+=step) {
   		vblevalue[vble] = i;
   		  var fxvalue = exprfx.evaluate(vblevalue);
   		  var fyvalue = exprfy.evaluate(vblevalue);
	   	  if(!isNaN(fxvalue) && !isNaN(fyvalue)) {   		
	   		points[j] = [];
	    	points[j][0] = fxvalue;	
			points[j++][1] = fyvalue;
		  }
		}
	}
	catch(error) {
	  mElement.getController().invokeAction("OnError");
	 } // do not complain
	
	// draw points
    drawAnalyticCurve(points, x-mx, y-my, size[0], size[1])     

	// set style
    var style = mElement.getStyle(); 
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas arrow
 */
EJSS_CANVASGRAPHICS.arrow = function(mContext, mElement) {  
	var Arrow = EJSS_DRAWING2D.Arrow;

	function createMarkDef(px, py, mark, width, height, stroke, color, sizex, sizey) {
		mContext.save();	
		// measure
		width /= 2;
		height /= 2;
		var angle = (sizex==0)? Math.PI/2 : Math.atan(sizey/sizex);
		angle += ((sizex < 0)?-Math.PI/2:Math.PI/2);
		switch(mark) {
			case Arrow.TRIANGLE:
	            mContext.beginPath();
	            mContext.translate(px,py);
	            mContext.rotate(angle);
	            mContext.moveTo(0,height/2);
	            mContext.lineTo(width,height/2);
	            mContext.lineTo(0,-height/2);
	            mContext.lineTo(-width,height/2);
	            mContext.closePath();           
				break;			
			case Arrow.ANGLE:
	            mContext.beginPath();
	            mContext.translate(px,py);
	            mContext.rotate(angle);
	            mContext.moveTo(width,height);
	            mContext.lineTo(0,0);
	            mContext.lineTo(-width,height);
				break;			
			case Arrow.LINE:
	            mContext.beginPath();
	            mContext.translate(px,py);
	            mContext.rotate(angle);
	            mContext.moveTo(width,0);
	            mContext.lineTo(-width,0);
				break;			
			case Arrow.RECTANGLE:
	            mContext.beginPath();
	            mContext.translate(px,py);
	            mContext.rotate(angle);
	            mContext.moveTo(0,-height);
	            mContext.lineTo(width,-height);
	            mContext.lineTo(width,height);
	            mContext.lineTo(-width,height);
	            mContext.lineTo(-width,-height);
	            mContext.closePath();
				break;			
			case Arrow.POINTED:
	            mContext.beginPath();
	            mContext.translate(px,py);
	            mContext.rotate(angle);
	            mContext.moveTo(0,height/4);
	            mContext.lineTo(width,height);
	            mContext.lineTo(0,-height/4);
	            mContext.lineTo(-width,height);
	            mContext.closePath();   	            	       
				break;
			case Arrow.CIRCLE:
				var size = (width < height) ? width : height;
	            mContext.beginPath();
	            mContext.translate(px,py);
	            mContext.arc(0, 0, size, 0, 2 * Math.PI, false);	            
				break;
			case Arrow.DIAMOND:
	            mContext.beginPath();
	            mContext.translate(px,py);
	            mContext.rotate(angle);
	            mContext.moveTo(0,-height);
	            mContext.lineTo(width,0);
	            mContext.lineTo(0,height);
	            mContext.lineTo(-width,0);
	            mContext.closePath();
				break;				
			case Arrow.INVTRIANGLE:					
	            mContext.beginPath();
	            mContext.translate(px,py);
	            mContext.rotate(angle);
	            mContext.moveTo(0,-height/2);
	            mContext.lineTo(-width,-height/2);
	            mContext.lineTo(0,height/2);
	            mContext.lineTo(width,-height/2);
	            mContext.closePath();           
				break;			
			case Arrow.INVANGLE:
	            mContext.beginPath();
	            mContext.translate(px,py);
	            mContext.rotate(angle);
	            mContext.moveTo(-width,-height);
	            mContext.lineTo(0,0);
	            mContext.lineTo(width,-height);
				break;	
										
			case Arrow.WEDGE:
			case Arrow.CURVE:
				console.log('Type of Arrow: not supported');
				break;								
		}		
		
		// set style
		if (color != 'none') {
			if(mark != Arrow.ANGLE && mark != Arrow.INVANGLE) {
				mContext.fillStyle = color;
		    	mContext.fill();
			}
			if(mark != Arrow.POINTED) {
		      	mContext.lineWidth = stroke;
			    mContext.strokeStyle = color;
			    mContext.stroke();		
			}
	    }
	    mContext.restore();
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;

    // draw
	if((size[0] != 0) || (size[1] != 0)) {
		// draw line
	  	mContext.beginPath();
		mContext.moveTo(x-mx, y-my);
	    mContext.lineTo(x+mx, y+my);
	
		// set style
	    var style = mElement.getStyle(); 
		if (style.getDrawFill()  && style.getFillColor() != 'none') {
	      mContext.fillStyle = style.getFillColor();
	      mContext.fill();
	    }
	    if (style.getDrawLines()) {
	      mContext.lineWidth = style.getLineWidth();
	      mContext.strokeStyle = style.getLineColor();
	      mContext.stroke();
	    }	

		// create end mark	
		if(mElement.getMarkEnd() != Arrow.NONE) {
			var endMax = (mElement.getMarkProportion() * EJSS_TOOLS.Mathematics.norm([size[0],size[1]])) ;
			var endW = (endMax != 0 && endMax < mElement.getMarkEndWidth()) ? endMax : mElement.getMarkEndWidth();
			var endH = (endMax != 0 && endMax < mElement.getMarkEndHeight()) ? endMax : mElement.getMarkEndHeight();
			var color = (mElement.getMarkEndColor() == "none") ? mElement.getStyle().getLineColor() : mElement.getMarkEndColor();
			var stroke = (mElement.getMarkEndStroke() < 0) ? mElement.getStyle().getLineWidth() : mElement.getMarkEndStroke();   

			// mElement.getMarkEndOrient() is ignored
			createMarkDef(x+mx, y+my, mElement.getMarkEnd(), endW, endH, stroke, color, mx, my);
		}
				
		// create mid mark
		if(mElement.getMarkMiddle() != Arrow.NONE) {
			var midMax = (mElement.getMarkProportion() * EJSS_TOOLS.Mathematics.norm([size[0],size[1]])) ;
			var midW = (midMax != 0 && midMax < mElement.getMarkMiddleWidth()) ? midMax : mElement.getMarkMiddleWidth();
			var midH = (midMax != 0 && midMax < mElement.getMarkMiddleHeight()) ? midMax : mElement.getMarkMiddleHeight();
			var color = (mElement.getMarkMiddleColor() == "none") ? mElement.getStyle().getLineColor() : mElement.getMarkMiddleColor();
			var stroke = (mElement.getMarkMiddleStroke() < 0) ? mElement.getStyle().getLineWidth() : mElement.getMarkMiddleStroke();   

			// mElement.getMarkEndOrient() is ignored
			createMarkDef(x, y, mElement.getMarkMiddle(), midW, midH, stroke, color, mx, my);
		}
		
		// create start mark
		if(mElement.getMarkStart() != Arrow.NONE) {
			var stMax = (mElement.getMarkProportion() * EJSS_TOOLS.Mathematics.norm([size[0],size[1]])) ;
			var stW = (stMax != 0 && stMax < mElement.getMarkStartWidth()) ? stMax : mElement.getMarkStartWidth();
			var stH = (stMax != 0 && stMax < mElement.getMarkStartHeight()) ? stMax : mElement.getMarkStartHeight();
			var color = (mElement.getMarkStartColor() == "none") ? mElement.getStyle().getLineColor() : mElement.getMarkStartColor();
			var stroke = (mElement.getMarkStartStroke() < 0) ? mElement.getStyle().getLineWidth() : mElement.getMarkStartStroke();   

			// mElement.getMarkStartOrient() is ignored
			createMarkDef(x-mx, y-my, mElement.getMarkStart(), stW, stH, stroke, color, mx, my);
		}		    	
	}
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas axis
 */
EJSS_CANVASGRAPHICS.axis = function(mContext, mElement) {  

	// draw axis line
	function drawLine(x, y, mx, my, orient, inverted) {
		if(orient == EJSS_DRAWING2D.Axis.AXIS_VERTICAL) {	// vertical axis
			if(inverted) {
				my = -my;
				y = -y;	
			}
			mContext.moveTo(x,y-my);
			mContext.lineTo(x,y+my);
		} else {	// horizontal axis
			mContext.moveTo(x-mx,y);
			mContext.lineTo(x+mx,y);
		}		
	}

	function drawTick(x, y, ticksize, orient) {
		if(orient == EJSS_DRAWING2D.Axis.AXIS_VERTICAL) {	// vertical axis
			mContext.moveTo((x-ticksize/2),y);
			mContext.lineTo((x+ticksize/2),y);
		} else {
			mContext.moveTo(x,(y-ticksize/2));
			mContext.lineTo(x,(y+ticksize/2));
		}					
	}
	
	// text for tick in position (x,y) with font and mode
	function tickText (x, y, text, font, horizontal) {
		mContext.save();

	 	var font = mElement.getFont();
	 	var fondtxt = "";
	 	fondtxt += (font.getFontStyle() != 'none')? font.getFontStyle() + ' ':'';
	 	fondtxt += (font.getFontWeight() != 'none')? font.getFontWeight() + ' ':'';
	 	fondtxt += font.getFontSizeString() + 'px ';
	 	fondtxt += font.getFontFamily();
		mContext.font = fondtxt;	
		mContext.fillStyle = font.getFillColor();

		if (horizontal) {
		  x -= mContext.measureText(text).width/2 + 0.5;
		}	
		else { // vertical axis
		  y += font.getFontSize()/2; // - 0.5;
		}

		mContext.fillText(text, x, y);		
				
		mContext.restore();		    
	}		
	
	if(mElement.getShow()) { // show

		// get position of the element center 
	    var pos = mElement.getPixelPosition();
	    var size = mElement.getPixelSizes();     
	    var offset = mElement.getRelativePositionOffset(size);  
	    var x = pos[0]+offset[0];
	    var y = pos[1]+offset[1];
		
		// get sizes 		
	    var mx = size[0]/2;
	    var my = size[1]/2;
				
		// properties	    
	    var style = mElement.getStyle();		// element style   	 	
		var orient = mElement.getOrient();		// axis orientation (vertical or horizontal)
		var inverted = mElement.getInvertedScaleY();
	    
	    // draw the line for axis
	  	mContext.beginPath();
		drawLine(x, y, mx, my, orient, inverted);
		 		
		// get axis size in pixel
		var segsize = (orient == EJSS_DRAWING2D.Axis.AXIS_VERTICAL)? Math.abs(size[1]):Math.abs(size[0]);		
		 		
		// draw axis (based on ticks mode)
		var ticksmode = mElement.getTicksMode();			// decimal or logarithmic
		if (ticksmode == EJSS_DRAWING2D.Axis.SCALE_LOG) { // logarithmic

		    var scale = mElement.getScale();		// axis scale 			
			if(scale[0] > 0 && scale[1] > 0) {  // valid scale			
				// get number of axis ticks
				var ticks = mElement.getTicks(); 	
	
			    // scale
				var scalePrecision = mElement.getScalePrecision();	// number of decimals for text
				
				// draw ticks 
				var ticksize = mElement.getTicksSize();				// ticks size in pixels
				EJSS_GRAPHICS.GraphicsUtils.drawLogTicks(x, y, mx, my, segsize, ticksize, scale, ticks, orient, drawTick);		
			
				// draw ticks text 
				var font = mElement.getFont();
			    var textPosition = mElement.getTextPosition();		// text position (UP or DOWN)
				EJSS_GRAPHICS.GraphicsUtils.drawLogTicksText (x, y, mx, my, ticks, ticksize, scale, scalePrecision, font, textPosition, orient, tickText)			
			}
			
		} else if (ticksmode == EJSS_DRAWING2D.Axis.SCALE_NUM) { // decimal
			 			 		
			// calculate step in pixels	
		    var step = 0;     // step for ticks (pixels)
		    var tickstep = 0; // step for ticks (real units)
			if (!mElement.getAutoTicks()) {		// no auto-ticks
		    	var ticks = mElement.getTicks();	// number of ticks
				// whether the number of ticks exits, changes step for ticks and scale 
			    if (ticks != 0) { step = segsize/ticks; } else {
			    	step = mElement.getStep();
			    	tickstep = mElement.getTickStep(); 
			    } 	    	
			} else {	// auto-ticks
				var stepmin = mElement.getAutoStepMin();		// step min in pixels
				var ticksrange = mElement.getAutoTicksRange();	// ticks range
				// find step based on ticks range
				for(var i=ticksrange.length-1; i>=0; i--)	{	
					step = Math.abs(segsize/ticksrange[i]);
					if (step*1.001 >= stepmin) break;
				}
			}
		    
			var scalePrecision = mElement.getScalePrecision();	// number of decimals for text
			var scale = mElement.getScale();		// axis scale 	
			// values for scale
		    if(tickstep == 0) {
				var scalestep = Math.abs((scale[1] - scale[0]) * step / segsize);  // step in axis scale
			} else {
				var scalestep = tickstep;
				step = Math.abs((scalestep * segsize) / (scale[1] - scale[0])); 
			}			
			
			// adjust step to decimals of precision
			var decimals = Math.pow(10,scalePrecision);
			var scalestepTmp = Math.round(scalestep * decimals) / decimals;
			if(scalestepTmp > 0) {
				scalestep = scalestepTmp; 
				step = Math.abs(scalestepTmp * segsize / (scale[1] - scale[0]));
			}
			
			// check fixed tick
			var fixedTicks = mElement.getFixedTick();	  
			var tickfixed = scale[1];
			if (!isNaN(fixedTicks)) {
			  if (fixedTicks < scale[0]) tickfixed = fixedTicks + (Math.floor((scale[0]-fixedTicks)/scalestep)+1)*scalestep;
			  else if (fixedTicks > scale[1])  tickfixed = fixedTicks - (Math.floor((fixedTicks-scale[1])/scalestep)+1)*scalestep;
		      else tickfixed = fixedTicks; 
			}		
	
			// tick fixed in axis scale
			var scaleshift = Math.abs((scale[0] - tickfixed) % scalestep);						
			var dist = Math.abs(scaleshift-scalestep);	// fitting shift
			if(scaleshift < 0.001 || dist < 0.001) scaleshift = 0;						
			var shift = segsize * scaleshift / Math.abs(scale[1] - scale[0]);	// shift in pixels				
										    		
			// draw ticks based on step and shift
			var ticksize = mElement.getTicksSize();				// ticks size in pixels
			EJSS_GRAPHICS.GraphicsUtils.drawDecTicks(x, y, mx, my, ticksize, step, shift, orient, inverted, drawTick);		
		
			// draw ticks text based on scaleshift, scalestep and scale
			var font = mElement.getFont();
		    var textPosition = mElement.getTextPosition();		// text position (UP or DOWN)
			EJSS_GRAPHICS.GraphicsUtils.drawDecTicksText (x, y, mx, my, ticksize, step, shift, scale, scalePrecision, scalestep, scaleshift, font, textPosition, orient, inverted, tickText)
								
		}

		// set style
	    var style = mElement.getStyle(); 
		if (style.getDrawFill() && style.getFillColor() != 'none') {
	      mContext.fillStyle = style.getFillColor();
	      mContext.fill();
	    }
	    if (style.getDrawLines()) {
	      mContext.lineWidth = style.getLineWidth();
	      mContext.strokeStyle = style.getLineColor();
	      mContext.stroke();
	    }						
	}

}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas byteRaster
 */
EJSS_CANVASGRAPHICS.byteRaster = function(mContext, mElement) {
	
	if(mElement.getDataChanged()) { // new data or colors
		var data = mElement.getData();
		var colors = mElement.getColorMapper().getColors();			
		var xlen = data.length;
		var ylen = data[0].length;
		var num = mElement.getColorMapper().getNumberOfColors();

		// get position of the mElement center 
	    var pos = mElement.getPixelPosition();
	    var size = mElement.getPixelSizes();     
	    var offset = mElement.getRelativePositionOffset(size);  
	    var x = pos[0]+offset[0];
	    var y = pos[1]+offset[1];
		
		// get half sizes 		
	    var mx = Math.abs(size[0]/2);
	    var my = Math.abs(size[1]/2);    
		
		var idcanvas = mContext.canvas.id;
		
		// using worker
		EJSS_SVGGRAPHICS.Utils.ImageDataCanvas (xlen, ylen, num, data, colors,
			function(img) {
				// I do not know why the reference mContext is lost
				var canvas = document.getElementById(idcanvas);
			  	mContext = canvas.getContext("2d");
				mContext.putImageData(img,x-mx,y-my);
			});			    	    
	    mElement.setDataChanged(false);

	}

}

/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas canvas
 */
EJSS_CANVASGRAPHICS.canvas = function(mContext, mElement) {  
  
	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	var dim = mElement.getDimensions();
  	var deltaX = dim[1] - dim[0];
  	var deltaY = dim[3] - dim[2];
    
	var pixWth = Math.abs(size[0]);
	var pixHgt = Math.abs(size[1]);
	var pxPerUnitX = pixWth / deltaX;
	var pxPerUnitY = pixHgt / deltaY;
	var pxTopLeftX = x - pixWth / 2;
	var pxTopLeftY = y + pixHgt / 2;
   
    //draws all objects within this.drawables
    var drawables = mElement.getDrawables();
    for (var i = 0; i < drawables.length; i++) {
      if(typeof drawables[i].imageField != "undefined")
        EJSS_GRAPHICS.GraphicsUtils.drawImageField(mContext, 
        	pxTopLeftX, pxTopLeftY, xMin, yMin, pixWth, pixHgt, pxPerUnitX, pxPerUnitY, drawables[i]);
      else if(typeof drawables[i].run != "undefined")
      	drawables[i].draw(mContext, pxTopLeftX, pxTopLeftY, xMin, yMin, pixWth, pixHgt, pxPerUnitX, pxPerUnitY);
    }
      
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas cellLattice
 */
EJSS_CANVASGRAPHICS.cellLattice = function(mContext, mElement) {  

	function drawGrid(left, top, right, bottom, stepx, stepy) {
		mContext.beginPath();
		
		// vertical lines
		if (stepx == 0) stepx = Math.abs(right-left);
	    for (var i = left; i <= right; i = i+stepx) {
	  	  mContext.moveTo(i,top);
	  	  mContext.lineTo(i,bottom); 
	    }
	    // horizontal lines
	    if (stepy == 0) stepy = Math.abs(top-bottom);
	    for (var i = bottom; i >= top; i = i-stepy) {
	  	  mContext.moveTo(left,i);
	  	  mContext.lineTo(right,i); 
	    }
	}  
	
	function rectCell(x, y, sx, sy, fill) {
		mContext.beginPath();
			    		
		var mx = sx/2, my = sy/2;
		mContext.moveTo(x-mx,y+my);
		mContext.lineTo(x+mx,y+my);
		mContext.lineTo(x+mx,y-my);
		mContext.lineTo(x-mx,y-my);
		mContext.lineTo(x-mx,y+my);
		
		mContext.fillStyle = fill;
        mContext.fill();
      	mContext.lineWidth = 1;
     	mContext.strokeStyle = fill;
     	mContext.stroke();
	}

	function gridCell(x, y, sx, sy, stepx, stepy, style) {
		// set attributes
		var mx = sx/2, my = sy/2;    	   	    
		var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
		drawGrid(left, top, right, bottom, stepx, stepy);
						
		// set style
		if (style.getDrawFill() && style.getFillColor() != 'none') {
	      mContext.fillStyle = style.getFillColor();
	      mContext.fill();
	    }
	    if (style.getDrawLines()) {
	      mContext.lineWidth = style.getLineWidth();
	      mContext.strokeStyle = style.getLineColor();
	      mContext.stroke();
	    }	
	}

	// get position of the mElement center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;

	// draw cells
	var style = mElement.getStyle();     
	var data = mElement.getData();
	var colors = mElement.getColorMapper().getColors();
	var xlen = data.length;
	var ylen = data[0].length;

    var stepx = Math.abs(size[0]/xlen);
    var stepy = Math.abs(size[1]/ylen);
   	
  	var left = x-mx+stepx/2, bottom = y-my-stepy/2;		
  	for(var i=0; i<ylen; i++) { 
  		for(var j=0; j<xlen; j++) {
  			// draw rectangle
  			rectCell(left+stepx*j, bottom-stepy*i, stepx, stepy, colors[data[j][i]]);
  		}  		
  	}
  	 	
  	// draw grid
  	var showGrid = mElement.getShowGrid();
	if (showGrid) gridCell(x, y, size[0], size[1], stepx, stepy, style);	
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas cursor
 */
EJSS_CANVASGRAPHICS.cursor = function(mContext, mElement) {  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    
  	// type of cursor
	var cursorType = mElement.getCursorType();

    // cursor size
    var panel = mElement.getGroupPanel();
    var bounds = panel.getRealWorldCoordinates(); // xmin,xmax,ymin,ymax
	var bottomLeft = panel.toPixelPosition([bounds[0],bounds[2]]);    
	var topRight   = panel.toPixelPosition([bounds[1],bounds[3]]);    

  	mContext.beginPath();	    
	if (cursorType!=EJSS_DRAWING2D.Cursor.VERTICAL) {
	  var x1 = bottomLeft[0], x2 = topRight[0];
	  var y = pos[1];
	  // keep the cursor inside the panel
	  if (y>bottomLeft[1]) y = bottomLeft[1];
	  else if (y<topRight[1]) y = topRight[1];
	  
	  mContext.moveTo(x1,y);
	  mContext.lineTo(x2,y);
	}

	if (cursorType!=EJSS_DRAWING2D.Cursor.HORIZONTAL) {
	  var x = pos[0];
	  var y1 = bottomLeft[1], y2 = topRight[1];
	  // keep the cursor inside the panel
	  if (x<bottomLeft[0]) x = bottomLeft[0];
	  else if (x>topRight[0]) x = topRight[0];
	  
	  mContext.moveTo(x,y1);
	  mContext.lineTo(x,y2);
	}
      
	// set style
    var style = mElement.getStyle(); 
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas ellipse
 */
EJSS_CANVASGRAPHICS.ellipse = function(mContext, mElement) {  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var xm = pos[0]+offset[0];
    var ym = pos[1]+offset[1];
	
	// get sizes 		
    var w = Math.abs(size[0]);
    var h = Math.abs(size[1]);

    // draw
    var kappa = .5522848,
      // x = x-middle, y = y-middle
      x = xm - w / 2,       // x-middle
      y = ym - h / 2;       // y-middle      
      ox = (w / 2) * kappa, // control point offset horizontal
      oy = (h / 2) * kappa, // control point offset vertical
      xe = x + w,           // x-end
      ye = y + h,           // y-end

  	mContext.beginPath();
  	mContext.moveTo(x, ym);
  	mContext.bezierCurveTo(x, ym - oy, xm - ox, y, xm, y);
  	mContext.bezierCurveTo(xm + ox, y, xe, ym - oy, xe, ym);
  	mContext.bezierCurveTo(xe, ym + oy, xm + ox, ye, xm, ye);
  	mContext.bezierCurveTo(xm - ox, ye, x, ym + oy, x, ym);
      
	// set style
    var style = mElement.getStyle(); 
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas grid
 */
EJSS_CANVASGRAPHICS.grid = function(mContext, mElement) {  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;

    var showx = mElement.getShowX();
    var showy = mElement.getShowY(); 
    
	// draw axis X (based on ticks mode)
	if(showx) {
		mContext.beginPath();

		var ticksXmode = mElement.getTicksXMode();			// decimal or logarithmic
		if (ticksXmode == EJSS_DRAWING2D.Grid.SCALE_LOG) { // logarithmic
				
		    // scale
		    var scaleX = mElement.getScaleX();		// X axis scale 	
			
			if(scaleX[0] > 0 && scaleX[1] > 0) {  // valid scale
				// get number of axis ticks
			    var ticksX = mElement.getTicksX();
			    
				// set attributes
			    var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
		    	EJSS_GRAPHICS.GraphicsUtils.drawLogGrid(left, top, right, bottom, ticksX, scaleX, 1,
					function(xx,yy) { mContext.moveTo(xx,yy) }, 
					function(xx,yy) { mContext.lineTo(xx,yy) }); 
			} 
					    		
		} else if (ticksXmode == EJSS_DRAWING2D.Grid.SCALE_NUM) { // decimal
	    	    
		 	// create path	    	   	    
		   	var stepx = 0;	// step for ticks
		   	var tickstepx = 0;
			if (!mElement.getAutoTicksX()) {	  // no auto-ticks    
			    var ticksX = mElement.getTicksX();
			    // whether the number of sticks exits, changes step for ticks
			    if (ticksX != 0) { stepx = Math.abs(size[0]/ticksX); } else {
			    	stepx = mElement.getStepX();
			    	tickstepx = mElement.getTickStepX(); 
			    } 	    	

			} else { // auto-ticks
				var stepxmin = mElement.getAutoStepXMin();		// step y min
				var ticksxrange = mElement.getAutoTicksRangeX();		// ticks x range
		
				// find stepx
				for(var i=ticksxrange.length-1; i>=0; i--)	{	
					stepx = Math.abs(size[0]/ticksxrange[i]);
					if (stepx >= stepxmin) break;
				}
			}	
			
		    // step for scale
			var scalePrecisionX = mElement.getScalePrecisionX();	// number of decimals for scale
		    var scaleX = mElement.getScaleX();		// X axis scale 	
		    if(tickstepx == 0) {
				var scalestepX = Math.abs((scaleX[1] - scaleX[0]) * stepx / size[0]);  // step in X axis scale
			} else {
				var scalestepX = tickstepx;
				stepx = Math.abs((scalestepX * size[0]) / (scaleX[1] - scaleX[0])); 
			}			
						
			// adjust step to decimals of precision	
			var decimalsX = Math.pow(10,scalePrecisionX);
			var scalestepXTmp = Math.round(scalestepX * decimalsX) / decimalsX;
			if(scalestepXTmp > 0) {
				scalestepX = scalestepXTmp;
				stepx = Math.abs(scalestepXTmp * size[0] / (scaleX[1] - scaleX[0]));
			}

			// check fixed tick
			var fixedTicks = mElement.getFixedTickX();	  
			var tickfixed = scaleX[1];
			if (!isNaN(fixedTicks)) {
			  if (fixedTicks < scaleX[0]) tickfixed = fixedTicks + (Math.floor((scaleX[0]-fixedTicks)/scalestepX)+1)*scalestepX;
			  else if (fixedTicks > scaleX[1])  tickfixed = fixedTicks - (Math.floor((fixedTicks-scaleX[1])/scalestepX)+1)*scalestepX;
		      else tickfixed = fixedTicks; 
			}		
				
			// tick fixed in axis scale
			var scaleshift = Math.abs((scaleX[0] - tickfixed) % scalestepX);						
			var dist = Math.abs(scaleshift-scalestepX);	// fitting shift
			if(scaleshift < 0.001 || dist < 0.001) scaleshift = 0;						
			var shiftx = Math.abs(size[0] * scaleshift / (scaleX[1] - scaleX[0]));	// shift in pixels				

			// set attributes
		    var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
	    	EJSS_GRAPHICS.GraphicsUtils.drawDecGrid(left, top, right, bottom, shiftx, stepx, 1,
					function(xx,yy) { mContext.moveTo(xx,yy) }, 
					function(xx,yy) { mContext.lineTo(xx,yy) }); 
		}		
		
		// set style
		mContext.lineWidth = mElement.getLineWidthX();
		mContext.strokeStyle = mElement.getLineColorX();
		mContext.stroke();
	}

	// draw axis Y (based on ticks mode)
	if(showy) {
		mContext.beginPath();

		var ticksYmode = mElement.getTicksYMode();			// decimal or logarithmic
		if (ticksYmode == EJSS_DRAWING2D.Grid.SCALE_LOG) { // logarithmic
						
		    // scale
		    var scaleY = mElement.getScaleY();		// Y axis scale 	

			if(scaleY[0] > 0 && scaleY[1] > 0) {  // valid scale
				// get number of axis ticks
			    var ticksY = mElement.getTicksY();
			
				// set attributes
			    var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
		    	EJSS_GRAPHICS.GraphicsUtils.drawLogGrid(left, top, right, bottom, ticksY, scaleY, -1,
					function(xx,yy) { mContext.moveTo(xx,yy) }, 
					function(xx,yy) { mContext.lineTo(xx,yy) }); 			
			}
					    		
		} else if (ticksYmode == EJSS_DRAWING2D.Grid.SCALE_NUM) { // decimal
	    
		 	// create path	    	   	    
		   	var stepy = 0;	// step for ticks
		   	var tickstepy = 0;
			if (!mElement.getAutoTicksY()) {	  // no auto-ticks    
			    var ticksY = mElement.getTicksY();
			    // whether the number of sticks exits, changes step for ticks
			    if (ticksY != 0) { stepy = Math.abs(size[1]/ticksY); } else {
			    	stepy = mElement.getStepY();
			    	tickstepy = mElement.getTickStepY(); 
			    } 	    	
			} else { // auto-ticks
				var stepymin = mElement.getAutoStepYMin();		// step y min		
				var ticksyrange = mElement.getAutoTicksRangeY();		// ticks y range
		
				// find stepy
				for(var i=ticksyrange.length-1; i>=0; i--)	{	
					stepy = Math.abs(size[1]/ticksyrange[i]);
					if (stepy >= stepymin) break;
				}						
			}	
			
		    // step for scale
			var scalePrecisionY = mElement.getScalePrecisionY();	// number of decimals for scale
		    var scaleY = mElement.getScaleY();		// Y axis scale 	
		    if(tickstepy == 0) {
				var scalestepY = Math.abs((scaleY[1] - scaleY[0]) * stepy / size[1]);  // step in Y axis scale 
			} else {
				var scalestepY = tickstepy;
				stepy = Math.abs((scalestepY * size[1]) / (scaleY[1] - scaleY[0])); 
			}			
			
			// adjust step to decimals of precision	
			var decimalsY = Math.pow(10,scalePrecisionY);
			var scalestepYTmp = Math.round(scalestepY * decimalsY) / decimalsY;
			if(scalestepYTmp > 0) {
				scalestepY = scalestepYTmp;
				stepy = Math.abs(scalestepYTmp * size[1] / (scaleY[1] - scaleY[0]));
			}
			
			// check fixed tick
			var fixedTicks = mElement.getFixedTickY();	  
			var tickfixed = scaleY[1];
			if (!isNaN(fixedTicks)) {
			  if (fixedTicks < scaleY[0]) tickfixed = fixedTicks + (Math.floor((scaleY[0]-fixedTicks)/scalestepY)+1)*scalestepY;
			  else if (fixedTicks > scaleY[1])  tickfixed = fixedTicks - (Math.floor((fixedTicks-scaleY[1])/scalestepY)+1)*scalestepY;
		      else tickfixed = fixedTicks; 
			}		
	
			// tick fixed in axis scale
			var scaleshift = Math.abs((scaleY[0] - tickfixed) % scalestepY);						
			var dist = Math.abs(scaleshift-scalestepY);	// fitting shift
			if(scaleshift < 0.001 || dist < 0.001) scaleshift = 0;						
			var shifty = Math.abs(size[1] * scaleshift / (scaleY[1] - scaleY[0]));	// shift in pixels				
																								            				            
			// set attributes
		    var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
	    	EJSS_GRAPHICS.GraphicsUtils.drawDecGrid(left, top, right, bottom, shifty, stepy, -1,
					function(xx,yy) { mContext.moveTo(xx,yy) }, 
					function(xx,yy) { mContext.lineTo(xx,yy) }); 
		}

		// set style
		mContext.lineWidth = mElement.getLineWidthY();
		mContext.strokeStyle = mElement.getLineColorY();
		mContext.stroke();
	}	
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas histogram
 */
EJSS_CANVASGRAPHICS.histogram = function(mContext, mElement) {  

	// draw area bar	
	function areaBar(barx1, barx2, bary, axis, barstyle) {		
		mContext.beginPath();
		mContext.moveTo(barx1,axis);
		mContext.lineTo(barx1,bary);
		mContext.lineTo(barx2,bary);
		mContext.lineTo(barx2,axis);

		// set style
		if (barstyle.getDrawFill() && barstyle.getFillColor() != 'none') {
		  mContext.fillStyle = barstyle.getFillColor();
		  mContext.fill();
		}
		if (barstyle.getDrawLines()) {
		  mContext.lineWidth = barstyle.getLineWidth();
		  mContext.strokeStyle = barstyle.getLineColor();
		  mContext.stroke();
		}		
	}  	

	// get position of the element center 	
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
	
	// get style        
	var style = mElement.getStyle(); 
        
	// get histogram parameters
	var discrete =  mElement.getDiscrete();
 	var halfBin = mElement.getBinWidth()/2;
 	var bars = mElement.getBars();

	var axisx = mElement.getGroupPanel().getPixelPositionWorldOrigin()[1];
	for (var j=0; j < bars.length; j++) { // create bars
	  var barx1, barx2;
	  var cbin = bars[j][0];
	  if (discrete) {
        barx1 = barx2 = (x-mx) + cbin*size[0];
	  } 
	  else {
		barx1 = (x-mx) + (cbin-halfBin)*size[0];	// Bar x
		barx2 = (x-mx) + (cbin+halfBin)*size[0];	// Bar x
	  }
	  var bary = (y-my) + bars[j][1]*size[1];  // Bar y
	  areaBar(barx1, barx2, bary, axisx, style);
    }
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas image
 */
EJSS_CANVASGRAPHICS.image = function(mContext, mElement, transformCallBack) {  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];

	// get sizes 		
    var mx = Math.abs(size[0]/2);
    var my = Math.abs(size[1]/2);
    
    // draw
    var img;
    if (!mElement.getChangedImage()) img = mElement.getCustomObject();
    if (img) {
		mContext.save();
		transformCallBack(mElement);
    	mContext.drawImage(img, (x-mx), (y-my), Math.abs(size[0]), Math.abs(size[1]));
		mContext.restore();
    }
    else{
    	img = new Image();
    	mElement.setCustomObject(img);
  	    img.onload = function () {
    			mContext.save();
    			transformCallBack(mElement);
    	    	mContext.drawImage(img, (x-mx), (y-my), Math.abs(size[0]), Math.abs(size[1]));
    			mContext.restore();
    	};
	 	var code = mElement.getEncode();
	 	if(code.length > 0) { // http://webcodertools.com/imagetobase64converter/Create
	 		img.src = 'data:png;base64,'+ code;
	 	} else {
	    	img.src = mElement.getImageUrl();
	    }
    }

    // indica que la fuente de la imagen ha sido actualizada  
    mElement.setChangedImage(false);
}  
/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas mesh
 */
EJSS_CANVASGRAPHICS.mesh = function(mContext, mElement) {  
  var mA; 			// int[]
  var mB; 			// int[]
  var mCells;		// {cellA, cellB, cellZ, cellVector}
  var mBoundary;	// {boundaryA (int[][]), boundaryB (int[][])}
  
  // fill cell
  function fillCell(xs, ys, numpoints, color, levelIndex) {
  	mContext.beginPath();

	for(var i=0; i<numpoints; i++) {
      if(i==0) 
        mContext.moveTo(xs[0], ys[0]);
      else 
      	mContext.lineTo(xs[i], ys[i]);
	}  	  	
    mContext.lineTo(xs[0], ys[0]);

  	var fill = color.indexToColor(levelIndex);

  	mContext.fillStyle = fill;
  	mContext.fill();
  	mContext.lineWidth = 1.25;
  	mContext.strokeStyle = fill;
  	mContext.stroke();
  }   
 
  // draw cell
  function drawCell(xs, ys, fill, style) {
  	mContext.beginPath();

	var numpoints = xs.length;
	for(var i=0; i<numpoints; i++) {
      if(i==0) 
        mContext.moveTo(xs[0], ys[0]);
      else 
      	mContext.lineTo(xs[i], ys[i]);
	}  	  	
    mContext.lineTo(xs[0], ys[0]);
    
	// set style
	if (fill && style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	    
  }  

  function drawSegment(xs, ys, color, lineWidth) {
  	mContext.beginPath();

	var numpoints = xs.length;
	for(var i=0; i<numpoints; i++) {
      if(i==0) 
        mContext.moveTo(xs[0], ys[0]);
      else 
      	mContext.lineTo(xs[i], ys[i]);
	}  	  	
    mContext.lineTo(xs[0], ys[0]);
    
	// set style
  	mContext.lineWidth = lineWidth + 0.5;
  	mContext.strokeStyle = color;
  	mContext.stroke();
  }  
  
  function drawVectors(cellA,cellB,field,lenVector) { 
  	for(var i=0; i<cellA.length; i++) {
	
		// draw line
		var a = cellA[i];
		var b = cellB[i];
		var a1 = field[i][0];
		var b1 = field[i][1];
		var len = EJSS_TOOLS.Mathematics.norm([a1,b1]);
		var sa = a1*lenVector/len;
		var sb = b1*lenVector/len;
		
	    // draw
		if((sa != 0) || (sb != 0)) {    			
			var stMax = (0.3 * EJSS_TOOLS.Mathematics.norm([sa,sb])) ;
			var stW = (stMax != 0 && stMax < 12) ? stMax : 12;
		    var angle = Math.atan2(sa,sb);
		  	mContext.beginPath();
			mContext.moveTo(a, b);
		    mContext.lineTo(a+sa - stW*Math.cos(angle), b+sb - stW*Math.sin(angle));
		    mContext.lineTo(a+sa - stW*Math.cos(angle-Math.PI/6), b+sb - stW*Math.sin(angle-Math.PI/6));
		    mContext.lineTo(a+sa, y+my);
		    mContext.lineTo(a+sa - stW*Math.cos(angle+Math.PI/6), b+sb - stW*Math.sin(angle+Math.PI/6));
		    mContext.lineTo(a+sa - stW*Math.cos(angle), b+sb - stW*Math.sin(angle));
		
			// set style
	      	mContext.lineWidth = 1;
	     	mContext.strokeStyle = 'black';
	      	mContext.stroke();
		}
	}  	
  }
	
	// get position of the mElement center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
    
	var mPoints = mElement.getPoints();	
    if (mPoints == null) return;    
    
  	// project the points
    var nPoints = mPoints.length;
    mA = new Array(nPoints);
    mB = new Array(nPoints);
    for (var i=0; i<nPoints; i++) {
      mA[i] = x-mx + mPoints[i][0]*size[0];
      mB[i] = y-my + mPoints[i][1]*size[1];
    }

	// build cells
	var cells = mElement.getCells();
	var field = mElement.getFieldAtPoints();
	if (field) {
		mCells = EJSS_GRAPHICS.GraphicsUtils.buildCells(mA,mB,cells,field,false);
	} else {
	  field = mElement.getFieldAtCells();
	  mCells = EJSS_GRAPHICS.GraphicsUtils.buildCells(mA,mB,cells,field,true);
	}

    // first draw the cells
    if (cells != null) {    
	  var style = mElement.getStyle();
      if (field != null) {
	      // draw polygons    	  
		  var color = mElement.getColorCoded();
	      EJSS_GRAPHICS.GraphicsUtils.drawPolygons(mCells, color, 
	      	function(xs, ys) { drawCell(xs, ys, false, style) },
	      	function(xs, ys, numpoints, color, levelIndex) { fillCell(xs, ys, numpoints, color, levelIndex) });
      }
      else {
	      EJSS_GRAPHICS.GraphicsUtils.drawPolygons(mCells, color, 
	      	function(xs, ys) { drawCell(xs, ys, true, style) },
	      	function(xs, ys, numpoints, color, levelIndex) { });
      }
    }

	// build boundary
	var boundary = mElement.getBoundary();
	if(mElement.getDrawBoundary() && boundary != null) {	 
	  var labels = mElement.getBoundaryLabels();
	  var colors = mElement.getBoundaryColors();
	  var lineWidth = mElement.getBoundaryLineWidth();	
	  mBoundary = EJSS_GRAPHICS.GraphicsUtils.buildBoundary(mA, mB, boundary);
 
      // draw boundary
      EJSS_GRAPHICS.GraphicsUtils.drawBoundary(mBoundary, labels, colors, 
      	function(xs, ys, color) { drawSegment(xs,ys,color,lineWidth) }); 
	}

	// draw vectors
	if(mElement.getDataType() >= 4 && field != null) {
		var lenVector = mElement.getVectorLength() * size[0]; // scale for size[0]		
        for (var i=0, n=cells.length; i<n; i++) {
          // draw vectors
          drawVectors(mCellA[i],mCellB[i],field[i],lenVector);
        }
    }	

}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas pipe
 */
EJSS_CANVASGRAPHICS.pipe = function(mContext, mElement) {  
    var points = mElement.getPoints();
    if (!points || points.length<=0) return;  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];

	
	// get half sizes 		
    var sx = Math.abs(size[0]);
    var sy = Math.abs(size[1]);

 	var xmin = x-sx/2;
 	var ymin = y+sy/2;

    var style = mElement.getStyle(); 

	// draw container
    mContext.beginPath();
    mContext.moveTo(xmin,ymin);
    var x = xmin;
    var y = ymin; 
    for (var i=0, n=points.length; i<n; i++) {
      var point = points[i];
      x += Math.round(point[0]*sx);
      y += Math.round(-point[1]*sy);
      mContext.lineTo(x,y);
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = 2*style.getLineWidth()+mElement.getPipeWidth();
	  mContext.strokeStyle = style.getLineColor();
	  mContext.stroke();
    }

	// draw liquid
    mContext.beginPath();
    mContext.moveTo(xmin,ymin);
    var x = xmin;
    var y = ymin;
    for (var i=0, n=points.length; i<n; i++) {
      var point = points[i];
      x += Math.round(point[0]*sx);
      y += Math.round(-point[1]*sy);
      mContext.lineTo(x,y);
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = mElement.getPipeWidth();
	  mContext.strokeStyle = style.getFillColor();
	  mContext.stroke();
    }    
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas point
 */
EJSS_CANVASGRAPHICS.point = function(mContext, mElement) {  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var xm = pos[0]+offset[0];
    var ym = pos[1]+offset[1];
	
    // draw
  	mContext.beginPath();
  	mContext.arc(xm,ym,2,0,2*Math.PI);
      
	// set style
    var style = mElement.getStyle(); 
    mContext.fillStyle = style.getFillColor();
    mContext.fill();
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas polygon
 */
EJSS_CANVASGRAPHICS.polygon = function(mContext, mElement) {  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
    
    // draw
  	mContext.beginPath();
	var points = mElement.getPoints();	    	    
	for(var i=0; i<points.length; i++) {
	  var point = points[i];	  
      var xx = (x-mx) + point[0]*size[0];
      var yy = (y-my) + point[1]*size[1];
      var type = point[2];
	  if ((i==0) || (type == 0)) // 0 is NOT CONNECTION
        mContext.moveTo(xx, yy);
      else 
      	mContext.lineTo(xx, yy);
	}  	  	
      
	// set style
    var style = mElement.getStyle(); 
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas rectangle
 */
EJSS_CANVASGRAPHICS.rectangle = function(mContext, mElement) {  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
    
    // draw
  	mContext.beginPath();
  	mContext.moveTo(x-mx,y+my);
  	mContext.lineTo(x+mx,y+my);
  	mContext.lineTo(x+mx,y-my);
  	mContext.lineTo(x-mx,y-my);
  	mContext.lineTo(x-mx,y+my);

	// set style
    var style = mElement.getStyle(); 
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas round rectangle
 */
EJSS_CANVASGRAPHICS.roundRectangle = function(mContext, mElement) {  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var mx = pos[0]+offset[0];
    var my = pos[1]+offset[1];
	
	// get sizes 		
    var width = Math.abs(size[0]);
    var height = Math.abs(size[1]);
    var x = mx - width / 2;
    var y = my - height / 2;

    // get element radius
    var radius = Math.abs(mElement.getCornerRadius());
    
    // draw
    mContext.beginPath();
    mContext.moveTo(x + radius, y);
    mContext.lineTo(x + width - radius, y);
    mContext.quadraticCurveTo(x + width, y, x + width, y + radius);
    mContext.lineTo(x + width, y + height - radius);
    mContext.quadraticCurveTo(x + width, y + height, x + width - radius, y + height);
    mContext.lineTo(x + radius, y + height);
    mContext.quadraticCurveTo(x, y + height, x, y + height - radius);
    mContext.lineTo(x, y + radius);
    mContext.quadraticCurveTo(x, y, x + radius, y);
    mContext.closePath();
      
	// set style
    var style = mElement.getStyle(); 
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas canvas
 */
EJSS_CANVASGRAPHICS.canvas = function(mContext, mElement) {  
  
	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
  	var deltaX = mElement.getMaximumX() - mElement.getMinimumX();
  	var deltaY = mElement.getMaximumY() - mElement.getMinimumY();
    
	var pixWth = Math.abs(size[0]);
	var pixHgt = Math.abs(size[1]);
	var pxPerUnitX = pixWth / deltaX;
	var pxPerUnitY = pixHgt / deltaY;
	var pxTopLeftX = x - pixWth / 2;
	var pxTopLeftY = y + pixHgt / 2;
   
    //draws all objects within this.drawables
    EJSS_GRAPHICS.GraphicsUtils.drawImageField(context, 
    	pxTopLeftX, pxTopLeftY, mElement.getMinimumX(), mElement.getMinimumY(), 
    	pixWth, pixHgt, pxPerUnitX, pxPerUnitY, {data: mElement.getData()});
    
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas segment
 */
EJSS_CANVASGRAPHICS.segment = function(mContext, mElement) {  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;

    // draw
  	mContext.beginPath();
  	mContext.moveTo(x-mx, y-my);
  	mContext.lineTo(x+mx, y+my);
      
	// set style
    var style = mElement.getStyle(); 
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas spring
 */
EJSS_CANVASGRAPHICS.spring = function(mContext, mElement) {  
	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;

 	// get spring radius	    	
    var radius = mElement.getGroupPanel().toPixelMod([mElement.getRadius(),0])[0];

	// set attributes
	mContext.beginPath();
	EJSS_GRAPHICS.GraphicsUtils.drawSpring(mElement.getLoops(), mElement.getPointsPerLoop(), 
			radius, mElement.getSolenoid(), mElement.getThinExtremes(), x-mx, y-my, size[0], size[1],
			function(xx,yy) { mContext.moveTo(xx,yy) }, 
			function(xx,yy) { mContext.lineTo(xx,yy) });    
      
	// set style
    var style = mElement.getStyle(); 
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas tank
 */
EJSS_CANVASGRAPHICS.tank = function(mContext, mElement) {  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get sizes 		
    var mx = Math.abs(size[0]/2);
    var my = Math.abs(size[1]/2);

    var level = mElement.getPixelSizeOf(0,mElement.getLevel());  

 	var xmin = x-mx, xmax = x+mx;
 	var ymin = y+my, ymax = y-my;
 	var ylevel = ymin + level[1];
    var style = mElement.getStyle();
    
    // draw liquid
  	mContext.beginPath();
    mContext.moveTo(xmin + level[0],ylevel);
    mContext.lineTo(xmin,ymin);
    mContext.lineTo(xmax,ymin);
    mContext.lineTo(xmax + level[0],ylevel);
    mContext.fillStyle = mElement.getLevelColor();
    mContext.fill();

	// draw container
	mContext.beginPath();
	mContext.moveTo(xmin,ymax);
	mContext.lineTo(xmin,ymin);
	mContext.lineTo(xmax,ymin);
	mContext.lineTo(xmax,ymax);
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    mContext.lineWidth = style.getLineWidth();
    mContext.strokeStyle = style.getLineColor();
    mContext.stroke();
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas text
 */
EJSS_CANVASGRAPHICS.text = function(mContext, mElement) {  
	// font 
 	var font = mElement.getFont();
 	var fondtxt = "";
 	fondtxt += (font.getFontStyle() != 'none')? font.getFontStyle() + ' ':'';
 	fondtxt += (font.getFontWeight() != 'none')? font.getFontWeight() + ' ':'';
 	fondtxt += font.getFontSizeString() + ((font.getFontSizeString().toString().indexOf('px') > -1)?' ':'px ');
 	fondtxt += font.getFontFamily();
	mContext.font = fondtxt;

	// txt 	
 	var txt = mElement.getText(); 	

	// get position of the element center 
    var pos = mElement.getPixelPosition();
	var marginFrame = mElement.getFramed()?4:0; 
    var size = [mContext.measureText(txt).width + marginFrame*2, font.getFontSize() + marginFrame*2];
    var offset = EJSS_DRAWING2D.Element.getSWRelativePositionOffset(mElement.getRelativePosition(), size[0], size[1]);  
    var xmargin = mElement.getMarginX();
    var ymargin = mElement.getMarginY();
    var x = pos[0]+offset[0]+xmargin+marginFrame;
    var y = pos[1]+offset[1]+ymargin-marginFrame;

    if (mElement.getFramed()) { // framed    
    	var tbmargin = marginFrame;
    	var rlmargin = marginFrame;
    	var wlen = size[0] - marginFrame*2;
    	var hlen = size[1] - marginFrame*2;    	
    	
	    // draw box
	    if(wlen > 0) {
	    	var bottom = y + tbmargin, top = y - hlen - tbmargin;
	    	var left = x - rlmargin, rigth = x + wlen + tbmargin;
	    	mContext.save();	 		 
		  	mContext.beginPath();
  			mContext.moveTo(left, bottom);
  			mContext.lineTo(rigth, bottom);
  			mContext.lineTo(rigth, top);
  			mContext.lineTo(left, top);
			mContext.lineTo(left, bottom);

			// set style
		    var style = mElement.getStyle(); 
			if (style.getDrawFill() && style.getFillColor() != 'none') {
		      mContext.fillStyle = style.getFillColor();
		      mContext.fill();
		    }
		    if (style.getDrawLines()) {
		      mContext.lineWidth = style.getLineWidth();
		      mContext.strokeStyle = style.getLineColor();
		      mContext.stroke();
		    }	
		    mContext.restore();
		}
	}
		    
	// angle of text  	 
	var angletext = 0;
	switch(mElement.getWritingMode()) {
		case EJSS_DRAWING2D.Text.MODE_TOPDOWN : angletext = Math.PI / 2; break;
		case EJSS_DRAWING2D.Text.MODE_RIGTHLEFT : angletext = Math.PI; break;
		case EJSS_DRAWING2D.Text.MODE_DOWNTOP : angletext = 3 * Math.PI / 2; break;
		case EJSS_DRAWING2D.Text.MODE_LEFTRIGHT: angletext = 0; break;
	} 	  	  	

  	mContext.translate(pos[0]+xmargin,pos[1]+ymargin);
  	mContext.rotate(angletext);
  	mContext.translate(-(pos[0]+xmargin),-(pos[1]+ymargin));

	// draw text		
	mContext.fillStyle = font.getFillColor();
	mContext.fillText(mElement.getText(), x, y);
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas trace
 */
EJSS_CANVASGRAPHICS.trace = function(mContext, mElement) {  

	// draw path for trail
	function drawTrail(points, x, y, sx, sy) {
	  	mContext.beginPath();
		for(var i=0; i<points.length; i++) {
		  var point = points[i];	  
	      var xx = x + point[0]*sx;
	      var yy = y + point[1]*sy;
	      var type = point[2];		
		  if ((i==0) || (type == 0)) // 0 is NOT CONNECTION
		  	mContext.moveTo(xx,yy);		// move 
	      else       	
	      	mContext.lineTo(xx,yy);		// line
		}  	  	
	}  	

	// creates ellipse mark
	function ellipseMark(markx, marky, size, markstyle) {
		// get position of the element center 
	    var xm = markx;
	    var ym = marky;
		
		// get sizes 		
	    var w = Math.abs(size[0]);
	    var h = Math.abs(size[1]);
	
	    // draw
	    var kappa = .5522848,
	      // x = x-middle, y = y-middle
	      x = xm - w / 2,       // x-middle
	      y = ym - h / 2;       // y-middle      
	      ox = (w / 2) * kappa, // control point offset horizontal
	      oy = (h / 2) * kappa, // control point offset vertical
	      xe = x + w,           // x-end
	      ye = y + h,           // y-end
	
	  	mContext.beginPath();
	  	mContext.moveTo(x, ym);
	  	mContext.bezierCurveTo(x, ym - oy, xm - ox, y, xm, y);
	  	mContext.bezierCurveTo(xm + ox, y, xe, ym - oy, xe, ym);
	  	mContext.bezierCurveTo(xe, ym + oy, xm + ox, ye, xm, ye);
	  	mContext.bezierCurveTo(xm - ox, ye, x, ym + oy, x, ym);
	      
		// set style
		if (markstyle.getDrawFill() && markstyle.getFillColor() != 'none') {
	      mContext.fillStyle = markstyle.getFillColor();
	      mContext.fill();
	    }
	    if (markstyle.getDrawLines()) {
	      mContext.lineWidth = markstyle.getLineWidth();
	      mContext.strokeStyle = markstyle.getLineColor();
	      mContext.stroke();
	    }	  	
	}  	

	// creates rectangle mark
	function rectangleMark(markx, marky, size, markstyle) {
		// get position of the element center 
	    var x = markx;
	    var y = marky;
		
		// get sizes 		
	    var sx = Math.abs(size[0]);
	    var sy = Math.abs(size[1]);

	    // draw
	  	mContext.beginPath();
	  	mContext.rect(x-sx/2, y-sy/2, sx, sy);
	      
		// set style
		if (markstyle.getDrawFill() && markstyle.getFillColor() != 'none') {
	      mContext.fillStyle = markstyle.getFillColor();
	      mContext.fill();
	    }
	    if (markstyle.getDrawLines()) {
	      mContext.lineWidth = markstyle.getLineWidth();
	      mContext.strokeStyle = markstyle.getLineColor();
	      mContext.stroke();
	    }	
	}  	

	// creates area mark
	function areaMark(lastmarkx, lastmarky, markx, marky, axis, markstyle) {	
	    // draw
	  	mContext.beginPath();
	  	mContext.moveTo(lastmarkx,axis);
	  	mContext.lineTo(lastmarkx,lastmarky);
	  	mContext.lineTo(markx,marky);
	  	mContext.lineTo(markx,axis);
	
		// set style
		if (markstyle.getDrawFill() && markstyle.getFillColor() != 'none') {
	      mContext.fillStyle = markstyle.getFillColor();
	      mContext.fill();
	    }
	    if (markstyle.getDrawLines()) {
	      mContext.lineWidth = markstyle.getLineWidth();
	      mContext.strokeStyle = markstyle.getLineColor();
	      mContext.stroke();
	    }		
	}  	

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
        
	// draw trail
    drawTrail(mElement.getPoints(), x-mx, y-my, size[0], size[1]);

	// set style
    var style = mElement.getStyle(); 
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
     
	// draw marks     
	var points = mElement.getPoints();
	var lastpointX, lastpointY;	// for mark area
	for(var i=0; i<points.length; i++) {
		var point = points[i];	  
		var markx = (x-mx) + point[0]*size[0];		// mark x
		var marky = (y-my) + point[1]*size[1];     // mark y
	    var markType = point[3];    		// mark type
		var markstyle = point[4];			// mark style
		var marksize = [point[5],point[6]];		// mark size

	    if (markType == EJSS_DRAWING2D.Trace.ELLIPSE) { // circle
			ellipseMark(markx, marky, marksize, markstyle);
	    } else if (markType == EJSS_DRAWING2D.Trace.RECTANGLE) { // rectangle
			rectangleMark(markx, marky, marksize, markstyle);
	    } else if (markType == EJSS_DRAWING2D.Trace.BAR) { // bar
		    // create bar  		    	
			//var axisx = mElement.getGroupPanel().getPixelPositionWorldOrigin()[1];
			var axisx = mElement.getGroupPanel().toPixelPosition([0,0])[1];
			
	    	marksize[1] = axisx-marky;			// bar size
	    	marky = marky + marksize[1]/2;		// new mark y	    	
	    	rectangleMark(markx, marky, marksize, markstyle);
		}
		else if (markType == EJSS_DRAWING2D.Trace.AREA) { // area
	    	if(i!=0) {				  
				// create area
				//var axisx = mElement.getGroupPanel().getPixelPositionWorldOrigin()[1];
				var axisx = mElement.getGroupPanel().toPixelPosition([0,0])[1];
				var lastmarkx = (x-mx) + lastpointX*size[0];		// mark x
				var lastmarky = (y-my) + lastpointY*size[1];     // mark y
				areaMark(mMark, lastmarkx, lastmarky, markx, marky, axisx, markstyle);
	    	} 
	    	lastpointX = point[0];    	
	    	lastpointY = point[1];
	    }		  
	}  	  		  

}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas trail
 */
EJSS_CANVASGRAPHICS.trail = function(mContext, mElement) {  

  /** adds a segment to the trail
   */
   function addSegment(points,num,style,pX,pY,sX,sY) {
     if (num<=0) return; // Nothing to add
	 	 	
	mContext.beginPath();
	for (var i=0; i<num; i++) {
	  var point = points[i];	  
	  var xx = pX + point[0]*sX;
	  var yy = pY + point[1]*sY;
	  var type = point[2];		
	  if ((i==0) || (type == 0)) // 0 is NOT CONNECTION
	    mContext.moveTo(xx,yy);
      else
        mContext.lineTo(xx,yy);       	
 	}  	  	

/*
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
*/
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
   }

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
	
	var pX = x-mx;
	var pY = y-my;
	var sX = size[0];
	var sY = size[1];
	
	var segmentCount = mElement.getSegmentsCount();
	if (segmentCount>0) {
	  for (var i=0; i<segmentCount; i++) { 
	  	var num = mElement.getSegmentPoints(i).length;
	    addSegment(mElement.getSegmentPoints(i),num,mElement.getSegmentStyle(i),pX,pY,sX,sY);
	  }
	}
	
	//mElement.dataCollected(); // add temporary points
	var num = mElement.getCurrentPoints().length; 
	addSegment(mElement.getCurrentPoints(),num,mElement.getStyle(),pX,pY,sX,sY);
	
}/**
 * Deployment for 2D Canvas drawing.
 * @module CanvasGraphics 
 */

var EJSS_CANVASGRAPHICS = EJSS_CANVASGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A Canvas wheel
 */
EJSS_CANVASGRAPHICS.wheel = function(mContext, mElement) {  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var xm = pos[0]+offset[0];
    var ym = pos[1]+offset[1];
	
	// get sizes 		
    var w = Math.abs(size[0]);
    var h = Math.abs(size[1]);

	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
    
    // draw
    var kappa = .5522848,
      // x = x-middle, y = y-middle
      x = xm - w / 2,       // x-middle
      y = ym - h / 2;       // y-middle      
      ox = (w / 2) * kappa, // control point offset horizontal
      oy = (h / 2) * kappa, // control point offset vertical
      xe = x + w,           // x-end
      ye = y + h,           // y-end

  	mContext.beginPath();
	// set circle atts
  	mContext.moveTo(x, ym);
  	mContext.bezierCurveTo(x, ym - oy, xm - ox, y, xm, y);
  	mContext.bezierCurveTo(xm + ox, y, xe, ym - oy, xe, ym);
  	mContext.bezierCurveTo(xe, ym + oy, xm + ox, ye, xm, ye);
  	mContext.bezierCurveTo(xm - ox, ye, x, ym + oy, x, ym);
      
	// set cross atts
	mContext.moveTo(xm-mx,ym);
	mContext.lineTo(xm+mx,ym);
	mContext.moveTo(xm,ym-my);
	mContext.lineTo(xm,ym+my);
	
	// set style
    var style = mElement.getStyle(); 
	if (style.getDrawFill() && style.getFillColor() != 'none') {
      mContext.fillStyle = style.getFillColor();
      mContext.fill();
    }
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
}/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Deployment for 2D drawing.
 * @module 2Dgraphics 
 */

var EJSS_GRAPHICS = EJSS_GRAPHICS || {};

/**
 * SVGGraphics class
 * @class SVGGraphics 
 * @constructor  
 */
EJSS_GRAPHICS.CanvasGraphics = {

  /**
   * Return document box
   * @return box  
   */
  getOffsetRect: function(graphics) {
  	return EJSS_GRAPHICS.GraphicsUtils.getOffsetRect(graphics);
  }
};

/**
 * Constructor for Element
 * @param mName Identifier Canvas element in HTML document
 * @returns A Canvas graphics
 */
EJSS_GRAPHICS.canvasGraphics = function(mName) {
  var self = EJSS_INTERFACE.canvas(mName);	// reference returned     
  var mInterfaceGraphics = self.getDOMElement();
  var mContext = self.getContext();

  /**
   * Return canvas for event handle
   * @return canvas element 
   */
  self.getEventContext = function() {  	
    return mInterfaceGraphics;
  };

  /**
   * Return canvas document box
   * @return width integer 
   */
  self.getBox = function() {
  	var box = EJSS_GRAPHICS.CanvasGraphics.getOffsetRect(self)
  	box.width -= 1;
  	box.height -= 1;
  	return box;
  };

  /**
   * Reset the Canvas graphics
   */
  self.reset = function() {
  	var rect = EJSS_GRAPHICS.CanvasGraphics.getOffsetRect(self);
  	mContext.clearRect(0, 0, rect.width, rect.height);
  };

  /**
   * Draw Elements
   * @param elements array of Elements and Element Sets
   * @param force whether force draw elements
   */
  self.draw = function(elements,force) {  	
    for (var i=0, n=elements.length; i<n; i++) { 
      var element = elements[i];
      if (element.isGroupVisible()) {
		  	self.drawElement(element, self.transformElement);
	  }  	
    }
  };
    
  /**
   * Draw Element
   * @param element
   */
  self.drawElement = function (element, transformCallBack) {
  	var Shape = EJSS_DRAWING2D.Shape;
  	var cl = element.getClass();
	
	if (cl === 'ElementImage') {
		EJSS_CANVASGRAPHICS.image(mContext,element,transformCallBack);
	} else {
		mContext.save();
		transformCallBack(element);

	  	switch (cl) {
			case "ElementCustom": 
			  {
			    var customFunction = element.getFunction();
			    if (customFunction) customFunction(mContext,element);	
			  }
			  break;
	  		case "ElementShape": 
				switch (element.getShapeType()) {
				  default : 
				  case Shape.ELLIPSE         : EJSS_CANVASGRAPHICS.ellipse(mContext,element); 	break;
				  case Shape.RECTANGLE       : EJSS_CANVASGRAPHICS.rectangle(mContext,element);	break;
				  case Shape.ROUND_RECTANGLE : EJSS_CANVASGRAPHICS.roundRectangle(mContext,element); break;
				  case Shape.WHEEL           : EJSS_CANVASGRAPHICS.wheel(mContext,element); 		break;
				  case Shape.NONE            : // do not break;
				  case Shape.POINT           : EJSS_CANVASGRAPHICS.point(mContext,element); 		break;
				}
				break;
			case "ElementSegment": 	EJSS_CANVASGRAPHICS.segment(mContext,element);	break;
			case "ElementArrow":	EJSS_CANVASGRAPHICS.arrow(mContext,element);	break;
			case "ElementText":		EJSS_CANVASGRAPHICS.text(mContext,element);		break;
			case "ElementSpring":	EJSS_CANVASGRAPHICS.spring(mContext,element);		break;
			case "ElementTrail":	EJSS_CANVASGRAPHICS.trail(mContext,element);		break;
			case "ElementTrace":	EJSS_CANVASGRAPHICS.trace(mContext,element);		break;
			case "ElementGrid":		EJSS_CANVASGRAPHICS.grid(mContext,element);		break;
			case "ElementAxis":		EJSS_CANVASGRAPHICS.axis(mContext,element);		break;
			case "ElementCursor":	EJSS_CANVASGRAPHICS.cursor(mContext,element);		break;
			case "ElementPolygon":	EJSS_CANVASGRAPHICS.polygon(mContext,element);	break;
			case "ElementAnalyticCurve":	EJSS_CANVASGRAPHICS.analyticCurve(mContext,element);	break;
			case "ElementCellLattice": 	EJSS_CANVASGRAPHICS.cellLattice(mContext,element);	break; 			
			case "ElementByteRaster": 	EJSS_CANVASGRAPHICS.byteRaster(mContext,element);	break;			
			case "ElementMesh": 	EJSS_CANVASGRAPHICS.mesh(mContext,element);		break;			
			case "ElementTank":   	EJSS_CANVASGRAPHICS.tank(mContext,element);		break;
			case "ElementPipe":   	EJSS_CANVASGRAPHICS.pipe(mContext,element);		break;
			case "ElementHistogram":EJSS_CANVASGRAPHICS.histogram(mContext,element);		break;

			case "ElementScalarField":	EJSS_CANVASGRAPHICS.scalarField(mContext,element);		break;
			case "ElementCanvas":	EJSS_CANVASGRAPHICS.canvas(mContext,element);		break;

			case "ElementVideo":   	Console.log('Not supported ElementVideo in Canvas!');	break;
		}		
		mContext.restore();
	}
  };

  /**
   * Transform Element
   * @param element
   */
  self.transformElement = function (element) {
  	// first, group transform 
  	if(element.getGroup() != null)
  		self.transformElement(element.getGroup())
	
	// second, my transform  	
	var trans = element.getTransformation();
  	var rot = element.getRotate();
  	if (Array.isArray(trans) && trans.length > 0) {  
    	console.log("Array transformations are not supported in Canvas!");  	
	} else if (typeof trans === "string") {  // SVG format
    	console.log("SVG-based transformations are not supported in Canvas!");  	
	} else if (rot != 0) { // && trans!=0) {  // rotation angle
	    var pos = element.getPixelPosition();	
      	mContext.translate(pos[0],pos[1]);
      	mContext.rotate(-rot);
      	mContext.translate(-pos[0],-pos[1]);
	}	  	
  }

  /**
   * Draw gutters
   * @param gutters
   */
  self.drawGutters = function(panel) {
  	var gutters = panel.getGutters();
  	
  	if (gutters.visible) {    	 			
	 	// get gutters position     
	    var inner = panel.getInnerRect();
	    
	 	// get panel position
	    var x = 0.5, y = 0.5;
	    var box = self.getBox();
	    var sx = box.width, sy = box.height;
	    
	 	var guttersStyle = panel.getGuttersStyle();	 
	 	var panelStyle = panel.getStyle();   

        if (guttersStyle.getDrawFill()) { // create path	    	   	    
	      mContext.beginPath();
		  mContext.moveTo(x,y);
		  mContext.lineTo(x,y+sy);
		  mContext.lineTo((x+sx),(y+sy));
		  mContext.lineTo((x+sx),y);
		  mContext.lineTo(x,y);
		  mContext.moveTo(inner.x,inner.y);
		  mContext.lineTo(inner.x + inner.width, inner.y);
		  mContext.lineTo(inner.x + inner.width, inner.y + inner.height);
		  mContext.lineTo(inner.x, inner.y + inner.height);
		  mContext.lineTo(inner.x,inner.y);
		  mContext.moveTo(x,y);
	      mContext.closePath();
	      mContext.fillStyle = guttersStyle.getFillColor();
	      mContext.fill();
	    }
	    if (guttersStyle.getDrawLines()) { // Outer border
	      mContext.beginPath();
		  mContext.moveTo(x,y);
		  mContext.lineTo(x,y+sy);
		  mContext.lineTo((x+sx),(y+sy));
		  mContext.lineTo((x+sx),y);
		  mContext.lineTo(x,y);
	      mContext.closePath();
	      mContext.lineWidth = guttersStyle.getLineWidth();
	      mContext.strokeStyle = guttersStyle.getLineColor();
	      mContext.stroke();
        }
	    if (guttersStyle.getDrawLines()) { // Inner border
	      mContext.beginPath();
		  mContext.moveTo(inner.x,inner.y);
		  mContext.lineTo(inner.x + inner.width, inner.y);
		  mContext.lineTo(inner.x + inner.width, inner.y + inner.height);
		  mContext.lineTo(inner.x, inner.y + inner.height);
		  mContext.lineTo(inner.x,inner.y);
	      mContext.closePath();
	      mContext.lineWidth = panelStyle.getLineWidth();
	      mContext.strokeStyle = panelStyle.getLineColor();
	      mContext.stroke();
        }
	}  	
  };
  
  /**
   * Draw panel
   * @param panel
   */
  self.drawPanel = function(panel) {
 	// get position
    var x = 0.5, y = 0.5;
    var box = self.getBox();
	
    mContext.beginPath();
    mContext.rect(x, y, box.width, box.height);
    
    // style
    var style = panel.getStyle();
    mContext.fillStyle = style.getFillColor();
    mContext.fill();
    if (style.getDrawLines()) {
      mContext.lineWidth = style.getLineWidth();
      mContext.strokeStyle = style.getLineColor();
      mContext.stroke();
    }	
	
  };
  
 return self;     
      
}
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Deployment for 2D drawing.
 * @module 2Dgraphics 
 */

var EJSS_GRAPHICS = EJSS_GRAPHICS || {};

/**
 * GraphicsUtils class
 * @class GraphicsUtils 
 * @constructor  
 */
EJSS_GRAPHICS.GraphicsUtils = {

	/**
	 * Return document box
	 * @return box  
	 */
	getOffsetRect: function(graphics) {
	    var width, height, top, left;
	    var element = graphics.getDOMElement();
	
		// (0) width and height
		var strw = graphics.getWidth();
		var strh = graphics.getHeight();  		
		width = parseFloat(strw);
		var isWidthNumber = (!isNaN(width) && isFinite(strw));	 
		height = parseFloat(strh);	 
		var isHeightNumber = (!isNaN(height) && isFinite(strh));
	
	    // (1) left and top
	    if(element.getBoundingClientRect) { // this option is better
			// **myPanel shows the VISIBLE elements
			var mypanel = element.getElementById? element.getElementById(".myPanel"):null;
			if(mypanel && mypanel.childElementCount>0) { 
				var box = mypanel.getBoundingClientRect();
			} else {
		    	var box = element.getBoundingClientRect();
		    }
			top = box.top;
			left = box.left;	  		
			if(!isWidthNumber) width = box.width;
			if(!isHeightNumber) height = box.height;
		} else if(element.offsetTop !== undefined) { // else this one
	    	top = element.offsetTop;
	    	left = element.offsetLeft;
			if(!isWidthNumber) width = element.offsetWidth;
			if(!isHeightNumber) height = element.offsetHeight;
	
	    } else { // use parent offset (buggy!)
			// **the parent MUST fit the SVG
	   		var parent = element.parentNode || document.getElementById(element.id).parentNode;
			while ((parent && !parent.offsetTop) && (parent && !parent.previousElementSibling && !parent.nextElementSibling)) {
	    		parent = parent.parentNode;
			};
			top = parent.offsetTop;
			left = parent.offsetLeft;
			if(!isWidthNumber) width = parent.offsetWidth;
			if(!isHeightNumber) height = parent.offsetHeight;
		}				    			 
	
	    var body = document.body;
	    var docElem = document.documentElement;
	     
	    // (2) we don't need absolute position, so scroll is ignored
	    // var scrollTop = window.pageYOffset || docElem.scrollTop || body.scrollTop;
	    // var scrollLeft = window.pageXOffset || docElem.scrollLeft || body.scrollLeft;
	     
	    // (3)
	    var clientTop = docElem.clientTop || body.clientTop || 0;
	    var clientLeft = docElem.clientLeft || body.clientLeft || 0;
	     
	    // (4)
	    top =  top - clientTop;
	    left = left - clientLeft;
	
	    return { 
			top: top,
			left: left,
			width: width,
			height: height,
			bottom: top + height,
			right: left + width
		};     
	},

	/**
	 * Draw a spring
	*/
	drawSpring: function(loops, pointsPerLoop, radius, solenoid, thinExtremes, x, y, sx, sy, moveTo, lineTo) {
		var segments = loops * pointsPerLoop;
		var delta = 2.0 * Math.PI / pointsPerLoop;
		if (radius < 0) delta *= -1;
		var pre = pointsPerLoop / 2;
		// normalize sizes        	
		var size = [sx, sy, 0];
		var u1 = EJSS_TOOLS.Mathematics.normalTo(size);
		var u2 = EJSS_TOOLS.Mathematics.normalize(EJSS_TOOLS.Mathematics.crossProduct(size, u1));
		
	    for (var i=0; i<=segments; i++) {
	      var k;
	      if (thinExtremes) {	// extremes
	        if (i < pre) k = 0;
	        else if (i < pointsPerLoop) k = i - pre;
	        else if (i > (segments - pre)) k = 0;
	        else if (i > (segments - pointsPerLoop)) k = segments - i - pre;
	        else k = pre;
	      }
	      else k = pre;
	      var angle = Math.PI/2 + i*delta;
	      var cos = Math.cos(angle), sin = Math.sin(angle);
	      var xx = x + i*sx/segments + k*radius*(cos*u1[0] + sin*u2[0])/pre;
	      var yy = y + i*sy/segments + k*radius*(cos*u1[1] + sin*u2[1])/pre;
	      if (solenoid != 0.0)  {
	        var cte = k*Math.cos(i*2*Math.PI/pointsPerLoop)/pre;
	        xx += solenoid*cte*sx;
	        yy += solenoid*cte*sy;
	      }
	      if (i==0)
	      	moveTo(xx,yy); 
	      else 
	      	lineTo(xx,yy);
	    }
	},	
  
	/**
	 * Draw ticks for axis
	*/
	drawDecTicks: function(x, y, mx, my, ticksize, step, shift, orient, inverted, drawTick) {
		var UTILS = EJSS_SVGGRAPHICS.Utils;
		
		// bounds	
		var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
		var cright = UTILS.crispValue(right);
		var cbottom = UTILS.crispValue(bottom);
	
		// drawing ticks (only support mode NUM (decimal))
	    if(orient == EJSS_DRAWING2D.Axis.AXIS_VERTICAL) {
	    	// draw vertical axis (inverted or not are the same)
	    	for (var i = cbottom-shift; i >= top-0.5; i -= step) {		
				drawTick(x, i, ticksize, orient);
			}
	    } else {
	    	// draw horizontal axis
	    	for (var i = left+shift; i <= cright+0.5; i += step) {	// left to right	    		
				drawTick(i, y, ticksize, orient);
			}
		}					
	},
	
  
	/**
	 * Draw ticks text for axis
	*/
	drawDecTicksText: function(x, y, mx, my, ticksize, step, shift, scale, scalePrecision, scalestep, scaleshift, font, textPosition, orient, inverted, tickText) {
		var UTILS = EJSS_SVGGRAPHICS.Utils;
		
		// bounds	
		//console.log ("Axis = "+ orient+" scaleshif = "+scaleshift);
		var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
		var cright = UTILS.crispValue(right);
		var cbottom = UTILS.crispValue(bottom);
	
		// drawing ticks (only support mode NUM (decimal))
	    if(orient == EJSS_DRAWING2D.Axis.AXIS_VERTICAL) {
			// draw tick text			
	    	for (var j = 0, i = cbottom-shift; i >= top-0.5; i -= step, j++) {
	    		if(inverted)
	    			var number = scale[1] - scaleshift - (scalestep*j); // number for text
	    		else							
					var number = scale[0] + scaleshift + (scalestep*j); // number for text
		    	var text = parseFloat(number.toFixed(scalePrecision)).toFixed(scalePrecision);  // fixed precision				
				if(textPosition == EJSS_DRAWING2D.Axis.TICKS_DOWN) {
					var dx = ticksize/2; // tick size middle + 2
					tickText(mGroup, x+dx, i, text, font, false);
				} else {
					var dx = text.length * Math.floor(font.getFontSize()/2); // text long plus tick size 
					dx += text.length * font.getNumberLetterSpacing();
					dx += Math.floor(ticksize/2); 				
					tickText(x-dx, i, text, font, false);
				}
			}    	
	    } else {
			// draw tick text
	    	for (var j = 0, i = left+shift; i <= cright+0.5; i += step, j++) {	// left to right	    		
				var number = scale[0] + scaleshift + (scalestep*j); // number for text
		    	var text = parseFloat(number.toFixed(scalePrecision)).toFixed(scalePrecision);  // fixed precision		    					
				if(textPosition == EJSS_DRAWING2D.Axis.TICKS_DOWN) {
					var dy = Math.floor(ticksize/2) + 2; // tick size middle + 2
					tickText(i, y-ticksize, text, font, true);
				} else {
					var dy = font.getFontSize() + Math.floor(ticksize/2); // size font + tick size middle 
					tickText(i, y+dy, text, font, true);
				}			
		   	}
		}
	
	},
	
	/**
	 * Draw ticks for log axis
	*/
	drawLogTicks: function(x, y, mx, my, segsize, ticksize, scale, numticks, orient, drawTick) {
		var UTILS = EJSS_SVGGRAPHICS.Utils;
		
		// bounds	
		var left = x-mx, bottom = y-my;
		var cbottom = UTILS.crispValue(bottom);
	
		// drawing ticks (only support mode NUM (decimal))
	    if(orient == EJSS_DRAWING2D.Axis.AXIS_VERTICAL) {
	    	// draw vertical axis
	    	var minscale = Math.log(scale[0])/Math.log(10);		// min log value
	    	var maxscale = Math.log(scale[1])/Math.log(10);		// max log value
	    	var scalesize = maxscale - minscale;			// size of log scale
	    	var numbands = Math.ceil(scalesize);			// num of log bands
	    	var sizeratio = segsize / scalesize;			// pixel per log value
	    	
	    	// generate ticks in each band
	    	for (var i = 0; i <= numbands; i++) {
	    		var bandmin = Math.pow(10,i+minscale);		// band min
	    		var bandmax = Math.pow(10,i+minscale+1);	// band max
	    		var bandshift = bandmax / numticks;			// band part
		    	for (var j = 0; j < numticks-1; j++) {	    		
		    		// log pos	    		
		    		var number = bandmin + j * bandshift;  // number for text
					if(number != 0 && number >= scale[0] && number <= scale[1]) {
			    		var step = Math.log(number) / Math.log(10);
			    		var pos = cbottom - (step - minscale) * sizeratio;   		
						drawTick(x, pos, ticksize, orient);
					}  	   		    		    					
				}
			}
	    } else {
	    	// draw horizontal axis
	    	var minscale = Math.log(scale[0])/Math.log(10);		// min log value
	    	var maxscale = Math.log(scale[1])/Math.log(10);		// max log value
	    	var scalesize = maxscale - minscale;			// size of log scale
	    	var numbands = Math.ceil(scalesize);			// num of log bands
	    	var sizeratio = segsize / scalesize;			// pixel per log value
	    	
	    	// generate ticks in each band
	    	for (var i = 0; i <= numbands; i++) {
	    		var bandmin = Math.pow(10,i+minscale);		// band min
	    		var bandmax = Math.pow(10,i+minscale+1);	// band max
	    		var bandshift = bandmax / numticks;			// band part
		    	for (var j = 0; j < numticks-1; j++) {	    		
		    		// log pos	    		
		    		var number = bandmin + j * bandshift;  // number for text
					if(number != 0 && number >= scale[0] && number <= scale[1]) {
			    		var step = Math.log(number) / Math.log(10);
			    		var pos = left + (step - minscale) * sizeratio;   		
						drawTick(pos, y, ticksize, orient);
					}  	   		    		    					
				}
			}
		}					
	},
		
	/**
	 * Draw ticks text for log axis
	*/
	drawLogTicksText: function(x, y, mx, my, segsize, numticks, ticksize, scale, scalePrecision, font, textPosition, orient, tickText) {
		var UTILS = EJSS_SVGGRAPHICS.Utils;
		
		// bounds	
		var left = x-mx, bottom = y-my;
		var cbottom = UTILS.crispValue(bottom);
	
		// drawing ticks (only support mode NUM (decimal))
	    if(orient == EJSS_DRAWING2D.Axis.AXIS_VERTICAL) {
			// draw tick text			
	    	var minscale = Math.log(scale[0])/Math.log(10);		// min log value
	    	var maxscale = Math.log(scale[1])/Math.log(10);		// max log value
	    	var scalesize = maxscale - minscale;			// size of log scale
	    	var numbands = Math.ceil(scalesize);			// num of log bands
	    	var sizeratio = segsize / scalesize;			// pixel per log value
	    	
	    	// generate ticks in each band
	    	for (var i = 0; i <= numbands; i++) {
	    		var bandmin = Math.pow(10,i+minscale);		// band min
	    		var bandmax = Math.pow(10,i+minscale+1);	// band max
	    		var bandshift = bandmax / numticks;			// band part
		    	for (var j = 0; j < numticks-1; j++) {	    		
		    		// log pos	 		    		   		
		    		var number = bandmin + j * bandshift;  // number for text
					if(number != 0 && number >= scale[0] && number <= scale[1]) {
			    		var step = Math.log(number) / Math.log(10);
			    		var pos = cbottom - (step - minscale) * sizeratio;   		
						
						// draw text
						if(j == 0) { // exp format
							var text = number.toExponential(scalePrecision); 		    											
						} else {  // fixed precision
				    		var text = parseFloat((number/bandmin).toFixed(scalePrecision)).toFixed(scalePrecision); 		    					
						}
						if(textPosition == EJSS_DRAWING2D.Axis.TICKS_DOWN) {
							var dx = ticksize/2; // tick size middle + 2
							tickText(x+dx, pos, text, font, false);
						} else {
							var dx = text.length * Math.floor(font.getFontSize()/2); // text long plus tick size 
							dx += text.length * font.getNumberLetterSpacing();
							dx += Math.floor(ticksize/2); 				
							tickText(x-dx, pos, text, font, false);
						}
					}  	   		    		    					
				}
			}
	    } else {
	    	// draw horizontal axis
	    	var minscale = Math.log(scale[0])/Math.log(10);		// min log value
	    	var maxscale = Math.log(scale[1])/Math.log(10);		// max log value
	    	var scalesize = maxscale - minscale;			// size of log scale
	    	var numbands = Math.ceil(scalesize);			// num of log bands
	    	var sizeratio = segsize / scalesize;			// pixel per log value
	    	
	    	// generate ticks in each band
	    	for (var i = 0; i <= numbands; i++) {
	    		var bandmin = Math.pow(10,i+minscale);		// band min
	    		var bandmax = Math.pow(10,i+minscale+1);	// band max
	    		var bandshift = bandmax / numticks;			// band part
		    	for (var j = 0; j < numticks-1; j++) {	    		
		    		// log pos	 		    		   		
		    		var number = bandmin + j * bandshift;  // number for text
					if(number != 0 && number >= scale[0] && number <= scale[1]) {
			    		var step = Math.log(number) / Math.log(10);
			    		var pos = left + (step - minscale) * sizeratio;   		
						
						// draw text
						if(j == 0) { // exp format
							var text = number.toExponential(scalePrecision); 		    											
						} else {  // fixed precision
				    		var text = parseFloat((number/bandmin).toFixed(scalePrecision)).toFixed(scalePrecision); 		    					
						}
						if(textPosition == EJSS_DRAWING2D.Axis.TICKS_DOWN) {
							var dy = Math.floor(ticksize/2) + 2; // tick size middle + 2
							tickText(pos, y-ticksize, text, font, true);
						} else {
							var dy = font.getFontSize() + Math.floor(ticksize/2); // size font + tick size middle 
							tickText(pos, y+dy, text, font, true);
						}			
					}  	   		    		    					
				}
			}
		}
	},

	/**
	 * Draw grid
	*/
	drawDecGrid: function (left, top, right, bottom, sh, step, orientation, moveTo, lineTo) {
		var UTILS = EJSS_SVGGRAPHICS.Utils;

		var cleft = UTILS.crispValue(left), cright = UTILS.crispValue(right);
		var ctop = UTILS.crispValue(top), cbottom = UTILS.crispValue(bottom);

    	if(orientation == -1) {
    		if (step == 0) step = Math.abs(ctop-cbottom);
		    for (var i = cbottom-sh; i >= top-0.5; i -= step) { // bottom to top
	    		moveTo(left,i);
	    		lineTo(right,i);
		    }		    	
    	} else {
    		if (step == 0) step = Math.abs(cright-cleft);
		    for (var i = left+sh; i <= cright+0.5; i += step) { // left to right
		    	moveTo(i,top);
		    	lineTo(i,bottom);
		    }    		
    	}
	},
	
	/**
	 * Draw log grid
	*/
	drawLogGrid: function (left, top, right, bottom, numTicks, scale, orientation, moveTo, lineTo) {	
		var UTILS = EJSS_SVGGRAPHICS.Utils;

		var cleft = UTILS.crispValue(left), cright = UTILS.crispValue(right);
		var ctop = UTILS.crispValue(top), cbottom = UTILS.crispValue(bottom);

		var path = "";   

    	var minscale = Math.log(scale[0])/Math.log(10);		// min log value
    	var maxscale = Math.log(scale[1])/Math.log(10);		// max log value
    	var scalesize = maxscale - minscale;			// size of log scale
    	var numbands = Math.ceil(scalesize);			// num of log bands
    	if(orientation == -1) 
    		var sizeratio = (cbottom-ctop) / scalesize;		// pixel per log value
    	else
    		var sizeratio = (cright-cleft) / scalesize;		// pixel per log value
    	
    	// generate ticks in each band
    	for (var i = 0; i <= numbands; i++) {
    		var bandmin = Math.pow(10,i+minscale);		// band min
    		var bandmax = Math.pow(10,i+minscale+1);	// band max
    		var bandshift = bandmax / numTicks;			// band part
	    	for (var j = 0; j < numTicks-1; j++) {	    		
	    		// log pos	    		
	    		var number = bandmin + j * bandshift;  // number for text
				if(number != 0 && number >= scale[0] && number <= scale[1]) {
		    		var step = Math.log(number) / Math.log(10);
		    		if(orientation == -1) {
		    			var pos = cbottom - (step - minscale) * sizeratio;
						moveTo(cleft,pos);
						lineTo(cright,pos);
		    		} else { 
		    			var pos = cleft + (step - minscale) * sizeratio;
						moveTo(pos,ctop);
						lineTo(pos,cbottom);
					}
				}  	   		    		    					
			}
		}
	    return path;
	},
	
	/**
	 * Build the cells for mesh
	*/
	buildCells: function(a, b, cells, field, fieldGivenPerCell) {    
		var cellA;
		var cellB;
		var cellZ;
		var cellVector;
		
	    if (cells != null && cells.length>0) {
	      var nCells = cells.length;
	      cellA = new Array(nCells);
	      cellB = new Array(nCells);
	      cellZ = new Array(nCells);
	      cellVector = new Array(nCells);
	      var dimension = 1;
	      if (field != null) {
	        if (fieldGivenPerCell) dimension = field[0][0].length;
	        else dimension = field[0].length;
	      }
	      for (var i=0; i<nCells; i++) {
	        var cell = cells[i];
	        var nCellPoints = cell.length;
	        cellA[i] = new Array(nCellPoints);
	        cellB[i] = new Array(nCellPoints);
	        for (var j=0; j<nCellPoints; j++) {
	          var p = cell[j];
	          cellA[i][j] = a[p];
	          cellB[i][j] = b[p];
	        }
	        if (field != null) {
		      cellZ[i] = new Array(nCellPoints);        
		      cellVector[i] = new Array(nCellPoints);        
	          if (!fieldGivenPerCell) {
	            for (var j=0; j<nCellPoints; j++) {
	              var p = cell[j];
	              cellZ[i][j] = (dimension<=1) ? field[p][0] : EJSS_TOOLS.Mathematics.norm(field[p]);
	              cellVector[i][j] = new Array(dimension);
	              for (var k=0; k<dimension; k++) {
	                cellVector[i][j][k] = field[p][k];
	              }
	            }
	          }
	          else {
	            var cellField = field[i];
	            for (var j=0; j<nCellPoints; j++) {
	              cellZ[i][j] = (dimension<=1) ? cellField[j][0] : EJSS_TOOLS.Mathematics.norm(cellField[j]);
	              cellVector[i][j] = new Array(dimension);
	              for (var k=0; k<dimension; k++) {
	                cellVector[i][j][k] = cellField[j][k];
	              }
	            }
	          }
	        }        
	      }
	    }
	   	return {cellA: cellA, cellB: cellB, cellZ: cellZ, cellVector: cellVector};
	},
		
	/**
	 * Fill cells with the given values for mesh
	 */
	drawPolygons: function(cells, color, drawCell, fillCell){ 
		var cellA = cells.cellA;
		var cellB = cells.cellB;
		var cellZ = cells.cellZ;
		
	    for (var i=0, n=cellA.length; i<n; i++) {
		    var a = cellA[i];
		    var b = cellB[i];
			var values = cellZ[i]; 
		
			if(typeof values != 'undefined') {
				// fill cell	
			    var nVertex = a.length;
			    var vertexIndex = new Array(nVertex);
			
			    // Compute vertex indexes and their range
			    var minIndex = Number.MAX_VALUE, maxIndex = Number.MIN_VALUE; 
			    for(var j=0; j<nVertex; j++) {
			      var valueIndex = color.doubleToIndex(values[j]); // from -1 to numColors
			      minIndex = Math.min(valueIndex, minIndex);
			      maxIndex = Math.max(valueIndex, maxIndex);
			      vertexIndex[j] = valueIndex; 
			    }
			
			    // Computes the subpoligon in each region
			    var mThresholds = color.getColorThresholds()
			    
			    var newCornersX = new Array(2*nVertex);
			    var newCornersY = new Array(2*nVertex);    
			    for (var levelIndex = minIndex; levelIndex<=maxIndex; levelIndex++) { // for each level affected
			      var pointsAdded = 0;
			      for (var point = 0; point<nVertex; point++) { // for each point
			        var nextPoint = (point+1) % nVertex;
			
			        if (vertexIndex[point]<=levelIndex && vertexIndex[nextPoint]>=levelIndex) { // There is a bottom-up change of level
			          if (vertexIndex[point]==levelIndex) { // the point is on the current level
			            newCornersX[pointsAdded] = a[point];
			            newCornersY[pointsAdded] = b[point];
			            pointsAdded++;
			          }
			          else { // Add the point where the change occurs
			            //(x,y,levels[l]) = (x_p, y_p, z_p) + lambda((x_n, y_n, z_n)-(x_p, y_p, z_p))
			            //where (x_p, y_p) are point's coord and z_p can be calculated as the projection of (x_p, y_p) over levels vector 
			            //and (x_n, y_n) are next's coord and z_n can be calculated as the projection of (x_n, y_n) over levels vector
			            var lambda = (mThresholds[levelIndex]-values[point])/(values[nextPoint]-values[point]);
			            newCornersX[pointsAdded] = Math.round(a[point]+lambda*(a[nextPoint]-a[point]));
			            newCornersY[pointsAdded] = Math.round(b[point]+lambda*(b[nextPoint]-b[point]));
			            /*
					    if (newCornersX[pointsAdded]>300) {
					     console.log("level 1 added = "+newCornersX[pointsAdded]+ " division by "+(values[nextPoint]-values[point]));
					     console.log("Values = "+values[point]+ ", "+values[nextPoint]);
					     console.log("point index = "+vertexIndex[point]+ " next point index = "+vertexIndex[nextPoint]);
					     console.log("Should be point index = "+color.doubleToIndex(values[point])+ " next point index = "+color.doubleToIndex(values[nextPoint]));
					     console.log("Points= "+a[point]+ ", "+a[nextPoint]);
					     console.log("lambda= "+lambda);
			       for (var i=0,n=mThresholds.length; i<n; i++) console.log("Thresholds "+mThresholds[i]);
					     console.log("mThreshold= "+mThresholds[levelIndex]+ " level index = "+levelIndex);
					    }
					    */
			            pointsAdded++;        
			          }
			          if (vertexIndex[nextPoint]>levelIndex) { // This segment contributes with a second point
			            var lambda = (mThresholds[levelIndex+1]-values[point])/(values[nextPoint]-values[point]);
			            newCornersX[pointsAdded] = Math.round(a[point]+lambda*(a[nextPoint]-a[point]));
			            newCornersY[pointsAdded] = Math.round(b[point]+lambda*(b[nextPoint]-b[point]));
			            pointsAdded++;
			          }
			        } //end bottom-up
			        else if (vertexIndex[point]>=levelIndex && vertexIndex[nextPoint]<=levelIndex) { // There is a top-down change of level
			          if (vertexIndex[point]==levelIndex) { // the point is on the current level
			            newCornersX[pointsAdded] = a[point];
			            newCornersY[pointsAdded] = b[point];
			            pointsAdded++;
			          }
			          else { // Add the point where the change occurs
			            var lambda = (mThresholds[levelIndex+1]-values[point])/(values[nextPoint]-values[point]);
			            newCornersX[pointsAdded] = Math.round(a[point]+lambda*(a[nextPoint]-a[point]));
			            newCornersY[pointsAdded] = Math.round(b[point]+lambda*(b[nextPoint]-b[point]));
			            pointsAdded++;
			          }
			          if (vertexIndex[nextPoint]<levelIndex) { // This segment contributes with a second point
			            var lambda = (mThresholds[levelIndex]-values[point])/(values[nextPoint]-values[point]);
			            newCornersX[pointsAdded] = Math.round(a[point]+lambda*(a[nextPoint]-a[point]));
			            newCornersY[pointsAdded] = Math.round(b[point]+lambda*(b[nextPoint]-b[point]));
			            pointsAdded++;
			          }
			        }    
			      } // end for each point
			
			      if (pointsAdded>0) { // Draw the subpoligon
			      	  fillCell(newCornersX, newCornersY, pointsAdded, color, levelIndex);
			      }      	  
			    } //end for each level
			}
			
			// draw cell
			drawCell(a,b);
		}
	},
	
	
	/**
	 * Build the boundary for mesh
	*/
	buildBoundary: function(a, b, boundary) {        
	  	var boundaryA;
	  	var boundaryB;
	  	
	    if (boundary!=null && boundary.length>0) {
	      var num = boundary.length;
	      boundaryA = new Array(num);
	      boundaryB = new Array(num);
	      for (var i=0; i<num; i++) {
	        var segment = boundary[i];
	        var len = segment.length;
	        boundaryA[i] = new Array(len);
	        boundaryB[i] = new Array(len);
	        for (var j=0; j<len; j++) {
	          var p = segment[j];
	          boundaryA[i][j] = a[p];
	          boundaryB[i][j] = b[p];
	        }
	      }
	    }
	    return {boundaryA: boundaryA, boundaryB: boundaryB};
	},
	
	/**
	 * Build the boundary for mesh
	*/
	drawBoundary: function(boundary, labels, colors, drawSegment) {   
 		var boundaryA = boundary.boundaryA;
 		var boundaryB = boundary.boundaryB;
 		var segmentColor = 'black';
	    for (var i=0, n=boundaryA.length; i<n; i++) {
	        if (labels != null) {
	          var colorIndex = labels[i];
	          if ( colors!=null && colorIndex<colors.length) 
	          	segmentColor = colors[colorIndex];
	        }
	        drawSegment(boundaryA[i],boundaryB[i],segmentColor);
	    }     
	},
	
	/**
	 * Draw Image Field in Canvas
	*/
	drawImageField: function(ctx, pxTopLeftX, pxTopLeftY, xMin, yMin, pixWth, pixHgt, pxPerUnitX, pxPerUnitY, shape) {
	     function xToPix(x) {
		   return pxTopLeftX +(x - xMin) * pxPerUnitX;
		 }
		  
		 function yToPix(y) {
		   return pxTopLeftY -(y - yMin) * pxPerUnitY;
		 }
	
	    var left=Math.round(xToPix(shape.xMin));
	    var top=Math.round(yToPix(shape.yMax)); 
	    var width=Math.round(pixWth); 
	    var height=Math.round(pixHgt);
	    var imageData = ctx.getImageData(left, top, width, height);  // parameters must be integers
	    var maskpx=this.maskRadius*pxPerUnitX;
	    var r=0;
	    var g=0;
	    var b=0;
	    for (var i=0; i<width; i++) {
	      var ii= Math.floor(i*shape.nx/width);
	      //console.log("i="+i+"  ii="+ii);
	      for (var j=0; j<height; j++) {
	        var jj=Math.floor(j*shape.ny/height);
	        g=0;
	        if(shape.data[ii][jj]<shape.lower){
	          b=255;  // blue below lower
	          r=0;
	        }else if(shape.data[ii][jj]<shape.center){
	          b=255*(shape.center-shape.data[ii][jj])/(shape.center-shape.lower);
	          r=0;
	        }else if(shape.data[ii][jj]<shape.upper){
	          b=0;
	          r=255*(shape.data[ii][jj]-shape.center)/(shape.upper-shape.center);
	        }else{  // red above upper
	          b=0;
	          r=255;
	        }
	        if(shape.maskRadius>0){  // check for circular mask
	          //var ri=Math.sqrt(this.xPos[ii]*this.xPos[ii]+this.yPos[jj]*this.yPos[jj]);
	          //if(ri>this.maskRadius){
	          if(Math.sqrt((i-width/2)*(i-width/2)+(j-height/2)*(j-height/2))>maskpx){
	            r=g=224;
	            b=255;
	          }
	        }
	        var index = 4*i+4*width*j;
	        imageData.data[index+0] = r;  //red
	        imageData.data[index+1] = g;  //green
	        imageData.data[index+2] = b;  // blue
	      }
	    }
	    ctx.putImageData(imageData, left, top);		
	}	
};

/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG analyticCurve
 */
EJSS_SVGGRAPHICS.analyticCurve = function(mGraphics, mElement) {
	
	// generates path for analyticCurve
	function pathForanalyticCurve(points, x, y, sx, sy) {
		var path = "";
		for(var i=0; i<points.length; i++) {
		  var point = points[i];	  
	      var xx = x + point[0]*sx;
	      var yy = y + point[1]*sy;
	      var type = point[2];		
		  if ((i==0) || (type == 0)) // 0 is NOT CONNECTION
		  	path += " M " + xx + " " + yy;		// move 
	      else       	
	      	path += " L " + xx + " " + yy;		// line
		}  	  	
	    return path;
	}  	

	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());	
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);	    
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;

	// calculate points	    	 
	var numPoints = mElement.getNumPoints();
	var mMinimun = mElement.getMinimun();
	var mMaximun = mElement.getMaximun();
		var min = ( (typeof mMinimun == "undefined" || mMinimun === null) ?  mElement.getPanel().getRealWorldXMin() : mMinimun);
		var max = ( (typeof mMaximun == "undefined" || mMaximun === null) ?  mElement.getPanel().getRealWorldXMax() : mMaximun);
	
//	var min = (mElement.getMinimun()? mElement.getMinimun() : mElement.getPanel().getRealWorldXMin());
//	var max = (mElement.getMaximun()? mElement.getMaximun() : mElement.getPanel().getRealWorldXMax());
	var vble = mElement.getVariable();
	var fx = mElement.getFunctionX();
	var fy = mElement.getFunctionY();	
   	var parser = EJSS_DRAWING2D.functionsParser();
   	
    var exprfx;
	var exprfy;
	var mustReturn = false;
	try {
	  exprfx = parser.parse(fx);
	}
	catch (errorfx) {
  	  console.log ("Analytic curve error parsing FunctionX: "+fx);
	  mustReturn = true;
	}
	if (!mustReturn) {
	  try {
	    exprfy = parser.parse(fy);
	  }
	  catch (errorfy) {
  	    console.log ("Analytic curve error parsing FunctionY: "+fy);
	   	mustReturn = true;
	  }
	}
	if (mustReturn) {
	  mElement.getController().invokeAction("OnError");
	  return;
	}
	
   	var step = (max-min)/(numPoints-1);
   	
	var points = [];
    var vblevalue = {};
    var parameters = mElement.getParameters();
    for (var param in parameters) { 
      vblevalue[param] = parameters[param];
	 }
    
   	try {
   	  for(var j=0, i=min; i<=max; i+=step) {
   		vblevalue[vble] = i;
   		  var fxvalue = exprfx.evaluate(vblevalue);
   		  var fyvalue = exprfy.evaluate(vblevalue);
	   	  if(!isNaN(fxvalue) && !isNaN(fyvalue)) {   		
	   		points[j] = [];
	    	points[j][0] = fxvalue;	
			points[j++][1] = fyvalue;
		  }
		}
	}
	catch(error) {
	  mElement.getController().invokeAction("OnError");
	 } // do not complain
	
	// draw points
    var analyticCurvepath = pathForanalyticCurve(points, x-mx, y-my, size[0], size[1])     
    if(analyticCurvepath !== "") {
    	mShape.setAttribute('d', analyticCurvepath);
    		
		// set style
	    var style = mElement.getStyle(); 
	    if(style.getDrawFill())	
	    	mShape.setAttribute("fill",style.getFillColor());
	    else 
	    	mShape.setAttribute("fill","none");    
	    if(style.getDrawLines()) {
	    	mShape.setAttribute("stroke",style.getLineColor());
	    	mShape.setAttribute("stroke-width",style.getLineWidth());
	    } else {
	    	mShape.setAttribute("stroke","none");
	    	mShape.setAttribute("stroke-width",0);    	
	    }        
		mShape.setAttribute("shapeRendering",style.getShapeRendering());  	
		
	    var attributes = style.getAttributes();
	    for (var attr in attributes) {
	      mShape.setAttribute(attr,attributes[attr]);
	    }		
	}

	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG arrow
 */
EJSS_SVGGRAPHICS.arrow = function(mGraphics, mElement) {  
	var UTILS = EJSS_SVGGRAPHICS.Utils;
	var Arrow = EJSS_DRAWING2D.Arrow;
	
	function createMarkDef(name, mark, orient, width, height, stroke, color, color2, sizex, sizey) {
		// create marker
		var path, path2;		
		var marker = mGraphics.getElementById(name);						
		if(marker === null) {
			marker = document.createElementNS('http://www.w3.org/2000/svg', 'marker');
			marker.setAttribute('id', name);
			path = document.createElementNS('http://www.w3.org/2000/svg', 'path');
			path.setAttribute('id', name + ".markpath");
			marker.appendChild(path); 					
			path2 = document.createElementNS('http://www.w3.org/2000/svg', 'path');
			path2.setAttribute('id', name + ".markpath2");
			marker.appendChild(path2); 					
			mGraphics.getElementsByTagName('defs')[0].appendChild(marker);
		} else {
			path = mGraphics.getElementById(name + ".markpath");		
			path2 = mGraphics.getElementById(name + ".markpath2");
			// for bug in IE
			mGraphics.getElementsByTagName('defs')[0].insertBefore(marker, marker);						
		}
				
		var vwidth = width + 2*stroke;
		var vheight = height + 2*stroke;
		marker.setAttribute('viewBox', (-stroke) + " " + (-stroke) + " " + vwidth + " " + vheight);
		marker.setAttribute('refX', width/2);
		marker.setAttribute('refY', height/2);
		marker.setAttribute('markerUnits', 'userSpaceOnUse');
		marker.setAttribute('markerWidth', vwidth);
		marker.setAttribute('markerHeight', vheight);
		marker.setAttribute('orient', orient);

		// create path
		path.setAttribute('fill', color);								
		path.setAttribute('stroke', color);   
		path.setAttribute('stroke-width', stroke);		
		path.setAttribute('style', '  marker-start :none;   marker-end :none; ');
		path2.setAttribute('fill', color2);								
		path2.setAttribute('stroke', color2);   
		path2.setAttribute('stroke-width', stroke);		
		path2.setAttribute('style', '  marker-start :none;   marker-end :none; ');
		switch(mark) {
			case Arrow.TRIANGLE:				
				var str = "M 0 0 L " + width + " " + (height/2) + " L 0 " + height + " z";
				path.setAttribute('d', str);
				break;			
			case Arrow.INVTRIANGLE:		
				var str = "M " + width + " 0 L 0 " + (height/2) + " L " + width + " " + height + " z";
				path.setAttribute('d', str);
				break;			
			case Arrow.ANGLE:
				// cut extremes
				// marker.setAttribute('viewBox', "0 0 " + vwidth + " " + height);
				marker.setAttribute('markerHeight', height);
				marker.setAttribute('refX', width);
				// create angle
				var p1 = -(stroke-1);
				var p2 = height+(stroke-1);					
				var str = "M 0 " + p1 + " L " + width + " " + (height/2) + " L 0 " + p2;				  
				path.setAttribute('d', str);
				path.setAttribute('fill', 'none');								
				break;			
			case Arrow.INVANGLE:
				// cut extremes
				// marker.setAttribute('viewBox', "0 0 " + vwidth + " " + height);
				marker.setAttribute('markerHeight', height);
				marker.setAttribute('refX', 1);
				// create angle
				var p1 = -(stroke-1);
				var p2 = height+(stroke-1);					
				var str = "M " + width + " " + p1 + " L 0 " + (height/2) + " L " + width + " " + p2;				  
				path.setAttribute('d', str);
				path.setAttribute('fill', 'none');								
				break;			
			case Arrow.LINE:				
				var str = "M " + (width/2) + " 0 L " + (width/2) + " " + height + " z"; 
				path.setAttribute('d', str);
				break;			
			case Arrow.RECTANGLE:			
				var str = "M 0 0 L " + width + " 0 L " + width + " " + height + " L 0 " + height + " z"; 
				path.setAttribute('d', str);
				break;			
			case Arrow.DIAMOND:
				var str = "M " + (width/2) + " 0 L " + width + " " + (height/2) + " L " + (width/2) + " " + height + " L 0 " + (height/2) + " z"; 
				path.setAttribute('d', str);
				break;			
			case Arrow.CIRCLE:
				var radius = (width < height) ? width : height;
				radius /= 2;
				var str = "M " + (width/2) + " " + (height/2) + " m " + (-radius) + ", 0 a " + radius + "," + radius + " 0 1,0 " + (radius*2) + ",0" + 
					" a " + radius + "," + radius + " 0 1,0 " + (-radius*2) + ",0"; 				  
				path.setAttribute('d', str);
				break;
			case Arrow.WEDGE:
				// without stroke
				path.setAttribute('stroke-width', 0);
				path2.setAttribute('stroke-width', 0.5);
				// create wedge (fixed position)
				var angle_rot = ((typeof sizey === "undefined") || (sizey===0))? 0 : Math.atan(sizex/sizey);
				var radius = (width < height) ? width : height;
				radius /= 2;
				var x = (width/2);
				var y = (height/2);
				var p11 = x + radius*Math.cos(angle_rot);
				var p12 = y - radius*Math.sin(angle_rot);		
				var p21 = x + radius*Math.cos(Math.PI/3.25 - angle_rot);
				var p22 = y + radius*Math.sin(Math.PI/3.25 - angle_rot);				
				var wedge = " M " + x + " " + y + " L " + p11 + "," + p12 + " A " + radius + "," + radius + " 0 0,1 " + p21 + "," + p22; 				  
				var circle = "M " + x + " " + y + " m " + (-radius) + ", 0 a " + radius + "," + radius + " 0 1,0 " + (radius*2) + ",0" + 
					 " a " + radius + "," + radius + " 0 1,0 " + (-radius*2) + ",0 ";
				path.setAttribute('d', circle);
				path2.setAttribute('d', wedge);				
				break;		
			case Arrow.POINTED:
				// without stroke
				marker.setAttribute('refX', width);
				path.setAttribute('stroke-width', 0);
				// create pointed mark
				var str = "M 0 0 L " + width + " " + (height/2) + " L 0 " + height + 
					" L " + (width/2) + " " + (height/2) + " L 0 0 z";
				path.setAttribute('d', str);
				break;			
			case Arrow.CURVE:
				// without stroke
				marker.setAttribute('refX', width);
				path.setAttribute('stroke-width', 0);
				// create curve mark
				var radius1 = width*3;
				var radius2 = height/2.5;
				var str = "M 0 0 " +
					" A " + radius1 + "," + radius1 + " 1 0,0 " + width + "," + (height/2) + 
					" A " + radius1 + "," + radius1 + " 1 0,0 " + " 0," + height + 
					" A " + radius2 + "," + radius2 + " 1 0,0 " + " 0," + (height/2) + 
					" A " + radius2 + "," + radius2 + " 1 0,0 " + " 0,0 z";
				path.setAttribute('d', str);
				break;								
		}		
		
	}
		
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get line
	var mLine = mGraphics.getElementById(mElement.getName());	
	if (mLine === null) { 	// exits?
	    mLine = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    mLine.setAttribute("id", mElement.getName());
	    group.appendChild(mLine);	    
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
    
	// draw tip
	if((size[0] != 0) || (size[1] != 0)) {    			
		// create end mark	
		if(mElement.getMarkEnd() != Arrow.NONE) {
			var endMax = (mElement.getMarkProportion() * EJSS_TOOLS.Mathematics.norm([size[0],size[1]])) ;
			var endW = (endMax != 0 && endMax < mElement.getMarkEndWidth()) ? endMax : mElement.getMarkEndWidth();
			var endH = (endMax != 0 && endMax < mElement.getMarkEndHeight()) ? endMax : mElement.getMarkEndHeight();
			var color = (mElement.getMarkEndColor() == "none") ? mElement.getStyle().getLineColor() : mElement.getMarkEndColor();
			var stroke = (mElement.getMarkEndStroke() < 0) ? mElement.getStyle().getLineWidth() : mElement.getMarkEndStroke();   
			createMarkDef(mElement.getName() + ".end", mElement.getMarkEnd(), mElement.getMarkEndOrient(), endW, endH, stroke, color);
			mLine.setAttribute('marker-end', 'url(#' + mElement.getName() + ".end" + ')');
		} else {
			mLine.removeAttribute('marker-end');
		}
				
		// create mid mark
		if(mElement.getMarkMiddle() != Arrow.NONE) {
			var midMax = (mElement.getMarkProportion() * EJSS_TOOLS.Mathematics.norm([size[0],size[1]])) ;
			var midW = (midMax != 0 && midMax < mElement.getMarkMiddleWidth()) ? midMax : mElement.getMarkMiddleWidth();
			var midH = (midMax != 0 && midMax < mElement.getMarkMiddleHeight()) ? midMax : mElement.getMarkMiddleHeight();
			var color = (mElement.getMarkMiddleColor() == "none") ? mElement.getStyle().getLineColor() : mElement.getMarkMiddleColor();
			var colorsnd = (mElement.getMarkMiddleColorSnd() == "none") ? mElement.getStyle().getLineColor() : mElement.getMarkMiddleColorSnd();
			var stroke = (mElement.getMarkMiddleStroke() < 0) ? mElement.getStyle().getLineWidth() : mElement.getMarkMiddleStroke();						
			createMarkDef(mElement.getName() + ".mid", mElement.getMarkMiddle(), mElement.getMarkMiddleOrient(), midW, midH, stroke, color, colorsnd, size[0], size[1]);
			mLine.setAttribute('marker-mid', 'url(#' + mElement.getName() + ".mid" + ')');
		} else {
			mLine.removeAttribute('marker-mid');
		}
		
		// create start mark
		if(mElement.getMarkStart() != Arrow.NONE) {
			var stMax = (mElement.getMarkProportion() * EJSS_TOOLS.Mathematics.norm([size[0],size[1]])) ;
			var stW = (stMax != 0 && stMax < mElement.getMarkStartWidth()) ? stMax : mElement.getMarkStartWidth();
			var stH = (stMax != 0 && stMax < mElement.getMarkStartHeight()) ? stMax : mElement.getMarkStartHeight();		
			var color = (mElement.getMarkStartColor() == "none") ? mElement.getStyle().getLineColor() : mElement.getMarkStartColor();
			var stroke = (mElement.getMarkStartStroke() < 0) ? mElement.getStyle().getLineWidth() : mElement.getMarkStartStroke();						
			createMarkDef(mElement.getName() + ".start", mElement.getMarkStart(), mElement.getMarkStartOrient(), stW, stH, stroke, color);		
			mLine.setAttribute('marker-start', 'url(#' + mElement.getName() + ".start" + ')');			
		} else {
			mLine.removeAttribute('marker-start');
		}
	} else {
		mLine.removeAttribute('marker-end');
		mLine.removeAttribute('marker-mid');
		mLine.removeAttribute('marker-start');		
	}
		
	// draw line
	  var CR = UTILS.crispValue;

	  var style = mElement.getStyle();
    switch (mElement.getSplineType()) {
      default : 
      case Arrow.SPLINE_NONE :
        if(style.getShapeRendering() == "crispEdges") {
          if(mElement.getMarkMiddle() != Arrow.NONE) {  // point in the middle
              mLine.setAttribute('d', "M " + CR(x-mx) + " " + CR(y-my) + 
                " L " + CR(x) + " " + CR(y) +
                " L " + CR(x+mx) + " " + CR(y+my));   
          } else {
              mLine.setAttribute('d', "M " + CR(x-mx) + " " + CR(y-my) +
                " L " + CR(x+mx) + " " + CR(y+my));
            }
          } else {
          if(mElement.getMarkMiddle() != Arrow.NONE) {  // point in the middle
              mLine.setAttribute('d', "M " + (x-mx) + " " + (y-my) + 
                  " L " + (x) + " " + (y) + 
                  " L " + (x+mx) + " " + (y+my));   
          } else {
              mLine.setAttribute('d', "M " + (x-mx) + " " + (y-my) + 
                  " L " + (x+mx) + " " + (y+my));
           }
        }       
        break;
      case Arrow.SPLINE_CUBIC :
        var controlX = Math.abs(mx)*0.25;
        if(style.getShapeRendering() == "crispEdges") {
          if(mElement.getMarkMiddle() != Arrow.NONE) {  // point in the middle
              mLine.setAttribute('d', "M " + CR(x-mx) + " " + CR(y-my) + 
                  " Q " + CR(x-controlX) + " " + CR(y-my) +", " + CR(x) + " " + CR(y) +
                  " Q " + CR(x+controlX) + " " + CR(y+my)+ ", " + CR(x+mx) + " " + CR(y+my));   
          } else {
            mLine.setAttribute('d', "M " + CR(x-mx) + " " + CR(y-my) +
                " C " + CR(x) + " " + CR(y-my) +", " + CR(x) +
                " " + CR(y+my)+ ", " + CR(x+mx) + " " + CR(y+my));
            }
          } else {
          if(mElement.getMarkMiddle() != Arrow.NONE) {  // point in the middle
              mLine.setAttribute('d', "M " + (x-mx) + " " + (y-my) + 
                  " Q " + (x-controlX) + " " + (y-my) + ", " + (x)    + " " + (y) +
                  " Q " + (x+controlX) + " " + (y+my) + ", " + (x+mx) + " " + (y+my) 
                  );   
//                  " C " + (x)    + " " + (y-my) +", " + (x) +" " + (y-my)+ ", " + (x)    + " " + (y) +
//                  " C " + (x) + " " + (y+my) +", " + (x)    +" " + (y+my)+ ", " + (x+mx) + " " + (y+my));   
          } else {
              mLine.setAttribute('d', "M " + (x-mx) + " " + (y-my) + 
                  " C " + x + " " + (y-my) +", " + x +" " + (y+my)+ ", " + (x+mx) + " " + (y+my) +
                  " L " + (x+mx) + " " + (y+my));
           }
        }       
        break;
    }

	// set style
    if(style.getDrawFill()) 	
    	mLine.setAttribute("fill",style.getFillColor());    	
    else  
    	mLine.setAttribute("fill","none");
    if(style.getDrawLines()) {
    	mLine.setAttribute("stroke",style.getLineColor());
    	mLine.setAttribute("stroke-width",style.getLineWidth());
    } else {
    	mLine.setAttribute("stroke","none");
    	mLine.setAttribute("stroke-width",0);    	
    }        
	mLine.setAttribute("shapeRendering",style.getShapeRendering());  	

    var attributes = style.getAttributes();
    for (var attr in attributes) {
      mLine.setAttribute(attr,attributes[attr]);
    }

	return mLine;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG axis
 */
EJSS_SVGGRAPHICS.axis = function(mGraphics, mElement) {
	var UTILS = EJSS_SVGGRAPHICS.Utils;

	// draw axis line
	function drawLine(group, x, y, mx, my, style, orient, inverted) {
	    var mLine = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    group.appendChild(mLine);	    

		if(orient == EJSS_DRAWING2D.Axis.AXIS_VERTICAL) {	// vertical axis
			if(inverted) {
				my = -my;
				y = -y;	
			}
		 	// set attributes	    	
		    if(style.getShapeRendering() == "crispEdges") {
		    	mLine.setAttribute('d', "M " + UTILS.crispValue(x) + " " + UTILS.crispValue(y-0.5-my) 
		    		+ " L " + UTILS.crispValue(x) + " " + UTILS.crispValue(y-0.5+my));
		    } else {
		    	mLine.setAttribute('d', "M " + (x) + " " + (y-my) + " L " + (x) + " " + (y+my));
			}  	   		    		    		
		} else {	// horizontal axis
		 	// set attributes	    	
		    if(style.getShapeRendering() == "crispEdges") {
		    	mLine.setAttribute('d', "M " + UTILS.crispValue(x+0.5-mx) + " " + UTILS.crispValue(y) 
		    		+ " L " + UTILS.crispValue(x+0.5+mx) + " " + UTILS.crispValue(y));
		    } else {
		    	mLine.setAttribute('d', "M " + (x-mx) + " " + (y) + " L " + (x+mx) + " " + (y));
			}  	   		
		}		
	}

	function drawTick(group, x, y, ticksize, style, orient) {
		if(orient == EJSS_DRAWING2D.Axis.AXIS_VERTICAL) {	// vertical axis
		    var mTick = document.createElementNS("http://www.w3.org/2000/svg","path"); 
		    group.appendChild(mTick);
		    if(style.getShapeRendering() == "crispEdges") {
		    	mTick.setAttribute('d', "M " + UTILS.crispValue(x-ticksize/2) + " " + UTILS.crispValue(y) 
		    		+ " L " + UTILS.crispValue(x+ticksize/2) + " " + UTILS.crispValue(y));
		    } else {
		    	mTick.setAttribute('d', "M " + (x-ticksize/2) + " " + (y) + " L " + (x+ticksize/2) + " " + (y));
			}  	   					
		} else {
		    var mTick = document.createElementNS("http://www.w3.org/2000/svg","path"); 
		    group.appendChild(mTick);
		    if(style.getShapeRendering() == "crispEdges") {
		    	mTick.setAttribute('d', "M " + UTILS.crispValue(x) + " " + UTILS.crispValue(y-ticksize/2) 
		    		+ " L " + UTILS.crispValue(x) + " " + UTILS.crispValue(y+ticksize/2));
		    } else {
		    	mTick.setAttribute('d', "M " + (x) + " " + (y-ticksize/2) + " L " + (x) + " " + (y+ticksize/2));
			}  	   		    		    								
		}					
	}	
	
	// text for tick in position (x,y) with font and mode
	function tickText (group, x, y, text, font, horizontal) {
	    // create SVG element group
	    var mTxtGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    group.appendChild(mTxtGroup);	    
	    // create text element
	    var mText = document.createElementNS("http://www.w3.org/2000/svg","text"); 
	    mTxtGroup.appendChild(mText);	    
				    
		// set attributes	    
		mText.setAttribute("font-family",font.getFontFamily());
		mText.setAttribute("font-size",font.getFontSizeString());
	    mText.setAttribute("fill",font.getFillColor());
    	mText.setAttribute("stroke",font.getOutlineColor());    	
	    mText.setAttribute("stroke-width",font.getFontWeight());
	    mText.setAttribute("letterSpacing",font.getLetterSpacing());
	    mText.textContent = text;
		if (horizontal) {
		  x -= mText.getComputedTextLength()/2 + 0.5;
		}	
		else { // vertical axis
		  y += font.getFontSize()/2; // - 0.5;
		}
		mText.setAttribute("x",x);
		mText.setAttribute("y",y);
	    
		// angle of text  	 
/*		var angletext = 0;
		switch(mode) {
			case EJSS_DRAWING2D.Text.MODE_TOPDOWN : angletext = 90; break;
			case EJSS_DRAWING2D.Text.MODE_RIGTHLEFT : angletext = 180; break;
			case EJSS_DRAWING2D.Text.MODE_DOWNTOP : angletext = 270; break;
			case EJSS_DRAWING2D.Text.MODE_LEFTRIGHT: angletext = 0; break;
		}
	    mText.setAttribute("transform","rotate(" + angletext + " " + x + " " + y +")"); 		
*/ 	  	  
	}		

	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // remove SVG element (not reusing element)
	var mGroup = mGraphics.getElementById(mElement.getName());	
	if (mGroup !== null) group.removeChild(mGroup);
	
	if(mElement.getShow()) { // show
	
	    // create SVG element
	    mGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    mGroup.setAttribute("id", mElement.getName());
	    group.appendChild(mGroup);	    
	
		// get position of the mElement center 
	    var pos = mElement.getPixelPosition();
	    var size = mElement.getPixelSizes();     
	    var offset = mElement.getRelativePositionOffset(size);  
	    var x = pos[0]+offset[0];
	    var y = pos[1]+offset[1];
		
		// get half sizes 		
	    var mx = size[0]/2;
	    var my = size[1]/2;

		// properties	    
	    var style = mElement.getStyle();		// element style   	 	
		var orient = mElement.getOrient();		// axis orientation (vertical or horizontal)
		var inverted = mElement.getInvertedScaleY();
	    
	    // draw the line for axis
		drawLine(mGroup, x, y, mx, my, style, orient, inverted);
		 		
		// get axis size in pixel
		var segsize = (orient == EJSS_DRAWING2D.Axis.AXIS_VERTICAL)? Math.abs(size[1]):Math.abs(size[0]);		
		 		
		// draw axis (based on ticks mode)
		var ticksmode = mElement.getTicksMode();			// decimal or logarithmic
		if (ticksmode == EJSS_DRAWING2D.Axis.SCALE_LOG) { // logarithmic

		    var scale = mElement.getScale();		// axis scale 			
			if(scale[0] > 0 && scale[1] > 0) {  // valid scale			
				// get number of axis ticks
				var ticks = mElement.getTicks(); 	
	
			    // scale
				var scalePrecision = mElement.getScalePrecision();	// number of decimals for text
				
				// draw ticks 
				var ticksize = mElement.getTicksSize();				// ticks size in pixels
				EJSS_GRAPHICS.GraphicsUtils.drawLogTicks(x, y, mx, my, segsize, ticksize, scale, ticks, orient, 
						function(x, y, ticksize, orient) {
							drawTick(mGroup, x, y, ticksize, style, orient);			
						});		
			
				// draw ticks text 
				var font = mElement.getFont();
			    var textPosition = mElement.getTextPosition();		// text position (UP or DOWN)
				EJSS_GRAPHICS.GraphicsUtils.drawLogTicksText (x, y, mx, my, segsize, ticks, ticksize, scale, scalePrecision, font, textPosition, orient,
						function(x, y, text, font, horizontal) {
							tickText (mGroup, x, y, text, font, horizontal);
						});			
			}
			
		} else if (ticksmode == EJSS_DRAWING2D.Axis.SCALE_NUM) { // decimal
			 			 		
			// calculate step in pixels	
		    var step = 0;     // step for ticks (pixels)
		    var tickstep = 0; // step for ticks (real units)
			if (!mElement.getAutoTicks()) {		// no auto-ticks
		    	var ticks = mElement.getTicks();	// number of ticks
				// whether the number of ticks exits, changes step for ticks and scale 
			    if (ticks != 0) { step = segsize/ticks; } else {
			    	step = mElement.getStep();
			    	tickstep = mElement.getTickStep(); 
			    } 	    	
			} else {	// auto-ticks
				var stepmin = mElement.getAutoStepMin();		// step min in pixels
				var ticksrange = mElement.getAutoTicksRange();	// ticks range
				// find step based on ticks range
				for(var i=ticksrange.length-1; i>=0; i--)	{	
					step = Math.abs(segsize/ticksrange[i]);
					if (step*1.001 >= stepmin) break;
				}
			}
		    
			var scalePrecision = mElement.getScalePrecision();	// number of decimals for text
			var scale = mElement.getScale();		// axis scale 	
			// values for scale
		    if(tickstep == 0) {
				var scalestep = Math.abs((scale[1] - scale[0]) * step / segsize);  // step in axis scale
			} else {
				var scalestep = tickstep;
				step = Math.abs((scalestep * segsize) / (scale[1] - scale[0])); 
			}			
			
			// adjust step to decimals of precision
			var decimals = Math.pow(10,scalePrecision);
			var scalestepTmp = Math.round(scalestep * decimals) / decimals;
			if(scalestepTmp > 0) {
				scalestep = scalestepTmp; 
				step = Math.abs(scalestepTmp * segsize / (scale[1] - scale[0]));
			}
			
			// check fixed tick
			var fixedTicks = mElement.getFixedTick();	  
			var tickfixed = scale[1];
			if (!isNaN(fixedTicks)) {
			  if (fixedTicks < scale[0]) tickfixed = fixedTicks + (Math.floor((scale[0]-fixedTicks)/scalestep)+1)*scalestep;
			  else if (fixedTicks > scale[1])  tickfixed = fixedTicks - (Math.floor((fixedTicks-scale[1])/scalestep)+1)*scalestep;
		      else tickfixed = fixedTicks; 
			}		
	
			// tick fixed in axis scale
			var scaleshift = Math.abs((scale[0] - tickfixed) % scalestep);						
			var dist = Math.abs(scaleshift-scalestep);	// fitting shift
			if(scaleshift < 0.001 || dist < 0.001) scaleshift = 0;						
			var shift = segsize * scaleshift / Math.abs(scale[1] - scale[0]);	// shift in pixels				
										    		
			// draw ticks based on step and shift
			var ticksize = mElement.getTicksSize();				// ticks size in pixels
			EJSS_GRAPHICS.GraphicsUtils.drawDecTicks(x, y, mx, my, ticksize, step, shift, orient, inverted, 
					function(x, y, ticksize, orient) {
						drawTick(mGroup, x, y, ticksize, style, orient);
					});		
		
			// draw ticks text based on scaleshift, scalestep and scale
			var font = mElement.getFont();
		    var textPosition = mElement.getTextPosition();		// text position (UP or DOWN)
			EJSS_GRAPHICS.GraphicsUtils.drawDecTicksText (x, y, mx, my, ticksize, step, shift, scale, scalePrecision, scalestep, scaleshift, font, textPosition, orient, inverted, 
					function(x, y, text, font, horizontal) {
						tickText(mGroup, x, y, text, font, horizontal);
					});
		}
					
		// set style
	    if(style.getDrawFill()) 	
	    	mGroup.setAttribute("fill",style.getFillColor());
	    else 
	    	mGroup.setAttribute("fill","none");    
	    if(style.getDrawLines()) {
	    	mGroup.setAttribute("stroke",style.getLineColor());
	    	mGroup.setAttribute("stroke-width",style.getLineWidth());
	    } else {
	    	mGroup.setAttribute("stroke","none");
	    	mGroup.setAttribute("stroke-width",0);    	
	    }        
		mGroup.setAttribute("shapeRendering",style.getShapeRendering());  	
	
	    var attributes = style.getAttributes();
	    for (var attr in attributes) {
	      mGroup.setAttribute(attr,attributes[attr]);
	    }
	}
	
	return mGroup;         
}
/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG byteRaster
 */
EJSS_SVGGRAPHICS.byteRaster = function(mGraphics, mElement) {
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape		
	var mGroup = mGraphics.getElementById(mElement.getName());
	var mShape;
	if (mGroup == null) { 	// exists?
	    // create SVG mElement group
	    mGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    mGroup.setAttribute("id", mElement.getName());
	    group.appendChild(mGroup);	    
	    
		mShape = document.createElementNS("http://www.w3.org/2000/svg","image");
		mShape.setAttribute("id", mElement.getName() + ".img");
		mGroup.appendChild(mShape); 	    
	} else {
		mShape = mGraphics.getElementById(mElement.getName() + ".img");
	}
	
	if(mElement.getDataChanged()) { // new data or colors
		var data = mElement.getData();
		var colors = mElement.getColorMapper().getColors();			
		var xlen = data.length;
		if (xlen>0) {
		  var ylen = data[0].length;
		  var num = mElement.getColorMapper().getNumberOfColors();
		
		// Worker previous
		// var png = EJSS_SVGGRAPHICS.Utils.PNGCanvas (xlen, ylen, num, data, colors);		
	    // mShape.setAttributeNS('http://www.w3.org/1999/xlink', 'xlink:href', png);	    
	    // mElement.setDataChanged(false);		 	    	    	

		  // using worker
		  EJSS_SVGGRAPHICS.Utils.PNGCanvas (xlen, ylen, num, data, colors,
			function(png) {
				mShape.setAttributeNS('http://www.w3.org/1999/xlink', 'xlink:href', png);		
			});		
	    }	    	    
	    mElement.setDataChanged(false);		 	    	    	
	}

	// get position of the mElement center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = Math.abs(size[0]/2);
    var my = Math.abs(size[1]/2);    

	mShape.setAttribute("x",x-mx);
	mShape.setAttribute("y",y-my);
	mShape.setAttribute("width",Math.abs(size[0]));
	mShape.setAttribute("height",Math.abs(size[1]));

	var style = mElement.getStyle();				// element style
    var attributes = style.getAttributes();
    for (var attr in attributes) {
      mShape.setAttribute(attr,attributes[attr]);
    }

	return mGroup;         
}

/*  // Alternative using SVG

    // create other SVG mElement group
    var mPixels = document.createElementNS("http://www.w3.org/2000/svg","g"); 
    mPixels.setAttribute("id", mElement.getName() + ".pixels");    	   

	var myBasePixel = document.createElementNS("http://www.w3.org/2000/svg","path");	
	myBasePixel.setAttribute("stroke-width",1.5);	    
     	
  	for(var i=0; i<xlen; i++) { 
  		for(var j=0; j<ylen; j++) {
  			// draw pixel
			color = colors[data[i][j]];
      		var myPixel = myBasePixel.cloneNode(true);      		
			myPixel.setAttribute('d', "M " + (i) + " " + (j) + " l 0 1");
			myPixel.setAttribute("stroke",color);
			mPixels.appendChild(myPixel);			
  		}  		
	}	

	// scale  	  	  	  		
  	var left = x-mx, top = y+my;
    var stepx = Math.abs(size[0]/xlen);
    var stepy = Math.abs(size[1]/ylen);
    mPixels.setAttribute("transform","scale(" + (stepx) + "," + (stepy) +") translate(" + left + "," + top + ")"); 
*/
/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG canvas
 */
EJSS_SVGGRAPHICS.canvas = function(mGraphics, mElement) {  
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());	
	var myCanvasSrc;	
	if (mShape === null) { 	// exits?
	    // create SVG element
	   	mShape = document.createElementNS("http://www.w3.org/2000/svg","foreignObject");
	    mShape.setAttribute("id",mElement.getName());	   
	    group.appendChild(mShape);
	    // create canvas src 
		myCanvasSrc = document.createElement('canvas');
		myCanvasSrc.setAttribute("id", mElement.getName() + ".canvas");

		mShape.appendChild(myCanvasSrc);	    
	}
	else {
		myCanvasSrc = document.getElementById(mElement.getName() + ".canvas");
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];

	myCanvasSrc.width = Math.abs(size[0]);
	myCanvasSrc.height = Math.abs(size[1]);
	
	mShape.width = Math.abs(size[0]);
	mShape.height = Math.abs(size[1]);
		
	// get half sizes 		
    var mx = Math.abs(size[0]/2);
    var my = Math.abs(size[1]/2);
    
 	// set SVG element	
	mShape.setAttribute("x",(x-mx));
	mShape.setAttribute("y",(y-my));    	    					  	
	
	var dim = mElement.getDimensions();
  	var deltaX = dim[1] - dim[0];
  	var deltaY = dim[3] - dim[2];
    
	var pixWth = Math.abs(size[0]);
	var pixHgt = Math.abs(size[1]);
	var pxPerUnitX = pixWth / deltaX;
	var pxPerUnitY = pixHgt / deltaY;
	var pxTopLeftX = x - pixWth / 2;
	var pxTopLeftY = y + pixHgt / 2;
   
    //draws all objects within this.drawables
    var drawables = mElement.getDrawables();
    var context = myCanvasSrc.getContext('2d');
    context.fillRect(0,0,myCanvasSrc.width-0.5,myCanvasSrc.height-0.5);
    for (var i = 0; i < drawables.length; i++) {
      if(typeof drawables[i].imageField != "undefined")
        EJSS_GRAPHICS.GraphicsUtils.drawImageField(context, 
        	pxTopLeftX, pxTopLeftY, xMin, yMin, pixWth, pixHgt, pxPerUnitX, pxPerUnitY, drawables[i]);
      else if(typeof drawables[i].run != "undefined")
      	drawables[i].draw(context,
      		pxTopLeftX, pxTopLeftY, xMin, yMin, pixWth, pixHgt, pxPerUnitX, pxPerUnitY);
    }

	var attributes = mElement.getStyle().getAttributes();
	for (var attr in attributes) {
	  	mShape.setAttribute(attr,attributes[attr]);
	}    	    	    					  	
	
	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG cellLattice
 */
EJSS_SVGGRAPHICS.cellLattice = function(mGraphics, mElement) {	
	var UTILS = EJSS_SVGGRAPHICS.Utils;   

	function pathForGrid_crispEdges(left, top, right, bottom, stepx, stepy) {
		var path = "";   
		
		var cleft = UTILS.crispValue(left), cright = UTILS.crispValue(right);
		var ctop = UTILS.crispValue(top), cbottom = UTILS.crispValue(bottom);
		// vertical lines
		if (stepx == 0) stepx = Math.abs(right-left);
	    for (var i = left; i <= Math.max(right,cright)+0.5; i = i+stepx) {
	  	  path += " M " + UTILS.crispValue(i) + " " + ctop + " L " + UTILS.crispValue(i) + " " + cbottom; 
	    }
	    // horizontal lines
	    if (stepy == 0) stepy = Math.abs(top-bottom);
	    for (var i = bottom; i >= Math.min(top,ctop)-0.5; i = i-stepy) {
	  	  path += " M " + cleft + " " + UTILS.crispValue(i) + " L " + cright + " " + UTILS.crispValue(i); 
	    }
	    return path;
	}  
	
	function pathForGrid(left, top, right, bottom, stepx, stepy) {	 	
		var path = "";   
		// vertical lines
		if (stepx == 0) stepx = Math.abs(right-left);
	    for (var i = left; i <= right; i = i+stepx) {
	  	  path += " M " + i + " " + top + " L " + i + " " + bottom; 
	    }
	    // horizontal lines
	    if (stepy == 0) stepy = Math.abs(top-bottom);
	    for (var i = bottom; i >= top; i = i-stepy) {
	  	  path += " M " + left + " " + i + " L " + right + " " + i; 
	    }
	    return path;
	}  
	
	function rectCell(group, x, y, sx, sy, rendering, fill) {
		// create rectangle
	    var mShape = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    group.appendChild(mShape);	   
	    		
		// create path	
		var mx = sx/2, my = sy/2;    	   	    
		if(rendering == "crispEdges") {
		    mShape.setAttribute('d', 
		    	 "M " + UTILS.crispValue(x-mx) + " " + UTILS.crispValue(y+my) + 
		    	" L " + UTILS.crispValue(x+mx) + " " + UTILS.crispValue(y+my) + 
		    	" L " + UTILS.crispValue(x+mx) + " " + UTILS.crispValue(y-my) + 
		    	" L " + UTILS.crispValue(x-mx) + " " + UTILS.crispValue(y-my) + " z");
		} else {
		    mShape.setAttribute('d', 
		    	 "M " + (x-mx) + " " + (y+my) + 
		    	" L " + (x+mx) + " " + (y+my) + 
		    	" L " + (x+mx) + " " + (y-my) + 
		    	" L " + (x-mx) + " " + (y-my) + " z");
		}  	 		
		mShape.setAttribute("fill",fill);
		mShape.setAttribute("stroke",fill);
	}

	function gridCell(group, x, y, sx, sy, stepx, stepy, style) {
		// create rectangle
	    var mShape = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    group.appendChild(mShape);	   

		// set attributes
		var mx = sx/2, my = sy/2;    	   	    
		var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
	    if(style.getShapeRendering() == "crispEdges") 
	    	mShape.setAttribute('d', pathForGrid_crispEdges(left, top, right, bottom, stepx, stepy));
	    else {
	    	mShape.setAttribute('d', pathForGrid(left, top, right, bottom, stepx, stepy));
	    }
						
	    if(style.getDrawLines()) {
	    	mShape.setAttribute("stroke",style.getLineColor());
	    	mShape.setAttribute("stroke-width",style.getLineWidth());
	    } else {
	    	mShape.setAttribute("stroke","none");
	    	mShape.setAttribute("stroke-width",0);    	
	    }        
	    
		var attributes = style.getAttributes();
		for (var attr in attributes) {
		    mShape.setAttribute(attr,attributes[attr]);
		}	    
	}
	
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mGroup = mGraphics.getElementById(mElement.getName());			
	if (mGroup !== null) { 	// exits?
		group.removeChild(mGroup);
	}
    // create SVG mElement group
    mGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
    mGroup.setAttribute("id", mElement.getName());
    group.appendChild(mGroup);	    

	// get position of the mElement center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;

	// draw cells
	var style = mElement.getStyle();     
	var data = mElement.getData();
	var colors = mElement.getColorMapper().getColors();
	var xlen = data.length;
	var ylen = data[0].length;

    var stepx = Math.abs(size[0]/xlen);
    var stepy = Math.abs(size[1]/ylen);
   	
  	var left = x-mx+stepx/2, bottom = y-my-stepy/2;		
  	for(var i=0; i<ylen; i++) { 
  		for(var j=0; j<xlen; j++) {
  			// draw rectangle
  			rectCell(mGroup, left+stepx*j, bottom-stepy*i, stepx, stepy, style.getShapeRendering(), colors[data[j][i]]);
  		}  		
  	}
  	 	
  	// draw grid
  	var showGrid = mElement.getShowGrid();
	if (showGrid) gridCell(mGroup, x, y, size[0], size[1], stepx, stepy, style);
	  	
	return mGroup;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG cursor
 */
EJSS_SVGGRAPHICS.cursor = function(mGraphics, mElement) {  
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape	
	var mGroup = mGraphics.getElementById(mElement.getName());
	var mLine1, mLine2;
	if (mGroup === null) { 	// not exists
	    // create SVG element group
	    mGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    mGroup.setAttribute("id", mElement.getName());
	    group.appendChild(mGroup);
	    // create line1 element
	    mLine1 = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    mLine1.setAttribute("id", mElement.getName() + ".line1");
	    mGroup.appendChild(mLine1);	    
	    // create line2 element
	    mLine2 = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    mLine2.setAttribute("id", mElement.getName() + ".line2");
	    mGroup.appendChild(mLine2);	    	    	    
	} else {
		mLine1 = mGraphics.getElementById(mElement.getName() + ".line1");
		mLine2 = mGraphics.getElementById(mElement.getName() + ".line2");
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    
  	// type of cursor
	var cursorType = mElement.getCursorType();

    // cursor size
    var panel = mElement.getGroupPanel();
    var bounds = panel.getRealWorldCoordinates(); // xmin,xmax,ymin,ymax
	var bottomLeft = panel.toPixelPosition([bounds[0],bounds[2]]);    
	var topRight   = panel.toPixelPosition([bounds[1],bounds[3]]);    
	    
	var style = mElement.getStyle();
	mLine1.setAttribute("stroke-width",0);   	 	
	if (cursorType!=EJSS_DRAWING2D.Cursor.VERTICAL) {
	  var x1 = bottomLeft[0], x2 = topRight[0];
	  var y = pos[1];
	  // keep the cursor inside the panel
	  if (y>bottomLeft[1]) y = bottomLeft[1];
	  else if (y<topRight[1]) y = topRight[1];
	  if (style.getShapeRendering() == "crispEdges") {
	    mLine1.setAttribute('d', "M " + UTILS.crispValue(x1) + " " + UTILS.crispValue(y) 
	    		+ " L " + UTILS.crispValue(x2) + " " + UTILS.crispValue(y));
	  } 
	  else {
	    mLine1.setAttribute('d', "M " + x1 + " " + y + " L " + x2 + " " + y);
	  }  	  
	  mLine1.setAttribute("stroke-width",style.getLineWidth());   	 	
	}
	mLine2.setAttribute("stroke-width",0);   	 	
	if (cursorType!=EJSS_DRAWING2D.Cursor.HORIZONTAL) {
	  var x = pos[0];
	  var y1 = bottomLeft[1], y2 = topRight[1];
	  // keep the cursor inside the panel
	  if (x<bottomLeft[0]) x = bottomLeft[0];
	  else if (x>topRight[0]) x = topRight[0];
	  if (style.getShapeRendering() == "crispEdges") {
	    mLine2.setAttribute('d', "M " + UTILS.crispValue(x) + " " + UTILS.crispValue(y1) 
	    		+ " L " + UTILS.crispValue(x) + " " + UTILS.crispValue(y2));
	  } 
	  else {
	    mLine2.setAttribute('d', "M " + x + " " + y1 + " L " + x + " " + y2);
	  }  	  
	  mLine2.setAttribute("stroke-width",style.getLineWidth());   	 	
	}
	if (style.getDrawFill()) mGroup.setAttribute("fill",style.getFillColor());
    else mGroup.setAttribute("fill","none");    
	mGroup.setAttribute("stroke",style.getLineColor());
	//mGroup.setAttribute("stroke-width",style.getLineWidth());
	mGroup.setAttribute("shapeRendering",style.getShapeRendering());  
    
    var attributes = style.getAttributes();
	for (var attr in attributes) {
      mGroup.setAttribute(attr,attributes[attr]);
    }			 	  	
	
	return mGroup;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG ellipse
 */
EJSS_SVGGRAPHICS.ellipse = function(mGraphics, mElement) {  
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());	
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","ellipse"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);	    
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = Math.abs(size[0]/2);
    var my = Math.abs(size[1]/2);
    
 	// set attributes	    	    
    mShape.setAttribute("cx",x);
    mShape.setAttribute("cy",y);
    mShape.setAttribute("rx",mx);
    mShape.setAttribute("ry",my);
	
	// set style
    var style = mElement.getStyle(); 
    if(style.getDrawFill())	
    	mShape.setAttribute("fill",style.getFillColor());
    else 
    	mShape.setAttribute("fill","none");    
    if(style.getDrawLines()) {
    	mShape.setAttribute("stroke",style.getLineColor());
    	mShape.setAttribute("stroke-width",style.getLineWidth());
    } else {
    	mShape.setAttribute("stroke","none");
    	mShape.setAttribute("stroke-width",0);    	
    }        
	mShape.setAttribute("shapeRendering",style.getShapeRendering());  	

	var attributes = style.getAttributes();
	for (var attr in attributes) {
	  	mShape.setAttribute(attr,attributes[attr]);
	}

	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG grid
 */
EJSS_SVGGRAPHICS.grid = function(mGraphics, mElement) {
	
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // remove SVG element (not reusing element)
	var mGroup = mGraphics.getElementById(mElement.getName());	
	if (mGroup !== null) group.removeChild(mGroup);
	
    // create SVG element
    mGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
    mGroup.setAttribute("id", mElement.getName());
    group.appendChild(mGroup);	    
    var mShapeX = document.createElementNS("http://www.w3.org/2000/svg","path"); 
    mShapeX.setAttribute("id", mElement.getName() + "X");
    mGroup.appendChild(mShapeX);	    
    var mShapeY = document.createElementNS("http://www.w3.org/2000/svg","path"); 
    mShapeY.setAttribute("id", mElement.getName() + "Y");
    mGroup.appendChild(mShapeY);

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;

    var showx = mElement.getShowX();
    var showy = mElement.getShowY(); 
    
    var pathX = "";
    var pathY = "";
    
	// draw axis X (based on ticks mode)
	if(showx) {
		var ticksXmode = mElement.getTicksXMode();			// decimal or logarithmic
		if (ticksXmode == EJSS_DRAWING2D.Grid.SCALE_LOG) { // logarithmic
				
		    // scale
		    var scaleX = mElement.getScaleX();		// X axis scale 	
			
			if(scaleX[0] > 0 && scaleX[1] > 0) {  // valid scale
				// get number of axis ticks
			    var ticksX = mElement.getTicksX();
			    
				// set attributes
			    var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
		    	EJSS_GRAPHICS.GraphicsUtils.drawLogGrid(left, top, right, bottom, ticksX, scaleX, 1,
					function(xx,yy) { pathX += " M " + xx + " " + yy; }, 
					function(xx,yy) { pathX += " L " + xx + " " + yy; }); 
			} 
					    		
		} else if (ticksXmode == EJSS_DRAWING2D.Grid.SCALE_NUM) { // decimal
	    	    
		 	// create path	    	   	    
		   	var stepx = 0;	// step for ticks
		   	var tickstepx = 0;
			if (!mElement.getAutoTicksX()) {	  // no auto-ticks    
			    var ticksX = mElement.getTicksX();
			    // whether the number of sticks exits, changes step for ticks
			    if (ticksX != 0) { stepx = Math.abs(size[0]/ticksX); } else {
			    	stepx = mElement.getStepX();
			    	tickstepx = mElement.getTickStepX(); 
			    } 	    	

			} else { // auto-ticks
				var stepxmin = mElement.getAutoStepXMin();		// step y min
				var ticksxrange = mElement.getAutoTicksRangeX();		// ticks x range
		
				// find stepx
				for(var i=ticksxrange.length-1; i>=0; i--)	{	
					stepx = Math.abs(size[0]/ticksxrange[i]);
					if (stepx >= stepxmin) break;
				}
			}	
			
		    // step for scale
			var scalePrecisionX = mElement.getScalePrecisionX();	// number of decimals for scale
		    var scaleX = mElement.getScaleX();		// X axis scale 	
		    if(tickstepx == 0) {
				var scalestepX = Math.abs((scaleX[1] - scaleX[0]) * stepx / size[0]);  // step in X axis scale
			} else {
				var scalestepX = tickstepx;
				stepx = Math.abs((scalestepX * size[0]) / (scaleX[1] - scaleX[0])); 
			}			
						
			// adjust step to decimals of precision	
			var decimalsX = Math.pow(10,scalePrecisionX);
			var scalestepXTmp = Math.round(scalestepX * decimalsX) / decimalsX;
			if(scalestepXTmp > 0) {
				scalestepX = scalestepXTmp;
				stepx = Math.abs(scalestepXTmp * size[0] / (scaleX[1] - scaleX[0]));
			}

			// check fixed tick
			var fixedTicks = mElement.getFixedTickX();	  
			var tickfixed = scaleX[1];
			if (!isNaN(fixedTicks)) {
			  if (fixedTicks < scaleX[0]) tickfixed = fixedTicks + (Math.floor((scaleX[0]-fixedTicks)/scalestepX)+1)*scalestepX;
			  else if (fixedTicks > scaleX[1])  tickfixed = fixedTicks - (Math.floor((fixedTicks-scaleX[1])/scalestepX)+1)*scalestepX;
		      else tickfixed = fixedTicks; 
			}		
				
			// tick fixed in axis scale
			var scaleshift = Math.abs((scaleX[0] - tickfixed) % scalestepX);						
			var dist = Math.abs(scaleshift-scalestepX);	// fitting shift
			if(scaleshift < 0.001 || dist < 0.001) scaleshift = 0;						
			var shiftx = Math.abs(size[0] * scaleshift / (scaleX[1] - scaleX[0]));	// shift in pixels				

			// set attributes
		    var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
	    	EJSS_GRAPHICS.GraphicsUtils.drawDecGrid(left, top, right, bottom, shiftx, stepx, 1,
					function(xx,yy) { pathX += " M " + xx + " " + yy; }, 
					function(xx,yy) { pathX += " L " + xx + " " + yy; }); 
		}		
	}

	// draw axis Y (based on ticks mode)
	if(showy) {
		var ticksYmode = mElement.getTicksYMode();			// decimal or logarithmic
		if (ticksYmode == EJSS_DRAWING2D.Grid.SCALE_LOG) { // logarithmic
						
		    // scale
		    var scaleY = mElement.getScaleY();		// Y axis scale 	

			if(scaleY[0] > 0 && scaleY[1] > 0) {  // valid scale
				// get number of axis ticks
			    var ticksY = mElement.getTicksY();
			
				// set attributes
			    var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
		    	EJSS_GRAPHICS.GraphicsUtils.drawLogGrid(left, top, right, bottom, ticksY, scaleY, -1,
					function(xx,yy) { pathY += " M " + xx + " " + yy; }, 
					function(xx,yy) { pathY += " L " + xx + " " + yy; }); 	
			}
					    		
		} else if (ticksYmode == EJSS_DRAWING2D.Grid.SCALE_NUM) { // decimal
	    
		 	// create path	    	   	    
		   	var stepy = 0;	// step for ticks
		   	var tickstepy = 0;
			if (!mElement.getAutoTicksY()) {	  // no auto-ticks    
			    var ticksY = mElement.getTicksY();
			    // whether the number of sticks exits, changes step for ticks
			    if (ticksY != 0) { stepy = Math.abs(size[1]/ticksY); } else {
			    	stepy = mElement.getStepY();
			    	tickstepy = mElement.getTickStepY(); 
			    } 	    	
			} else { // auto-ticks
				var stepymin = mElement.getAutoStepYMin();		// step y min		
				var ticksyrange = mElement.getAutoTicksRangeY();		// ticks y range
		
				// find stepy
				for(var i=ticksyrange.length-1; i>=0; i--)	{	
					stepy = Math.abs(size[1]/ticksyrange[i]);
					if (stepy >= stepymin) break;
				}						
			}	
			
		    // step for scale
			var scalePrecisionY = mElement.getScalePrecisionY();	// number of decimals for scale
		    var scaleY = mElement.getScaleY();		// Y axis scale 	
		    if(tickstepy == 0) {
				var scalestepY = Math.abs((scaleY[1] - scaleY[0]) * stepy / size[1]);  // step in Y axis scale 
			} else {
				var scalestepY = tickstepy;
				stepy = Math.abs((scalestepY * size[1]) / (scaleY[1] - scaleY[0])); 
			}			
			
			// adjust step to decimals of precision	
			var decimalsY = Math.pow(10,scalePrecisionY);
			var scalestepYTmp = Math.round(scalestepY * decimalsY) / decimalsY;
			if(scalestepYTmp > 0) {
				scalestepY = scalestepYTmp;
				stepy = Math.abs(scalestepYTmp * size[1] / (scaleY[1] - scaleY[0]));
			}
			
			// check fixed tick
			var fixedTicks = mElement.getFixedTickY();	  
			var tickfixed = scaleY[1];
			if (!isNaN(fixedTicks)) {
			  if (fixedTicks < scaleY[0]) tickfixed = fixedTicks + (Math.floor((scaleY[0]-fixedTicks)/scalestepY)+1)*scalestepY;
			  else if (fixedTicks > scaleY[1])  tickfixed = fixedTicks - (Math.floor((fixedTicks-scaleY[1])/scalestepY)+1)*scalestepY;
		      else tickfixed = fixedTicks; 
			}		
	
			// tick fixed in axis scale
			var scaleshift = Math.abs((scaleY[0] - tickfixed) % scalestepY);						
			var dist = Math.abs(scaleshift-scalestepY);	// fitting shift
			if(scaleshift < 0.001 || dist < 0.001) scaleshift = 0;						
			var shifty = Math.abs(size[1] * scaleshift / (scaleY[1] - scaleY[0]));	// shift in pixels				
																								            				            
			// set attributes
		    var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
	    	EJSS_GRAPHICS.GraphicsUtils.drawDecGrid(left, top, right, bottom, shifty, stepy, -1,
					function(xx,yy) { pathY += " M " + xx + " " + yy; }, 
					function(xx,yy) { pathY += " L " + xx + " " + yy; }); 	    	
		}
	}
	
	// set path and style X
	if(pathX.length > 0) {
		mShapeX.setAttribute('d', pathX);
		mShapeX.setAttribute("shapeRendering",mElement.getShapeRenderingX());  	
    	mShapeX.setAttribute("stroke",mElement.getLineColorX());
    	mShapeX.setAttribute("stroke-width",mElement.getLineWidthX());    		
	}

	// set path and style Y
	if(pathY.length > 0) {
		mShapeY.setAttribute('d', pathY);
		mShapeY.setAttribute("shapeRendering",mElement.getShapeRenderingY());  	
    	mShapeY.setAttribute("stroke",mElement.getLineColorY());
    	mShapeY.setAttribute("stroke-width",mElement.getLineWidthY());    		
	}

	var attributes = mElement.getStyle().getAttributes();
	for (var attr in attributes) {
	  	mShapeX.setAttribute(attr,attributes[attr]);
	  	mShapeY.setAttribute(attr,attributes[attr]);
	}

	return mGroup;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG group
 */
EJSS_SVGGRAPHICS.group = function(mGraphics, mElement) {

	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());	
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);	    
	}

	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG histogram
 */
 EJSS_SVGGRAPHICS.histogram = function(mGraphics, mElement) {  
	
	// creates area bar	
	function areaBar(bar, barx1, barx2, bary, axis, barstyle) {		
	 	// set attributes	    	    
	    bar.setAttribute('d', 
	    	 "M " + barx1 + " " + axis + 
	    	" L " + barx1 + " " + bary + 
	    	" L " + barx2 + " " + bary + 
	    	" L " + barx2 + " " + axis + " z");
	
		// set style
	    if(barstyle.getDrawFill())	
	    	bar.setAttribute("fill",barstyle.getFillColor());
	    else 
	    	bar.setAttribute("fill","none");    
	    if(barstyle.getDrawLines()) {
	    	bar.setAttribute("stroke",barstyle.getLineColor());
	    	bar.setAttribute("stroke-width",barstyle.getLineWidth());
	    } else {
	    	bar.setAttribute("stroke","none");
	    	bar.setAttribute("stroke-width",0);    	
	    }        
		bar.setAttribute("shapeRendering",barstyle.getShapeRendering());  	
	}  	
		
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape		
	var mGroup = mGraphics.getElementById(mElement.getName());
	if (mGroup === null) { 	// exists?
	    // create SVG element group
	    mGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    mGroup.setAttribute("id", mElement.getName());
	    group.appendChild(mGroup);	    
	} else {
		// remove Bars
		var mBars = mGraphics.getElementById(mElement.getName() + ".bars");
		mGroup.removeChild(mBars);
	}
    // create Bars element
    var mBars = document.createElementNS("http://www.w3.org/2000/svg","g"); 
    mBars.setAttribute("id", mElement.getName() + ".bars");
    mGroup.appendChild(mBars);	    

	// get position of the element center 	
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
	
	// get style        
	var style = mElement.getStyle(); 
        
    
	// get histogram parameters
	var discrete =  mElement.getDiscrete();
 	var halfBin = mElement.getBinWidth()/2;
 	var bars = mElement.getBars();

	var axisx = mElement.getGroupPanel().getPixelPositionWorldOrigin()[1];
	for (var j=0; j < bars.length; j++) { // create bars
      var mBar = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	  mBar.setAttribute("id", mElement.getName() + ".bar" + j);
	  mBars.appendChild(mBar);	 
	  var barx1, barx2;
	  var cbin = bars[j][0];
	  if (discrete) {
        barx1 = barx2 = (x-mx) + cbin*size[0];
	  } 
	  else {
		barx1 = (x-mx) + (cbin-halfBin)*size[0];	// Bar x
		barx2 = (x-mx) + (cbin+halfBin)*size[0];	// Bar x
	  }
	  var bary = (y-my) + bars[j][1]*size[1];  // Bar y
	  areaBar(mBar, barx1, barx2, bary, axisx, style);
    }

	var attributes = style.getAttributes();
	for (var attr in attributes) {
	  	mGroup.setAttribute(attr,attributes[attr]);
	}    	    	    			
		
	return mGroup;         
}
/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG image
 */
EJSS_SVGGRAPHICS.image = function(mGraphics, mElement) {  
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());	
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","image"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);	    
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = Math.abs(size[0]/2);
    var my = Math.abs(size[1]/2);
    
 	// set attributes
 	var current = mShape.getAttributeNS('http://www.w3.org/1999/xlink', 'href');
 	if(!current || mElement.getChangedImage()) {
	 	var code = mElement.getEncode();
	 	if(code.length > 0) { // http://webcodertools.com/imagetobase64converter/Create
	 		mShape.setAttributeNS('http://www.w3.org/1999/xlink', 'xlink:href', 'data:png;base64,'+code);
	 	} else {
	    	mShape.setAttributeNS('http://www.w3.org/1999/xlink', 'xlink:href', mElement.getImageUrl());
	    	// mShape.onerror = function() {  };
	    }
	    // indica que la fuente de la imagen ha sido actualizada  
	    mElement.setChangedImage(false);
	}
	
	mShape.setAttribute('preserveAspectRatio', 'none')

	mShape.setAttribute("x",(x-mx));
	mShape.setAttribute("y",(y-my));
	mShape.setAttribute("width",Math.abs(size[0]));
	mShape.setAttribute("height",Math.abs(size[1]));
	
	// set style
    var style = mElement.getStyle(); 
    if(style.getDrawFill())	
    	mShape.setAttribute("fill",style.getFillColor());
    else 
    	mShape.setAttribute("fill","none");    
    if(style.getDrawLines()) {
    	mShape.setAttribute("stroke",style.getLineColor());
    	mShape.setAttribute("stroke-width",style.getLineWidth());
    } else {
    	mShape.setAttribute("stroke","none");
    	mShape.setAttribute("stroke-width",0);    	
    }        
	mShape.setAttribute("shapeRendering",style.getShapeRendering());  	

	var attributes = style.getAttributes();
	for (var attr in attributes) {
	  	mShape.setAttribute(attr,attributes[attr]);
	}

	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG mesh
 */
EJSS_SVGGRAPHICS.mesh = function(mGraphics, mElement) {
  var mA; 			// int[]
  var mB; 			// int[]
  var mCells;		// {cellA, cellB, cellZ, cellVector}
  var mBoundary;	// {boundaryA (int[][]), boundaryB (int[][])}
  
  // fill cell
  function fillCell(group, xs, ys, numpoints, color, levelIndex) {
  	var poly = document.createElementNS("http://www.w3.org/2000/svg","polygon"); 
  	group.appendChild(poly);
	
	var path = "";
	for(var i=0; i<numpoints; i++) {
   	  path += xs[i] + "," + ys[i] + " ";
	}
	path += xs[0] + "," + ys[0] + " ";  	  	
	
	poly.setAttribute('points', path);
  	var fill = color.indexToColor(levelIndex);
  	poly.setAttribute("fill",fill);	  
  	poly.setAttribute("stroke",fill);
  	poly.setAttribute("stroke-width",1.25);  
  }   
 
  // draw cell
  function drawCell(group, xs, ys, fill, style) {
    var poly = document.createElementNS("http://www.w3.org/2000/svg","path"); 
    group.appendChild(poly);
    
	var path = "";
	var numpoints = xs.length-1;
	for(var i=0; i<numpoints; i++) {
	  if (i==0)
	  	path += " M " + xs[i] + " " + ys[i];	 
      else       	
      	path += " L " + xs[i] + " " + ys[i];	
	}  	  	
   	path += " L " + xs[i] + " " + ys[i] + " Z";
   	
    poly.setAttribute('d', path);
  
    // set style
    if(fill && style.getDrawFill()) 	
	  poly.setAttribute("fill",style.getFillColor());
    else 
	  poly.setAttribute("fill","none");    
    if(style.getDrawLines()) {
	  poly.setAttribute("stroke",style.getLineColor());
	  poly.setAttribute("stroke-width",style.getLineWidth());
    } else {
	  poly.setAttribute("stroke","none");
	  poly.setAttribute("stroke-width",0);    	
    }        
    poly.setAttribute("shapeRendering",style.getShapeRendering());      
  }  

  function drawSegment(group, xs, ys, color, lineWidth) {
    var mPoly = document.createElementNS("http://www.w3.org/2000/svg","path"); 
    group.appendChild(mPoly);
    
	var path = "";
	var numpoints = xs.length-1;
	for(var i = 0; i<numpoints; i++) {
	  if (i==0) // 0 is NOT CONNECTION
	  	path += " M " + xs[i] + " " + ys[i];	 
      else       	
      	path += " L " + xs[i] + " " + ys[i];	
	}  	  	
   	path += " L " + xs[i] + " " + ys[i] + " Z";
	
    mPoly.setAttribute('d', path);
	
    // set style
    mPoly.setAttribute("fill","none");    
    mPoly.setAttribute("stroke", color);
    mPoly.setAttribute("stroke-width", lineWidth + 0.5);
  }  
  
  function drawVectors(group,cellA,cellB,field,lenVector) { 
    var map = document.createElementNS("http://www.w3.org/2000/svg","g"); 
    group.appendChild(map);	    
  	for(var i=0; i<cellA.length; i++) {
	    // create SVG element group
	    var vector = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    map.appendChild(vector);	    
	    // create path element
	    var line = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    vector.appendChild(line);	    
	    // create cross element
	    var tip = document.createElementNS("http://www.w3.org/2000/svg","polygon"); 
	    vector.appendChild(tip);	    
	
		// draw line
		var a = cellA[i];
		var b = cellB[i];
		var a1 = field[i][0];
		var b1 = field[i][1];
		var len = EJSS_TOOLS.Mathematics.norm([a1,b1]);
		var sa = a1*lenVector/len;
		var sb = b1*lenVector/len;
		
		var str = "M " + a + " " + b + " L " + (a+sa) + " " + (b+sb);
	   	line.setAttribute('d', str);
	
		// draw tip
		if((sa != 0) || (sb != 0)) {
			var endsize;
		    var maxsize = 12; // tip max size
		    var proportion = 0.3; // 30% length
	
			// calculate tip size	    
			var len = EJSS_TOOLS.Mathematics.norm([sa,sb]);
	    	endsize = len * proportion;
	    	if (endsize > maxsize) endsize = maxsize;
		
		    // position of arrow end
		    var tipx = a + sa;
		    var tipy = b + sb;
		
		  	// draw arrow end (only triangle)
		  	var points = tipx +","+ tipy + " " +
		  				 (tipx-endsize) +","+ (tipy+endsize/3) + " " + 
		  				 (tipx-endsize) +","+ (tipy-endsize/3);	        
		    tip.setAttribute('points', points);	
			
			// angle of arrow end  	  	  	  	
		    var anglearrow = Math.atan2(sa,-sb);
		    anglearrow = anglearrow * 180 / Math.PI;
		    tip.setAttribute("transform","rotate(" + (anglearrow-90) + " " + tipx + " " + tipy +")"); // minus 90!
		}
		
	   	vector.setAttribute("stroke","black");
	   	vector.setAttribute("stroke-width",0.5);
	}  	
  }
	
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;

    // create SVG element
    var mGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
    mGroup.setAttribute("id", mElement.getName());
    	
    // remove SVG element (not reusing element)
	var mOldGroup = mGraphics.getElementById(mElement.getName());	
	if (mOldGroup !== null) mOldGroup.parentNode.replaceChild(mGroup, mOldGroup);		
	else group.appendChild(mGroup);	

	// get position of the mElement center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
    
	var mPoints = mElement.getPoints();	
    if (mPoints == null) return;    
    
  	// project the points
    var nPoints = mPoints.length;
    mA = new Array(nPoints);
    mB = new Array(nPoints);
    for (var i=0; i<nPoints; i++) {
      mA[i] = x-mx + mPoints[i][0]*size[0];
      mB[i] = y-my + mPoints[i][1]*size[1];
    }

	// build cells
	var cells = mElement.getCells();
	var field = mElement.getFieldAtPoints();
	if (field) {
		mCells = EJSS_GRAPHICS.GraphicsUtils.buildCells(mA,mB,cells,field,false);
	} else {
	  field = mElement.getFieldAtCells();
	  mCells = EJSS_GRAPHICS.GraphicsUtils.buildCells(mA,mB,cells,field,true);
	}

    // first draw the cells
    if (cells != null) {    
	  var style = mElement.getStyle();
      if (field != null) {
	      // draw polygons    	  
		  var color = mElement.getColorCoded();
	      EJSS_GRAPHICS.GraphicsUtils.drawPolygons(mCells, color, 
	      	function(xs, ys) { drawCell(mGroup, xs, ys, false, style) },
	      	function(xs, ys, numpoints, color, levelIndex) { fillCell(mGroup, xs, ys, numpoints, color, levelIndex) });
      }
      else {
	      EJSS_GRAPHICS.GraphicsUtils.drawPolygons(mCells, color, 
	      	function(xs, ys) { drawCell(mGroup, xs, ys, true, style) },
	      	function(xs, ys, numpoints, color, levelIndex) { });
      }
    }

	// build boundary
	var boundary = mElement.getBoundary();
	if(mElement.getDrawBoundary() && boundary != null) {	 
	  var labels = mElement.getBoundaryLabels();
	  var colors = mElement.getBoundaryColors();
	  var lineWidth = mElement.getBoundaryLineWidth();	
	  mBoundary = EJSS_GRAPHICS.GraphicsUtils.buildBoundary(mA, mB, boundary);
 
      // draw boundary
      EJSS_GRAPHICS.GraphicsUtils.drawBoundary(mBoundary, labels, colors, 
      	function(xs, ys, color) { drawSegment(mGroup,xs,ys,color,lineWidth) }); 
	}

	// draw vectors
	if(mElement.getDataType() >= 4 && field != null) {
		var lenVector = mElement.getVectorLength() * size[0]; // scale for size[0]		
        for (var i=0, n=cells.length; i<n; i++) {
          // draw vectors
          drawVectors(mGroup,mCellA[i],mCellB[i],field[i],lenVector);
        }
    }	
	
	var attributes = mElement.getStyle().getAttributes();
	for (var attr in attributes) {
	  	mGroup.setAttribute(attr,attributes[attr]);
	}    	    	    	
	
	return mGroup;         
}

 /**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG pipe
 */
EJSS_SVGGRAPHICS.pipe = function(mGraphics, mElement) {
	var UTILS = EJSS_SVGGRAPHICS.Utils;   
	
	// get element group
    var group, elementGroup = mElement.getGroup();
    if (elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);
	}
	var container = mGraphics.getElementById(mElement.getName()+":container");
	if (container === null) {	    
	    container = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    container.setAttribute("id", mElement.getName()+":container");
	    mShape.appendChild(container);
	 }	
	var liquid = mGraphics.getElementById(mElement.getName()+":liquid");
	if (liquid === null) {    
	    liquid = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    liquid.setAttribute("id", mElement.getName()+":liquid");
	    mShape.appendChild(liquid);	    
	}

    var points = mElement.getPoints();
    if (!points || points.length<=0) return;  

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];

	
	// get half sizes 		
    var sx = Math.abs(size[0]);
    var sy = Math.abs(size[1]);
        
 	// create path	    	   	    
 	var xmin = x-sx/2;
 	var ymin = y+sy/2;
    var style = mElement.getStyle(); 
    var crispy = (style.getShapeRendering() == "crispEdges"); 
    
    var containerPath = "";
    var liquidPath = "";
    var piece;
    if (crispy) piece = "M " + UTILS.crispValue(xmin) + " " + UTILS.crispValue(ymin);
    else        piece = "M " + xmin + " " + ymin; 
    containerPath = piece;
    liquidPath    = piece;
    for (var i=0, n=points.length; i<n; i++) {
      var point = points[i];
      var x = Math.round(point[0]*sx);
      var y = Math.round(-point[1]*sy);
      if (crispy) piece =  " l " + UTILS.crispValue(x) + " " + UTILS.crispValue(y);
      else        piece =  " l " + x + " " + y; 
      containerPath += piece;
      liquidPath += piece;
    }

	container.setAttribute('d',containerPath);
	liquid.setAttribute('d',liquidPath);
				
	// set style
	if (style.getDrawFill())  {
	  liquid.setAttribute("stroke",style.getFillColor());
      liquid.setAttribute("stroke-width",mElement.getPipeWidth());
    }
    else {
	  liquid.setAttribute("stroke","none");
      liquid.setAttribute("stroke-width",0);
    }  
    liquid.setAttribute("fill","none");    
	liquid.setAttribute("shapeRendering",style.getShapeRendering());  	

    if (style.getDrawLines()) {
      container.setAttribute("stroke",style.getLineColor());
      container.setAttribute("stroke-width",2*style.getLineWidth()+mElement.getPipeWidth());
    }
    else {
      container.setAttribute("stroke","none");
      container.setAttribute("stroke-width",0);
    }
    container.setAttribute("fill","none");    
	container.setAttribute("shapeRendering",style.getShapeRendering());  	

	var attributes = style.getAttributes();
	for (var attr in attributes) {
	  	liquid.setAttribute(attr,attributes[attr]);
	  	container.setAttribute(attr,attributes[attr]);
	}

	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG point
 */
EJSS_SVGGRAPHICS.point = function(mGraphics, mElement) {  
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());	
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","ellipse"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);	    
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	    
 	// set attributes	    	    
    mShape.setAttribute("cx",x);
    mShape.setAttribute("cy",y);
    mShape.setAttribute("rx",2);
    mShape.setAttribute("ry",2);
	
	// set style
    var style = mElement.getStyle(); 
   	mShape.setAttribute("fill",style.getFillColor());
   	
	var attributes = style.getAttributes();
	for (var attr in attributes) {
	  	mShape.setAttribute(attr,attributes[attr]);
	}
   	
	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG polygon
 */
EJSS_SVGGRAPHICS.polygon = function(mGraphics, mElement) {  
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());	
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);	    
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
    
	// set attributes
	var points = mElement.getPoints();	    	    
    var polpath = "";
	for(var i=0; i<points.length; i++) {
	  var point = points[i];	  
      var xx = (x-mx) + point[0]*size[0];
      var yy = (y-my) + point[1]*size[1];
      var type = point[2];
	  if ((i==0) || (type == 0)) // 0 is NOT CONNECTION
	  	polpath += " M " + xx + " " + yy;		// move 
      else       	
     	polpath += " L " + xx + " " + yy;		// line
	}  	  	

    if(polpath !== "") {
    	mShape.setAttribute('d', polpath);	

		// set style
	    var style = mElement.getStyle(); 
	    if(style.getDrawFill())	
	    	mShape.setAttribute("fill",style.getFillColor());
	    else 
	    	mShape.setAttribute("fill","none");    
	    if(style.getDrawLines()) {
	    	mShape.setAttribute("stroke",style.getLineColor());
	    	mShape.setAttribute("stroke-width",style.getLineWidth());
	    } else {
	    	mShape.setAttribute("stroke","none");
	    	mShape.setAttribute("stroke-width",0);    	
	    }        
		mShape.setAttribute("shapeRendering",style.getShapeRendering());

		var attributes = style.getAttributes();
		for (var attr in attributes) {
		  	mShape.setAttribute(attr,attributes[attr]);
		}		  	
	}
	
	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG rectangle
 */
EJSS_SVGGRAPHICS.rectangle = function(mGraphics, mElement) {
	var UTILS = EJSS_SVGGRAPHICS.Utils;   
	
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());		
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);	    
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
    
 	// create path	    	   	    
    var style = mElement.getStyle(); 
    if(style.getShapeRendering() == "crispEdges") {
	    mShape.setAttribute('d', 
	    	 "M " + UTILS.crispValue(x-mx) + " " + UTILS.crispValue(y+my) + 
	    	" L " + UTILS.crispValue(x+mx) + " " + UTILS.crispValue(y+my) + 
	    	" L " + UTILS.crispValue(x+mx) + " " + UTILS.crispValue(y-my) + 
	    	" L " + UTILS.crispValue(x-mx) + " " + UTILS.crispValue(y-my) + " z");
    } else {
	    mShape.setAttribute('d', 
	    	 "M " + (x-mx) + " " + (y+my) + 
	    	" L " + (x+mx) + " " + (y+my) + 
	    	" L " + (x+mx) + " " + (y-my) + 
	    	" L " + (x-mx) + " " + (y-my) + " z");
	}  	    
		
	// set style
    if(style.getDrawFill())	
    	mShape.setAttribute("fill",style.getFillColor());
    else 
    	mShape.setAttribute("fill","none");    
    if(style.getDrawLines()) {
    	mShape.setAttribute("stroke",style.getLineColor());
    	mShape.setAttribute("stroke-width",style.getLineWidth());
    } else {
    	mShape.setAttribute("stroke","none");
    	mShape.setAttribute("stroke-width",0);    	
    }        
	mShape.setAttribute("shapeRendering",style.getShapeRendering());  	

	var attributes = style.getAttributes();
	for (var attr in attributes) {
	  	mShape.setAttribute(attr,attributes[attr]);
	}

	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG round rectangle
 */
EJSS_SVGGRAPHICS.roundRectangle = function(mGraphics, mElement) {  
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());	
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","rect"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);	    
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = Math.abs(size[0]/2);
    var my = Math.abs(size[1]/2);
    
    // get element radius
    var radius = Math.abs(mElement.getCornerRadius());
    
 	// set attributes	   
	mShape.setAttribute("x", x-mx);
	mShape.setAttribute("y", y-my);
	mShape.setAttribute("width", Math.abs(size[0]));
	mShape.setAttribute("height", Math.abs(size[1]));	    	
	mShape.setAttribute("rx", radius);
	mShape.setAttribute("ry", radius);
	
	// set style
    var style = mElement.getStyle(); 
    if(style.getDrawFill())	
    	mShape.setAttribute("fill",style.getFillColor());
    else 
    	mShape.setAttribute("fill","none");    
    if(style.getDrawLines()) {
    	mShape.setAttribute("stroke",style.getLineColor());
    	mShape.setAttribute("stroke-width",style.getLineWidth());
    } else {
    	mShape.setAttribute("stroke","none");
    	mShape.setAttribute("stroke-width",0);    	
    }        
	mShape.setAttribute("shapeRendering",style.getShapeRendering());  	

	var attributes = style.getAttributes();
	for (var attr in attributes) {
	  	mShape.setAttribute(attr,attributes[attr]);
	}

	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG scalarField
 */
EJSS_SVGGRAPHICS.scalarField = function(mGraphics, mElement) {	
	var UTILS = EJSS_SVGGRAPHICS.Utils;   

	function pathForGrid_crispEdges(left, top, right, bottom, stepx, stepy) {
		var path = "";   
		
		var cleft = UTILS.crispValue(left), cright = UTILS.crispValue(right);
		var ctop = UTILS.crispValue(top), cbottom = UTILS.crispValue(bottom);
		// vertical lines
		if (stepx == 0) stepx = Math.abs(right-left);
	    for (var i = left; i <= Math.max(right,cright)+0.5; i = i+stepx) {
	  	  path += " M " + UTILS.crispValue(i) + " " + ctop + " L " + UTILS.crispValue(i) + " " + cbottom; 
	    }
	    // horizontal lines
	    if (stepy == 0) stepy = Math.abs(top-bottom);
	    for (var i = bottom; i >= Math.min(top,ctop)-0.5; i = i-stepy) {
	  	  path += " M " + cleft + " " + UTILS.crispValue(i) + " L " + cright + " " + UTILS.crispValue(i); 
	    }
	    return path;
	}  
	
	function pathForGrid(left, top, right, bottom, stepx, stepy) {	 	
		var path = "";   
		// vertical lines
		if (stepx == 0) stepx = Math.abs(right-left);
	    for (var i = left; i <= right; i = i+stepx) {
	  	  path += " M " + i + " " + top + " L " + i + " " + bottom; 
	    }
	    // horizontal lines
	    if (stepy == 0) stepy = Math.abs(top-bottom);
	    for (var i = bottom; i >= top; i = i-stepy) {
	  	  path += " M " + left + " " + i + " L " + right + " " + i; 
	    }
	    return path;
	}  
	
	function rectCell(group, x, y, sx, sy, rendering, fill) {
		// create rectangle
	    var mShape = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    group.appendChild(mShape);	   
	    		
		// create path	
		var mx = sx/2, my = sy/2;    	   	    
		if(rendering == "crispEdges") {
		    mShape.setAttribute('d', 
		    	 "M " + UTILS.crispValue(x-mx) + " " + UTILS.crispValue(y+my) + 
		    	" L " + UTILS.crispValue(x+mx) + " " + UTILS.crispValue(y+my) + 
		    	" L " + UTILS.crispValue(x+mx) + " " + UTILS.crispValue(y-my) + 
		    	" L " + UTILS.crispValue(x-mx) + " " + UTILS.crispValue(y-my) + " z");
		} else {
		    mShape.setAttribute('d', 
		    	 "M " + (x-mx) + " " + (y+my) + 
		    	" L " + (x+mx) + " " + (y+my) + 
		    	" L " + (x+mx) + " " + (y-my) + 
		    	" L " + (x-mx) + " " + (y-my) + " z");
		}  	 		
		mShape.setAttribute("fill",fill);
		mShape.setAttribute("stroke",fill);
	}

	function gridCell(group, x, y, sx, sy, stepx, stepy, style) {
		// create rectangle
	    var mShape = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    group.appendChild(mShape);	   

		// set attributes
		var mx = sx/2, my = sy/2;    	   	    
		var left = x-mx, right = x+mx, top = y+my, bottom = y-my;
	    if(style.getShapeRendering() == "crispEdges") 
	    	mShape.setAttribute('d', pathForGrid_crispEdges(left, top, right, bottom, stepx, stepy));
	    else {
	    	mShape.setAttribute('d', pathForGrid(left, top, right, bottom, stepx, stepy));
	    }
						
	    if(style.getDrawLines()) {
	    	mShape.setAttribute("stroke",style.getLineColor());
	    	mShape.setAttribute("stroke-width",style.getLineWidth());
	    } else {
	    	mShape.setAttribute("stroke","none");
	    	mShape.setAttribute("stroke-width",0);    	
	    }        
	    
		var attributes = style.getAttributes();
		for (var attr in attributes) {
		    mShape.setAttribute(attr,attributes[attr]);
		}	    
	}
	
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mGroup = mGraphics.getElementById(mElement.getName());			
	if (mGroup !== null) { 	// exits?
		group.removeChild(mGroup);
	}
    // create SVG mElement group
    mGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
    mGroup.setAttribute("id", mElement.getName());
    group.appendChild(mGroup);	    

	var data = mElement.getData();
	var xlen = data.length;	
	if (xlen>0) {
		// get position of the mElement center 
	    var pos = mElement.getPixelPosition();
	    var size = mElement.getPixelSizes();     
	    var offset = mElement.getRelativePositionOffset(size);  
	    var x = pos[0]+offset[0];
	    var y = pos[1]+offset[1];
		
		// get half sizes 		
	    var mx = size[0]/2;
	    var my = size[1]/2;
	    
		// draw cells
		var style = mElement.getStyle();     
		var colors = mElement.getColorMapper().getColors();
		if(mElement.getAutoscaleZ()) mElement.getColorMapper().setAutoscaleArray2(data);
		
		var ylen = data[0].length;
	
	    var stepx = Math.abs(size[0]/xlen);
	    var stepy = Math.abs(size[1]/ylen);
	   	
	  	var left = x-mx+stepx/2, bottom = y-my-stepy/2;	
	  	for(var i=0; i<ylen; i++) { 
	  		for(var j=0; j<xlen; j++) {
	  			// get color based on max and min
	  			var value = data[j][i];
	  			var color = mElement.getColorMapper().doubleToColor(value);
	  			// draw rectangle
	  			rectCell(mGroup, left+stepx*j, bottom-stepy*i, stepx, stepy, style.getShapeRendering(), color);
	  		}  		
	  	}
	  	 	
	  	// draw grid
	  	var showGrid = mElement.getShowGrid();
		if (showGrid) gridCell(mGroup, x, y, size[0], size[1], stepx, stepy, style);	
	}
	  	
	return mGroup;         
}
/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG segment
 */
EJSS_SVGGRAPHICS.segment = function(mGraphics, mElement) {
	var UTILS = EJSS_SVGGRAPHICS.Utils;   

	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());	
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);	    
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
    
 	// set attributes	    	
    var style = mElement.getStyle();   	 	
    if(style.getShapeRendering() == "crispEdges") {
    	mShape.setAttribute('d', "M " + UTILS.crispValue(x-mx) + " " + UTILS.crispValue(y-my) 
    		+ " L " + UTILS.crispValue(x+mx) + " " + UTILS.crispValue(y+my));
    } else {
    	mShape.setAttribute('d', "M " + (x-mx) + " " + (y-my) + " L " + (x+mx) + " " + (y+my));
	}  	    
	
	// set style
    if(style.getDrawFill())	
    	mShape.setAttribute("fill",style.getFillColor());
    else 
    	mShape.setAttribute("fill","none");    
    if(style.getDrawLines()) {
    	mShape.setAttribute("stroke",style.getLineColor());
    	mShape.setAttribute("stroke-width",style.getLineWidth());
    } else {
    	mShape.setAttribute("stroke","none");
    	mShape.setAttribute("stroke-width",0);    	
    }        
	mShape.setAttribute("shapeRendering",style.getShapeRendering());  	

    var attributes = style.getAttributes();
    for (var attr in attributes) {
      mShape.setAttribute(attr,attributes[attr]);
    }
    
	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG spring
 */
EJSS_SVGGRAPHICS.spring = function(mGraphics, mElement) {

	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());	
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);	    
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
    
 	// get spring radius	    	
    var radius = mElement.getGroupPanel().toPixelMod([mElement.getRadius(),0])[0];

	// set attributes
	var springpath = "";
	EJSS_GRAPHICS.GraphicsUtils.drawSpring(mElement.getLoops(), mElement.getPointsPerLoop(), 
			radius, mElement.getSolenoid(), mElement.getThinExtremes(), x-mx, y-my, size[0], size[1],
			function(xx,yy) { springpath += " M " + xx + " " + yy; }, 
			function(xx,yy) { springpath += " L " + xx + " " + yy; });   
			
    if (springpath !== "") {
	    mShape.setAttribute('d', springpath);
	    		
		// set style
	    var style = mElement.getStyle(); 	
	    if(style.getDrawFill())	
	    	mShape.setAttribute("fill",style.getFillColor());
	    else 
	    	mShape.setAttribute("fill","none");    
	    if(style.getDrawLines()) {
	    	mShape.setAttribute("stroke",style.getLineColor());
	    	mShape.setAttribute("stroke-width",style.getLineWidth());
	    } else {
	    	mShape.setAttribute("stroke","none");
	    	mShape.setAttribute("stroke-width",0);    	
	    }        
		mShape.setAttribute("shapeRendering",style.getShapeRendering());

		var attributes = style.getAttributes();
		for (var attr in attributes) {
		  	mShape.setAttribute(attr,attributes[attr]);
		}		  	
	}
	
	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG rectangle
 */
EJSS_SVGGRAPHICS.tank = function(mGraphics, mElement) {
	var UTILS = EJSS_SVGGRAPHICS.Utils;   
	
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    mShape.setAttribute("id", mElement.getName());
	    group.appendChild(mShape);
	}
	var container = mGraphics.getElementById(mElement.getName()+":container");
	if (container === null) {	    
	    container = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    container.setAttribute("id", mElement.getName()+":container");
	    mShape.appendChild(container);
	 }	
	var liquid = mGraphics.getElementById(mElement.getName()+":liquid");
	if (liquid === null) {    
	    liquid = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    liquid.setAttribute("id", mElement.getName()+":liquid");
	    mShape.appendChild(liquid);	    
	}


	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];

	
	// get half sizes 		
    var mx = Math.abs(size[0]/2);
    var my = Math.abs(size[1]/2);
        
    var level = mElement.getPixelSizeOf(0,mElement.getLevel());  

 	// create path	    	   	    
 	var xmin = x-mx, xmax = x+mx;
 	var ymin = y+my, ymax = y-my;
 	var ylevel = ymin + level[1];
    var style = mElement.getStyle(); 
    if(style.getShapeRendering() == "crispEdges") {
	    liquid.setAttribute('d', 
	    	 "M " + UTILS.crispValue(xlevel) + " " + UTILS.crispValue(ylevel) + 
	    	" L " + UTILS.crispValue(xmin) + " " + UTILS.crispValue(ymin) + 
	    	" L " + UTILS.crispValue(xmax) + " " + UTILS.crispValue(ymin) + 
	    	" L " + UTILS.crispValue(xlevel) + " " + UTILS.crispValue(ylevel) +
	    	" Z");
	    if (style.getDrawFill()) container.setAttribute('d', 
	    	 "M " + UTILS.crispValue(xmin) + " " + UTILS.crispValue(ymax) + 
	    	" L " + UTILS.crispValue(xmin) + " " + UTILS.crispValue(ymin) + 
	    	" L " + UTILS.crispValue(xmax) + " " + UTILS.crispValue(ymin) + 
	    	" L " + UTILS.crispValue(xmax) + " " + UTILS.crispValue(ymax) +" Z");
	    else container.setAttribute('d', 
	    	 "M " + UTILS.crispValue(xmin) + " " + UTILS.crispValue(ymax) + 
	    	" L " + UTILS.crispValue(xmin) + " " + UTILS.crispValue(ymin) + 
	    	" L " + UTILS.crispValue(xmax) + " " + UTILS.crispValue(ymin) + 
	    	" L " + UTILS.crispValue(xmax) + " " + UTILS.crispValue(ymax));
    } else {
	    liquid.setAttribute('d', 
	    	 "M " + (xmin + level[0]) + " " + (ylevel) + 
	    	" L " + (xmin) + " " + (ymin) + 
	    	" L " + (xmax) + " " + (ymin) + 
	    	" L " + (xmax + level[0]) + " " + (ylevel) +
	    	" Z");
	    if (style.getDrawFill()) container.setAttribute('d', 
	    	 "M " + (xmin) + " " + (ymax) + 
	    	" L " + (xmin) + " " + (ymin) + 
	    	" L " + (xmax) + " " + (ymin) + 
	    	" L " + (xmax) + " " + (ymax) + " Z");
	     else container.setAttribute('d', 
	    	 "M " + (xmin) + " " + (ymax) + 
	    	" L " + (xmin) + " " + (ymin) + 
	    	" L " + (xmax) + " " + (ymin) + 
	    	" L " + (xmax) + " " + (ymax) );
	}  	    
		
	// set style
    liquid.setAttribute("stroke","none");
    liquid.setAttribute("stroke-width",0);    	
    liquid.setAttribute("fill",mElement.getLevelColor());
	liquid.setAttribute("shapeRendering",style.getShapeRendering());  	

    if (style.getDrawFill()) container.setAttribute("fill",style.getFillColor());
    else container.setAttribute("fill","none");    
    container.setAttribute("stroke",style.getLineColor());
    container.setAttribute("stroke-width",style.getLineWidth());
	container.setAttribute("shapeRendering",style.getShapeRendering());  	

	var attributes = style.getAttributes();
	for (var attr in attributes) {
	  	liquid.setAttribute(attr,attributes[attr]);
	  	container.setAttribute(attr,attributes[attr]);
	}

	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG text
 */
EJSS_SVGGRAPHICS.text = function(mGraphics, mElement) {  
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mGroup = mGraphics.getElementById(mElement.getName());
	var mInnerGroup, mText, mBox;
	if (mGroup === null) { 	// exits?
	    // create SVG element group
	    mGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    mGroup.setAttribute("id", mElement.getName());
	    group.appendChild(mGroup);	    
		// create SVG inner element group
	    mInnerGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    mInnerGroup.setAttribute("id", mElement.getName() + ".inner");
	    mGroup.appendChild(mInnerGroup);	    
	    // create box element
	    mBox = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    mBox.setAttribute("id", mElement.getName() + ".box");
	    mInnerGroup.appendChild(mBox);	    
	    // create text element
	    mText = document.createElementNS("http://www.w3.org/2000/svg","text"); 
	    mText.setAttribute("id", mElement.getName() + ".text");
	    mInnerGroup.appendChild(mText);	    
	} else {
		mInnerGroup = mGraphics.getElementById(mElement.getName() + ".inner");
		mText = mGraphics.getElementById(mElement.getName() + ".text");
		mBox = mGraphics.getElementById(mElement.getName() + ".box");
	}

	// remove previous text
	while (mText.firstChild) {
	    mText.removeChild(mText.firstChild);
	}	

	// draw text	    		
	var font = mElement.getFont();
	mText.setAttribute("font-style",font.getFontStyle());
	mText.setAttribute("font-weight",font.getFontWeight());	    	
	mText.setAttribute("font-size",font.getFontSizeString());
	mText.setAttribute("font-family",font.getFontFamily());
	
    mText.setAttribute("fill",font.getFillColor());
   	mText.setAttribute("stroke",font.getOutlineColor());
   	mText.setAttribute("stroke-width",font.getOutlineWidth());
    mText.setAttribute("letterSpacing",font.getLetterSpacing());
    
	// get max length and max height	
	var text = mElement.getText();
	var arr = text.split("\n");
	var maxLength;
	var maxHeight;
	var drawingSize = mElement.getDrawingSize();
	if(drawingSize[0] == -1) {
		if (typeof text == 'undefined' || text == null) text = "";
		maxLength = 0;
		maxHeight = font.getFontSize();
	    mText.textContent = "";
		mText.setAttribute("visibility", "hidden");
	  	for (i = 0; i < arr.length; i++) {
	  		mText.textContent = arr[i];
	  		var len = mText.getComputedTextLength();
	  		if(len > maxLength) maxLength = len;
	  		if(i > 0) maxHeight += 1.8 * font.getFontSize();
		}
		mText.setAttribute("visibility", "inherit");
	    mText.textContent = "";		
	    mElement.setDrawingSize([maxLength,maxHeight]);
	} else {
		maxLength = drawingSize[0];
		maxHeight = drawingSize[1];
	}	
	
	// get position of the element center 
    var pos = mElement.getPixelPosition();
	var marginFrame = mElement.getFramed()?4:0; 
	// var bBox = mText.getBBox();	
    // var size = [bBox.width + marginFrame*2, bBox.height + marginFrame*2];
// Paco    var size = [1.15*maxLength + marginFrame*2, maxHeight + marginFrame*2];
    var size = [maxLength + marginFrame*2, maxHeight/1.5 + marginFrame*2];
    var offset = EJSS_DRAWING2D.Element.getSWRelativePositionOffset(
    		mElement.getRelativePosition(), size[0], size[1]);  
    var xmargin = mElement.getMarginX();
    var ymargin = mElement.getMarginY();
    var x = pos[0]+offset[0]+xmargin+marginFrame;
    var y = pos[1]+offset[1]+ymargin-marginFrame;

	// text position	
	mText.setAttribute("x",x);
	mText.setAttribute("y",y);

	// set text keeping newlines
  	for (i = arr.length-1; i >= 0; i--) {
		var ts = document.createElementNS("http://www.w3.org/2000/svg", "tspan");
        if(i == arr.length-1) ts.setAttribute("y", y);
        else ts.setAttribute("dy", "-1.2em");
        ts.setAttribute("x", x);
		ts.textContent = arr[i];
		mText.appendChild(ts);
	}

    if (mElement.getFramed()) { // framed    
    	var tbmargin = marginFrame;
    	var rlmargin = marginFrame;
    	var wlen = size[0] - marginFrame*2;
    	var hlen = size[1] - marginFrame*2;    	
    	
	    // draw box
	    if(wlen > 0) {
	    	var bottom = y + tbmargin, top = y - hlen - tbmargin;
	    	var left = x - rlmargin, rigth = x + wlen + tbmargin;		 		 
			mBox.setAttribute('d', 
			   	 "M " + left + " " + bottom + 
			   	" L " + rigth + " " + bottom + 
			   	" L " + rigth + " " + top + 
			   	" L " + left + " " + top + " z");    
		    var style = mElement.getStyle(); 
		    if(style.getDrawFill())	
		    	mBox.setAttribute("fill",style.getFillColor());
		    else 
		    	mBox.setAttribute("fill","none");    
		    if(style.getDrawLines()) {
		    	mBox.setAttribute("stroke",style.getLineColor());
		    	mBox.setAttribute("stroke-width",style.getLineWidth());
		    } else {
		    	mBox.setAttribute("stroke","none");
		    	mBox.setAttribute("stroke-width",0);    	
		    }       
			var attributes = style.getAttributes();
			for (var attr in attributes) {
			  	mShape.setAttribute(attr,attributes[attr]);
			}		    
		} else {
	    	mBox.setAttribute("stroke","none");
	    	mBox.setAttribute("fill","none");    		
		}
	}
		    
	// angle of text  	 
	var angletext = 0;
	switch(mElement.getWritingMode()) {
		case EJSS_DRAWING2D.Text.MODE_TOPDOWN : angletext = 90; break;
		case EJSS_DRAWING2D.Text.MODE_RIGTHLEFT : angletext = 180; break;
		case EJSS_DRAWING2D.Text.MODE_DOWNTOP : angletext = 270; break;
		case EJSS_DRAWING2D.Text.MODE_LEFTRIGHT: angletext = 0; break;
	} 	  	  	
    mInnerGroup.setAttribute("transform","rotate(" + angletext + " " + (pos[0]+xmargin) + " " + (pos[1]+ymargin) + ")"); 
    
    // avoid pointer events in mobiles
    mGroup.setAttribute('pointer-events', 'none');

	return mGroup;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG trace
 */
EJSS_SVGGRAPHICS.trace = function(mGraphics, mElement) {  
	
	// generates path for trail
	function pathForTrail(points, x, y, sx, sy) {
		var path = "";
		for(var i=0; i<points.length; i++) {
		  var point = points[i];	  
	      var xx = x + point[0]*sx;
	      var yy = y + point[1]*sy;
	      var type = point[2];		
		  if ((i==0) || (type == 0)) // 0 is NOT CONNECTION
		  	path += " M " + xx + " " + yy;		// move 
	      else       	
	      	path += " L " + xx + " " + yy;		// line
		}  	  	
	    return path;
	}  	

	// creates ellipse mark
	function ellipseMark(mark, markx, marky, size, markstyle) {
		// get position of the element center 
	    var x = markx;
	    var y = marky;
		
		// get half sizes 		
	    var mx = Math.abs(size[0]/2);
	    var my = Math.abs(size[1]/2);

	 	// set attributes	    	    
	    mark.setAttribute("cx",x);
	    mark.setAttribute("cy",y);
	    mark.setAttribute("rx",mx);
	    mark.setAttribute("ry",my);
		
		// set style
	    if(markstyle.getDrawFill())	
	    	mark.setAttribute("fill",markstyle.getFillColor());
	    else 
	    	mark.setAttribute("fill","none");    
	    if(markstyle.getDrawLines()) {
	    	mark.setAttribute("stroke",markstyle.getLineColor());
	    	mark.setAttribute("stroke-width",markstyle.getLineWidth());
	    } else {
	    	mark.setAttribute("stroke","none");
	    	mark.setAttribute("stroke-width",0);    	
	    }        
		mark.setAttribute("shapeRendering",markstyle.getShapeRendering());  	
	}  	

	// creates rectangle mark
	function rectangleMark(mark, markx, marky, size, markstyle) {
		// get position of the element center 
	    var x = markx;
	    var y = marky;
		
		// get half sizes 		
	    var mx = Math.abs(size[0]/2);
	    var my = Math.abs(size[1]/2);

	 	// set attributes	    	    
	    mMark.setAttribute('d', 
	    	 "M " + (x-mx) + " " + (y+my) + 
	    	" L " + (x+mx) + " " + (y+my) + 
	    	" L " + (x+mx) + " " + (y-my) + 
	    	" L " + (x-mx) + " " + (y-my) + " z");
	
		// set style
	    if(markstyle.getDrawFill())	
	    	mark.setAttribute("fill",markstyle.getFillColor());
	    else 
	    	mark.setAttribute("fill","none");    
	    if(markstyle.getDrawLines()) {
	    	mark.setAttribute("stroke",markstyle.getLineColor());
	    	mark.setAttribute("stroke-width",markstyle.getLineWidth());
	    } else {
	    	mark.setAttribute("stroke","none");
	    	mark.setAttribute("stroke-width",0);    	
	    }        
		mark.setAttribute("shapeRendering",markstyle.getShapeRendering());  	
	}  	

	// creates area mark
	function areaMark(mark, lastmarkx, lastmarky, markx, marky, axis, markstyle) {		
	 	// set attributes	    	    
	    mMark.setAttribute('d', 
	    	 "M " + lastmarkx + " " + axis + 
	    	" L " + lastmarkx + " " + lastmarky + 
	    	" L " + markx + " " + marky + 
	    	" L " + markx + " " + axis + " z");
	
		// set style
	    if(markstyle.getDrawFill())	
	    	mark.setAttribute("fill",markstyle.getFillColor());
	    else 
	    	mark.setAttribute("fill","none");    
	    if(markstyle.getDrawLines()) {
	    	mark.setAttribute("stroke",markstyle.getLineColor());
	    	mark.setAttribute("stroke-width",markstyle.getLineWidth());
	    } else {
	    	mark.setAttribute("stroke","none");
	    	mark.setAttribute("stroke-width",0);    	
	    }        
		mark.setAttribute("shapeRendering",markstyle.getShapeRendering());  	
	}  	
		
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
	// create SVG element
	var mTraceGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	group.appendChild(mTraceGroup);	    

    // remove SVG element (not reusing element)
	var lastGroup = mGraphics.getElementById(mElement.getName());	
	if (lastGroup !== null) { 
		group.insertBefore(mTraceGroup,lastGroup);
		group.removeChild(lastGroup);
	}
   	
    // create SVG element group
    mTraceGroup.setAttribute("id", mElement.getName());
    	    
    // create trail element
    var mTrail = document.createElementNS("http://www.w3.org/2000/svg","path"); 
    mTrail.setAttribute("id", mElement.getName() + ".trail");
    mTraceGroup.appendChild(mTrail);	    

    // create marks element
    var mMarks = document.createElementNS("http://www.w3.org/2000/svg","g"); 
    mMarks.setAttribute("id", mElement.getName() + ".marks");
    mTraceGroup.appendChild(mMarks);	    

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
        
	// draw trail
    var trailpath = pathForTrail(mElement.getPoints(), x-mx, y-my, size[0], size[1])     
    if(trailpath !== "") {
    	mTrail.setAttribute('d', trailpath);
    			
		// set style for trail
	    var style = mElement.getStyle(); 
	    if(style.getDrawFill()) 	
	    	mTrail.setAttribute("fill",style.getFillColor());
	    else 
	    	mTrail.setAttribute("fill","none");    
	    if(style.getDrawLines()) {
	    	mTrail.setAttribute("stroke",style.getLineColor());
	    	mTrail.setAttribute("stroke-width",style.getLineWidth());
	    } else {
	    	mTrail.setAttribute("stroke","none");
	    	mTrail.setAttribute("stroke-width",0);    	
	    }        
		mTrail.setAttribute("shapeRendering",style.getShapeRendering());  	
    			
		// draw marks     
		var points = mElement.getPoints();
		var styleList = mElement.getMarkStyleList();
		var lastpointX, lastpointY;	// for mark area
		for(var i=0; i<points.length; i++) {
			var point = points[i];	  
			var markx = (x-mx) + point[0]*size[0];		// mark x
			var marky = (y-my) + point[1]*size[1];     // mark y
		    var markType = point[3];    		// mark type
			var markstyle = styleList.length>i?styleList[i]:point[4];	// mark style
			var marksize = [point[5],point[6]];		// mark size
			var markaxis = point[7];			
	
		    if (markType == EJSS_DRAWING2D.Trace.ELLIPSE) { // circle
			    var mMark = document.createElementNS("http://www.w3.org/2000/svg","ellipse"); 
			    mMark.setAttribute("id", mElement.getName() + ".mark" + i);
			    mMarks.appendChild(mMark);	    		    	
				ellipseMark(mMark, markx, marky, marksize, markstyle);
		    } else if (markType == EJSS_DRAWING2D.Trace.RECTANGLE) { // rectangle
			    var mMark = document.createElementNS("http://www.w3.org/2000/svg","path"); 
			    mMark.setAttribute("id", mElement.getName() + ".mark" + i);
			    mMarks.appendChild(mMark);	    		    	
				rectangleMark(mMark, markx, marky, marksize, markstyle);
		    } else if (markType == EJSS_DRAWING2D.Trace.BAR) { // bar
			    var mMark = document.createElementNS("http://www.w3.org/2000/svg","path"); 
			    mMark.setAttribute("id", mElement.getName() + ".mark" + i);
			    mMarks.appendChild(mMark);	  
			    // create bar  		    	
				//var axisx = mElement.getGroupPanel().getPixelPositionWorldOrigin()[1];
				var axisx = mElement.getGroupPanel().toPixelPosition([0,markaxis])[1];
				
		    	marksize[1] = axisx-marky;			// bar size
		    	marky = marky + marksize[1]/2;		// new mark y	    	
		    	rectangleMark(mMark, markx, marky, marksize, markstyle);
			}
			else if (markType == EJSS_DRAWING2D.Trace.AREA) { // area
		    	if(i!=0) {				  
				    var mMark = document.createElementNS("http://www.w3.org/2000/svg","path"); 
				    mMark.setAttribute("id", mElement.getName() + ".mark" + i);
				    mMarks.appendChild(mMark);	    		    	
					// create area
					//var axisx = mElement.getGroupPanel().getPixelPositionWorldOrigin()[1];
					
    				var axisx = mElement.getGroupPanel().toPixelPosition([0,markaxis])[1];
					var lastmarkx = (x-mx) + lastpointX*size[0];		// mark x
					var lastmarky = (y-my) + lastpointY*size[1];     // mark y
					areaMark(mMark, lastmarkx, lastmarky, markx, marky, axisx, markstyle);
		    	} 
		    	lastpointX = point[0];    	
		    	lastpointY = point[1];
		    }		  
		}  	  		  
	
		var attributes = style.getAttributes();
		for (var attr in attributes) {
		  	mTraceGroup.setAttribute(attr,attributes[attr]);
		}    	    	    			
	}
		
	return mTraceGroup;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG trail
 */
EJSS_SVGGRAPHICS.trail = function(mGraphics, mElement) {

  /** adds a segment to the trail
   */
   function addSegment(group,points,num,style,pX,pY,sX,sY) {
     if (num<=0) return; // Nothing to add
	
	var segment = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	group.appendChild(segment);	    
	
    var path = "";
	for (var i=0; i<num; i++) {
	  var point = points[i];	  
	  var xx = pX + point[0]*sX;
	  var yy = pY + point[1]*sY;
	  var type = point[2];		
	  if ((i==0) || (type == 0)) // 0 is NOT CONNECTION
	  	path += " M " + xx + " " + yy;		// move 
      else       	
     	path += " L " + xx + " " + yy;		// line
 	}  	  	
    segment.setAttribute('d', path);
	if (style.getDrawFill()) segment.setAttribute("fill",style.getFillColor());
	else segment.setAttribute("fill","none");    
	if (style.getDrawLines()) {
	  segment.setAttribute("stroke",style.getLineColor());
	  segment.setAttribute("stroke-width",style.getLineWidth());
	} else {
	  segment.setAttribute("stroke","none");
	  segment.setAttribute("stroke-width",0);    	
	}        
	segment.setAttribute("shapeRendering",style.getShapeRendering());
		
	var attributes = style.getAttributes();
	for (var attr in attributes) segment.setAttribute(attr,attributes[attr]);
  }

	// get element group
    var group, elementGroup = mElement.getGroup();
    if (elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    
	// create SVG element
	var mTrail = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	group.appendChild(mTrail);	    

    // remove SVG element (not reusing element)
	var lastTrail = mGraphics.getElementById(mElement.getName());	
	if (lastTrail !== null) { 
		group.insertBefore(mTrail,lastTrail);
		group.removeChild(lastTrail);
	}

	// set name 
	mTrail.setAttribute("id", mElement.getName());
    	
	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
	
	var pX = x-mx;
	var pY = y-my;
	var sX = size[0];
	var sY = size[1];
	
	var segmentCount = mElement.getSegmentsCount();
	if (segmentCount>0) {
	  for (var i=0; i<segmentCount; i++) { 
	  	var num = mElement.getSegmentPoints(i).length;
	    addSegment(mTrail,mElement.getSegmentPoints(i),num,mElement.getSegmentStyle(i),pX,pY,sX,sY);
	  }
	}
	
	//mElement.dataCollected(); // add temporary points
	var num = mElement.getCurrentPoints().length; 
	addSegment(mTrail,mElement.getCurrentPoints(),num,mElement.getStyle(),pX,pY,sX,sY);
	
	return mTrail;


}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};
EJSS_SVGGRAPHICS.Utils = {};

/**
 * @param value
 * @returns crisp value
 */
EJSS_SVGGRAPHICS.Utils.crispValue = function(value) {
  	return (Math.round(+value.toFixed(4)+0.5)-0.5);  	
}

/**
 * @param value
 * @returns crisp top value
 */
EJSS_SVGGRAPHICS.Utils.crispTop = function(value) {
  	return (Math.round(+value.toFixed(4)+0.5)-0.5);  	
}

/**
 * @param value
 * @returns round value
 */
EJSS_SVGGRAPHICS.Utils.round = function(value) {
  	return +value.toFixed(4);  	
}

/**
 * @param HSL, where H ∈ [0°, 360°), S ∈ [0, 1], L ∈ [0, 1]
 * @return rgb
 */
EJSS_SVGGRAPHICS.Utils.hsl2rgb = function (hsl) {
	var H = hsl[0], S = hsl[1], L = hsl[2];
    /* calculate chroma */
    var C = (1 - Math.abs((2 * L) - 1)) * S;

    /* Find a point (R1, G1, B1) along the bottom three faces of the RGB cube, with the same hue and chroma as our color (using the intermediate value X for the second largest component of this color) */
    var H_ = H / 60;
    var X = C * (1 - Math.abs((H_ % 2) - 1));
    var R1, G1, B1;

    if (H === undefined || isNaN(H) || H === null) {
        R1 = G1 = B1 = 0;
    }
    else {
        if (H_ >= 0 && H_ < 1) { R1 = C; G1 = X; B1 = 0;
        } else if (H_ >= 1 && H_ < 2) { R1 = X; G1 = C; B1 = 0;
        } else if (H_ >= 2 && H_ < 3) { R1 = 0; G1 = C; B1 = X;
        } else if (H_ >= 3 && H_ < 4) { R1 = 0; G1 = X; B1 = C;
        } else if (H_ >= 4 && H_ < 5) { R1 = X; G1 = 0; B1 = C;
        } else if (H_ >= 5 && H_ < 6) { R1 = C; G1 = 0; B1 = X; }
    }

    /* Find R, G, and B by adding the same amount to each component, to match lightness */
    var m = L - (C / 2);
    var R, G, B;
    /* Normalise to range [0,255] by multiplying 255 */
    R = (R1 + m) * 255;
    G = (G1 + m) * 255;
    B = (B1 + m) * 255;

    R = Math.round(R);
    G = Math.round(G);
    B = Math.round(B);

    return { r: R, g: G, b: B };
}


// params: imageData, width, height, depth, data, colors
// return: imageData
EJSS_SVGGRAPHICS.Utils.PNGCanvasWorker = function(params) {
	var imageData = params.data.params.imageData;
	var width = params.data.params.width;
	var height = params.data.params.height;
	var depth = params.data.params.depth;
	var data = params.data.params.data;
	var colors = params.data.params.colors;
	var index = params.data.id;    
	
	function Rgb(rgb){
		var rgb = rgb.substring(4, rgb.length-1)
	     	.replace(/ /g, '')
	     	.split(',');
	    return {r: rgb[0], g: rgb[1], b: rgb[2]};
	}
	
	function Hsl(hsl){
		var hsl = hsl.substring(4, hsl.length-1)
	     	.replace(/ /g, '')
	     	.replace(/%/g, '')
	     	.split(',');
	    return {h: hsl[0], s: (hsl[1]/100), l: (hsl[2]/100)};
	}
		
	function setPixel(imageData, x, y, r, g, b, a) {
	    index = (x + y * imageData.width) * 4;
	    imageData.data[index+0] = r;
	    imageData.data[index+1] = g;
	    imageData.data[index+2] = b;
	    imageData.data[index+3] = a;
	}
	
	var cinf = colors[0];
	var csup = colors[colors.length-1];
	for(var i=0; i<width; i++) { 
		for(var j=0; j<height; j++) {
			// get color from palette
			var index = data[i][height-j-1];
			var color = index<0 ? cinf : (index<colors.length) ? colors[index] : csup;
			// check color
			if (typeof color == "undefined") color = cinf;
			// get rgb from color  			
			if (color.substring(0,3) == "rgb") {
				rgb = Rgb(color);
			} else if (color.substring(0,3) == "hsl") {
				rgb = EJSS_SVGGRAPHICS.Utils.hsl2rgb(Hsl(color));
			}  			
			// draw pixel
  			setPixel(imageData, i, j, rgb.r, rgb.g, rgb.b, 255);
  		}  		
	}
	
    self.postMessage({ result: imageData, id: index });					  
}

EJSS_SVGGRAPHICS.Utils.PNGCanvasNoWorker = function(imageData, width, height, depth, data, colors) {
	function Rgb(rgb){
		var rgb = rgb.substring(4, rgb.length-1)
         	.replace(/ /g, '')
         	.split(',');
	    return {r: rgb[0], g: rgb[1], b: rgb[2]};
	}

	function Hsl(hsl){
		var hsl = hsl.substring(4, hsl.length-1)
         	.replace(/ /g, '')
         	.replace(/%/g, '')
         	.split(',');
	    return {h: hsl[0], s: (hsl[1]/100), l: (hsl[2]/100)};
	}
		
	function setPixel(imageData, x, y, r, g, b, a) {
	    index = (x + y * imageData.width) * 4;
	    imageData.data[index+0] = r;
	    imageData.data[index+1] = g;
	    imageData.data[index+2] = b;
	    imageData.data[index+3] = a;
	}
	
	// set data
	var cinf = colors[0];
	var csup = colors[colors.length-1];
  	for(var i=0; i<width; i++) { 
  		for(var j=0; j<height; j++) {
  			// get color from palette
  			var index = data[i][height-j-1];
  			var color = index<0 ? cinf : (index<colors.length) ? colors[index] : csup;
  			// check color
  			if (typeof color == "undefined") color = cinf;
  			// get rgb from color  			
			if (color.substring(0,3) == "rgb") {
				rgb = Rgb(color);
			} else if (color.substring(0,3) == "hsl") {
				rgb = EJSS_SVGGRAPHICS.Utils.hsl2rgb(Hsl(color));
			}  			
  			// draw pixel
  			setPixel(imageData, i, j, rgb.r, rgb.g, rgb.b, 255);
  		}  		
	}		
		
};

EJSS_SVGGRAPHICS.Utils.PNGCanvas = function(width, height, depth, data, colors, callback) {	
	var element = document.createElement("canvas");
	element.setAttribute('width',width);
	element.setAttribute('height',height);
	var c = element.getContext("2d");
	
	// create a new pixel array
	var imageData = c.createImageData(width, height);
	
	if(EJSS_TOOLS.Worker && EJSS_TOOLS.Worker.runFunction) {
		EJSS_TOOLS.Worker.runFunction("PNGCanvasWorker", EJSS_SVGGRAPHICS.Utils.PNGCanvasWorker, 
			{ imageData: imageData, 
			  width: width, height: height, depth: depth, 
			  data: data, colors: colors}, 
			function(result){ 			
				c.putImageData(result.result, 0, 0); // at coords 0,0	
				callback(element.toDataURL());
			});		
	} else {
		EJSS_SVGGRAPHICS.Utils.PNGCanvasNoWorker(imageData, width, height, depth, data, colors);
				
		// copy the image data back onto the canvas
		c.putImageData(imageData, 0, 0); // at coords 0,0		
		callback(element.toDataURL());	
	}
	
};

EJSS_SVGGRAPHICS.Utils.ImageDataCanvas = function(width, height, depth, data, colors, callback) {	
	var element = document.createElement("canvas");
	element.setAttribute('width',width);
	element.setAttribute('height',height);
	var c = element.getContext("2d");
	
	// create a new pixel array
	var imageData = c.createImageData(width, height);
	
	EJSS_TOOLS.Worker.runFunction("PNGCanvasWorker", EJSS_SVGGRAPHICS.Utils.PNGCanvasWorker, 
		{ imageData: imageData, 
		  width: width, height: height, depth: depth, 
		  data: data, colors: colors}, 
		function(result){		
			callback(result);
		});
};/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG video
 */
EJSS_SVGGRAPHICS.video = function(mGraphics, mElement) {  
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape
	var mShape = mGraphics.getElementById(mElement.getName());	
	var myVidSrc;	
	if (mShape === null) { 	// exits?
	    // create SVG element
	   	mShape = document.createElementNS("http://www.w3.org/2000/svg","foreignObject");
	    mShape.setAttribute("id",mElement.getName());	   
	    group.appendChild(mShape);
	    // create video src 
		myVidSrc = document.createElement('video');
		myVidSrc.setAttribute("id", mElement.getName() + ".video");

		// select src
		navigator.getUserMedia = navigator.getUserMedia || navigator.webkitGetUserMedia || navigator.mozGetUserMedia || navigator.msGetUserMedia || navigator.oGetUserMedia;
		if (navigator.getUserMedia && mElement.getWebCam()) {       
	   		navigator.getUserMedia({video: true}, function(s) { myVidSrc.src = window.URL.createObjectURL(s); }, function(e) {} );
		} else {	
			myVidSrc.src = mElement.getVideoUrl();
		}		

		mShape.appendChild(myVidSrc);	    
	}
	else {
		myVidSrc = document.getElementById(mElement.getName() + ".video");
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = Math.abs(size[0]/2);
    var my = Math.abs(size[1]/2);
    
 	// set video src attrs
 	myVidSrc.loop = mElement.getLoop();	
 	myVidSrc.type = mElement.getType();
 	myVidSrc.poster = mElement.getPoster();
	myVidSrc.autoplay = mElement.getAutoPlay();
	myVidSrc.controls = mElement.getControls();
	if(!isNaN(myVidSrc.duration) && (myVidSrc.duration >= mElement.getCurrentTime()))
		 myVidSrc.currentTime = mElement.getCurrentTime();
	myVidSrc.width = Math.abs(size[0]);
	myVidSrc.height = Math.abs(size[1]);
	
 	// set SVG element	
	mShape.setAttribute("x",(x-mx));
	mShape.setAttribute("y",(y-my));
	
	// play video 
	if(mElement.isPlay()) 
		myVidSrc.play()
	else 
		myVidSrc.pause();	 	
	
	var attributes = mElement.getStyle().getAttributes();
	for (var attr in attributes) {
	  	mShape.setAttribute(attr,attributes[attr]);
	}    	    	    					  	
	
	return mShape;         
}/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_SVGGRAPHICS = EJSS_SVGGRAPHICS || {};

/**
 * @param mGraphics Element where draw
 * @param mElement Element to draw
 * @returns A SVG wheel
 */
EJSS_SVGGRAPHICS.wheel = function(mGraphics, mElement) {  
	// get element group
    var group, elementGroup = mElement.getGroup();
    if(elementGroup !== null) 
    	group = mGraphics.getElementById(elementGroup.getName());
    else 
    	group = mGraphics;
    	
    // get shape	
	var mGroup = mGraphics.getElementById(mElement.getName());
	var mCircle, mCross;
	if (mGroup === null) { 	// exits?
	    // create SVG element group
	    mGroup = document.createElementNS("http://www.w3.org/2000/svg","g"); 
	    mGroup.setAttribute("id", mElement.getName());
	    group.appendChild(mGroup);	    
	    // create ellipse element
	    mCircle = document.createElementNS("http://www.w3.org/2000/svg","ellipse"); 
	    mCircle.setAttribute("id", mElement.getName() + ".circle");
	    mGroup.appendChild(mCircle);	    
	    // create cross element
	    mCross = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    mCross.setAttribute("id", mElement.getName() + ".cross");
	    mGroup.appendChild(mCross);	    
	} else {
		mCircle = mGraphics.getElementById(mElement.getName() + ".circle");
		mCross = mGraphics.getElementById(mElement.getName() + ".cross");		
	}

	// get position of the element center 
    var pos = mElement.getPixelPosition();
    var size = mElement.getPixelSizes();     
    var offset = mElement.getRelativePositionOffset(size);  
    var x = pos[0]+offset[0];
    var y = pos[1]+offset[1];
	
	// get half sizes 		
    var mx = size[0]/2;
    var my = size[1]/2;
    
	// set circle atts
    mCircle.setAttribute("cx",x);
    mCircle.setAttribute("cy",y);
    mCircle.setAttribute("rx",Math.abs(mx));
    mCircle.setAttribute("ry",Math.abs(my));
      	
	// set cross atts
    mCross.setAttribute('d', 
    	 "M " + (x-mx) + " " + y + 
    	" L " + (x+mx) + " " + y + 
    	" M " + x + " " + (y-my) + 
    	" L " + x + " " + (y+my));
	
	// set style
    var style = mElement.getStyle(); 
    if(style.getDrawFill()) 	
    	mGroup.setAttribute("fill",style.getFillColor());
    else 
    	mGroup.setAttribute("fill","none");    
    if(style.getDrawLines()) {
    	mGroup.setAttribute("stroke",style.getLineColor());
    	mGroup.setAttribute("stroke-width",style.getLineWidth());
    } else {
    	mGroup.setAttribute("stroke","none");
    	mGroup.setAttribute("stroke-width",0);    	
    }        
	mGroup.setAttribute("shapeRendering",style.getShapeRendering());  	

	var attributes = style.getAttributes();
	for (var attr in attributes) {
	  	mGroup.setAttribute(attr,attributes[attr]);
	}    	    	    					  	

	return mGroup;         
}/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Deployment for 2D drawing.
 * @module 2Dgraphics 
 */

var EJSS_GRAPHICS = EJSS_GRAPHICS || {};

/**
 * SVGGraphics class
 * @class SVGGraphics 
 * @constructor  
 */
EJSS_GRAPHICS.SvgGraphics = {

  /**
   * Return document box
   * @return box  
   */
  getOffsetRect: function(graphics) {
  	return EJSS_GRAPHICS.GraphicsUtils.getOffsetRect(graphics);
  }
};

/**
 * Constructor for Element
 * @param mName Identifier SVG element in HTML document
 * @returns A SVG graphics
 */
EJSS_GRAPHICS.svgGraphics = function(mName) {
  var self = EJSS_INTERFACE.svgGraphics(mName);	// reference returned     
  var mInterfaceGraphics = self.getDOMElement();
  var mDefs = document.createElementNS("http://www.w3.org/2000/svg", "defs");
  mInterfaceGraphics.appendChild(mDefs);	     



  /**
   * Return SVG document for event handle
   * @return SVG element 
   */
  self.getEventContext = function() {  	
    return mInterfaceGraphics;
  };
  
  /**
   * Return SVG document box
   * @return width integer 
   */
  self.getBox = function() {
  	var box = EJSS_GRAPHICS.CanvasGraphics.getOffsetRect(self)
  	box.width -= 1;
  	box.height -= 1;
  	return box;
  };

  /**
   * Remove a shape
   * @param name
   */
  self.remove = function(name) {	
    var myElement = mInterfaceGraphics.getElementById(name);		// get the SVG element from document svg
    if (myElement !== null) { 
    	mInterfaceGraphics.removeChild(myElement);
    	return true;
    }
    return false;
  };

  /**
   * Reset the SVG graphics
   */
  self.reset = function() {	
	while (mInterfaceGraphics.firstChild) {
	    mInterfaceGraphics.removeChild(mInterfaceGraphics.firstChild);
	} 
  	// create defs
//  	mDefs = document.createElementNS("http://www.w3.org/2000/svg", "defs");
  	mInterfaceGraphics.appendChild(mDefs);	     
  };

  
  self.getDefs = function() { return mDefs; }
  
  /**
   * Draw Elements
   * @param elements array of Elements and Element Sets
   * @param force whether force draw elements
   */
  self.draw = function(elements,force) {
    for (var i=0, n=elements.length; i<n; i++) { 
      var element = elements[i];
      if(element.getElements) {	// whether element set
      	var lasts = element.getLastElements();
  		for(var j=0, m=lasts.length; j<m; j++) { // remove last removed elements
			var mShape = mInterfaceGraphics.getElementById(lasts[j].getName());
			if(mShape) mShape.parentNode.removeChild(mShape);
  		}
      }
  	  if(force || element.isChanged() || element.isGroupChanged() || element.isMustProject()) {
  		var build = self.drawElement(element);
		if(typeof build != "undefined" && build != null)  		 
			self.transformElement(element, build);
	  }
    }
  };
    
  /**
   * Draw Element
   * @param element
   */
  self.drawElement = function (element) {
  	var Shape = EJSS_DRAWING2D.Shape;			
    // console.log("Drawing element "+element.getName());
    var build = null;
	
  	switch (element.getClass()) {
		case "ElementCustom":
		  {
		    var customFunction = element.getFunction();
		    if (customFunction) build = customFunction(mInterfaceGraphics,element);	
		  }
		  break;
  		case "ElementGroup": 	build = EJSS_SVGGRAPHICS.group(mInterfaceGraphics,element);	break;
  		case "ElementShape": 
			switch (element.getShapeType()) {
			  default : 
			  case Shape.ELLIPSE         : build = EJSS_SVGGRAPHICS.ellipse(mInterfaceGraphics,element); 	break;
			  case Shape.RECTANGLE       : build = EJSS_SVGGRAPHICS.rectangle(mInterfaceGraphics,element);	break;
			  case Shape.ROUND_RECTANGLE : build = EJSS_SVGGRAPHICS.roundRectangle(mInterfaceGraphics,element); break;
			  case Shape.WHEEL           : build = EJSS_SVGGRAPHICS.wheel(mInterfaceGraphics,element); 		break;
			  case Shape.NONE            : // do not break;
			  case Shape.POINT           : build = EJSS_SVGGRAPHICS.point(mInterfaceGraphics,element); 		break;
			}
			break;
		case "ElementSegment": 	build = EJSS_SVGGRAPHICS.segment(mInterfaceGraphics,element);	break;
		case "ElementImage":   	build = EJSS_SVGGRAPHICS.image(mInterfaceGraphics,element);		break;
		case "ElementVideo":   	build = EJSS_SVGGRAPHICS.video(mInterfaceGraphics,element);		break;
		case "ElementArrow":	build = EJSS_SVGGRAPHICS.arrow(mInterfaceGraphics,element);		break;
		case "ElementText":		build = EJSS_SVGGRAPHICS.text(mInterfaceGraphics,element);		break;
		case "ElementSpring":	build = EJSS_SVGGRAPHICS.spring(mInterfaceGraphics,element);		break;
		case "ElementTrail":	build = EJSS_SVGGRAPHICS.trail(mInterfaceGraphics,element);		break;
		case "ElementTrace":	build = EJSS_SVGGRAPHICS.trace(mInterfaceGraphics,element);		break;
		case "ElementGrid":		build = EJSS_SVGGRAPHICS.grid(mInterfaceGraphics,element);		break;
		case "ElementAxis":		build = EJSS_SVGGRAPHICS.axis(mInterfaceGraphics,element);		break;
		case "ElementCursor":	build = EJSS_SVGGRAPHICS.cursor(mInterfaceGraphics,element);		break;
		case "ElementPolygon":	build = EJSS_SVGGRAPHICS.polygon(mInterfaceGraphics,element);	break;
		case "ElementAnalyticCurve":	build = EJSS_SVGGRAPHICS.analyticCurve(mInterfaceGraphics,element);	break;
		case "ElementCellLattice": 	build = EJSS_SVGGRAPHICS.cellLattice(mInterfaceGraphics,element);	break; 			
		case "ElementByteRaster": 	build = EJSS_SVGGRAPHICS.byteRaster(mInterfaceGraphics,element);	break;			
		case "ElementMesh": 	build = EJSS_SVGGRAPHICS.mesh(mInterfaceGraphics,element);		break;			
		case "ElementTank":   	build = EJSS_SVGGRAPHICS.tank(mInterfaceGraphics,element);		break;
		case "ElementPipe":   	build = EJSS_SVGGRAPHICS.pipe(mInterfaceGraphics,element);		break;
		case "ElementHistogram":build = EJSS_SVGGRAPHICS.histogram(mInterfaceGraphics,element);		break;

		case "ElementScalarField":	build = EJSS_SVGGRAPHICS.scalarField(mInterfaceGraphics,element);		break;
		case "ElementCanvas":	build = EJSS_SVGGRAPHICS.canvas(mInterfaceGraphics,element);		break;
	}
	if(build) {
		if (!element.isGroupVisible()) 
		   build.setAttribute("visibility","hidden");
		else
	       build.setAttribute("visibility","visible");
	}			

  	return build;
  };

  /**
   * Transform Element
   * @param element
   */
  self.transformElement = function (element, build) {
	var trstr = "";
	var trans = element.getTransformation();
	var rot = element.getRotate();
	var pos = element.getPixelPosition();	

    if (trans.length > 0) {  // matrix
    	trstr = "translate(" + pos[0] + " " + pos[1] + ")";
		trstr += "matrix(" + trans[0] + " " + trans[1] + " " + trans[2] + " " +
		 			trans[3] + " " + trans[4] + " " + trans[5] + ")";  	
    	trstr += "translate(-" + pos[0] + " -" + pos[1] + ")";
		// build.setAttribute("transform-origin", pos[0] + " " + pos[1]); // not supported in Safari and Mozilla
	}
	if (rot != 0) {  // rotation angle 	
	    var angle = EJSS_TOOLS.Mathematics.degrees(-rot);	
		trstr += " rotate(" + angle + " " + pos[0] + " " + pos[1]+ ")";			
	}	  
    //if (trstr.length>0) 
    	build.setAttribute("transform",trstr);
  }

  /**
   * Draw gutters
   * @param gutters
   */
  self.drawGutters = function(panel) {
  	var gutters = panel.getGutters();
  	
  	if (gutters.visible) {    	 			
	 	var UTILS = EJSS_SVGGRAPHICS.Utils;  	   	    

		var mShape = mInterfaceGraphics.getElementById(".myGutters");
		var mOuterBorder = mInterfaceGraphics.getElementById(".myOuterBorder");
		var mInnerBorder = mInterfaceGraphics.getElementById(".myInnerBorder");
		if (mShape === null) { 	// exits?
		    // create SVG element
		    mShape = document.createElementNS("http://www.w3.org/2000/svg","path"); 
		    mShape.setAttribute("id", ".myGutters");
		    mShape.setAttribute("stroke","none");
	    	mShape.setAttribute("stroke-width",0);    	
		    mInterfaceGraphics.appendChild(mShape);	    

		    mOuterBorder = document.createElementNS("http://www.w3.org/2000/svg","path"); 
		    mOuterBorder.setAttribute("id", ".myOuterBorder");
		    mOuterBorder.setAttribute("fill","none");    
		    mInterfaceGraphics.appendChild(mOuterBorder);	    

		    mInnerBorder = document.createElementNS("http://www.w3.org/2000/svg","path"); 
		    mInnerBorder.setAttribute("id", ".myInnerBorder");
		    mInnerBorder.setAttribute("fill","none");    
		    mInterfaceGraphics.appendChild(mInnerBorder);	
		        
		}

	 	// get gutters position    
	    var inner = panel.getInnerRect(); 

	 	// get panel position
	    var x = 0.5, y = 0.5;
	    var box = self.getBox();
    	var sx = box.width, sy = box.height;

	 	// create paths	    	   	    
	 	var guttersStyle = panel.getGuttersStyle();	 
	 	var panelStyle = panel.getStyle();   
	    if(panelStyle.getShapeRendering() == "crispEdges") {
		    mShape.setAttribute('d', 
		    	 "M " + UTILS.crispValue(x) + " " + UTILS.crispValue(y) + 
		    	" L " + UTILS.crispValue(x) + " " + UTILS.crispValue(y+sy) + 
		    	" L " + UTILS.crispValue(x+sx) + " " + UTILS.crispValue(y+sy) + 
		    	" L " + UTILS.crispValue(x+sx) + " " + UTILS.crispValue(y) +
		    	" L " + UTILS.crispValue(x) + " " + UTILS.crispValue(y) + 
		    	" M " + UTILS.crispValue(inner.x) + " " + UTILS.crispValue(inner.y) + 
		    	" L " + UTILS.crispValue(inner.x+inner.width) + " " + UTILS.crispValue(inner.y) + 
		    	" L " + UTILS.crispValue(inner.x+inner.width) + " " + UTILS.crispValue(inner.y+inner.height) + 
		    	" L " + UTILS.crispValue(inner.x) + " " + UTILS.crispValue(inner.y+inner.height) + 
		    	" L " + UTILS.crispValue(inner.x) + " " + UTILS.crispValue(inner.y) + " z");
		    mOuterBorder.setAttribute('d', 
		    	 "M " + UTILS.crispValue(x) + " " + UTILS.crispValue(y) + 
		    	" L " + UTILS.crispValue(x) + " " + UTILS.crispValue(y+sy) + 
		    	" L " + UTILS.crispValue(x+sx) + " " + UTILS.crispValue(y+sy) + 
		    	" L " + UTILS.crispValue(x+sx) + " " + UTILS.crispValue(y) +
		    	" L " + UTILS.crispValue(x) + " " + UTILS.crispValue(y) + " z");
		    mInnerBorder.setAttribute('d', 
		    	 "M " + UTILS.crispValue(inner.x) + " " + UTILS.crispValue(inner.y) + 
		    	" L " + UTILS.crispValue(inner.x+inner.width) + " " + UTILS.crispValue(inner.y) + 
		    	" L " + UTILS.crispValue(inner.x+inner.width) + " " + UTILS.crispValue(inner.y+inner.height) + 
		    	" L " + UTILS.crispValue(inner.x) + " " + UTILS.crispValue(inner.y+inner.height) + 
		    	" L " + UTILS.crispValue(inner.x) + " " + UTILS.crispValue(inner.y) + " z");
	    } else {
		    mShape.setAttribute('d', 
		    	 "M " + (x) + " " + (y) + 
		    	" L " + (x) + " " + (y+sy) + 
		    	" L " + (x+sx) + " " + (y+sy) + 
		    	" L " + (x+sx) + " " + (y) +
		    	" L " + (x) + " " + (y) + 
		    	" M " + (inner.x) + " " + (inner.y) + 
		    	" L " + (inner.x+inner.width) + " " + (inner.y) + 
		    	" L " + (inner.x+inner.width) + " " + (inner.y+inner.height) + 
		    	" L " + (inner.x) + " " + (inner.y+inner.height) + 
		    	" L " + (inner.x) + " " + (inner.y) + " z");
		    mOuterBorder.setAttribute('d', 
		    	 "M " + (x) + " " + (y) + 
		    	" L " + (x) + " " + (y+sy) + 
		    	" L " + (x+sx) + " " + (y+sy) + 
		    	" L " + (x+sx) + " " + (y) +
		    	" L " + (x) + " " + (y) + " z");
		    mInnerBorder.setAttribute('d',
		    	"M " + (inner.x) + " " + (inner.y) + 
		    	" L " + (inner.x+inner.width) + " " + (inner.y) + 
		    	" L " + (inner.x+inner.width) + " " + (inner.y+inner.height) + 
		    	" L " + (inner.x) + " " + (inner.y+inner.height) + 
		    	" L " + (inner.x) + " " + (inner.y) + " z");
		}  	    
		
		// set style (appling into gutters, i.e. panel, so panel style)
	    if(guttersStyle.getDrawFill())	
	    	mShape.setAttribute("fill",guttersStyle.getFillColor());
	    else 
	    	mShape.setAttribute("fill","none");    
	    if(guttersStyle.getDrawLines()) {
	    	mOuterBorder.setAttribute("stroke",guttersStyle.getLineColor());
	    	mOuterBorder.setAttribute("stroke-width",guttersStyle.getLineWidth());
	    } else {
	    	mOuterBorder.setAttribute("stroke","none");
	    	mOuterBorder.setAttribute("stroke-width",0);    	
	    }        
	    if(panelStyle.getDrawLines()) {
	    	mInnerBorder.setAttribute("stroke",panelStyle.getLineColor());
	    	mInnerBorder.setAttribute("stroke-width",panelStyle.getLineWidth());
	    } else {
	    	mInnerBorder.setAttribute("stroke","none");
	    	mInnerBorder.setAttribute("stroke-width",0);    	
	    }        
		mShape.setAttribute("shapeRendering",panelStyle.getShapeRendering());
		mOuterBorder.setAttribute("shapeRendering",guttersStyle.getShapeRendering());
		mInnerBorder.setAttribute("shapeRendering",panelStyle.getShapeRendering());
	}  	
  };
   /**
   * Draw panel
   * @param panel
   */
  self.drawPanel = function(panel) {
 	var UTILS = EJSS_SVGGRAPHICS.Utils;   

	var mShape = mInterfaceGraphics.getElementById(".myPanel");
	if (mShape === null) { 	// exits?
	    // create SVG element
	    mShape = document.createElementNS("http://www.w3.org/2000/svg","path"); 
	    mShape.setAttribute("id", ".myPanel");
	    mInterfaceGraphics.appendChild(mShape);	    
	}

 	// get position
    var x = 0.5, y = 0.5;
    var box = self.getBox();
    var sx = box.width, sy = box.height;

 	// create path	    	   	    
 	var style = panel.getStyle();	    	   	   
    if(style.getShapeRendering() == "crispEdges") {
	    mShape.setAttribute('d', 
	    	 "M " + UTILS.crispValue(x) + " " + UTILS.crispValue(y) + 
	    	" L " + UTILS.crispValue(x) + " " + UTILS.crispValue(y+sy) + 
	    	" L " + UTILS.crispValue(x+sx) + " " + UTILS.crispValue(y+sy) + 
	    	" L " + UTILS.crispValue(x+sx) + " " + UTILS.crispValue(y) + " z");
    } else {
	    mShape.setAttribute('d', 
	    	 "M " + (x) + " " + (y) + 
	    	" L " + (x) + " " + (y+sy) + 
	    	" L " + (x+sx) + " " + (y+sy) + 
	    	" L " + (x+sx) + " " + (y) + " z");
	}  	    
	
	// set style (appling into gutters, i.e. panel, so panel style)
    if(style.getDrawFill())	
    	mShape.setAttribute("fill",style.getFillColor());
    else 
    	mShape.setAttribute("fill","none");
    	
    if(style.getDrawLines()) {
    	mShape.setAttribute("stroke",style.getLineColor());
    	mShape.setAttribute("stroke-width",style.getLineWidth());
    } else {
    	mShape.setAttribute("stroke","none");
    	mShape.setAttribute("stroke-width",0);    	
    }        

	mShape.setAttribute("shapeRendering",style.getShapeRendering());
  };
   
  self.importSVG = function(callback) {
    var box = self.getBox();

	// canvas
    var canvas = document.createElement("canvas");
    canvas.width = box.width+1;
    canvas.height = box.height+1;
    var ctx = canvas.getContext("2d");

	// fix size
  	var dom = self.getDOMElement();
  	dom.setAttribute("width", canvas.width);
  	dom.setAttribute("height", canvas.height);
    var svg_xml = (new XMLSerializer()).serializeToString(dom);    

    var img = new Image();
//    img.src = "data:image/svg+xml;base64," + btoa(svg_xml);
    img.src = "data:image/svg+xml;utf8," + svg_xml;

    img.onload = function() {
    	console.log("tam " + canvas.width + " " + canvas.height);
        ctx.drawImage(img,0,0);
        if(callback) callback(canvas.toDataURL("image/png"));
    } 
    return img.src;
  }
    
 return self;     
      
}
/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * InfoText
 * @class InfoText 
 * @constructor  
 */
EJSS_DRAWING2D.InfoText = {

	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Text.registerProperties(element, controller);
		// super class
		
		controller.registerProperty("Info",  element.setInfo);
		controller.registerProperty("Format",  element.setFormat);		
	},
};

/**
 * Creates a 2D InfoText
 * @method infoText
 */
EJSS_DRAWING2D.infoText = function(name) {
	var self = EJSS_DRAWING2D.text(name);
	var super_getText = self.getText;
	
	var mInfo;
	var mFormat;			// replace ".^", ".##" por los números pasados

	/**
	 * Set info function or element with getInfo function
	 */
	self.setInfo = function(info) {
		mInfo = info;
	}

	self.getText = function() {
		if(typeof mInfo != "undefined") {	// exits info function 
			if(mInfo.getInfo) 
				return mInfo.getInfo();
			else if (typeof mInfo == "function") 
				return mInfo();
		} else { // uses text with the format
			var text = super_getText();			
			// process text
			return fillString(mFormat,text);
		}
	}

	self.setFormat = function(format) {
		mFormat = format;
	}

	self.isChanged = function() {
		return true;
	}

	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.InfoText.registerProperties(self, controller);
	};

	/*
	 * Fill formatted string
	 */
	function fillString(format, text) {
		var t = "";
		var txt = format;
		var numbers = format.match(/[0-9]*\.(\^|#)+/g);
		if (numbers==null) return text;
		if(!Array.isArray(text)) {
			text = JSON.parse("["+text+"]");
		}

		for (var i=0; i<numbers.length; i++) {
			var precision = numbers[i].split("#").length - 1;
			// check text
			if(isNaN(text[i])) 
				t = text[i];
			else
				t = text[i].toFixed(precision)
			txt = txt.replace(numbers[i], t);
		}
		
		return txt;		
	}
	
	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * EJSS framework for interface element.
 * @module Drawing2D 
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Knob
 * @class Knob 
 * @constructor  
 */
EJSS_DRAWING2D.Knob = {
  // ----------------------------------------------------
  // Static methods
  // ----------------------------------------------------

  /**
   * static registerProperties method
   */ 
   registerProperties : function(element,controller) {
	EJSS_DRAWING2D.Meter.registerProperties(element,controller); // super class

    controller.registerAction("OnPress" ,element.getValue); 
    controller.registerAction("OnRelease" ,element.getValue); 
    controller.registerAction("OnChange" ,element.getValue); 
    
  }

};

/**
 * Knob function
 * Creates a 2D element that looks and behaves like a knob
 * @method knob
 * @param mName the name of the element
 * @returns An abstract interface element
 */
EJSS_DRAWING2D.knob = function (mName) {
  var self = EJSS_DRAWING2D.meter(mName);
  
  var mThreshold = Math.PI/5;
  var pressed_shift = [0,0];  // initial dragging shift  

  // Implementation variables
  var mKnobExterior,mKnobMiddle,mKnobInterior;

  var mDummyContainer;
  var mSVGDefs = 
	  "<svg xmlns='http://www.w3.org/2000/svg' width='0' height='0' version='1.1'>"+
      "  <defs>"+
      "    <linearGradient id='KnobExteriorGradient' x1='0%' y1='0%' x2='100%' y2='100%'>"+
      "      <stop offset='0%'   stop-color='rgb(250,250,250)' />"+
      "      <stop offset='100%' stop-color='rgb(100,100,100)' />"+
      "    </linearGradient>"+
      "    <linearGradient id='KnobMiddleGradient' x1='0%' y1='0%' x2='100%' y2='100%'>"+
      "      <stop offset='0%'   stop-color='rgb(200,200,200)' />"+
      "      <stop offset='100%' stop-color='rgb(150,150,150)' />"+
      "    </linearGradient>"+
      "    <linearGradient id='KnobInteriorGradient' x1='0%' y1='0%' x2='100%' y2='100%'>"+
      "      <stop offset='0%'   stop-color='rgb(127,127,127)' />"+
      "      <stop offset='100%' stop-color='rgb(250,250,250)' />"+
      "    </linearGradient>"+
      "    <filter id='KnobFilter' x='0' y='0' width='200%' height='200%'>"+
      "      <feOffset result='offOut' in='SourceGraphic' dx='1' dy='1' />"+
      "      <feGaussianBlur result='blurOut' in='offOut' stdDeviation='2' />"+
      "      <feBlend in='SourceGraphic' in2='blurOut' mode='normal' />"+
      "    </filter>"+
      "  </defs>"+
      "</svg>";

  
  // ----------------------------------------------------
  // Properties
  // ----------------------------------------------------
  
  // ----------------------------------------------------
  // Properties overwritten
  // ----------------------------------------------------

  self.super_setRadius = self.setRadius;

  self.setRadius = function(value) {
	  self.super_setRadius(value);
	  mKnobExterior.setChanged(true);
	  mKnobMiddle.setChanged(true);
	  mKnobInterior.setChanged(true);
  }

  self.setFillColor = function(value) { mKnobInterior.getStyle().setFillColor(value); };

  // ----------------------------------------------------
  // Properties and copies
  // ----------------------------------------------------

  /**
   * Extended registerProperties method. To be used by promoteToControlElement
   * @method registerProperties
   */
  self.registerProperties = function(controller) {
	EJSS_DRAWING2D.Knob.registerProperties(self,controller);
  };
  
  // ----------------------------------------------------
  // private or protected functions
  // ----------------------------------------------------

  self.super_addParticularChildren = self.addParticularChildren;

  self.addParticularChildren = function() {
	  self.super_addParticularChildren();
	  mKnobExterior.setParent(self);
	  mKnobMiddle.setParent(self);
	  mKnobInterior.setParent(self);
  };

  self.createBasics = function() { 
	  /*
	  var stop1 = document.createElementNS("http://www.w3.org/2000/svg", "stop");
	  stop1.setAttribute("offset","0%");
	  stop1.setAttribute("stop-color","rgb(250,250,250)");

	  var stop2 = document.createElementNS("http://www.w3.org/2000/svg", "stop");
	  stop2.setAttribute("offset","100%");
	  stop2.setAttribute("stop-color","rgb(100,100,100)");

	  var linearGradient = document.createElementNS("http://www.w3.org/2000/svg", "linearGradient");
	  linearGradient.setAttribute("id", "#KnobExteriorGradient");
	  linearGradient.setAttribute("x1","0%");
	  linearGradient.setAttribute("y1","0%");
	  linearGradient.setAttribute("x2","100%");
	  linearGradient.setAttribute("y2","100%");
	  linearGradient.appendChild(stop1);
	  linearGradient.appendChild(stop2);

	  var feOffset = document.createElementNS("http://www.w3.org/2000/svg", "feOffset");
	  feOffset.setAttribute("result","offOut");
	  feOffset.setAttribute("in","SourceGraphic");
	  feOffset.setAttribute("dx","1");
	  feOffset.setAttribute("dy","1");

	  var gaussianFilter = document.createElementNS("http://www.w3.org/2000/svg", "feGaussianBlur");
	  gaussianFilter.setAttribute("result","blurOut");
	  gaussianFilter.setAttribute("in","offOut");
	  gaussianFilter.setAttribute("stdDeviation","2");

	  var feBlend = document.createElementNS("http://www.w3.org/2000/svg", "feBlend");
	  feOffset.setAttribute("in","SourceGraphic");
	  feOffset.setAttribute("in2","blurOut");
	  feOffset.setAttribute("mode","normal");

	  var filter = document.createElementNS("http://www.w3.org/2000/svg", "filter");
	  filter.setAttribute("id","#KnobFilter");
	  filter.setAttribute("x","0");
	  filter.setAttribute("y","0");
	  filter.setAttribute("width","200%");
	  filter.setAttribute("height","200%");
	  filter.appendChild(feOffset);
	  filter.appendChild(gaussianFilter);
	  filter.appendChild(feBlend);
	  
	  /*
	  var defs = self.getParentPanel()getGraphics().getDefs(); // document.createElementNS("http://www.w3.org/2000/svg", "defs");
	  defs.appendChild(linearGradient);
	  defs.appendChild(filter);
	  
	 // self.getGraphics().appendChild(defs);
	  */

	  var mDummyContainer = document.getElementById(mName+".dummy_container");	
	  if (mDummyContainer === null) { 	// exits?
		  mDummyContainer = document.createElement('div');
	      mDummyContainer.setAttribute("id", mName+".dummy_container");
	      mDummyContainer.style.width = "0px";
	      mDummyContainer.style.height = "0px";
		  document.body.appendChild(mDummyContainer);
		  mDummyContainer.innerHTML = mSVGDefs;	    
	  }	
	  
	  mKnobExterior = EJSS_DRAWING2D.shape(mName+".exterior_shape");
	  mKnobExterior.setSize([1,1]);
	  mKnobExterior.getStyle().setDrawLines(false);
	  mKnobExterior.getStyle().setAttributes({ fill: "url(#KnobExteriorGradient)", filter:"url(#KnobFilter)"});

	  mKnobMiddle = EJSS_DRAWING2D.shape(mName+".middle_shape");
	  mKnobMiddle.setSize([0.9,0.9]);
	  mKnobMiddle.getStyle().setDrawLines(false);
	  mKnobMiddle.getStyle().setAttributes({ fill: "url(#KnobMiddleGradient)"});

	  mKnobInterior = EJSS_CORE.promoteToControlElement(
			  EJSS_DRAWING2D.shape(mName+".inner_shape"),self.getView(),mName+".inner_shape");
	  mKnobInterior.setY(0.3);
	  mKnobInterior.setSize([0.2,0.2]);
	  mKnobInterior.getStyle().setDrawLines(false);
	  mKnobInterior.getStyle().setAttributes({ fill: "url(#KnobInteriorGradient)"});

	  mKnobInterior.setProperty("EnabledPosition","ENABLED_NO_MOVE");
      mKnobInterior.setProperty("Sensitivity",10);
	  mKnobInterior.setProperty("OnDrag",dragged);
	  mKnobInterior.setProperty("OnPress",pressed);
	  mKnobInterior.setProperty("OnRelease",released);
  }
    
  self.super_createForeground = self.createForeground;
  
  self.createForeground = function() {
	  self.super_createForeground();
	  self.getElement('brand').setPosition([0,-0.7]);
	  self.getElement('units').setPosition([0, 0.9]);
  };

  self.adjustPosition = function() {
	  var actualValue = self.getValue();
	  if (actualValue<self.getMinimum()) {
		  actualValue = self.getMinimum();
//		  mArrow.getStyle().setFillColor(mDangerColor);
	  }
	  else if (actualValue>self.getMaximum()) {
		  actualValue = self.getMaximum();
//		  mArrow.getStyle().setFillColor(mDangerColor);
	  }
	  else {
//		  mArrow.getStyle().setFillColor(mColor);
	  }
      var angle = self.interpolateAngle(actualValue);
	  mKnobInterior.setPosition([0.3*Math.cos(angle),0.3*Math.sin(angle)]);
  }

  self.foregroundIsInside = function() { return false; };

  function pressed(point,info) {
	  var controller = self.getController();    		
	  if (controller) controller.invokeImmediateAction("OnPress");
	  
	  // init dragging	 
	  var point = info.point;
	  var i_point = info.element.getAbsolutePosition(true);
	  pressed_shift = [point[0]-i_point[0], point[1]-i_point[1]];	  
  }

  function released(point,info) {
	  var controller = self.getController();    		
	  if (controller) controller.invokeImmediateAction("OnRelease");	  
  }

  function dragged(point,info) {
	  var point = info.point;
	  var element = info.element;
	  var drag_point = [point[0] - pressed_shift[0], point[1] - pressed_shift[1]];	  
	  var group_pos = self.getAbsolutePosition(true);	  
	  var group_size = self.getSize();
	  drag_point[0] = drag_point[0] - group_pos[0];
	  drag_point[1] = (drag_point[1] - group_pos[1]) * (group_size[0]/group_size[1]);
	  var angle = self.interpolateAngle(self.getValue());
	  var rotation = Math.atan2(-drag_point[0],drag_point[1])+Math.PI/2;
	  if (Math.abs(rotation-angle)>mThreshold) return;
	  rotation = self.valueFromAngle(rotation);
	  element.setPosition([0.3*Math.cos(rotation),0.3*Math.sin(rotation)]);
	  var controller = self.getController();    		
	  if (controller) {
		controller.immediatePropertyChanged("Value");
		controller.invokeImmediateAction("OnChange");
	  }	        
  }
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------
  
  return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * EJSS framework for interface element.
 * @module Drawing2D 
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Knob
 * @class Knob 
 * @constructor  
 */
EJSS_DRAWING2D.KnobOld = {
  // ----------------------------------------------------
  // Static methods
  // ----------------------------------------------------

  /**
   * static registerProperties method
   */ 
   registerProperties : function(element,controller) {
	EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class

	controller.registerProperty("Value", element.setValue, element.getValue);
    controller.registerProperty("Minimum", element.setMinimum, element.getMinimum);
    controller.registerProperty("Maximum", element.setMaximum, element.getMaximum);

    controller.registerAction("OnPress" ,element.getValue); 
    controller.registerAction("OnRelease" ,element.getValue); 
    controller.registerAction("OnChange" ,element.getValue); 
    
    controller.registerProperty("Size", element.setSize);
	
  }

};

/**
 * Knob function
 * Creates a 2D element that looks and behaves like a knob
 * @method knob
 * @param mName the name of the element
 * @returns An abstract interface element
 */
EJSS_DRAWING2D.knobOld = function (mName) {
  var self = EJSS_DRAWING2D.group(mName);
  
  // Configuration variables
  var mValue = 0.0;
  var mMaximum =  1.0;
  var mMinimum = -1.0;

  var mNumberOfMarks = 21;
  var mBigMark = 4;
  var mMediumMark = 2;
  var mDigits = 0;

  var mMinimumAngle = Math.PI/2 + 1.5*Math.PI/2;
  var mMaximumAngle = Math.PI/2 - 1.5*Math.PI/2;

  var mMinAngle = -Math.PI*0.7;
  var mMaxAngle =  Math.PI*0.7;
  var mThreshold = Math.PI/5;

  // Implementation variables
  var mKnobExterior,mKnobMiddle,mKnobInterior;
  var mSegments, mTexts, mBrand, mUnits;

  var mDummyContainer;
  var mSVGDefs = 
	  "<svg xmlns='http://www.w3.org/2000/svg' version='1.1'>"+
      "  <defs>"+
      "    <linearGradient id='KnobExteriorGradient' x1='0%' y1='0%' x2='100%' y2='100%'>"+
      "      <stop offset='0%'   stop-color='rgb(250,250,250)' />"+
      "      <stop offset='100%' stop-color='rgb(100,100,100)' />"+
      "    </linearGradient>"+
      "    <linearGradient id='KnobMiddleGradient' x1='0%' y1='0%' x2='100%' y2='100%'>"+
      "      <stop offset='0%'   stop-color='rgb(200,200,200)' />"+
      "      <stop offset='100%' stop-color='rgb(150,150,150)' />"+
      "    </linearGradient>"+
      "    <linearGradient id='KnobInteriorGradient' x1='0%' y1='0%' x2='100%' y2='100%'>"+
      "      <stop offset='0%'   stop-color='rgb(127,127,127)' />"+
      "      <stop offset='100%' stop-color='rgb(250,250,250)' />"+
      "    </linearGradient>"+
      "    <filter id='KnobFilter' x='0' y='0' width='200%' height='200%'>"+
      "      <feOffset result='offOut' in='SourceGraphic' dx='1' dy='1' />"+
      "      <feGaussianBlur result='blurOut' in='offOut' stdDeviation='2' />"+
      "      <feBlend in='SourceGraphic' in2='blurOut' mode='normal' />"+
      "    </filter>"+
      "  </defs>"+
      "</svg>";

  
  // ----------------------------------------------------
  // Properties
  // ----------------------------------------------------
  
  /**
   * Sets the value of the element. 
   * @method setValue
   * @param value
   */
  self.setValue = function(value) {
    if (mValue!=value) {
    	mValue = value;
    	adjustPosition();
    }
  }
  
  /**
   * @method getValue
   * @return current value
   */
  self.getValue = function() { 
    return mValue; 
  };    

  /**
   * Sets the minimum value of the element. 
   * @method setMinimum
   * @param value
   */
  self.setMinimum = function(value) {
    if (mMinimum!=value) {
    	mMinimum = value;
    	adjustPosition();
    }
  }
  
  /**
   * @method getMinimum
   * @return current value
   */
  self.getMinimum = function() { 
    return mMinimum; 
  };    

  /**
   * Sets the minimum value of the element. 
   * @method setMaximum
   * @param value
   */
  self.setMaximum = function(value) {
	if (mMaximum!=value) {
		mMaximum = value;
		adjustPosition();		  
	}
  }
  
  /**
   * @method getMaximum
   * @return current value
   */
  self.getMaximum = function() { 
    return mMaximum; 
  };    

  /**
   * Sets the size of the element. 
   * @method setSize
   * @param value
   */
  self.setSize = function(value) {
	  self.setSizeX(value);
	  self.setSizeY(value);
  }
  
  /**
   * Extended registerProperties method. To be used by promoteToControlElement
   * @method registerProperties
   */
  self.registerProperties = function(controller) {
	createKnob();
	EJSS_DRAWING2D.Knob.registerProperties(self,controller);
  };
  
  self.superSetParent = self.setParent;

  self.setParent = function(parent, sibling) {
		self.superSetParent(parent,sibling);
		mKnobExterior.setParent(self);
		mKnobMiddle.setParent(self);
		mKnobInterior.setParent(self);
	  }

  
  function createKnob() {
	  /*
	  var stop1 = document.createElementNS("http://www.w3.org/2000/svg", "stop");
	  stop1.setAttribute("offset","0%");
	  stop1.setAttribute("stop-color","rgb(250,250,250)");

	  var stop2 = document.createElementNS("http://www.w3.org/2000/svg", "stop");
	  stop2.setAttribute("offset","100%");
	  stop2.setAttribute("stop-color","rgb(100,100,100)");

	  var linearGradient = document.createElementNS("http://www.w3.org/2000/svg", "linearGradient");
	  linearGradient.setAttribute("id", "#KnobExteriorGradient");
	  linearGradient.setAttribute("x1","0%");
	  linearGradient.setAttribute("y1","0%");
	  linearGradient.setAttribute("x2","100%");
	  linearGradient.setAttribute("y2","100%");
	  linearGradient.appendChild(stop1);
	  linearGradient.appendChild(stop2);

	  var feOffset = document.createElementNS("http://www.w3.org/2000/svg", "feOffset");
	  feOffset.setAttribute("result","offOut");
	  feOffset.setAttribute("in","SourceGraphic");
	  feOffset.setAttribute("dx","1");
	  feOffset.setAttribute("dy","1");

	  var gaussianFilter = document.createElementNS("http://www.w3.org/2000/svg", "feGaussianBlur");
	  gaussianFilter.setAttribute("result","blurOut");
	  gaussianFilter.setAttribute("in","offOut");
	  gaussianFilter.setAttribute("stdDeviation","2");

	  var feBlend = document.createElementNS("http://www.w3.org/2000/svg", "feBlend");
	  feOffset.setAttribute("in","SourceGraphic");
	  feOffset.setAttribute("in2","blurOut");
	  feOffset.setAttribute("mode","normal");

	  var filter = document.createElementNS("http://www.w3.org/2000/svg", "filter");
	  filter.setAttribute("id","#KnobFilter");
	  filter.setAttribute("x","0");
	  filter.setAttribute("y","0");
	  filter.setAttribute("width","200%");
	  filter.setAttribute("height","200%");
	  filter.appendChild(feOffset);
	  filter.appendChild(gaussianFilter);
	  filter.appendChild(feBlend);
	  
	  /*
	  var defs = self.getParentPanel()getGraphics().getDefs(); // document.createElementNS("http://www.w3.org/2000/svg", "defs");
	  defs.appendChild(linearGradient);
	  defs.appendChild(filter);
	  
	 // self.getGraphics().appendChild(defs);
	  */
	  
	  mDummyContainer = document.createElement('div');
	  document.body.appendChild(mDummyContainer);
	  mDummyContainer.innerHTML = mSVGDefs;
	  
	  mKnobExterior = EJSS_DRAWING2D.shape(mName+".exterior_shape");
	  mKnobExterior.setSize([1,1]);
	  mKnobExterior.getStyle().setDrawLines(false);
	  mKnobExterior.getStyle().setAttributes({ fill: "url(#KnobExteriorGradient)", filter:"url(#KnobFilter)"});

	  mKnobMiddle = EJSS_DRAWING2D.shape(mName+".middle_shape");
	  mKnobMiddle.setSize([0.9,0.9]);
	  mKnobMiddle.getStyle().setDrawLines(false);
	  mKnobMiddle.getStyle().setAttributes({ fill: "url(#KnobMiddleGradient)"});

	  mKnobInterior = EJSS_CORE.promoteToControlElement(
			  EJSS_DRAWING2D.shape(mName+".inner_shape"),self.getView(),mName+".inner_shape");
	  mKnobInterior.setY(0.3);
	  mKnobInterior.setSize([0.2,0.2]);
	  mKnobInterior.getStyle().setDrawLines(false);
	  mKnobInterior.getStyle().setAttributes({ fill: "url(#KnobInteriorGradient)"});

	  mKnobInterior.setProperty("EnabledPosition","ENABLED_NO_MOVE");
      mKnobInterior.setProperty("Sensitivity",10);
	  mKnobInterior.setProperty("OnDrag",dragged);
	  mKnobInterior.setProperty("OnPress",pressed);
	  mKnobInterior.setProperty("OnRelease",released);

//	  self.addElement(mKnobExterior);
//	  self.addElement(mKnobMiddle);
//	  self.addElement(mKnobInterior);


  }
    
  function adjustPosition() {
	  mValue = Math.max(Math.min(mMaximum,mValue),mMinimum);
	  var angle = mMaxAngle - (mValue-mMinimum)/(mMaximum-mMinimum)*(mMaxAngle-mMinAngle) + Math.PI/2;
	  mKnobInterior.setPosition([0.3*Math.cos(angle),0.3*Math.sin(angle)]);
  }

  function pressed(point,info) {
	  var controller = self.getController();    		
	  if (controller) controller.invokeImmediateAction("OnPress");
  }

  function released(point,info) {
	  var controller = self.getController();    		
	  if (controller) controller.invokeImmediateAction("OnRelease");
  }

  function dragged(point,info) {
	  var point = info.point;
	  var element = info.element;
	  var group_pos = self.getAbsolutePosition(true);
	  point[0] -= group_pos[0];
	  point[1] -= group_pos[1];
	  var angle = mMaxAngle - (mValue-mMinimum)/(mMaximum-mMinimum)*(mMaxAngle-mMinAngle);
	  var rotation = Math.atan2(-point[0],point[1]);
	  if (Math.abs(rotation-angle)>mThreshold) return;
	  rotation = Math.min(Math.max(rotation,mMinAngle),mMaxAngle);
	  mValue = mMaximum - (rotation-mMinAngle)/(mMaxAngle-mMinAngle)*(mMaximum-mMinimum);
	  var angle = rotation+Math.PI/2;
	  element.setPosition([0.3*Math.cos(angle),0.3*Math.sin(angle)]);
	  var controller = self.getController();    		
	  if (controller) {
		controller.immediatePropertyChanged("Value");
		controller.invokeImmediateAction("OnChange");
	  }	        

  }
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------
  
  return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Mesh
 * @class Mesh 
 * @constructor  
 */
EJSS_DRAWING2D.Mesh = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy
  	
  	},

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class

		controller.registerProperty("Data", element.setData);
		controller.registerProperty("DataType", element.setDataType);
		controller.registerProperty("Mesh", element.setMesh);

		controller.registerProperty("Points", element.setPoints);
		controller.registerProperty("Cells", element.setCells);
		controller.registerProperty("FieldAtPoints", element.setFieldAtPoints);
		controller.registerProperty("FieldAtCells", element.setFieldAtCells);
		controller.registerProperty("VectorLength", element.setVectorLength);
		
		controller.registerProperty("Boundary", element.setBoundary);
		controller.registerProperty("BoundaryLabels", element.setBoundaryLabels);
		controller.registerProperty("BoundaryColors", element.setBoundaryColors);
		controller.registerProperty("BoundaryWidth", element.setBoundaryLineWidth);
		controller.registerProperty("DrawBoundary", element.setDrawBoundary);
	}
};


/**
 * Creates a 2D Mesh
 * A Mesh is a collection of polygons, where each vertex has associated a scalar values value to it.
 * The mesh can be drawn using a color for each polygon, or a color code that uses the vertex values to find areas of equal value.
 * This second feature can be configure setting the levels to distinguish and the color of each of the levels (and below and above them)
 * The Mesh can also be given a set of lines which can be drawn to show the boundaries or special areas of the mesh, each with a different color.  
 * 
 * @method mesh
 */
EJSS_DRAWING2D.mesh = function (name) {
  var self = EJSS_DRAWING2D.element(name);

  // Configuration variables
  var mDataType = -1;	    // An int that overrides the mData type
  var mPoints; 				// double[][]
  var mValues; 				// double[][]
  var mCellValues; 			// double[][][]
  var mMeshGeometry; 		// int[][]
  var mAutoscaleZ = true; 
  var mVectorLength = 0.03;

  var mBoundaryGeometry;		// int[][]
  var mBoundaryLabels;			// int[]
  var mBoundaryColors = null; 	// color[] an array with a color for each index. If null, the element color is used
  var mBoundaryStroke = 1;	  	// stroke
  var mBoundaryDraw = true;
  
  // Implementation variables
  var mColor = EJSS_DRAWING2D.colorCoded(16, EJSS_DRAWING2D.ColorMapper.SPECTRUM); 		// ColorCoded

  // -------------------------------------
  // Public methods for mesh geometry
  // -------------------------------------

  self.getClass = function() {
  	return "ElementMesh";
  }

  /**
   * Indicates what type to consider for the setData argument.
   * Allows using the same data for different displays.
   * @param type int or String // one of 0=MESH_2D, 1=MESH_3D, 2=SCALAR_2D_FIELD, 3=SCALAR_3D_FIELD, 4=VECTOR_2D_FIELD, 5=VECTOR_3D_FIELD
   */
  self.setDataType = function(type) {
    switch (type) {
      case "MESH_2D" : mDataType = 0; break;
      case "MESH_3D" : mDataType = 1; break;
      case "SCALAR_2D_FIELD" : mDataType = 2; break;
      case "SCALAR_3D_FIELD" : mDataType = 3; break;
      case "VECTOR_2D_FIELD" : mDataType = 4; break;
      case "VECTOR_3D_FIELD" : mDataType = 5; break;
      default : mDataType = type; 
     }
    self.setChanged(true);
  }
  
  self.getDataType = function() {
  	return mDataType;
  }

  /**
   * Provides the data based on the following structure
   * @method setData
   *     "data": {
   *		"type": int, // one of  0=MESH_2D, 1=MESH_3D, SCALAR_2D_FIELD, SCALAR_3D_FIELD, VECTOR_2D_FIELD, VECTOR_3D_FIELD
   *        "problem_mesh": mesh, // the mesh that defined the problem
   *        "solution_mesh": mesh, // the mesh that contains the solution
   *        "solution_values": double[][][], // the values of the field for the solution mesh
   *	  }
   * @param data 
   */
  self.setData = function(data) {
    if (!data) return false;
    if (mDataType<0 || mDataType>5) {
   		self.setDataType(data["type"]);
    }
    switch (mDataType) {
      case 0 : // 2D mesh
      case 1 : // 3D mesh
        self.setMesh(data["problem_mesh"]); 
        return true;
      case 2 : // scalar (real or complex) 2D solution
      case 3 : // scalar (real or complex) 3D solution
      case 4 : // vector (real or complex) 2D solution
      case 5 : // vector (real or complex) 3D solution
        self.setMesh(data["solution_mesh"]);
        self.setFieldAtCells(data["solution_values"]);
        return true;
     }
     return false; // no correct data
  }

  /**
   * Provides the mesh based on the following structure
   * @method setMesh
   *     "mesh": {
   *		"points":[[x0,y0],[x1,y1],...],
   *		"cells":[[p00,p01,p02],[p10,p11,p12],...],
   *		"boundary":[[p00,p01],[p10,p11],...], // end points for each segment of the boundary		
   *		"boundary_labels":int [l0, l1, l2,...] // one for each boundary segment
   *	} 
   * @param mesh 
   */
  self.setMesh = function(mesh) {
    self.setPoints(mesh["points"]);
    self.setCells(mesh["cells"]);
    self.setBoundary(mesh["boundary"]);
    self.setBoundaryLabels(mesh["boundary_labels"]);
    self.setChanged(true);
  }
  
//  self.getMesh = function() {
//  	return {"points": mPoints,  "cells": mCells, "values": mValues, "boundary": mBoundaryGeometry};
//  }

  /**
   * Provides the array of points that conform the mesh and its boundary
   * @method setPoints
   * @param meshs The double[nPoints][2] where nPoints is the number of points in the mesh
   */
  self.setPoints = function(points) {
    mPoints = points;
    self.setChanged(true);
  }
  
  self.getPoints = function() {
  	return mPoints;
  }

  /**
   * Provides the field value for each point in the mesh
   * @method setValues
   * @param values The double[nPoints]array with the value for each mesh vertex
   */
  self.setFieldAtPoints = function(values) {
    mValues = values;
    mCellValues = null;
    if(mAutoscaleZ && mValues != null) mColor.setAutoscaleArray2(mValues); 
//    console.log ("Field at points changed");
    self.setChanged(true);
  }
  
  self.getFieldAtPoints = function() {
  	return mValues;
  }

  /**
   * Provides the geometry of the mesh
   * @method setCells
   * @param cells the int[nCells][nPoints] array with the points in each mesh.
   * For example,if the first mesh is a triangle joining points 0,3,7, one gets: meshs[0] = { 0, 3, 7 }; 
   */
  self.setCells = function(cells) {
    mMeshGeometry = cells;
    self.setChanged(true);
  }

  self.getCells = function() {
  	return mMeshGeometry;
  }

  /**
   * Provides the field value for each point in the mesh
   * @method setValues
   * @param values The double[nPoints]array with the value for each mesh vertex
   */
  self.setFieldAtCells = function(values) {
    mCellValues = values;
    mValues = null;
    if(mAutoscaleZ && mCellValues != null) mColor.setAutoscaleArray3(mCellValues); 
    self.setChanged(true);
  }
  
    self.getFieldAtCells = function() {
  	return mCellValues;
  }
  
  /**
   * Provides the geometry of the mesh
   * @method setVectorLength
   * @param length the double length for vectors in the field 
   */
  self.setVectorLength = function(length) {
    if (mVectorLength !== length) {
      mVectorLength = length;
      self.setChanged(true);
    }
  }

  self.getVectorLength = function() {
  	return mVectorLength;
  }

  
  /**
   * Provides the data for the boundary.
   * @method setBoundary
   * @param meshs The int[nOfSegments][nPointsInSegment] array with the mesh information, where:
   * <ul>
   * <li>First index = nOfSegments : number of segments in the boundary</li>
   * <li>Second index = nPointsInSegment : the points in this segment</li>
   * </ul>
   */
  self.setBoundary = function(boundary) {
    mBoundaryGeometry = boundary;
    self.setChanged(true);
  }

  self.getBoundary = function() {
  	return mBoundaryGeometry;
  }

  /**
   * Provides the label for each boundary segment
   * @method setBoundaryLabels
   * @param values The int[nOfSegments] array with the label for each boundary segment
   */
  self.setBoundaryLabels = function(labels) {
    mBoundaryLabels = labels;
    self.setChanged(true);
  }

  self.getBoundaryLabels = function() {
  	return mBoundaryLabels;
  }

  /**
   * The color to use for each boundary index 
   * There must be a color for each index 
   * If not enough colors are given, or colors is null, the element draw color is used
   * @method setBoundaryColors
   * @param colors
   */
  self.setBoundaryColors = function(colors) {
    mBoundaryColors = colors;
    self.setChanged(true);
  }
  
  self.getBoundaryColors = function() {
    return mBoundaryColors;
  }
  
  self.setBoundaryLineWidth = function(width) {
    mBoundaryStroke = Math.max(1, width);
    self.setChanged(true);
  }

  self.getBoundaryLineWidth = function() {
    return mBoundaryStroke;
  }

  self.setBoundaryStroke = function(stroke) {
    mBoundaryStroke = stroke;
    self.setChanged(true);
  }

  self.getBoundaryStroke = function() {
    return mBoundaryStroke;
  }

  self.setDrawBoundary = function(draw) {
    mBoundaryDraw = draw;
    self.setChanged(true);
  }

  self.getDrawBoundary = function() {
    return mBoundaryDraw;
  }

  // -------------------------------------
  // Public methods for coloring the meshs
  // -------------------------------------

  /**
   * Returns the ColorCoded for customization
   * @method getDrawer
   * @return
   */
  self.getColorCoded = function() {
    return mColor;
  }
  
  /**
   * Sets the autoscale flag and the floor and ceiling values for the colors.
   *
   * If autoscaling is true, then the min and max values of z are span the colors.
   *
   * If autoscaling is false, then floor and ceiling values limit the colors.
   * Values below min map to the first color; values above max map to the last color.
   *
   * @method setAutoscaleZ
   * @param isAutoscale
   * @param floor
   * @param ceil
   */
  self.setAutoscaleZ = function (isAutoscale, floor, ceil) {
    mAutoscaleZ = isAutoscale;
    if (mAutoscaleZ) {
    	if(mValues != null)	mColor.setAutoscaleArray2(mValues); 
    	else if(mCellValues != null) mColor.setAutoscaleArray3(mCellValues); 
    } else mColor.setScale(floor, ceil);
    self.setChanged(true);
  }
  
  /**
   * Gets the autoscale flag for z.
   *
   * @method isAutoscaleZ
   * @return boolean
   */
  self.isAutoscaleZ = function() {
    return mAutoscaleZ;
  }

  /**
   * Returns bounds for an element
   * @override
   * @method getBounds
   * @return Object{left,rigth,top,bottom}
   */
  self.getBounds = function(element) {
  	var xmin=0, xmax=0, ymin=0, ymax=0;
  	var points = self.getPoints();
  	if (points && points.length>0) {
      var len = points.length;
      xmax = xmin = points[0][0];
	  ymax = ymin = points[0][1];    	
	  for(var j=1; j<len; j++) {
			var x = points[j][0], y = points[j][1];
			if(x>xmax) xmax=x; if(y>ymax) ymax=y; if(x<xmin) xmin=x; if(y<ymin) ymin=y;
	  }
	}    
    var x = self.getX(), y = self.getY();
    var sx = self.getSizeX(), sy = self.getSizeY();
  	var mx = sx/2, my = sy/2;  	
  	var d = self.getRelativePositionOffset(sx,sy);
	var result =  {
		left: ((x+d[0])-mx)+xmin*sx,
		right: ((x+d[0])-mx)+xmax*sx,
		top: ((y+d[1])-my)+ymax*sy,
		bottom: ((y+d[1])-my)+ymin*sy
	}
	return result;
  };  

  self.registerProperties = function(controller) {
	EJSS_DRAWING2D.Mesh.registerProperties(self, controller);
  };
	
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([1,1]);
  self.setRelativePosition("SOUTH_WEST");

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Shape
 * @class Shape 
 * @constructor  
 */
EJSS_DRAWING2D.Meter = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class

      controller.registerProperty("Value", element.setValue, element.getValue);
      controller.registerProperty("Minimum", element.setMinimum, element.getMinimum);
      controller.registerProperty("Maximum", element.setMaximum, element.getMaximum);

      controller.registerProperty("AngleForMinimum", element.setAngleForMinimum, element.getAngleForMinimum);
      controller.registerProperty("AngleForMaximum", element.setAngleForMaximum, element.getAngleForMaximum);

      controller.registerProperty("NumberOfMarks",element.setNumberOfMarks,element.getNumberOfMarks);
      controller.registerProperty("BigMark",element.setBigMark,element.getBigMark);
      controller.registerProperty("MediumMark",element.setMediumMark,element.getMediumMark);
      controller.registerProperty("Digits",element.setDigits,element.getDigits);

      controller.registerProperty("Radius", element.setRadius);
      controller.registerProperty("CircleColor",element.setCircleColor);
      controller.registerProperty("CircleWidth",element.setCircleWidth);
      controller.registerProperty("FillColor",element.setFillColor);
      
      controller.registerProperty("Foreground",element.setForeground);
      controller.registerProperty("LineWidth",element.setLineWidth);

      controller.registerProperty("ArrowColor",element.setArrowColor);
      controller.registerProperty("ArrowWidth",element.setArrowWidth);

      controller.registerProperty("DrawLines",element.setDrawLines);
      controller.registerProperty("DrawText",element.setDrawText);

      controller.registerProperty("Brand",element.setBrand);
      controller.registerProperty("Units",element.setUnits);
      
      controller.registerProperty("MarksFont",element.setMarksFont);
      controller.registerProperty("BrandFont",element.setBrandFont);
      controller.registerProperty("UnitsFont",element.setUnitsFont);
    }

};

/**
 * Creates a 2D circular meter
 * @method shape
 */
EJSS_DRAWING2D.meter = function (mName) {
  var self = EJSS_DRAWING2D.group(mName);

  // Configuration variables
  var mValue = 0.0;
  var mMinimum =   0;
  var mMaximum = 100;

  var mNumberOfMarks = 21;
  var mBigMark = 4;
  var mMediumMark = 2;
  var mDigits = 0;

  var mAngleForMinimum = Math.PI/2 + 1.5*Math.PI/2;
  var mAngleForMaximum = Math.PI/2 - 1.5*Math.PI/2;

  // Implementation variables
  var mArrow, mCircle, mSegments, mTexts, mBrand, mUnits;
  var mColor = "Black", mDangerColor = "Red";
  var mForegroundCreated = false;

  var mElements = [];
  
  self.getElement = function(keyword) {
	  return mElements[keyword];
  };
  
  // ----------------------------------------------------
  // public functions
  // ----------------------------------------------------
  
  /**
   * Sets the value of the element. 
   * @method setValue
   * @param value
   */
  self.setValue = function(value) {
    if (mValue!=value) {
    	mValue = value;
    	self.adjustPosition();
    }
  }
  
  /**
   * @method getValue
   * @return current value
   */
  self.getValue = function() { 
    return mValue; 
  };    

  /**
   * Sets the minimum value of the element. 
   * @method setMinimum
   * @param value
   */
  self.setMinimum = function(value) {
    if (mMinimum!=value) {
    	mMinimum = value;
        self.setChanged(true);
    }
  }
  
  /**
   * @method getMinimum
   * @return current minimum
   */
  self.getMinimum = function() { 
    return mMinimum; 
  };    

  /**
   * Sets the minimum value of the element. 
   * @method setMaximum
   * @param value
   */
  self.setMaximum = function(value) {
	if (mMaximum!=value) {
		mMaximum = value;
        self.setChanged(true);
	}
  }
  
  /**
   * @method getMaximum
   * @return current maximum
   */
  self.getMaximum = function() { 
    return mMaximum; 
  };    

  /**
   * Sets the angle at which to display the minimum value 
   * @method setAngleForMinimum
   * @param value
   */
  self.setAngleForMinimum = function(value) {
    if (mAngleForMinimum!=value) {
    	mAngleForMinimum = value;
        self.setChanged(true);
    }
  }
  
  /**
   * @method getAngleForMinimum
   * @return current angle at which to display the minimum value
   */
  self.getAngleForMinimum = function() { 
    return mAngleForMinimum; 
  };    

  /**
   * Sets the angle at which to display the maximum value 
   * @method setAngleForMaximum
   * @param value
   */
  self.setAngleForMaximum = function(value) {
	if (mAngleForMaximum!=value) {
		mAngleForMaximum = value;
        self.setChanged(true);
	}
  }
  
  /**
   * @method getAngleForMaximum
   * @return current angle at which to display the maximum value 
   */
  self.getAngleForMaximum = function() { 
    return mAngleForMaximum; 
  };    

  /***
   * Sets the number of marks
   * @method setNumberOfMarks(value)
   * @visibility public
   * @param value int
   */ 
  self.setNumberOfMarks = function(value) {
	  if (value!=mNumberOfMarks) {
		  mNumberOfMarks = value;
        self.setChanged(true);
	  }
  };

  /***
   * Gets the number of marks
   * @method getNumberOfMarks
   * @visibility public
   * @return int
   */
  self.getNumberOfMarks = function() {
  	return mNumberOfMarks;
  };

  /***
   * Sets the interval for big marks
   * @method setBigMark(value)
   * @visibility public
   * @param value int 
   */ 
  self.setBigMark = function(value) {
	  if (value!=mBigMark) {
        mBigMark = value;
        self.setChanged(true);
	  }
  };

  /***
   * Gets the interval for big marks
   * @method getBigMark
   * @visibility public
   * @return int
   */
  self.getBigMark = function() {
  	return mBigMark;
  };

  /***
   * Sets the interval for medium marks
   * @method setMediumMark(value)
   * @visibility public
   * @param value int
   */ 
  self.setMediumMark = function(value) {
	  if (value!=mMediumMark) {
		  mMediumMark = value;
        self.setChanged(true);
	  }
  };
 
  /***
   * Gets the interval for medium marks
   * @method getMediumMark
   * @visibility public
   * @return value int
   */ 
  self.getMediumMark = function() {
  	return mMediumMark;
  };


  /***
   * Sets the number of digits for the marks
   * @method setDigits(value)
   * @visibility public
   * @param value int 
   */ 
  self.setDigits = function(value) {
	  if (value!=mDigits) {
		  mDigits = value;
        self.setChanged(true);
	  }
  };

  /***
   * Gets the number of digits for the marks
   * @method getPrecision
   * @visibility public
   * @return int
   */
  self.getDigits = function() {
  	return mDigits;
  };
  
  // ----------------------------------------------------
  // Properties overwritten
  // ----------------------------------------------------

  /**
   * Sets the radius of the element. 
   * @method setRadius
   * @param value
   */
  self.setRadius = function(value) {
	  self.setSizeX(2*value);
	  self.setSizeY(2*value);
	  if (mCircle) mCircle.setChanged(true);
	  if (mArrow) mArrow.setChanged(true);
	  mSegments.setToEach(function(element) { element.setChanged(true); });
	  mTexts.setToEach(function(element) { element.setChanged(true); });
	  if (mBrand) mBrand.setChanged(true);
	  if (mUnits) mUnits.setChanged(true);
  };

  self.setCircleColor = function(value) { mCircle.getStyle().setLineColor(value); };
  
  self.setCircleWidth = function(value) { mCircle.getStyle().setLineWidth(value); };

  self.setFillColor = function(value) { mCircle.getStyle().setFillColor(value); };

  self.setArrowColor = function(value) {
	  if (value=="Red") mDangerColor = "Yellow";
	  else mDangerColor = "Red";
	  mColor = value;
	  mArrow.getStyle().setLineColor(value);
	  mArrow.getStyle().setFillColor(value);
  };

  self.setArrowWidth = function(value) { mArrow.getStyle().setLineWidth(value); }

  self.setForeground = function(value) { 
	  mSegments.setToEach(function(element) { element.getStyle().setLineColor(value); });
	  mTexts.setToEach(function(element) { element.getFont().setFillColor(value); });
	  mBrand.getFont().setFillColor(value);
	  mUnits.getFont().setFillColor(value);
}
  
  self.setLineWidth = function(value) { 
	  mSegments.setToEach(function(element) { element.getStyle().setLineWidth(value); });
  }

  self.setDrawLines = function(value) { 
	  mSegments.setToEach(function(element,value) { element.setVisible(value); },value);
  }
  
  self.setDrawText = function(value) { 
	  mTexts.setToEach(function(element,value) { element.setVisible(value); },value);
  }

  self.setFont = function(value) { 
	  mTexts.setToEach(function(element) { element.getFont().setFont(value); });
	  mBrand.getFont().setFont(value);
	  mUnits.getFont().setFont(value);
  }

  self.setFont = function(value) { 
	  mTexts.setToEach(function(element) { element.getFont().setFont(value); });
	  mBrand.getFont().setFont(value);
	  mUnits.getFont().setFont(value);
  }
  self.setMarksFont = function(value) { 
	  mTexts.setToEach(function(element) { element.getFont().setFont(value); });
  }
  self.setBrandFont = function(value) { mBrand.getFont().setFont(value); }
  self.setUnitsFont = function(value) { mUnits.getFont().setFont(value); }

  self.setBrand = function(value) { mBrand.setText(value); }
  self.setUnits= function(value) { mUnits.setText(value); }

  // ----------------------------------------------------
  // Properties and copies
  // ----------------------------------------------------

  /**
   * Extended registerProperties method. To be used by promoteToControlElement
   * @method registerProperties
   */
  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.Meter.registerProperties(self,controller);
  };

  // ----------------------------------------------------
  // private or protected functions
  // ----------------------------------------------------

  self.superSetParent = self.setParent;
  
  self.setParent = function(parent, sibling) {
	self.superSetParent(parent,sibling);
	if (!mForegroundCreated) {
	  self.createBasics();
	  self.createForeground();
	  mForegroundCreated = true;
    }
	self.addParticularChildren();
	self.adjustForeground();
  }

  self.dataCollected = function() {
	  if (self.isChanged()) {
		  self.adjustForeground();
		  self.setChanged(false);
	  }
  }; 
  
  self.addParticularChildren = function() {
		if (mCircle) mCircle.setParent(self);
		if (mSegments) mSegments.setToEach(function(element) { element.setParent(self); });
		if (mTexts) mTexts.setToEach(function(element) { element.setParent(self); });
		if (mBrand) mBrand.setParent(self);
		if (mUnits) mUnits.setParent(self);
		if (mArrow) mArrow.setParent(self);
  };

  self.createBasics = function() {
    mCircle = EJSS_DRAWING2D.shape(mName+".circle");
    mCircle.setSize([1,1]);
    mCircle.setShapeType("ELLIPSE");
    mCircle.getStyle().setLineWidth(3);
    mCircle.getStyle().setLineColor("Grey");
    mCircle.getStyle().setFillColor("White");
    mElements['circle'] = mCircle;

    mArrow = EJSS_DRAWING2D.arrow(mName+".arrow");
    mArrow.setSize([0.45,0]);
    mArrow.getStyle().setLineWidth(2);
    mArrow.setMarkStart("WEDGE");
    mArrow.setMarkEnd("POINTED");
    mElements['arrow'] = mArrow;
  };
  
  self.createForeground = function() {
    mSegments = EJSS_DRAWING2D.segmentSet(mName+".segments");
    mSegments.setToEach(function(element) { element.setRelativePosition("SOUTH_WEST"); });
    mElements['segments'] = mSegments;

    mTexts = EJSS_DRAWING2D.textSet(mName+".texts");
    mTexts.setToEach(function(element) { element.getFont().setFont("normal bold 12px \"Courier New\", Courier, monospace"); });
    mElements['texts'] = mTexts;

    mBrand = EJSS_DRAWING2D.text(mName+".brand");
    mBrand.setPosition([0,0.20]);
    mBrand.setText("Meter");
    mBrand.getFont().setFont("normal bold 12px \"Courier New\", Courier, monospace");
    mElements['brand'] = mBrand;

    mUnits = EJSS_DRAWING2D.text(mName+".Units");
    mUnits.setPosition([0,-0.25]);
    mUnits.setText("%");
    mUnits.getFont().setFont("normal bold 12px \"Courier New\", Courier, monospace");
    mElements['units'] = mUnits;
}
  
  self.interpolateAngle= function(value) {
	 return mAngleForMinimum + (value-mMinimum)/(mMaximum-mMinimum)*(mAngleForMaximum-mAngleForMinimum);
  };

  self.valueFromAngle= function(angle) {
	  angle = Math.max(Math.min(angle,mAngleForMinimum),mAngleForMaximum);
	  mValue = mMaximum - (angle-mAngleForMaximum)/(mAngleForMinimum-mAngleForMaximum)*(mMaximum-mMinimum);
      return angle;
  };

  self.adjustPosition = function() {
	  var actualValue = mValue;
	  if (mValue<mMinimum) {
		  actualValue = mMinimum;
		  mArrow.getStyle().setLineColor(mDangerColor);
		  mArrow.getStyle().setFillColor(mDangerColor);
	  }
	  else if (mValue>mMaximum) {
		  actualValue = mMaximum;
		  mArrow.getStyle().setLineColor(mDangerColor);
		  mArrow.getStyle().setFillColor(mDangerColor);
	  }
	  else {
		  actualValue = mValue;
		  mArrow.getStyle().setLineColor(mColor);
		  mArrow.getStyle().setFillColor(mColor);
	  }
	  var angle = mAngleForMinimum + (actualValue-mMinimum)/(mMaximum-mMinimum)*(mAngleForMaximum-mAngleForMinimum);
	  mArrow.setTransformation(angle);
  }

  self.foregroundIsInside = function() { return true; };
  
  self.adjustForeground = function() {
	  var MARK = 1.0/12;
	  
	  mSegments.setNumberOfElements(mNumberOfMarks);
	  var dAngle = (mAngleForMaximum-mAngleForMinimum)/(mNumberOfMarks-1);
	  var angle = mAngleForMinimum;
	  var mediumMarksCounter = 0;
	  var bigMarksCounter = 0;
	  var direction =  self.foregroundIsInside() ? -MARK : MARK;
	  
	  for (var i=0; i<mNumberOfMarks; i++) {
	    var segment = mSegments.getElement(i);
	    var xPos = Math.cos(angle)/2;
	    var yPos = Math.sin(angle)/2;
	    segment.setPosition([xPos,yPos]);
	    if (i%mBigMark==0) {
	      bigMarksCounter++;
	      segment.setSize([2*direction*xPos,2*direction*yPos]);
	    }
	    else if (i%mMediumMark==0) {
	      mediumMarksCounter++;
	      segment.setSize([1.5*direction*xPos,1.5*direction*yPos]);
	    }
	    else {
	      segment.setSize([direction*xPos,direction*yPos]);
	    }
	    angle += dAngle;
	  }
	  
	  mTexts.setNumberOfElements(bigMarksCounter); //+mediumMarksCounter);
	  var text_counter=0;
      var fontSize =  mTexts.getElement(0).getFont().getFontSize();
      direction =  self.foregroundIsInside() ? 1.7 : 2.5;
      if (fontSize>30) direction *= 1.5;
      else if (fontSize>20) direction *= 1.3;
      else if (fontSize>15) direction *= 1.1;
	  for (i=0; i<mNumberOfMarks; i++) {
	    var segment = mSegments.getElement(i);
	    if (i%mBigMark==0) { // || i%mediumMark==0) {
	      var text = mTexts.getElement(text_counter);
	      var pos = segment.getPosition();
	      var size = segment.getSize();
	      text.setPosition([pos[0]+direction*size[0],pos[1]+direction*size[1]]);
	      var mark_value = ((mNumberOfMarks-1-i)*mMinimum + i*mMaximum)/(mNumberOfMarks-1);
	      text.setText(""+mark_value.toFixed(mDigits));
	      text_counter++;
	    }
	  }
	  self.adjustPosition();  
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  return self;
};


/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Shape
 * @class Shape 
 * @constructor  
 */
EJSS_DRAWING2D.Pipe = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class

      controller.registerProperty("Points",element.setPoints,element.getPoints);
      controller.registerProperty("PipeWidth",element.setPipeWidth,element.getPipeWidth);
    },

    /**
     * static copyTo method, to be used by sets
     */
    copyTo : function(source, dest) {
      EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy

      dest.setPoints(source.getPoints());
      dest.setPipeWidth(source.getPipeWidth());
    }

};

/**
 * Creates a 2D shape
 * @method shape
 */
EJSS_DRAWING2D.pipe = function (name) {
  var self = EJSS_DRAWING2D.element(name);

  var mPoints = [[0.1,0.2],[0.1,0.3]];
  var mPipeWidth = 5;

  self.getClass = function() {
  	return "ElementPipe";
  };

  // ----------------------------------------------------
  // public functions
  // ----------------------------------------------------

  self.setPoints = function(points) {
      mPoints = points;
      self.setChanged(true);
  };

  self.getPoints = function() {
  	return mPoints;
  };

  self.setPipeWidth = function(width) { 
    mPipeWidth = width; 
    self.setChanged(true);
  };
    
  self.getPipeWidth = function() { 
    return mPipeWidth; 
  };
  
  // ----------------------------------------------------
  // Properties and copies
  // ----------------------------------------------------

  /**
   * Extended registerProperties method. To be used by promoteToControlElement
   * @method registerProperties
   */
  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.Pipe.registerProperties(self,controller);
  };

  /**
   * Copies itself to another element
   * Extended copyTo method. To be used by Sets
   * @method copyTo
   */
  self.copyTo = function(element) {
    EJSS_DRAWING2D.Pipe.copyTo(self,element);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([1,1]);
  self.setRelativePosition("SOUTH_WEST");
  self.getStyle().setLineWidth(1);
  self.getStyle().setFillColor("Blue");
  return self;
};


/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia and Félix J. García 
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*/

/**
 * Framework for 2D drawing.
 * @module 2Ddrawing
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * A PlottingPanel is a 2D drawing panel with added decoration. The decoration includes axes, gutters and titles.
 * @class EJSS_DRAWING2D.PlottingPanel
 * @parent EJSS_DRAWING2D.DrawingPanel
 * @constructor
 */

EJSS_DRAWING2D.PlottingPanel = {

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.DrawingPanel.registerProperties(element, controller);
		// super class

		// title (text and font)
     /*** 
	  * The title at the top center of the panel
	  * @property Title
	  * @type String
	  * @default "Plot"
	  */ 
		controller.registerProperty("Title", element.getTitle().setText);
		controller.registerProperty("TitleFont", element.getTitle().getFont().setFont);
		controller.registerProperty("TitleColor", element.getTitle().getFont().setFillColor);
		controller.registerProperty("TitleMargin", element.getTitle().setMarginY);

		controller.registerProperty("TitleX", element.getTitleX().setText);
		controller.registerProperty("TitleXFont", element.getTitleX().getFont().setFont);
		controller.registerProperty("TitleXColor", element.getTitleX().getFont().setFillColor);
		controller.registerProperty("TitleXMargin", element.getTitleX().setMarginY);

		controller.registerProperty("TitleY", element.getTitleY().setText);
		controller.registerProperty("TitleYFont", element.getTitleY().getFont().setFont);
		controller.registerProperty("TitleYColor", element.getTitleY().getFont().setFillColor);
		controller.registerProperty("TitleYMargin", element.getTitleY().setMarginX);
 
		// Grid
		controller.registerProperty("GridXShow", element.getGrid().setShowX, element.getGrid().getShowX);
		controller.registerProperty("GridXTicks", element.getGrid().setTicksX, element.getGrid().getTicksX);
		controller.registerProperty("GridXStep", element.getGrid().setStepX, element.getGrid().getStepX);
		controller.registerProperty("GridXAutoStepMin", element.getGrid().setAutoStepXMin, element.getGrid().getAutoStepXMin);
		controller.registerProperty("GridXAutoTicksRange", element.getGrid().setAutoTicksXRange, element.getGrid().getAutoTicksXRange);
		controller.registerProperty("GridXScale", element.getGrid().setScaleX, element.getGrid().getScaleX);
		controller.registerProperty("GridXScalePrecision", element.getGrid().setScalePrecisionX, element.getGrid().getScalePrecisionX);
		controller.registerProperty("GridXFixedTick", element.getGrid().setFixedTickX, element.getGrid().getFixedTickX);				
		controller.registerProperty("GridXAutoTicks", element.getGrid().setAutoTicksX, element.getGrid().getAutoTicksX);
		controller.registerProperty("GridXLineColor", element.getGrid().setLineColorX);
		controller.registerProperty("GridXLineWidth", element.getGrid().setLineWidthX);
		controller.registerProperty("GridXShapeRendering", element.getGrid().setShapeRenderingX);

		controller.registerProperty("GridYShow", element.getGrid().setShowY, element.getGrid().getShowY);
		controller.registerProperty("GridYTicks", element.getGrid().setTicksY, element.getGrid().getTicksY);
		controller.registerProperty("GridYStep", element.getGrid().setStepY, element.getGrid().getStepY);
		controller.registerProperty("GridYAutoStepMin", element.getGrid().setAutoStepYMin, element.getGrid().getAutoStepYMin);
		controller.registerProperty("GridYAutoTicksRange", element.getGrid().setAutoTicksYRange, element.getGrid().getAutoTicksYRange);
		controller.registerProperty("GridYScale", element.getGrid().setScaleY, element.getGrid().getScaleY);
		controller.registerProperty("GridYScalePrecision", element.getGrid().setScalePrecisionY, element.getGrid().getScalePrecisionY);		
		controller.registerProperty("GridYFixedTick", element.getGrid().setFixedTickY, element.getGrid().getFixedTickY);
		controller.registerProperty("GridYAutoTicks", element.getGrid().setAutoTicksY, element.getGrid().getAutoTicksY);
		controller.registerProperty("GridYLineColor", element.getGrid().setLineColorY);
		controller.registerProperty("GridYLineWidth", element.getGrid().setLineWidthY);
		controller.registerProperty("GridYShapeRendering", element.getGrid().setShapeRenderingY);

		// axis
 		controller.registerProperty("AxisXShow", element.getAxisX().setShow);
		controller.registerProperty("AxisXLineColor", element.getAxisX().getStyle().setLineColor);
		controller.registerProperty("AxisXLineWidth", element.getAxisX().getStyle().setLineWidth);
		controller.registerProperty("AxisXShapeRendering", element.getAxisX().getStyle().setShapeRendering);		
		controller.registerProperty("AxisXFixedTick", element.getAxisX().setFixedTick, element.getAxisX().getFixedTick);
		controller.registerProperty("AxisXAutoTicks", element.getAxisX().setAutoTicks, element.getAxisX().getAutoTicks);
		controller.registerProperty("AxisXTicks", element.getAxisX().setTicks, element.getAxisX().getTicks);
		controller.registerProperty("AxisXStep", element.getAxisX().setStep, element.getAxisX().getStep);
		controller.registerProperty("AxisXTickStep", element.getAxisX().setTickStep, element.getAxisX().getTickStep);
		controller.registerProperty("AxisXAutoStepMin", element.getAxisX().setAutoStepMin, element.getAxisX().getAutoStepMin);
		controller.registerProperty("AxisXAutoTicksRange", element.getAxisX().setAutoTicksRange, element.getAxisX().getAutoTicksRange);
		controller.registerProperty("AxisXScale", element.getAxisX().setScale, element.getAxisX().getScale);
		controller.registerProperty("AxisXScalePrecision", element.getAxisX().setScalePrecision, element.getAxisX().getScalePrecision);	    
	    controller.registerProperty("AxisXFont", element.getAxisX().getFont().setFont);
		controller.registerProperty("AxisXFontColor", element.getAxisX().getFont().setFillColor);

 		controller.registerProperty("AxisYShow", element.getAxisY().setShow);
		controller.registerProperty("AxisYLineColor", element.getAxisY().getStyle().setLineColor);
		controller.registerProperty("AxisYLineWidth", element.getAxisY().getStyle().setLineWidth);
		controller.registerProperty("AxisYShapeRendering", element.getAxisY().getStyle().setShapeRendering);
		controller.registerProperty("AxisYFixedTick", element.getAxisY().setFixedTick, element.getAxisY().getFixedTick);
		controller.registerProperty("AxisYAutoTicks", element.getAxisY().setAutoTicks, element.getAxisY().getAutoTicks);
		controller.registerProperty("AxisYTicks", element.getAxisY().setTicks, element.getAxisY().getTicks);
		controller.registerProperty("AxisYStep", element.getAxisY().setStep, element.getAxisY().getStep);
		controller.registerProperty("AxisYTickStep", element.getAxisY().setTickStep, element.getAxisY().getTickStep);
		controller.registerProperty("AxisYAutoStepMin", element.getAxisY().setAutoStepMin, element.getAxisY().getAutoStepMin);
		controller.registerProperty("AxisYAutoTicksRange", element.getAxisY().setAutoTicksRange, element.getAxisY().getAutoTicksRange);
		controller.registerProperty("AxisYScale", element.getAxisY().setScale, element.getAxisY().getScale);
		controller.registerProperty("AxisYScalePrecision", element.getAxisY().setScalePrecision, element.getAxisY().getScalePrecision);
		controller.registerProperty("AxisYFont", element.getAxisY().getFont().setFont);
		controller.registerProperty("AxisYFontColor", element.getAxisY().getFont().setFillColor);

		// Axis and Grid
		controller.registerProperty("XFixedTick", function(v) {
			element.getAxisX().setFixedTick(v);
			element.getGrid().setFixedTickX(v);
		});
		controller.registerProperty("YFixedTick", function(v) {
			element.getAxisY().setFixedTick(v);
			element.getGrid().setFixedTickY(v);
		});

		//controller.registerProperty("GridXTicks", element.getGrid().setTicksX, element.getGrid().getTicksX);
		//controller.registerProperty("AxisXTicks", element.getAxisX().setTicks, element.getAxisX().getTicks);
		controller.registerProperty("XTicks", function(v) {
			element.getAxisX().setTicks(v);
			element.getGrid().setTicksX(v);
		});
		controller.registerProperty("YTicks", function(v) {
			element.getAxisY().setTicks(v);
			element.getGrid().setTicksY(v);
		});

		//controller.registerProperty("GridXStep", element.getGrid().setStepX, element.getGrid().getStepX);
		//controller.registerProperty("AxisXStep", element.getAxisX().setStep, element.getAxisX().getStep);
		controller.registerProperty("XStep", function(v) {
			element.getAxisX().setStep(v);
			element.getGrid().setStepX(v);
		});
		controller.registerProperty("YStep", function(v) {
			element.getAxisY().setStep(v);
			element.getGrid().setStepY(v);
		});

		controller.registerProperty("XTickStep", function(v) {
			element.getAxisX().setTickStep(v);
			element.getGrid().setTickStepX(v);
		});
		controller.registerProperty("YTickStep", function(v) {
			element.getAxisY().setTickStep(v);
			element.getGrid().setTickStepY(v);
		});

		//controller.registerProperty("GridXAutoStepMin", element.getGrid().setAutoStepXMin, element.getGrid().getAutoStepXMin);
		//controller.registerProperty("AxisXAutoStepMin", element.getAxisX().setAutoStepMin, element.getAxisX().getAutoStepMin);
		controller.registerProperty("XAutoStepMin", function(v) {
			element.getAxisX().setAutoStepMin(v);
			element.getGrid().setAutoStepXMin(v);
		});
		controller.registerProperty("YAutoStepMin", function(v) {
			element.getAxisY().setAutoStepMin(v);
			element.getGrid().setAutoStepYMin(v);
		});

		//controller.registerProperty("GridXAutoTicksRange", element.getGrid().setAutoTicksXRange, element.getGrid().getAutoTicksXRange);
		//controller.registerProperty("AxisXAutoTicksRange", element.getAxisX().setAutoTicksRange, element.getAxisX().getAutoTicksRange);
		controller.registerProperty("XAutoTicksRange", function(v) {
			element.getAxisX().setAutoTicksRange(v);
			element.getGrid().setAutoTicksXRange(v);
		});
		controller.registerProperty("YAutoTicksRange", function(v) {
			element.getAxisY().setAutoTicksRange(v);
			element.getGrid().setAutoTicksyRange(v);
		});

		//controller.registerProperty("GridXScale", element.getGrid().setScaleX, element.getGrid().getScaleX);
		//controller.registerProperty("AxisXScale", element.getAxisX().setScale, element.getAxisX().getScale);
		controller.registerProperty("XScale", function(v) {
			element.getAxisX().setScale(v);
			element.getGrid().setScaleX(v);
		});
		controller.registerProperty("YScale", function(v) {
			element.getAxisY().setScale(v);
			element.getGrid().setScaleY(v);
		});

		//controller.registerProperty("GridXScalePrecision", element.getGrid().setScalePrecisionX, element.getGrid().getScalePrecisionX);
		//controller.registerProperty("AxisXScalePrecision", element.getAxisX().setScalePrecision, element.getAxisX().getScalePrecision);
		controller.registerProperty("XScalePrecision", function(v) {
			element.getAxisX().setScalePrecision(v);
			element.getGrid().setScalePrecisionX(v);
		});
		controller.registerProperty("YScalePrecision", function(v) {
			element.getAxisY().setScalePrecision(v);
			element.getGrid().setScalePrecisionY(v);
		});

		//controller.registerProperty("GridXAutoTicks", element.getGrid().setAutoTicksX, element.getGrid().getAutoTicksX);
		//controller.registerProperty("AxisXAutoTicks", element.getAxisX().setAutoTicks, element.getAxisX().getAutoTicks);
		controller.registerProperty("XAutoTicks", function(v) {
			element.getAxisX().setAutoTicks(v);
			element.getGrid().setAutoTicksX(v);
		});
		controller.registerProperty("YAutoTicks", function(v) {
			element.getAxisY().setAutoTicks(v);
			element.getGrid().setAutoTicksY(v);
		});
		
	}
};

EJSS_DRAWING2D.plottingPanel = function(mName,mGraphicsMode) {
	var self = EJSS_DRAWING2D.drawingPanel(mName,mGraphicsMode);
	
	/**
	 * @method getGrid
	 * @return grid element
	 */
	self.getGrid = function() {
		return mGrid;
	};
	
	/***
	 * @method getAxisX()
	 * @return the axis element for the X dimension
     * @visibility public
	 * @see EJSS_DRAWING2D.Axis
	 */
	self.getAxisX = function() {
		return mAxisX;
	};
	
	/***
	 * @method getAxisY()
	 * @return the axis element for the Y dimension
     * @visibility public
	 * @see EJSS_DRAWING2D.Axis
	 */
	self.getAxisY = function() {
		return mAxisY;
	};
	
	/**
	 * @method getTitleX
	 * @return axis X title element
	 */
	self.getTitleX = function() {
		return mTitleX;
	};
	
	/**
	 * @method getTitleY
	 * @return axis Y title element
	 */
	self.getTitleY = function() {
		return mTitleY;
	};
	
	/**
	 * @method getTitle
	 * @return plot title element
	 */
	self.getTitle = function() {
		return mTitle;
	};
	
	/**
	 * @method getFixedTickX
	 * @return x fixed tick
	 */
	self.getFixedTickX = function() {
		return getGrid().getFixedTickX();
	};

	/**
	 * @method getFixedTickY
	 * @return y fixed tick
	 */
	self.getFixedTickY = function() {
		return getGrid().getFixedTickY();
	};

	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.PlottingPanel.registerProperties(self, controller);
	};
	
	var mBottomGroup = EJSS_DRAWING2D.group(mName + ".bottomgroup");
	mBottomGroup.setRelativePosition("SOUTH_WEST");
	mBottomGroup.panelChangeListener = function(event) {
		if (event == "bounds") {
			var xmin = self.getRealWorldXMin();
			var xmax = self.getRealWorldXMax();
			var ymin = self.getRealWorldYMin();
			var ymax = self.getRealWorldYMax();
			var sx = Math.abs(xmax - xmin); 
			var sy = Math.abs(ymax - ymin);
			mBottomGroup.setSize([sx, sy]);
			mBottomGroup.setPosition([xmin, ymin]);
			return true;
		}
	};
	mBottomGroup.panelChangeListener("bounds");

	var mGrid = EJSS_DRAWING2D.grid(mName + ".grid");
	mGrid.setGroup(mBottomGroup);
	mGrid.setRelativePosition("SOUTH_WEST");
	mGrid.setLineColorX("lightgray");
	mGrid.setShapeRenderingX("RENDER_CRISPEDGES");
	mGrid.setLineColorY("lightgray");
	mGrid.setShapeRenderingY("RENDER_CRISPEDGES");
	mGrid.setSize([1,1]);
	mGrid.setPosition([0, 0]);
	mGrid.panelChangeListener = function(event) {
		if (event == "bounds") {
			var xmin = self.getRealWorldXMin();
			var xmax = self.getRealWorldXMax();
			mGrid.setScaleX([xmin, xmax]);
			mGrid.setTicksXMode(self.getTypeScaleX());
			var ymin = self.getRealWorldYMin();
			var ymax = self.getRealWorldYMax();
			mGrid.setScaleY([ymin, ymax]);
			mGrid.setTicksYMode(self.getTypeScaleY());		
			
			// mGrid.setInvertedScaleY(self.getInvertedScaleY());
			return true;
		}
	};
	var mTopGroup = EJSS_DRAWING2D.group(mName + ".topgroup");
	mTopGroup.setRelativePosition("SOUTH_WEST");
	mTopGroup.panelChangeListener = function(event) {
		if (event == "bounds") {
			var xmin = self.getRealWorldXMin();
			var xmax = self.getRealWorldXMax();
			var ymin = self.getRealWorldYMin();
			var ymax = self.getRealWorldYMax();
			var sx = Math.abs(xmax - xmin); 
			var sy = Math.abs(ymax - ymin);
			mTopGroup.setSize([sx, sy]);
			mTopGroup.setPosition([xmin, ymin]);
			return true;
		}
	};
	mTopGroup.panelChangeListener("bounds");

	var mAxisX = EJSS_DRAWING2D.axis(mName + ".axisX");
	mAxisX.setGroup(mTopGroup);
	mAxisX.setRelativePosition("WEST");
	mAxisX.getStyle().setLineColor("black");
	mAxisX.getStyle().setShapeRendering("RENDER_CRISPEDGES");
	mAxisX.getFont().setFontSize(10);
	mAxisX.setSize([1, 1]);
	mAxisX.setPosition([0, 0]);
	mAxisX.setOrient("AXIS_HORIZONTAL");
	mAxisX.panelChangeListener = function(event) {
		if (event == "bounds") {
			var xmin = self.getRealWorldXMin();
			var xmax = self.getRealWorldXMax();
			mAxisX.setScale([xmin, xmax]);
			mAxisX.setTicksMode(self.getTypeScaleX());
			
			if(self.getInvertedScaleY()) {
				mAxisX.setTextPosition("TICKS_DOWN");
				mTitleX.setMarginY(-30);
				mTitle.setMarginY(20);
			} else {
				mAxisX.setTextPosition("TICKS_UP");
				mTitleX.setMarginY(30);
				mTitle.setMarginY(-20);
			}
			return true;
		}
	};
	mAxisX.setTicksMode(self.getTypeScaleX());
	
	
	var mAxisY = EJSS_DRAWING2D.axis(mName + ".axisY");
	mAxisY.setGroup(mTopGroup);
	mAxisY.setRelativePosition("SOUTH");
	mAxisY.getStyle().setLineColor("black");
	mAxisY.getStyle().setShapeRendering("RENDER_CRISPEDGES");
	mAxisY.getFont().setFontSize(10);
	mAxisY.setSize([1, 1]);
	mAxisY.setPosition([0, 0]);
	mAxisY.setOrient("AXIS_VERTICAL");
	mAxisY.panelChangeListener = function(event) {
		if (event == "bounds") {
			var ymin = self.getRealWorldYMin();
			var ymax = self.getRealWorldYMax();
			mAxisY.setScale([ymin, ymax]);
			mAxisY.setTicksMode(self.getTypeScaleY());
			
			mAxisY.setInvertedScaleY(self.getInvertedScaleY());			
			return true;
		}
	};
	mAxisY.setTicksMode(self.getTypeScaleY());
	
	var mTitleX = EJSS_DRAWING2D.text(mName + ".titleX");
	mTitleX.setGroup(mTopGroup);
	mTitleX.setRelativePosition("CENTER");
	mTitleX.getFont().setFontSize(12);
	mTitleX.setPosition([0.5, 0]);
	mTitleX.setMarginY(30);
	mTitleX.setText("x");

	var mTitleY = EJSS_DRAWING2D.text(mName + ".titleY");
	mTitleY.setGroup(mTopGroup);
	mTitleY.setRelativePosition("CENTER");
	mTitleY.getFont().setFontSize(12);
	mTitleY.setWritingMode("MODE_DOWNTOP");
	mTitleY.setPosition([0, 0.5]);
	mTitleY.setMarginX(-40);
	mTitleY.setText("y");

	var mTitle = EJSS_DRAWING2D.text(mName + ".title");
	mTitle.setGroup(mTopGroup);
	mTitle.setRelativePosition("CENTER");
	mTitle.getFont().setFontSize(12);
	mTitle.setPosition([0.5, 1.0]);
	mTitle.setMarginY(-20);
	mTitle.setText("Plot");

	self.getStyle().setLineColor('black');
	self.getStyle().setFillColor('white');
	self.getStyle().setShapeRendering("RENDER_CRISPEDGES");

	self.setGutters([50, 50, 50, 50]);
	self.getGuttersStyle().setLineColor('black');
	self.getGuttersStyle().setFillColor('rgb(211,216,255)');
	self.getGuttersStyle().setShapeRendering("RENDER_CRISPEDGES");

	self.addDecoration(mBottomGroup);
	self.addDecoration(mGrid);
	self.addDecoration(mTopGroup, -1, true);
	self.addDecoration(mAxisX, -1, true);
	self.addDecoration(mAxisY, -1, true);
	self.addDecoration(mTitleX, -1, true);
	self.addDecoration(mTitleY, -1, true);
	self.addDecoration(mTitle, -1, true);

	return self;
};
/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Polygon
 * @class Polygon 
 * @constructor  
 */
EJSS_DRAWING2D.Polygon = {
  	NO_CONNECTION : 0, 	// The next point will not be connected to the previous one
 	LINE_CONNECTION : 1,	// The next point will be connected to the previous one by a segment

	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy
  	
		dest.setPoints(source.getPoints());
  	},


	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class

		controller.registerProperty("Points", element.setPoints, element.getPoints);
		controller.registerProperty("PointsX", element.setPointsX);
		controller.registerProperty("PointsY", element.setPointsY);
		controller.registerProperty("LastPoint", element.addPoint, element.getLastPoint);		
	}
			
};

/**
 * Creates a 2D Polygon
 * @method polygon
 */
EJSS_DRAWING2D.polygon = function(name) {
	var self = EJSS_DRAWING2D.element(name);

  	var mPointList = []; 	// The current list of points  	
	var mTempX = null;
	var mTempY = null; 
	 
	self.getClass = function() {
		return "ElementPolygon";
	}
		  
  /**
   * Adds a new point to the Polygon.
   * @method addPoint
   * @param x double The X coordinate of the point 
   * 		or point double[] The double[2] array with the coordinates of the point.
   * @param y double The Y coordinate of the point.
   * @param style connected or not.
   */
  self.addPoint = function (x, y, style) {
    if(x instanceof Array)    	
    	addPoint(x[0], x[1], x[2]);
    else 
    	addPoint (x, y, style);
    self.setChanged(true);
  }

  /**
   * Adds an array of points to the Polygon.
   * @method addPoints
   * @param xInput double The double[] array with the X coordinates of the points.
   * 	or  point double[][] The double[nPoints][2] array with the coordinates of the points.
   * @param yInput double The double[] array with the Y coordinates of the points.
   */
  self.addPoints  = function(x, y) {
    if (x==undefined || x==null || x[0]==undefined) return;
  	if(x[0] instanceof Array) {    
    	for (var i=0,n=x.length; i<n; i++) addPoint (x[i][0],x[i][1],x[i][2]);  		
  	}
    else {
    	var n = Math.min(x.length,y.length);
    	for (var i=0; i<n; i++) addPoint (x[i],y[i]);
    }
    self.setChanged(true);
  }

  /**
   * Sets an array of points to the Polygon.
   * @method setPoints
   * @param x double The double[] array with the X coordinates of the points.
   * 	or  point double[][] The double[nPoints][2] array with the coordinates of the points.
   * @param y double The double[] array with the Y coordinates of the points.
   */
  self.setPoints  = function(x, y) {
  	self.clear();
  	self.addPoints(x,y);
  	self.setChanged(true);
  }


  /**
   * Adds the X coordinates of an array of points to the Polygon.
   * @method addXPoints
   * @param x double The double[] array with the X coordinates of the points.
   */
  self.setPointsX  = function(x) {
    if (mTempY==null) mTempX = x;
    else {
      self.setPoints(x,mTempY);
      mTempY = null;
    }
  }

  /**
   * Adds the Y coordinates of an array of points to the Polygon.
   * @method addYPoints
   * @param y double The double[] array with the Y coordinates of the points.
   */
  self.setPointsY  = function(y) {
    if (mTempX==null) mTempY = y;
    else {
      self.setPoints(mTempX,y);
      mTempX = null;
    }
  }

  /**
   * Returns all points.
   * @method getPoints
   * @return points
   */
  self.getPoints = function () {
    return mPointList; 
  }

  /**
   * Gets last point.
   * @method getLastPoint
   * @return point
   */
  self.getLastPoint = function () {
  	if(mPointList.length > 0) 
    	return mPointList[mPointList.length-1];
    else return [];
  }

  /**
   * Clears all points from all segments of the Polygon.
   * @method clear
   */
  self.clear = function() {
    mPointList = [];
  	self.setChanged(true);
  }
  
  /**
   * Returns bounds for an element
   * @override
   * @method getBounds
   * @return Object{left,rigth,top,bottom}
   */
  self.getBounds = function(element) {
  	var xmin, xmax, ymin, ymax;
  	var points = self.getPoints();
    var len = points.length;
    if(len == 0) {
    	xmin = xmax = ymin = ymax = 0;
    }             
    else {
		xmax = xmin = points[0][0];
		ymax = ymin = points[0][1];    	
		for(var j=1; j<len; j++) {
			var x = points[j][0], y = points[j][1];
			if(x>xmax) xmax=x; if(y>ymax) ymax=y; if(x<xmin) xmin=x; if(y<ymin) ymin=y;
		}
	}    
    var x = self.getX(), y = self.getY();
    var sx = self.getSizeX(), sy = self.getSizeY();
  	var mx = sy/2, my = sy/2;  	
  	var d = self.getRelativePositionOffset(sx,sy);
	return {
		left: ((x+d[0])-mx)+xmin*sx,
		right: ((x+d[0])-mx)+xmax*sx,
		top: ((y+d[1])-my)+ymax*sy, 
		bottom: ((y+d[1])-my)+ymin*sy
	}
  };  
  
  function addPoint(_x, _y, _style) {
    if (isNaN(_x) || isNaN(_y)) { 
    	return; 
    }
    
    mPointList[mPointList.length] = [_x, _y, _style];
  }


  self.registerProperties = function(controller) {
	EJSS_DRAWING2D.Polygon.registerProperties(self, controller);
  };
  
  self.copyTo = function(element) {
    EJSS_DRAWING2D.Polygon.copyTo(self,element);
  };
  
	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	self.setSize([1,1]);
	self.setRelativePosition("SOUTH_WEST");
	
	return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//PolygonSet
//---------------------------------

/**
 * PolygonSet
 * @class PolygonSet 
 * @constructor  
 */
EJSS_DRAWING2D.PolygonSet = {
    
    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING2D.ElementSet;
      
      ElementSet.registerProperties(set,controller);
      controller.registerProperty("Points", 
          function(v) { set.setToEach(function(element,value) { element.setPoints(value); }, v); }
      );
      controller.registerProperty("LastPoint", 
          function(v) { set.setToEach(function(element,value) { element.addPoint(value); }, v); }
      );           
    }        
};

/**
 * Creates a set of Segments
 * @method polygonSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING2D.polygonSet = function (mName) {
  var self = EJSS_DRAWING2D.elementSet(EJSS_DRAWING2D.polygon, mName);

  // Static references
  var PolygonSet = EJSS_DRAWING2D.PolygonSet;		// reference for PolygonSet

  /**
   * Registers properties in a ControlElement
   * @method registerProperties
   * @param controller A ControlElement that becomes the element controller
   */
  self.registerProperties = function(controller) {
    PolygonSet.registerProperties(self,controller);
  };

  return self;
};/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Ruler
 * @class Ruler 
 * @constructor  
 */
EJSS_DRAWING2D.Ruler = {
  sROTATE_ICON : "data:image/png;base64,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",
  sMOVE_ICON   : "data:image/png;base64,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",

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class

      controller.registerProperty("Minimum", element.setMinimum);
      controller.registerProperty("Maximum", element.setMaximum);

      controller.registerProperty("NumberOfMarks",element.setNumberOfMarks,element.getNumberOfMarks);
      controller.registerProperty("BigMark",element.setBigMark);
      controller.registerProperty("MediumMark",element.setMediumMark);
      controller.registerProperty("MarkFactor",element.setMarkFactor);
      controller.registerProperty("Digits",element.setDigits);

      controller.registerProperty("BorderWidth",element.setBorderWidth);
      controller.registerProperty("BorderColor",element.setBorderColor);
      controller.registerProperty("FillColor",element.setFillColor);
      controller.registerProperty("LineWidth",element.setLineWidth);
      controller.registerProperty("LineColor",element.setLineColor);
      controller.registerProperty("TextColor",element.setTextColor);
      controller.registerProperty("TextFont",element.setTextFont);

      controller.registerProperty("Width",element.setWidth);
      controller.registerProperty("Length",element.setLength);
      
      controller.registerProperty("DrawLines",element.setDrawLines);
      controller.registerProperty("DrawText",element.setDrawText);

      controller.registerProperty("Enabled",element.setEnabled);
      controller.registerProperty("Rotates",element.setRotates);

      controller.registerAction("OnDrag",element.getOnRotationInformation);
      controller.registerAction("OnRotation",element.getOnRotationInformation);

    }
};

/**
 * Creates a 2D ruler
 * @method ruler
 */
EJSS_DRAWING2D.ruler = function (mName) {
  var self = EJSS_DRAWING2D.group(mName);

  // Configuration variables
  var mLength = 1.1;
  var mWidth = 0.2;
  
  var mMinimum = 0;
  var mMaximum = 1;

  var mNumberOfMarks = 101;
  var mBigMark = 10;
  var mMediumMark = 5;
  var mMarkFactor = 0.1;
  var mDigits = 0;

  // Implementation variables
  var mGroup,mBody;
  var mSegments, mTexts;
  var mLeftHandle, mRightHandle;
  var mLeftRotate, mRightRotate;
  var mForegroundCreated = false;

  var mElements = [];
  var mScaleChanged = true;
  
  self.getElement = function(keyword) {
	  return mElements[keyword];
  };
  
  self.superSetChanged = self.setChanged;
  
  self.setChanged = function(changed) {
	self.superSetChanged(changed);
	if (changed) for (var i=0,n=mElements.length; i<n; i++) mElements[i].setChanged(changed);
  }

  
  // ----------------------------------------------------
  // public functions
  // ----------------------------------------------------

  self.setWidth = function(value) { 
	  if (value!=mWidth) {
		  mWidth = value;
		  self.mustAdjust();
	  }
  }
  self.getWidth = function() { return mWidth; }

  self.setLength = function(value) { 
	  if (value!=mLength) {
		  mLength = value;
		  self.mustAdjust();
      }
  }
  self.getLength = function() { return mLength; }

  /**
   * Sets the minimum value of the element. 
   * @method setMinimum
   * @param value
   */
  self.setMinimum = function(value) {
    if (mMinimum!=value) {
    	mMinimum = value;
    	mScaleChanged = true;
    }
  }
  
  /**
   * @method getMinimum
   * @return current minimum
   */
  self.getMinimum = function() { 
    return mMinimum; 
  };    

  /**
   * Sets the minimum value of the element. 
   * @method setMaximum
   * @param value
   */
  self.setMaximum = function(value) {
	if (mMaximum!=value) {
		mMaximum = value;
    	mScaleChanged = true;
	}
  }
  
  /**
   * @method getMaximum
   * @return current maximum
   */
  self.getMaximum = function() { 
    return mMaximum; 
  };    
  
  /***
   * Sets the number of marks
   * @method setNumberOfMarks(value)
   * @visibility public
   * @param value int
   */ 
  self.setNumberOfMarks = function(value) {
	  if (value!=mNumberOfMarks) {
		  mNumberOfMarks = value;
	      mScaleChanged = true;
	  }
  };

  /***
   * Gets the number of marks
   * @method getNumberOfMarks
   * @visibility public
   * @return int
   */
  self.getNumberOfMarks = function() {
  	return mNumberOfMarks;
  };


  /***
   * Sets the interval for big marks
   * @method setBigMark(value)
   * @visibility public
   * @param value int 
   */ 
  self.setBigMark = function(value) {
	  if (value!=mBigMark) {
        mBigMark = value;
    	mScaleChanged = true;
	  }
  };

  /***
   * Gets the interval for big marks
   * @method getBigMark
   * @visibility public
   * @return int
   */
  self.getBigMark = function() {
  	return mBigMark;
  };

  /***
   * Sets the interval for medium marks
   * @method setMediumMark(value)
   * @visibility public
   * @param value int 
   */ 
  self.setMediumMark = function(value) {
	  if (value!=mMediumMark) {
        mMediumMark = value;
    	mScaleChanged = true;
	  }
  };

  /***
   * Gets the interval for medium marks
   * @method getMediumMark
   * @visibility public
   * @return int
   */
  self.getMediumMark = function() {
  	return mMediumMark;
  };
  
  /***
   * Sets the factor that divide the value in marks
   * @method setMarkFactor(value)
   * @visibility public
   * @param value int|double 
   */ 
  self.setMarkFactor = function(value) {
	  if (value!=mMarkFactor) {
        mMarkFactor = value;
    	mScaleChanged = true;
	  }
  };

  /***
   * Gets the factor that divide the displayed value in marks
   * @method getMarkFactor
   * @visibility public
   * @return int|double
   */
  self.getMarkFactor = function() {
  	return mMarkFactor;
  };

  /***
   * Sets the maximum value to display in a mark
   * @method setMaximumMark(value)
   * @visibility public
   * @param value int|double 
   */ 
  self.setMaximumMark = function(value) {
	  if (value!=mMaximumMark) {
        mMaximumMark = value;
    	mScaleChanged = true;
	  }
  };

  /***
   * Gets the maximum value to display in a mark
   * @method getMaximumMark
   * @visibility public
   * @return value int|double 
   */ 
  self.getMaximumMark = function() {
  	return mMaximumMark;
  };

  /***
   * Sets the number of digits for the marks
   * @method setDigits(value)
   * @visibility public
   * @param value int 
   */ 
  self.setDigits = function(value) {
	  if (value!=mDigits) {
		mDigits = value;
	    mScaleChanged = true;
	  }
  };

  /***
   * Gets the number of digits for the marks
   * @method getPrecision
   * @visibility public
   * @return int
   */
  self.getDigits = function() {
  	return mDigits;
  };
  

  // ----------------------------------------------------
  // Properties overwritten
  // ----------------------------------------------------

  self.mustAdjust = function() { mScaleChanged = true; }
  
  self.setBorderWidth = function(value) {  mBody.getStyle().setLineWidth(value); }
  self.setBorderColor = function(value) {  mBody.getStyle().setLineColor(value); }
  self.setFillColor = function(value) { mBody.getStyle().setFillColor(value); }

  self.setLineWidth = function(value) { 
	  mSegments.setToEach(function(element) { element.getStyle().setLineWidth(value); });
  }
  self.setLineColor = function(value) { 
	  mSegments.setToEach(function(element) { element.getStyle().setLineColor(value); });
  }
  self.setDrawLines = function(value) { 
	  mSegments.setToEach(function(element,value) { element.setVisible(value); },value);
  }
  
  self.setTextColor = function(value) { 
	  mTexts.setToEach(function(element) { element.getFont().setFillColor(value); });
  }
  self.setTextFont = function(value) { 
	  mTexts.setToEach(function(element) { element.getFont().setFont(value); });
  }
  self.setDrawText = function(value) { 
	  mTexts.setToEach(function(element,value) { element.setVisible(value); },value);
  }

  self.setEnabled = function(value) { 
	  mLeftHandle.setVisible(value);
	  mRightHandle.setVisible(value);
  };

  self.setRotates = function(value) { 
	  mLeftRotate.setVisible(value);
	  mRightRotate.setVisible(value);
  };
  
  // ----------------------------------------------------
  // Properties and copies
  // ----------------------------------------------------

  /**
   * Extended registerProperties method. To be used by promoteToControlElement
   * @method registerProperties
   */
  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.Ruler.registerProperties(self,controller);
  };

  // ----------------------------------------------------
  // private functions
  // ----------------------------------------------------

  self.superSetParent = self.setParent;
  
  self.setParent = function(parent, sibling) {
	self.superSetParent(parent,sibling);
	if (!mForegroundCreated) {
      self.createRuler();
	  mForegroundCreated = true;
	}
	self.addParticularChildren();
	self.adjustForeground();
  }

  self.dataCollected = function() {
	  if (mScaleChanged) {
		  self.adjustForeground();
		  mScaleChanged = false;
	  }
  };

  self.addParticularChildren = function() { 
	mGroup.setParent(self);
	mBody.setParent(mGroup);
	mSegments.setToEach(function(element) { element.setParent(mGroup); });
	mTexts.setToEach(function(element) { element.setParent(mGroup); });
	mLeftHandle.setParent(self);
	mRightHandle.setParent(self);
	mLeftRotate.setParent(self);
	mRightRotate.setParent(self);
  };

  self.createBody = function() {
	var body = EJSS_DRAWING2D.shape(mName+".body");
	body.setShapeType("ROUND_RECTANGLE");
	body.setCornerRadius(5);
    return body;
  }

  self.createRuler = function() {
	mGroup = EJSS_DRAWING2D.group(mName+".group");
	mElements['group'] = mGroup;

	mBody = self.createBody();
	mBody.getStyle().setFillColor("rgba(0,255,0,0.2)");
	mElements['body'] = mBody;

	mSegments = EJSS_DRAWING2D.segmentSet(mName+".segments");
	mSegments.setToEach(function(element) { element.setRelativePosition("SOUTH_WEST"); });
	mSegments.setToEach(function(element) { element.getStyle().setLineWidth(0.5); });
	mElements['segments'] = mSegments;

	mTexts = EJSS_DRAWING2D.textSet(mName+".texts");
	mTexts.setToEach(function(element) { element.getFont().setFont("normal normal 8px \"Courier New\", Courier, monospace"); });
	mTexts.setToEach(function(element) { element.setRelativePosition("NORTH"); });
	mElements['texts'] = mTexts;

	mLeftHandle = EJSS_CORE.promoteToControlElement(
				EJSS_DRAWING2D.image(mName+".left_handle"),self.getView(),mName+".left_handle");
	mLeftHandle.setImageUrl(EJSS_DRAWING2D.Ruler.sMOVE_ICON);
	mLeftHandle.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setMotionEnabled("ENABLED_ANY");
	mLeftHandle.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setAffectsGroup(true);
	mLeftHandle.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setSensitivity(10);
	mLeftHandle.setSize([10,10]);
	mLeftHandle.setPixelSize(true);
	mLeftHandle.setProperty("OnDrag",moved);
	mElements['left_handle'] = mLeftHandle;

	mRightHandle = EJSS_CORE.promoteToControlElement(
				EJSS_DRAWING2D.image(mName+".right_handle"),self.getView(),mName+".right_handle");
	mRightHandle.setImageUrl(EJSS_DRAWING2D.Ruler.sMOVE_ICON);
	mRightHandle.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setMotionEnabled("ENABLED_ANY");
	mRightHandle.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setAffectsGroup(true);
	mRightHandle.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setSensitivity(10);
	mRightHandle.setProperty("OnDrag",moved);
	mRightHandle.setSize([10,10]);
	mRightHandle.setPixelSize(true);
	mElements['right_handle'] = mRightHandle;
		
	mLeftRotate = EJSS_CORE.promoteToControlElement(
			EJSS_DRAWING2D.image(mName+".left_rotate"),self.getView(),mName+".left_rotate");
	mLeftRotate.setImageUrl(EJSS_DRAWING2D.Ruler.sROTATE_ICON);
	mLeftRotate.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setMotionEnabled("ENABLED_NO_MOVE");
	mLeftRotate.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setSensitivity(10);
	mLeftRotate.setSize([8,8]);
	mLeftRotate.setPixelSize(true);
	mLeftRotate.setProperty("OnDrag",rotated);
	mElements['left_rotate'] = mLeftRotate;

	mRightRotate = EJSS_CORE.promoteToControlElement(
			EJSS_DRAWING2D.image(mName+".right_rotate"),self.getView(),mName+".right_rotate");
	mRightRotate.setImageUrl(EJSS_DRAWING2D.Ruler.sROTATE_ICON);
	mRightRotate.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setMotionEnabled("ENABLED_NO_MOVE");
	mRightRotate.getInteractionTarget(EJSS_DRAWING2D.PanelInteraction.TARGET_POSITION).setSensitivity(10);
	mRightRotate.setSize([8,8]);
	mRightRotate.setPixelSize(true);
	mRightRotate.setProperty("OnDrag",rotated);
	mElements['right_rotate'] = mRightRotate;

  };

  self.adjustBody = function() {
		mBody.setSize([mLength,mWidth]);
  };

  self.adjustForeground = function() {
	mLeftHandle.setPosition([-0.125*mLength,0]);
	mRightHandle.setPosition([0.125*mLength,0]);
	mLeftRotate.setPosition([-0.35*mLength,0]);
	mRightRotate.setPosition([0.35*mLength,0]);
	
	mGroup.setY(mWidth*0.3);
	self.adjustBody();
	
	var MARGIN = (mLength - Math.abs(mMaximum-mMinimum))/2;
	var MARK = mWidth/6;
	  
	mSegments.setNumberOfElements(mNumberOfMarks);
	var dx = Math.abs(mMaximum-mMinimum)/(mNumberOfMarks-1);
	var xPos = -mLength/2+MARGIN;
	var mediumMarksCounter = 0;
	var bigMarksCounter = 0;

	for (var i=0; i<mNumberOfMarks; i++) {
	  var segment = mSegments.getElement(i);
	  segment.setPosition([xPos,mWidth/2]);
	  if (i%mBigMark==0) {
	    bigMarksCounter++;
	    segment.setSize([0,-2*MARK]);
	  }
	  else if (i%mMediumMark==0) {
	    mediumMarksCounter++;
	    segment.setSize([0,-1.5*MARK]);
	  }
	  else segment.setSize([0,-MARK]);
	  xPos += dx;
	}
	
	mTexts.setNumberOfElements(bigMarksCounter);
	var text_counter=0;
	for (i=0; i<mNumberOfMarks; i++) {
	  var segment = mSegments.getElement(i);
	  if (i%mMediumMark==0) {
		  // locate units
	  }
	  if (i%mBigMark==0) {
	    var text = mTexts.getElement(text_counter);
	    text.setX(segment.getX());
	    text.setY(mWidth/2+segment.getSizeY()*1.1);
	    var mark_value = ((mNumberOfMarks-1-i)*mMinimum + i*mMaximum)/(mNumberOfMarks-1) /mMarkFactor;
	    text.setText(""+mark_value.toFixed(mDigits));
	    text_counter++;
	  }
	}
  };
   
  self.getOnRotationInformation = function() {
	  return { position : self.getPosition(), element : self, 
		  point : self.getGroupPanel().getPanelInteraction().getInteractionPoint(),
		  angle : self.getTransformation() };
  };

  function moved(point,info) {
	  var controller = self.getController();    		
	  if (controller) {
		controller.invokeImmediateAction("OnDrag");
	  }
  };

  self.computeRotation = function(rightIcon,point) {
	  if (rightIcon) return Math.atan2(point[1],point[0]);
	  else return Math.atan2(-point[1],-point[0]);
  };
  
  function rotated(point,info) {
	  var point = info.point;
	  var element = info.element;
	  var group_pos = self.getAbsolutePosition(true);
	  point[0] -= group_pos[0];
	  point[1] -= group_pos[1];
	  var rotation = self.computeRotation(element==mRightRotate,point);
	  self.setTransformation(rotation);
	  self.setChanged(true);
	  var controller = self.getController();    		
	  if (controller) {
//		controller.immediatePropertyChanged("Value");
		controller.invokeImmediateAction("OnRotation");
	  }	        

  }
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------
  
  return self;
};


/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * ScalarField
 * @class ScalarField 
 * @constructor  
 */
EJSS_DRAWING2D.ScalarField = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class

		controller.registerProperty("ZData", element.setData, element.getData);
		controller.registerProperty("MinimumZ", element.setMinimumZ, element.getMinimumZ);
		controller.registerProperty("MaximumZ", element.setMaximumZ, element.getMaximumZ);
		controller.registerProperty("AutoscaleZ", element.setAutoscaleZ, element.getAutoscaleZ);
		controller.registerProperty("SymmetricZ", element.setSymmetricZ, element.getSymmetricZ);
		//controller.registerProperty("ExpandedZ", element.setMinimumX, element.getMinimumX);

		controller.registerProperty("FloorColor", element.setFloorColor, element.getFloorColor);
		controller.registerProperty("CeilColor", element.setCeilColor, element.getCeilColor);

		controller.registerProperty("ShowGrid", element.setShowGrid, element.getShowGrid);
		controller.registerProperty("NumColors", element.getColorMapper().setNumberOfColors);
		controller.registerProperty("Colors", element.getColorMapper().setColorPalette);
		controller.registerProperty("Palette", element.getColorMapper().setPaletteType);
        controller.registerProperty("AutoUpdate", element.setAutoupdate);  

    },
};

/**
 * Creates a 2D scalarField
 * @method ScalarField
 */
EJSS_DRAWING2D.scalarField = function (name) {
	var self = EJSS_DRAWING2D.element(name);
 
  	var mColorMapper = EJSS_DRAWING2D.colorCoded(16, EJSS_DRAWING2D.ColorMapper.SPECTRUM); 
	var mCeilColor = mColorMapper.getCeilColor();
	var mFloorColor = mColorMapper.getFloorColor();
 	var mShowGrid = true;
	var mData = [];
	var mAutoUpdate=true;
	var mzMax = 1;
	var mzMin = -1;
	var mAutoscaleZ = true;
	var mSymmetricZ = false;
 
	self.getClass = function() {
  		return "ElementScalarField";
	}

   	/**
   	* @method setFloorColor
   	* @param color
   	*/
  	self.setFloorColor = function (color) {
	  	if(mFloorColor != color) {
    		mFloorColor = color;
    		mColorMapper.setFloorCeilColor(mFloorColor,mCeilColor);
    		self.setChanged(true);
    	}
  	}
  
  	/**
   	* @method getFloorColor
   	* @return color
   	*/
  	self.getFloorColor = function() {
    	return mFloorColor;
  	}

   	/**
   	* @method setCeilColor
   	* @param color
   	*/
  	self.setCeilColor = function (color) {
	  	if(mCeilColor != color) {
    		mCeilColor = color;
    		mColorMapper.setFloorCeilColor(mFloorColor,mCeilColor);
    		self.setChanged(true);
    	}
  	}
  
  	/**
   	* @method getCeilColor
   	* @return color
   	*/
  	self.getCeilColor = function() {
    	return mCeilColor;
  	}

	/** 
	 * @method setData
	 * @param data
	 */
	self.setData = function (data) {
	  if (mAutoUpdate || mData != data) {
	    mData = data;
		mDataChanged = true;
		self.setChanged(true);
	  }
	}
	  
	/**
	 * @method getData
	 * @return
	 */
	self.getData = function() { 
		return mData; 
	}
	 
	self.setAutoupdate= function (auto) {
	  mAutoUpdate = auto;
      if (mAutoUpdate) mDataChanged = true;
    }
    
	/** 
	 * @method getMaximumZ
	 * @return max
	 */
  	self.getMaximumZ = function() {
  		return mzMax;
  	}

	/** 
	 * Sets ceiling value for the colors 
	 *
	 * @method setMaximumZ
	 * @param max
	 */
  	self.setMaximumZ = function(max) {
  		if(mzMax != max) {
  			mzMax = max;
	    	if (!mAutoscaleZ) {
	    		mColorMapper.setScale(mzMin, mzMax);		
	  		}
			self.setChanged(true);
  		};
  	}

	/** 
	 * @method getMinimumZ
	 * @return min
	 */
  	self.getMinimumZ = function() {
  		return mzMin;
  	}

	/** 
	 * Sets floor value for the colors 
	 *
	 * @method setMinimumZ
	 * @param min
	 */
  	self.setMinimumZ = function(min) {
  		if(mzMin != min) {
  			mzMin = min;
	    	if (!mAutoscaleZ) {
	    		mColorMapper.setScale(mzMin, mzMax);		
	  		}
			self.setChanged(true);
  		};
  	}

  	/**
   	* Sets the autoscale flag.
   	*
   	* If autoscaling is true, then the min and max values of z are span the colors.
   	*
   	* If autoscaling is false, then MaximumZ and MinimumZ values limit the colors.
   	* Values below min map to the first color; values above max map to the last color.
   	*
   	* @method setAutoscaleZ
   	* @param isAutoscale
   	*/
  	self.setAutoscaleZ = function (isAutoscale) {
	  	if(mAutoscaleZ != isAutoscale) {
	    	mAutoscaleZ = isAutoscale;
	    	self.setChanged(true);
	    }
  	}
  
  	/**
   	* Gets the autoscale flag for z.
   	*
   	* @method getAutoscaleZ
   	* @return boolean
   	*/
  	self.getAutoscaleZ = function() {
    	return mAutoscaleZ;
  	}

   	/**
   	* @method setSymmetricZ
   	* @param isAutoscale
   	*/
  	self.setSymmetricZ = function (symmetric) {
	  	if(mSymmetricZ != symmetric) {
    		mSymmetricZ = symmetric;
    		mColorMapper.setSymmetricZ(mSymmetricZ);
    		self.setChanged(true);
    	}
  	}
  
  	/**
   	* @method getSymmetricZ
   	* @return boolean
   	*/
  	self.getSymmetricZ = function() {
    	return mSymmetricZ;
  	}

	/** 
	 * @method setColorMapper
	 * @param colormapper
	 */
	self.setColorMapper = function (colormapper) {
	  	if(mColorMapper != colormapper) {
	  		mColorMapper = colormapper;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getColorMapper
	 * @return
	 */
	self.getColorMapper = function() { 
		return mColorMapper; 
	}
	 
	/** 
	 * @method setShowGrid
	 * @param showgrid
	 */
	self.setShowGrid = function (showgrid) {
	  	if(mShowGrid != showgrid) {
	  		mShowGrid = showgrid;
	  		self.setChanged(true);
	  	}
	}
	  
	/**
	 * @method getShowGrid
	 * @return
	 */
	self.getShowGrid = function() { 
		return mShowGrid; 
	}

	self.registerProperties = function(controller) {
    	EJSS_DRAWING2D.ScalarField.registerProperties(self,controller);
	};
  
	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * A Segment is a 2D drawing element that displays a single line 
 * @class EJSS_DRAWING2D.Segment
 * @parent EJSS_DRAWING2D.Element
 * @constructor  
 */
EJSS_DRAWING2D.Segment = {

    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class

      controller.registerProperty("Offset", element.setRelativePosition, element.getRelativePosition);
    },  


};

/**
 * Creates a 2D Segment
 * @method segment
 */
EJSS_DRAWING2D.segment = function (name) {
  var self = EJSS_DRAWING2D.element(name);

  self.getClass = function() {
  	return "ElementSegment";
  };
  
    self.registerProperties = function(controller) {
    EJSS_DRAWING2D.Segment.registerProperties(self,controller);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([0.1,0.1]);
  self.setRelativePosition("SOUTH_WEST");

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//SegmentSet
//---------------------------------

/**
 * SegmentSet
 * @class SegmentSet 
 * @constructor  
 */
EJSS_DRAWING2D.SegmentSet = {
    
    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING2D.ElementSet;
      
      ElementSet.registerProperties(set,controller);
      controller.registerProperty("Offset", 
          function(v) { set.setToEach(function(element,value) { element.setRelativePosition(value); }, v); }
      );
    }
    
};


/**
 * Creates a set of Segments
 * @method segmentSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING2D.segmentSet = function (mName) {
  var self = EJSS_DRAWING2D.elementSet(EJSS_DRAWING2D.segment, mName);

  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.SegmentSet.registerProperties(self,controller);
  };
  
  return self;
};/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * SetSquare
 * @class SetSquare 
 * @constructor  
 */
EJSS_DRAWING2D.SetSquare = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Ruler.registerProperties(element,controller); // super class

      controller.registerProperty("Height",element.setHeight);
    }

};

/**
 * Creates a 2D shape
 * @method shape
 */
EJSS_DRAWING2D.setSquare = function (mName) {
  var self = EJSS_DRAWING2D.ruler(mName);
  
  var mHeight = self.getLength();

  // Implementation variables
  // ----------------------------------------------------
  // public functions
  // ----------------------------------------------------

  // ----------------------------------------------------
  // Properties and copies
  // ----------------------------------------------------

  self.setHeight = function(value) { 
	  if (value!=mHeight) {
		  mHeight = value;
		  self.mustAdjust();
      }
  }
  self.getHeight = function() { return mHeight; }

  /**
   * Extended registerProperties method. To be used by promoteToControlElement
   * @method registerProperties
   */
  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.SetSquare.registerProperties(self,controller);
  };

  // ----------------------------------------------------
  // private functions
  // ----------------------------------------------------

  self.createBody = function() {
	var body = EJSS_DRAWING2D.polygon(mName+".body");
	return body;
  }

  self.adjustBody = function() {
	    var w = self.getWidth(), l = self.getLength(), h = self.getHeight();
	    var hyp = Math.sqrt(l*l+h*h);
	    var l2 = (h-w)*l/h - w - w*hyp/h; // interior length
	    var h2 = (h-w) - w*h/l - w*hyp/l; // interior height
	    
	    var points = [[0,0,1],[l,0,1],[0,-h,1],[0,0,1],
	                 [w,-w,0],[w,-w-h2,1],[w+l2,-w,1],[w,-w,1]];
	    for (var i=0; i<points.length; i++) {
	    	points[i][0] -= l/2;
	    	points[i][1] += w/2;
	    };
		self.getElement('body').setPoints(points);
		
		self.getElement('left_handle').setPosition([w+l2/4-l/2,0]);
		self.getElement('right_handle').setPosition([w+3*l2/4-l/2,0]);
		self.getElement('left_rotate').setPosition([w/2-l/2,0]);
		self.getElement('right_rotate').setPosition([w+l2+w/2-l/2,0]);	  
  };
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------
    self.setMaximum(0.99);
	self.setNumberOfMarks(100);

  return self;
};


/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * A Shape is a 2D drawing element that displays a 2D shape (such as a rectangle, ellipse, etc.) 
 * @class EJSS_DRAWING2D.Shape
 * @parent EJSS_DRAWING2D.Element
 */
EJSS_DRAWING2D.Shape = {
    NONE : 0,
    ELLIPSE : 1,
    RECTANGLE : 2,
    ROUND_RECTANGLE : 3,
    WHEEL : 4,
    POINT : 5,

    /**
     * Static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller);

	 /*** 
	  * Type of shape
	  * @property ShapeType 
	  * @type int|String
	  * @values 0:"NONE", 1:"ELLIPSE", 2:"RECTANGLE", 3:"ROUND_RECTANGLE", 4:"WHEEL", 5:"POINT" 
	  * @default "ELLIPSE" 
	  */  
      controller.registerProperty("ShapeType",element.setShapeType);
      
	 /*** 
	  * Radius for the corners of a round rectangular shape
	  * @property CornerRadius 
	  * @type int
	  * @default 10 
	  */  
      controller.registerProperty("CornerRadius",element.setCornerRadius);
    },

    /**
     * Static copyTo method, to be used by sets
     */
    copyTo : function(source, dest) {
      EJSS_DRAWING2D.Element.copyTo(source,dest);

      dest.setShapeType(source.getShapeType());
      dest.setCornerRadius(source.getCornerRadius());
    }

};

/**
 * Creates a 2D shape
 * @method shape
 */
EJSS_DRAWING2D.shape = function (name) {
  var self = EJSS_DRAWING2D.element(name);

  // Instance variables
  var mCornerRadius = 10;
  var mShapeType = EJSS_DRAWING2D.Shape.ELLIPSE;

  // ----------------------------------------------------
  // Public functions
  // ----------------------------------------------------

  /***
   * Sets the value of the property ShapeType
   * @method setShapeType(shapeType)
   * @visibility public
   * @param shapeType int|String 
   */ 
  self.setShapeType = function(shapeType) {
    if (typeof shapeType === 'string')
      mShapeType = EJSS_DRAWING2D.Shape[shapeType.toUpperCase()];
    else 
      mShapeType = shapeType;
  };

  /***
   * Gets the value of the property ShapeType
   * @method getShapeType
   * @visibility public
   * @return int|String 
   */ 
  self.getShapeType = function() {
  	return mShapeType;
  }

  /***
   * Sets the value of the property CornerRadius
   * @method setCornerRadius
   * @param radius int
   */
  self.setCornerRadius = function(radius) {
    if(mCornerRadius != radius) {
    	mCornerRadius = radius;
    	self.setChanged(true);
    }
  };

  /***
   * Gets the value of the property CornerRadius
   * @method getCornerRadius
   * @return int
   */
  self.getCornerRadius = function() {
    return mCornerRadius;
  };

  // ----------------------------------------------------
  // Private functions
  // ----------------------------------------------------

  /***
   * Returns the class name
   * @method getClass
   * @visibility private
   * @return String
   */
  self.getClass = function() {
  	return "ElementShape";
  }

  /***
   * Extended registerProperties method
   * @method registerProperties
   * @param controller
   * @visibility private
   */
  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.Shape.registerProperties(self,controller);
  };

  /***
   * Extended copyTo method
   * @method copyTo
   * @param Element
   * @visibility private
   */
  self.copyTo = function(element) {
    EJSS_DRAWING2D.Shape.copyTo(self,element);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  /*** @property Size @default [0.1,0.1] */
  self.setSize([0.1,0.1]);
  
  /*** @property RelativePosition @default "CENTER" */
  self.setRelativePosition("CENTER");
  
  /*** @property FillColor @default "Blue" */  
  self.getStyle().setFillColor("Blue");
  
  /*** @property LineColor @default "Black" */  
  self.getStyle().setLineColor("Black");
    
  return self;
};


/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//ShapeSet
//---------------------------------

/**
 * ShapeSet
 * @class ShapeSet 
 * @constructor  
 */
EJSS_DRAWING2D.ShapeSet = {

    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING2D.ElementSet;
      
      ElementSet.registerProperties(set,controller);
      controller.registerProperty("ShapeType", 
          function(v) { set.setToEach(function(element,value) { element.setShapeType(value); }, v); }
      );
      controller.registerProperty("CornerRadius", 
          function(v) { set.setToEach(function(element,value) { element.setCornerRadius(value); }, v); }
      );
    }

};


/**
 * Creates a set of shapes
 * @method shapeSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING2D.shapeSet = function (mName) {
  var self = EJSS_DRAWING2D.elementSet(EJSS_DRAWING2D.shape, mName);

  // Static references
  var ShapeSet = EJSS_DRAWING2D.ShapeSet;		// reference for ShapeSet
  
  self.registerProperties = function(controller) {
    ShapeSet.registerProperties(self,controller);
  };

  return self;
};
/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia and Félix J. García 
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*/

/**
 * Framework for 2D drawing.
 * @module 2Ddrawing
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * A SimplePanel is a 2D drawing panel with no decoration. 
 * @class EJSS_DRAWING2D.SimplePanel
 * @see EJSS_DRAWING2D.PlottingPanel
 * @constructor
 */
EJSS_DRAWING2D.SimplePanel = {
	GRAPHICS2D_SVG : "SVG",
	GRAPHICS2D_CANVAS : "Canvas",

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		// No super class
		// EJSS_INTERFACE.Element.registerProperties(element.getGraphics(), controller);

		controller.registerProperty("MinimumX", element.setWorldXMin, element.getWorldXMin);
		controller.registerProperty("MaximumX", element.setWorldXMax, element.getWorldXMax);
		controller.registerProperty("MinimumY", element.setWorldYMin, element.getWorldYMin);
		controller.registerProperty("MaximumY", element.setWorldYMax, element.getWorldYMax);
		controller.registerProperty("Bounds", element.setWorldCoordinates, element.getWorldCoordinates);


		controller.registerProperty("MarginX", element.setMarginX,element.getMarginX);
		controller.registerProperty("MarginY", element.setMarginY,element.getMarginY);

		controller.registerProperty("Parent", element.getGraphics().setParent, element.getGraphics().getParent);
		controller.registerProperty("Width", element.getGraphics().setWidth, element.getGraphics().getWidth);
		controller.registerProperty("Height", element.getGraphics().setHeight, element.getGraphics().getHeight);

		controller.registerProperty("Graphics", element.setGraphics);

		controller.registerProperty("Background", element.getStyle().setFillColor);
		controller.registerProperty("Foreground", element.getStyle().setLineColor);
		controller.registerProperty("LineColor", element.getStyle().setLineColor);
		controller.registerProperty("LineWidth", element.getStyle().setLineWidth);
		controller.registerProperty("DrawLines", element.getStyle().setDrawLines);
		controller.registerProperty("FillColor", element.getStyle().setFillColor);
		controller.registerProperty("DrawFill", element.getStyle().setDrawFill);
		controller.registerProperty("ShapeRendering", element.getStyle().setShapeRendering);

	    controller.registerProperty("Visibility", element.getGraphics().getStyle().setVisibility);      
        controller.registerProperty("Display", element.getGraphics().getStyle().setDisplay); 
		controller.registerProperty("CSS", element.getGraphics().getStyle().setCSS);
	}
};

/**
 * Constructor for SimplePanel
 * @method simplePanel
 * @param mName string
 * @returns An abstract 2D drawing panel
 */
EJSS_DRAWING2D.simplePanel = function(mName) {
	var self = {}; // reference returned	
	var mGraphics = EJSS_GRAPHICS.svgGraphics(mName); // graphics implementation (default: SVG)

	// Instance variables
	var mStyle = EJSS_DRAWING2D.style(mName);	// style for panel
	var mElements = [];							// elements list for panel
	var mElementsChanged = false;				// whether elements list has changed
	var mCollectersList = [];		            // Array of all control elements that need a call to dataCollected() after data collection

	// Configuration variables	
	var mWorld = {
		// preferred dimensions
		xminPreferred : -1, xmaxPreferred : 1, yminPreferred : -1, ymaxPreferred : 1,
		// origin in panel
		xorigin : 0, yorigin : 0,
		// pixel per unit for panel
		xscale : 1, yscale : 1		
	};

	// Implementation variables
	var mPanelChanged = true;	// whether panel changed (style, decorations, gutters)
	var mMustScale = true;		// whether panel must scale

	// ----------------------------------------
	// Instance functions
	// ----------------------------------------

	/**
	 * Get name for drawing panel
	 * @method getName
	 * @return string
	 */
	self.getName = function() {
		return mName;
	};

	/**
	 * Returns the graphics implementation
	 * @method getGraphics
	 * @return Graphics
	 */
	self.getGraphics = function() {
		return mGraphics;
	};

	/**
	 * Returns the svg image in Base64 format
	 * @method importGraphics
	 * @return string 
	 */
    self.importGraphics = function(callback) {
    	return mGraphics.importSVG(callback);
    }
    
	/**
	 * Return the drawing style of the inner rectangle for panel
	 * @method getStyle
	 * @return Style
	 */
	self.getStyle = function() {
		return mStyle;
	};

	/**
	 * Set graphics
	 * @method setGraphics
	 * @param type
	 */
	self.setGraphics = function(type) {
		if (type == EJSS_DRAWING2D.SimplePanel.GRAPHICS2D_SVG) {
			mGraphics = EJSS_GRAPHICS.svgGraphics(mName);
		} else if (type == EJSS_DRAWING2D.SimplePanel.GRAPHICS2D_CANVAS) {
			// mGraphics = EJSS_GRAPHICS.canvasGraphics(mName);
			console.log("WARNING: setGraphics() - Canvas not supported");
		} else {
			console.log("WARNING: setGraphics() - Graphics not supported");
		}
	};
	
	// ----------------------------------------
	// World coordinates
	// ----------------------------------------

	/**
	 * Sets the preferred minimum X coordinate for the panel
	 * @method setWorldXMin
	 * @param xmin
	 */
	self.setWorldXMin = function(xmin) {
		if (xmin !== mWorld.xminPreferred) {
			mWorld.xminPreferred = xmin;
			mMustScale = true;
		}
	};

	/**
	 * Returns the preferred minimum X coordinate for the panel
	 * @method getWorldXMin
	 * @return double
	 */
	self.getWorldXMin = function() {
		return mWorld.xminPreferred;
	};

	/**
	 * Sets the preferred maximum X coordinate for the panel
	 * @method setWorldXMax
	 * @param xmax
	 */
	self.setWorldXMax = function(xmax) {
		if (xmax !== mWorld.xmaxPreferred) {
			mWorld.xmaxPreferred = xmax;
			mMustScale = true;
		}
	};

	/**
	 * Returns the preferred maximum X coordinate for the panel
	 * @method getWorldXMax
	 * @return double
	 */
	self.getWorldXMax = function() {
		return mWorld.xmaxPreferred;
	};

	/**
	 * Sets the preferred minimum Y coordinate for the panel
	 * @method setWorldYMin
	 * @param ymin
	 */
	self.setWorldYMin = function(ymin) {
		if (ymin !== mWorld.yminPreferred) {
			mWorld.yminPreferred = ymin;
			mMustScale = true;
		}
	};

	/**
	 * Returns the preferred minimum Y coordinate for the panel
	 * @method getWorldYMin
	 * @return double
	 */
	self.getWorldYMin = function() {
		return mWorld.yminPreferred;
	};

	/**
	 * Sets the preferred maximum Y coordinate for the panel
	 * @method setWorldYMax
	 * @param ymax
	 */
	self.setWorldYMax = function(ymax) {
		if (ymax !== mWorld.ymaxPreferred) {
			mWorld.ymaxPreferred = ymax;
			mMustScale = true;
		}
	};

	/**
	 * Returns the preferred maximum Y coordinate for the panel
	 * @method getWorldYMax
	 * @return double
	 */
	self.getWorldYMax = function() {
		return mWorld.ymaxPreferred;
	};

	/**
	 * Sets the preferred user coordinates for the panel
	 * @method setWorldCoordinates
	 * @param bounds
	 */
	self.setWorldCoordinates = function(bounds) {
		self.setWorldXMin(bounds[0]);
		self.setWorldXMax(bounds[1]);
		self.setWorldYMin(bounds[2]);
		self.setWorldYMax(bounds[3]);
	};

	/**
	 * Gets the preferred user coordinates for the panel
	 * @method getWorldCoordinates
	 * @return bounds
	 */
	self.getWorldCoordinates = function() {
		return [self.getWorldXMin(), self.getWorldXMax(), self.getWorldYMin(), self.getWorldYMax()];
	};

	// ----------------------------------------
	// Elements
	// ----------------------------------------

	/**
	 * Add a element to the panel. Elements are asked to draw themselves
	 * whenever the panel needs to render. For this purpose, they will receive a
	 * calls to draw().
	 * Elements are reported of changes in the world coordinates of the panel, in case
	 * they need to recalculate themselves.
	 * @method addElement
	 * @param element Element
	 * @param position int
	 */
	self.addElement = function(element, position) {
		EJSS_TOOLS.addToArray(mElements, element, position);
		// set this panel to decoration element
		element.setPanel(self);
		if (element.dataCollected) mCollectersList.push(element);
		// elements list has changed
		mElementsChanged = true;
	};

	/**
	 * Remove a element to the panel.
	 * @method removeElement
	 * @param element Element
	 */
	self.removeElement = function(element) {
		EJSS_TOOLS.removeFromArray(mElements, element);
		element.setPanel(null);
		if (element.dataCollected) EJSS_TOOLS.removeFromArray(mCollectersList, element);
		// elements list has changed
		mElementsChanged = true;
	};

	/**
	 * Return the array of a elements.
	 * @method getElements
	 * @return Elements
	 */
	self.getElements = function() {
		return mElements;
	};

	/**
	 * Return the position of a element.
	 * @method indexOfElement
	 * @param element Element
	 * @return integer
	 */
	self.indexOfElement = function(element) {
		return mElements.indexOf(element);
	};

	// ----------------------------------------
	// Apply transformations and conversions
	// ----------------------------------------

	/**
	 * Converts a Y pixel value so that 0 is at the bottom
	 * @method toPixelAxisY
	 * @param y double
	 */
	self.toPixelAxisY = function(y) {
		return (mWorld.yorigin - y) - (mWorld.yscale * mWorld.yminPreferred);
	};

	/**
	 * Converts a X pixel value so that 0 is at the left
	 * @method toPixelAxisX
	 * @param x double
	 */
	self.toPixelAxisX = function(x) {
		return (mWorld.xorigin + x) - (mWorld.xscale * mWorld.xminPreferred);
	};

	/**
	 * To be used only after a call to render()!
	 * Projects a point from world coordinates to pixel coordinates.
	 * @method toPixelPosition
	 * @param point double[] The original coordinates
	 * @return double[] The same array once transformed
	 */
	self.toPixelPosition = function(point) {
		var pos = [];
		pos[0] = mWorld.xorigin + mWorld.xscale * (point[0] - mWorld.xminPreferred);		
		pos[1] = mWorld.yorigin + mWorld.yscale * (point[1] - mWorld.yminPreferred);			
		return pos;
	};

	/**
	 * To be used only after a call to render()!
	 * Projects a module from world coordinates to pixel coordinates
	 * @method toPixelMod
	 * @param point double[] The original module
	 * @return double[] The same array once transformed
	 */
	self.toPixelMod = function(mod) {
		var pmod = [];
		pmod[0] = mod[0] * mWorld.xscale;
		pmod[1] = mod[1] * mWorld.yscale;		
		return pmod;
	};

	/**
	 * To be used only after a call to render()!
	 * Projects a point from pixel coordinates to world coordinates
	 * @method toPanelPosition
	 * @param point double[] The original coordinates
	 * @return double[] The same array once transformed
	 */
	self.toPanelPosition = function(point) {	
		var pos = [];
		pos[0] = mWorld.xminPreferred + (point[0] - mWorld.xorigin) / mWorld.xscale;
		pos[1] = mWorld.yminPreferred + (point[1] - mWorld.yorigin) / mWorld.yscale;
		return pos;
	};

	/**
	 * To be used only after a call to render()!
	 * Projects a module from pixel coordinates to world coordinates
	 * @method toPanelMod
	 * @param point double[] The original module
	 * @return double[] The same array once transformed
	 */
	self.toPanelMod = function(mod) {
		var pmod = [];
		pmod[0] = ((mWorld.xscale == 0)? 0 : mod[0]/mWorld.xscale);
		pmod[1] = ((mWorld.yscale == 0)? 0 : mod[1]/mWorld.yscale);		
		return pmod;
	};

	/**
	 * Get pixel position of the origin
	 * @return array pixel position
	 */
	self.getPixelPositionWorldOrigin = function() {
		return [mWorld.xorigin,mWorld.yorigin];
	}

	/**
	 * Recomputes the scales of the panel.
	 * @method recomputeScales
	 */
	self.recomputeScales = function() {	
		// get sizes and scale
		var width = mGraphics.getWidth(); // width in pixels
		var height = mGraphics.getHeight(); // height in pixels
		var xPixPerUnit = width / (mWorld.xmaxPreferred - mWorld.xminPreferred); // the x scale in pixels
		var yPixPerUnit = height / (mWorld.ymaxPreferred - mWorld.yminPreferred); // the y scale in pixels
		
		// centered
		mWorld.xscale = xPixPerUnit;
		mWorld.yscale = -yPixPerUnit;
		mWorld.xorigin =  0.5;
		mWorld.yorigin = height + 0.5;
		
		mMustScale = false;
	};
	
	// ----------------------------------------
	// Drawing functions
	// ----------------------------------------

	/**
	 * Reset the scene
	 * @method reset
	 */
	self.reset = function() {
		mGraphics.reset();
	};
	
	/**
	 * Render the scene
	 * @method render
	 */
	self.render = function() {
		var reseted = false;
		if (mElementsChanged) {// whether elements added or removed, reset the scene
			mGraphics.reset();
			reseted = true;
			mElementsChanged = false;
		}

        // check for data collection
		for (var i = 0, n = mCollectersList.length; i < n; i++)
			mCollectersList[i].dataCollected();

		// recompute scales
		if (mMustScale) 
			self.recomputeScales();

		if (mPanelChanged || reseted) {// whether panel changed or reseted
			mGraphics.drawPanel(self);
		}

		// draw visible elements
		mGraphics.draw(mElements, reseted);

		// set changed to false
		mPanelChanged = false;
		for (var i = 0, n = mElements.length; i < n; i++)
			mElements[i].setChanged(false);
	};

	// ----------------------------------------------------
	// Properties
	// ----------------------------------------------------

	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.SimplePanel.registerProperties(self, controller);
	};

	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	mStyle.setLineColor('black');
	mStyle.setFillColor('rgb(239,239,255)');
	mStyle.setChangeListener(function(change) {
		mPanelChanged = true;
	});


	return self;
};

/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Spring
 * @class Spring 
 * @constructor  
 */
EJSS_DRAWING2D.Spring = {
  DEF_RADIUS : 0.05,
  DEF_LOOPS : 8,
  DEF_PPL : 15,

	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy
  	
		dest.setRadius(source.getRadius());
		dest.setSolenoid(source.getSolenoid());
		dest.setThinExtremes(source.getThinExtremes());
		dest.setLoops(source.getLoops());
		dest.setPointsPerLoop(source.getPointsPerLoop());
  	},

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class

		controller.registerProperty("Radius", element.setRadius, element.getRadius);
		controller.registerProperty("Solenoid", element.setSolenoid, element.getSolenoid);
		controller.registerProperty("ThinExtremes", element.setThinExtremes, element.getThinExtremes);
		controller.registerProperty("Loops", element.setLoops, element.getLoops);
		controller.registerProperty("PointsPerLoop", element.setPointsPerLoop, element.getPointsPerLoop);
	},
};

/**
 * Creates a 2D Spring
 * @method spring
 */
EJSS_DRAWING2D.spring = function(name) {
	var self = EJSS_DRAWING2D.element(name);
  	
	  // Configuration variables
	var mRadius = EJSS_DRAWING2D.Spring.DEF_RADIUS;
	var mSolenoid = 0.0;
	var mThinExtremes = true;
	var mLoops;
	var mPointsPerLoop;

	self.getClass = function() {
		return "ElementSpring";
	}

	/***
	 * Set radius
	 * @method setRadius
 	 * @param radius
	 */
	self.setRadius = function(radius) {
		if(mRadius != radius) {
			mRadius = radius;
			self.setChanged(true);
		}		
	}

	self.getRadius = function() {
		return mRadius;
	}

	self.setSolenoid = function(solenoid) {
		if(mSolenoid != solenoid) {
			mSolenoid = solenoid;
			self.setChanged(true);
		}
	}

	self.getSolenoid = function() {
		return mSolenoid;
	}

	self.setThinExtremes = function(thinExtremes) {
		if(mThinExtremes != thinExtremes) {
			mThinExtremes = thinExtremes;
			self.setChanged(true);
		}
	}

	self.getThinExtremes = function() {
		return mThinExtremes;
	}

	self.setLoops = function(loops) {
		if(mLoops != loops) {
			mLoops = loops;
			self.setChanged(true);
		}
	}

	self.getLoops = function() {
		return mLoops;
	}

	self.setPointsPerLoop = function(pointsPerLoop) {
		if(mPointsPerLoop != pointsPerLoop) {
			mPointsPerLoop = pointsPerLoop;
			self.setChanged(true);
		}
	}

	self.getPointsPerLoop = function() {
		return mPointsPerLoop;
	}

	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.Spring.registerProperties(self, controller);
	};

  	self.copyTo = function(element) {
    	EJSS_DRAWING2D.Spring.copyTo(self,element);
  	};

	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

  	self.setSize([0.1,0.1]);
  	self.setLoops(EJSS_DRAWING2D.Spring.DEF_LOOPS);
  	self.setPointsPerLoop(EJSS_DRAWING2D.Spring.DEF_PPL);
	self.setRelativePosition("SOUTH_WEST");
	
	return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//SpringSet
//---------------------------------

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * SpringSet
 * @class SpringSet 
 * @constructor  
 */
EJSS_DRAWING2D.SpringSet = {
	
    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING2D.ElementSet;
      
      ElementSet.registerProperties(set,controller);
      controller.registerProperty("Radius", 
          function(v) { set.setToEach(function(element,value) { element.setRadius(value); }, v); }
      );
      controller.registerProperty("Solenoid", 
          function(v) { set.setToEach(function(element,value) { element.setSolenoid(value); }, v); }
      );           
      controller.registerProperty("ThinExtremes", 
          function(v) { set.setToEach(function(element,value) { element.setThinExtremes(value); }, v); }
      );           
      controller.registerProperty("Loops", 
          function(v) { set.setToEach(function(element,value) { element.setLoops(value); }, v); }
      );           
      controller.registerProperty("PointsPerLoop", 
          function(v) { set.setToEach(function(element,value) { element.setPointsPerLoop(value); }, v); }
      );           
    }        
};

/**
 * Creates a set of Segments
 * @method springSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING2D.springSet = function (mName) {
  var self = EJSS_DRAWING2D.elementSet(EJSS_DRAWING2D.spring, mName);

  // Static references
  var SpringSet = EJSS_DRAWING2D.SpringSet;		// reference for SpringSet
  
  self.registerProperties = function(controller) {
    SpringSet.registerProperties(self,controller);
  };

  return self;
};/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Shape
 * @class Shape 
 * @constructor  
 */
EJSS_DRAWING2D.Tank = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class

      controller.registerProperty("LevelColor",element.setLevelColor);
      controller.registerProperty("Level",element.setLevel);
      controller.registerProperty("Closed",element.setClosed);
    },

    /**
     * static copyTo method, to be used by sets
     */
    copyTo : function(source, dest) {
      EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy

      dest.setLevel(source.getLevel());
      dest.setLevelColor(source.getLevelColor());
    }

};

/**
 * Creates a 2D shape
 * @method shape
 */
EJSS_DRAWING2D.tank = function (name) {
  var self = EJSS_DRAWING2D.element(name);

  var mLevel = 0;
  var mLevelColor = "blue";

  // Implementation variables
  var Tank = EJSS_DRAWING2D.Tank;

  self.getClass = function() {
  	return "ElementTank";
  }

  // ----------------------------------------------------
  // public functions
  // ----------------------------------------------------

  self.setLevel = function(level) {
      mLevel = level;
      self.setChanged(true);
  };

  self.getLevel = function() {
  	return mLevel;
  }

  /**
   * Set the color of the level
   * @param color a stroke style
   */
  self.setLevelColor = function(color) { 
    if (typeof color !== "string") color = EJSS_TOOLS.DisplayColors.getLineColor(color);
    if (color!=mLevelColor) {
      mLevelColor = color; 
      self.setChanged(true);
    }
    return self;
  };
    
  /**
   * Get the line color
   */
  self.getLevelColor = function() { 
    return mLevelColor; 
  };
  
  // ----------------------------------------------------
  // Properties and copies
  // ----------------------------------------------------

  /**
   * Extended registerProperties method. To be used by promoteToControlElement
   * @method registerProperties
   */
  self.registerProperties = function(controller) {
    Tank.registerProperties(self,controller);
  };

  /**
   * Copies itself to another element
   * Extended copyTo method. To be used by Sets
   * @method copyTo
   */
  self.copyTo = function(element) {
    Tank.copyTo(self,element);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([0.1,0.1]);
  self.setRelativePosition("CENTER");
  self.getStyle().setFillColor("White");
  self.getStyle().setLineColor("Black");
  self.getStyle().setDrawFill(false);
    
  return self;
};


/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * A Text is a 2D drawing element that displays a text 
 * @class EJSS_DRAWING2D.Text
 * @parent EJSS_DRAWING2D.Element
 */
EJSS_DRAWING2D.Text = {
	MODE_TOPDOWN : 0,
	MODE_RIGTHLEFT : 1,
	MODE_DOWNTOP : 2,
	MODE_LEFTRIGHT: 3,
	
	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy  	
  		EJSS_DRAWING2D.Font.copyTo(source.getFont(),dest.getFont());
  		
		dest.setText(source.getText());
		dest.setWritingMode(source.getWritingMode());
  	},
  	
	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class

	 	/*** 
	  	* Text
	  	* @property Text 
	  	* @type String
	  	* @default "" 
	  	*/  
		controller.registerProperty("Text", element.setText);
		controller.registerProperty("Framed", element.setFramed);
		controller.registerProperty("WritingMode", element.setWritingMode);

	 	/*** 
	  	* Font description in one declaration
	  	* @property Font 
	  	* @type String - https://www.w3schools.com/cssref/pr_font_font.asp
	  	* @default "20px Arial" 
	  	*/  
		controller.registerProperty("Font", function (e) {
				element.setDrawingSize([-1,-1]);
				element.getFont().setFont(e);
			});

	 	/*** 
	  	* Font family
	  	* @property FontFamily 
	  	* @type String
	  	* @default "Arial" 
	  	*/  
		controller.registerProperty("FontFamily", element.getFont().setFontFamily);

	 	/*** 
	  	* Font size
	  	* @property FontSize 
	  	* @type String
	  	* @default "20" 
	  	*/  
		controller.registerProperty("FontSize", function (e) {
				element.setDrawingSize([-1,-1]);
				element.getFont().setFontSize(e);
			});

		controller.registerProperty("LetterSpacing", element.getFont().setLetterSpacing);

	 	/*** 
	  	* Outline color
	  	* @property OutlineColor 
	  	* @type String
	  	* @default "none" 
	  	*/  
		controller.registerProperty("OutlineColor", element.getFont().setOutlineColor);

		controller.registerProperty("FontWeight", element.getFont().setFontWeight);

	 	/*** 
	  	* Fill color
	  	* @property FillColor 
	  	* @type String
	  	* @default "none" 
	  	*/  
		controller.registerProperty("FillColor", element.getFont().setFillColor);

	 	/*** 
	  	* Font style
	  	* @property FontStyle 
	  	* @type String
	  	* @default "normal" 
	  	*/  
		controller.registerProperty("FontStyle", element.getFont().setFontStyle);

		controller.registerProperty("MarginX", element.getMarginX().setMarginX);
		controller.registerProperty("MarginY", element.getMarginY().setMarginY);
	},
};

/**
 * Creates a 2D Text
 * @method text
 */
EJSS_DRAWING2D.text = function(mName) {
	var self = EJSS_DRAWING2D.element(mName);

	var mText = "";
	var mWritingMode = EJSS_DRAWING2D.Text.MODE_LEFTRIGHT;
	var mFont = EJSS_DRAWING2D.font(mName);	// font for text
	var mFramed = false; 
	var mMarginX = 0;
	var mMarginY = 0;
	
	var mDrawingSize = [-1,-1]; // autocalculated 

	self.getClass = function() {
		return "ElementText";
	}

  	/***
   	* Sets text
   	* @method setText
   	* @visibility public
   	* @param string
   	*/
	self.setText = function(text) {
		text = text + ""; // convert text to string
		if(mText != text) {
			mText = text;
			self.setChanged(true);
			mDrawingSize = [-1,-1];
		}
	}

  	/***
   	* Gets text
   	* @method getText
   	* @visibility public
   	* @return string
   	*/
	self.getText = function() {
		return mText;
	}

	self.setDrawingSize = function(drawingSize) {
		mDrawingSize = drawingSize;
	}

	self.getDrawingSize = function() {
		return mDrawingSize;
	}

	self.getSizeX = function() {		
		return mDrawingSize[0]; 
	}

	self.getSizeY = function() {
		return mDrawingSize[1];
	}

	self.setMarginX = function(margin) {
		if(mMarginX != margin) {
			mMarginX = margin;
			self.setChanged(true);
		}
	}

	self.getMarginX = function() {
		return mMarginX;
	}

	self.setMarginY = function(margin) {
		if(mMarginY != margin) {
			mMarginY = margin;
			self.setChanged(true);
		}
	}

	self.getMarginY = function() {
		return mMarginY;
	}

	self.getFont = function() {
		return mFont;
	}

	self.setFramed = function(framed) {
		if(mFramed != framed) {
			mFramed = framed;
			self.setChanged(true);
		}
	}

	self.getFramed = function() {
		return mFramed;
	}

	self.setWritingMode = function(writemode) {
	    if (typeof writemode == "string") writemode = EJSS_DRAWING2D.Text[writemode.toUpperCase()];
	    if (mWritingMode != writemode) {
	      mWritingMode = writemode;
	      self.setChanged(true);
	    }
	}

	self.getWritingMode = function() {
		return mWritingMode;
	}

	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.Text.registerProperties(self, controller);
	};

  	self.copyTo = function(element) {
    	EJSS_DRAWING2D.Text.copyTo(self,element);
  	};

	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

  	mFont.setChangeListener(function (change) { self.setChanged(true); });
	self.setPixelSize(true); 
	return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//TextSet
//---------------------------------

/**
 * TextSet
 * @class TextSet 
 * @constructor  
 */
EJSS_DRAWING2D.TextSet = {
	
    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING2D.ElementSet;
      
      ElementSet.registerProperties(set,controller);
      
      controller.registerProperty("Text", 
          function(v) { set.setToEach(function(element,value) { element.setText(value); }, v); }
      );
      controller.registerProperty("Framed", 
          function(v) { set.setToEach(function(element,value) { element.setFramed(value); }, v); }
      );           
      controller.registerProperty("WritingMode", 
          function(v) { set.setToEach(function(element,value) { element.setWritingMode(value); }, v); }
      );           
      controller.registerProperty("Font", 
          function(v) { set.setToEach(function(element,value) { element.getFont().setFont(value); }, v); }
      );          
      controller.registerProperty("FontFamily", 
          function(v) { set.setToEach(function(element,value) { element.getFont().setFontFamily(value); }, v); }
      );           
      controller.registerProperty("FontSize", 
          function(v) { set.setToEach(function(element,value) { element.getFont().setFontSize(value); }, v); }
      );           
      controller.registerProperty("LetterSpacing", 
          function(v) { set.setToEach(function(element,value) { element.getFont().setLetterSpacing(value); }, v); }
      );           
      controller.registerProperty("OutlineColor", 
          function(v) { set.setToEach(function(element,value) { element.getFont().setOutlineColor(value); }, v); }
      );           
      controller.registerProperty("FontWeight", 
          function(v) { set.setToEach(function(element,value) { element.getFont().setFontWeight(value); }, v); }
      );           
      controller.registerProperty("FillColor", 
          function(v) { set.setToEach(function(element,value) { element.getFont().setFillColor(value); }, v); }
      );           
    }         
    
};


/**
 * Creates a set of Texts
 * @method textSet
 * @param mName
 */
EJSS_DRAWING2D.textSet = function (mName) {
  var self = EJSS_DRAWING2D.elementSet(EJSS_DRAWING2D.text, mName);

  // Static references
  var TextSet = EJSS_DRAWING2D.TextSet;		// reference for TextSet

  /**
   * Registers properties in a ControlElement
   * @method registerProperties
   * @param controller A ControlElement that becomes the element controller
   */
  self.registerProperties = function(controller) {
    TextSet.registerProperties(self,controller);
  };

  return self;
};

/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * A Trace is a sophistication of the Trail element. 
 * The extra features are the possibility to specify markers that will be displayed at 
 * each point of the trace.
 * @class EJSS_DRAWING2D.Trace 
 * @parent EJSS_DRAWING2D.Trail
 * @constructor  
 */
EJSS_DRAWING2D.Trace = {
  ELLIPSE : 0, 	
  RECTANGLE : 1,	
  AREA : 2,	
  BAR : 3,

	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy  	
  		EJSS_DRAWING2D.Trail.copyTo(source,dest);
  		EJSS_DRAWING2D.Style.copyTo(source.getMarkStyle(),dest.getMarkStyle());
  		
		dest.setMarkType(source.getMarkType());
		dest.setMarkSize(source.getMarkSize());
  	},
  	
	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Trail.registerProperties(element, controller);
		// super class

	 /*** 
	  * Type of marker to draw
	  * @property MarkType 
	  * @type int|String
	  * @values 0="ELLIPSE", 1="RECTANGLE", 2:"AREA", 3:"BAR" 
	  * @default "ELLIPSE" 
	  */ 
	  controller.registerProperty("MarkType", element.setMarkType, element.getMarkType);

	 /*** 
	  * Axis for bar or area markers
	  * @property MarkAxisY 
	  * @type int
	  * @default 0 
	  */ 
	  controller.registerProperty("MarkAxisY", element.setMarkAxisY, element.getMarkAxisY);

	 /*** 
	  * Size of the marker to draw
	  * @property MarkSize 
	  * @type int[2] providing the width and height in pixels
	  * @default [0,0] 
	  */ 
		controller.registerProperty("MarkSize", element.setMarkSize, element.getMarkSize);

	 /*** 
	  * Color for the lines of the markers
	  * @property MarkLineColor
	  * @type String
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "Black"
	  */ 
      	controller.registerProperty("MarkLineColor",  element.getMarkStyle().setLineColor);

	 /*** 
	  * Marker stroke width
	  * @property MarkLineWidth 
	  * @type double
	  * @default 0.5
	  */ 
      	controller.registerProperty("MarkLineWidth",  element.getMarkStyle().setLineWidth);

	 /*** 
	  * Whether the marker lines are drawn
	  * @property MarkDrawLines 
	  * @type boolean
	  * @default true 
	  */ 
      	controller.registerProperty("MarkDrawLines",  element.getMarkStyle().setDrawLines);

	 /*** 
	  * The fill color for the markers
	  * @property MarkFillColor 
	  * @type String
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "Blue"
	  */ 
      	controller.registerProperty("MarkFillColor",  element.getMarkStyle().setFillColor);

	 /*** 
	  * Whether the marker are filled
	  * @property MarkDrawFill 
	  * @type boolean
	  * @default true
	  */ 
      	controller.registerProperty("MarkDrawFill",   element.getMarkStyle().setDrawFill);

	}

};

/**
 * Creates a 2D Trace
 * @method trace
 */
EJSS_DRAWING2D.trace = function(mName) {
	var self = EJSS_DRAWING2D.trail(mName);
 
    // Instance variables
	var mMarkType = EJSS_DRAWING2D.Trace.ELLIPSE;
    var mMarkStyle = EJSS_DRAWING2D.style(mName);	// style for mark
    var mMarkRelativePosition = EJSS_DRAWING2D.Element.CENTER;  
	var mMarkSizeX = 0;
	var mMarkSizeY = 0;
	var mMarkAxisY = 0;
	var mMarkStyleList = [];
		 
	self.getClass = function() {
		return "ElementTrace";
	}

	/***
	 * Set a list of styles
	 * @param list
     * @visibility public
	 */
	self.setMarkStyleList = function(list) {
	    if (typeof list == "undefined" || list === null) { 
	    	mMarkStyleList = [];
	    } else if (Array.isArray(list)) {  // list
	      	mMarkStyleList = list.slice();
	      	self.setChanged(true);
	    } 
	};
	
	/***
	 * Get the list of styles
	 * @method getMarkStyleList()
	 * @return list
     * @visibility public
	 */
	self.getMarkStyleList = function() {
	  return mMarkStyleList;
	};

	/**
	 * Sets the mark relative position of the element with respect to its (x,y) coordinates.
	 */
	self.setMarkRelativePosition = function(position) {
	  if (typeof position == "string") position = EJSS_DRAWING2D.Element[position.toUpperCase()];
	  if (mMarkRelativePosition != position) {
	    mMarkRelativePosition = position;
	  	self.setChanged(true);
	  }
	};
	  
	self.getMarkRelativePosition = function() { 
	  return mMarkRelativePosition;
	};

	/***
	 * Return the style of the mark
	 * @method getMarkStyle() 
     * @visibility public
	 */
	self.getMarkStyle = function() { 
	  return mMarkStyle; 
	};

	/***
	 * Set the size of the mark
	 * @method setMarkSize(size)
	 * @param size int[2]|double[2] array with the size in pixels
     * @visibility public
	 */
	self.setMarkSize = function(size) {
	  if((mMarkSizeX != size[0]) || (mMarkSizeY != size[1])) { 
	  	mMarkSizeX = size[0];
	  	mMarkSizeY = size[1];
	  	self.setChanged(true);
	  }
	};
	
	/***
	 * Get the sizes of the mark
	 * @method getMarkSize()
	 * @return double[]
     * @visibility public
	 */
	self.getMarkSize = function() {
	  return [mMarkSizeX, mMarkSizeY];
	};

	/***
	 * Set the type of the mark
	 * @method setMarkType(type)
	 * @param type int|String One of: "CENTER":0,"NORTH":1,"SOUTH":2,"EAST":3,"WEST":4,"NORTH_EAST":5,"NORTH_WEST":6,
	 *                 "SOUTH_EAST":7,"SOUTH_WEST":8 
     * @visibility public
	 */
	self.setMarkType = function(type) {
	  if (typeof type == "string") type = EJSS_DRAWING2D.Trace[type.toUpperCase()];
	  if(mMarkType != type) {
	  	mMarkType = type;
	  	self.setChanged(true);
	  }
	};
	
	/***
	 * Get the type of the mark
	 * @method getMarkType()
	 * @return int
     * @visibility public
	 */
	self.getMarkType = function() {
	  return mMarkType;
	};

	/***
	 * Set the limit axis for bar or area marks
	 * @method setMarkAxisY()
	 * @param int
     * @visibility public
	 */
	self.setMarkAxisY = function(axisY) {
	  if(mMarkAxisY != axisY) {
	  	mMarkAxisY = axisY;
	  	self.setChanged(true);
	  }
	};
  
	/***
	 * Get the limit axis for bar or area marks
	 * @method getMarkAxisY()
	 * @return int
     * @visibility public
	 */
	self.getMarkAxisY = function() {
	  return mMarkAxisY;
	};
  
   self.formatPoint = function(src) {
   		// override
   		var size = src.length;
   		src[size++] = mMarkType;
   		src[size] = EJSS_DRAWING2D.style();
   		EJSS_DRAWING2D.Style.copyTo(mMarkStyle,src[size++]);
		src[size++] = mMarkSizeX;
		src[size++] = mMarkSizeY;
		src[size] = mMarkAxisY;
		return src;
   }
  
	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.Trace.registerProperties(self, controller);
	};
  
  	self.copyTo = function(element) {
    	EJSS_DRAWING2D.Trace.copyTo(self,element);
  	};
  
	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//TraceSet
//---------------------------------

/**
 * TraceSet
 * @class TraceSet 
 * @constructor  
 */
EJSS_DRAWING2D.TraceSet = {
  
    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var TrailSet = EJSS_DRAWING2D.TrailSet;
      
      TrailSet.registerProperties(set,controller);
      
	 /*** 
	  * Type of marker to draw
	  * @property MarkType 
	  * @type int|String
	  * @values 0="ELLIPSE", 1="RECTANGLE", 2:"AREA", 3:"BAR" 
	  * @default "ELLIPSE" 
	  */ 
	  controller.registerProperty("MarkType", 
          function(v) { set.setToEach(function(element,value) { element.setMarkType(value); }, v); }
      );
      
	 /*** 
	  * Size of the marker to draw
	  * @property MarkSize 
	  * @type int[2] providing the width and height in pixels
	  * @default [0,0] 
	  */ 
	  controller.registerProperty("MarkSize", 
          function(v) { set.setToEach(function(element,value) { element.setMarkSize(value); }, v); }
      );           
      
	 /*** 
	  * Color for the lines of the markers
	  * @property MarkLineColor
	  * @type String
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "Black"
	  */ 
	  controller.registerProperty("MarkLineColor", 
          function(v) { set.setToEach(function(element,value) { element.getMarkStyle().setLineColor(value); }, v); }
      );   
      
      /*** 
	  * Marker stroke width
	  * @property MarkLineWidth 
	  * @type double
	  * @default 0.5
	  */         
      controller.registerProperty("MarkLineWidth", 
          function(v) { set.setToEach(function(element,value) { element.getMarkStyle().setLineWidth(value); }, v); }
      );           
      
      /*** 
	  * Whether the marker lines are drawn
	  * @property MarkDrawLines 
	  * @type boolean
	  * @default true 
	  */
      controller.registerProperty("MarkDrawLines", 
          function(v) { set.setToEach(function(element,value) { element.getMarkStyle().setDrawLines(value); }, v); }
      );           
      
      /*** 
	  * The fill color for the markers
	  * @property MarkFillColor 
	  * @type String
	  * @see http://www.w3schools.com/cssref/css_colornames.asp
	  * @default "Blue"
	  */
	  controller.registerProperty("MarkFillColor", 
          function(v) { set.setToEach(function(element,value) { element.getMarkStyle().setFillColor(value); }, v); }
      );           
      
      /*** 
	  * Whether the marker are filled
	  * @property MarkDrawFill 
	  * @type boolean
	  * @default true
	  */
	  controller.registerProperty("MarkDrawFill", 
          function(v) { set.setToEach(function(element,value) { element.getMarkStyle().setDrawFill(value); }, v); }
      );           
      
      /*** 
	  * Position of the marker relative to the point 
	  * @property MarkRelativePosition 
	  * @type int|String
	  * @values "CENTER":0,"NORTH":1,"SOUTH":2,"EAST":3,"WEST":4,"NORTH_EAST":5,"NORTH_WEST":6,
	  * "SOUTH_EAST":7,"SOUTH_WEST":8 
	  * @default "CENTER
	  */
	  controller.registerProperty("MarkRelativePosition", 
          function(v) { set.setToEach(function(element,value) { element.setMarkRelativePosition(value); }, v); }
      );           
    }    
};


/**
 * Creates a set of Segments
 * @method traceSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING2D.traceSet = function (mName) {
  var self = EJSS_DRAWING2D.elementSet(EJSS_DRAWING2D.trace, mName);

  // Static references
  var TraceSet = EJSS_DRAWING2D.TraceSet;		// reference for TraceSet
  
  self.registerProperties = function(controller) {
    TraceSet.registerProperties(self,controller);
  };

  return self;
};/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*
* This code is Open Source and is provided "as is".
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/***
 * A Trail is a two-dimensional drawable that displays a collection of points in the plane. 
 * The points are added to the trail either through the <tt>InputX</tt> and <tt>InputY</tt>
 * properties, or using the element’s <tt>addPoint</tt> or <tt>moveToPoint</tt> functions. 
 * The points are displayed using lines which connect the points, each with the previous one, 
 * thus forming a polygonal line (which looks like a trail). The trail can be disconnected 
 * at some points if the <tt>Connected</tt> property is set temporarily to false, or if 
 * the <tt>moveToPoint</tt> function is used.<br/> 
 * The number of points in a trail can be limited through the <tt>Maximum</tt> property. 
 * Adding points past the limit causes the trail to remove the first points in it.<br/>
 *<br/>
 * Trails can be broken into several segments using the <tt>newSegment</tt> function. 
 * Each segment behaves like a sub-trail, but segments can have different drawing 
 * styles (line width and color). Also, the user can delete the points in the last 
 * segment without affecting the previous segments.
 * @class EJSS_DRAWING2D.Trail 
 * @parent EJSS_DRAWING2D.Element
 * @constructor  
 */
EJSS_DRAWING2D.Trail = {
  NO_CONNECTION : 0, 	// The next point will not be connected to the previous one
  LINE_CONNECTION : 1,	// The next point will be connected to the previous one by a segment

	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING2D.Element.copyTo(source,dest); // super class copy  	
  		
		dest.setActive(source.getActive());
		dest.setNoRepeat(source.getNoRepeat());
		dest.setClearAtInput(source.getClearAtInput());
		dest.setSkip(source.getSkip());
		dest.setInputLabels(source.getInputLabels());
		dest.setMaximumPoints(source.getMaximumPoints());
		dest.setConnectionType(source.getConnectionType());
		dest.setPoints(source.getPoints());
  	},
  	
	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING2D.Element.registerProperties(element, controller);
		// super class

	 /*** 
	  * Whether the trail should accept input points. 
	  * @property Active
	  * @type boolean
	  * @default true 
	  */  
		controller.registerProperty("Active", element.setActive, element.getActive);

	 /*** 
	  * Whether the trail ignores repeated input points 
	  * @property NoRepeat
	  * @type boolean
	  * @default false 
	  */  
		controller.registerProperty("NoRepeat", element.setNoRepeat, element.getNoRepeat);

	 /*** 
	  * Whether the trail clears all points when receiving new input. (Useful if input is an array.)
	  * @property ClearAtInput
	  * @type boolean
	  * @default false 
	  */  
		controller.registerProperty("ClearAtInput", element.setClearAtInput, element.getClearAtInput);

	 /*** 
	  * Whether the trail should display only one oout of many inpout points. 
	  * If Skip is positive, it indicates how many points the trail will take to display one.
	  * @property Skip
	  * @type int
	  * @default 0 
	  */  
		controller.registerProperty("Skip", element.setSkip, element.getSkip);

		controller.registerProperty("ColumnNames", element.setInputLabels, element.getInputLabels);

	 /*** 
	  * The maximum number of points the trail accepts. If more input is received, the trail
	  * will discard the corresponding first points.
	  * @property Maximum
	  * @type int
	  * @default 0 indicating no limit
	  */  
		controller.registerProperty("Maximum", element.setMaximumPoints, element.getMaximumPoints);

		controller.registerProperty("ConnectionType", element.setConnectionType, element.getConnectionType);		

	 /*** 
	  * Whether the next input point should be connected to the previous one. 
	  * @property Connected
	  * @type boolean
	  * @default true 
	  */  		
	  controller.registerProperty("Connected", element.setConnected, element.isConnected);		
		
		controller.registerProperty("Points", element.setPoints, element.getPoints);		
		controller.registerProperty("LastPoint", element.addPoint, element.getLastPoint);

	 /*** 
	  * The next input point for the trail. 
	  * @property Input
	  * @type int[2]|double[2] An array with a pair of [x,y] coordinates, 
	  * or int[n][2]|double[n][2] a double array with n pairs of [x,y] coordinates. 
	  * @default "none" 
	  */ 
	  controller.registerProperty("Input", element.addPoints);

	 /*** 
	  * The X coordinate for the next input point for the trail. 
	  * @property InputX
	  * @type int|double The X coordinate for the input point 
	  * or int[n]|double[n] a double array with the X coordinates for n input points. 
	  * @default "none" 
	  */ 
	  controller.registerProperty("InputX", element.addXPoints);				

	 /*** 
	  * The Y coordinate for the next input point for the trail. 
	  * @property InputY
	  * @type int|double The Y coordinate for the input point 
	  * or int[n]|double[n] a double array with the Y coordinates for n input points. 
	  * @default "none" 
	  */ 
	  controller.registerProperty("InputY", element.addYPoints);
	}
			
};

/**
 * Creates a 2D Trail
 * Trail implements dataCollected(), which makes it a Collector of data 
 * @see _view._collectData
 * @method trail
 */
EJSS_DRAWING2D.trail = function(name) {
	var self = EJSS_DRAWING2D.element(name);
 
    // Configuration variables
	var mActive = true;
	var mNoRepeat = false;
	var mClearAtInput = false;
	var mSkip = 0;
	var mInputLabels = ["x", "y"];	  
	var mMaximum = 0;
	var mConnectionType = EJSS_DRAWING2D.Trail.LINE_CONNECTION;
	
  	// Implementation variables
  	var mCounter = 0;
  	var mIsEmpty = true;
	//  private int firstPoint = 0; // the first point of the current path 
  	var mLastPoint = 0;	
  	var mFlushPoint = 0; // This is the last point which was not added because of the skip parameter  	
  	var mCurrentList = []; 	// The current list of points  	
  	var mSegmentList = []; // The list of past Segments, if any  	
  	var mSegmentStyle = []; // The list of past segment styles, if any
	var mTmpList = []; // The temporal list of new points
	var mTmpLastXAdded = 0;
	var mTmpLastYAdded = 0;
	 
	self.getClass = function() {
		return "ElementTrail";
	}

  /***
   * Sets the active state of the trail. An inactive trail ignores all input. 
   * @method setActive(active)
   * @param active boolean
   * @visibility public
   */
  self.setActive = function(active) {
    if(mActive != active) {
    	mActive = active;
    	self.setChanged(true);
    }
  }
  
  /***
   * Whether the trail is in active mode.
   * @method getActive()
   * @return boolean
   */
  self.getActive = function() { 
  	return mActive; 
  }
  
  /***
   * Sets the no repeat state of the trail.
   * When set, a trail will ignore (x,y) points which equal the last added point.
   * @method setNoRepeat(noRepeat)
   * @param noRepeat boolean
   * @visibility public
   */
  self.setNoRepeat = function (noRepeat) {
    if(mNoRepeat != noRepeat) {
    	mNoRepeat = noRepeat;
    	self.setChanged(true);
    }
  }
  
  /***
   * Whether the trail is in no repeat mode.
   * @method getNoRepeat()
   * @return boolean
   */
  self.getNoRepeat = function() { 
  	return mNoRepeat; 
  }

  /***
   * Sets the trail to clear existing points when receiving 
   * a new point or array of points.
   * @method setClearAtInput(clear)
   * @param clear boolean
   * @visibility public
   */
  self.setClearAtInput = function(clear) {
    if(mClearAtInput != clear) {
    	mClearAtInput = clear;
    	self.setChanged(true);
    }
  }
  
  /***
   * Whether the trail is in clear at input mode.
   * @method getClearAtInput()
   * @return boolean
   * @visibility public
   */
  self.getClearAtInput = function() { 
  	return mClearAtInput; 
  }

  /***
   * Sets the skip parameter. When the skip parameter is larger than zero,
   * the trail only considers one of every 'skip' points. That is, if skip is 3, 
   * the trail will consider only every third point sent to it. 
   * The default is zero, meaning all points must be considered.
   * @method setSkip(skip)
   * @param skip int
   * @visibility public
   */
  self.setSkip = function (skip) {
    if (mSkip != skip) {
      mSkip = skip;
      mCounter = 0;
      self.setChanged(true);
    }
  }
  
  /***
   * Returns the skip parameter of the trail.
   * @method getSkip()
   * @return int
   * @visibility public
   */
  self.getSkip = function () { 
  	return mSkip; 
  }

  /**
   * Sets the labels of the X,Y coordinates when the data is displayed in a table
   * @method setInputLabels(label)
   * @param label
   */
  self.setInputLabels = function (label) {
  	if((mInputLabels[0] != label[0]) || (mInputLabels[1] != label[1])) { 
  		mInputLabels[0] = label[0]; 
  		mInputLabels[1] = label[1]; 
  		self.setChanged(true);
  	}
  }

  /**
   * Returns the labels of the X,Y coordinates
   * @method getInputLabels()
   * @return labels
   */
  self.getInputLabels = function () { 
  	return mInputLabels; 
  }
  
  /***
   * Sets the maximum number of points for the trail.
   * Once the maximum is reached, adding a new point will cause
   * remotion of the first one. This is useful to keep trails
   * down to a reasonable size, since very long trails can slow
   * down the rendering (in certain implementations).
   * If the value is 0 (the default) the trail grows forever
   * without discarding old points.
   * @method setMaximumPoints(maximum)
   * @param maximum int
   * @visibility public
   */
  self.setMaximumPoints = function(maximum) {
  	var max = Math.max(maximum, 2);
  	if(mMaximum != max) {
    	mMaximum = max;
    	self.setChanged(true);
    }
  }

  /***
   * Returns the maximum number of points allowed for the trail
   * @method getMaximumPoints()
   * @return int
   * @visibility public
   */
  self.getMaximumPoints = function() {
    return mMaximum;
  }

  /***
   * Whether to connect next input point with the previous one
   * @method setConnected(connected)
   * @param connected boolean
   * @visibility public
   */
  self.setConnected = function (connected) {
   if (connected) self.setConnectionType(EJSS_DRAWING2D.Trail.LINE_CONNECTION);
   else self.setConnectionType(EJSS_DRAWING2D.Trail.NO_CONNECTION);
  }

  /***
   * Gets the connection state.
   * @method getConnected()
   * @return boolean
   * @visibility public
   */
  self.isConnected = function() {
    return mConnectionType===EJSS_DRAWING2D.Trail.LINE_CONNECTION;
  }

  /**
   * Sets the type of connection for the next point.
   * @method setConnectionType(type)
   * @param type int
   */
  self.setConnectionType = function (type) {
    if (typeof type == "string") type = EJSS_DRAWING2D.Trail[type.toUpperCase()];      	
    if(mConnectionType != type) {
    	mConnectionType = type;
    	self.setChanged(true);
    }
  }

  /**
   * Gets the connection type.
   * @method getConnectionType()
   * @see #setConnectionType(int)
   */
  self.getConnectionType = function() {
    return mConnectionType;
  }

  /***
   * Adds a new point to the trail.
   * @method addPoint(x,y,style)
   * @param x double The X coordinate of the point 
   * 		or point double[] The double[2] array with the coordinates of the point.
   * @param y double The Y coordinate of the point.
   * @param style int an optional connection style: 0 = EJSS_DRAWING2D.Trail.NO_CONNECTION, 1 = EJSS_DRAWING2D.Trail.LINE_CONNECTION
   * @visibility public
   */
  self.addPoint = function (x, y, style) {
    if (!mActive) return;
  	if (mClearAtInput) self.initialize();
    if(x instanceof Array)    	
    	addPoint(x[0], x[1], x[2]);
    else 
    	addPoint (x,y,style);
    self.setChanged(true);
  }


  /***
   * Adds an array of points to the trail.
   * @method addPoints(x,y)
   * @param x double The double[] array with the X coordinates of the points.
   * 	or  point double[][] The double[nPoints][2] array with the coordinates of the points.
   * @param y double The double[] array with the Y coordinates of the points.
   * @visibility public
   * @param len int The number of points to copy.
   * @visibility public
   */
  self.addPoints  = function(x, y, len) {
    if (!mActive) return;
  	if (mClearAtInput) self.initialize();
  	// number of points to copy
  	if (typeof len == "undefined") len = Number.MAX_SAFE_INTEGER;
	if ((typeof y == "undefined") || (y === null)) {
      if(x[0] instanceof Array) {    
    	for (var i=0,n=Math.min(len,x.length); i<n; i++) addPoint (x[i][0],x[i][1],x[i][2]);  		
  	  }
  	  else addPoint (x[0],x[1]);
	}
	else { // y is defined
    	for (var i=0,n=Math.min(len,x.length,y.length); i<n; i++) addPoint (x[i],y[i]);
  	}
    self.setChanged(true);
  }

  /**
   * Sets the array of X points for the input of the trail.
   * @method addXPoints(x)
   */
  self.addXPoints  = function(x) {
    if (!mActive) return;
  	var i = 0;
  	if(x instanceof Array) {
  		// fill points with the values  		
    	for (var n=x.length; i<n; i++) {
    		if(mTmpList[i]) mTmpList[i] = [x[i],mTmpList[i][1],mTmpList[i][2]];
    		else mTmpList[i] = [x[i],mTmpLastYAdded,mConnectionType];
    	}  		
    	mTmpLastXAdded = x[i];
  	}
    else { // only one value
    	if(mTmpList[i]) mTmpList[i] = [x,mTmpList[i][1],mTmpList[i][2]];
    	else mTmpList[i] = [x,mTmpLastYAdded,mConnectionType];
    	i++;
    	mTmpLastXAdded = x;    	
    }

	// fill points with the last value
	while (mTmpList[i]) {
		mTmpList[i] = [mTmpLastXAdded,mTmpList[i][1],mTmpList[i][2]];
		i++;
	}    	
  }

  /**
   * Adds an array of Y points to the trail.
   * @method addYPoints(y)
   */
  self.addYPoints  = function(y) {
    if (!mActive) return;
  	var i = 0;
  	if(y instanceof Array) {
  		// fill points with the values  		
    	for (var n=y.length; i<n; i++) {
    		if(mTmpList[i]) mTmpList[i] = [mTmpList[i][0],y[i],mTmpList[i][2]];
    		else mTmpList[i] = [mTmpLastXAdded,y[i],mConnectionType];
    	}  		
    	mTmpLastYAdded = y[i];
  	}
    else { // only one value
    	if(mTmpList[i]) mTmpList[i] = [mTmpList[i][0],y,mTmpList[i][2]];
    	else mTmpList[i] = [mTmpLastXAdded,y,mConnectionType];
    	i++;
    	mTmpLastYAdded = y;    	
    }

	// fill points with the last value
	while (mTmpList[i]) {
		mTmpList[i] = [mTmpList[i][0],mTmpLastYAdded,mTmpList[i][2]];
		i++;
	}    	
  }

  /**
   * Adds the temporal array of point to the trail.
   * @method dataCollected()
   */
  self.dataCollected  = function() {
  	if (mTmpList.length > 0) {
  	  self.addPoints(mTmpList);
      mTmpList = [];
  	  self.setChanged(true);
  	}
  }  

  /**
   * Sets an array of points to the trail. Equivalent to clear() + addPoints(x,y)
   * @method setPoints(x,y)
   * @param xInput double The double[] array with the X coordinates of the points.
   * 	or  point double[][] The double[nPoints][2] array with the coordinates of the points.
   * @param yInput double The double[] array with the Y coordinates of the points.
   */
  self.setPoints  = function(x, y) {
    if (!mActive) return;
  	self.clear();
  	self.addPoints(x,y);
  	self.setChanged(true);
  }
    
  /***
   * Moves to the new point without drawing.
   * (Equivalent to setting the connection type
   * to NO_CONNECTION and adding one single point, then setting the 
   * type back to its previous value.)
   * @method moveToPoint(x,y)
   * @param x double The X coordinate of the point.
   * 	or point double[] The double[2] array with the coordinates of the point.
   * @param y double The Y coordinate of the point.
   * @visibility public
   */
  self.moveToPoint = function(x, y) { 
    if (!mActive) return;
  	if (mClearAtInput) self.initialize();
  	if(x instanceof Array) 
    	addPoint(x[0], x[1], EJSS_DRAWING2D.Trail.NO_CONNECTION);    
    else	
    	addPoint(x,y, EJSS_DRAWING2D.Trail.NO_CONNECTION);
    self.setChanged(true);
  }

  self.getSegmentsCount = function() {
    return mSegmentList.length; 
  }

  self.getSegmentPoints = function(index) {
    return mSegmentList[index]; 
  }

  self.getSegmentStyle = function(index) {
    return mSegmentStyle[index]; 
  }

  /**
   * Returns current points.
   * @method getCurrentPoints
   * @return points
   */
  self.getCurrentPoints = function () {
    return mCurrentList;
  }
  
  /**
   * Returns all points.
   * @method getPoints
   * @return points
   */
  self.getPoints = function () {
    var points = [];
    var size = mSegmentList.length;          	
	for(var i=0; i<size; i++ )
		points = points.concat(mSegmentList[i]);
	points = points.concat(mCurrentList);

    return points; 
  }

  
  /**
   * Gets last point.
   * @method getLastPoint
   * @return point
   */
  self.getLastPoint = function () {
    return mLastPoint;
  }

  /***
   * Same as clear
   * @method reset()
   * @visibility public
   */
  self.reset = function() {
    self.clear();
  }

  /***
   * Clears all points from all segments of the trail.
   * @method clear()
   * @visibility public
   */
  self.clear = function() {
    mSegmentList = [];
    mSegmentStyle = [];
    self.initialize();
  }

  /***
   * Clears all points from the last segment of the trail.
   * @method clearLastSegment()
   * @visibility public
   */
  self.clearLastSegment = function() {
	if (mCurrentList.length<=0) {
      mSegmentList.pop();
      mSegmentStyle.pop();
	  self.setChanged(true);
    }
	self.initialize();
  }
  
  /***
   * Clears all points from the last segment of the trail, 
   * respecting previous segments.
   * @method initialize()
   * @visibility public
   */
  self.initialize  = function() {
  	  mTmpList = [];
	  mCurrentList = [];	  
	  mFlushPoint = 0;
	  mIsEmpty = true;
	  mCounter = 0;
	  mLastPoint = 0;
	  self.setChanged(true);
  }

  /***
   * Creates a new segment of the trail.
   * @method newSegment()
   * @visibility public
   */
  self.newSegment = function() {
    if (mFlushPoint != 0) {
    	var size = mCurrentList.length;
    	mCurrentList[size] = self.formatPoint(mFlushPoint);
    }
    var size = mSegmentList.length;
    mSegmentList[size] = mCurrentList;
    var segmentStyle = EJSS_DRAWING2D.style(self.getName()+" #"+size);
    EJSS_DRAWING2D.Style.copyTo(self.getStyle(),segmentStyle);
    mSegmentStyle[size] = segmentStyle;

	self.initialize();
  }  
  
  /**
   * Returns bounds for an element
   * @override
   * @method getBounds
   * @return Object{left,rigth,top,bottom}
   */
  self.getBounds = function(element) {
  	var xmin, xmax, ymin, ymax;
  	var points = self.getPoints();
  	
  	// add mTmpList
  	//if (mTmpList.length > 0) points = points.concat(mTmpList);
  	
  	  	
    var len = points.length;
    if(len == 0) {
    	xmin = xmax = ymin = ymax = 0;
    }             
    else {
		xmax = xmin = points[0][0];
		ymax = ymin = points[0][1];    	
		for(var j=1; j<len; j++) {
			var x = points[j][0], y = points[j][1];
			if(x>xmax) xmax=x; if(y>ymax) ymax=y; if(x<xmin) xmin=x; if(y<ymin) ymin=y;
		}
	}    
    var x = self.getX(), y = self.getY();
    var sx = self.getSizeX(), sy = self.getSizeY();
  	var mx = sx/2, my = sy/2;  	
  	var d = self.getRelativePositionOffset(sx,sy);
  	
	return {
		left: ((x+d[0])-mx)+xmin*sx,
		right: ((x+d[0])-mx)+xmax*sx,
		top: ((y+d[1])-my)+ymax*sy,
		bottom: ((y+d[1])-my)+ymin*sy
	}
  };  
  
  function addPoint(_x, _y, _type) {
    //if (!mActive) return;
    if (isNaN(_x) || isNaN(_y)) { 
    	mIsEmpty = true; 
    	return; 
    }
    if (mNoRepeat && mLastPoint[0]==_x && mLastPoint[1]==_y) 
    	return;
	if ((typeof _type == "undefined") || (_type === null))
		_type = mConnectionType;

    mFlushPoint = 0;
    mLastPoint = self.formatPoint([_x,_y,_type]);
    
    if (mSkip > 0) { // Only if the counter is 0      
      if(mCounter > 0) mFlushPoint = self.formatPoint([_x, _y, _type]);      
      if(++mCounter >= mSkip) mCounter = 0;	        
    }
    
    if (mFlushPoint == 0) {
	    if (mMaximum > 2 && mCurrentList.length >= mMaximum) {
	    	mCurrentList.splice(0,1) // remove the firstPoint
	    }
	    
	    mCurrentList[mCurrentList.length] = self.formatPoint([_x, _y, _type]);
	    mIsEmpty = false;
	}
  }

	// override for subclases (for example Trace)
   self.formatPoint = function(src) {
   		return src;   		
   }

	self.registerProperties = function(controller) {
		EJSS_DRAWING2D.Trail.registerProperties(self, controller);
	};
	
  	self.copyTo = function(element) {
    	EJSS_DRAWING2D.Trail.copyTo(self,element);
  	};
  
	// ----------------------------------------------------
	// Final start-up
	// ----------------------------------------------------

	self.setSize([1,1]);
  /*** @property RelativePosition @default "SOUTH_WEST" */  
	self.setRelativePosition("SOUTH_WEST");

	return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//TrailSet
//---------------------------------

/**
 * TrailSet
 * @class TrailSet 
 * @constructor  
 */
EJSS_DRAWING2D.TrailSet = {
  
    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING2D.ElementSet;
      
      ElementSet.registerProperties(set,controller);
      controller.registerProperty("Active", 
          function(v) { set.setToEach(function(element,value) { element.setActive(value); }, v); }
      );
      controller.registerProperty("NoRepeat", 
          function(v) { set.setToEach(function(element,value) { element.setNoRepeat(value); }, v); }
      );           
      controller.registerProperty("ClearAtInput", 
          function(v) { set.setToEach(function(element,value) { element.setClearAtInput(value); }, v); }
      );           
      controller.registerProperty("Skip", 
          function(v) { set.setToEach(function(element,value) { element.setSkip(value); }, v); }
      );           
      controller.registerProperty("ColumnNames", 
          function(v) { set.setToEach(function(element,value) { element.setInputLabels(value); }, v); }
      );           
      controller.registerProperty("Maximum", 
          function(v) { set.setToEach(function(element,value) { element.setMaximumPoints(value); }, v); }
      );           
      controller.registerProperty("ConnectionType", 
          function(v) { set.setToEach(function(element,value) { element.setConnectionType(value); }, v); }
      );           
      controller.registerProperty("Connected", 
          function(v) { set.setToEach(function(element,value) { element.setConnected(value); }, v); }
      );           
      controller.registerProperty("Points", 
          function(v) { set.setToEach(function(element,value) { element.setPoints(value); }, v); }
      );           
      controller.registerProperty("LastPoint", 
          function(v) { set.setToEach(function(element,value) { element.addPoint(value); }, v); }
      );           
      controller.registerProperty("Input", 
          function(v) { set.setToEach(function(element,value) { element.addPoints(value); }, v); }
      );           
      controller.registerProperty("InputX", 
          function(v) { set.setToEach(function(element,value) { element.addXPoints(value); }, v); }
      );           
      controller.registerProperty("InputY", 
          function(v) { set.setToEach(function(element,value) { element.addYPoints(value); }, v); }
      );           
      
    }    
};


/**
 * Creates a set of Segments
 * @method trailSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING2D.trailSet = function (mName) {
  var self = EJSS_DRAWING2D.elementSet(EJSS_DRAWING2D.trail, mName);

  // Static references
  var TrailSet = EJSS_DRAWING2D.TrailSet;		// reference for TrailSet
  
  // ----------------------------------------------------
  // Set actions
  // ----------------------------------------------------

  /**
   * Resets all elements
   * @method reset
   */
  self.reset = function() {
    var elementList = self.getElements();
    for (var i=0,n=elementList.length;i<n;i++) elementList[i].reset();
  }
  
  /**
   * Initializes all elements
   * @method clear
   */
  self.initialize = function() {
    var elementList = self.getElements();
    for (var i=0,n=elementList.length;i<n;i++) elementList[i].initialize();
  }
  
  /**
   * Clears all elements
   * @method clear
   */
  self.clear = function() {
    var elementList = self.getElements();
    for (var i=0,n=elementList.length;i<n;i++) elementList[i].clear();
  }
  
  // ----------------------------------------------------
  // Custom methods
  // ----------------------------------------------------
  
  /**
   * Creates a new segment of one of the trails in the set
   * @method newSegment
   */
    self.newSegment = function(index) {
      self.getElements()[index].newSegment();
    }
  
  /**
   * Moves to the new point without drawing.
   * (Equivalent to setting the connection type
   * to NO_CONNECTION and adding one single point, then setting the 
   * type back to its previous value.)
   * @method moveToPoint
   * @param index int The index of the trail to move to 
   * @param x double The X coordinate of the point.
   * 	or point double[] The double[2] array with the coordinates of the point.
   * @param y double The Y coordinate of the point.
   */
   self.moveToPoint = function (index, x, y) {
      self.getElements()[index].moveToPoint(x,y);
    }
    
  /**
   * Adds a new point to one of the trails in the set
   * @method addPoint
   * @param index int The index of the trail to add to 
   * @param x double The X coordinate of the point 
   * 		or point double[] The double[2] array with the coordinates of the point.
   * @param y double The Y coordinate of the point.
   * @param style
   */
   self.addPoint = function (index, x, y, style) {
      self.getElements()[index].addPoint(x,y,style);
    }
  
  /**
   * Adds an array of points to one of the trails in the set
   * @method addPoints
   * @param index int The index of the trail to add to 
   * @param x double The double[] array with the X coordinates of the points.
   * 	or  point double[][] The double[nPoints][2] array with the coordinates of the points.
   * @param y double The double[] array with the Y coordinates of the points.
   */
    self.addPoints = function (index, x, y) {
      self.getElements()[index].addPoints(x,y);
    }
  

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------
  
  self.registerProperties = function(controller) {
    TrailSet.registerProperties(self,controller);
  };

  return self;
};/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Creates a ColorCoded object for 2D drawing
 * ColorCoded is a utility class that allows drawing a collection of polygons using a color code.
 * Each point in the polygon has a double value associated to it and the drawer uses a color mapper to find color subpolygons 
 * for each of the levels of the mapper
 * 
 */
EJSS_DRAWING2D.colorCoded = function (mNumColors, mPalette) {
  var self = EJSS_DRAWING2D.colorMapper(mNumColors, mPalette);  
    
  // Configuration variables
  var mSymmetricZ = false;
      
  // -------------------------------------
  // Public methods for coloring the tiles
  // -------------------------------------
      
  /**
   * Forces the z-scale to be symmetric about zero.
   * Forces zmax to be positive and zmin=-zmax when in autoscale mode.
   * @method setSymmetricZ
   * @param symmetric
   */
  self.setSymmetricZ = function(symmetric) {
    mSymmetricZ = symmetric;
  }
 
  /**
   * Gets the symmetric z flag.
   * @method isSymmetricZ 
   */
  self.isSymmetricZ = function(){
    return mSymmetricZ;
  }
  
  // -------------------------------------
  // Utility methods
  // -------------------------------------

  /**
   * Computes the extrema and sets the scales for the ColorMapper
   * @method setAutoscale
   * @param values
   */
  self.setAutoscale = function(values) {
    var min = Number.MAX_VALUE;
    var max = Number.MIN_VALUE;
    for (var i=0, n=values.length; i<n; i++) {
//      for (var j=0,m=values[i].length; j<m; j++) {        
//        var value = values[i][j];
		var value = values[i];
        max = Math.max (max, value);
        min = Math.min (min, value);            
//      }
    }
    var ceil = max;
    var floor = min;
    if (mSymmetricZ) {
      ceil = Math.max(Math.abs(min),Math.abs(max));
      floor = -ceil;
    }
    self.setScale(floor, ceil);
  }
 
   /**
   * Computes the extrema and sets the scales for the ColorMapper
   * @method setAutoscale2
   * @param values double[][]
   */
  self.setAutoscaleArray2 = function(values) {
    var min = Number.MAX_VALUE;
    var max = Number.MIN_VALUE;
    for (var i=0, n=values.length; i<n; i++) {
      for (var j=0,m=values[i].length; j<m; j++) {        
        var value = values[i][j];
        max = Math.max (max, value);
        min = Math.min (min, value);            
      }
    }
    var ceil = max;
    var floor = min;
    if (mSymmetricZ) {
      ceil = Math.max(Math.abs(min),Math.abs(max));
      floor = -ceil;
    }
    self.setScale(floor, ceil);
  }
  
     /**
   * Computes the extrema and sets the scales for the ColorMapper
   * @method setAutoscale3
   * @param values double[][][]
   */
  self.setAutoscaleArray3 = function(values) {
    var min = Number.MAX_VALUE;
    var max = Number.MIN_VALUE;
    for (var i=0, n=values.length; i<n; i++) {
      for (var j=0,m=values[i].length; j<m; j++) {        
        for (var k=0,p=values[i][j].length; k<p; k++) {        
          var value = values[i][j][k];
          max = Math.max (max, value);
          min = Math.min (min, value);
        }            
      }
    }
    var ceil = max;
    var floor = min;
    if (mSymmetricZ) {
      ceil = Math.max(Math.abs(min),Math.abs(max));
      floor = -ceil;
    }
    self.setScale(floor, ceil);
  }
  
  return self;	
}/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

EJSS_DRAWING2D.ColorMapper = {
  CUSTOM : -1,
  SPECTRUM : 0,
  GRAYSCALE : 1,
  DUALSHADE : 2,
  RED : 3,
  GREEN : 4,
  BLUE : 5,
  BLACK : 6,
  WIREFRAME : 7,     // special SurfacePlotter palette
  NORENDER : 8,      // special SurfacePlotter palette
  REDBLUE_SHADE : 9, // special SurfacePlotter palette

  /**
   * Gets a array of colors for use in data visualization.
   *
   * Colors are similar to the colors returned by a color mapper instance.
   * @param numColors
   * @param paletteType
   * @return
   */
  getColorPalette : function(numColors,paletteType) {

	function HSB2RGB (h,s,b){
		var hsb = { "h": Math.round(360*h), "s": s, "b": b };
		var rgb = { "r": 0, "g": 0, "b": 0 };
 
		if (hsb.s === 0){
			rgb.r = rgb.g = rgb.b = hsb.h;
			return rgb;
		}else{
			var Hi = Math.floor(hsb.h/60),
				f = (hsb.h/60) - Hi,
				p = hsb.b * (1 - hsb.s),
				q = hsb.b * (1 - (f * hsb.s)),
				t = hsb.b * (1 - (1 - f) * hsb.s);
			switch(Hi){
				case 0: rgb.r = hsb.b; rgb.g = t; rgb.b = p; break;
				case 1: rgb.r = q; rgb.g = hsb.b; rgb.b = p; break;
				case 2: rgb.r = p; rgb.g = hsb.b; rgb.b = t; break;
				case 3: rgb.r = p; rgb.g = q; rgb.b = hsb.b; break;
				case 4: rgb.r = t; rgb.g = p; rgb.b = hsb.b; break;
				case 5: rgb.r = hsb.b; rgb.g = p; rgb.b = q; break;
			}
			rgb.r = Math.round(rgb.r * 255);
			rgb.g = Math.round(rgb.g * 255);
			rgb.b = Math.round(rgb.b * 255);
 
			return rgb;
		}
	}
  	
  	var ColorMapper = EJSS_DRAWING2D.ColorMapper;
    var colors = [];    
	
    if(numColors<2) numColors = 2;
    
    for(var i = 0; i < numColors; i++) {
      var level = i/(numColors-1)*0.8;
      var porc1 = 1, porc2 = 1;
      var r = 0, g = 0, b = 0;
      switch(paletteType) {
         case ColorMapper.REDBLUE_SHADE :
           r = Math.floor((Math.max(0, -numColors-1+i*2)*255)/(numColors-1));           
           b = Math.floor((Math.max(0, numColors-1-i*2)*255)/(numColors-1));
           colors[i] = "rgb("+r+","+g+","+b+")";
           break;
         case ColorMapper.SPECTRUM :
           level = 0.8-level;     
           var rgb = HSB2RGB (level,porc1,porc2);      
           colors[i] = "rgb("+rgb.r+","+rgb.g+","+rgb.b+")";
           break;
         case ColorMapper.GRAYSCALE :
         case ColorMapper.BLACK :
           r = g = b = Math.floor(i*255/(numColors-1));
           colors[i] = "rgb("+r+","+g+","+b+")";           
           break;
         case ColorMapper.RED :
           r = Math.floor(i*255/(numColors-1));
           colors[i] = "rgb("+r+","+g+","+b+")";
           break;
         case ColorMapper.GREEN :
           g = Math.floor(i*255/(numColors-1));
           colors[i] = "rgb("+r+","+g+","+b+")";
           break;
         case ColorMapper.BLUE :
           b = Math.floor(i*255/(numColors-1));
           colors[i] = "rgb("+r+","+g+","+b+")";
           break;
         case ColorMapper.DUALSHADE :
         default :
           var tmp = i/(numColors-1);
           level = 0.8 * (1-tmp);
           porc2 = 0.2 + 1.6 * Math.abs(0.5-tmp);
           var rgb = HSB2RGB (level,porc1,porc2);      
           colors[i] = "rgb("+rgb.r+","+rgb.g+","+rgb.b+")";
           break;
      }
    }
    return colors;
  }

};

/**
 * Creates a Font object for 2D drawing
 */
EJSS_DRAWING2D.colorMapper = function (mNumColors, mPalette) {
  var ColorMapper = EJSS_DRAWING2D.ColorMapper;
  var self = {};  
  
  var mColors = [];  
  var mFloorColor = "darkgray";
  var mCeilColor = "lightgray";
  var mFloor = -1;
  var mCeil = 1;
  var mChangeListener;    
  var mPaletteType;
  var mThresholds;
      
  /**
   * Set a listener that will be called whenever there are Font changes.
   * I.e. a call to listener("change"); will be issued
   */
  self.setChangeListener = function(listener) {
    mChangeListener = listener;
  };
  
  /**
   * Sets the color palette.
   * @param _paletteType
   */
  self.setPaletteType = function(_paletteType) {
	if (typeof _paletteType == "string") _paletteType = EJSS_DRAWING2D.ColorMapper[_paletteType.toUpperCase()];
	mPaletteType = _paletteType;
    mFloorColor = "darkgray";
    mCeilColor = "lightgray";
    if((mPaletteType == EJSS_DRAWING2D.ColorMapper.GRAYSCALE) || 
    	(mPaletteType == EJSS_DRAWING2D.ColorMapper.BLACK)) {
      mFloorColor = "rgb(64,64,128)";
      mCeilColor = "rgb(255,191,191)";
    }
    mNumColors = Math.max(2, mNumColors); // need at least 2 colors
    mColors = ColorMapper.getColorPalette(mNumColors, mPaletteType);
    if (mChangeListener) mChangeListener("palette");
  }

  /**
   * Gets the colors;
   * @return
   */
  self.getColors = function() {
    return mColors;
  }

  /**
   * Gets the floor color;
   * @return
   */
  self.getFloorColor = function() {
    return mFloorColor;
  }

  /**
   * Gets the ceiling color.
   * @return
   */
  self.getCeilColor = function() {
    return mCeilColor;
  }

  /**
   * Gets the number of colors between the floor and ceiling values.
   * @return
   */
  self.getNumColors = function() {
    return mNumColors;
  }

  /**
   * Sets the floor and ceiling colors.
   *
   * @param _floorColor
   * @param _ceilColor
   */
  self.setFloorCeilColor = function(_floorColor,_ceilColor) {
    mFloorColor = _floorColor;
    mCeilColor = _ceilColor;
  }

  /**
   * Sets the floor and ceiling index.
   *
   * @param _floor
   * @param _ceil
   */
  self.setScale = function(_floor,_ceil) {
    mFloor = _floor;
    mCeil = _ceil;
    
    mThresholds = new Array(mColors.length + 1);
    var delta = (mCeil-mFloor) / mColors.length;
    for (var i=0,n=mColors.length; i<n; i++) mThresholds[i] = mFloor + i*delta;
    mThresholds[mColors.length] = mCeil;
  }

  /**
   * Returns the color palette.
   * @return mode
   */
  self.getPaletteType = function() {
    return mPaletteType;
  }

  /**
   * Sets the color palette.
   * @param _colors
   */
  self.setColorPalette = function(_colors) {  	
  	mFloorColor = "darkgray";
  	mCeilColor = "lightgray";
    mColors = _colors;
    mNumColors = _colors.length;
    mPaletteType = EJSS_DRAWING2D.ColorMapper.CUSTOM;
    if (mChangeListener) mChangeListener("colors");
  }

  /**
   * Sets the number of colors
   * @param _numColors
   */
  self.setNumberOfColors = function(_numColors) {
    if(_numColors == mNumColors) {
      return;
    }
    mNumColors = _numColors;
    if(mPaletteType == EJSS_DRAWING2D.ColorMapper.CUSTOM) {
      var newColors = [];
      for(var i = 0, n = Math.min(colors.length, mNumColors); i<n; i++) {
        newColors[i] = colors[i];
      }
      for(var i = colors.length; i<numColors; i++) {
        newColors[i] = colors[colors.length-1];
      }
      colors = newColors;
    } else {
      self.setPaletteType(mPaletteType);
    }
    if (mChangeListener) mChangeListener("numColors");
  }

  /**
   * Gets the number of colors
   * @param _numColors
   */
  self.getNumberOfColors = function() {
  	return mNumColors;
  }

 /**
   * Converts a double to an index in the color array.
   * @method doubleToIndex
   * @param value
   * @return the index in the array with the following exceptions:
   * <ul>
   *   <li>-1 if floor color</li>
   *   <li>colors.length if ceil color</li>
   * </ul> 
   */
  self.doubleToIndex = function(value) { 
    if(mFloor-value>Number.MIN_VALUE) {
      return -1;
    } else if(value-mCeil>Number.MIN_VALUE) {
      return mColors.length;
    }
    var index = 0;
    if(mCeil != mFloor)
    	index = Math.floor((mColors.length*(value-mFloor)/(mCeil-mFloor)));
    index = Math.max(0, index);
    return Math.min(index, mColors.length-1);
  }
  
  /**
   * Returns the color for an index
   * @method indexToColor
   */
  self.indexToColor = function(index) { 
    if (index<0) return mFloorColor;
    if (index>=mColors.length) return mCeilColor;
    return mColors[index];
  }

  /**
   * Returns the color for an index
   * @method indexToColor
   */
  self.doubleToColor = function(value) { return self.indexToColor(self.doubleToIndex(value)); } 
  
  /**
   * Returns the thresholds for color change. One more than colors, includes ceil and floor
   * @method getColorThresholds
   */
  self.getColorThresholds = function() { 
  	return mThresholds;
  }

  self.setPaletteType(mPalette); // default colors
  self.setScale(-1,1);
  return self;	
}/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Font object for 2D drawing
 * @class Font 
 * @constructor  
 */
EJSS_DRAWING2D.Font = {
  FONTSTYLES: ["normal", "italic", "oblique", "initial"],
  FONTWEIGHTS: ["normal", "lighter", "bold", "bolder", "initial", 
  	"100", "200", "300", "400", "500", "600", "700", "800", "900"],

  // ----------------------------------------------------
  // Static methods
  // ----------------------------------------------------

  /**
   * Copies one element into another
   */
  copyTo : function(source, dest) {
  	dest.setFontFamily(source.getFontFamily());
  	dest.setFontSize(source.getFontSizeString());
	dest.setLetterSpacing(source.getLetterSpacing());
	dest.setOutlineColor(source.getOutlineColor());
	dest.setOutlineWidth(source.getOutlineWidth());
	dest.setFontWeight(source.getFontWeight());
	dest.setFillColor(source.getFillColor());  	
  },

};

/**
 * Creates a Font object for 2D drawing
 */
EJSS_DRAWING2D.font = function (mName) {
  var self = {};
  
  var mFontFamily = "Arial"; 	// the type of font to use, for instance 'Arial' or 'Verdana'.
  var mFontSize = "20";			// the size of the font, for instance '12' or '24'.
  var mLetterSpacing = "normal";// spacing between letters, for instance '2' or '3'. Similar to kerning.
  var mLineColor = "none";		// the outline color of the font. By default text only has fill color, not stroke.
  var mLineWidth = "1";			// the outline width of the font. By default 1.
  var mFontWeight = "normal";	// the weight of the font.
  var mFillColor = "black";		// the fill color of the font. 	    
  var mFontStyle = "none";
  var mChangeListener;           
      
  /**
   * Set a listener that will be called whenever there are Font changes.
   * I.e. a call to listener("change"); will be issued
   */
  self.setChangeListener = function(listener) {
    mChangeListener = listener;
  };

  /**
   * Set the family of the font.
   * @method setFontFamily
   * @param fontFamily
   */
  self.setFontFamily = function(fontFamily) {
    if (mFontFamily != fontFamily) {
      mFontFamily = fontFamily;
      if (mChangeListener) mChangeListener("fontFamily");
    }
  };
      
  /**
   * Get the family of the font.
   * @method getFontFamily
   * @return font family
   */
  self.getFontFamily = function() { 
    return mFontFamily; 
  };

  /**
   * Set the size of the font.
   * @method setFontSize
   * @param fontSize
   */
  self.setFontSize = function(fontSize) {
    if (mFontSize != fontSize) {
      mFontSize = fontSize;
      if (mChangeListener) mChangeListener("fontSize");
    }
  };
      
  /**
   * Get the size of the font in pixels.
   * @method getFontSize
   * @return font size
   */
  self.getFontSize = function() {
  	var size;
  	if(!isNaN(mFontSize)) { 
  		size = +mFontSize; // number
    } else if(mFontSize.indexOf("em") != -1) {
    	size = 	+(mFontSize.substr(0,mFontSize.indexOf("em"))) * 10;	// aprox
    } else if(mFontSize.indexOf("px") != -1) { 
    	size = +(mFontSize.substr(0,mFontSize.indexOf("px"))); // px
    } else if(mFontSize.indexOf("vh") != -1) { 
		var h = Math.max(document.documentElement.clientHeight, window.innerHeight || 0);
    	size = (h/100)*(+(mFontSize.substr(0,mFontSize.indexOf("vh")))); // vh    	
    } else if(mFontSize.indexOf("vw") != -1) { 
		var w = Math.max(document.documentElement.clientWidth, window.innerWidth || 0);
    	size = (w/100)*(+(mFontSize.substr(0,mFontSize.indexOf("vw")))); // vw
    } else {
    	size = 0; // for example, rem
    }
    
    return size; 
  };

  /**
   * Get the size of the font as a String
   * @method getFontSizeString
   * @return font size as a String
   */
  self.getFontSizeString = function() { 
    return mFontSize; 
  };

  /**
   * Set the letter spacing of the font.
   * @method setLetterSpacing
   * @param letterSpacing
   */
  self.setLetterSpacing = function(letterSpacing) {
    if (mLetterSpacing != letterSpacing) {
      mLetterSpacing = letterSpacing;
      if (mChangeListener) mChangeListener("letterSpacing");
    }
  };

  /**
   * Get the letter spacing of the font.
   * @method getLetterSpacing
   * @return letter spacing
   */
  self.getLetterSpacing = function() {
    return mLetterSpacing;
  };
      
  /**
   * Get the letter spacing of the font.
   * @method getNumberLetterSpacing
   * @return letter spacing
   */
  self.getNumberLetterSpacing = function() {
  	var spacing;
  	if(!isNaN(mLetterSpacing)) { 
  		spacing = +mLetterSpacing; // number
    } else if(mLetterSpacing.indexOf("px") != -1) { 
    	spacing = +(mLetterSpacing.substr(0,mLetterSpacing.indexOf("px"))); // px
    } else {
    	spacing = 1; // for example, normal
    }
    return spacing; 
  };

  /**
   * Set the line color of the font.
   * @method setOutlineColor
   * @param lineColor
   */
  self.setOutlineColor = function(lineColor) {
    if (mLineColor != lineColor) {
      mLineColor = lineColor;
      if (mChangeListener) mChangeListener("lineColor");
    }
  };
      
  /**
   * Get the line color of the font.
   * @method getOutlineColor
   * @return line color
   */
  self.getOutlineColor = function() { 
    return mLineColor; 
  };

  /**
   * Set the line width of the font.
   * @method setOutlineWidth
   * @param lineWidth
   */
  self.setOutlineWidth = function(lineWidth) {
    if (mLineWidth != lineWidth) {
      mLineWidth = lineWidth;
      if (mChangeListener) mChangeListener("lineWidth");
    }
  };
      
  /**
   * Get the line color of the font.
   * @method getOutlineWidth
   * @return line width
   */
  self.getOutlineWidth = function() { 
    return mLineWidth; 
  };

  /**
   * Set the weight of the font.
   * @method setFontWeight
   * @param fontWeight
   */
  self.setFontWeight = function(fontWeight) {
	  fontWeight = fontWeight.toLowerCase(); 

    if (mFontWeight != fontWeight && (EJSS_DRAWING2D.Font.FONTWEIGHTS.indexOf(fontWeight) > -1)) {
      mFontWeight = fontWeight;
      if (mChangeListener) mChangeListener("fontWeight");
    }
  };
      
  /**
   * Get the weight of the font.
   * @method getFontWeight
   * @return fontWeight
   */
  self.getFontWeight = function() { 
    return mFontWeight; 
  };

  /**
   * Set the fill color of the font.
   * @method setFillColor
   * @param fillColor
   */
  self.setFillColor = function(fillColor) {
    if (mFillColor != fillColor) {
      mFillColor = fillColor;
      if (mChangeListener) mChangeListener("fillColor");
    }
  };
      
  /**
   * Get the fill color of the font.
   * @method getFillColor
   * @return fill color
   */
  self.getFillColor = function() { 
    return mFillColor; 
  };
  
  /**
   * Set the font style (normal, italic, oblique, initial, none).
   * @method setFontStyle
   * @param fontStyle
   */
  self.setFontStyle = function(fontstyle) {
	fontstyle = fontstyle.toLowerCase(); 
    if (mFontStyle != fontstyle && (EJSS_DRAWING2D.Font.FONTSTYLES.indexOf(fontstyle) > -1)) {
      mFontStyle = fontstyle;
      if (mChangeListener) mChangeListener("fontstyle");
    }
  };
      
  /**
   * Get the font style.
   * @method getFontStyle
   * @return fontStyle
   */
  self.getFontStyle = function() { 
    return mFontStyle; 
  };

  /**
   * @method getFont
   * @return font
   */
  self.getFont = function() {
  	return mFontStyle + " " + mLineWidth + " " + mFontSize + " " + mFontFamily;
  };

  /**
   * @method setFont
   * @param recap string, format: [style weight size[/lineHeight] [family]]
   */ 
  self.setFont = function (recap) {  
    if ((typeof(recap) !== 'string') || recap.length<=0) return;	
    var params = recap.split(" ");        
   	mFontStyle = params[0];		// style   	
   	mFontWeight = params[1];	// weight	
	var sizes = params[2].split("/");   
   	mFontSize = sizes[0];			// size
   	// ignoring sizes[1] lineHeight   	
   	if (params[3]) 
   		mFontFamily = recap.substring(recap.indexOf(params[3])); // family  
  }
      

  
  //---------------------------------
  // final initialization
  //---------------------------------
  
  return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 *
 *  
 * This code is Open Source and is provided "as is".
 */

/*
 Based on the parser by Matthew Crumley (email@matthewcrumley.com, http://silentmatt.com/)
*/

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};
// TODO: "Math.rint", "Math.toDegrees", "Math.toRadians", "Math.IEEEremainder"

EJSS_DRAWING2D.functionsParser = function() {
	var self = {};
	  	
	function object(o) {
		function F() {}
		F.prototype = o;
		return new F();
	}

	var TNUMBER = 0;
	var TOP1 = 1;
	var TOP2 = 2;
	var TVAR = 3;
	var TFUNCALL = 4;

	function Token(type_, index_, prio_, number_) {
		this.type_ = type_;
		this.index_ = index_ || 0;
		this.prio_ = prio_ || 0;
		this.number_ = (number_ !== undefined && number_ !== null) ? number_ : 0;
		this.toString = function () {
			switch (this.type_) {
			case TNUMBER:
				return this.number_;
			case TOP1:
			case TOP2:
			case TVAR:
				return this.index_;
			case TFUNCALL:
				return "CALL";
			default:
				return "Invalid Token";
			}
		};
	}

	function Expression(tokens, ops1, ops2, functions) {
		this.tokens = tokens;
		this.ops1 = ops1;
		this.ops2 = ops2;
		this.functions = functions;
	}

	// Based on http://www.json.org/json2.js
    var cx = /[\u0000\u00ad\u0600-\u0604\u070f\u17b4\u17b5\u200c-\u200f\u2028-\u202f\u2060-\u206f\ufeff\ufff0-\uffff]/g,
        escapable = /[\\\'\x00-\x1f\x7f-\x9f\u00ad\u0600-\u0604\u070f\u17b4\u17b5\u200c-\u200f\u2028-\u202f\u2060-\u206f\ufeff\ufff0-\uffff]/g,
        meta = {    // table of character substitutions
            '\b': '\\b',
            '\t': '\\t',
            '\n': '\\n',
            '\f': '\\f',
            '\r': '\\r',
            "'" : "\\'",
            '\\': '\\\\'
        };

	function escapeValue(v) {
		if (typeof v === "string") {
			escapable.lastIndex = 0;
	        return escapable.test(v) ?
	            "'" + v.replace(escapable, function (a) {
	                var c = meta[a];
	                return typeof c === 'string' ? c :
	                    '\\u' + ('0000' + a.charCodeAt(0).toString(16)).slice(-4);
	            }) + "'" :
	            "'" + v + "'";
		}
		return v;
	}

	Expression.prototype = {
		simplify: function (values) {
			values = values || {};
			var nstack = [];
			var newexpression = [];
			var n1;
			var n2;
			var f;
			var L = this.tokens.length;
			var item;
			var i = 0;
			for (i = 0; i < L; i++) {
				item = this.tokens[i];
				var type_ = item.type_;
				if (type_ === TNUMBER) {
					nstack.push(item);
				}
				else if (type_ === TVAR && (item.index_ in values)) {
					item = new Token(TNUMBER, 0, 0, values[item.index_]);
					nstack.push(item);
				}
				else if (type_ === TOP2 && nstack.length > 1) {
					n2 = nstack.pop();
					n1 = nstack.pop();
					f = this.ops2[item.index_];
					item = new Token(TNUMBER, 0, 0, f(n1.number_, n2.number_));
					nstack.push(item);
				}
				else if (type_ === TOP1 && nstack.length > 0) {
					n1 = nstack.pop();
					f = this.ops1[item.index_];
					item = new Token(TNUMBER, 0, 0, f(n1.number_));
					nstack.push(item);
				}
				else {
					while (nstack.length > 0) {
						newexpression.push(nstack.shift());
					}
					newexpression.push(item);
				}
			}
			while (nstack.length > 0) {
				newexpression.push(nstack.shift());
			}

			return new Expression(newexpression, object(this.ops1), object(this.ops2), object(this.functions));
		},

		substitute: function (variable, expr) {
			if (!(expr instanceof Expression)) {
				expr = new Parser().parse(String(expr));
			}
			var newexpression = [];
			var L = this.tokens.length;
			var item;
			var i = 0;
			for (i = 0; i < L; i++) {
				item = this.tokens[i];
				var type_ = item.type_;
				if (type_ === TVAR && item.index_ === variable) {
					for (var j = 0; j < expr.tokens.length; j++) {
						var expritem = expr.tokens[j];
						var replitem = new Token(expritem.type_, expritem.index_, expritem.prio_, expritem.number_);
						newexpression.push(replitem);
					}
				}
				else {
					newexpression.push(item);
				}
			}

			var ret = new Expression(newexpression, object(this.ops1), object(this.ops2), object(this.functions));
			return ret;
		},

		evaluate: function (values) {
			values = values || {};
			var nstack = [];
			var n1;
			var n2;
			var f;
			var L = this.tokens.length;
			var item;
			var i = 0;
			for (i = 0; i < L; i++) {
				item = this.tokens[i];
				var type_ = item.type_;
				if (type_ === TNUMBER) {
					nstack.push(item.number_);
				}
				else if (type_ === TOP2) {
					n2 = nstack.pop();
					n1 = nstack.pop();
					f = this.ops2[item.index_];
					nstack.push(f(n1, n2));
				}
				else if (type_ === TVAR) {
					if (item.index_ in values) {
						nstack.push(values[item.index_]);
					}
					else if (item.index_ in this.functions) {
						nstack.push(this.functions[item.index_]);
					}
					else {
						throw new Error("undefined variable: " + item.index_);
					}
				}
				else if (type_ === TOP1) {
					n1 = nstack.pop();
					f = this.ops1[item.index_];
					nstack.push(f(n1));
				}
				else if (type_ === TFUNCALL) {
					n1 = nstack.pop();
					f = nstack.pop();
					if (f.apply && f.call) {
						if (Object.prototype.toString.call(n1) == "[object Array]") {
							nstack.push(f.apply(undefined, n1));
						}
						else {
							nstack.push(f.call(undefined, n1));
						}
					}
					else {
						throw new Error(f + " is not a function");
					}
				}
				else {
					throw new Error("invalid Expression");
				}
			}
			if (nstack.length > 1) {
				throw new Error("invalid Expression (parity)");
			}
			return nstack[0];
		},

		toString: function (toJS) {
			var nstack = [];
			var n1;
			var n2;
			var f;
			var L = this.tokens.length;
			var item;
			var i = 0;
			for (i = 0; i < L; i++) {
				item = this.tokens[i];
				var type_ = item.type_;
				if (type_ === TNUMBER) {
					nstack.push(escapeValue(item.number_));
				}
				else if (type_ === TOP2) {
					n2 = nstack.pop();
					n1 = nstack.pop();
					f = item.index_;
					if (toJS && f == "^") {
						nstack.push("Math.pow(" + n1 + "," + n2 + ")");
					}
					else {
						nstack.push("(" + n1 + f + n2 + ")");
					}
				}
				else if (type_ === TVAR) {
					nstack.push(item.index_);
				}
				else if (type_ === TOP1) {
					n1 = nstack.pop();
					f = item.index_;
					if (f === "-") {
						nstack.push("(" + f + n1 + ")");
					}
					else {
						nstack.push(f + "(" + n1 + ")");
					}
				}
				else if (type_ === TFUNCALL) {
					n1 = nstack.pop();
					f = nstack.pop();
					nstack.push(f + "(" + n1 + ")");
				}
				else {
					throw new Error("invalid Expression");
				}
			}
			if (nstack.length > 1) {
				throw new Error("invalid Expression (parity)");
			}
			return nstack[0];
		},

		variables: function () {
			var L = this.tokens.length;
			var vars = [];
			for (var i = 0; i < L; i++) {
				var item = this.tokens[i];
				if (item.type_ === TVAR && (vars.indexOf(item.index_) == -1)) {
					vars.push(item.index_);
				}
			}

			return vars;
		},

		toJSFunction: function (param, variables) {
			var f = new Function(param, "with(Parser.values) { return " + this.simplify(variables).toString(true) + "; }");
			return f;
		}
	};

	function add(a, b) {
		return Number(a) + Number(b);
	}
	function sub(a, b) {
		return a - b; 
	}
	function mul(a, b) {
		return a * b;
	}
	function div(a, b) {
		return a / b;
	}
	function mod(a, b) {
		return a % b;
	}
	function concat(a, b) {
		return "" + a + b;
	}

	function neg(a) {
		return -a;
	}

	function random(a) {
		return Math.random() * (a || 1);
	}
	function fac(a) { //a!
		a = Math.floor(a);
		var b = a;
		while (a > 1) {
			b = b * (--a);
		}
		return b;
	}

	// TODO: use hypot that doesn't overflow
	function pyt(a, b) {
		return Math.sqrt(a * a + b * b);
	}

	function ifelse(a, b, c) {
		if(a) return b; else return c;
	}
	
	function step(a) {
	  if (a<0) return 0;
	  return 1;
	}

	function append(a, b) {
		if (Object.prototype.toString.call(a) != "[object Array]") {
			return [a, b];
		}
		a = a.slice();
		a.push(b);
		return a;
	}

	function greaterthan(a, b) {
		return (a > b)? 1 : 0;
	}

	function lessthan(a, b) {
		return (a < b)? 1 : 0;
	}

	function equalto(a, b) {
		return (a == b)? 1 : 0;
	}

	function notequalto(a, b) {
		return (a != b)? 1 : 0;
	}

	function Parser() {
		this.success = false;
		this.errormsg = "";
		this.expression = "";

		this.pos = 0;

		this.tokennumber = 0;
		this.tokenprio = 0;
		this.tokenindex = 0;
		this.tmpprio = 0;

		this.ops1 = {
			"sin": Math.sin,
			"cos": Math.cos,
			"tan": Math.tan,
			"asin": Math.asin,
			"acos": Math.acos,
			"atan": Math.atan,
			"sqrt": Math.sqrt,
			"log": Math.log,
			"abs": Math.abs,
			"ceil": Math.ceil,
			"floor": Math.floor,
			"round": Math.round,
			"-": neg,
			"neg": neg,
			"exp": Math.exp
		};

		this.ops2 = {
			"+": add,
			"-": sub,
			"*": mul,
			"/": div,
			"%": mod,
			"^": Math.pow,
			",": append,
			"||": concat,
			">": greaterthan,
			"<": lessthan,
			"==": equalto,
			"!=": notequalto
		};

		this.functions = {
			"random": random,
			"fac": fac,
			"min": Math.min,
			"max": Math.max,
			"pyt": pyt,
			"step": step,
			"pow": Math.pow,
			"atan2": Math.atan2,
			"if": ifelse
		};

		this.consts = {
		    //"e": Math.E,
			"E": Math.E,
			"pi": Math.PI,
			"PI": Math.PI
		};
	}

	self.parse = function (expr) {
		return new Parser().parse(expr);
	};

	self.evaluate = function (expr, variables) {
		return Parser.parse(expr).evaluate(variables);
	};

	Parser.Expression = Expression;

	Parser.values = {
		sign: Math.sign,
		sin: Math.sin,
		cos: Math.cos,
		tan: Math.tan,
		asin: Math.asin,
		acos: Math.acos,
		atan: Math.atan,
		sqrt: Math.sqrt,
		log: Math.log,
		abs: Math.abs,
		ceil: Math.ceil,
		floor: Math.floor,
		round: Math.round,
		random: random,
		fac: fac,
		exp: Math.exp,
		min: Math.min,
		max: Math.max,
		pyt: pyt,
		ifelse: ifelse,
		step: step,
		pow: Math.pow,
		atan2: Math.atan2,
		//e: Math.E,
		E: Math.E,
		pi: Math.PI,
		PI: Math.PI
	};

	var PRIMARY      = 1 << 0;
	var OPERATOR     = 1 << 1;
	var FUNCTION     = 1 << 2;
	var LPAREN       = 1 << 3;
	var RPAREN       = 1 << 4;
	var COMMA        = 1 << 5;
	var SIGN         = 1 << 6;
	var CALL         = 1 << 7;
	var NULLARY_CALL = 1 << 8;

	Parser.prototype = {
		parse: function (expr) {
			this.errormsg = "";
			this.success = true;
			var operstack = [];
			var tokenstack = [];
			this.tmpprio = 0;
			var expected = (PRIMARY | LPAREN | FUNCTION | SIGN);
			var noperators = 0;
			this.expression = expr;
			this.pos = 0;

			while (this.pos < this.expression.length) {
				if (this.isOperator()) {
					if (this.isSign() && (expected & SIGN)) {
						if (this.isNegativeSign()) {
							this.tokenprio = 2;
							this.tokenindex = "-";
							noperators++;
							this.addfunc(tokenstack, operstack, TOP1);
						}
						expected = (PRIMARY | LPAREN | FUNCTION | SIGN);
					}
					else if (this.isComment()) {

					}
					else {
						if ((expected & OPERATOR) === 0) {
							this.error_parsing(this.pos, "unexpected operator");
						}
						noperators += 2;
						this.addfunc(tokenstack, operstack, TOP2);
						expected = (PRIMARY | LPAREN | FUNCTION | SIGN);
					}
				}
				else if (this.isNumber()) {
					if ((expected & PRIMARY) === 0) {
						this.error_parsing(this.pos, "unexpected number");
					}
					var token = new Token(TNUMBER, 0, 0, this.tokennumber);
					tokenstack.push(token);

					expected = (OPERATOR | RPAREN | COMMA);
				}
				else if (this.isString()) {
					if ((expected & PRIMARY) === 0) {
						this.error_parsing(this.pos, "unexpected string");
					}
					var token = new Token(TNUMBER, 0, 0, this.tokennumber);
					tokenstack.push(token);

					expected = (OPERATOR | RPAREN | COMMA);
				}
				else if (this.isLeftParenth()) {
					if ((expected & LPAREN) === 0) {
						this.error_parsing(this.pos, "unexpected \"(\"");
					}

					if (expected & CALL) {
						noperators += 2;
						this.tokenprio = -2;
						this.tokenindex = -1;
						this.addfunc(tokenstack, operstack, TFUNCALL);
					}

					expected = (PRIMARY | LPAREN | FUNCTION | SIGN | NULLARY_CALL);
				}
				else if (this.isRightParenth()) {
				    if (expected & NULLARY_CALL) {
						var token = new Token(TNUMBER, 0, 0, []);
						tokenstack.push(token);
					}
					else if ((expected & RPAREN) === 0) {
						this.error_parsing(this.pos, "unexpected \")\"");
					}

					expected = (OPERATOR | RPAREN | COMMA | LPAREN | CALL);
				}
				else if (this.isComma()) {
					if ((expected & COMMA) === 0) {
						this.error_parsing(this.pos, "unexpected \",\"");
					}
					this.addfunc(tokenstack, operstack, TOP2);
					noperators += 2;
					expected = (PRIMARY | LPAREN | FUNCTION | SIGN);
				}
				else if (this.isConst()) {
					if ((expected & PRIMARY) === 0) {
						this.error_parsing(this.pos, "unexpected constant");
					}
					var consttoken = new Token(TNUMBER, 0, 0, this.tokennumber);
					tokenstack.push(consttoken);
					expected = (OPERATOR | RPAREN | COMMA);
				}
				else if (this.isOp2()) {
					if ((expected & FUNCTION) === 0) {
						this.error_parsing(this.pos, "unexpected function");
					}
					this.addfunc(tokenstack, operstack, TOP2);
					noperators += 2;
					expected = (LPAREN);
				}
				else if (this.isOp1()) {
					if ((expected & FUNCTION) === 0) {
						this.error_parsing(this.pos, "unexpected function");
					}
					this.addfunc(tokenstack, operstack, TOP1);
					noperators++;
					expected = (LPAREN);
				}
				else if (this.isVar()) {
					if ((expected & PRIMARY) === 0) {
						this.error_parsing(this.pos, "unexpected variable");
					}
					var vartoken = new Token(TVAR, this.tokenindex, 0, 0);
					tokenstack.push(vartoken);

					expected = (OPERATOR | RPAREN | COMMA | LPAREN | CALL);
				}
				else if (this.isWhite()) {
				}
				else {
					if (this.errormsg === "") {
						this.error_parsing(this.pos, "unknown character");
					}
					else {
						this.error_parsing(this.pos, this.errormsg);
					}
				}
			}
			if (this.tmpprio < 0 || this.tmpprio >= 10) {
				this.error_parsing(this.pos, "unmatched \"()\"");
			}
			while (operstack.length > 0) {
				var tmp = operstack.pop();
				tokenstack.push(tmp);
			}
			if (noperators + 1 !== tokenstack.length) {
				//print(noperators + 1);
				//print(tokenstack);
				this.error_parsing(this.pos, "parity");
			}

			return new Expression(tokenstack, object(this.ops1), object(this.ops2), object(this.functions));
		},

		evaluate: function (expr, variables) {
			return this.parse(expr).evaluate(variables);
		},

		error_parsing: function (column, msg) {
			this.success = false;
			this.errormsg = "parse error [column " + (column) + "]: " + msg;
			throw new Error(this.errormsg);
		},

//\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\

		addfunc: function (tokenstack, operstack, type_) {
			var operator = new Token(type_, this.tokenindex, this.tokenprio + this.tmpprio, 0);
			while (operstack.length > 0) {
				if (operator.prio_ <= operstack[operstack.length - 1].prio_) {
					tokenstack.push(operstack.pop());
				}
				else {
					break;
				}
			}
			operstack.push(operator);
		},

		isNumber: function () {
			var r = false;
			var str = "";
			while (this.pos < this.expression.length) {
				var code = this.expression.charCodeAt(this.pos);
				if ((code >= 48 && code <= 57) || code === 46) {
					str += this.expression.charAt(this.pos);
					this.pos++;
					this.tokennumber = parseFloat(str);
					r = true;
				}
				else {
					break;
				}
			}
			return r;
		},

		// Ported from the yajjl JSON parser at http://code.google.com/p/yajjl/
		unescape: function(v, pos) {
			var buffer = [];
			var escaping = false;

			for (var i = 0; i < v.length; i++) {
				var c = v.charAt(i);
	
				if (escaping) {
					switch (c) {
					case "'":
						buffer.push("'");
						break;
					case '\\':
						buffer.push('\\');
						break;
					case '/':
						buffer.push('/');
						break;
					case 'b':
						buffer.push('\b');
						break;
					case 'f':
						buffer.push('\f');
						break;
					case 'n':
						buffer.push('\n');
						break;
					case 'r':
						buffer.push('\r');
						break;
					case 't':
						buffer.push('\t');
						break;
					case 'u':
						// interpret the following 4 characters as the hex of the unicode code point
						var codePoint = parseInt(v.substring(i + 1, i + 5), 16);
						buffer.push(String.fromCharCode(codePoint));
						i += 4;
						break;
					default:
						throw this.error_parsing(pos + i, "Illegal escape sequence: '\\" + c + "'");
					}
					escaping = false;
				} else {
					if (c == '\\') {
						escaping = true;
					} else {
						buffer.push(c);
					}
				}
			}
	
			return buffer.join('');
		},

		isString: function () {
			var r = false;
			var str = "";
			var startpos = this.pos;
			if (this.pos < this.expression.length && this.expression.charAt(this.pos) == "'") {
				this.pos++;
				while (this.pos < this.expression.length) {
					var code = this.expression.charAt(this.pos);
					if (code != "'" || str.slice(-1) == "\\") {
						str += this.expression.charAt(this.pos);
						this.pos++;
					}
					else {
						this.pos++;
						this.tokennumber = this.unescape(str, startpos);
						r = true;
						break;
					}
				}
			}
			return r;
		},

		isConst: function () {
			var str;
			for (var i in this.consts) {
				var L = i.length;
				str = this.expression.substr(this.pos, L);
				if (i === str) {
					this.tokennumber = this.consts[i];
					this.pos += L;
					return true;
				}
			}
			return false;
		},

		isOperator: function () {
			var code = this.expression.charCodeAt(this.pos);
			if (code === 43) { // +
				this.tokenprio = 0;
				this.tokenindex = "+";
			}
			else if (code === 45) { // -
				this.tokenprio = 0;
				this.tokenindex = "-";
			}
			else if (code === 124) { // |
				if (this.expression.charCodeAt(this.pos + 1) === 124) {
					this.pos++;
					this.tokenprio = 0;
					this.tokenindex = "||";
				}
				else {
					return false;
				}
			}
			else if (code === 61) { // ==
				if (this.expression.charCodeAt(this.pos + 1) === 61) {
					this.pos++;
					this.tokenprio = 0;
					this.tokenindex = "==";
				}
				else {
					return false;
				}
			}
			else if (code === 33) { // !=
				if (this.expression.charCodeAt(this.pos + 1) === 61) {
					this.pos++;
					this.tokenprio = 0;
					this.tokenindex = "!=";
				}
				else {
					return false;
				}
			}
			else if (code === 42) { // *
				this.tokenprio = 1;
				this.tokenindex = "*";
			}
			else if (code === 47) { // /
				this.tokenprio = 2;
				this.tokenindex = "/";
			}
			else if (code === 37) { // %
				this.tokenprio = 2;
				this.tokenindex = "%";
			}
			else if (code === 94) { // ^
				this.tokenprio = 3;
				this.tokenindex = "^";
			}
			else if (code === 60) { // <
				this.tokenprio = 2;
				this.tokenindex = "<";
			}
			else if (code === 62) { // >
				this.tokenprio = 2;
				this.tokenindex = ">";
			}
			else {
				return false;
			}
			this.pos++;
			return true;
		},

		isSign: function () {
			var code = this.expression.charCodeAt(this.pos - 1);
			if (code === 45 || code === 43) { // -
				return true;
			}
			return false;
		},

		isPositiveSign: function () {
			var code = this.expression.charCodeAt(this.pos - 1);
			if (code === 43) { // -
				return true;
			}
			return false;
		},

		isNegativeSign: function () {
			var code = this.expression.charCodeAt(this.pos - 1);
			if (code === 45) { // -
				return true;
			}
			return false;
		},

		isLeftParenth: function () {
			var code = this.expression.charCodeAt(this.pos);
			if (code === 40) { // (
				this.pos++;
				this.tmpprio += 10;
				return true;
			}
			return false;
		},

		isRightParenth: function () {
			var code = this.expression.charCodeAt(this.pos);
			if (code === 41) { // )
				this.pos++;
				this.tmpprio -= 10;
				return true;
			}
			return false;
		},

		isComma: function () {
			var code = this.expression.charCodeAt(this.pos);
			if (code === 44) { // ,
				this.pos++;
				this.tokenprio = -1;
				this.tokenindex = ",";
				return true;
			}
			return false;
		},

		isWhite: function () {
			var code = this.expression.charCodeAt(this.pos);
			if (code === 32 || code === 9 || code === 10 || code === 13) {
				this.pos++;
				return true;
			}
			return false;
		},

		isOp1: function () {
			var str = "";
			for (var i = this.pos; i < this.expression.length; i++) {
				var c = this.expression.charAt(i);
				if (c.toUpperCase() === c.toLowerCase()) {
					if (i === this.pos || (c != '_' && (c < '0' || c > '9'))) {
						break;
					}
				}
				str += c;
			}
			if (str.length > 0 && (str in this.ops1)) {
				this.tokenindex = str;
				this.tokenprio = 5;
				this.pos += str.length;
				return true;
			}
			return false;
		},

		isOp2: function () {
			var str = "";
			for (var i = this.pos; i < this.expression.length; i++) {
				var c = this.expression.charAt(i);
				if (c.toUpperCase() === c.toLowerCase()) {
					if (i === this.pos || (c != '_' && (c < '0' || c > '9'))) {
						break;
					}
				}
				str += c;
			}
			if (str.length > 0 && (str in this.ops2)) {
				this.tokenindex = str;
				this.tokenprio = 5;
				this.pos += str.length;
				return true;
			}
			return false;
		},

		isVar: function () {
			var str = "";
			for (var i = this.pos; i < this.expression.length; i++) {
				var c = this.expression.charAt(i);
				if (c.toUpperCase() === c.toLowerCase()) {
					if (i === this.pos || (c != '_' && (c < '0' || c > '9'))) {
						break;
					}
				}
				str += c;
			}
			if (str.length > 0) {
				this.tokenindex = str;
				this.tokenprio = 4;
				this.pos += str.length;
				return true;
			}
			return false;
		},

		isComment: function () {
			var code = this.expression.charCodeAt(this.pos - 1);
			if (code === 47 && this.expression.charCodeAt(this.pos) === 42) {
				this.pos = this.expression.indexOf("*/", this.pos) + 2;
				if (this.pos === 1) {
					this.pos = this.expression.length;
				}
				return true;
			}
			return false;
		}
	};

	return self;
}
/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia and Félix J. García
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*/

/**
 * Framework for 2D drawing.
 * @module 2Ddrawing
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * InteractionTarget class
 * @class InteractionTarget
 * @constructor
 */
EJSS_DRAWING2D.InteractionTarget = {
	ENABLED_FIXED: 0,
	ENABLED_NONE : 0,
	ENABLED_ANY : 1,
	ENABLED_X : 2,
	ENABLED_Y : 3,
	ENABLED_NO_MOVE : 4,

    SENSITIVITY_BOTH : 0,
    SENSITIVITY_HORIZONTAL : 1,
    SENSITIVITY_VERTICAL : 2,
    SENSITIVITY_ANY : 3,
    
    
	// ----------------------------------------------------
	// Static methods
	// ----------------------------------------------------

	/**
	 * Copies one element into another
	 */
	copyTo : function(source, dest) {
		dest.setMotionEnabled(source.getMotionEnabled());
		dest.setAffectsGroup(source.getAffectsGroup());
		dest.setActionCommand(source.getActionCommand());
		dest.setDataObject(source.getDataObject());
		dest.setSensitivity(source.getSensitivity());
	}
};

/**
 * Creates an interaction object for 2D interaction
 */
EJSS_DRAWING2D.interactionTarget = function(mElement, mType, mPosition) {
	var InteractionTarget = EJSS_DRAWING2D.InteractionTarget;
	var self = {};

	var Element = EJSS_DRAWING2D.Element;	// reference for Element

	var mMotionEnabled = InteractionTarget.ENABLED_NONE;
	var mAffectsGroup = false;
	var mCommand = null;
	var mDataObject = null;
	var mSensitivity = 20;		// reference to interact, distance to element center
	var mSensitivityType = InteractionTarget.SENSITIVITY_BOTH;

	/**
	 * Returns the owner element
	 */
	self.getElement = function() {
		return mElement;
	};

	/**
	 * Returns the type of target
	 */
	self.getType = function() {
		return mType;
	};

	/**
	 * Sets the motion capability of the target
	 * @param value One of:
	 * <ul>
	 *   <li> EJSS_DRAWING2D.InteractionTarget.ENABLED_NONE: the target is not responsive</li>
	 *   <li> EJSS_DRAWING2D.InteractionTarget.ENABLED_ANY: any motion (x,y) is allowed</li>
	 *   <li> EJSS_DRAWING2D.InteractionTarget.ENABLED_X: the target only responds to motion in the X direction</li>
	 *   <li> EJSS_DRAWING2D.InteractionTarget.ENABLED_Y: the target only responds to motion in the Y direction</li>
	 *   <li> EJSS_DRAWING2D.InteractionTarget.ENABLED_NO_MOVE: the target fires events but cannot be moved</li>
	 * </ul>
	 * If not specified, then ENABLED_ANY is assumed.
	 */
	self.setMotionEnabled = function(motion) {
		//var result = InteractionTarget.ENABLED_ANY;
		if ( typeof motion == "string") {
			value = InteractionTarget[motion.toUpperCase()];
			mMotionEnabled = (typeof value === 'undefined')?InteractionTarget.ENABLED_NONE:value;
		}
	    else mMotionEnabled = motion;
	};

	/**
	 * Returns the (perhaps partial) interaction capability of the target
	 */
	self.getMotionEnabled = function() {
		return mMotionEnabled;
	};
	
	/**
	 * Sets the sensitivity type
	 * @param value One of:
	 * <ul>
	 *   <li> EJSS_DRAWING2D.InteractionTarget.SENSITIVITY_ANY: the target is selected if (x,y) is close to its position (default)</li>
	 *   <li> EJSS_DRAWING2D.InteractionTarget.SENSITIVITY_HORIZONTAL: the target is selected if x is close to its horizontal position</li>
	 *   <li> EJSS_DRAWING2D.InteractionTarget.SENSITIVITY_VERTICAL: the target is selected if y is close to its vertical position</li>
	 * </ul>
	 */
	self.setSensitivityType = function(sensitivityType) {
		if (typeof sensitivityType == "string")
			mSensitivityType = InteractionTarget[sensitivityType.toUpperCase()];
	    else mSensitivityType = sensitivityType;
	};

	/**
	 * Returns the (perhaps partial) interaction capability of the target
	 */
	self.getSensitivityType = function() {
		return mSensitivityType;
	};

	/**
	 * Returns the (perhaps partial) interaction capability of the target
	 */
	self.isEnabled = function() {
		return mMotionEnabled != InteractionTarget.ENABLED_NONE;
	};
	
	self.setPositionOffset = function(position) { 
		mPosition = position;
	}

	self.getPositionOffset = function() { 
		return mPosition;
	}

	/**
	 * Gets pixel position for interaction target
	 */
	self.getPixelPosition = function() {
		// get position of the element center
		var pos = mElement.getPixelPosition(true);
		var size = mElement.getPixelSizes(true);
		var offset = mElement.getRelativePositionOffset(size);
		var x = pos[0] + offset[0];
		var y = pos[1] + offset[1];

		// get target relative position
		var inverted = mElement.getPanel().getInvertedScaleY();
		var d = Element.getRelativePositionOffset(mPosition, size[0], size[1], inverted);

		return [x - d[0], y - d[1]];
	};
	
	/**
	 * Gets position for interaction target
	 */
	self.getPosition = function() {
		// get position of the element center
		var pos = mElement.getAbsolutePosition(true);
		var size = mElement.getAbsoluteSize();
		// get target relative position
		var inverted = mElement.getPanel().getInvertedScaleY();
		var d = Element.getRelativePositionOffset(mPosition, size[0], size[1], inverted);
		var offset = mElement.getRelativePositionOffset(size);
		var x = pos[0] + offset[0] - d[0];
		var y = pos[1] + offset[1] - d[1];
		return [x,y];
	};
	
	/**
	 * Whether the interaction with the target affects the top-level group
	 * of the element that contains it (instead of only affecting the element).
	 * Default is false.
	 * @param value boolean
	 */
	self.setAffectsGroup = function(value) {
		mAffectsGroup = value;
	};

	/**
	 * Whether the target affects the top-level group
	 * @return boolean
	 */
	self.getAffectsGroup = function() {
		return mAffectsGroup;
	};

	/**
	 * Sets the action command for this target
	 * @param command String
	 */
	self.setSensitivity = function(sense) {
		mSensitivity = sense;
	};

	/**
	 * Returns the action command of this target
	 * @return String
	 */
	self.getSensitivity = function() {
		return mSensitivity;
	};

	/**
	 * Sets the action command for this target
	 * @param command String
	 */
	self.setActionCommand = function(command) {
		mCommand = command;
	};

	/**
	 * Returns the action command of this target
	 * @return String
	 */
	self.getActionCommand = function() {
		return mCommand;
	};

	/**
	 * A place holder for data objects
	 * @param _object Object
	 */
	self.setDataObject = function(object) {
		mDataObject = object;
	};

	/**
	 * Returns the data object
	 * @return Object
	 */
	self.getDataObject = function() {
		return mDataObject;
	};

	return self;
};

/*
* Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia and Félix J. García
* This code is part of the Easy Javascript Simulations authoring and simulation tool
*/

/**
 * Framework for 2D drawing.
 * @module 2Ddrawing
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * PanelInteraction class
 * @class PanelInteraction
 * @constructor
 */
EJSS_DRAWING2D.PanelInteraction = {
	TARGET_POSITION : 0,
	TARGET_SIZE : 1
};

/**
 * Constructor for Interaction
 * @param drawing panel
 * @returns An interaction
 */
EJSS_DRAWING2D.panelInteraction = function(mPanel) {
	var self = {};
	// reference returned

	// Static references
	var PanelInteraction = EJSS_DRAWING2D.PanelInteraction;	// reference for PanelInteraction

	var mEnabled = false;		// whether interaction is enabled
	var mStopGestures = false;  // whether gesture propagation is enabled
	var mStopMoveEvents = false;// wheteher move event listener is enabled when not clicking (recomended in CANVAS)
	var mLastPoint = [];		// last position on panel trying interaction
	var mFirstPoint = [];		// first position on panel trying interaction
	var mPreviousPoint = [];	// position previous to last position
	var mOnlyTarget = null;		// only InteractionTarget with interaction
	var mIsPanelOnlyTarget = false;	// whether the target is the panel 
	var mMouseIsDown = false;	// last event is mouse down
	var mLastTime = 0;			// last timestamp for mouse up
	var mPinching = 0;			// pinching (0: nothing, 1: panel, 2: element)
	var mPinchInitDistance = [0,0];		// distance between touches in pinching
	var mPinchRelDistance = [0,0];		// relative distance when move in pinching
	var mZoomDelta = 0;					// zooming delta
	var mCursorTypeForMove = "move"; // type of cursor for move event (ex., "move", "crosshair")

	var mElementInteracted = 0;	// index of interacted element into its element array

	/**
	 * Reload the interaction configuration
	 */
	self.reload = function() {
		mLastPoint = [];
		mFirstPoint = [];		
		mPreviousPoint = [];
		mOnlyTarget = null;		
		mMouseIsDown = false;	
		mLastTime = 0;			
		mPinching = 0;
		mPinchInitDistance = [0,0];
		mPinchRelDistance = [0,0];
		mZoomDelta = 0;
		mElementInteracted = 0;	
		
		mEnabled = !mEnabled;
		self.setEnabled(!mEnabled);
		mStopGestures = !mStopGestures;
		self.setStopGestures(!mStopGestures);			
	}
	
	/**
	 * Return panel
	 * @return drawing panel
	 */
	self.getPanel = function() {
		return mPanel;
	};

	self.getEnabled = function() {
		return mEnabled;
	};

	self.setStopMoveEvents = function(stop) {
		if (mStopMoveEvents != stop) {
			mStopMoveEvents = stop;

			if (mStopMoveEvents) {
				self.setHandler("move", (function() { return true; }));
			} else {
				self.setHandler("move", self.handleMouseMoveEvent);
			}
		}		
	}

	self.setStopGestures = function(stop) {
		if (mStopGestures != stop) {
			mStopGestures = stop;

			if (mStopGestures) {
				self.setHandler("gesturestart", function(e) {e.preventDefault();e.stopPropagation();return false;});
				self.setHandler("gesturechange", function(e) {e.preventDefault();e.stopPropagation();return false;});
				self.setHandler("gestureend", function(e) {e.preventDefault();e.stopPropagation();return false;});
			} else {
				self.setHandler("gesturestart", function() { return true; });
				self.setHandler("gesturechange", function() { return true; });
				self.setHandler("gestureend", function() { return true; });
			}
		}		
	}

	self.setEnabled = function(enabled) {
		if (mEnabled != enabled) {
			mEnabled = enabled;

			if (mEnabled) {
				if (!mStopMoveEvents) {
					self.setHandler("move", self.handleMouseMoveEvent);
				}
				self.setHandler("down", self.handleMouseDownEvent);
				self.setHandler("up", self.handleMouseUpEvent);
				self.setHandler("mousewheel",self.handleMouseWheelEvent);
			} else {
				self.setHandler("move", (function() { return true; }));
				self.setHandler("down", (function() { return true; }));
				self.setHandler("up", (function() { return true; }));
				self.setHandler("mousewheel", (function() { return true; }));
			}
		}
	};
	
	self.setCursorTypeForMove = function(type) {
		mCursorTypeForMove = type;
	}
	
	/**
	 * Return the last interaction point 
	 * @method getInteractionPoint
	 * @return double[]
	 */
	self.getInteractionPoint = function() {
		return mPanel.toPanelPosition(mLastPoint);
	};

	/**
	 * Return the distance between the previous and last interaction
	 * @method getInteractionDistance
	 * @return double[]
	 */
	self.getInteractionDistance = function() {
		if(mPreviousPoint.length == 0 || mLastPoint.length == 0) return [];
		var p = mPanel.toPanelPosition(mPreviousPoint);
		var l = mPanel.toPanelPosition(mLastPoint);
		return [p[0] - l[0], p[1] - l[1]];
	};

	/**
	 * Return the bounds of the last interaction rectangle 
	 * @method getInteractionBounds
	 * @return double[]
	 */
	self.getInteractionBounds = function() {
		if(mFirstPoint.length == 0 || mLastPoint.length == 0) return [];
		var f = mPanel.toPanelPosition(mFirstPoint);
		var l = mPanel.toPanelPosition(mLastPoint);
		return [f[0], l[0], f[1], l[1]];
	};

	/**
	 * Return the zooming delta
	 * @method getInteractionZoomDelta
	 * @return double[]
	 */
	self.getInteractionZoomDelta = function() {
		// zooming via mousewheel or pinching 
		if(mZoomDelta == 0 && mPinchRelDistance[0] == 0) return 0;
		if(mZoomDelta > 0 || mPinchRelDistance[0] > 0) return 1;
		return -1;
	};
	
	/**
	 * Return the only interaction target (only one touch)
	 * @method getInteractionElement
	 * @return element
	 */
	self.getInteractionElement = function() {
		return mOnlyTarget;
	};

  /**
   * Clears the current interaction elementn if it equals the given one
   * (for example, if it was deleted)
   * @method clearInteractionElement
   * @return element
   */
  self.clearInteractionElement = function(element) {
    if (mOnlyTarget && element===mOnlyTarget.getElement()) mOnlyTarget = null;
  };

	/**
	 * Return the screen orientation
	 * @method getOrientation
	 * @return orientation
	 */
	self.getOrientation = function() {
		return window.orientation;
	};
	
	/**
	 * Handler for mouse move event
	 * @method handleMouseMoveEvent
	 * @param event
	 * @param location
	 */
	self.handleMouseMoveEvent = function(event) {
		if(mStopGestures) {
		  // Prevent the browser from doing its default thing (scroll, zoom)
		  event.preventDefault();
		  // prevent the browser from propagation
		  event.stopPropagation();
		}

		// number of fingers over screen	
		var nFingers = (typeof event.touches != "undefined")? event.touches.length:1;

		// manage pinch		 
		if(nFingers == 2) {
    		var e0 = event.touches[0]; var e1 = event.touches[1];
    		var e0x = (e0.clientX || e0.x); var e0y = (e0.clientY || e0.y);
    		var e1x = (e1.clientX || e1.x); var e1y = (e1.clientY || e1.y);
			if(mPinching != 0) {
	    		var newDistance = [Math.abs(e0x-e1x),Math.abs(e0y-e1y)];
	    		var delta = [mPinchInitDistance[0]-newDistance[0],mPinchInitDistance[1]-newDistance[1]]; // used in zooming
	    		if (delta[0]<8 && delta[0]>-8 && delta[1]<8 && delta[1]>-8) { // minimum filter
	    			mPinchRelDistance = [0,0];
	    		} else {
		    		mPinchRelDistance = delta;
		    		mPinchInitDistance = newDistance;
	    		}
			} else {
				mPinching = 1;
	    		mPinchInitDistance = [Math.abs(e0x-e1x),Math.abs(e0y-e1y)];
				mPinchRelDistance = [0,0];			
			}
		} else {
			mPinching = 0;
			mPinchInitDistance = [0,0];
		}
		
		// touch locations 
		var locations = getEventLocations(event);

		// last and previous location		
		mPreviousPoint = mLastPoint;
		mLastPoint = locations[locations.length-1];
		
		if (mMouseIsDown) { // Mouse or finger(s) is down (drag)
			if (mOnlyTarget != null) { // whether mOnlyTarget exists, only one touch and target
				// get element or group
				var element = mOnlyTarget.getAffectsGroup() ? mOnlyTarget.getElement().getGroup() : mOnlyTarget.getElement();
				// register properties changed
				var point = mPanel.toPanelPosition(mLastPoint);
				var refpoint = mPanel.toPanelPosition(mPreviousPoint);
				propertiesChanged(element, mOnlyTarget, point, refpoint);
				// invoke actions
				element.getController().invokeAction("OnDrag");				
				if (mEnabled)// also onMove over panel
					mPanel.getController().invokeAction("OnMove");
			} else if(mIsPanelOnlyTarget) { // the only target is the panel
				if (mEnabled) { // onDrag over panel
					mPanel.getController().invokeAction("OnDrag");
				}
			} else { // multiple touches
				// invoke actions
				var targetHitList = [];
				for(var i=0; i<locations.length; i++) {					
					var targetHit = self.findInteractionTarget(mPanel.getElements(), locations[i]);
					if(targetHit != null) {
						targetHitList.push(targetHit);
						// get element or group
						var element = targetHit.getAffectsGroup() ? targetHit.getElement().getGroup() : targetHit.getElement();
						// register properties changed
						var point = mPanel.toPanelPosition(locations[i]);
						propertiesChanged(element, targetHit, point, targetHit.getPosition());
						// invoke actions
						element.getController().invokeAction("OnDrag");				
						if (mEnabled) // also onMove over panel
							mPanel.getController().invokeAction("OnMove");			
					}
				}
				// actions over panel
				if (mEnabled) { 
					if(mPinching == 0)
						mPanel.getController().invokeAction("OnDrag");
					else if (mPinching == 1 && targetHitList.length == 0) 
						mPanel.getController().invokeAction("OnZoom");						
				}
			}
				
		} else { // Mouse is up (move over panel and elements using mouse)
			var targetHit = self.findInteractionTarget(mPanel.getElements(), locations[0]);
			if (targetHit === null) { // no target
				if (mOnlyTarget != null)// whether mOnlyTarget exists
					mOnlyTarget.getElement().getController().invokeAction("OnExit");
				if (mEnabled) // onMove over panel
					mPanel.getController().invokeAction("OnMove");
				event.target.style.cursor = 'default';
			} else {
				if (mOnlyTarget != targetHit) {
					if (mOnlyTarget != null)// whether mOnlyTarget exists
						mOnlyTarget.getElement().getController().invokeAction("OnExit");
					targetHit.getElement().getController().invokeAction("OnEnter");
				}
				if (targetHit.isEnabled())
					event.target.style.cursor = mCursorTypeForMove;
				else
					event.target.style.cursor = 'default';
			}
			mOnlyTarget = targetHit;				
		}

		// all interactions are reported to View
		mPanel.getController().reportInteractions();
		
		return false;
	};

	/**
	 * Handler for mouse down event
	 * @method handleMouseDownEvent
	 * @param event
	 * @param location
	 */
	self.handleMouseDownEvent = function(event) {
		mMouseIsDown = true;

		if (mStopMoveEvents) {
			// for drag&drop 
			self.setHandler("move", self.handleMouseMoveEvent);
		}
					
		if(mStopGestures) {
		  // prevent the browser from doing its default thing (scroll, zoom)
		  event.preventDefault();
		  // prevent the browser from propagation
		  event.stopPropagation();
		}

		// number of fingers over screen	
		var nFingers = (typeof event.touches != "undefined")? event.touches.length:1;
			
		// manage pinch		 
		if(nFingers != 2) {
			mPinching = 0;
		}
		
		// touch locations 
		var locations = getEventLocations(event);

		// first, previous and last points
		mFirstPoint = locations[0]; // first location
		mPreviousPoint = mFirstPoint;
		mLastPoint = locations[locations.length-1];	// last location
				 
		// invoke actions
		var isPanelHit = false;
		var targetHit = null;
		for(var i=0; i<locations.length; i++) {
			// find target
			targetHit = self.findInteractionTarget(mPanel.getElements(), locations[i]);
			if (targetHit !== null) { // one target
				targetHit.getElement().getController().invokeAction("OnPress");
				event.target.style.cursor = mCursorTypeForMove;
			} else if (mEnabled) { // no target, then onPress over panel
				mPanel.getController().invokeAction("OnPress");
				isPanelHit = true;
			}
		}		
		
		// mOnlyTarget and mIsPanelOnlyTarget are usefull whether only one touch
		if(locations.length == 1 && nFingers == 1) { 					
			// targetHit is null if no target, so the target is the panel 
			mOnlyTarget = targetHit;	// last target
			mIsPanelOnlyTarget = isPanelHit;
		} else {
			// no only target
			mOnlyTarget = null;
			mIsPanelOnlyTarget = false;			
		}
		
		// all interactions are reported to View
		mPanel.getController().reportInteractions();		
		
		return false;
	};

	/**
	 * Handler for mouse up event
	 * @method handleMouseUpEvent
	 * @param event
	 * @param location
	 */
	self.handleMouseUpEvent = function(event) {
		mMouseIsDown = false;

		if (mStopMoveEvents) {
			// for drag&drop
			self.setHandler("move", (function() { return true; }));
		}

		if(mStopGestures) {
		  // prevent the browser from doing its default thing (scroll, zoom)
		  event.preventDefault();
		  // prevent the browser from propagation
		  event.stopPropagation();
		}

		// manage pinch		 
		if(mPinching != 0) { 
    		mPinching = 0;
    		mPinchRelDistance = [0,0];
    		mPinchInitDistance = [0,0];
		}

		// touch locations 
		var locations = getEventLocations(event);

		// last location
		mPreviousPoint = mLastPoint;
		mLastPoint = locations[locations.length-1];
			
		// whether event is dbclick
		var dblclick = (event.timeStamp - mLastTime < 500); // < 500 ms is dblclick
			
		if(mOnlyTarget != null) { // only one touch
			// mOnlyTarget is null if no element touched						
			mOnlyTarget.getElement().getController().invokeAction("OnRelease");
			if (dblclick) mOnlyTarget.getElement().getController().invokeAction("OnDoubleClick");
		} else {				
			// invoke actions		
			for(var i=0; i<locations.length; i++) {
				// find target
				var targetHit = self.findInteractionTarget(mPanel.getElements(), locations[i]);
				if (targetHit !== null) { // one target
					targetHit.getElement().getController().invokeAction("OnRelease");
					if (dblclick) targetHit.getElement().getController().invokeAction("OnDoubleClick");
				} else if (mEnabled) { // no target, then onRelease over panel
					mPanel.getController().invokeAction("OnRelease");
					if (dblclick) mPanel.getController().invokeAction("OnDoubleClick");
				}
			}
		}
				
		// last time to manage double click
		mLastTime = event.timeStamp; 

		// cursor style
		event.target.style.cursor = 'default';

		// all interactions are reported to View
		mPanel.getController().reportInteractions();
		
		return false;
	};

	/**
	 * Handler for mouse wheel event
	 * @method handleMouseWheelEvent
	 * @param event
	*/
	self.handleMouseWheelEvent = function(event) {
		// number of fingers over screen	
		var nFingers = (typeof event.touches != "undefined")? event.touches.length:0;
		
		if(nFingers == 0) {  	
			if(mStopGestures) {
			  // prevent the browser from doing its default thing (scroll, zoom)
			  event.preventDefault();
			  // prevent the browser from propagation
			  event.stopPropagation();
			}
	
			mZoomDelta = Math.max(-1, Math.min(1, (event.wheelDelta || -event.detail)));
			mPanel.getController().invokeAction("OnZoom");	
	        
		    // all interactions are reported to View
		    mPanel.getController().reportInteractions();
		}
	};
	
	/**
	 * Find target element for the position
	 * @method findInteractionTarget
	 * @param location double[]
	 * @return double[]
	 */
	self.findInteractionTarget = function(elements, location) {
	    var InteractionTarget = EJSS_DRAWING2D.InteractionTarget;
		var target = null;
		for (var i = elements.length - 1; i >= 0; i--) {
			var element = elements[i];
			if (element.isGroupVisible()) {
				if (element.getElements) {// array of elements
					target = self.findInteractionTarget(element.getElements(), location);
					if (target !== null)
						return target;
				} else {// one element
					// find interaction target for element
					var targets = element.getInteractionTargets();
					for (var j = 0; j < targets.length; j++) {
						if (targets[j].isEnabled()) {
							var sensitivity = targets[j].getSensitivity();
							if (sensitivity > 0) {
								var projectedPosition = targets[j].getPixelPosition();
								var selected = false;
								switch(targets[j].getSensitivityType()) {
								  case InteractionTarget.SENSITIVITY_HORIZONTAL :
								    selected = (Math.abs(projectedPosition[1] - location[1]) < sensitivity); 
								    break;
								  case InteractionTarget.SENSITIVITY_VERTICAL :
								    selected = (Math.abs(projectedPosition[0] - location[0]) < sensitivity); 
								    break;
								  case InteractionTarget.SENSITIVITY_ANY :
								    selected = (Math.abs(projectedPosition[0] - location[0]) < sensitivity || Math.abs(projectedPosition[1] - location[1]) < sensitivity);
								    break;
								  default : // InteractionTarget.SENSITIVITY_BOTH
								    selected = (Math.abs(projectedPosition[0] - location[0]) < sensitivity && Math.abs(projectedPosition[1] - location[1]) < sensitivity);
								    break;
								}
								if (selected) {
									mElementInteracted = i;
									// index of element into element array
									// var pp = element.toGroupSpace(element.getPosition(),true);
									// console.log('pp' + pp[0] + ' ' + pp[1]);
									
									return targets[j];
								}
							}
					    }
					}
					// maybe, any location into element (pinching?)
					if(mPinching != 0) {
						target = element.getInteractionTarget(PanelInteraction.TARGET_SIZE);
						if (target.isEnabled() && !target.getSensitivity()) {
							var rect = element.getAbsoluteBounds();
							var pos = mPanel.toPanelPosition(location);
							if ((rect.left < pos[0]) && (rect.right > pos[0]) && (rect.bottom < pos[1]) && (rect.top > pos[1])) {
								mElementInteracted = i;
								// index of element into element array
								mPinching = 2; 
								return target;
							}
						}
					} 
					// maybe, any location into element (position interaction?)
					target = element.getInteractionTarget(PanelInteraction.TARGET_POSITION);
					if (target.isEnabled() && !target.getSensitivity()) {
						var rect = element.getAbsoluteBounds();
						var pos = mPanel.toPanelPosition(location);
						if ((rect.left < pos[0]) && (rect.right > pos[0]) && (rect.bottom < pos[1]) && (rect.top > pos[1])) {
							mElementInteracted = i;
							// index of element into element array
							return target;
						}
					}						
				}
			}
		}
		return null;
	};

	/**
	 * Return index of interacted element into element array
	 */
	self.getIndexElement = function() {
		return mElementInteracted;
	};
	
	/**
	 * Register properties changed for a motion of the mouse when dragging the element
	 * For subclasses use only!
	 */
	function propertiesChanged(element, target, point, targetPosition) {
		var InteractionTarget = EJSS_DRAWING2D.InteractionTarget;
		var PanelInteraction = EJSS_DRAWING2D.PanelInteraction;

		// reports changes for element
		if (target.getType() === PanelInteraction.TARGET_POSITION) {
    	    var dx = point[0]-targetPosition[0];
    	    var dy = point[1]-targetPosition[1];
			switch (target.getMotionEnabled()) {
				case InteractionTarget.ENABLED_ANY :
					element.setX(element.getX() + dx);
					element.setY(element.getY() + dy);
					element.getController().propertiesChanged("Position", "X", "Y");
					target.getElement().setChanged(true);
					element.setChanged(true);
					break;
				case InteractionTarget.ENABLED_X :
					element.setX(element.getX() + dx);
					element.getController().propertiesChanged("Position", "X");
					target.getElement().setChanged(true);
					element.setChanged(true);
					break;
				case InteractionTarget.ENABLED_Y :
					element.setY(element.getY() + dy);
					element.getController().propertiesChanged("Position", "Y");
					target.getElement().setChanged(true);
					element.setChanged(true);
					break;
			}
		} else if (target.getType() === PanelInteraction.TARGET_SIZE) {
			if(mPinching == 2) {
				// not used targetPosition, directly we have the pinching distance
	    	    var diff = mPanel.toPanelMod(mPinchRelDistance);
	    	    var dx = diff[0];
	    	    var dy = -diff[1];		
			} else {
	    	    var dx = point[0]-targetPosition[0];
	    	    var dy = point[1]-targetPosition[1];
			}
				
		    var targetElement = target.getElement();
			var absSize = targetElement.getAbsoluteSize();
			if (absSize[0]===0) absSize[0] = 1.0e-5;
			if (absSize[1]===0) absSize[1] = 1.0e-5;
			var rsizes = [dx/absSize[0], dy/absSize[1]];
			
			var size = element.getSize();
			if (size[0]===0 && rsizes[0]!==0) size[0] = 1.0e-5;
			if (size[1]===0 && rsizes[0]!==0) size[1] = 1.0e-5;				

			switch (target.getMotionEnabled()) {
				case InteractionTarget.ENABLED_ANY :
					element.setSizeX(size[0] + size[0] * rsizes[0]);
					element.setSizeY(size[1] + size[1] * rsizes[1]);					
					element.getController().propertiesChanged("Size", "SizeX", "SizeY");
					target.getElement().setChanged(true);
					element.setChanged(true);
					break;
				case InteractionTarget.ENABLED_X :
					element.setSizeX(size[0] + size[0] * rsizes[0]);
					element.getController().propertiesChanged("Size", "SizeX");
					target.getElement().setChanged(true);
					element.setChanged(true);
					break;
				case InteractionTarget.ENABLED_Y :
					element.setSizeY(size[1] + size[1] * rsizes[1]);					
					element.getController().propertiesChanged("Size", "SizeY");
					target.getElement().setChanged(true);
					element.setChanged(true);
					break;
			}
		}
	}

	/**
	 * Get locations for the touch events
	 * @method getEventLocations
	 * @param {Object} e
	 */
	function getEventLocations(e) {
		var box = mPanel.getGraphics().getBox();
		var oleft = box.left; // offset left in pixels
		var otop = box.top; // offset top in pixels
		var locations = [];

		if ( typeof e.changedTouches != "undefined") {
			for(var i=0; i<e.changedTouches.length; i++) {
				locations[i] = [];		
				// ignored window.pageYOffset and window.pageXOffset
		    	locations[i][0] = e.changedTouches[i].clientX - oleft;
		    	locations[i][1] = e.changedTouches[i].clientY - otop;
		   }
		} else {
			locations[0] = [];
			locations[0][0] = (e.clientX || e.x) - oleft;
			locations[0][1] = (e.clientY || e.y) - otop;
		}

		return locations;
	}

	self.setHandler = function(type, handler) {
		var graphics = mPanel.getGraphics();
		var context = graphics.getEventContext();
		switch (type) {
			case "move" :
				context.addEventListener('mousemove', handler, false);
				context.addEventListener('touchmove', handler, false);
				break;
			case "down" :
				context.addEventListener('mousedown', handler, false);
				context.addEventListener('touchstart', handler, false);
				break;
			case "up" :
				context.addEventListener('mouseup', handler, false);
				context.addEventListener('touchend', handler, false);
				break;
		   	case "mousewheel" :
		      	context.addEventListener('mousewheel', handler, false);
			  	break;	      

			// not used 
			case "gesturestart" : // Sent when the second finger touches the surface.
				context.addEventListener('gesturestart', handler, false);
				break;
			case "gesturechange" : //  Sent when both fingers move while still touching the surface.
				context.addEventListener('gesturechange', handler, false);
				break;
			case "gestureend" : //  Sent when the second finger lifts from the surface.
				context.addEventListener('gestureend', handler, false);
				break;
			default :
		}
		return false;
	};
	
	// change orientation
	var _super_onorientationchange = window.onorientationchange; 
		
	window.onorientationchange = function() {		
		if(_super_onorientationchange) _super_onorientationchange();
		mPanel.getController().invokeAction("OnOrientationChange");
		mPanel.getController().reportInteractions();
	};
	
	// resize
	var _super_onresize = window.onresize; 
	window.onresize = function() {
		if(_super_onresize) _super_onresize();		
		mPanel.getController().invokeAction("OnResize");
		mPanel.getController().reportInteractions();
	};

	// for default, stop propagation
	self.setStopGestures(true);
	
	return self;
};
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Style object for 2D drawing
 * @class Style 
 * @constructor  
 */
EJSS_DRAWING2D.Style = {  
  RENDER_AUTO : "auto",
  RENDER_OPTSPEED: "optimizeSpeed",
  RENDER_CRISPEDGES: "crispEdges",
  RENDER_GEOPRECISION: "geometricPrecision", 
  
  // ----------------------------------------------------
  // Static methods
  // ----------------------------------------------------

  /**
   * Copies one element into another
   */
  copyTo : function(source, dest) {
    dest.setDrawLines(source.getDrawLines());
    dest.setLineColor(source.getLineColor());
    dest.setLineWidth(source.getLineWidth());
    dest.setDrawFill(source.getDrawFill());
    dest.setFillColor(source.getFillColor());
    dest.setAttributes(source.getAttributes());
    dest.setShapeRendering(source.getShapeRendering());
  }
  
};

/**
 * Creates a Style object for 2D drawing
 */
EJSS_DRAWING2D.style = function (mName) {
  var Style = EJSS_DRAWING2D.Style;
  var self = {};
   
  var mDrawLines = true;
  var mLineColor = 'black';
  var mLineWidth = 0.5;
  var mDrawFill = true;
  var mFillColor = 'none';
  var mShapeRendering = Style.RENDER_AUTO;
  var mChangeListener; 
  var mAttributes = {};     

  /**
   * Set a listener that will be called whenever there are style changes.
   * I.e. a call to listener("change"); will be issued
   */
  self.setChangeListener = function(listener) {
    mChangeListener = listener;
  };
  
  //---------------------------------
  // lines
  //---------------------------------

  /**
   * Whether to draw the lines of the element
   */
  self.setDrawLines = function(draw) {
    if (draw!=mDrawLines) {
      mDrawLines = draw;
      if (mChangeListener) mChangeListener("drawlines");
    }
  };
  
  /**
   * Get the draw lines flag
   */
  self.getDrawLines = function() { 
    return mDrawLines; 
  };
  
  /**
   * Set the line color of the element
   * @param color a stroke style
   */
  self.setLineColor = function(color) { 
    if (typeof color !== "string") color = EJSS_TOOLS.DisplayColors.getLineColor(color);
    if (color!=mLineColor) {
      mLineColor = color; 
      if (mChangeListener) mChangeListener("linecolor");
    }
    return self;
  };
    
  /**
   * Get the line color
   */
  self.getLineColor = function() { 
    return mLineColor; 
  };

  /**
   * Set the line width of the element
   * @param width a stroke width (may be double, such as 0.5, the default)
   */
  self.setLineWidth = function(width) { 
    if (width!=mLineWidth) {
      mLineWidth = width; 
      if (mChangeListener) mChangeListener("linewidth");
    }
  };

  /**
   * Get the line width
   */
  self.getLineWidth = function() { return mLineWidth; };
  
  //---------------------------------
  // interior fill
  //---------------------------------

  /**
   * Whether to fill the interior of the element
   */
  self.setDrawFill = function(draw) {
    if (draw!=mDrawFill) {
      mDrawFill = draw;
      if (mChangeListener) mChangeListener("drawfill");
    }
  };
  
  /**
   * Get the draw fill flag
   */
  self.getDrawFill = function() { 
    return mDrawFill; 
  };

  /**
   * Set the fill color of the element
   * @param color a fill style
   */
  self.setFillColor = function(color) {
    if (typeof color !== "string") color = EJSS_TOOLS.DisplayColors.getLineColor(color);
    if (color!=mFillColor) {
      mFillColor = color; 
      if (mChangeListener) mChangeListener("fillcolor");
    }
  };
  
  /**
   * Get the fill color
   */
  self.getFillColor = function() { 
    return mFillColor; 
  };

  /**
   * Sets shape rendering
   */
  self.setShapeRendering = function(rendering) {
    if (rendering.substring(0,6) == "RENDER") rendering = Style[rendering.toUpperCase()];
    if (mShapeRendering != rendering) {
      mShapeRendering = rendering;
      if (mChangeListener) mChangeListener("shaperendering");
    }
  };
  
  self.getShapeRendering = function() { 
    return mShapeRendering;
  };

  /**
   * Sets optional attributes used by the element (such as SVG attributes)
   */
  self.setAttributes = function(attr) {
    if (attr != mAttributes) {
      mAttributes = attr;
      if (mChangeListener) mChangeListener("attributes");
    }
  };
  
  self.getAttributes = function() { 
    return mAttributes;
  };
  
  /***
   * Get JSON object with private variables
   * @method serialize
   * @visibility private
   */
  self.serialize = function() {
  	return { 
		mDrawLines: mDrawLines, mLineColor: mLineColor, mLineWidth: mLineWidth,
		mDrawFill: mDrawFill, mFillColor: mFillColor, mShapeRendering: mShapeRendering,
		mAttributes: mAttributes
  	};
  }
  
  /***
   * Set JSON object with private variables
   * @method unserialize
   * @parem json JSON object
   * @visibility private
   */
  self.unserialize = function(json) {
	mDrawLines = json.mDrawLines, mLineColor = json.mLineColor, mLineWidth = json.mLineWidth,
	mDrawFill = json.mDrawFill, mFillColor = json.mFillColor, mShapeRendering = json.mShapeRendering,
	mAttributes = json.mAttributes;
  }  
  
  //---------------------------------
  // final initialization
  //---------------------------------
  
  return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Transformation
 * @class Transformation 
 * @constructor  
 */
EJSS_DRAWING2D.Transformation = {
    
    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

    /**
     * @method compare
     */
    compare : function(mtr1, mtr2) {    	
    	return( mtr1.radians == mtr2.radians && mtr1.a == mtr2.a && mtr1.b == mtr2.b && mtr1.c == mtr2.c && mtr1.d == mtr2.d
    		 && mtr1.e == mtr2.e && mtr1.f == mtr2.f  )
    }
}

/**
 * Transformation object for 2D drawing
 * @class transformation 
 * @constructor  
 */

/**
 * Creates a transformation object for 2D drawing
 */
EJSS_DRAWING2D.transformation = function(transform) {
  transform = transform || { a:1, b:0, c:0, d:1, e:0, f:0 };
  var self = {
    a : transform.a, c : transform.c, e : transform.e,
    b : transform.b, d : transform.d, f : transform.f,
    radians : 0
  };
  
  self.toString = function() {
    return "a="+self.a+", c="+self.c+", e="+self.e+"\n"+
           "b="+self.b+", d="+self.d+", f="+self.f;  
  };
   
  self.setToIdentity = function() {
    self.a = 1; self.c = 0; self.e = 0; 
    self.b = 0; self.d = 1; self.f = 0;
    radians = 0;
  };
  
  self.setTransform = function(tr) {
    if (Array.isArray(tr) && tr.length>5) {
      self.a = tr[0]; self.c = tr[2]; self.e = tr[4];
      self.b = tr[1]; self.d = tr[3]; self.f = tr[5];
    } 
    else {
      self.a = tr.a; self.c = tr.c; self.e = tr.e; 
      self.b = tr.b; self.d = tr.d; self.f = tr.f;
      self.radians = tr.radians;
    }
  };
  
  self.setToTranslation = function(x,y) {
    self.a = 1; self.c = 0; self.e = x; 
    self.b = 0; self.d = 1; self.f = y;
  };

  self.setToRotation = function(radians) {
    var cos = Math.cos(radians), sin = Math.sin(radians);
    self.a = cos;  self.c = -sin; self.e = 0; 
    self.b = sin; self.d = cos; self.f = 0;
    self.radians = radians;
  };

  self.getRotation = function() {
    return self.radians;
  };
  
  self.translate = function(dx,dy) {
    self.e += dx;
    self.f += dy;
  };
  
  self.scale = function(sx,sy) {
    self.a *= sx; self.b *= sx;
    self.c *= sy; self.d *= sy;
  };
  
  self.concatenate = function(tr) {
    var a = self.a, b=self.b, c=self.c, d=self.d, e=self.e, f=self.f;
    self.a = a*tr.a + c*tr.b;
    self.b = b*tr.a + d*tr.b;
    self.c = a*tr.c + c*tr.d;
    self.d = b*tr.c + d*tr.d;
    self.e = a*tr.e + c*tr.f + e,
    self.f = b*tr.e + d*tr.f + f;
    self.radians += tr.radians;
  };
  
  self.transform = function(point) {
    var x = point[0], y = point[1];
    point[0] = self.a*x + self.c*y + self.e;
    point[1] = self.b*x + self.d*y + self.f;
    return point;
  };

  self.transformVector = function(vector) {
    var x = vector[0], y = vector[1];
    vector[0] = self.a*x + self.c*y;
    vector[1] = self.b*x + self.d*y;
    return vector;
  };
  
  self.inverseTransform = function(point) {
    var a = self.a, b=self.b, c=self.c, d=self.d, e=self.e, f=self.f;
    var den = a*d-b*c;
    if (den===0) return null;
    var x = point[0], y = point[1];
    point[0] = ( d*x - c*y + (c*f - d*e))/den;
    point[1] = (-b*x + a*y + (b*e - a*f))/den;
    return point;
  };
  
  //---------------------------------
  // final initialization
  //---------------------------------
  
  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * Video
 * @class Video 
 * @constructor  
 */
EJSS_DRAWING2D.Video = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

    /**
     * static registerProperties method
     */
    registerProperties : function(element,controller) {
      EJSS_DRAWING2D.Element.registerProperties(element,controller); // super class

      controller.registerProperty("VideoUrl",element.setVideoUrl);
      controller.registerProperty("CurrentTime",element.setCurrentTime);
      controller.registerProperty("ShowControls",element.setControls);
      controller.registerProperty("AutoPlay",element.setAutoPlay);
      controller.registerProperty("Type",element.setType);
      controller.registerProperty("Poster",element.setPoster);
      controller.registerProperty("Loop",element.setLoop);
      controller.registerProperty("WebCam",element.setWebCam);      
    },
};

/**
 * Creates a 2D Segment
 * @method video
 */
EJSS_DRAWING2D.video = function (name) {
  var self = EJSS_DRAWING2D.element(name);

  var mLoop = false;
  var mWebCam = false;
  var mType = "";
  var mPoster = "";
  var mAutoPlay = true;
  var mCurrentTime = 0;
  var mControls = true;
  var mPlay = true;
  var mUrl;		// Video url

  self.getClass = function() {
  	return "ElementVideo";
  }

  self.setLoop = function(loop) {
  	if(mLoop != loop) {
  		mLoop = loop;
  		self.setChanged(true);
  	}
  }	

  self.getLoop = function() {
  	return mLoop;
  }	

  self.setPoster = function(poster) {
  	if(mPoster != poster) {
  		mPoster = poster;
  		self.setChanged(true);
  	}
  }	

  self.getPoster = function() {
  	return mPoster;
  }	

  self.setType = function(type) {
  	if(mType != type) {
  		mType = type;
  		self.setChanged(true);
  	}
  }	

  self.getType = function() {
  	return mType;
  }	

  self.setAutoPlay = function(autoPlay) {
  	if(mAutoPlay != autoPlay) {
  		mAutoPlay = autoPlay;
  		mPlay = autoPlay;
  		self.setChanged(true);
  	}
  }	

  self.getAutoPlay = function() {
  	return mAutoPlay;
  }	

  self.setWebCam = function(webCam) {
  	if(mWebCam != webCam) {
  		mWebCam = webCam;
  		self.setChanged(true);
  	}
  }	

  self.getWebCam = function() {
  	return mWebCam;
  }	

// play,stop,controls

  self.setVideoUrl = function(url) {
  	if(mUrl != url) {
  		mUrl = url;
  		self.setChanged(true);
  	}
  }	

  self.getVideoUrl = function() {
  	return mUrl;
  }	

  self.setCurrentTime = function(currentTime) {
  	if(mCurrentTime != currentTime) {
  		mCurrentTime = currentTime;
  		self.setChanged(true);
  	}
  }	

  self.getCurrentTime = function() {
  	return mCurrentTime;
  }	

  self.setControls = function(controls) {
  	if(mControls != controls) {
  		mControls = controls;
  		self.setChanged(true);
  	}
  }	

  self.getControls = function() {
  	return mControls;
  }	

  self.play = function() {
  	if(!mPlay) {
	  	mPlay = true;
	  	self.setChanged(true);
	}
  }	

  self.stop = function() {
  	if(mPlay) {
	  	mPlay = false;
	  	mAutoPlay = false;
	  	self.setChanged(true);
	}
  }	
  
  self.isPlay = function() {
  	return mPlay;
  }

  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.Video.registerProperties(self,controller);
  };
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * EJSS framework for drawing2D elements
 * @module drawing2D 
 */

var EJSS_DRAWING2D = EJSS_DRAWING2D || {};

/**
 * WebCamImage
 * @class WebCamImage 
 * @extends Image
 * @constructor  
 */
EJSS_DRAWING2D.WebCamImage = {
  // ----------------------------------------------------
  // Static methods
  // ----------------------------------------------------

  /**
   * static registerProperties method
   */ 
   registerProperties : function(element,controller) {
    EJSS_DRAWING2D.Image.registerProperties(element,controller); // super class

    controller.registerProperty("Url",element.setUrl);
    controller.registerProperty("On",element.setOnOff, element.isOnOff)

  },

};

/**
 * motionJPEG function
 * Creates a basic MotionJPEG image element
 * @method motionJPEG
 * @param name the name of the element
 * @returns A MotionJPEG image element
 */
EJSS_DRAWING2D.webCamImage = function (mName) {
  var self = EJSS_DRAWING2D.image(mName);
  navigator.getUserMedia  = navigator.getUserMedia ||
                          navigator.webkitGetUserMedia ||
                          navigator.mozGetUserMedia ||
                          navigator.msGetUserMedia;
  
  var mPlaying = true;
  var mUrl;
  var mPreviousUrl;
  
  self.setUrl = function(url) {
    //if (self.getResourcePath) url = self.getResourcePath(url);
    mUrl = url;
  }	

  self.getUrl = function() {
  	return mUrl;
  }	

  self.setOnOff = function(on) {
	if (on) {
	  if (!mPlaying) self.play();
	}
	else {
	  if (mPlaying) self.pause();
	}
  }

  self.isOnOff = function() {
	return mPlaying;
  }

  self.play = function() {
	if (mUrl) {
      if (mUrl != self.getImageUrl()) {
        mPreviousUrl = self.getImageUrl();
        if (mUrl=="local:") {
          if (navigator.getUserMedia) {
            navigator.getUserMedia({video: true}, 
              function(stream) { self.forceImageUrl(stream); }, 
              errorCallback);
          } 
          else self.setImageUrl(mPreviousUrl);
        }
        else self.setImageUrl(mUrl);
	  }
	}
	mPlaying = true;
  }

  self.pause = function() {
	mPlaying = false;
	if (mPreviousUrl) self.setImageUrl(mPreviousUrl);
  }

/*
  self.dataCollectedno = function() {
  	if (mPlaying && mUrl) {
    	// self.setChanged(true);
    	self.setChangedImage(true);
    }
  }
  
*/

  /**
   * Extended registerProperties method. To be used by promoteToControlElement
   * @method registerProperties
   */
  self.registerProperties = function(controller) {
    EJSS_DRAWING2D.WebCamImage.registerProperties(self,controller);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  return self;
};
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/**
 * Surface
 * @class Surface 
 * @constructor  
 */
EJSS_DRAWING3D.AnalyticCurve = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------


	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING3D.Element.registerProperties(element, controller);
		// super class

		controller.registerProperty("NumPoints", element.setNumPoints);
		controller.registerProperty("MinValue", element.setMinValue);
		controller.registerProperty("MaxValue", element.setMaxValue);
		controller.registerProperty("Variable", element.setVariable);

		controller.registerProperty("FunctionX", element.setExpressionX);
		controller.registerProperty("FunctionY", element.setExpressionY);
		controller.registerProperty("FunctionZ", element.setExpressionZ);
	}

};

/**
 * Creates a 3D Surface
 * @method surface
 */
EJSS_DRAWING3D.analyticCurve = function (name) {
  var self = EJSS_DRAWING3D.element(name);

  // Implementation variables
  var mParser = EJSS_DRAWING2D.functionsParser();
  var mExpressionX = "t";
  var mExpressionY = "t";
  var mExpressionZ = "0";

  var mNumPoints = screen.width / 10;
  var mMinValue = 0;
  var mMaxValue = 0;
  var mVariable = "t";
  
  self.getClass = function() {
  	return "ElementAnalyticCurve";
  }

  self.setNumPoints = function(n) {
  	if(mNumPoints != n) {
  	  mNumPoints = n;
  	  self.setMeshChanged(true);
  	}
  }

  self.setMinValue = function(n) {
  	if(mMinValue != n) {
  	  mMinValue = n;
  	  self.setMeshChanged(true);
  	}
  }

  self.setMaxValue = function(n) {
  	if(mMaxValue != n) {
  	  mMaxValue = n;
  	  self.setMeshChanged(true);
  	}
  }

  self.setVariable = function(n) {
  	if(mVariable != n) {
  	  mVariable = n;
  	  self.setMeshChanged(true);
  	}
  }

  self.setExpressionX = function(exp) {
  	if(mExpressionX != exp) {
  	  mExpressionX = exp;
  	  self.setMeshChanged(true);
  	}
  }

  self.setExpressionY = function(exp) {
  	if(mExpressionY != exp) {
  	  mExpressionY = exp;
  	  self.setMeshChanged(true);
  	}
  }

  self.setExpressionZ = function(exp) {
  	if(mExpressionZ != exp) {
  	  mExpressionZ = exp;
  	  self.setMeshChanged(true);
  	}
  }

  self.getNumPoints = function() {
  	return mNumPoints;
  }

  self.getMinValue = function() {
  	return mMinValue;
  }

  self.getMaxValue = function() {
  	return mMaxValue;
  }

  self.getVariable = function() {
  	return mVariable;
  }

  self.getExpressionX = function() {
  	return mParser.parse(mExpressionX);
  }

  self.getExpressionY = function() {
  	return mParser.parse(mExpressionY);
  }

  self.getExpressionZ = function() {
  	return mParser.parse(mExpressionZ);
  }

  self.registerProperties = function(controller) {
	EJSS_DRAWING3D.AnalyticCurve.registerProperties(self, controller);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([1,1,1]);
  self.getStyle().setDrawFill(false);
  self.getStyle().setDrawLines(true);

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/**
 * Surface
 * @class Surface 
 * @constructor  
 */
EJSS_DRAWING3D.AnalyticSurface = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------


	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING3D.Element.registerProperties(element, controller);
		// super class

		controller.registerProperty("NumPoints1", element.setNumPoints1);
		controller.registerProperty("MinValue1", element.setMinValue1);
		controller.registerProperty("MaxValue1", element.setMaxValue1);
		controller.registerProperty("Variable1", element.setVariable1);

		controller.registerProperty("NumPoints2", element.setNumPoints2);
		controller.registerProperty("MinValue2", element.setMinValue2);
		controller.registerProperty("MaxValue2", element.setMaxValue2);
		controller.registerProperty("Variable2", element.setVariable2);

		controller.registerProperty("FunctionX", element.setExpressionX);
		controller.registerProperty("FunctionY", element.setExpressionY);
		controller.registerProperty("FunctionZ", element.setExpressionZ);
	}

};

/**
 * Creates a 3D Surface
 * @method surface
 */
EJSS_DRAWING3D.analyticSurface = function (name) {
  var self = EJSS_DRAWING3D.element(name);

  // Implementation variables
  var mParser = EJSS_DRAWING2D.functionsParser();
  var mExpressionX = "u";
  var mExpressionY = "v";
  var mExpressionZ = "0";

  var mNumPoints1 = 100;
  var mMinValue1 = 0;
  var mMaxValue1 = 0;
  var mVariable1 = "u";
  
  var mNumPoints2 = 100;
  var mMinValue2 = 0;
  var mMaxValue2 = 0;
  var mVariable2 = "v";  

  self.getClass = function() {
  	return "ElementAnalyticSurface";
  }

  self.setNumPoints1 = function(n) {
  	if(mNumPoints1 != n) {
  	  mNumPoints1 = n;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.setMinValue1 = function(n) {
  	if(mMinValue1 != n) {
  	  mMinValue1 = n;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.setMaxValue1 = function(n) {
  	if(mMaxValue1 != n) {
  	  mMaxValue1 = n;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.setVariable1 = function(n) {
  	if(mVariable1 != n) {
  	  mVariable1 = n;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.setNumPoints2 = function(n) {
  	if(mNumPoints2 != n) {
  	  mNumPoints2 = n;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.setMinValue2 = function(n) {
  	if(mMinValue2 != n) {
  	  mMinValue2 = n;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.setMaxValue2 = function(n) {
  	if(mMaxValue2 != n) {
  	  mMaxValue2 = n;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.setVariable2 = function(n) {
  	if(mVariable2 != n) {
  	  mVariable2 = n;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.setExpressionX = function(exp) {
  	if(mExpressionX != exp) {
  	  mExpressionX = exp;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.setExpressionY = function(exp) {
  	if(mExpressionY != exp) {
  	  mExpressionY = exp;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.setExpressionZ = function(exp) {
  	if(mExpressionZ != exp) {
  	  mExpressionZ = exp;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.getNumPoints1 = function() {
  	return mNumPoints1;
  }

  self.getMinValue1 = function() {
  	return mMinValue1;
  }

  self.getMaxValue1 = function() {
  	return mMaxValue1;
  }

  self.getVariable1 = function() {
  	return mVariable1;
  }

  self.getNumPoints2 = function() {
  	return mNumPoints2;
  }

  self.getMinValue2 = function() {
  	return mMinValue2;
  }

  self.getMaxValue2 = function() {
  	return mMaxValue2;
  }

  self.getVariable2 = function() {
  	return mVariable2;
  }

  self.getExpressionX = function() {
  	return mParser.parse(mExpressionX);
  }

  self.getExpressionY = function() {
  	return mParser.parse(mExpressionY);
  }

  self.getExpressionZ = function() {
  	return mParser.parse(mExpressionZ);
  }

  self.registerProperties = function(controller) {
	EJSS_DRAWING3D.AnalyticSurface.registerProperties(self, controller);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([1,1,1]);

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/***
 * Arrow is a class to display vectors in 3D
 * @class EJSS_DRAWING3D.Arrow 
 * @constructor  
 */
EJSS_DRAWING3D.Arrow = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING3D.Element.copyTo(source,dest); // super class copy

		dest.setHeadHeight(source.getHeadHeight());
		dest.setHeadWidth(source.getHeadWidth());
        dest.setLineWidth(source.getLineWidth());

  	},


	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING3D.Element.registerProperties(element, controller);
		// super class

  	 /*** 
	  * Ratio arrow_length/head_length 
	  * @property HeadHeight 
	  * @type double
	  * @default "8"
	  */ 
      	controller.registerProperty("HeadHeight", element.setHeadHeight);
  	 /*** 
	  * Ratio arrow_length/head_width 
	  * @property HeadWidth 
	  * @type double
	  * @default "20"
	  */ 
      	controller.registerProperty("HeadWidth", element.setHeadWidth);
     
      	controller.registerProperty("LineWidth",  element.setLineWidth, element.getLineWidth);
      	
	}

};

/**
 * Creates a 3D Arrow
 * @method arrow
 */
EJSS_DRAWING3D.arrow = function (name) {
  var self = EJSS_DRAWING3D.element(name);

  // Implementation variables
  var mX = 0;
  var mY = 0;
  var mZ = 0;
  var mSizeX = 1;				
  var mSizeY = 1;					
  var mSizeZ = 1;		
  var mHeadHeight = 8;  // ratio arrow length / head long
  var mHeadWidth = 20;  // ratio arrow length / head width
  var mLineWidth = 0.01;

  self.getClass = function() {
  	return "ElementArrow";
  } 

  self.setHeadHeight = function(h) {
    if (mHeadHeight!=h) { 
      mHeadHeight = h; 
      self.setMeshChanged(true); 
    }   	
  }

  self.getHeadHeight = function() { 
    return mHeadHeight; 
  };

  self.setHeadWidth = function(w) {
    if (mHeadWidth!=w) { 
      mHeadWidth = w; 
      self.setMeshChanged(true); 
    }   	
  }

  self.getHeadWidth = function() { 
    return mHeadWidth; 
  };

  self.setX = function(x) { 
    if (mX!=x) { 
      mX = x; 
      self.setMeshChanged(true); 
    } 
  };

  self.getX = function() { 
    return mX; 
  };

  self.setY = function(y) {  
    if (mY!=y) { 
      mY = y; 
      self.setMeshChanged(true);
    } 
  };

  self.getY = function() { 
    return mY; 
  };

  self.setZ = function(z) {  
    if (mZ!=z) { 
      mZ = z; 
      self.setMeshChanged(true);
    } 
  };

  self.getZ = function() { 
    return mZ; 
  };

  self.setPosition = function(position) {
    self.setX(position[0]);
    self.setY(position[1]);
    self.setZ(position[2]);
  };

  self.getPosition = function() { 
    return [mX, mY, mZ]; 
  };

  self.setSizeX = function(sizeX) { 
    if (mSizeX!=sizeX) { 
      mSizeX = sizeX; 
      self.setMeshChanged(true);
    } 
  };

  self.setSizeY = function(sizeY) { 
    if (mSizeY!=sizeY) { 
      mSizeY = sizeY; 
      self.setMeshChanged(true);
    }
  };

  self.setSizeZ = function(sizeZ) { 
    if (mSizeZ!=sizeZ) { 
      mSizeZ = sizeZ; 
      self.setMeshChanged(true);
    }
  };

  self.getSizeX = function() { 
    return mSizeX; 
  };

  self.getSizeY = function() { 
    return mSizeY; 
  };

  self.getSizeZ = function() { 
    return mSizeZ; 
  };

  self.setSize = function(size) {
    self.setSizeX(size[0]);
    self.setSizeY(size[1]);
    self.setSizeZ(size[2]);
  };

  self.getSize = function() {
    return [self.getSizeX(), self.getSizeY(), self.getSizeZ()];
  };

  /**
   * Set the line width
   */
  self.setLineWidth = function(width) { 
    if (width!=mLineWidth) {
      mLineWidth = width; 
  	  self.setMeshChanged(true);
    }
  };

  /**
   * Get the line width
   */
  self.getLineWidth = function() { 
  	return mLineWidth; 
  };

  self.registerProperties = function(controller) {
	EJSS_DRAWING3D.Arrow.registerProperties(self, controller);
  };

  self.copyTo = function(element) {
	EJSS_DRAWING3D.Arrow.copyTo(self,element);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.getStyle().setDrawFill(true);
  self.getStyle().setDrawLines(false);
  self.setResolution([10,2]);
  // self.setPrivateTransformation([[54.735,-0.5,0.5,0,0,0,0]]); // 54.735 angle (arcsin(1/sqrt(3)))

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//ArrowSet
//---------------------------------

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

/***
 * ArrowSet is a set of Arrows
 * @class EJSS_DRAWING3D.ArrowSet 
 * @constructor  
 */
EJSS_DRAWING3D.ArrowSet = {

    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING3D.ElementSet;
      ElementSet.registerProperties(set,controller); 
     
  	 /*** 
	  * Ratio arrow_length/head_length 
	  * @property HeadHeight 
	  * @type double
	  * @default "8"
	  */ 
	  controller.registerProperty("HeadHeight", 
          function(v) { set.setToEach(function(element,value) { element.setHeadHeight(value); }, v); }
      );
      
      /*** 
	  * Ratio arrow_length/head_width 
	  * @property HeadWidth 
	  * @type double
	  * @default "20"
	  */ 
      controller.registerProperty("HeadWidth", 
          function(v) { set.setToEach(function(element,value) { element.setHeadWidth(value); }, v); }
      );
        
      controller.registerProperty("LineWidth", 
          function(v) { set.setToEach(function(element,value) { element.setLineWidth(value); }, v); }
      );      
        
    }

};


/**
 * Creates a set of arrows
 * @method arrowSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING3D.arrowSet = function (mName) {
  var self = EJSS_DRAWING3D.elementSet(EJSS_DRAWING3D.arrow,mName);

  // Static references
  var ArrowSet = EJSS_DRAWING3D.ArrowSet;		// reference for ArrowSet
  
  self.registerProperties = function(controller) {
    ArrowSet.registerProperties(self,controller);
  };

  return self;
};
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/**
 * Basic
 * @class Basic 
 * @constructor  
 */
EJSS_DRAWING3D.Basic = {
    object_files:{},
    
    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING3D.Element.copyTo(source,dest); // super class copy
  	
		dest.setVertices(source.getVertices());
		dest.setTriangles(source.getTriangles());
		dest.setNormals(source.getNormals());
		dest.setColors(source.getColors());
  	},


	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING3D.Element.registerProperties(element, controller);
		// super class

		controller.registerProperty("Description", element.setDescription);
		controller.registerProperty("Vertices", element.setVertices);
		controller.registerProperty("Triangles", element.setTriangles);		
		controller.registerProperty("Normals", element.setNormals);
		controller.registerProperty("Colors", element.setColors);
	}

};

/**
 * Creates a 3D Basic
 * @method basic
 */
EJSS_DRAWING3D.basic = function (name) {
  var self = EJSS_DRAWING3D.element(name);

  // Implementation variables
  var mDataChanged = true;

  var mVertices = [];
  var mTriangles = [];
  var mNormals = [];
  var mColors = [];

  self.getClass = function() {
  	return "ElementBasic";
  }

  /**
   * Sets all parts of an object
   */
  self.setDescription = function(description) {
    var panel = self.getPanel();
    if (!panel) return;
    if (!panel.supportsWebGL()) return; // Or an error will block the rest of the interface
//      console.log("Setting description to "+description);
    if (typeof description === 'string') { // read description from file. The file must have an object called "Object3D_" + the file name (without extensions)
      var descName = "Object3D_"+EJSS_TOOLS.File.plainName(description);
//      console.log(description+" variable = "+descName);
      self.setDescription(window[descName]);
      //EJSS_TOOLS.File.loadJSfile(description, function() { console.log ("Setting description object to "+descName); self.setDescription(window[descName]); });
      return;
    }
//    console.log("Do setting description to "+description);
  
    if (description["vertices"])  self.setVertices(description["vertices"]);
    if (description["triangles"]) self.setTriangles(description["triangles"]);
    if (description["normals"])   self.setNormals(description["normals"]);
    if (description["colors"])    self.setColors(description["colors"]);
  }

  self.setVertices = function(vertices) {
    if (!EJSS_TOOLS.compareArrays(mVertices,vertices)) {
        mVertices = vertices.slice(); 
  	    self.setMeshChanged(true);      
    }
  }

  self.getVertices = function() {
  	return mVertices;
  }

  self.setTriangles = function(triangles) {
    if (!EJSS_TOOLS.compareArrays(mTriangles,triangles)) {
        mTriangles = triangles.slice(); 
  	    self.setMeshChanged(true);      
    }
  }

  self.getTriangles = function() {
  	return mTriangles;
  }

  self.setNormals = function(normals) {
    if (!EJSS_TOOLS.compareArrays(mNormals,normals)) {
        mNormals = normals.slice(); 
  	    self.setMeshChanged(true);      
    }
  }

  self.getNormals = function() {
  	return mNormals;
  }

  self.setColors = function(colors) {
    if (!EJSS_TOOLS.compareArrays(mColors,colors)) {
        mColors = colors.slice(); 
  	    self.setMeshChanged(true);      
    }
  }

  self.getColors = function() {
  	return mColors;
  }

  self.registerProperties = function(controller) {
	EJSS_DRAWING3D.Basic.registerProperties(self, controller);
  };

  self.copyTo = function(element) {
	EJSS_DRAWING3D.Basic.copyTo(self,element);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([1,1,1]);

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//BasicSet
//---------------------------------

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

/**
 * BasicSet
 * @class BasicSet 
 * @constructor  
 */
EJSS_DRAWING3D.BasicSet = {

    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING3D.ElementSet;
      
      ElementSet.registerProperties(set,controller);
      controller.registerProperty("Description", 
          function(v) { set.setToEach(function(element,value) { element.setDescription(value); }, v); }
      );
      controller.registerProperty("Vertices", 
          function(v) { set.setToEach(function(element,value) { element.setVertices(value); }, v); }
      );
      controller.registerProperty("Triangles", 
          function(v) { set.setToEach(function(element,value) { element.setTriangles(value); }, v); }
      );           
      controller.registerProperty("Normals", 
          function(v) { set.setToEach(function(element,value) { element.setNormals(value); }, v); }
      );           
      controller.registerProperty("Colors", 
          function(v) { set.setToEach(function(element,value) { element.setColors(value); }, v); }
      );           
    }        

};


/**
 * Creates a set of basics
 * @method basicSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING3D.basicSet = function (mName) {
  var self = EJSS_DRAWING3D.elementSet(EJSS_DRAWING3D.basic,mName);

  // Static references
  var BasicSet = EJSS_DRAWING3D.BasicSet;		// reference for BasicSet
  
  self.registerProperties = function(controller) {
    BasicSet.registerProperties(self,controller);
  };

  return self;
};
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/**
 * Box
 * @class Box 
 * @constructor  
 */
EJSS_DRAWING3D.Box = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING3D.Element.copyTo(source,dest); // super class copy
  	
		dest.setReduceZby(source.getReduceZby());
  	
  	},

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING3D.Element.registerProperties(element, controller);
		// super class
		
      	controller.registerProperty("ReduceZby", element.setReduceZby);

	}
};

/**
 * Creates a 3D Box
 * @method box
 */
EJSS_DRAWING3D.box = function (name) {
  var self = EJSS_DRAWING3D.element(name);
  var mReduceZby = 0.0;

  // Implementation variables
  self.getClass = function() {
  	return "ElementBox";
  }

  self.setReduceZby = function(reduce) {
  	if(mReduceZby != reduce) {
  	  mReduceZby = reduce;
  	  self.setMeshChanged(true);
  	}
  }

  self.getReduceZby = function() {
  	return mReduceZby;
  }

  self.registerProperties = function(controller) {
	EJSS_DRAWING3D.Box.registerProperties(self, controller);
  };

  self.copyTo = function(element) {
	EJSS_DRAWING3D.Box.copyTo(self,element);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([1,1,1]);

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//BoxSet
//---------------------------------

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

/**
 * BoxSet
 * @class BoxSet 
 * @constructor  
 */
EJSS_DRAWING3D.BoxSet = {

    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING3D.ElementSet;
      ElementSet.registerProperties(set,controller);
      controller.registerProperty("ReduceZby", 
          function(v) { set.setToEach(function(element,value) { element.setReduceZby(value); }, v); }
      );
    }

};


/**
 * Creates a set of boxs
 * @method boxSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING3D.boxSet = function (mName) {
  var self = EJSS_DRAWING3D.elementSet(EJSS_DRAWING3D.box,mName);

  // Static references
  var BoxSet = EJSS_DRAWING3D.BoxSet;		// reference for BoxSet
  
  self.registerProperties = function(controller) {
    BoxSet.registerProperties(self,controller);
  };

  return self;
};
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/**
 * Cone
 * @class Cone 
 * @constructor  
 */
EJSS_DRAWING3D.Cone = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING3D.Element.copyTo(source,dest); // super class copy
  		EJSS_DRAWING3D.Cylinder.copyTo(source,dest);  	
  	},

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING3D.Cylinder.registerProperties(element, controller);
		// super class
		
	}
};

/**
 * Creates a 3D Cone
 * @method cone
 */
EJSS_DRAWING3D.cone = function (name) {
  var self = EJSS_DRAWING3D.cylinder(name);

  // Implementation variables
  self.getClass = function() {
  	return "ElementCone";
  }

  self.registerProperties = function(controller) {
	EJSS_DRAWING3D.Cone.registerProperties(self, controller);
  };

  self.copyTo = function(element) {
	EJSS_DRAWING3D.Cone.copyTo(self,element);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([1,1,1]);
  self.setTopRadius(0);

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//ConeSet
//---------------------------------

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

/**
 * ConeSet
 * @class ConeSet 
 * @constructor  
 */
EJSS_DRAWING3D.ConeSet = {

    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING3D.ElementSet;
      ElementSet.registerProperties(set,controller); 
      
    }

};


/**
 * Creates a set of cones
 * @method coneSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING3D.coneSet = function (mName) {
  var self = EJSS_DRAWING3D.elementSet(EJSS_DRAWING3D.cone,mName);

  // Static references
  var ConeSet = EJSS_DRAWING3D.ConeSet;		// reference for ConeSet
  
  self.registerProperties = function(controller) {
    ConeSet.registerProperties(self,controller);
  };

  return self;
};
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/***
 * Cylinder
 * @class EJSS_DRAWING3D.Cylinder
 * @constructor
 */
EJSS_DRAWING3D.Cylinder = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING3D.Element.copyTo(source,dest); // super class copy
  	
		dest.setBottomRadius(source.getBottomRadius());  	
		dest.setTopRadius(source.getTopRadius());  	

  	},

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING3D.Element.registerProperties(element, controller);
		// super class

	 /*** 
	  * Bottom radius
	  * @property BottomRadius 
	  * @type number
	  * @default 1
	  */ 
		controller.registerProperty("BottomRadius", element.setBottomRadius);
	 /*** 
	  * Top radius
	  * @property TopRadius 
	  * @type number
	  * @default 1
	  */ 	
		controller.registerProperty("TopRadius", element.setTopRadius);
		
	 /*** 
	  * Minimum angle in drawing
	  * @property MinAngleU 
	  * @type number
	  * @default 0
	  */ 		
        controller.registerProperty("MinAngleU", element.setMinAngleU);
	 /*** 
	  * Maximum angle in drawing
	  * @property MaxAngleU 
	  * @type number
	  * @default 360
	  */ 		
        controller.registerProperty("MaxAngleU", element.setMaxAngleU);		
	}
};

/**
 * Creates a 3D Cylinder
 * @method cylinder
 */
EJSS_DRAWING3D.cylinder = function (name) {
  var self = EJSS_DRAWING3D.element(name);

  // Implementation variables
  var mMinAngleU = 0; 		// the start angle (in degrees) for the parallels
  var mMaxAngleU = 360;		// the end angle (in degrees) for the parallels

  var mTopRadius = 1;
  var mBottomRadius = 1;

  self.getClass = function() {
  	return "ElementCylinder";
  }
  
	 /*** 
	  * Changes the position and Size Z and adds an extra transformation
	  * so that the axis of the element has the prescribed origin and end points   
	  * @method setOriginAndEnd 
	  * @param origin a double[3] array with the origin of the axis
	  * @param end a double[3] array with the end point of the axis
	  */ 
  self.setOriginAndEnd = function(origin,end) {
  	var xc = (origin[0]+end[0])/2;
    var yc = (origin[1]+end[1])/2;
    var zc = (origin[2]+end[2])/2;
    var dx = end[0]-origin[0], dy = end[1]-origin[1], dz = end[2]-origin[2];
    var distance = Math.sqrt(dx*dx+dy*dy+dz*dz);
    var angle = Math.acos(dz/distance);
    self.setPosition([xc,yc,zc]);
    self.setSizeZ(distance);
    self.setFirstTransformation([[angle,-dy, dx, 0, xc,yc,zc]]);
  }

  self.setMinAngleU = function(angle) {
  	if(mMinAngleU != angle) {
  	  mMinAngleU = angle;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.getMinAngleU = function() {
  	return mMinAngleU;
  }

  self.setMaxAngleU = function(angle) {
  	if(mMaxAngleU != angle) {
  	  mMaxAngleU = angle;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.getMaxAngleU = function() {
  	return mMaxAngleU;
  }

  self.setBottomRadius = function(radius) {
  	if(mBottomRadius != radius) {
  	  mBottomRadius = radius;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.getBottomRadius = function() {
  	return mBottomRadius;
  }

  self.setTopRadius = function(radius) {
  	if(mTopRadius != radius) {
  	  mTopRadius = radius;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.getTopRadius = function() {
  	return mTopRadius;
  }

  self.registerProperties = function(controller) {
	EJSS_DRAWING3D.Cylinder.registerProperties(self, controller);
  };

  self.copyTo = function(element) {
	EJSS_DRAWING3D.Cylinder.copyTo(self,element);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([1,1,1]);

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//CylinderSet
//---------------------------------

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

/**
 * CylinderSet
 * @class CylinderSet 
 * @constructor  
 */
EJSS_DRAWING3D.CylinderSet = {

    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING3D.ElementSet;
      ElementSet.registerProperties(set,controller); 

      controller.registerProperty("MinAngleU", 
          function(v) { set.setToEach(function(element,value) { element.setMinAngleU(value); }, v); }
      );      
      controller.registerProperty("MaxAngleU", 
          function(v) { set.setToEach(function(element,value) { element.setMaxAngleU(value); }, v); }
      );      
      controller.registerProperty("BottomRadius", 
          function(v) { set.setToEach(function(element,value) { element.setBottomRadius(value); }, v); }
      );      
      controller.registerProperty("TopRadius", 
          function(v) { set.setToEach(function(element,value) { element.setTopRadius(value); }, v); }
      );      
    }

};


/**
 * Creates a set of cylinders
 * @method cylinderSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING3D.cylinderSet = function (mName) {
  var self = EJSS_DRAWING3D.elementSet(EJSS_DRAWING3D.cylinder,mName);

  // Static references
  var CylinderSet = EJSS_DRAWING3D.CylinderSet;		// reference for CylinderSet
  
  self.registerProperties = function(controller) {
    CylinderSet.registerProperties(self,controller);
  };

  return self;
};
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};


/***
 * Disk
 * @class EJSS_DRAWING3D.Disk
 * @constructor
 */
EJSS_DRAWING3D.Disk = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING3D.Element.copyTo(source,dest); // super class copy
  		EJSS_DRAWING3D.Cylinder.copyTo(source,dest);
  	
  	},

	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING3D.Cylinder.registerProperties(element, controller);
		// super class
		
	      controller.registerProperty("Size", function(s) {
	      		s[2] = Math.min(s[0],s[1])/1000;
	      		element.setSize(s);
	      	});
	      controller.registerProperty("SizeX", function(sx) {
	      		var sy = element.getSizeY();
	      		element.setSizeZ(Math.min(sy,sx)/1000)
	      		element.setSizeX(sx);
	      	});
	      controller.registerProperty("SizeY",function(sy) {
	      		var sx = element.getSizeX();
	      		element.setSizeZ(Math.min(sy,sx)/1000)
	      		element.setSizeY(sy);
	      	});		
	}
};

/**
 * Creates a 3D Disk
 * @method disk
 */
EJSS_DRAWING3D.disk = function (name) {
  var self = EJSS_DRAWING3D.cylinder(name);

  // Implementation variables
  self.getClass = function() {
  	return "ElementDisk";
  }

  self.registerProperties = function(controller) {
	EJSS_DRAWING3D.Disk.registerProperties(self, controller);
  };

  self.copyTo = function(element) {
	EJSS_DRAWING3D.Disk.copyTo(self,element);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([1,1,0.001]);

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/***
 * A DrawingPanel is a 3D drawing panel. 
 * @class EJSS_DRAWING3D.DrawingPanel
 * @constructor
 */
 EJSS_DRAWING3D.DrawingPanel = {  
  GRAPHICS3D_WEBGL : "WebGL",

  /**
   * static registerProperties method
   */
  registerProperties : function(element, controller) {
    // No super class
    
    element.setController(controller);
    
	 /*** 
	  * The minimum value for the X coordinates of elements. 
	  * @property MinimumX 
	  * @type int or double
	  * @default "-1"
	  */ 
    controller.registerProperty("MinimumX",element.setWorldXMin,element.getWorldXMin);
	 /*** 
	  * The maximum value for the X coordinates of elements. 
	  * @property MaximumX 
	  * @type int or double
	  * @default "+1"
	  */ 
    controller.registerProperty("MaximumX",element.setWorldXMax,element.getWorldXMax);
	 /*** 
	  * The minimum value for the Y coordinates of elements. 
	  * @property MinimumY 
	  * @type int or double
	  * @default "-1"
	  */ 
    controller.registerProperty("MinimumY",element.setWorldYMin,element.getWorldYMin);
	 /*** 
	  * The maximum value for the Y coordinates of elements. 
	  * @property MaximumY
	  * @type int or double
	  * @default "+1"
	  */ 
    controller.registerProperty("MaximumY",element.setWorldYMax,element.getWorldYMax);
	 /*** 
	  * The minimum value for the Z coordinates of elements. 
	  * @property MinimumZ 
	  * @type int or double
	  * @default "-1"
	  */ 
    controller.registerProperty("MinimumZ",element.setWorldZMin,element.getWorldZMin);
	 /*** 
	  * The maximum value for the Z coordinates of elements. 
	  * @property MaximumZ 
	  * @type int or double
	  * @default "+1"
	  */ 
    controller.registerProperty("MaximumZ",element.setWorldZMax,element.getWorldZMax);
    controller.registerProperty("Bounds",element.setWorldCoordinates,element.getWorldCoordinates);
    
    controller.registerProperty("Size",element.setSize,element.getSize);
	 /*** 
	  * Size for the X axis.
	  * @property SizeX
	  * @type int or double 
	  * @default 1
	  */ 
    controller.registerProperty("SizeX",element.setSizeX,element.getSizeX);
	 /*** 
	  * Size for the Y axis.
	  * @property SizeY
	  * @type int or double 
	  * @default 1
	  */ 
    controller.registerProperty("SizeY",element.setSizeY,element.getSizeY);
	 /*** 
	  * Size for the Z axis.
	  * @property SizeZ
	  * @type int or double 
	  * @default 1
	  */ 
    controller.registerProperty("SizeZ",element.setSizeZ,element.getSizeZ);

	 /*** 
	  * Camera rotation in Z axis.
	  * @property CameraAzimuth
	  * @type int or double (degrees) 
	  * @default 0
	  */ 
    controller.registerProperty("CameraAzimuth",element.setCamAzimuth,element.getCamAzimuth);
	 /*** 
	  * Camera rotation in Y axis.
	  * @property CameraAltitude
	  * @type int or double (degrees) 
	  * @default 0
	  */ 
    controller.registerProperty("CameraAltitude",element.setCamAltitude,element.getCamAltitude);
	 /*** 
	  * Camera rotation in X axis.
	  * @property CameraTilt
	  * @type int or double (degrees) 
	  * @default 0
	  */ 
    controller.registerProperty("CameraTilt",element.setCamTilt,element.getCamTilt);
	 /*** 
	  * Camera position in X axis.
	  * @property CameraX
	  * @type int or double 
	  * @default 4
	  */ 
    controller.registerProperty("CameraX",element.setCamLocX,element.getCamLocX);
	 /*** 
	  * Camera position in Y axis.
	  * @property CameraY
	  * @type int or double 
	  * @default 0
	  */ 
    controller.registerProperty("CameraY",element.setCamLocY,element.getCamLocY);
	 /*** 
	  * Camera position in Z axis.
	  * @property CameraZ
	  * @type int or double 
	  * @default 0
	  */ 
    controller.registerProperty("CameraZ",element.setCamLocZ,element.getCamLocZ);
    controller.registerProperty("CameraLocation",element.setCamLoc,element.getCamLoc);
	 /*** 
	  * Camera focus in X axis.
	  * @property CameraFocusX
	  * @type int or double 
	  * @default 0
	  */ 
    controller.registerProperty("CameraFocusX",element.setCamFocusX,element.getCamFocusX);
	 /*** 
	  * Camera focus in Y axis.
	  * @property CameraFocusY
	  * @type int or double 
	  * @default 0
	  */ 
    controller.registerProperty("CameraFocusY",element.setCamFocusY,element.getCamFocusY);
	 /*** 
	  * Camera focus in Z axis.
	  * @property CameraFocusZ
	  * @type int or double 
	  * @default 0
	  */ 
    controller.registerProperty("CameraFocusZ",element.setCamFocusZ,element.getCamFocusZ);
    controller.registerProperty("CameraFocus",element.setCamFocus,element.getCamFocus);
	 /*** 
	  * Camera up vector in X axis.
	  * @property CameraUpVectorX
	  * @type int or double 
	  * @default 0
	  */ 
    controller.registerProperty("CameraUpVectorX",element.setCamUpVectorX,element.getCamUpVectorX);
	 /*** 
	  * Camera up vector in Y axis.
	  * @property CameraUpVectorY
	  * @type int or double 
	  * @default 0
	  */ 
    controller.registerProperty("CameraUpVectorY",element.setCamUpVectorY,element.getCamUpVectorY);
	 /*** 
	  * Camera up vector in Z axis.
	  * @property CameraUpVectorZ
	  * @type int or double 
	  * @default 0
	  */ 
    controller.registerProperty("CameraUpVectorZ",element.setCamUpVectorZ,element.getCamUpVectorZ);
    controller.registerProperty("CameraUpVector",element.setCamUpVector,element.getCamUpVector);
	 /*** 
	  * Camera zoom rate.
	  * @property CameraZoomRate
	  * @type int or double 
	  * @default 1.10
	  */ 
    controller.registerProperty("CameraZoomRate",element.setZoomRate,element.getZoomRate);
	 /*** 
	  * Near plane in 3D projection.
	  * @property CameraNear 
	  * @type int
	  * @default 1
	  */ 
    controller.registerProperty("CameraNear",element.setNear,element.getNear);
	 /*** 
	  * Far plane in 3D projection.
	  * @property CameraFar 
	  * @type int
	  * @default 10000
	  */ 
    controller.registerProperty("CameraFar",element.setFar,element.getFar);
	 /*** 
	  *  Type of projection.  
	  * @property Projection 
	  * @type int or string 
	  * @values "PLANAR_XY": 0; "PLANAR_XZ": 1; "PLANAR_YZ": 2; "PERSPECTIVE_OFF": 3; "PERSPECTIVE_ON": 4;
	  * @default "PERSPECTIVE_ON"
	  */ 
	controller.registerProperty("Projection",element.setProjection, element.getProjection);
	 /*** 
	  * Field of view when perspective on. A parameter that gives more or less perspective to the projection.
	  * @property Perspective 
	  * @type int (degrees)
	  * @default "45"
	  */ 
	controller.registerProperty("Perspective",element.setFOV, element.getFOV);
    controller.registerProperty("FieldOfView",element.setFOV,element.getFOV);

	 /*** 
	  * Lights in scene.
	  * @property Lights 
	  * @type array of array 
	  * @default [[1,2,3]]
	  */ 
    controller.registerProperty("Lights",element.setLights,element.getLights);
    
    controller.registerProperty("DecorationType",element.setDecorationType);
    controller.registerProperty("VisibleDecorationBox",element.setVisibleDecorationBox);
    controller.registerProperty("VisibleDecorationAxis",element.setVisibleDecorationAxis);
    controller.registerProperty("VisibleDecorationBasicAxis",element.setVisibleDecorationBasicAxis);

    controller.registerProperty("Parent",element.getGraphics().setParent,element.getGraphics().getParent);
    controller.registerProperty("ClassName", element.getGraphics().setClassName);
    controller.registerProperty("Width",element.getGraphics().setWidth, element.getGraphics().getWidth);
    controller.registerProperty("Height",element.getGraphics().setHeight,element.getGraphics().getHeight);
    
    controller.registerProperty("ImageUrl",element.setImageUrl);

    controller.registerProperty("Graphics",element.setGraphics);
    controller.registerProperty("Enabled",element.setEnabled);    
    controller.registerProperty("Draggable",element.setDraggable);    

	controller.registerProperty("Background", element.getStyle().setFillColor);
    controller.registerProperty("Foreground",  element.getGraphics().getStyle().setBorderColor);
    controller.registerProperty("LineColor",  element.getGraphics().getStyle().setBorderColor);
    controller.registerProperty("LineWidth",  element.getGraphics().getStyle().setBorderWidth);
    controller.registerProperty("DrawLines",  element.getStyle().setDrawLines);
    controller.registerProperty("FillColor",  element.getStyle().setFillColor);
    controller.registerProperty("DrawFill",   element.getStyle().setDrawFill);

    controller.registerProperty("Visibility", element.getGraphics().getStyle().setVisibility);      
    controller.registerProperty("Display", element.getGraphics().getStyle().setDisplay); 
	controller.registerProperty("CSS", element.getGraphics().getStyle().setCSS);

    controller.registerAction("OnDoubleClick", element.getInteraction().getInteractionPoint);      
    controller.registerAction("OnMove", element.getInteraction().getInteractionPoint);      
    controller.registerAction("OnPress", element.getInteraction().getInteractionPoint);    
    controller.registerAction("OnDrag", element.getInteraction().getInteractionPoint, element.getOnDragHandler);    
	controller.registerAction("OnRelease", element.getInteraction().getInteractionBounds);
    controller.registerAction("OnZoom", element.getInteraction().getZoomDelta, element.getOnZoomHandler);    
        
  }

};

/**
 * Constructor for DrawingPanel
 * @method drawingPanel
 * @param mName string
 * @returns An abstract 3D drawing panel
 */
EJSS_DRAWING3D.drawingPanel = function (mName) {
  var self = {};										// reference returned 
  var mGraphics = EJSS_GRAPHICS.webGLGraphics(mName);	// graphics implementation (default: WebGL) 

  // Instance variables
  var mStyle = EJSS_DRAWING3D.style(mName);			// style for panel 
  var mDecorations = [];							// decorations list for panel
  var mElements = [];								// elements list for panel
  var mElementsChanged = true;						// whether elements list has changed 
  var mDraggable = 1;								// how panel responds to interaction
  var mImageBackground = false;						// whether background image 
  var mCollectersList = [];		            // Array of all control elements that need a call to dataCollected() after data collection
   
  // Configuration variables 
  var mWorld = {
  	  // preferred dimensions
      xminPreferred: -1, xmaxPreferred: +1, yminPreferred: -1, ymaxPreferred: +1, zminPreferred: -1, zmaxPreferred: +1,  
  };
  
  // Camera
  var mChangeCamera = true;
  var mAutoCamera = true;
  var mZoomRate = 1.10;
  var mCamera = {
  	location: new Vector(4, 0, 0),
  	focus: new Vector(0, 0, 0),
  	upvector: new Vector(0, 0, 1),
  	azimuth: 0,
  	altitude: 0,
  	tilt: 0,
  	delta: 0
  }    
  var mFOV = 45;
  var mNear = 1;
  var mFar = 10000;
  var mOrthographic = false;
  var mProjection = 4;
  
  // Size  
  var mSizeX = 1;					// size X
  var mSizeY = 1;					// size Y
  var mSizeZ = 1;					// size Z
  
  // Lights
  var mLights = [[1,2,3]];
  
  // Implementation variables
  var mPanelChanged = true;			// whether panel changed (style, decorations)
  var mChangeWorld = true;			// whether decorations must change
  var mInteraction;					// user interaction on panel
  
  var mController = { // dummy controller object
          propertiesChanged : function() {},
          invokeAction : function() {}
  };
  
  // ----------------------------------------
  // Instance functions
  // ----------------------------------------

  /***
   * Get name for drawing panel
   * @method getName
   * @return string
   */
  self.getName = function() {
    return mName;
  };

  /***
   * Returns the graphics implementation
   * @method getGraphics
   * @return Graphics
   */
  self.getGraphics = function() {
    return mGraphics;
  };
 
  /***
   * Whether webgl is supported
   * @method supportsWebGL
   * @return boolean
   */
  self.supportsWebGL = function() {
    return mGraphics.supportsWebGL();
  };
 
  /***
   * Return the drawing style of the inner rectangle for panel
   * @method getStyle
   * @return Style
   */
  self.getStyle = function() { 
    return mStyle; 
  };

  /***
   * Set graphics (only Webgl supported)
   * @method setGraphics
   * @param type
   */
  self.setGraphics = function(type) {
  	if(type == EJSS_DRAWING3D.DrawingPanel.GRAPHICS3D_WEBGL) {
		mGraphics = EJSS_GRAPHICS.webGLGraphics(mName); 
	} else {
		console.log("WARNING: setGraphics() - Graphics not supported");
	}	
  }

  // ----------------------------------------
  // World coordinates
  // ----------------------------------------

  /***
   * Sets the preferred minimum X coordinate for the panel
   * @method setWorldXMin
   * @param xmin
   */
  self.setWorldXMin = function(xmin)  {
    if (xmin !== mWorld.xminPreferred) { 
      mWorld.xminPreferred = xmin;
      mChangeWorld = true; 
    }
  };

  /***
   * Returns the preferred minimum X coordinate for the panel
   * @method getWorldXMin
   * @return double
   */
  self.getWorldXMin = function() {
    return mWorld.xminPreferred;
  };
  
  /***
   * Sets the preferred maximum X coordinate for the panel
   * @method setWorldXMax
   * @param xmax
   */
  self.setWorldXMax = function(xmax)  {
    if (xmax !== mWorld.xmaxPreferred) { 
      mWorld.xmaxPreferred = xmax;
      mChangeWorld = true; 
    }
  };

  /***
   * Returns the preferred maximum X coordinate for the panel
   * @method getWorldXMax
   * @return double
   */
  self.getWorldXMax = function() {
    return mWorld.xmaxPreferred;
  };

  /***
   * Sets the preferred minimum Y coordinate for the panel
   * @method setWorldYMin
   * @param ymin
   */
  self.setWorldYMin = function(ymin)  {
    if (ymin !== mWorld.yminPreferred) { 
      mWorld.yminPreferred = ymin;
      mChangeWorld = true; 
    }
  };
  
  /***
   * Returns the preferred minimum Y coordinate for the panel
   * @method getWorldYMin
   * @return double
   */
  self.getWorldYMin = function() {
    return mWorld.yminPreferred;
  };

  /***
   * Sets the preferred maximum Y coordinate for the panel
   * @method setWorldYMax
   * @param ymax
   */
  self.setWorldYMax = function(ymax)  {
    if (ymax !== mWorld.ymaxPreferred) { 
      mWorld.ymaxPreferred = ymax;
      mChangeWorld = true; 
    }
  };

  /***
   * Returns the preferred maximum Y coordinate for the panel
   * @method getWorldYMax
   * @return double
   */
  self.getWorldYMax = function() {
    return mWorld.ymaxPreferred;
  };

  /***
   * Sets the preferred minimum Y coordinate for the panel
   * @method setWorldZMin
   * @param zmin
   */
  self.setWorldZMin = function(zmin)  {
    if (zmin !== mWorld.zminPreferred) { 
      mWorld.zminPreferred = zmin;
      mChangeWorld = true; 
    }
  };
  
  /***
   * Returns the preferred minimum Z coordinate for the panel
   * @method getWorldZMin
   * @return double
   */
  self.getWorldZMin = function() {
    return mWorld.zminPreferred;
  };

  /***
   * Sets the preferred maximum Z coordinate for the panel
   * @method setWorldZMax
   * @param zmax
   */
  self.setWorldZMax = function(zmax)  {
    if (zmax !== mWorld.zmaxPreferred) { 
      mWorld.zmaxPreferred = zmax;
      mChangeWorld = true; 
    }
  };

  /***
   * Returns the preferred maximum Z coordinate for the panel
   * @method getWorldZMax
   * @return double
   */
  self.getWorldZMax = function() {
    return mWorld.zmaxPreferred;
  };

  /***
   * Sets the preferred user coordinates for the panel
   * @method setWorldCoordinates
   * @param bounds
   */
  self.setWorldCoordinates = function(bounds)  {
    self.setWorldXMin(bounds[0]);
    self.setWorldXMax(bounds[1]);
    self.setWorldYMin(bounds[2]);
    self.setWorldYMax(bounds[3]);
    self.setWorldZMin(bounds[4]);
    self.setWorldZMax(bounds[5]);
  };

  /***
   * Gets the preferred user coordinates for the panel
   * @method getWorldCoordinates
   * @return bounds
   */
  self.getWorldCoordinates = function()  {
  	return [self.getWorldXMin(),self.getWorldXMax(),
  			self.getWorldYMin(),self.getWorldYMax(),
  			self.getWorldZMin(),self.getWorldZMax()];
  };
  
  /***
   * Gets X coordinate for camera
   * @method getCamLocX
   * @return x
   */
  self.getCamLocX = function() {
  	return mCamera.location.getx();
  }

  /***
   * Gets Y coordinate for camera
   * @method getCamLocY
   * @return y
   */
  self.getCamLocY = function() {
  	return mCamera.location.gety();
  }

  /***
   * Gets Z coordinate for camera
   * @method getCamLocZ
   * @return z
   */
  self.getCamLocZ = function() {
  	return mCamera.location.getz();
  }

  /***
   * Gets coordinates for camera
   * @method getCamLoc
    * @return vector
  */
  self.getCamLoc = function() {
  	return mCamera.location.toArray();
  }

  /***
   * Sets X coordinate for camera
   * @method setCamLocX
   * @param x
   */
  self.setCamLocX = function(x) {
  	if(mCamera.location.getx() != x) {
  	  mCamera.location.setx(x);
      mChangeCamera = true;   		
  	}
    mAutoCamera = false;
  }

  /***
   * Sets Y coordinate for camera
   * @method setCamLocY
   * @param y
   */
  self.setCamLocY = function(y) {
  	if(mCamera.location.gety() != y) {
  	  mCamera.location.sety(y);
      mChangeCamera = true;   		
  	}
    mAutoCamera = false;
  }

  /***
   * Sets Z coordinate for camera
   * @method setCamLocZ
   * @param z
   */
  self.setCamLocZ = function(z) {
  	if(mCamera.location.getz() != z) {
  	  mCamera.location.setz(z);
      mChangeCamera = true;   		
  	}
    mAutoCamera = false;
  }
  
  /***
   * Sets coordinates for camera
   * @method setCamLoc
   * @param vector [x,y,z]
   */
  self.setCamLoc = function(v) {
  	self.setCamLocX(v[0]);
  	self.setCamLocY(v[1]);
  	self.setCamLocZ(v[2]);
  }

  /***
   * Sets tilt for camera
   * @method setCamTilt
   * @param tilt (degrees)
   */
  self.setCamTilt = function(t) {
  	if(mCamera.tilt != t) {
  	  mCamera.tilt = t;
      mChangeCamera = true;   		
  	}
  }

  /***
   * Gets tilt for camera
   * @method getCamTilt
   * @return tilt
   */
  self.getCamTilt = function() {
  	return mCamera.tilt;
  }
  
  /***
   * Sets azimuth for camera
   * @method setCamAzimuth
   * @param azimuth (degrees)
   */
  self.setCamAzimuth = function(az) {
  	if(mCamera.azimuth != az) {
  	  mCamera.azimuth = az;
      mChangeCamera = true;   		
  	}
  }

  /***
   * Gets azimuth for camera
   * @method getCamAzimuth
   * @return azimuth
   */
  self.getCamAzimuth = function() {
  	return mCamera.azimuth;
  }

  /***
   * Sets altitude for camera
   * @method setCamAltitude
   * @param altitude (degrees)
   */
  self.setCamAltitude = function(al) {
  	if(mCamera.altitude != al) {
  	  mCamera.altitude = al;
      mChangeCamera = true;   		
  	}
  }

  /***
   * Gets altitude for camera
   * @method getCamAltitude
   * @return altitude
   */
  self.getCamAltitude = function() {
  	return mCamera.altitude;
  }

  /***
   * Gets X focus for camera
   * @method getCamFocusX
   * @return x
   */
  self.getCamFocusX = function() {
  	return mCamera.focus.getx();
  }

  /***
   * Gets Y focus for camera
   * @method getCamFocusY
   * @return y
   */
  self.getCamFocusY = function() {
  	return mCamera.focus.gety();
  }

  /***
   * Gets Z focus for camera
   * @method getCamFocusZ
   * @return z
   */
  self.getCamFocusZ = function() {
  	return mCamera.focus.getz();
  }

  /***
   * Gets focus for camera
   * @method getCamFocus
   * @return vector
   */
  self.getCamFocus = function() {
  	mCamera.focus.toArray();
  }

  /***
   * Sets X focus for camera
   * @method setCamFocusX
   * @param x
   */
  self.setCamFocusX = function(x) {
  	if(mCamera.focus.getx() != x) {
  	  mCamera.focus.setx(x);
      mChangeCamera = true;   		
  	}
    mAutoCamera = false;
  }

  /***
   * Sets Y focus for camera
   * @method setCamFocusY
   * @param y
   */
  self.setCamFocusY = function(y) {
  	if(mCamera.focus.gety() != y) {
  	  mCamera.focus.sety(y);
      mChangeCamera = true;   		
  	}
    mAutoCamera = false;
  }

  /***
   * Sets Z focus for camera
   * @method setCamFocusZ
   * @param z
   */
  self.setCamFocusZ = function(z) {
  	if(mCamera.focus.getz() != z) {
  	  mCamera.focus.setz(z);
      mChangeCamera = true;   		
  	}
    mAutoCamera = false;
  }
  
  /***
   * Sets focus for camera
   * @method setCamFocus
   * @param vector [x,y,z]
   */
  self.setCamFocus = function(v) {
  	self.setCamFocusX(v[0]);
  	self.setCamFocusY(v[1]);
  	self.setCamFocusZ(v[2]);
  }

  /***
   * Gets X Up Vector for camera
   * @method getCamUpVectorX
   * @return x
   */
  self.getCamUpVectorX = function() {
  	return mCamera.upvector.getx();
  }

  /***
   * Gets Y Up Vector for camera
   * @method getCamUpVectorY
   */
  self.getCamUpVectorY = function() {
  	return mCamera.upvector.gety();
  }

  /***
   * Gets Z Up Vector for camera
   * @method getCamUpVectorZ
   * @return z
   */
  self.getCamUpVectorZ = function() {
  	return mCamera.upvector.getz();
  }

  /***
   * Gets Up Vector for camera
   * @method getCamUpVector
   * @return vector [x,y,z]
   */
  self.getCamUpVector = function() {
  	mCamera.upvector.toArray();
  }

  /***
   * Sets X Up Vector for camera
   * @method setCamUpVectorX
   * @param x
   */
  self.setCamUpVectorX = function(x) {
  	if(mCamera.upvector.getx() != x) {
  	  mCamera.upvector.setx(x);
      mChangeCamera = true;   		
  	}
  }

  /***
   * Sets Y Up Vector for camera
   * @method setCamUpVectorY
   * @param y
   */
  self.setCamUpVectorY = function(y) {
  	if(mCamera.upvector.gety() != y) {
  	  mCamera.upvector.sety(y);
      mChangeCamera = true;   		
  	}
  }

  /***
   * Sets Z Up Vector for camera
   * @method setCamUpVectorZ
   * @param z
   */
  self.setCamUpVectorZ = function(z) {
  	if(mCamera.upvector.getz() != z) {
  	  mCamera.upvector.setz(z);
      mChangeCamera = true;   		
  	}
  }
  
  /***
   * Sets Up Vector for camera
   * @method setCamUpVector
   * @param vector [x,y,z]
   */
  self.setCamUpVector = function(v) {
  	self.setCamUpVectorX(v[0]);
  	self.setCamUpVectorY(v[1]);
  	self.setCamUpVectorZ(v[2]);
  }

  /***
   * Gets auto camera
   * @method getAutoCamera
   * @return boolean
   */
  self.getAutoCamera = function() {
  	return mAutoCamera;
  }

  /**
   * Sets auto camera
   * @method setAutoCamera
   * @param boolean
   */
  self.setAutoCamera = function(auto) {
  	if(mAutoCamera != auto) {
  		mAutoCamera = auto;
  		mChangeCamera = true; 
  	}
  }

  /***
   * Gets zoom rate
   * @method getZoomRate
   * @return double
   */
  self.getZoomRate = function() {
  	return mZoomRate;
  }

  /***
   * Sets zoom rate
   * @method setZoomRate
   * @param double
   */
  self.setZoomRate = function(rate) {
	mZoomRate = rate;
  }

  /***
   * Set lights
   * @method setLights
   * @param array of array
   */
  self.setLights = function(lights) {
	mLights = lights;
  }
  
  /***
   * Gets lights
   * @method getLights
   * @return array of array
   */
  self.getLights = function() {
  	return mLights;
  }
  
  // ----------------------------------------
  // Size of the element
  // ----------------------------------------

  /***
   * Set the size along the X axis of the elements
   * @method setSizeX
   * @param sizeX double
   */
  self.setSizeX = function(sizeX) {
    if (mSizeX !== sizeX) { 
      mSizeX = sizeX;
      mChangeWorld = true; 
    }  	 
  };

  /***
   * Get the size along the X coordinate of the elements
   * @method getSizeX
   * @return double
   */
  self.getSizeX = function() { 
    return mSizeX; 
  };

  /***
   * Set the size along the Y axis of the elements
   * @method setSizeY
   * @param sizeY double
   */
  self.setSizeY = function(sizeY) { 
    if (mSizeY !== sizeY) { 
      mSizeY = sizeY;
      mChangeWorld = true; 
    }  	 
  };

  /***
   * Get the size along the Y coordinate of the elements
   * @method getSizeY
   * @return double
   */
  self.getSizeY = function() { 
    return mSizeY; 
  };

  /***
   * Set the size along the Z axis of the elements
   * @method setSizeZ
   * @param sizeZ double
   */
  self.setSizeZ = function(sizeZ) { 
    if (mSizeZ !== sizeZ) { 
      mSizeZ = sizeZ;
      mChangeWorld = true; 
    }  	 
  };

  /***
   * Get the size along the Z coordinate of the elements
   * @method getSizeZ
   * @return double
   */
  self.getSizeZ = function() { 
    return mSizeZ; 
  };

  /***
   * Set the size of the elements
   * @method setSize
   * @param position double[] an array of dimension 3 or an object with {x,y,z} properties
   */
  self.setSize = function(size) {
    self.setSizeX(size[0]);
    self.setSizeY(size[1]);
    self.setSizeZ(size[2]);
  };

  /***
   * Get the sizes of the elements
   * @method getSize
   * @return double[]
   */
  self.getSize = function() {
    return [self.getSizeX(), self.getSizeY(), self.getSizeZ()];
  };
  
  /***
   * Set background image url 
   * @param url
   */
  self.setImageUrl = function(url) {
  	mImageBackground = true;
  	
  	if (self.getResourcePath != null) {
  	  url = self.getResourcePath(url);
  	}
  	else {
  		console.log ("No getResourcePath function for panel. Texture = " + url);
  	}
  	self.getGraphics().getStyle().setBackgroundImage(url);
  }
  
  /***
   * Whether background image exists
   * @return boolean
   */
  self.isImageUrl = function() {
  	return mImageBackground;
  }
  
  // ----------------------------------------
  // Decorations and elements
  // ----------------------------------------

  /***
   * Adds a decoration to the panel. Decorations are drawn before any other elements.
   * @method addDecoration
   * @param drawable decoration element
   * @param position integer
   */
  self.addDecoration = function(drawable, position) {
  	EJSS_TOOLS.addToArray(mDecorations, drawable, position);
    if (drawable.setPanel) // set this panel to decoration element 
    	drawable.setPanel(self);	 
    return self;
  };

  /***
   * Removes a decoration
   * @method removeDecoration
   * @param drawable decoration element
   */
  self.removeDecoration = function(drawable) {
    EJSS_TOOLS.removeFromArray(mDecorations,drawable);
    if (drawable.setPanel) // remove this panel to decoration element 
    	drawable.setPanel(null);
    return self;
  };

  /***
   * Set visible decoration shown basic axis
   * @method setVisibleDecorationBasicAxis
   * @param boolean
   */
  self.setVisibleDecorationBasicAxis = function(visible) {
  	mBasicAxisXDecoration.setVisible(visible);
  	mBasicAxisYDecoration.setVisible(visible);
  	mBasicAxisZDecoration.setVisible(visible);
  }

  /***
   * Set visible decoration shown traditional axis
   * @method setVisibleDecorationAxis
   * @param boolean
   */
  self.setVisibleDecorationAxis = function(visible) {
  	mAxisXDecoration.setVisible(visible);
  	mAxisYDecoration.setVisible(visible);
  	mAxisZDecoration.setVisible(visible);
  }

  /***
   * Set visible decoration shown box
   * @method setVisibleDecorationBox
   * @param boolean
   */
  self.setVisibleDecorationBox = function(visible) {
  	mBoxDecoration.setVisible(visible);  	
  }

  /***
   * Set visible a specific type of decoration
   * @method setDecorationType
   * @param type "NONE", "AXES", "CUBE", "CENTERED_AXIS"
   */
  self.setDecorationType = function(type) {
    if (typeof(type)=='string') {
      switch (type) {
        default : 
        case "NONE" : type = 0; break; 
        case "AXES" : type = 1; break; 
        case "CUBE" : type = 2; break; 
        case "CENTERED_AXES" : type = 3; break; 
      }    
    }
    switch(type) {
      case 0 : 
        self.setVisibleDecorationBox(false); 
        self.setVisibleDecorationBasicAxis(false);
        self.setVisibleDecorationAxis(false);
        break;
      case 1 : 
        self.setVisibleDecorationBox(false); 
        self.setVisibleDecorationBasicAxis(false);
        self.setVisibleDecorationAxis(true);
        break;
      default:
      case 2 : 
        self.setVisibleDecorationBox(true); 
        self.setVisibleDecorationBasicAxis(false);
        self.setVisibleDecorationAxis(false);
        break;
      case 3 : 
        self.setVisibleDecorationBox(false); 
        self.setVisibleDecorationBasicAxis(true);
        self.setVisibleDecorationAxis(false);
        break;
     }
   }

  /***
   * Add a element to the panel. Elements are asked to draw themselves
   * whenever the panel needs to render. For this purpose, they will receive a
   * calls to draw().
   * Elements are reported of changes in the world coordinates of the panel, in case
   * they need to recalculate themselves.
   * @method addElement
   * @param element Element
   * @param position int
   */
  self.addElement = function(element, position) {
    EJSS_TOOLS.addToArray(mElements,element,position);
    // set this panel to decoration element
    element.setPanel(self);
	if (element.dataCollected) mCollectersList.push(element);
	// elements list has changed
	mElementsChanged = true;		
  };
  
  /***
   * Remove a element to the panel.
   * @method removeElement
   * @param element Element 
   */
  self.removeElement = function(element) {
    EJSS_TOOLS.removeFromArray(mElements,element);
    element.setPanel(null);
	if (element.dataCollected) EJSS_TOOLS.removeFromArray(mCollectersList, element);
	// elements list has changed
	mElementsChanged = true;
  };

  /***
   * Return the array of a elements.
   * @method getElements
   * @return Elements 
   */
  self.getElements = function() {
    return mElements;
  };
    
  /***
   * Return the position of a element.
   * @method indexOfElement
   * @param element Element
   * @return integer 
   */
  self.indexOfElement = function(element) {
    return mElements.indexOf(element);
  };
           
  // ----------------------------------------
  // Drawing functions
  // ----------------------------------------

  /***
   * Get the type of projection
   * @method getProjection
   * @return type 
   */
  self.getProjection = function() {
  	return mProjection;
  }
  
  /***
   * Set the type of projection
   * @method setProjection
   * @return type "PLANAR_XY", "PLANAR_XZ", "PLANAR_YZ", "PERSPECTIVE_OFF", "PERSPECTIVE_ON"
   */
  self.setProjection = function(proj) {
    if (typeof(proj)=='string') {
      switch (proj) {
        case "PLANAR_XY" : proj = 0; break;
      	case "PLANAR_XZ" : proj = 1; break; 
      	case "PLANAR_YZ" : proj = 2; break; 
      	case "PERSPECTIVE_OFF" : proj = 3; break; // PLANAR_ZY
        default : 
      	case "PERSPECTIVE_ON" : proj = 4; break; // PLANAR_ZY
      }    
    }
    if(mProjection != proj) {
	    switch (proj) {
	      case 0 : // PLANAR_XY
	  			mProjection = 0;
	  			self.setOrthographic(true);
	  			self.setCamUpVector([0,1,0]);
	  			self.setCamAzimuth(0);
	  			self.setCamAltitude(0);
	  			self.setCamTilt(0);
	  			break; 
	      case 1 : // PLANAR_XZ
	  			mProjection = 1;
	  			self.setOrthographic(true);
	  			self.setCamUpVector([0,0,-1]);
	  			self.setCamAzimuth(0);
	  			self.setCamAltitude(0);
	  			self.setCamTilt(0);
	  			break; 
	      case 2 : // PLANAR_YZ
	  			mProjection = 2;
	  			self.setOrthographic(true);
	  			self.setCamUpVector([0,1,0]);
	  			self.setCamAzimuth(0);
	  			self.setCamAltitude(0);
	  			self.setCamTilt(0);
	  			break; 
	      case 3 : // PLANAR_ZY
	  			mProjection = 3;
	  			self.setOrthographic(true);
	  			self.setCamUpVector([0,0,1]);
	  			self.setCamAzimuth(0);
	  			self.setCamAltitude(0);
	  			self.setCamTilt(0);
	  			break; 
	      default : 
	      case 4 : // PLANAR_ZY
	  			mProjection = 4;
	  			self.setOrthographic(false);
	  			self.setCamUpVector([0,0,1]);
	  			self.setCamAzimuth(0);
	  			self.setCamAltitude(0);
	  			self.setCamTilt(0);
	  			break; 
	    }        	
	    mChangeWorld = true;
    }
  }

  /***
   * Whether the projection is orthographic
   * @method getOrthographic
   * @return boolean
   */
  self.getOrthographic = function() {
  	return mOrthographic;
  }
  
  /***
   * Set orthographic projection
   * @method setOrthographic
   * @param boolean
   */
  self.setOrthographic = function(orth) {
    if (mOrthographic !== orth) { 
      mOrthographic = orth;
      mChangeWorld = true; 
    }  	 
  }

  /***
   * Get field of view in non-orthographic projection
   * @method getFOV
   * @param int
   */
  self.getFOV = function() {
  	return mFOV;
  }
  
  /***
   * Set field of view in non-orthographic projection
   * @method setFOV
   * @return int
   */
  self.setFOV = function(fov) {
    if (mFOV !== fov) { 
      mFOV = fov;
      mChangeWorld = true; 
    }  	 
  }

  /***
   * Get far plane in non-orthographic projection
   * @method getFar
   * @param number
   */
  self.getFar = function() {
  	return mFar;
  }

  /***
   * Set far plane in non-orthographic projection
   * @method setFar
   * @return number
   */
  self.setFar = function(far) {
    if (mFar !== far) { 
      mFar = far;
      mChangeWorld = true; 
    }  	 
  }

  /***
   * Get near plane in non-orthographic projection
   * @method getNear
   * @param number
   */
  self.getNear = function() {
  	return mNear;
  }
  
  /***
   * Set near plane in non-orthographic projection
   * @method setNear
   * @return number
   */
  self.setNear = function(near) {
    if (mNear !== near) { 
      mNear = near;
      mChangeWorld = true; 
    }  	 
  }

  /***
   * Reset the scene
   * @method reset
   */
  self.reset = function() {
  	// dimensions for orthographic view
  	var dim = {};
  	if(mOrthographic) {
 		// using world
		var aspect = mGraphics.getAspect(); // width / height
		var size = new Vector(mSizeX,mSizeY,mSizeZ);
		var min = new Vector(mWorld.xminPreferred,mWorld.yminPreferred,mWorld.zminPreferred).multiply(size);
		var max = new Vector(mWorld.xmaxPreferred,mWorld.ymaxPreferred,mWorld.zmaxPreferred).multiply(size);
		var diff = max.subtract(min);

        switch (mProjection) {
          case 0: // "PLANAR_XY"
			var ratio = (diff.y/diff.x) * aspect;
            if(ratio>=1.0) {
            	var width = (diff.x * ratio) / 2; var height = diff.y / 2;
			} else {
            	var width = diff.x / 2; var height = (diff.y * (1 / ratio - 1)) / 2;
          	} 
          	break; 
          case 1: // "PLANAR_XZ"
			var ratio = (diff.z/diff.x) * aspect;
            if(ratio>=1.0) {
            	var width = (diff.x * ratio) / 2; var height = diff.z / 2;
			} else {
            	var width = diff.x / 2; var height = (diff.z * (1 / ratio - 1)) / 2;
          	} 
          	break; 
          case 2: // "PLANAR_YZ"
			var ratio = (diff.y/diff.z) * aspect;
            if(ratio>=1.0) {
            	var width = (diff.z * ratio) / 2; var height = diff.y / 2;
			} else {
            	var width = diff.z / 2; var height = (diff.y * (1 / ratio - 1)) / 2;
          	} 
            break;
       	  default: // "PERSPECTIVE_OFF" "PLANAR_ZY"
			var ratio = (diff.z/diff.y) * aspect;
            if(ratio>=1.0) {
            	var width = (diff.y * ratio) / 2; var height = diff.z / 2;
			} else {
            	var width = diff.y / 2; var height = (diff.z * (1 / ratio - 1)) / 2;
          	} 
            break;
		}					
		// with margin
		var rate = 1.10;
   		dim = {left: -width*rate, right: width*rate, bottom: -height*rate, top: height*rate}; 
  	} // else pespective view based on fov

    mGraphics.reset(mOrthographic, dim, mFOV, mNear, mFar);
  }
  
  /***
   * Render the scene
   * @method render
   */
  self.render = function() {
    if(mElementsChanged || mChangeWorld)  // whether elements added or removed, reset the scene 
		self.reset();

    // check for data collection
	for (var i = 0, n = mCollectersList.length; i < n; i++)
		mCollectersList[i].dataCollected();
			
    if (mPanelChanged || mElementsChanged) // whether style panel changed or reseted    	    	 
      	mGraphics.drawPanel(self);

	// change decorations
    if(mChangeWorld) {
		// listener for decorations
		for (i = 0, n = mDecorations.length; i < n; i++) {
			if (mDecorations[i].panelChangeListener)
				mDecorations[i].panelChangeListener("bounds");
		}    	

		// auto change camera position	
		if(mAutoCamera) {
			// focus
			var size = new Vector(mSizeX,mSizeY,mSizeZ);
			var min = new Vector(mWorld.xminPreferred,mWorld.yminPreferred,mWorld.zminPreferred).multiply(size);
			var max = new Vector(mWorld.xmaxPreferred,mWorld.ymaxPreferred,mWorld.zmaxPreferred).multiply(size);
	  		mCamera.focus = max.add(min).divide(2);			
			if(!mOrthographic) { // Axis YZ
				// location (it depends on the fov)
				var aspect = mGraphics.getAspect();
				var diff = max.subtract(min);
            	var apply = (diff.z*aspect>diff.y)?diff.z:diff.y; 				
				var near = 1.1 * (apply/2) / Math.tan(mFOV*Math.PI/360); // near, but far 1.1
				mCamera.location = mCamera.focus.clone();
				mCamera.location.x = max.x + near;
			} else {
				mCamera.location = mCamera.focus.clone();
		        switch (mProjection) {
		          case 0: // "PLANAR_XY"
					mCamera.location.z = mWorld.zmaxPreferred*mSizeZ*4; // a little far		          
		          	break; 
		          case 1: // "PLANAR_XZ"
   					mCamera.location.y = mWorld.ymaxPreferred*mSizeY*4; // a little far
		          	break; 
		          case 2: // "PLANAR_YZ"
		            mCamera.location.x = mWorld.xmaxPreferred*mSizeX*4; // a little far
		            break;
		          default: // "PERSPECTIVE_OFF" 
					mCamera.location.x = mWorld.xmaxPreferred*mSizeX*4; // a little far
		          	break;
				}				
			}
	    }
    }
        	
    // check upvector
    if(mCamera.upvector.cross(mCamera.location.subtract(mCamera.focus)).length() == 0) {
    	// upvector and camera direction are paralell 
    	mCamera.upvector = mCamera.upvector.rotate(); // for upvector (0,0,1) to (1,0,0)
    }
    
    // locate camera
    // if(mChangeCamera)
    	mGraphics.putCamera(mOrthographic,mCamera);

    // draw visible elements
   	mGraphics.draw(mDecorations); // draw the decorations
	mGraphics.draw(mElements);
	
	// set changed to false
	mChangeWorld = false;
	mPanelChanged = false;
	mChangeCamera = false;
	mElementsChanged = false;	
	for (var i=0, n=mElements.length; i<n; i++) {
		mElements[i].setProjChanged(false);
		mElements[i].setMeshChanged(false);
	} 
  };

  // ----------------------------------------
  // Interaction
  // ----------------------------------------
  
  /***
   * Whether the panel should respond to user interaction
   * @method setEnabled
   * @param enabled boolean
   */
  self.setEnabled = function(enabled) { 
  	mInteraction.setEnabled(enabled);
  };
 
  /***
   * How the panel should respond to user interaction
   * @method setDraggable
   * @param draggable "NONE", "ANY", "AZIMUTH", "ALTITUDE"
   */
  self.setDraggable = function(draggable) {
    if (typeof(draggable)=='string') {
      switch (draggable) {
        default : 
        case "NONE" : draggable = 0; break; 
        case "ANY" : draggable = 1; break; 
        case "AZIMUTH" : draggable = 2; break; 
        case "ALTITUDE" : draggable = 3; break;
      }    
    }
	mDraggable = draggable;
  };

  /***
   * Return panel interaction
   * @method getPanelInteraction
   * @return panel interaction
   */
  self.getPanelInteraction = function() { 
  	return mInteraction;
  };
 
 
  /***
   * Returns the controller object
   * @method getController
   * @return Controller
   */
  self.getController = function () {
    return mController;
  };

  /***
   * Set the controller
   * @method setController
   * @param Controller
   */
  self.setController = function (controller) {
    mController = controller;
  };
    

  // ----------------------------------------------------
  // Properties
  // ----------------------------------------------------

  self.getInteraction = function() {
    return mInteraction;
  };

  self.registerProperties = function(controller) {
    EJSS_DRAWING3D.DrawingPanel.registerProperties(self,controller);
  };
  
  self.getOnDragHandler = function() {
  	if(mDraggable > 0) {
	  	var deltas = self.getPanelInteraction().getInteractionDeltas();

		// delta[0] move right/left
        switch (mProjection) {
          case 0: // "PLANAR_XY"
		    if(mDraggable == 2 || mDraggable == 1) {
			    mCamera.altitude += deltas[0];
		    }
          	break; 
          case 1: // "PLANAR_XZ"
		    if(mDraggable == 2 || mDraggable == 1) {	    
		    	mCamera.azimuth -= deltas[0];
			}  		
          	break; 
          case 2: // "PLANAR_YZ"
		    if(mDraggable == 2 || mDraggable == 1) {
			    mCamera.altitude += deltas[0];
		    }
          	break; 
          default: // "PERSPECTIVE_OFF": 
		    if(mDraggable == 2 || mDraggable == 1) {
			    mCamera.azimuth += deltas[0];
		    }
		    break;
		}

		// delta[1] move up/down
        switch (mProjection) {
          case 0: // "PLANAR_XY"
		    if(mDraggable == 3 || mDraggable == 1) {	    
		    	mCamera.tilt += deltas[1];
			}  		
          	break; 
          case 1: // "PLANAR_XZ"
		    if(mDraggable == 3 || mDraggable == 1) {	    
		    	mCamera.tilt += deltas[1];
			}  		
          	break; 
          case 2: // "PLANAR_YZ" 
		    if(mDraggable == 3 || mDraggable == 1) {	    
		    	mCamera.azimuth -= deltas[1];
			}  		
		    break;
          default: // "PERSPECTIVE_OFF"
		    if(mDraggable == 3 || mDraggable == 1) {	    
		    	mCamera.altitude += deltas[1];
			}  		
		    break;
		}
		
  	}

	self.getController().propertiesChanged("CameraAzimuth","CameraAltitude","CameraTilt");  	
  	
  	mChangeCamera = true;
  }     

  self.getOnZoomHandler = function() {
  	var delta = self.getPanelInteraction().getInteractionZoomDelta();
	var eyevector = mCamera.focus.subtract(mCamera.location);
	if(delta > 0) { // move 10% distance from location to focus camera
	  	mCamera.location = mCamera.focus.subtract(eyevector.divide(mZoomRate)); 
	  	mCamera.delta = mZoomRate;
	} else if(delta < 0) {
	  	mCamera.location = mCamera.focus.subtract(eyevector.multiply(mZoomRate));
	  	mCamera.delta = 2.0 - mZoomRate; 
	}
	self.getController().propertiesChanged("CameraLocation","CameraX","CameraY","CameraZ");

  	mChangeCamera = true;
  }     
    
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  mStyle.setFillColor([0.83,0.85,1]);
  mStyle.setMeshChangeListener(function (change) { mPanelChanged = true; });
  mStyle.setProjChangeListener(function (change) { mPanelChanged = true; });

  // box decoration
  var mBoxDecoration = EJSS_DRAWING3D.box(mName + "box");
  mBoxDecoration.getStyle().setDrawFill(false);
  mBoxDecoration.getStyle().setDrawLines(true);
  mBoxDecoration.getStyle().setLineColor("black");
  mBoxDecoration.setVisible(true);
  mBoxDecoration.setSize([2,2,2]);  
  mBoxDecoration.panelChangeListener = function(event) {
	if (event == "bounds") {
		var posx = (mWorld.xmaxPreferred + mWorld.xminPreferred)/2;
		var posy = (mWorld.ymaxPreferred + mWorld.yminPreferred)/2;
		var posz = (mWorld.zmaxPreferred + mWorld.zminPreferred)/2;
		var sx = Math.abs(mWorld.xmaxPreferred - mWorld.xminPreferred);
		var sy = Math.abs(mWorld.ymaxPreferred - mWorld.yminPreferred);
		var sz = Math.abs(mWorld.zmaxPreferred - mWorld.zminPreferred);		
		mBoxDecoration.setPosition([posx,posy,posz]);
		mBoxDecoration.setSize([sx,sy,sz]);
	}
  };  
  self.addDecoration(mBoxDecoration, 0);

  // axis decoration
  var mAxisXDecoration = EJSS_DRAWING3D.arrow(mName + "axisX");
  mAxisXDecoration.setColor("red");
  mAxisXDecoration.setVisible(false);
  mAxisXDecoration.panelChangeListener = function(event) {
	if (event == "bounds") {
  		mAxisXDecoration.setPosition([mWorld.xminPreferred,mWorld.yminPreferred,mWorld.zminPreferred]);
		var sx = Math.abs(mWorld.xmaxPreferred - mWorld.xminPreferred);
  		mAxisXDecoration.setSize([sx,0,0]);
  	}
  };  
  self.addDecoration(mAxisXDecoration, 0);

  var mAxisYDecoration = EJSS_DRAWING3D.arrow(mName + "axisY");
  mAxisYDecoration.setPosition([0,0,0]);
  mAxisYDecoration.setSize([0,1,0]);
  mAxisYDecoration.setColor("green");
  mAxisYDecoration.setVisible(false);
  mAxisYDecoration.panelChangeListener = function(event) {
	if (event == "bounds") {
  		mAxisYDecoration.setPosition([mWorld.xminPreferred,mWorld.yminPreferred,mWorld.zminPreferred]);
		var sy = Math.abs(mWorld.ymaxPreferred - mWorld.yminPreferred);  		
  		mAxisYDecoration.setSize([0,sy,0]);
  	}
  };  
  self.addDecoration(mAxisYDecoration, 0);

  var mAxisZDecoration = EJSS_DRAWING3D.arrow(mName + "axisZ");
  mAxisZDecoration.setPosition([0,0,0]);
  mAxisZDecoration.setSize([0,0,1]);
  mAxisZDecoration.setColor("blue");
  mAxisZDecoration.setVisible(false);
  mAxisZDecoration.panelChangeListener = function(event) {
	if (event == "bounds") {		
  		mAxisZDecoration.setPosition([mWorld.xminPreferred,mWorld.yminPreferred,mWorld.zminPreferred]);
		var sz = Math.abs(mWorld.zmaxPreferred - mWorld.zminPreferred);		
  		mAxisZDecoration.setSize([0,0,sz]);
  	}
  };  
  self.addDecoration(mAxisZDecoration, 0);

  // basic axis decoration
  var mBasicAxisXDecoration = EJSS_DRAWING3D.arrow(mName + "baxisX");
  mBasicAxisXDecoration.setPosition([0,0,0]);
  mBasicAxisXDecoration.setSize([1,0,0]);
  mBasicAxisXDecoration.setColor("red");
  mBasicAxisXDecoration.setVisible(false);
  mBasicAxisXDecoration.panelChangeListener = function(event) {
	if (event == "bounds") {
		var sx = Math.abs(mWorld.xmaxPreferred - mWorld.xminPreferred)/2;
  		mBasicAxisXDecoration.setSize([sx,0,0]);
  	}
  };    
  self.addDecoration(mBasicAxisXDecoration, 0);

  var mBasicAxisYDecoration = EJSS_DRAWING3D.arrow(mName + "baxisY");
  mBasicAxisYDecoration.setPosition([0,0,0]);
  mBasicAxisYDecoration.setSize([0,1,0]);
  mBasicAxisYDecoration.setColor("green");
  mBasicAxisYDecoration.setVisible(false);
  mBasicAxisYDecoration.panelChangeListener = function(event) {
	if (event == "bounds") {
		var sy = Math.abs(mWorld.ymaxPreferred - mWorld.yminPreferred)/2;  		
  		mBasicAxisYDecoration.setSize([0,sy,0]);
  	}
  };    
  self.addDecoration(mBasicAxisYDecoration, 0);

  var mBasicAxisZDecoration = EJSS_DRAWING3D.arrow(mName + "baxisZ");
  mBasicAxisZDecoration.setPosition([0,0,0]);
  mBasicAxisZDecoration.setSize([0,0,1]);
  mBasicAxisZDecoration.setColor("blue");
  mBasicAxisZDecoration.setVisible(false);
  mBasicAxisZDecoration.panelChangeListener = function(event) {
	if (event == "bounds") {
		var sz = Math.abs(mWorld.zmaxPreferred - mWorld.zminPreferred)/2;		
  		mBasicAxisZDecoration.setSize([0,0,sz]);
  	}
  };  
  self.addDecoration(mBasicAxisZDecoration, 0);
  
  mInteraction = EJSS_DRAWING3D.panelInteraction(self);
  
  return self;
};






/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/***
 * Element is the basic class for 3D elements
 * @class EJSS_DRAWING3D.Element 
 * @constructor  
 */
EJSS_DRAWING3D.Element = {
    
    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

    /**
     * Copies one element into another
     * @method copyTo
     */
    copyTo : function(source, dest) {
      EJSS_DRAWING3D.Style.copyTo(source.getStyle(),dest.getStyle());
      dest.setVisible(source.isVisible());
      dest.setMeasured(source.isMeasured());

      dest.setPosition(source.getPosition());
      dest.setSize(source.getSize());
      dest.setTransformation(source.getTransformation());
      dest.setExtraTransformation(source.getExtraTransformation());
      dest.setTextureUrl(source.getTextureUrl());

	  dest.setResolution(source.getResolution());

      dest.setParent(source.getParent());
    },
    
    /**
     * Registers properties in a ControlElement
     * @method registerProperties
     * @param element The element with the properties
     * @param controller A ControlElement that becomes the element controller
     */
    registerProperties : function(element,controller) {
      element.setController(controller);

	 /*** 
	  * Parent of the element
	  * @property Parent 
	  * @type Panel|Group
	  */  
      controller.registerProperty("Parent", element.setParent, element.getParent); 

	 /*** 
	  * Position in X
	  * @property X 
	  * @type double
	  * @default 0
	  */  
      controller.registerProperty("X",element.setX,element.getX);
	 /*** 
	  * Position in Y
	  * @property Y 
	  * @type double
	  * @default 0
	  */  
      controller.registerProperty("Y",element.setY,element.getY);
	 /*** 
	  * Position in Z
	  * @property Z 
	  * @type double
	  * @default 0
	  */  
      controller.registerProperty("Z",element.setZ,element.getZ);
	 /*** 
	  * Coordinates X, Y, and Z
	  * @property Position 
	  * @type double[3]
	  * @default [0,0,0]
	  */        
      controller.registerProperty("Position",element.setPosition,element.getPosition);

	 /*** 
	  * Size along the X axis
	  * @property SizeX 
	  * @type double
	  * @default 1
	  */                          
      controller.registerProperty("SizeX",element.setSizeX,element.getSizeX);
	 /*** 
	  * Size along the Y axis
	  * @property SizeY 
	  * @type double
	  * @default 1
	  */                          
      controller.registerProperty("SizeY",element.setSizeY,element.getSizeY);
	 /*** 
	  * Size along the Z axis
	  * @property SizeZ 
	  * @type double
	  * @default 1
	  */                          
      controller.registerProperty("SizeZ",element.setSizeZ,element.getSizeZ);
	 /*** 
	  * Size along the X, Y, and Z axes 
	  * @property Size 
	  * @type double[3]
	  * @default [1,1,1]
	  */                          
      controller.registerProperty("Size",element.setSize,element.getSize);

      controller.registerProperty("TextureUrl",element.setTextureUrl,element.getTextureUrl);

      controller.registerProperty("Transformation",element.setTransformation);

      controller.registerProperty("Visibility",  element.setVisible, element.isVisible);
      controller.registerProperty("Measured", element.setMeasured, element.isMeasured);

      controller.registerProperty("AlwaysUpdated", element.setAlwaysUpdated, element.isAlwaysUpdated);

      controller.registerProperty("Transparency",  element.getStyle().setTransparency, element.getStyle().getTransparency);

      controller.registerProperty("AmbientColor",  element.getStyle().setAmbientColor, element.getStyle().getAmbientColor);
      controller.registerProperty("FillColor",  element.getStyle().setFillColor, element.getStyle().getFillColor);
      controller.registerProperty("SpecularColor",  element.getStyle().setSpecularColor, element.getStyle().getSpecularColor);
      controller.registerProperty("AmbientReflection",  element.getStyle().setAmbientReflection, element.getStyle().getAmbientReflection);
      controller.registerProperty("ColorReflection",  element.getStyle().setColorReflection, element.getStyle().getColorReflection);
      controller.registerProperty("SpecularReflection",  element.getStyle().setSpecularReflection, element.getStyle().getSpecularReflection);
      controller.registerProperty("Shininess",  element.getStyle().setShininessVal, element.getStyle().getShininessVal);

      controller.registerProperty("PaletteFloor",  element.getStyle().setPaletteFloor, element.getStyle().getPaletteFloor);
      controller.registerProperty("PaletteCeil",  element.getStyle().setPaletteCeil, element.getStyle().getPaletteCeil);
      controller.registerProperty("PaletteFloorColor",  element.getStyle().setPaletteFloorColor, element.getStyle().getPaletteFloorColor);
      controller.registerProperty("PaletteCeilColor",  element.getStyle().setPaletteCeilColor, element.getStyle().getPaletteCeilColor);

      controller.registerProperty("LineColor",  element.getStyle().setLineColor, element.getStyle().getLineColor);
      controller.registerProperty("LineWidth",  element.getStyle().setLineWidth, element.getStyle().getLineWidth);
      controller.registerProperty("DrawLines",  element.getStyle().setDrawLines, element.getStyle().getDrawLines);
      controller.registerProperty("DrawFill",   element.getStyle().setDrawFill, element.getStyle().getDrawFill);
      controller.registerProperty("ClosedTop",   element.getStyle().setClosedTop, element.getStyle().getClosedTop);
      controller.registerProperty("ClosedBottom",   element.getStyle().setClosedBottom, element.getStyle().getClosedBottom);
      controller.registerProperty("ClosedLeft",   element.getStyle().setClosedLeft, element.getStyle().getClosedLeft);
      controller.registerProperty("ClosedRight",   element.getStyle().setClosedRight, element.getStyle().getClosedRight);

	  controller.registerProperty("Resolution", element.setResolution);
      controller.registerProperty("Color", element.setColor);

	 /*** 
	  * Whether the user could change the position   
	  * @property EnabledPosition 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("EnabledPosition",function(enabled) {
        //element.getInteractionTarget(TARGET_POSITION).setMotionEnabled(enabled);
      });

	 /*** 
	  * Whether the group position also changes when the element position changes    
	  * @property MovesGroup 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("MovesGroup", function(affects) {
        //element.getInteractionTarget(TARGET_POSITION).setAffectsGroup(affects);
      });

	 /*** 
	  * Whether the user could change the size   
	  * @property EnabledSize 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("EnabledSize",function(enabled) {
        //element.getInteractionTarget(TARGET_SIZE).setMotionEnabled(enabled);
      });

	 /*** 
	  * Whether the group size also changes when the element size changes    
	  * @property ResizesGroup 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("ResizesGroup", function(affects) {
        //element.getInteractionTarget(TARGET_SIZE).setAffectsGroup(affects);
      });


      // Actions
      controller.registerAction("OnEnter",   element.getPosition);
      controller.registerAction("OnExit",    element.getPosition);
      controller.registerAction("OnPress",   element.getPosition);
      controller.registerAction("OnDrag",    element.getPosition);
      controller.registerAction("OnRelease", element.getPosition);
      
      controller.registerAction("OnLoadTexture", null, null, function() { element.setProjChanged(true);});
    }
};

/**
 * Constructor for Element
 * @method element
 * @param mName string
 * @returns An abstract 3D element
 */
EJSS_DRAWING3D.element = function(mName) {
  var self = {};							// reference returned 

  // Static references
  var Element = EJSS_DRAWING3D.Element;		// reference for Element

  // Instance variables
  var mStyle = EJSS_DRAWING3D.style(mName);	// style for element 
  var mVisible = true;								// whether visible in drawing
  var mMeasured = true;								// whether measure for element

  var mTextureUrl = "";

  // Position and size
  var mX = 0;						// position X
  var mY = 0;						// position Y
  var mZ = 0;						// position Z
  var mSizeX = 1;					// size X
  var mSizeY = 1;					// size Y
  var mSizeZ = 1;					// size Z
  var mTransformation = [];			// transformations for element
  var mExtraTransformations = [];			// extra transformations for element
  var mFirstTransformation = []; // very first transformation, not to be set by the user directly

  // Implementation variables    
  var mPanel = null;				// drawing panel for element
  var mGroup = null;				// group for element
  var mSet = null;				    // The set it belongs to (if any)  
  var mIndexInSet = -1;				// The index of the element in a set (if any)  
  var mMeshChanged = true;		    // whether element mesh changed (resolution, ...)
  var mProjChanged = true;			// whether element projection changed (position, size, or group)
  var mAlwaysUpdated = false;		// whether element mesh must be always drawn

  var mResolutionU = 20;
  var mResolutionV = 20;
  var mResolutionR = 0;
  
  var mController = { 				// dummy controller object
      propertiesChanged : function() {},
      invokeAction : function() {}
  };

  // ----------------------------------------
  // Instance functions
  // ----------------------------------------

  /***
   * Copies itself to another element
   * @method copyTo
   * @param element Element
   */
  self.copyTo = function(element) {
    Element.copyTo(self,element);
  };
  
  /***
   * Get name for element
   * @method getName
   * @return string
   */
  self.getName = function() {
    return mName.replace(/\[|\]/g,"");
  };  

  /***
   * Set texture url
   * @method setTextureUrl
   * @param url
   */
  self.setTextureUrl = function(url) {
//  	console.log ("Texture = "+url);
//  	console.log ("Set = "+mSet);
//  	if (mSet!=null) console.log ("ResPathF = "+mSet.getResourcePath);
  	var resPathFunction = (mSet!=null) ? mSet.getResourcePath : self.getResourcePath; 
  	if (resPathFunction!=null) {
  	  url = resPathFunction(url);
  	  //alert ("  set to = "+url+"\n");
  	}
  	else { 
  		console.log ("No getResourcePath function for "+self.getName()+". Texture = " + url);
  	}
  	mTextureUrl = url;
  }

  /***
   * Get texture url
   * @method getTextureUrl
   * @return url
   */
  self.getTextureUrl = function() {
  	return mTextureUrl;
  }

  // -------------------------------------
  // Visible, style, measure
  // -------------------------------------

  /***
   * Sets the visibility of the element
   * @method setVisible
   * @param visible boolean
   */
  self.setVisible = function(visible) { 
    mVisible = visible; 
  };

  /***
   * Whether the element is visible
   * @method isVisible
   * @return boolean
   */
  self.isVisible = function() { 
    return mVisible; 
  };

  /***
   * Returns the real visibility status of the element, which will be false if
   * it belongs to an invisible group
   * @method isGroupVisible
   * @return boolean
   */
  self.isGroupVisible = function() {
    var el = mGroup;
    while (typeof el != "undefined" && el !== null) {
      if (!el.isVisible()) return false;
      el = el.getGroup();
    }
    return mVisible;
  };

  /***
   * Return the style (defined in DrawingPanel.js) of the inner rectangle
   * @method getStyle 
   * @return boolean
   */
  self.getStyle = function() { 
    return mStyle; 
  };

  /***
   * Sets the measurability of the element
   * @method setMeasured
   * @param measured boolean
   */
  self.setMeasured = function(measured) { 
    mMeasured = measured; 
  };

  /***
   * Whether the element is measured
   * @method isMeasured
   * @return boolean
   */
  self.isMeasured = function() { 
    return mMeasured; 
  };

  /***
   * Set color 
   * @method setColor
   * @param color
   */
  self.setColor = function(color) {
  	self.getStyle().setFillColor(color);
  	self.getStyle().setLineColor(color);
  }

  /***
   * Set color 
   * @method setColor
   * @param color
   */
  self.setResolution = function(resolution) {
  	if (Array.isArray(resolution)) {
  		self.setResolutionU(resolution[0]);
  		self.setResolutionV(resolution[1]);
  		if(resolution[2]) self.setResolutionR(resolution[2]);
  	} else {
  		self.setResolutionU(resolution);
  		self.setResolutionV(resolution);  		
  	}
  }
	
  /** TODO **/
  self.getResolution = function() {
  	return [mResolutionU,mResolutionV,mResolutionR];
  }

  self.setResolutionU = function(resolution) {
  	if (mResolutionU!=resolution) {
  		mResolutionU = resolution;
  		mMeshChanged = true; 
  	}
  }

  self.getResolutionU = function() {
  	return mResolutionU;
  }

  self.setResolutionV = function(resolution) {
  	if (mResolutionV!=resolution) {
  		mResolutionV = resolution;
  		mMeshChanged = true; 
  	}
  }

  self.getResolutionV = function() {
  	return mResolutionV;
  }

  self.setResolutionR = function(resolution) {
  	if (mResolutionR!=resolution) {
  		mResolutionR = resolution;
  		mMeshChanged = true; 
  	}
  }

  self.getResolutionR = function() {
  	return mResolutionR;
  }

  // ----------------------------------------
  // Position of the element
  // ----------------------------------------

  /***
   * Set the X coordinate of the element
   * @method setX
   * @param x double
   */
  self.setX = function(x) { 
    if (mX!=x) { 
      mX = x; 
      mProjChanged = true; 
    } 
  };

  /***
   * Get the X coordinate of the element
   * @method getX
   * @return double
   */
  self.getX = function() { 
    return mX; 
  };

  /***
   * Set the Y coordinate of the element
   * @method setY
   * @param y double
   */
  self.setY = function(y) {  
    if (mY!=y) { 
      mY = y; 
      mProjChanged = true; 
    } 
  };

  /***
   * Get the Y coordinate of the element
   * @method getY
   * @return double
   */
  self.getY = function() { 
    return mY; 
  };

  /***
   * Set the Z coordinate of the element
   * @method setZ
   * @param z double
   */
  self.setZ = function(z) {  
    if (mZ!=z) { 
      mZ = z; 
      mProjChanged = true; 
    } 
  };

  /***
   * Get the Z coordinate of the element
   * @method getZ
   * @return double
   */
  self.getZ = function() { 
    return mZ; 
  };

  /***
   * Set the coordinates of the element
   * @method setPosition
   * @param position double[] an array of dimension 3
   */
  self.setPosition = function(position) {
    self.setX(position[0]);
    self.setY(position[1]);
    self.setZ(position[2]);
  };

  /***
   * Get the coordinates of the element
   * @method getPosition
   * @return double[3]
   */
  self.getPosition = function() { 
    return [mX, mY, mZ]; 
  };
    
  // ----------------------------------------
  // Size of the element
  // ----------------------------------------

  /***
   * Set the size along the X axis of the element
   * @method setSizeX
   * @param sizeX double
   */
  self.setSizeX = function(sizeX) { 
    if (mSizeX!=sizeX) { 
      mSizeX = sizeX; 
      mProjChanged = true; 
    } 
  };

  /***
   * Get the size along the X coordinate of the element
   * @method getSizeX
   * @return double
   */
  self.getSizeX = function() { 
    return mSizeX; 
  };

  /***
   * Set the size along the Y axis of the element
   * @method setSizeY
   * @param sizeY double
   */
  self.setSizeY = function(sizeY) { 
    if (mSizeY!=sizeY) { 
      mSizeY = sizeY; 
      mProjChanged = true; 
    }
  };

  /***
   * Get the size along the Y coordinate of the element
   * @method getSizeY
   * @return double
   */
  self.getSizeY = function() { 
    return mSizeY; 
  };

  /***
   * Set the size along the Z axis of the element
   * @method setSizeZ
   * @param sizeZ double
   */
  self.setSizeZ = function(sizeZ) { 
    if (mSizeZ!=sizeZ) { 
      mSizeZ = sizeZ; 
      mProjChanged = true; 
    }
  };

  /***
   * Get the size along the Z coordinate of the element
   * @method getSizeZ
   * @return double
   */
  self.getSizeZ = function() { 
    return mSizeZ; 
  };

  /***
   * Set the size of the element
   * @method setSize
   * @param position double[] an array of dimension 3 or an object with {x,y,z} properties
   */
  self.setSize = function(size) {
    self.setSizeX(size[0]);
    self.setSizeY(size[1]);
    self.setSizeZ(size[2]);
  };

  /***
   * Get the sizes of the element
   * @method getSize
   * @return double[]
   */
  self.getSize = function() {
    return [self.getSizeX(), self.getSizeY(), self.getSizeZ()];
  };
      
  // ----------------------------------------
  // Panel and group
  // ----------------------------------------

  /***
   * To be used internally by DrawingPanel only! Sets the panel for this element.
   * @method setPanel
   * @param panel DrawingPanel
   */
  self.setPanel = function(panel) {
    mPanel = panel;
    mProjChanged = true;     
  };

  /***
   * For internal use only, use getGroupPanel() instead. Gets the panel for this element.
   * @method getPanel
   * @return DrawingPanel
   */
  self.getPanel = function() { 
    return mPanel;
  };

  /***
   * Returns the DrawingPanel in which it (or its final ancestor group) is displayed.
   * @method getGroupPanel
   * @return drawing3D.DrawingPanel
   */
  self.getGroupPanel = function() { 
    var el = self;
    while (el.getGroup()) el = el.getGroup();
    return el.getPanel();
  };

  /***
   * To be used internally by Group only! Sets the group of this element.
   * @method setGroup
   * @param group Group
   */
  self.setGroup = function(group) {
    mGroup = group;
    mProjChanged = true; 
  };

  /***
   * Get the group of this element, if any
   * @method getGroup
   * @return Group
   */
  self.getGroup = function() { 
    return mGroup; 
  };

  /***
   * To be used internally by ElementSet only! Sets the index of this element in the set
   * @method setSet
   * @param set ElementSet
   * @param index int
   */
  self.setSet = function(set,index) {
    mSet = set;
    mIndexInSet = index;
  };

  /***
   * Get the index of this element in a set, if any
   * @method getSetIndex
   * @return int
   */
  self.getSetIndex = function() { 
    return mIndexInSet; 
  };

  /***
   * Set the parent
   * @method setParent
   * @param parent Panel or Element
   */
  self.setParent = function(parent) {
  	if(parent.render) { // is a panel  		
  		self.setGroup(null);
  		parent.addElement(self);
  		self.setPanel(parent);
  	} else if (parent.getClass() == "ElementGroup") { // is a group
  		self.setGroup(parent);
  		self.getGroupPanel().addElement(self);
  		self.setPanel(self.getGroupPanel());
  	} else { //
  		console.log("WARNING: setParent() - Parent not valid : "+ parent.getName()); 
  	}
  };

  /***
   * Get the parent
   * @method getParent
   * @return parent Panel or Element
   */
  self.getParent = function() {
  	var parent;
  	if(mGroup !== null) parent = self.getGroup();
  	else parent = self.getPanel();
  	return parent  
  };

  // ----------------------------------------------------
  // Transformations
  // ----------------------------------------------------

  /***
   * Sets the internal transformation of the element.
   * @method setTransformation
   * @param tr list of transformation arrays supports: rotation based on vector [angle, x, y, z, cx, cy, cz], 
   *  rotation based on element center [angle, x, y, z], custom axes [x1, x2, x3, y1, y2, y3, z1, z2, z3] or
   *  full transformation with a matrix 4x4 
   */
  self.setTransformation = function(tr) {
  	if (typeof tr == "undefined" || tr === null) tr = [];
  	if(!EJSS_TOOLS.compareArrays(mTransformation, tr)) {
  		mTransformation = tr;
    	mProjChanged = true;
    }
  };

  self.setFirstTransformation = function(tr) {
  	if (typeof tr == "undefined" || tr === null) tr = [];
  	if(!EJSS_TOOLS.compareArrays(mTransformation, tr)) {
  		mFirstTransformation = tr;
    	mProjChanged = true;
    }
  };

  self.setExtraTransformation = function(tr) {
    mExtraTransformations = tr;
  }

  self.getExtraTransformation = function() {
    return mExtraTransformations;
  }

  self.addExtraTransformation = function(tr) {
  	if (typeof tr == "undefined" || tr === null) return;
    mExtraTransformations.push(tr);
    mProjChanged = true;
  };

  self.removeExtraTransformation = function(tr) {
  	if (typeof tr == "undefined" || tr === null) return;
  	for (var i=0; i<mExtraTransformations.length; i++) {
      var oneTr = mExtraTransformations[i];
      if (oneTr===tr) {
        mExtraTransformations.splice(i,1);
        mProjChanged = true;
        return;
      }
    }
  };

  function centerTransform(transform) {
    var centeredTr = [];
    for (var i=0; i<transform.length; i++) { 
      var tr = transform[i]; 
      if (tr.length>4) centeredTr.push(tr);
      else centeredTr.push(tr.concat(self.getPosition())); // set element center
    }
    return centeredTr;
  };
    
  /***
   * Get the internal transformation of the element.
   * @method getTransformation
   * @return Transformation 
   */
  self.getFullTransformation = function() {
    var fullTr = centerTransform(mFirstTransformation).concat(centerTransform(mTransformation)); //.concat(mFirstTransformation);
    for (var i=0; i<mExtraTransformations.length; i++) {
      fullTr.push(mExtraTransformations[i].getArray());
    }
    return fullTr;
  };
  
  /***
   * Get the internal transformation of the element.
   * @method getTransformation
   * @return Transformation 
   */
  self.getTransformation = function() {
    return mTransformation;
  };

  /***
   * Get the group transformation of the element.
   * @method getGroupTransformation
   * @return Transformation 
   */
  self.getGroupTransformation = function() {
  	var tr = EJSS_DRAWING3D.transformation();
    var el = self;
    while (typeof el != "undefined" && el !== null) {
      tr.concatenate(el.getFullTransformation());
      el = el.getGroup();
    }
    return tr;
  };

  // ----------------------------------------
  // Changes
  // ----------------------------------------

  /***
   * Whether element mesh must be always drawn
   * @method isAlwaysUpdated
   * @return boolean
   */
  self.isAlwaysUpdated = function() {
    return mAlwaysUpdated;
  };

  /***
   * Tells whether element mesh must be always drawn
   * @method setAlwaysUpdated
   * @param always boolean
   */
  self.setAlwaysUpdated = function(always) {
    mAlwaysUpdated = always;
  };

  /***
   * Whether the element mesh has changed
   * @method isChanged
   * @return boolean
   */
  self.isMeshChanged = function() {
    return mMeshChanged;
  };

  /***
   * Tells the element mesh that it has changed.
   * Typically used by subclasses when they change something.
   * @method setMeshChanged
   * @param changed boolean
   */
  self.setMeshChanged = function(changed) {
    mMeshChanged = changed;
  };

  /***
   * Whether the element projection has changed
   * @method isChanged
   * @return boolean
   */
  self.isProjChanged = function() {
    return mProjChanged;
  };

  /***
   * Tells the element projection that it has changed.
   * Typically used by subclasses when they change something.
   * @method setProjChanged
   * @param changed boolean
   */
  self.setProjChanged = function(changed) {
    mProjChanged = changed;
  };

  /***
   * Returns whether the element group has changed.
   * @method isGroupChanged
   * @return boolean
   */
  self.isGroupChanged = function() {
    var el = self.getGroup();
    while (typeof el != "undefined" && el !== null) {
      if (el.isChanged()) return true;
      el = el.getGroup();
    }
    return false;
  };

  // ----------------------------------------
  // Interaction
  // ----------------------------------------

  /***
   * Returns the controller object
   * @method getController
   * @return Controller
   */
  self.getController = function () {
    return mController;
  };

  /***
   * Set the controller
   * @method setController
   * @param Controller
   */
  self.setController = function (controller) {
    mController = controller;
  };

  // ----------------------------------------------------
  // Properties
  // ----------------------------------------------------

  /***
   * Registers properties in a ControlElement
   * @method registerProperties
   * @param controller A ControlElement that becomes the element controller
   */
  self.registerProperties = function(controller) {
    Element.registerProperties(self,controller);
  };

  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  mStyle.setMeshChangeListener(function (change) { mMeshChanged = true; });
  mStyle.setProjChangeListener(function (change) { mProjChanged = true; });
  
  return self;

};

//--------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------
//--------------------------------------------------------------------------------------------------------




/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/***
 * ElementSet is the basic class for a set of Elements
 * @class EJSS_DRAWING3D.ElementSet 
 * @constructor  
 */
EJSS_DRAWING3D.ElementSet = {
    
    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

    /**
     * Registers properties in a ControlElement
     * @method 
     * @param element The element with the properties
     * @param controller A ControlElement that becomes the element controller
     */
    registerProperties : function(set,controller) {
      set.setController(controller); // remember it, in case you change the number of elements
      set.setToEach(function(element,value) { element.setController(value); }, controller); // make all elements in the set report to the same controller

      controller.registerProperty("Parent", 
      	  function(panel) {
      	  	set.setParent(panel); 
      		set.setToEach(function(element,value) { element.setParent(value); }, set); 
      	  } );

      controller.registerProperty("NumberOfElements", set.setNumberOfElements);

      controller.registerProperty("ElementInteracted", set.setElementInteracted, set.getElementInteracted);

//      controller.registerProperty("x",function(v) { set.foreach("setX",v); }, function() { return set.getall("getX"); });
      controller.registerProperty("X",
          function(v) { set.setToEach(function(element,value) { element.setX(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getX(); } ); }      
          );
      controller.registerProperty("Y",
          function(v) { set.setToEach(function(element,value) { element.setY(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getY(); } ); }      
      );
      controller.registerProperty("Z",
          function(v) { set.setToEach(function(element,value) { element.setZ(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getZ(); } ); }      
      );
      controller.registerProperty("Position",
          function(v) { set.setToEach(function(element,value) { element.setPosition(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getPosition(); } ); }      
      );

      controller.registerProperty("SizeX",
          function(v) { set.setToEach(function(element,value) { element.setSizeX(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getSizeX(); } ); }      
          );
      controller.registerProperty("SizeY",
          function(v) { set.setToEach(function(element,value) { element.setSizeY(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getSizeY(); } ); }      
      );
      controller.registerProperty("SizeZ",
          function(v) { set.setToEach(function(element,value) { element.setSizeZ(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getSizeZ(); } ); }      
      );
      controller.registerProperty("Size",
          function(v) { set.setToEach(function(element,value) { element.setSize(value); }, v); },
          function()  { return set.getFromEach(function(element) { return element.getSize(); } ); }      
      );
      controller.registerProperty("TextureUrl", 
          function(v) { set.setToEach(function(element,value) { element.setTextureUrl(value); }, v); }
      );

      controller.registerProperty("Transformation",
      	  function(v) { set.setToEach(function(element,value) { element.setTransformation(value); }, v);
      });

      controller.registerProperty("Visibility", 
          function(v) { set.setToEach(function(element,value) { element.setVisible(value); }, v); }
      );
      controller.registerProperty("Measured", 
          function(v) { set.setToEach(function(element,value) { element.setMeasured(value); }, v); }
      );

      controller.registerProperty("Transparency", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setTransparency(value); }, v); }
      );
      controller.registerProperty("AmbientColor", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setAmbientColor(value); }, v); }
      );
      controller.registerProperty("FillColor", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setFillColor(value); }, v); }
      );
      controller.registerProperty("SpecularColor", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setSpecularColor(value); }, v); }
      );
      controller.registerProperty("AmbientReflection", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setAmbientReflection(value); }, v); }
      );
      controller.registerProperty("ColorReflection", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setColorReflection(value); }, v); }
      );
      controller.registerProperty("SpecularReflection", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setSpecularReflection(value); }, v); }
      );
      controller.registerProperty("Shininess", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setShininessVal(value); }, v); }
      );

      controller.registerProperty("LineColor", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setLineColor(value); }, v); }
      );
      controller.registerProperty("LineWidth", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setLineWidth(value); }, v); }
      );
      controller.registerProperty("DrawLines", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setDrawLines(value); }, v); }
      );
      controller.registerProperty("FillColor", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setFillColor(value); }, v); }
      );
      controller.registerProperty("DrawFill", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setDrawFill(value); }, v); }
      );

      controller.registerProperty("ClosedTop", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setAttributes(value); }, v); }
      );
      controller.registerProperty("ClosedBottom", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setAttributes(value); }, v); }
      );
      controller.registerProperty("ClosedLeft", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setAttributes(value); }, v); }
      );
      controller.registerProperty("ClosedRight", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setAttributes(value); }, v); }
      );

      controller.registerProperty("Resolution", 
          function(v) { set.setToEach(function(element,value) { element.getStyle().setAttributes(value); }, v); }
      );
      controller.registerProperty("Color", 
          function(v) { set.setToEach(function(element,value) { element.setColor(value); }, v); }
      );

	 /*** 
	  * Whether the user could change the position   
	  * @property EnabledPosition 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("EnabledPosition",
        function(v) { set.setToEach(function(element,value) {  }, v); }
        //element.getInteractionTarget(TARGET_POSITION).setMotionEnabled(enabled);
      );

	 /*** 
	  * Whether the group position also changes when the element position changes    
	  * @property MovesGroup 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("MovesGroup", 
        function(v) { set.setToEach(function(element,value) {  }, v); }
        //element.getInteractionTarget(TARGET_POSITION).setAffectsGroup(affects);
      );

	 /*** 
	  * Whether the user could change the size   
	  * @property EnabledSize 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("EnabledSize",
        function(v) { set.setToEach(function(element,value) {  }, v); }
        //element.getInteractionTarget(TARGET_SIZE).setMotionEnabled(enabled);
      );

	 /*** 
	  * Whether the group size also changes when the element size changes    
	  * @property ResizesGroup 
	  * @type boolean
	  * @default false
	  */                                
      controller.registerProperty("ResizesGroup", 
        function(v) { set.setToEach(function(element,value) {  }, v); }
        //element.getInteractionTarget(TARGET_SIZE).setAffectsGroup(affects);
      );


      var dataFunction = function() { 
        //var index = set.getGroupPanel().getPanelInteraction().getIndexElement();
        //set.setElementInteracted(index);
        //return { index: index, position: set.getElements()[index].getPosition() };
        var panel = set.getGroupPanel();
        var index = panel.getPanelInteraction().getIndexElement();
        var element = panel.getElements()[index];
        var elementIndex = element.getSetIndex(); 
        set.setElementInteracted(elementIndex);
  	    controller.propertiesChanged("ElementInteracted");
	    //controller.reportInteractions();	 
        return { index: elementIndex, position: element.getPosition() };   
      };
      
      // Actions
      controller.registerAction("OnEnter",   dataFunction);
      controller.registerAction("OnExit",    dataFunction);
      controller.registerAction("OnPress",   dataFunction);
      controller.registerAction("OnDrag",    dataFunction);
      controller.registerAction("OnRelease", dataFunction);
      
      // controller.registerAction("OnLoadTexture", null, null, function() { element.setProjChanged(true);});
    }
};

/**
 * Element set
 * Creates a basic abstract ElementSet
 * @method elementSet
 * @param mConstructor the function that creates new elements (will be used as element = mConstructor(name))
 * @returns An abstract 2D element set
 */
EJSS_DRAWING3D.elementSet = function (mConstructor, mName) {  
  var self = EJSS_DRAWING3D.group(mName);

  // Static references
  var ElementSet = EJSS_DRAWING3D.ElementSet;		// reference for ElementSet
  
  // Configuration variables
  var mElementList = []; 
  var mNumberOfElementsSet = false;

  // Implementation variables  
  var mElementInteracted = -1;

  // Last list of removed elements
  var mLastElementList = [];

  // ----------------------------------------
  // Configuration methods
  // ----------------------------------------

  /**
   * Sets the number of element of this set
   * @method setNumberOfElements
   * @param numberOfElements the number of elements, must be >= 1
   */
  self.setNumberOfElements = function(numberOfElements) {
	mNumberOfElementsSet = true;
	adjustNumberOfElements(numberOfElements);
  };
  
  /*
   * Adjusts the number of element of this set
   * @method adjustNumberOfElements
   * @param numberOfElements the number of elements, must be >= 1
   */
  function adjustNumberOfElements(numberOfElements) {
    // keep original settings for the new elements
    var name = self.getName ? self.getName() : "unnamed";
    numberOfElements = Math.max(1,numberOfElements);
    var diff = mElementList.length-numberOfElements;
    if (diff > 0) {
    	mLastElementList = mElementList.splice(numberOfElements, diff);
		for (var j = 0; j < mLastElementList.length; j++) { 
			var panel = mLastElementList[j].getPanel(); // remove element from panel
			if(panel) panel.removeElement(mLastElementList[j])
		} 
    } else if (diff < 0) {
    	mLastElementList = [];
    	var controller = self.getController();
    	var oldElement = mElementList[mElementList.length-1];
		for (var i = mElementList.length; i < numberOfElements; i++) {
			var element = mConstructor(name+"["+i+"]"); // new element			
			element.setSet(self,i);
			oldElement.copyTo(element);
			element.setController(controller);
		  	mElementList.push(element);	
		} 
    }
  };

  /**
   * Returns the array with all elements in the set
   * @method getElements
   * @return array of Elements
   */
  self.getElements = function() {
    return mElementList;
  };

  /**
   * Returns last list of removed elements and reset the value
   * @method getLastElements
   * @return last list of Elements
   */
  self.getLastElements = function() {
  	var ret = mLastElementList.slice();
  	mLastElementList = [];
    return ret;
  };
  
  self.setElementInteracted = function(index) {
    mElementInteracted = index;
  };

  self.getElementInteracted = function() {
    return mElementInteracted;
  };
  
  // ----------------------------------------
  // Relation to its panel
  // ----------------------------------------
  
  var super_setPanel = self.setPanel;

  self.setPanel = function(panel) {
    super_setPanel(panel);
    for (var i=0,n=mElementList.length;i<n;i++) mElementList[i].setPanel(panel);
  };

  self.setProjChanged = function(needsIt) {
    for (var i=0,n=mElementList.length;i<n;i++) mElementList[i].setProjChanged(needsIt);
  };
  
  // ----------------------------------------------------
  // Properties
  // ----------------------------------------------------

  /**
   * Applies to each element in the set the function f with argument v, or v[i], if it is an array
   * @method setToEach
   * @param f function
   * @param v arguments
   */
  self.setToEach = function(f,v) {
    if (Array.isArray(v)) {
      if (!mNumberOfElementsSet) { 
    	if (mElementList.length < v.length) adjustNumberOfElements(v.length);
      }
      for (var i=0,n=Math.min(mElementList.length,v.length);i<n;i++) f(mElementList[i],v[i]);
    }
    else {
      for (var i=0,n=mElementList.length;i<n;i++) f(mElementList[i],v);
    }
  };

  /**
   * Returns an array with the result of applying the function to each element of the set
   * @method getFromEach
   * @param f function
   * @return f function return 
   */
  self.getFromEach = function(f) {
    var value = [];
    for (var i=0, n=mElementList.length;i<n;i++) value[i] = f(mElementList[i]);
    return value;
  };

  /**
   * Registers properties in a ControlElement
   * @method registerProperties
   * @param controller A ControlElement that becomes the element controller
   */
  self.registerProperties = function(controller) {
    ElementSet.registerProperties(self,controller);
  };
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------
	
  var element = mConstructor(mName + "[0]"); // new element			
  element.setController(self.getController());
  element.setSet(self,0);
  mElementList.push(element);
  	
  return self;
};
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/**
 * Ellipsoid
 * @class Ellipsoid 
 * @constructor  
 */
EJSS_DRAWING3D.Ellipsoid = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING3D.Element.copyTo(source,dest); // super class copy
  	
		dest.setMinAngleU(source.getMinAngleU());
		dest.setMaxAngleU(source.getMaxAngleU());
		dest.setMinAngleV(source.getMinAngleV());
		dest.setMaxAngleV(source.getMaxAngleV());
		
  	},


	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING3D.Element.registerProperties(element, controller);
		// super class

      controller.registerProperty("MinAngleU", element.setMinAngleU);
      controller.registerProperty("MaxAngleU", element.setMaxAngleU);
      controller.registerProperty("MinAngleV", element.setMinAngleV);
      controller.registerProperty("MaxAngleV", element.setMaxAngleV);
		
	}

};

/**
 * Creates a 3D Ellipsoid
 * @method ellipsoid
 */
EJSS_DRAWING3D.ellipsoid = function (name) {
  var self = EJSS_DRAWING3D.element(name);
  var mMinAngleU = 0; 		// the start angle (in degrees) for the parallels
  var mMaxAngleU = 360;		// the end angle (in degrees) for the parallels
  var mMinAngleV = -90;		// the start angle (in degrees) for the meridians
  var mMaxAngleV = 90;		// the end angle (in degrees) for the meridians

  // Implementation variables
  self.getClass = function() {
  	return "ElementEllipsoid";
  }

  self.setMinAngleU = function(angle) {
  	if(mMinAngleU != angle) {
  	  mMinAngleU = angle;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.getMinAngleU = function() {
  	return mMinAngleU;
  }

  self.setMaxAngleU = function(angle) {
  	if(mMaxAngleU != angle) {
  	  mMaxAngleU = angle;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.getMaxAngleU = function() {
  	return mMaxAngleU;
  }
  
  self.setMinAngleV = function(angle) {
  	if(mMinAngleV != angle) {
  	  mMinAngleV = angle;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.getMinAngleV = function() {
  	return mMinAngleV;
  }

  self.setMaxAngleV = function(angle) {
  	if(mMaxAngleV != angle) {
  	  mMaxAngleV = angle;
  	  self.setMeshChanged(true);
  	}  	
  }

  self.getMaxAngleV = function() {
  	return mMaxAngleV;
  }
    
  self.copyTo = function(element) {
	EJSS_DRAWING3D.Ellipsoid.copyTo(self,element);
  };
      
  self.registerProperties = function(controller) {
	EJSS_DRAWING3D.Ellipsoid.registerProperties(self, controller);
  };
	    
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  self.setSize([1,1,1]);
  self.setResolution([30,20]);

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

//---------------------------------
//EllipsoidSet
//---------------------------------

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

/**
 * EllipsoidSet
 * @class EllipsoidSet 
 * @constructor  
 */
EJSS_DRAWING3D.EllipsoidSet = {

    /**
     * static registerProperties method
     */
    registerProperties : function(set,controller) {
      var ElementSet = EJSS_DRAWING3D.ElementSet;
      ElementSet.registerProperties(set,controller);
       
      controller.registerProperty("MinAngleU", 
          function(v) { set.setToEach(function(element,value) { element.setMinAngleU(value); }, v); }
      );    
      controller.registerProperty("MaxAngleU", 
          function(v) { set.setToEach(function(element,value) { element.setMaxAngleU(value); }, v); }
      );    
      controller.registerProperty("MinAngleV", 
          function(v) { set.setToEach(function(element,value) { element.setMinAngleV(value); }, v); }
      );    
      controller.registerProperty("MaxAngleV", 
          function(v) { set.setToEach(function(element,value) { element.setMaxAngleV(value); }, v); }
      );                             
    }

};


/**
 * Creates a set of ellipsoids
 * @method ellipsoidSet
 * @param mView
 * @param mName
 */
EJSS_DRAWING3D.ellipsoidSet = function (mName) {
  var self = EJSS_DRAWING3D.elementSet(EJSS_DRAWING3D.ellipsoid,mName);

  // Static references
  var EllipsoidSet = EJSS_DRAWING3D.EllipsoidSet;		// reference for EllipsoidSet
  
  self.registerProperties = function(controller) {
    EllipsoidSet.registerProperties(self,controller);
  };

  return self;
};
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/**
 * Group
 * @class Group 
 * @constructor  
 */
EJSS_DRAWING3D.Group = {

};

/**
 * Creates a group
 * @method group
 */
EJSS_DRAWING3D.group = function (name) {
  var self = EJSS_DRAWING3D.element(name);

  self.getClass = function() {
  	return "ElementGroup";
  }
  
  self.isGroup = function() {
  	return true;
  }
  
  // ----------------------------------------------------
  // Final start-up
  // ----------------------------------------------------

  return self;
};



/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/**
 * Constructor for Interaction
 * @param drawing panel
 * @returns An interaction
 */
EJSS_DRAWING3D.panelInteraction = function(mPanel) {
  var self = {};	// reference returned 

  var mEnabled = false;			// whether interaction is enable 
  var mLastLocation = [0,0];	// last position on panel trying interaction 
  var mDeltaX = 0;
  var mDeltaY = 0;
  var mMouseIsDown = false;		// last event is mouse down
  var mUpLastTime = 0;			// last timestamp for mouse up  
  var mZoomLastTime = 0;		// last timestamp for pinching  
  var mZoomDelta = 0;
  var mLastZoom = 0;
  var mPinching = false;
  var mPinchInitDistance = [0,0];

  /**
   * Return panel
   * @return drawing panel 
   */
  self.getPanel = function() {  	
    return mPanel;
  };  

  self.getEnabled = function() {
  	return mEnabled;
  } 

  self.setEnabled = function(enabled) {
  	if(mEnabled != enabled) {
	  	mEnabled = enabled;
	  	
	  	if(mEnabled) {
		  self.setHandler("move",self.handleMouseMoveEvent);
		  self.setHandler("down",self.handleMouseDownEvent);
		  self.setHandler("up",self.handleMouseUpEvent);     		
		  self.setHandler("mousewheel",self.handleMouseWheelEvent);
		  // self.setHandler("pinch",self.handlePinchGesture)
	  	} else { 
		  self.setHandler("move",(function() {}));
		  self.setHandler("down",(function() {}));
		  self.setHandler("up",(function() {}));    			  		
		  self.setHandler("mousewheel",(function() {}));
		  // self.setHandler("pinch",(function() {}));
	  	}
  	}
  } 

  /**
   * Return the last interaction point
   * @method getInteractionPoint
   * @return double[]
   */
  self.getInteractionPoint = function() {
    // mLastPoint = mPanel.toPanelPosition([location[0],location[1]]);
    return mLastLocation;
  };

  /**
   * Return the last interaction point deltas
   * @method getInteractionDeltas
   * @return double[]
   */
  self.getInteractionDeltas = function() {
    return [mDeltaX, mDeltaY];
  };

  /**
   * Return the last interaction zoom delta
   * @method getInteractionZoomDelta
   * @return double
   */
  self.getInteractionZoomDelta = function() {
    return mZoomDelta;
  };

  /**
   * Handler for mouse wheel event
   * @method handleMouseWheelEvent
   * @param event
   */
  self.handleMouseWheelEvent = function(event) {
	// number of fingers over screen	
	var nFingers = (typeof event.touches != "undefined")? event.touches.length:0;
	
	if(nFingers == 0) {  	
        // Prevent the browser from doing its default thing (scroll, zoom)
	    event.preventDefault(); 

		mZoomDelta = Math.max(-1, Math.min(1, (event.wheelDelta || -event.detail)));	
        mPanel.getController().invokeAction("OnZoom");
        
	    // all interactions are reported to View
	    mPanel.getController().reportInteractions();
	}
  };

  /**
   * Handler for pinch gesture (only supported by safari and iOS)
   * @method handlePinchGesture
   * @param event
   */
  self.handlePinchGesture = function(event) {  	
	if(event.timeStamp - mZoomLastTime > 100) { 
		// Prevent the browser from doing its default thing (scroll, zoom)
		event.preventDefault();
				
		if (event.scale < mLastZoom) {
		    // User moved fingers closer together
		    mZoomDelta = 1;                
		} else if (event.scale >= mLastZoom) {
		    // User moved fingers further apart
		    mZoomDelta = -1;
		}
		mLastZoom = event.scale;
		
		mPanel.getController().invokeAction("OnZoom");
		
		// all interactions are reported to View
		mPanel.getController().reportInteractions();
		
		mZoomLastTime = event.timeStamp;
	} 
	
  };
  
  /**
   * Handler for mouse move event
   * @method handleMouseMoveEvent
   * @param event
   * @param location
   */
  self.handleMouseMoveEvent = function(event) {
	// number of fingers over screen	
	var nFingers = (typeof event.touches != "undefined")? event.touches.length:1;
	
	if(nFingers == 1) {  	
	  	var currentX = mLastLocation[0];
	  	var currentY = mLastLocation[1];  	
	    mLastLocation = getEventLocation(event); 
	    
	    mDeltaX = mLastLocation[0] - currentX;
	    mDeltaY = mLastLocation[1] - currentY; 
	    
	    if (mMouseIsDown) { // Mouse is down
      	  mPanel.getController().invokeAction("OnDrag");
	      event.target.style.cursor = 'move';
	    }
	    else { // Mouse is up
	      event.target.style.cursor = 'default';
	    }
	
	}
	
	if (nFingers == 2) {
		if (event.timeStamp - mZoomLastTime > 25) { // time filter
			// Prevent the browser from doing its default thing (scroll, zoom)
			event.preventDefault();
			
			var e0 = event.touches[0]; var e1 = event.touches[1];
			var e0x = (e0.clientX || e0.x); var e0y = (e0.clientY || e0.y);
			var e1x = (e1.clientX || e1.x); var e1y = (e1.clientY || e1.y);
			if(mPinching) {
	    		var newDistance = [Math.abs(e0x-e1x),Math.abs(e0y-e1y)];
	    		var delta = newDistance[0]-mPinchInitDistance[0]; 
	    		if (delta<3 && delta>-3) { // minimum filter
	    			mZoomDelta = 0;
	    		} else {
	    			mZoomDelta = delta;
	    			mPinchInitDistance = newDistance;								
	    		}
				mPanel.getController().invokeAction("OnZoom");
			} else {
				mPinching = true;
	    		mPinchInitDistance = [Math.abs(e0x-e1x),Math.abs(e0y-e1y)];
	    		mZoomDelta = 0;
			}					
			mZoomLastTime = event.timeStamp;
		}
	} else {
		mPinching = false;
		mPinchInitDistance = [0,0];
	}		
	
    // all interactions are reported to View
    mPanel.getController().reportInteractions();
  };

  /**
   * Handler for mouse down event
   * @method handleMouseDownEvent
   * @param event
   * @param location
   */
  self.handleMouseDownEvent = function(event) {
	// number of fingers over screen	
	var nFingers = (typeof event.touches != "undefined")? event.touches.length:1;
	
	if(nFingers == 1) {  	
	  	mMouseIsDown = true;
	  	
	  	mLastLocation = getEventLocation(event);   		
    	mPanel.getController().invokeAction("OnPress");
	    self.pick(mLastLocation[0],mLastLocation[1]);
	    
	    // Prevent the browser from doing its default thing (scroll, zoom)
	    event.preventDefault(); 
	    
	    // all interactions are reported to View
	    mPanel.getController().reportInteractions();
	}    
	
	mPinching = false;
  };

  /**
   * Handler for mouse up event
   * @method handleMouseUpEvent
   * @param event
   * @param location
   */
  self.handleMouseUpEvent = function(event) {
	// number of fingers over screen	
	var nFingers = (typeof event.touches != "undefined")? event.touches.length:1;
	
	if(nFingers == 1) {  	  	
	    mMouseIsDown = false; 
	
	  	mLastLocation = getEventLocation(event);
	
      	mPanel.getController().invokeAction("OnRelease");       
	    event.target.style.cursor = 'default';
	
		if(event.timeStamp - mUpLastTime < 500) { // < 500 ms is dblclick
			mPanel.getController().invokeAction("OnDoubleClick");
		}
		mUpLastTime = event.timeStamp; 
	    
	    // all interactions are reported to View
	    mPanel.getController().reportInteractions();
	} 
	
	mPinching = false;   
  };  

  /**
   * Get location for the touch event 
   * @method getEventLocation 
   * @param {Object} e
   */
  function getEventLocation(e) {
  	var x, y;
	var box = mPanel.getGraphics().getBox();
	var oleft = box.left; // offset left in pixels
	var otop = box.top; // offset top in pixels
  	
    if ((typeof e.changedTouches != "undefined") && (e.changedTouches.length === 1)) {    	
      	x = e.changedTouches[0].pageX;
      	y = e.changedTouches[0].pageY;      
	} else {
	    x = e.x || e.clientX;
	    y = e.y || e.clientY;
	}  	  	

  	return [x-oleft, y-otop];
  }
 
  self.setHandler = function(type, handler) {
	  var graphics = mPanel.getGraphics();
	  var context = graphics.getEventContext();
	  switch (type) {
	    case "move" : 
		  context.addEventListener( 'mousemove', handler, false );
		  context.addEventListener( 'touchmove', handler, false );
	      break;
	    default : 
	    case "down" :
	      context.addEventListener( 'mousedown', handler, false );
	      context.addEventListener( 'touchstart', handler, false );
	      break;
	    case "up" :
	      context.addEventListener( 'mouseup', handler, false );
	      context.addEventListener( 'touchend', handler, false );
	      break;
	    case "mousewheel" :
	      context.addEventListener( 'mousewheel', handler, false );
		  break;	      
	    case "pinch" :
	      context.addEventListener( 'gesturestart', function(event) { mLastZoom = 1; }, false );
	      context.addEventListener( 'gesturechange', handler, false );
	      break;
	  }
	  return false;
	}

  /**
   * Picking in webgl
   */   
  self.pick = function (mousex, mousey) {
    // we want to read the pixel at x, y -- so we really need a rectangle from x-1,y-1 with witdth and height equal to 1
    var x = mousex - 1;
    var y = mousey - 1;
    var w = 1;
    var h = 1;
    
    var gl = mPanel.getGraphics().getContext();
    if(!gl) return;
        
    var data = new Uint8Array(w*h*4); // w * h * 4
    gl.readPixels(x, y, w, h, gl.RGBA, gl.UNSIGNED_BYTE, data);
    if(data.data) data=data.data;

// http://stackoverflow.com/questions/7156971/webgl-readpixels-is-always-returning-0-0-0-0
// http://webgldemos.thoughtsincomputation.com/engine_tests/picking

/*    
    var indices = null, index, i;
    for (i = 2; i < data.length; i += 4) {
      if (data[i] > 0) // check the 'blue' key (2)
      {
        index = decodeFromColor(data[i-2], data[i-1], data[i], data[i+1]);
        if (index)
        {
          if (!indices) indices = {};
          indices[index] = index;
        }
      }
    }
    self.context.bindFramebuffer(GL_FRAMEBUFFER, null);
    self.context.viewport(0,0,self.context.gl.viewportWidth,self.context.gl.viewportHeight);

    self.context.enable(GL_BLEND);
        
    if (indices) {
      var ind = [];
      for (i in indices) ind.push(indices[i]);
      
      if (ind[0]) return this.objects[ind];
    }
      
    return null;  	
*/  }
    
  return self;          
}  
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/**
 * Style object for 3D drawing
 * @class Style 
 * @constructor  
 */
EJSS_DRAWING3D.Style = {
    
  // ----------------------------------------------------
  // Static methods
  // ----------------------------------------------------

  /**
   * Copies one element into another
   */
  copyTo : function(source, dest) {
    dest.setClosedTop(source.getClosedTop());
    dest.setClosedBottom(source.getClosedBottom());
    dest.setClosedLeft(source.getClosedLeft());
    dest.setClosedRight(source.getClosedRight());
    dest.setDrawLines(source.getDrawLines());
    dest.setLineColor(source.getLineColor());
    dest.setLineWidth(source.getLineWidth());
    dest.setDrawFill(source.getDrawFill());
    dest.setTransparency(source.getTransparency());
    
	dest.setAmbientColor(source.getAmbientColor());
	dest.setFillColor(source.getFillColor());
	dest.setSpecularColor(source.getSpecularColor());
	dest.setAmbientReflection(source.getAmbientReflection());
	dest.setColorReflection(source.getColorReflection());
	dest.setSpecularReflection(source.getSpecularReflection());
	dest.setShininessVal(source.getShininessVal());
	
	dest.setPaletteFloor(source.getPaletteFloor());
	dest.setPaletteCeil(source.getPaletteCeil());
	dest.setPaletteFloorColor(source.getPaletteFloorColor());
	dest.setPaletteCeilColor(source.getPaletteCeilColor());	    
  },
  
};

/**
 * Creates a Style object for 3D drawing
 */
EJSS_DRAWING3D.style = function (mName) {
  var self = {};
  
  var mClosedTop = true;
  var mClosedBottom = true;
  var mClosedLeft = true;
  var mClosedRight = true;
  var mLineColor = [0,0,0];
  var mLineWidth = 0.5;     
  
  // type of material
  var mAmbientColor = [0.0,0.0,0.0];
  var mFillColor = [0.0,0.0,0.0];
  var mSpecularColor = [1.0,1.0,1.0];
  var mAmbientReflection = 1.0;
  var mColorReflection = 1.0;
  var mSpecularReflection = 1.0;
  var mShininessVal = 10;

  // color palette of material
  var mPaletteFloor; // e.g. [0.0,0.0,-5.0];
  var mPaletteCeil;  // e.g. [0.0,0.0,5.0];
  var mPaletteFloorColor = [1.0,0.0,-1.0];
  var mPaletteCeilColor = [-1.0,0.0,1.0];

  var mDrawLines = false;
  var mDrawFill = true;
  var mTransparency = 0.0;  
  var mProjChangeListener;       
  var mMeshChangeListener;       

  /**
   * Set a listener that will be called whenever there are style changes.
   * I.e. a call to listener("change"); will be issued
   */
  self.setProjChangeListener = function(listener) {
    mProjChangeListener = listener;
  };

  self.setMeshChangeListener = function(listener) {
    mMeshChangeListener = listener;
  };

  //---------------------------------
  // Closed
  //---------------------------------

  self.setClosedTop = function(closed) {
    if (closed!=mClosedTop) {
      mClosedTop = closed;
      if (mMeshChangeListener) mMeshChangeListener("closedtop");
    }
  };
  
  self.getClosedTop = function() { 
    return mClosedTop; 
  };

  self.setClosedBottom = function(closed) {
    if (closed!=mClosedBottom) {
      mClosedBottom = closed;
      if (mMeshChangeListener) mMeshChangeListener("closedbottom");
    }
  };
  
  self.getClosedBottom = function() { 
    return mClosedBottom; 
  };
  
  self.setClosedLeft = function(closed) {
    if (closed!=mClosedLeft) {
      mClosedLeft = closed;
      if (mMeshChangeListener) mMeshChangeListener("closedleft");
    }
  };
  
  self.getClosedLeft = function() { 
    return mClosedLeft; 
  };
  
  self.setClosedRight = function(closed) {
    if (closed!=mClosedRight) {
      mClosedRight = closed;
      if (mMeshChangeListener) mMeshChangeListener("closedright");
    }
  };
  
  self.getClosedRight = function() { 
    return mClosedRight; 
  };
  
  //---------------------------------
  // lines
  //---------------------------------

  /**
   * Whether to draw the lines of the element
   */
  self.setDrawLines = function(draw) {
    if (draw!=mDrawLines) {
      mDrawLines = draw;
      if (mProjChangeListener) mProjChangeListener("drawlines");
    }
  };
  
  /**
   * Get the draw lines flag
   */
  self.getDrawLines = function() { 
    return mDrawLines; 
  };
  
  /**
   * Set the line color of the element
   * @param color a stroke style
   */
  self.setLineColor = function(color) { 
    if (typeof color === "string") color = EJSS_TOOLS.DisplayColors.getArrayColor(color);
    if (!EJSS_TOOLS.compareArrays(color,mLineColor)) {
      mLineColor = color.slice(); 
      if (mProjChangeListener) mProjChangeListener("linecolor");
    }
    return self;
  };
    
  /**
   * Get the line color
   */
  self.getLineColor = function() { 
    return mLineColor; 
  };

  /**
   * Set the line width of the element
   * @param width a stroke width (may be double, such as 0.5, the default)
   */
  self.setLineWidth = function(width) { 
    if (width!=mLineWidth) {
      mLineWidth = width; 
      if (mProjChangeListener) mProjChangeListener("linewidth");
    }
  };

  /**
   * Get the line width
   */
  self.getLineWidth = function() { return mLineWidth; };
  
  //---------------------------------
  // interior fill
  //---------------------------------

  /**
   * Whether to fill the interior of the element
   */
  self.setDrawFill = function(draw) {
    if (draw!=mDrawFill) {
      mDrawFill = draw;
      if (mProjChangeListener) mProjChangeListener("drawfill");
    }
  };
  
  /**
   * Get the draw fill flag
   */
  self.getDrawFill = function() { 
    return mDrawFill; 
  };

  /**
   * Set the fill color of the element
   * @param color a fill style
   */
  self.setFillColor = function(color) {
    if (typeof color === "string") color = EJSS_TOOLS.DisplayColors.getArrayColor(color);
    if (!EJSS_TOOLS.compareArrays(color,mFillColor)) {
      mFillColor = color.slice(); 
      if (mProjChangeListener) mProjChangeListener("fillcolor");
    }
  };

  /**
   * Get the fill color
   */
  self.getFillColor = function() { 
    return mFillColor; 
  };
  
  /**
   * Set the Ambient color of the element
   * @param color ambient color
   */
  self.setAmbientColor = function(color) {
    if (typeof color === "string") color = EJSS_TOOLS.DisplayColors.getArrayColor(color);
    if (!EJSS_TOOLS.compareArrays(color,mAmbientColor)) {
      mAmbientColor = color.slice(); 
      if (mProjChangeListener) mProjChangeListener("ambientcolor");
    }
  };
  
  /**
   * Get the Ambient color
   */
  self.getAmbientColor = function() { 
    return mAmbientColor; 
  };

  /**
   * Set the Specular color of the element
   * @param color specular color
   */
  self.setSpecularColor = function(color) {
    if (typeof color === "string") color = EJSS_TOOLS.DisplayColors.getArrayColor(color);
    if (!EJSS_TOOLS.compareArrays(color,mSpecularColor)) {
      mSpecularColor = color.slice(); 
      if (mProjChangeListener) mProjChangeListener("specularcolor");
    }
  };
  
  /**
   * Get the Specular color
   */
  self.getSpecularColor = function() { 
    return mSpecularColor; 
  };

  /**
   * Set the Ambient Reflection of the element
   * @param k coefficient
   */
  self.setAmbientReflection = function(k) {
    if (mAmbientReflection != k) {
      mAmbientReflection = k; 
      if (mProjChangeListener) mProjChangeListener("ambientreflection");
    }
  };
  
  /**
   * Get the Ambient reflection
   */
  self.getAmbientReflection = function() { 
    return mAmbientReflection; 
  };

  /**
   * Set the Color Reflection of the element
   * @param k coefficient
   */
  self.setColorReflection = function(k) {
    if (mColorReflection != k) {
      mColorReflection = k; 
      if (mProjChangeListener) mProjChangeListener("colorreflection");
    }
  };
  
  /**
   * Get the Color reflection
   */
  self.getColorReflection = function() { 
    return mColorReflection; 
  };

  /**
   * Set the Specular Reflection of the element
   * @param k coefficient
   */
  self.setSpecularReflection = function(k) {
    if (mSpecularReflection != k) {
      mSpecularReflection = k; 
      if (mProjChangeListener) mProjChangeListener("specularreflection");
    }
  };
  
  /**
   * Get the Specular reflection
   */
  self.getSpecularReflection = function() { 
    return mSpecularReflection; 
  };

  /**
   * Set the shininess of the element
   * @param shininess coefficient
   */
  self.setShininessVal = function(shininess) {
    if (mShininessVal != shininess) {
      mShininessVal = shininess; 
      if (mProjChangeListener) mProjChangeListener("shininess");
    }
  };
  
  /**
   * Get the shininess 
   */
  self.getShininessVal = function() { 
    return mShininessVal; 
  };

  /**
   * Set transparency
   * @param transparency
   */
  self.setTransparency = function(transparency) {
    if (transparency!=mTransparency) {
      mTransparency = transparency; 
      if (mProjChangeListener) mProjChangeListener("transparency");
    }
  };
  
  /**
   * Get transparency
   */
  self.getTransparency = function() { 
    return mTransparency; 
  };

  /**
   * Set the palette floor of the element
   * @param floor palette floor
   */
  self.setPaletteFloor = function(floor) {
    if (floor && !EJSS_TOOLS.compareArrays(floor,mPaletteFloor)) {
      mPaletteFloor = floor.slice(); 
      if (mProjChangeListener) mProjChangeListener("palettefloor");
    }
  };

  /**
   * Get the palette floor
   */
  self.getPaletteFloor = function() { 
    return mPaletteFloor; 
  };

  /**
   * Set the palette ceil of the element
   * @param ceil palette ceil
   */
  self.setPaletteCeil = function(ceil) {
    if (ceil && !EJSS_TOOLS.compareArrays(ceil,mPaletteCeil)) {
      mPaletteCeil = ceil.slice(); 
      if (mProjChangeListener) mProjChangeListener("paletteceil");
    }
  };

  /**
   * Get the palette ceil
   */
  self.getPaletteCeil = function() { 
    return mPaletteCeil; 
  };

  /**
   * Set the palette floor color of the element
   * @param color palette floor color
   */
  self.setPaletteFloorColor = function(color) {
    if (!EJSS_TOOLS.compareArrays(color,mPaletteFloorColor)) {
      mPaletteFloorColor = color.slice(); 
      if (mProjChangeListener) mProjChangeListener("palettefloorcolor");
    }
  };

  /**
   * Get the palette floor color
   */
  self.getPaletteFloorColor = function() { 
    return mPaletteFloorColor; 
  };

  /**
   * Set the palette ceil color of the element
   * @param color palette ceil color
   */
  self.setPaletteCeilColor = function(color) {
    if (!EJSS_TOOLS.compareArrays(color,mPaletteCeilColor)) {
      mPaletteCeilColor = color.slice(); 
      if (mProjChangeListener) mProjChangeListener("paletteceilcolor");
    }
  };

  /**
   * Get the palette ceil color
   */
  self.getPaletteCeilColor = function() { 
    return mPaletteCeilColor; 
  };


  //---------------------------------
  // final initialization
  //---------------------------------
  
  return self;
};

/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/**
 * Transformation object for 3D elements.
 * A transformation is an array of 7 numbers indicating
 *  [ rotation angle (in degrees), 
 *    x,y,z coordinates of the rotation axis, 
 *    x,y,z coordinates of the origin of the rotation ] 
 * @class Transformation 
 * @constructor  
 */
EJSS_DRAWING3D.Transformation = {
    /**
     * Registers properties in a ControlElement
     * @method registerProperties
     * @param element The element with the properties
     * @param controller A ControlElement that becomes the element controller
     */
    registerProperties : function(element,controller) {
      element.setController(controller);
      controller.registerProperty("Parent", element.setParent, element.getParent); 
      controller.registerProperty("Axis", element.setAxis, element.getAxis); 
      controller.registerProperty("Angle", element.setAngle, element.getAngle); 
      controller.registerProperty("Origin", element.setOrigin, element.getOrigin); 
    }
};

/**
 * Creates a Rotation around the X axis
 */
EJSS_DRAWING3D.rotationX = function (mName) {
  var self = EJSS_DRAWING3D.transformation();
  self.setAxis([1,0,0]);
  return self;  
}
  
/**
 * Creates a Rotation around the Y axis
 */
EJSS_DRAWING3D.rotationY = function (mName) {
  var self = EJSS_DRAWING3D.transformation();
  self.setAxis([0,1,0]);
  return self;  
}

/**
 * Creates a Rotation around the Z axis
 */
EJSS_DRAWING3D.rotationZ = function (mName) {
  var self = EJSS_DRAWING3D.transformation();
  self.setAxis([0,0,1]);
  return self;  
}

/**
 * Creates a Transformation object for 3D element
 */
EJSS_DRAWING3D.transformation = function (mName) {
  var self = {};
  
  var mParent = null;
  var mArray = [0,0,0,1,0,0,0];

  var mController = { 				// dummy controller object
      propertiesChanged : function() {},
      invokeAction : function() {}
  };
  
  self.getArray = function() {
    return mArray;
  };
  
  self.setChanged = function(changed) {
    if (changed) mParent.setProjChanged(true);
  };

  /**
   * Set the parent
   * @method setParent
   * @param parent Element
   */
  self.setParent = function(parent) {
    if (mParent) {
      mParent.removeExtraTransformation(self);
    }
    mParent = parent;
    if (mParent) {
      mParent.addExtraTransformation(self);
    }
  };

  self.getParent = function() {
    return mParent;
  };
  
  //---------------------------------
  // Customization
  //---------------------------------

  /**
   * Set the rotation angle
   * @method setAngle
   * @param angle int the angle in degrees
   */
  self.setAngle = function(angle) {
    mArray[0] = angle;
  };

  self.getAngle = function() {
    return mArray[0];
  };

  /**
   * Set the rotation axis
   * @method setAxis
   * @param axis double[3] the coordinates of the axis
   */
  self.setAxis = function(axis) {
    mArray[1] = axis[0];
    mArray[2] = axis[1];
    mArray[3] = axis[2];
  };

  self.getAxis = function() {
    return mArray.slice(1,4);
  };
  
  /**
   * Set the rotation origin
   * @method setOrigin
   * @param origin double[3] the coordinates of the origin
   */
  self.setOrigin = function(origin) {
    mArray[4] = origin[0];
    mArray[5] = origin[1];
    mArray[6] = origin[2];
  };

  self.getOrigin = function() {
    return mArray.slice(5);
  };
  
  // ----------------------------------------------------
  // Properties
  // ----------------------------------------------------

  /**
   * Returns the controller object
   * @method getController
   * @return Controller
   */
  self.getController = function () {
    return mController;
  };

  /**
   * Set the controller
   * @method setController
   * @param Controller
   */
  self.setController = function (controller) {
    mController = controller;
  };
  
  /**
   * Registers properties in a ControlElement
   * @method registerProperties
   * @param controller A ControlElement that becomes the element controller
   */
  self.registerProperties = function(controller) {
    EJSS_DRAWING3D.Transformation.registerProperties(self,controller);
  };
  
  //---------------------------------
  // final initialization
  //---------------------------------
  
  return self;
};

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL custom
 */

EJSS_WEBGLGRAPHICS.analyticCurve = function(mGL, mElement) {         
      evaluate = function(expression, v1, p1, v2, p2) {
       var vblevalue = {};
       vblevalue[v1] = p1;
	   vblevalue[v2] = p2;
       return expression.evaluate(vblevalue);
      };

	  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName(), 
	  	{coords:false, normals:false, triangles:false, lines:true, colors:false});

	  var numRows = mElement.getNumPoints();
	  var minRows = mElement.getMinValue();
	  var maxRows = mElement.getMaxValue();
	  var d1 = Math.abs((maxRows - minRows) / (numRows - 1));
	  
	  var exprX = mElement.getExpressionX();		
	  var exprY = mElement.getExpressionY();
	  var exprZ = mElement.getExpressionZ();
	  
	  var var1 = mElement.getVariable();
	  
	  var point1 = minRows;	  					 
	  for (var i = 0; i < numRows; i++) {
		// expresion X
		var x = evaluate(exprX, var1, point1);
		// expresion Y
		var y = evaluate(exprY, var1, point1); 
		// expresion Z		 
		var z = evaluate(exprZ, var1, point1);

	  	mesh.vertices.push([x,y,z]);
	  	if(i!=0) mesh.lines.push([i-1, i]);

		point1 = point1 + d1;
	  }
	  
	  mesh.compile();
	  	  	  
	  mGL.addElement(mElement.getName() + ".mesh", mesh);

      return mesh;
}

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL custom
 */

EJSS_WEBGLGRAPHICS.analyticSurface = function(mGL, mElement) {   
    
    createSurfaceBuffers = function(xpoints, ypoints, data3D, mesh){
        for (var i = 0; i < xpoints - 1; i++) {
            for (var j = 0; j < ypoints - 1; j++) {
                // Create surface vertices.
                var rawP1 = data3D[j + (i * ypoints)];
                var rawP2 = data3D[j + (i * ypoints) + ypoints];
                var rawP3 = data3D[j + (i * ypoints) + ypoints + 1];
                var rawP4 = data3D[j + (i * ypoints) + 1];
                
                mesh.vertices.push(rawP1);
                mesh.vertices.push(rawP2);
                mesh.vertices.push(rawP3);
                mesh.vertices.push(rawP4);                
            }
        }
                
        var numQuads = ((xpoints - 1) * (ypoints - 1)) / 2;        
        for (var i = 0; i < (numQuads * 8); i += 4) {
            mesh.triangles.push([i, i+1, i+2]);
            mesh.triangles.push([i, i+2, i+3]);

            mesh.lines.push([i,i+1]);             
            mesh.lines.push([i,i+3]);             
            mesh.lines.push([i+2,i+3]);             
            mesh.lines.push([i+2,i+1]);             			

            mesh.coords.push([0,0]);
            mesh.coords.push([0,1]);
            mesh.coords.push([1,0]);
            mesh.coords.push([1,1]);            
        }        
    };
  
  evaluate = function(expression, v1, p1, v2, p2) {
    var vblevalue = {};
    vblevalue[v1] = p1;
	vblevalue[v2] = p2;
    return expression.evaluate(vblevalue);
  };

	  var numRows = mElement.getNumPoints1();
	  var minRows = mElement.getMinValue1();
	  var maxRows = mElement.getMaxValue1();
	  var numCols = mElement.getNumPoints2();
	  var minCols = mElement.getMinValue2();
	  var maxCols = mElement.getMaxValue2();	 
	  var d1 = Math.abs((maxRows - minRows) / (numRows - 1));
	  var d2 = Math.abs((maxCols - minCols) / (numCols - 1));
	  
	  var exprX = mElement.getExpressionX();		
	  var exprY = mElement.getExpressionY();
	  var exprZ = mElement.getExpressionZ();
	  
	  var var1 = mElement.getVariable1();
	  var var2 = mElement.getVariable2();
	  
	  var data3ds = new Array();
	  var index = 0;
	  var point1 = minRows;	  					 
	  for (var i = 0; i < numRows; i++) {
		var point2 = minCols;								
		for (var j = 0; j < numCols; j++) {
			// expresion X
			var x = evaluate(exprX, var1, point1, var2, point2);
			// expresion Y
			var y = evaluate(exprY, var1, point1, var2, point2); 
			// expresion Z		 
			var z = evaluate(exprZ, var1, point1, var2, point2);
			data3ds[index] = [x,y,z];
			index++;			
			point2 = point2 + d2;
		}				
		point1 = point1 + d1;
	  }
	  	  	  
	  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName());
	  
	  createSurfaceBuffers(numRows, numCols, data3ds, mesh);

	  // compute normals
	  mesh.computeNormals();
	  
	  mesh.compile();
	  	  	  
	  mGL.addElement(mElement.getName() + ".mesh", mesh);

      return mesh;
}

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL arrow
 */
EJSS_WEBGLGRAPHICS.arrow = function(mGL, mElement) {
  	  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName(), 
	  	{coords:false, normals:true, triangles:true, lines:false, colors:false});
  	  	
  	  // end point
  	  var endpoint = new Vector(mElement.getSize());
  	  var lenendpoint = endpoint.length();
  	  
  	  // resolution used to arrow head and segment
	  var resolutionU = mElement.getResolutionU();
	  var headWidth = mElement.getHeadWidth();
	  var headHeight = mElement.getHeadHeight();
	  
	  // generate vertex for head (head point at (0,0,0))
	  var hvertices3D = [new Vector(0, 0, 0)];
	  var hradius = (lenendpoint / headWidth); 	  
	  for (var x = 0; x < resolutionU; x++ ) {
		var u = x / (resolutionU - 1);

		var xpos = hradius * Math.sin( u * Math.PI * 2 );
		var ypos = hradius * Math.cos( u * Math.PI * 2 );
		var zpos = (- lenendpoint / headHeight);   

		hvertices3D.push(new Vector(xpos, ypos, zpos));
	  }

	  // generate vertex for segment
	  var svertices3D = [];
	  var sradius = mElement.getLineWidth() / 2;
	  for (var x = 0; x < resolutionU; x++ ) {
		var u = x / (resolutionU - 1);
		var xpos = sradius * Math.sin( u * Math.PI * 2 );
		var ypos = sradius * Math.cos( u * Math.PI * 2 );
		var zpos = (- lenendpoint / headHeight);   

		svertices3D.push(new Vector([xpos, ypos, zpos])); // top vertex
		svertices3D.push(new Vector([xpos, ypos, 0])); // bottom vertex
	  }
	  	  
	  // rotate and translate head
	  var cross = endpoint.cross(new Vector(0,0,1));
	  if(cross.length()==0) cross.y = 1; // if endoint in axis Z, get axis Y to rotate 
	  // console.log("cross:" + cross.x + " " + cross.y + " " + cross.z + " ");	  
	  var angle = endpoint.angle(new Vector(0,0,1));
	  // console.log("angle:" + angle);
	  var result = new Matrix();
	  Matrix.rotate(-angle,cross.x,cross.y,cross.z,0,0,0,result);
	  for(var i=0; i<hvertices3D.length; i++) { // head vertex
	  	var vector = result.transformPoint(hvertices3D[i]);
	  	mesh.vertices.push(vector.add(endpoint).toArray());	  		
	  }
	  for(var i=0; i<svertices3D.length; i++) { // segment vertex
	  	var vector = result.transformPoint(svertices3D[i]);
	  	if (i % 2 == 0) {
	  		// add enpoint to translate	  	
	  		mesh.vertices.push(vector.add(endpoint).toArray());	  		
	  	} else {
	  		mesh.vertices.push(vector.toArray());
	  	}	  		  		
	  }
	  	  
	  // create surface triangles for head
	  for (var x = 1; x < resolutionU; x++ ) {
	    var first = x;		    
	    var second = x + 1;		
        mesh.triangles.push([0, first, second]);
	  }		
			  
	  // create surface triangles for segment
	  for (var x = 0; x < resolutionU - 1; x++ ) {
	    var first = 2*x + hvertices3D.length;		    
	    var fourth = 2*x + 2 + hvertices3D.length;
	    var second = 2*x + 1 + hvertices3D.length;
	    var third = 2*x + 3 + hvertices3D.length;
			
        mesh.triangles.push([third, second, first]);      
        mesh.triangles.push([fourth, third, first]);
	  }		
			  
	  // bottom cap
 	  mesh.vertices.push([0, 0, 0]);
      for (var i = 0; i < resolutionU - 1; i++) {
      	var first = 2*i+1 + hvertices3D.length;;
      	var second = 2*i+3 + hvertices3D.length;;
      	var third = mesh.vertices.length - 1;
       	mesh.triangles.push([first, second, third]);
      }        
			  
      mesh.computeNormals();	  
	        
	  mesh.compile();	
	
	  mGL.addElement(mElement.getName() + ".mesh", mesh);
          
      return mesh;
}

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL basic
 */

EJSS_WEBGLGRAPHICS.basic = function(mGL, mElement) {   
	  mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName());
	  // set values
  	  mesh.setVertices(mElement.getVertices());	  
	  mesh.setCoords(mElement.getVertices());
	  mesh.setTriangles(mElement.getTriangles());
      mesh.setColors(mElement.getColors());

	  // compute normals
	  if (mElement.getNormals().length != 0) {
	  	mesh.setNormals(mElement.getNormals());
	  } else {
	  	mesh.computeNormals();
	  };
  
	  // compute wire
	  mesh.computeWireframe();
	  
	  mesh.compile();
	  
	  mGL.addElement(mElement.getName() + ".mesh", mesh);
      
      return mesh;
}

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL box
 */
EJSS_WEBGLGRAPHICS.box = function(mGL, mElement) {
	  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName());
	  
	  var octant = [[-1,-1,-1], [1,-1,-1], [-1,1,-1], [1,1,-1], [-1,-1,1], [1,-1,1], [-1,1,1], [1,1,1]];
	  
	  var coords = [[0,0],[1,0],[0,1],[1,1]];

	  var normals = [
	   [-1, 0, 0], // -x
	   [+1, 0, 0], // +x
	   [0, -1, 0], // -y
	   [0, +1, 0], // +y
	   [0, 0, -1], // -z
	   [0, 0, +1]  // +z
	  ];
	  
	  var boxData = [
	   [0, 2, 4, 6], // -x
	   [1, 3, 5, 7], // +x
	   [0, 4, 1, 5], // -y
	   [2, 6, 3, 7], // +y
	   [0, 2, 1, 3], // -z
	   [4, 6, 5, 7]  // +z
	  ];
	
  	  var isClosedTop = mElement.getStyle().getClosedTop();
  	  var isClosedBottom = mElement.getStyle().getClosedBottom();
	  var isClosedRight = mElement.getStyle().getClosedRight();
	  var isClosedLeft = mElement.getStyle().getClosedLeft();	
	  var reduceZby = mElement.getReduceZby();
	
	  // build box
	  for (var i = 0; i < boxData.length; i++) {
	    var data = boxData[i], v = i * 4;
	    for (var j = 0; j < 4; j++) {
	      var d = data[j];
	      if(d==6 || d==7)
	      	mesh.vertices.push([octant[d][0], octant[d][1], octant[d][2] - (2*reduceZby)]);
	      else
      	  	mesh.vertices.push(octant[d]);
      	  mesh.coords.push(coords[j]);
      	  mesh.normals.push(normals[i]);
	    }
	    if(!(i == 2 && !isClosedLeft) &&
	       !(i == 3 && !isClosedRight) &&
	       !(i == 5 && !isClosedTop) &&
	       !(i == 4 && !isClosedBottom)) {
	    	mesh.triangles.push([v, v + 1, v + 2]);
	    	mesh.triangles.push([v + 2, v + 1, v]);    
	    	mesh.triangles.push([v + 2, v + 1, v + 3]);    
	    	mesh.triangles.push([v + 3, v + 1, v + 2]);
	    }
	    mesh.lines.push([v,v+1]);
	    mesh.lines.push([v,v+2]);
	    mesh.lines.push([v+2,v+3]);
	    mesh.lines.push([v+1,v+3]);
	  }
	  	  		  	 
	  mesh.compile();
	
	  mGL.addElement(mElement.getName() + ".mesh", mesh);
         
      return mesh;
}

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL cylinder
 */
EJSS_WEBGLGRAPHICS.cylinder = function(mGL, mElement) {
  	  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName());
  	  	
	  var resolutionV = mElement.getResolutionV();
	  var resolutionU = mElement.getResolutionU() + 1;
	  
	  var radiusTop = mElement.getTopRadius();
	  var radiusBottom = mElement.getBottomRadius();      
	  var mMinAngleU = mElement.getMinAngleU();
	  var mMaxAngleU = mElement.getMaxAngleU();
      
	  // generate vertex for body
      var maxNumber = 0;
      var minNumber = 0;
	  for (var z = 0; z < resolutionV; z ++ ) {	
		var v = z / (resolutionV - 1);
		var radius = v * ( radiusBottom - radiusTop ) + radiusTop;

	  	minNumber = Math.floor(mMinAngleU * resolutionU / 360); 
	  	maxNumber = Math.floor(mMaxAngleU * resolutionU / 360);		  	
		for (var x = minNumber; x < maxNumber; x ++ ) {
			var u = x / (resolutionU - 1);

			var xpos = radius * Math.sin( u * Math.PI * 2 );
			var ypos = radius * Math.cos( u * Math.PI * 2 );
			var zpos = - v * 2 + 1;

			mesh.vertices.push([xpos, ypos, zpos]);
	        mesh.coords.push([u,v]);      			
		}
	  }
	  
	  // build central axis
	  for (var z = 0; z < resolutionV; z ++ ) {	
	      var zpos = - 1 + (z * (2 / (resolutionV - 1))); // proportional in axis
	      	      
		  mesh.vertices.push([0, 0, zpos]);
		  mesh.coords.push([1, z / (resolutionV - 1)]);		
  	  }	  
	  
	  // create surface triangles
	  var sizeU = (maxNumber - minNumber);
      for (var z = 0; z < resolutionV - 1; z++) {			
		  for (var x = 0; x < sizeU - 1; x++ ) {
		    var first = (z * sizeU) + x;		    
		    var second = ((z + 1) * sizeU) + x;
		    var third = ((z + 1) * sizeU) + x + 1;
		    var fourth = (z * sizeU) + x + 1;
			
	        mesh.triangles.push([third, second, first]);
	        mesh.triangles.push([fourth, third, first]);      
	        
		    mesh.lines.push([third, second]);
		    mesh.lines.push([fourth, third]);
		    mesh.lines.push([first, fourth]);
		  }
		
		  // last close sphera
	      if((mMinAngleU != 0) || (mMaxAngleU != 360)) { // quesito 		      		  			  	
		    
			// close left and right
	        var firstAxis = mesh.vertices.length - (z + 1); // points central axis
			if(mElement.getStyle().getClosedLeft()) {
		    	mesh.triangles.push([third, fourth, firstAxis - 1]);
		    	mesh.triangles.push([fourth, firstAxis, firstAxis - 1]);

		    	mesh.lines.push([third, firstAxis - 1]);
		    	mesh.lines.push([fourth, firstAxis]);
		    	mesh.lines.push([firstAxis, firstAxis - 1]);
		    }
		    if(mElement.getStyle().getClosedRight()) {
				// points first latitude
		    	var firstMin = z * sizeU;
				var secondMin = (z + 1) * sizeU;
		    	mesh.triangles.push([firstAxis - 1, firstMin, secondMin]);
		    	mesh.triangles.push([firstAxis - 1, firstAxis, firstMin]);

		    	mesh.lines.push([secondMin, firstMin]);
		    	mesh.lines.push([secondMin, firstAxis - 1]);
		    	mesh.lines.push([firstMin, firstAxis]);
		    	mesh.lines.push([firstAxis, firstAxis - 1]);
		    }			    
		  }			      			
	  }
		
	  // top cap
  	  var isClosedTop = mElement.getStyle().getClosedTop();
	  if ( isClosedTop && radiusTop > 0 ) {
		mesh.vertices.push([0, 0, 1]);		
        for (var i = 0; i < sizeU - 1; i++) {
           	mesh.triangles.push([mesh.vertices.length - 1, i+1, i]);
	        mesh.coords.push([1, (i / (sizeU - 2))]);
	               	
		    mesh.lines.push([mesh.vertices.length - 1, i]);
		    mesh.lines.push([i, i+1]);	               	
        }        
	  }

	  // bottom cap
  	  var isClosedBottom = mElement.getStyle().getClosedBottom();
	  if ( isClosedBottom && radiusBottom > 0 ) {
		mesh.vertices.push([0, 0, -1]);
        for (var i = (resolutionV-1)*sizeU; i < sizeU*resolutionV; i++) {
           	mesh.triangles.push([i, i+1, mesh.vertices.length - 1]);
           	mesh.coords.push([1, (i / (sizeU - 2))]);

		    mesh.lines.push([mesh.vertices.length - 1, i]);
		    mesh.lines.push([i, i+1]);	               	
        }        
	  }			  
	  
	  // compute normals
	  if((mMinAngleU == 0) && (mMaxAngleU == 360)) 
	  	mesh.normals = mesh.vertices;
	  else
	    mesh.computeNormals();	  
	        
	  mesh.compile();	
	
	  mGL.addElement(mElement.getName() + ".mesh", mesh);
        
     return mesh;
}

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL ellipsoid
 */
EJSS_WEBGLGRAPHICS.ellipsoid = function(mGL, mElement) {
	  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName());
	  	  
	  var indexer = EJSS_WEBGLGRAPHICS.indexer();
	
	  // generate vertex for body
	  var latitudeBands = mElement.getResolutionV();  	
	  var longitudeBands = mElement.getResolutionU();
	  for (var latNumber = 0; latNumber <= latitudeBands; latNumber++) {
		  var theta = latNumber * Math.PI / latitudeBands;
		  var sinTheta = Math.sin(theta);
		  var cosTheta = Math.cos(theta);
		
		  for (var longNumber = 0; longNumber <= longitudeBands; longNumber++) {
		    var phi = longNumber * 2 * Math.PI / longitudeBands;
		    var sinPhi = Math.sin(phi);
		    var cosPhi = Math.cos(phi);
		
		    var y = cosPhi * sinTheta;
		    var z = -cosTheta;
		    var x = -sinPhi * sinTheta;
		    var u = 1 - (longNumber / longitudeBands);
		    var v = (latNumber / latitudeBands);
		
			mesh.vertices.push([-x, y, z]);
			mesh.coords.push([u,v]);
		  }		  	
	  }
	  	
      // generate vertex for central axis
	  for (var latNumber = 0; latNumber <= latitudeBands; latNumber++) {		
	      var v =  latNumber / latitudeBands;
		  var theta = latNumber * Math.PI / latitudeBands;	      
	      var z = -Math.cos(theta);
	      // var z = - 1 + (latNumber * (2 / latitudeBands)); // proportional in axis
		
	      mesh.vertices.push([0, 0, z]);
		  mesh.coords.push([0,v]);
	  }	  
  		
	  // limits for sphera
	  var mMinAngleU = mElement.getMinAngleU();
	  var mMaxAngleU = mElement.getMaxAngleU();
	  var mMinAngleV = mElement.getMinAngleV() + 90;
	  var mMaxAngleV = mElement.getMaxAngleV() + 90;
	  
	  // build triangles
	  var latNumber;
  	  var minLatNumber = Math.floor(mMinAngleV * latitudeBands / 180); 
  	  var maxLatNumber = Math.floor(mMaxAngleV * latitudeBands / 180);
	  for (latNumber = minLatNumber; latNumber < maxLatNumber; latNumber++) {
	  	  var longNumber;	  	  
	  	  // close sphera
	  	  var minLongNumber = Math.floor(mMinAngleU * longitudeBands / 360); 
	  	  var maxLongNumber = Math.floor(mMaxAngleU * longitudeBands / 360);
	  	  var first, second;
		  for (longNumber = minLongNumber; longNumber < maxLongNumber; longNumber++) {
		    first = (latNumber * (longitudeBands + 1)) + longNumber;
		    second = first + longitudeBands + 1;
		    mesh.triangles.push([first, second, first + 1]);
		    mesh.triangles.push([second, second + 1, first + 1]);
		    
		    mesh.lines.push([first, second]);
		    mesh.lines.push([first, first+1]);
		    mesh.lines.push([second, second+1]);
		  }
		  			  
		  // last close sphera
	      if((mMinAngleU != 0) || (mMaxAngleU != 360)) { 	// quesito					  	  	
			// close left and right
	        var firstAxis = mesh.vertices.length - (latitudeBands - latNumber + 1); // points central axis
			if(mElement.getStyle().getClosedLeft()) {
		    	mesh.triangles.push([first + 1, second + 1, firstAxis]);
		    	mesh.triangles.push([second + 1, firstAxis + 1, firstAxis]);

		    	mesh.lines.push([first+1, second+1]);
		    	mesh.lines.push([first+1, firstAxis]);
		    	mesh.lines.push([second+1, firstAxis+1]);
		    	mesh.lines.push([firstAxis, firstAxis+1]);
		    }
		    if(mElement.getStyle().getClosedRight()) {
				// points first latitude
		    	var firstMin = (latNumber * (longitudeBands + 1)) + minLongNumber;
				var secondMin = firstMin + longitudeBands + 1;
		    	mesh.triangles.push([firstAxis, secondMin, firstMin]);
		    	mesh.triangles.push([firstAxis, firstAxis + 1, secondMin]);

		    	mesh.lines.push([firstMin, secondMin]);
		    	mesh.lines.push([firstMin, firstAxis]);
		    	mesh.lines.push([secondMin, firstAxis+1]);
		    	mesh.lines.push([firstAxis, firstAxis+1]);
		    }			    
		  }			      	
	  }
	  
	  // close bottom
      if(mElement.getStyle().getClosedBottom() && (mMinAngleV > 0)) {      	  
		  var first, longNumber, axis;
		  for (longNumber = minLongNumber; longNumber < maxLongNumber; longNumber++) {
		    first = (minLatNumber * (longitudeBands + 1)) + longNumber;
		    axis = mesh.vertices.length - (latitudeBands - minLatNumber + 1);
		    mesh.triangles.push([first + 1, axis, first]);
		    
		    mesh.lines.push([first+1,axis]);
		    mesh.lines.push([first,first+1]);
		  }      	
      }
      
	  // close top
      if(mElement.getStyle().getClosedTop() && (mMaxAngleV < 180)) {
      	  var first, longNumber, axis;
		  for (longNumber = minLongNumber; longNumber < maxLongNumber; longNumber++) {
		    first = (maxLatNumber * (longitudeBands + 1)) + longNumber;
		    axis = mesh.vertices.length - (latitudeBands - maxLatNumber + 1);
		    mesh.triangles.push([first, axis, first + 1]);

		    mesh.lines.push([first+1,axis]);
		    mesh.lines.push([first,first+1]);
		  }      	      	
      }		      
	    	  
	  // compute normals
	  if((mMinAngleU == 0) && (mMaxAngleU == 360)) 
	  	mesh.normals = mesh.vertices;
	  else
	    mesh.computeNormals();	  
  	 
	  // compute wire
	  // mesh.computeWireframe();
	  
	  mesh.compile();
	  
	  mGL.addElement(mElement.getName() + ".mesh", mesh);
   
      return mesh;
}

// Represents a 4x4 matrix stored in row-major order that uses Float32Arrays
// when available. Matrix operations can either be done using convenient
// methods that return a new matrix for the result or optimized methods
// that store the result in an existing matrix to avoid generating garbage.

var hasFloat32Array = (typeof Float32Array != 'undefined');

// ### new GL.Matrix([elements])
// 
// This constructor takes 9 or 16 arguments in row-major order, which can be passed
// individually, as a list, or even as four lists, one for each row. If the
// arguments are omitted then the identity matrix is constructed instead.
function Matrix() {
  var m = Array.prototype.concat.apply([], arguments);
  if (!m.length) {
    m = [
      1, 0, 0, 0,
      0, 1, 0, 0,
      0, 0, 1, 0,
      0, 0, 0, 1
    ];
  } else if (m.length == 9) {
    m = [
      m[0], m[3], m[6], 0,
      m[1], m[4], m[7], 0,
      m[2], m[5], m[8], 0,
         0,    0,    0, 1
    ];  	
  }
  this.m = hasFloat32Array ? new Float32Array(m) : m;
}

Matrix.prototype = {
  // ### .inverse()
  // 
  // Returns the matrix that when multiplied with this matrix results in the
  // identity matrix.
  inverse: function() {
    return Matrix.inverse(this, new Matrix());
  },

  // ### .transpose()
  // 
  // Returns this matrix, exchanging columns for rows.
  transpose: function() {
    return Matrix.transpose(this, new Matrix());
  },

  // ### .multiply(matrix)
  // 
  // Returns the concatenation of the transforms for this matrix and `matrix`.
  // This emulates the OpenGL function `glMultMatrix()`.
  multiply: function(matrix) {
    return Matrix.multiply(this, matrix, new Matrix());
  },

  // ### .transformPoint(point)
  // 
  // Transforms the vector as a point with a w coordinate of 1. This
  // means translations will have an effect, for example.
  transformPoint: function(v) {
    var m = this.m;
    return new Vector(
      m[0] * v.x + m[1] * v.y + m[2] * v.z + m[3],
      m[4] * v.x + m[5] * v.y + m[6] * v.z + m[7],
      m[8] * v.x + m[9] * v.y + m[10] * v.z + m[11]
    ).divide(m[12] * v.x + m[13] * v.y + m[14] * v.z + m[15]);
  },

  // ### .transformPoint(vector)
  // 
  // Transforms the vector as a vector with a w coordinate of 0. This
  // means translations will have no effect, for example.
  transformVector: function(v) {
    var m = this.m;
    return new Vector(
      m[0] * v.x + m[1] * v.y + m[2] * v.z,
      m[4] * v.x + m[5] * v.y + m[6] * v.z,
      m[8] * v.x + m[9] * v.y + m[10] * v.z
    );
  }
};

// ### GL.Matrix.inverse(matrix[, result])
// 
// Returns the matrix that when multiplied with `matrix` results in the
// identity matrix. You can optionally pass an existing matrix in `result`
// to avoid allocating a new matrix. This implementation is from the Mesa
// OpenGL function `__gluInvertMatrixd()` found in `project.c`.
Matrix.inverse = function(matrix, result) {
  result = result || new Matrix();
  var m = matrix.m, r = result.m;

  r[0] = m[5]*m[10]*m[15] - m[5]*m[14]*m[11] - m[6]*m[9]*m[15] + m[6]*m[13]*m[11] + m[7]*m[9]*m[14] - m[7]*m[13]*m[10];
  r[1] = -m[1]*m[10]*m[15] + m[1]*m[14]*m[11] + m[2]*m[9]*m[15] - m[2]*m[13]*m[11] - m[3]*m[9]*m[14] + m[3]*m[13]*m[10];
  r[2] = m[1]*m[6]*m[15] - m[1]*m[14]*m[7] - m[2]*m[5]*m[15] + m[2]*m[13]*m[7] + m[3]*m[5]*m[14] - m[3]*m[13]*m[6];
  r[3] = -m[1]*m[6]*m[11] + m[1]*m[10]*m[7] + m[2]*m[5]*m[11] - m[2]*m[9]*m[7] - m[3]*m[5]*m[10] + m[3]*m[9]*m[6];

  r[4] = -m[4]*m[10]*m[15] + m[4]*m[14]*m[11] + m[6]*m[8]*m[15] - m[6]*m[12]*m[11] - m[7]*m[8]*m[14] + m[7]*m[12]*m[10];
  r[5] = m[0]*m[10]*m[15] - m[0]*m[14]*m[11] - m[2]*m[8]*m[15] + m[2]*m[12]*m[11] + m[3]*m[8]*m[14] - m[3]*m[12]*m[10];
  r[6] = -m[0]*m[6]*m[15] + m[0]*m[14]*m[7] + m[2]*m[4]*m[15] - m[2]*m[12]*m[7] - m[3]*m[4]*m[14] + m[3]*m[12]*m[6];
  r[7] = m[0]*m[6]*m[11] - m[0]*m[10]*m[7] - m[2]*m[4]*m[11] + m[2]*m[8]*m[7] + m[3]*m[4]*m[10] - m[3]*m[8]*m[6];

  r[8] = m[4]*m[9]*m[15] - m[4]*m[13]*m[11] - m[5]*m[8]*m[15] + m[5]*m[12]*m[11] + m[7]*m[8]*m[13] - m[7]*m[12]*m[9];
  r[9] = -m[0]*m[9]*m[15] + m[0]*m[13]*m[11] + m[1]*m[8]*m[15] - m[1]*m[12]*m[11] - m[3]*m[8]*m[13] + m[3]*m[12]*m[9];
  r[10] = m[0]*m[5]*m[15] - m[0]*m[13]*m[7] - m[1]*m[4]*m[15] + m[1]*m[12]*m[7] + m[3]*m[4]*m[13] - m[3]*m[12]*m[5];
  r[11] = -m[0]*m[5]*m[11] + m[0]*m[9]*m[7] + m[1]*m[4]*m[11] - m[1]*m[8]*m[7] - m[3]*m[4]*m[9] + m[3]*m[8]*m[5];

  r[12] = -m[4]*m[9]*m[14] + m[4]*m[13]*m[10] + m[5]*m[8]*m[14] - m[5]*m[12]*m[10] - m[6]*m[8]*m[13] + m[6]*m[12]*m[9];
  r[13] = m[0]*m[9]*m[14] - m[0]*m[13]*m[10] - m[1]*m[8]*m[14] + m[1]*m[12]*m[10] + m[2]*m[8]*m[13] - m[2]*m[12]*m[9];
  r[14] = -m[0]*m[5]*m[14] + m[0]*m[13]*m[6] + m[1]*m[4]*m[14] - m[1]*m[12]*m[6] - m[2]*m[4]*m[13] + m[2]*m[12]*m[5];
  r[15] = m[0]*m[5]*m[10] - m[0]*m[9]*m[6] - m[1]*m[4]*m[10] + m[1]*m[8]*m[6] + m[2]*m[4]*m[9] - m[2]*m[8]*m[5];

  var det = m[0]*r[0] + m[1]*r[4] + m[2]*r[8] + m[3]*r[12];
  for (var i = 0; i < 16; i++) r[i] /= det;
  return result;
};

// ### GL.Matrix.transpose(matrix[, result])
// 
// Returns `matrix`, exchanging columns for rows. You can optionally pass an
// existing matrix in `result` to avoid allocating a new matrix.
Matrix.transpose = function(matrix, result) {
  result = result || new Matrix();
  var m = matrix.m, r = result.m;
  r[0] = m[0]; r[1] = m[4]; r[2] = m[8]; r[3] = m[12];
  r[4] = m[1]; r[5] = m[5]; r[6] = m[9]; r[7] = m[13];
  r[8] = m[2]; r[9] = m[6]; r[10] = m[10]; r[11] = m[14];
  r[12] = m[3]; r[13] = m[7]; r[14] = m[11]; r[15] = m[15];
  return result;
};

// ### GL.Matrix.multiply(left, right[, result])
// 
// Returns the concatenation of the transforms for `left` and `right`. You can
// optionally pass an existing matrix in `result` to avoid allocating a new
// matrix. This emulates the OpenGL function `glMultMatrix()`.
Matrix.multiply = function(left, right, result) {
  result = result || new Matrix();
  var a = left.m, b = right.m, r = result.m;

  r[0] = a[0] * b[0] + a[1] * b[4] + a[2] * b[8] + a[3] * b[12];
  r[1] = a[0] * b[1] + a[1] * b[5] + a[2] * b[9] + a[3] * b[13];
  r[2] = a[0] * b[2] + a[1] * b[6] + a[2] * b[10] + a[3] * b[14];
  r[3] = a[0] * b[3] + a[1] * b[7] + a[2] * b[11] + a[3] * b[15];

  r[4] = a[4] * b[0] + a[5] * b[4] + a[6] * b[8] + a[7] * b[12];
  r[5] = a[4] * b[1] + a[5] * b[5] + a[6] * b[9] + a[7] * b[13];
  r[6] = a[4] * b[2] + a[5] * b[6] + a[6] * b[10] + a[7] * b[14];
  r[7] = a[4] * b[3] + a[5] * b[7] + a[6] * b[11] + a[7] * b[15];

  r[8] = a[8] * b[0] + a[9] * b[4] + a[10] * b[8] + a[11] * b[12];
  r[9] = a[8] * b[1] + a[9] * b[5] + a[10] * b[9] + a[11] * b[13];
  r[10] = a[8] * b[2] + a[9] * b[6] + a[10] * b[10] + a[11] * b[14];
  r[11] = a[8] * b[3] + a[9] * b[7] + a[10] * b[11] + a[11] * b[15];

  r[12] = a[12] * b[0] + a[13] * b[4] + a[14] * b[8] + a[15] * b[12];
  r[13] = a[12] * b[1] + a[13] * b[5] + a[14] * b[9] + a[15] * b[13];
  r[14] = a[12] * b[2] + a[13] * b[6] + a[14] * b[10] + a[15] * b[14];
  r[15] = a[12] * b[3] + a[13] * b[7] + a[14] * b[11] + a[15] * b[15];

  return result;
};

// ### GL.Matrix.identity([result])
// 
// Returns an identity matrix. You can optionally pass an existing matrix in
// `result` to avoid allocating a new matrix. This emulates the OpenGL function
// `glLoadIdentity()`.
Matrix.identity = function(result) {
  result = result || new Matrix();
  var m = result.m;
  m[0] = m[5] = m[10] = m[15] = 1;
  m[1] = m[2] = m[3] = m[4] = m[6] = m[7] = m[8] = m[9] = m[11] = m[12] = m[13] = m[14] = 0;
  return result;
};

// ### GL.Matrix.perspective(fov, aspect, near, far[, result])
// 
// Returns a perspective transform matrix, which makes far away objects appear
// smaller than nearby objects. The `aspect` argument should be the width
// divided by the height of your viewport and `fov` is the top-to-bottom angle
// of the field of view in degrees. You can optionally pass an existing matrix
// in `result` to avoid allocating a new matrix. This emulates the OpenGL
// function `gluPerspective()`.
Matrix.perspective = function(fov, aspect, near, far, result) {
  var y = Math.tan(fov * Math.PI / 360) * near;
  var x = y * aspect;
  return Matrix.frustum(-x, x, -y, y, near, far, result);
};

// ### GL.Matrix.frustum(left, right, bottom, top, near, far[, result])
// 
// Sets up a viewing frustum, which is shaped like a truncated pyramid with the
// camera where the point of the pyramid would be. You can optionally pass an
// existing matrix in `result` to avoid allocating a new matrix. This emulates
// the OpenGL function `glFrustum()`.
Matrix.frustum = function(l, r, b, t, n, f, result) {
  result = result || new Matrix();
  var m = result.m;

  m[0] = 2 * n / (r - l);
  m[1] = 0;
  m[2] = (r + l) / (r - l);
  m[3] = 0;

  m[4] = 0;
  m[5] = 2 * n / (t - b);
  m[6] = (t + b) / (t - b);
  m[7] = 0;

  m[8] = 0;
  m[9] = 0;
  m[10] = -(f + n) / (f - n);
  m[11] = -2 * f * n / (f - n);

  m[12] = 0;
  m[13] = 0;
  m[14] = -1;
  m[15] = 0;

  return result;
};

// ### GL.Matrix.ortho(left, right, bottom, top, near, far[, result])
// 
// Returns an orthographic projection, in which objects are the same size no
// matter how far away or nearby they are. You can optionally pass an existing
// matrix in `result` to avoid allocating a new matrix. This emulates the OpenGL
// function `glOrtho()`.
Matrix.ortho = function(l, r, b, t, n, f, result) {
  result = result || new Matrix();
  var m = result.m;

  m[0] = 2 / (r - l);
  m[1] = 0;
  m[2] = 0;
  m[3] = -(r + l) / (r - l);

  m[4] = 0;
  m[5] = 2 / (t - b);
  m[6] = 0;
  m[7] = -(t + b) / (t - b);

  m[8] = 0;
  m[9] = 0;
  m[10] = -2 / (f - n);
  m[11] = -(f + n) / (f - n);

  m[12] = 0;
  m[13] = 0;
  m[14] = 0;
  m[15] = 1;

  return result;
};

// ### GL.Matrix.scale(x, y, z[, result])
// 
// This emulates the OpenGL function `glScale()`. You can optionally pass an
// existing matrix in `result` to avoid allocating a new matrix. 
Matrix.scale = function(x, y, z, result) {
  result = result || new Matrix();
  var m = result.m;

  m[0] = x;
  m[1] = 0;
  m[2] = 0;
  m[3] = 0;

  m[4] = 0;
  m[5] = y;
  m[6] = 0;
  m[7] = 0;

  m[8] = 0;
  m[9] = 0;
  m[10] = z;
  m[11] = 0;

  m[12] = 0;
  m[13] = 0;
  m[14] = 0;
  m[15] = 1;

  return result;
};

// ### GL.Matrix.translate(x, y, z[, result])
// 
// This emulates the OpenGL function `glTranslate()`. You can optionally pass
// an existing matrix in `result` to avoid allocating a new matrix. 
Matrix.translate = function(x, y, z, result) {
  result = result || new Matrix();
  var m = result.m;

  m[0] = 1;
  m[1] = 0;
  m[2] = 0;
  m[3] = x;

  m[4] = 0;
  m[5] = 1;
  m[6] = 0;
  m[7] = y;

  m[8] = 0;
  m[9] = 0;
  m[10] = 1;
  m[11] = z;

  m[12] = 0;
  m[13] = 0;
  m[14] = 0;
  m[15] = 1;

  return result;
};

// ### GL.Matrix.rotate(a, x, y, z, cx, cy, cz[, result])
// 
// Returns a matrix that rotates by `a` radians around the vector `x, y, z` and center 'cx, cy, cz'.
// You can optionally pass an existing matrix in `result` to avoid allocating
// a new matrix. This emulates the OpenGL function `glRotate()`. 
Matrix.rotate = function(a, x, y, z, cx, cy, cz, result) {
  if (!a || (!x && !y && !z)) {
    return Matrix.identity(result);
  }

  result = result || new Matrix();
  var m = result.m;

  var d = Math.sqrt(x*x + y*y + z*z);
  // a *= Math.PI / 180; 
  x /= d; y /= d; z /= d;
  var c = Math.cos(a), s = Math.sin(a), t = 1 - c;

  m[0] = x * x * t + c;
  m[1] = x * y * t - z * s;
  m[2] = x * z * t + y * s;
  m[3] = cx - m[0]*cx - m[1]*cy - m[2]*cz;

  m[4] = y * x * t + z * s;
  m[5] = y * y * t + c;
  m[6] = y * z * t - x * s;
  m[7] = cy - m[4]*cx - m[5]*cy - m[6]*cz;

  m[8] = z * x * t - y * s;
  m[9] = z * y * t + x * s;
  m[10] = z * z * t + c;
  m[11] = cz - m[8]*cx - m[9]*cy - m[10]*cz;

  m[12] = 0;
  m[13] = 0;
  m[14] = 0;
  m[15] = 1;

  return result;
};

// ### GL.Matrix.lookAt(ex, ey, ez, cx, cy, cz, ux, uy, uz[, result])
// 
// Returns a matrix that puts the camera at the eye point `ex, ey, ez` looking
// toward the center point `cx, cy, cz` with an up direction of `ux, uy, uz`.
// You can optionally pass an existing matrix in `result` to avoid allocating
// a new matrix. This emulates the OpenGL function `gluLookAt()`.
Matrix.lookAt = function(ex, ey, ez, cx, cy, cz, ux, uy, uz, result) {
  result = result || new Matrix();
  var m = result.m;

  var e = new Vector(ex, ey, ez);
  var c = new Vector(cx, cy, cz);
  var u = new Vector(ux, uy, uz);
  var f = e.subtract(c).unit();
  var s = u.cross(f).unit();
  var t = f.cross(s).unit();

  m[0] = s.x;
  m[1] = s.y;
  m[2] = s.z;
  m[3] = -s.dot(e);

  m[4] = t.x;
  m[5] = t.y;
  m[6] = t.z;
  m[7] = -t.dot(e);

  m[8] = f.x;
  m[9] = f.y;
  m[10] = f.z;
  m[11] = -f.dot(e);

  m[12] = 0;
  m[13] = 0;
  m[14] = 0;
  m[15] = 1;

  return result;
};
/**
 * Deployment for 3D SVG drawing.
 * @module SVGGraphics 
 */
	
// Represents indexed triangle geometry with arbitrary additional attributes.
// You need a shader to draw a mesh; meshes can't draw themselves.
// 
// A mesh is a collection of `Buffer` objects which are either vertex buffers
// (holding per-vertex attributes) or index buffers (holding the order in which
// vertices are rendered). By default, a mesh has a position vertex buffer called
// `vertices` and a triangle index buffer called `triangles`. New buffers can be
// added using `addVertexBuffer()` and `addIndexBuffer()`. Two strings are
// required when adding a new vertex buffer, the name of the data array on the
// mesh instance and the name of the GLSL attribute in the vertex shader.
// 
// Example usage:
// 
//     var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL, { coords: true, lines: true });
// 
//     // Default attribute "vertices", available as "gl_Vertex" in
//     // the vertex shader
//     mesh.setVertices([[0, 0, 0], [1, 0, 0], [0, 1, 0], [1, 1, 0]]);
// 
//     // Optional attribute "coords" enabled in constructor,
//     // available as "gl_TexCoord" in the vertex shader
//     mesh.setCoords([[0, 0], [1, 0], [0, 1], [1, 1]]);
// 
//     // Custom attribute "weights", available as "weight" in the
//     // vertex shader
//     mesh.addVertexBuffer('weights', 'weight');
//     mesh.setBuffer('weights',[1, 0, 0, 1]);
// 
//     // Default index buffer "triangles"
//     mesh.setTriangles([[0, 1, 2], [2, 1, 3]]);
// 
//     // Optional index buffer "lines" enabled in constructor
//     mesh.setLines([[0, 1], [0, 2], [1, 3], [2, 3]]);
// 
//     // Upload provided data to GPU memory
//     mesh.compile();

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

// Represents a collection of vertex buffers and index buffers. Each vertex
// buffer maps to one attribute in GLSL and has a corresponding property set
// on the Mesh instance. There is one vertex buffer by default: `vertices`,
// which maps to `gl_Vertex`. The `coords`, `normals`, and `colors` vertex
// buffers map to `gl_TexCoord`, `gl_Normal`, and `gl_Color` respectively,
// and can be enabled by setting the corresponding options to true. There are
// two index buffers, `triangles` and `lines`, which are used for rendering
// `mGL.TRIANGLES` and `mGL.LINES`, respectively. Only `triangles` is enabled by
// default, although `computeWireframe()` will add a normal buffer if it wasn't
// initially enabled.

/**
 * @param mGL Element where draw
 * @returns A WebGL mesh
 */
EJSS_WEBGLGRAPHICS.mesh = function(mGL, id, options) {
  var self = {};
  options = options || {coords:true, normals:true, triangles:true, lines:true, colors:true};
  id = id || "mesh";  
  var vertexBuffers = {};
  var indexBuffers = {};
  
  self.getId = function() {
  	return id;
  }
  
  self.getVertexBuffers = function() {
  	return vertexBuffers;
  }

  self.getIndexBuffers = function() {
  	return indexBuffers;
  }
  
  self.setVertices = function(b) {
  	self.vertices = b;
  }

  self.getVertices = function() {
  	return self.vertices;
  }

  self.setCoords = function(b) {
  	self.coords = b;
  }

  self.getCoords = function() {
  	return self.coords;
  }

  self.setNormals = function(b) {
  	self.normals = b;
  }

  self.getNormals = function() {
  	return self.normals;
  }

  self.setColors = function(b) {
  	self.colors = b;
  }

  self.getColors = function() {
  	return self.colors;
  }

  self.setTriangles = function(b) {
  	self.triangles = b;
  }

  self.getTriangles = function() {
  	return self.triangles;
  }

  self.setLines = function(b) {
  	self.lines = b;
  }

  self.getLines = function() {
  	return self.lines;
  }
  
  self.setBuffer = function(name, b) {
  	self[name] = b;
  }
  
  // Add a new vertex buffer with a list as a property called `name` on this object
  // and map it to the attribute called `attribute` in all shaders that draw this mesh.
  self.addVertexBuffer = function(name, attribute) {
    var buffer = vertexBuffers[attribute] = EJSS_WEBGLGRAPHICS.buffer(mGL, mGL.ARRAY_BUFFER, Float32Array);
    buffer.setName(name);
    self[name] = [];
  }

  // Add a new index buffer with a list as a property called `name` on this object.
  self.addIndexBuffer = function(name) {
    var buffer = indexBuffers[name] = EJSS_WEBGLGRAPHICS.buffer(mGL, mGL.ELEMENT_ARRAY_BUFFER, Uint16Array);
    self[name] = [];
  }

  // Upload all attached buffers to the GPU in preparation for rendering. This
  // doesn't need to be called every frame, only needs to be done when the data
  // changes.
  self.compile = function() {
    for (var attribute in vertexBuffers) {
      var buffer = vertexBuffers[attribute];
      buffer.setData(self[buffer.getName()]);
      buffer.compile();
    }

    for (var name in indexBuffers) {
      var buffer = indexBuffers[name];
      buffer.setData(self[name]);
      buffer.compile();
    }
  }

  // Transform all vertices by `matrix` and all normals by the inverse transpose
  // of `matrix`.
  self.transform = function(matrix) {
    self.vertices = self.vertices.map(function(v) {
      return matrix.transformPoint(Vector.fromArray(v)).toArray();
    });
    if (self.normals) {
      var invTrans = matrix.inverse().transpose();
      self.normals = self.normals.map(function(n) {
        return invTrans.transformVector(Vector.fromArray(n)).unit().toArray();
      });
    }
    self.compile();
    return self;
  }

  // Computes a new normal for each vertex from the average normal of the
  // neighboring triangles. This means adjacent triangles must share vertices
  // for the resulting normals to be smooth.
  self.computeNormals = function() {
    if (!self.normals) self.addVertexBuffer('normals', 'gl_Normal');
    for (var i = 0; i < self.vertices.length; i++) {
      self.normals[i] = new Vector();
    }
    for (var i = 0; i < self.triangles.length; i++) {
      var t = self.triangles[i];
      var a = Vector.fromArray(self.vertices[t[0]]);
      var b = Vector.fromArray(self.vertices[t[1]]);
      var c = Vector.fromArray(self.vertices[t[2]]);
      var normal = b.subtract(a).cross(c.subtract(a)).unit();
      self.normals[t[0]] = self.normals[t[0]].add(normal);
      self.normals[t[1]] = self.normals[t[1]].add(normal);
      self.normals[t[2]] = self.normals[t[2]].add(normal);
    }
    for (var i = 0; i < self.vertices.length; i++) {
      self.normals[i] = self.normals[i].unit().toArray();
    }
    self.compile();
    return self;
  }

  // Populate the `lines` index buffer from the `triangles` index buffer.
  self.computeWireframe = function() {
    var indexer = EJSS_WEBGLGRAPHICS.indexer();
    for (var i = 0; i < self.triangles.length; i++) {
      var t = self.triangles[i];
      for (var j = 0; j < t.length; j++) {
        var a = t[j], b = t[(j + 1) % t.length];
        indexer.add([Math.min(a, b), Math.max(a, b)]);
      }
    }
    if (!self.lines) self.addIndexBuffer('lines');
    self.lines = indexer.getUnique();
    self.compile();
    return self;
  }

  // Computes the axis-aligned bounding box, which is an object whose `min` and
  // `max` properties contain the minimum and maximum coordinates of all vertices.
  self.getAABB = function() {
    var aabb = { min: new Vector(Number.MAX_VALUE, Number.MAX_VALUE, Number.MAX_VALUE) };
    aabb.max = aabb.min.negative();
    for (var i = 0; i < self.vertices.length; i++) {
      var v = Vector.fromArray(self.vertices[i]);
      aabb.min = Vector.min(aabb.min, v);
      aabb.max = Vector.max(aabb.max, v);
    }
    return aabb;
  }

  // Computes a sphere that contains all vertices (not necessarily the smallest
  // sphere). The returned object has two properties, `center` and `radius`.
  self.getBoundingSphere = function() {
    var aabb = self.getAABB();
    var sphere = { center: aabb.min.add(aabb.max).divide(2), radius: 0 };
    for (var i = 0; i < self.vertices.length; i++) {
      sphere.radius = Math.max(sphere.radius,
        Vector.fromArray(self.vertices[i]).subtract(sphere.center).length());
    }
    return sphere;
  }

  self.addVertexBuffer('vertices', 'gl_Vertex');
  if (options.coords) self.addVertexBuffer('coords', 'gl_TexCoord');
  if (options.normals) self.addVertexBuffer('normals', 'gl_Normal');
  if (options.colors) self.addVertexBuffer('colors', 'gl_Color');
  if (!('triangles' in options) || options.triangles) self.addIndexBuffer('triangles');
  if (options.lines) {
  		self.addIndexBuffer('lines');
  }

  return self;
}	

// 
// Generates indices into a list of unique objects from a stream of objects
// that may contain duplicates. This is useful for generating compact indexed
// meshes from unindexed data.
EJSS_WEBGLGRAPHICS.indexer = function() {
  var self = {};
  var unique = [];
  // var indices = [];
  var map = {};

  self.getUnique = function() {
  	return unique;
  }

  // 
  // Adds the object `obj` to `unique` if it hasn't already been added. Returns
  // the index of `obj` in `unique`.
  self.add = function(obj) {
    var key = JSON.stringify(obj);
    if (!(key in map)) {
      map[key] = unique.length;
      unique.push(obj);
    }
    return map[key];
  }
  
  return self;
};

// 
// Provides a simple method of uploading data to a GPU buffer. Example usage:
// 
//     var vertices = EJSS_WEBGLGRAPHICS.buffer(mGL, mGL.ARRAY_BUFFER, Float32Array);
//     var indices = EJSS_WEBGLGRAPHICS.buffer(mGL, mGL.ELEMENT_ARRAY_BUFFER, Uint16Array);
//     vertices.setData([[0, 0, 0], [1, 0, 0], [0, 1, 0], [1, 1, 0]]);
//     indices.setData([[0, 1, 2], [2, 1, 3]]);
//     vertices.compile();
//     indices.compile();
// 
EJSS_WEBGLGRAPHICS.buffer = function(mGL, target, type) {
  var self = {};
  var buffer = null;
  var target = target;
  var type = type;
  var mName = "";
  var mData = [];

  self.setData = function(b) {
  	mData = b;
  }

  self.setName = function(name) {
  	mName = name;
  }

  self.getBuffer = function() {
  	return buffer;
  }

  self.getName = function() {
  	return mName;
  }

  // 
  // Upload the contents of `data` to the GPU in preparation for rendering. The
  // data must be a list of lists where each inner list has the same length. For
  // example, each element of data for vertex normals would be a list of length three.
  // This will remember the data length and element length for later use by shaders.
  // The type can be either `mGL.STATIC_DRAW` or `mGL.DYNAMIC_DRAW`, and defaults to
  // `mGL.STATIC_DRAW`.
  // 
  // This could have used `[].concat.apply([], this.data)` to flatten
  // the array but Google Chrome has a maximum number of arguments so the
  // concatenations are chunked to avoid that limit.
  self.compile = function(tp) {
    var data = [];
    for (var i = 0, chunk = 10000; i < mData.length; i += chunk) {
      data = Array.prototype.concat.apply(data, mData.slice(i, i + chunk));
    }
    var spacing = mData.length ? data.length / mData.length : 0;
    if (spacing != Math.round(spacing)) throw 'buffer elements not of consistent size, average size is ' + spacing;
    buffer = buffer || mGL.createBuffer();
    buffer.length = data.length;
    buffer.spacing = spacing;
    mGL.bindBuffer(target, buffer);
    mGL.bufferData(target, new type(data), tp || mGL.STATIC_DRAW);
  }
  
  return self;
}
/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL plane
 */
EJSS_WEBGLGRAPHICS.plane = function(mGL, mElement) {
	  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL, mElement.getName());
	  		 
	  var dirA = mElement.getDirectionA();
	  var dirB = mElement.getDirectionB();
	  var sizeA = mElement.getSizeA();
	  var sizeB = mElement.getSizeB();
	  var vecZ = Vector.cross(new Vector(dirA), new Vector(dirB), new Vector()); 
	  var dirZ = vecZ.unit().toArray();
	  
	  var trans = new Matrix( [
	  		dirA[0],dirB[0],dirZ[0],0,
	  		dirA[1],dirB[1],dirZ[1],0,
	  		dirA[2],dirB[2],dirZ[2],0,
	  		0,0,0,1,
	  		]);
	  
	  var resolution = mElement.getResolutionU();     // tesselation level 	
	  for (var y = 0; y <= resolution; y++) {
	    var t = (y / resolution) * sizeB;
	    for (var x = 0; x <= resolution; x++) {
	      var s = (x / resolution) * sizeA;	      
	      var vec = new Vector([2 * s - 1, 2 * t - 1, 0]);
	      var res = trans.transformVector(vec);	      
	      mesh.vertices.push(res.toArray());
	      mesh.coords.push([s, t]);
	      mesh.normals.push([0, 0, 1]);	      
	      if (x < resolution && y < resolution) {
	        var i = x + y * (resolution + 1);
	        mesh.triangles.push([i + resolution + 1, i + 1, i]);
	        mesh.triangles.push([i + resolution + 2, i + 1, i + resolution + 1]);
	        
	        mesh.lines.push([i + resolution + 1, i]);
	        mesh.lines.push([i + 1, i]);
	        mesh.lines.push([i + resolution + 2, i + 1]);
	        mesh.lines.push([i + resolution + 2, i + resolution + 1]);
	      }
	    }
	  }
	  
	  mesh.compile();
	
	  mGL.addElement(mElement.getName() + ".mesh", mesh);
        
      return mesh;
}

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL segment
 */
EJSS_WEBGLGRAPHICS.segment = function(mGL, mElement) {
  	  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName(), 
	  	{coords:false, normals:true, triangles:true, lines:false, colors:false});

	  var resolutionU = mElement.getResolutionU();
	  var radius = mElement.getLineWidth() / 2;
      
	  // generate vertex for body
	  var vertices3D = [];	  
	  for (var x = 0; x < resolutionU; x++ ) {
		var u = x / (resolutionU - 1);
		var xpos = radius * Math.sin( u * Math.PI * 2 );
		var ypos = radius * Math.cos( u * Math.PI * 2 );

		vertices3D.push(new Vector([xpos, ypos, 0])); // top vertex
		vertices3D.push(new Vector([xpos, ypos, 0])); // bottom vertex
	  }
	  
	  // rotate and translate top and bottom
  	  var endpoint = new Vector(mElement.getSize());
	  var cross = endpoint.cross(new Vector(0,0,1));
	  if(cross.length()==0) cross.y = 1; // if endoint in axis Z, get axis Y to rotate 
	  // console.log("cross:" + cross.x + " " + cross.y + " " + cross.z + " ");	  
	  var angle = endpoint.angle(new Vector(0,0,1));
	  // console.log("angle:" + angle);
	  var result = new Matrix();
	  Matrix.rotate(-angle,cross.x,cross.y,cross.z,0,0,0,result);
	  for(var i=0; i<vertices3D.length; i++) {
	  	var vector = result.transformPoint(vertices3D[i]);
	  	if (i % 2 == 0) {
	  		// add enpoint to translate	  	
	  		mesh.vertices.push(vector.add(endpoint).toArray());	  		
	  	} else {
	  		mesh.vertices.push(vector.toArray());
	  	}
	  }
	  
	  // create surface triangles
	  for (var x = 0; x < resolutionU - 1; x++ ) {
	    var first = 2*x;		    
	    var fourth = 2*x + 2;
	    var second = 2*x + 1;
	    var third = 2*x + 3;
			
        mesh.triangles.push([third, second, first]);      
        mesh.triangles.push([fourth, third, first]);
	  }		
		
	  // top cap
	  mesh.vertices.push(endpoint.toArray());		
      for (var i = 0; i < resolutionU - 1; i++) {
      	var first = mesh.vertices.length - 1;
      	var second = 2*i+2;
      	var third = 2*i;
      	mesh.triangles.push([first, second, third]);
      }        

	  // bottom cap
 	  mesh.vertices.push([0, 0, 0]);
      for (var i = 0; i < resolutionU - 1; i++) {
      	var first = 2*i+1;
      	var second = 2*i+3;
      	var third = mesh.vertices.length - 1;
       	mesh.triangles.push([first, second, third]);
      }        
	  	        
	  mesh.computeNormals();	
	  
	  mesh.compile();	
	
	  mGL.addElement(mElement.getName() + ".mesh", mesh);
        
     return mesh;

/* Previously it was possible to manage lineWidth and then the segment was easy ...
 	  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName(), 
	  	{coords:false, normals:false, triangles:false, lines:true, colors:false});
	  
      mesh.vertices.push([0,0,0]);
      mesh.vertices.push([2,2,2]);

	  mesh.lines.push([0,1]);
	  
	  mesh.compile();
           
      return mesh;
*/      
}

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

EJSS_WEBGLGRAPHICS.Shader = {
	//
	// Basic shaders (no lights), using for lines
	// (https://mattdesl.svbtle.com/drawing-lines-is-hard)
	// (https://jsfiddle.net/vy0we5wb/4)
	BasicVS: 
     [
     'void main() {',
     '  mat4 m = gl_ProjectionMatrix * gl_ViewMatrix * gl_ModelMatrix;',
     '  gl_Position = m * vec4(gl_Vertex.xyz, 1.0);',
     '}'
     ].join( '\r\n' ),
          
    BasicFS: 
     [
     'uniform vec3 color;',
     'uniform float opacity;',
     'void main() {',
     '    gl_FragColor = vec4(color, opacity);',
     '}'          
     ].join( '\r\n' ),
     
    //
    // Shaders using color array (only basic element)
    //
	ColorVS:
     [
     'varying vec4 color;',
     'void main() {',
     ' color = gl_Color;',
     ' gl_Position = gl_ProjectionMatrix * gl_ViewMatrix * gl_ModelMatrix * vec4(gl_Vertex.xyz, 1.0);',
     '}'
     ].join( '\r\n' ),
     
    ColorFS: 
     [
     'varying vec4 color;',
     'void main() {',
     ' gl_FragColor = color;',
     '}'    
     ].join( '\r\n' ),
	
	//
	// Shaders with light 
	// see: http://multivis.net/lecture/phong.html
	//
	LightVS:
     [
     'uniform float ka;',
     'uniform float kd;',
     'uniform float ks;',
     'uniform float shininessVal;',
     'uniform vec3 ambientColor;',
     'uniform vec3 diffuseColor;',
     'uniform vec3 specularColor;',
     'uniform vec3 lights[10];',
     'uniform int numlight;',
     'varying vec3 tmpcolor;',
     'void main() {',
     '  vec4 vmPos = gl_ViewMatrix * gl_ModelMatrix * vec4(gl_Vertex.xyz, 1.0);',
     '  vec3 normal = normalize(gl_NormalMatrix * gl_Normal);',
     '  tmpcolor = vec3(0);',
     '	for(int i = 0; i < 10; i++) { ',
     '		if(i == numlight) break;',
     '		vec3 light = normalize(lights[i] - vmPos.xyz);',
     '      float lambertian = abs(dot(normal, light));',
     '      float specular = 0.0;',
     '      if(lambertian > 0.0) {',
     '        vec3 reflect = reflect(-light, normal);',
     '        vec3 vector = normalize(-vmPos.xyz);',
     '        float specAngle = max(dot(reflect, vector), 0.0);',
     '        specular = pow(specAngle, shininessVal);',
     '      }',
     '      tmpcolor += vec3(ka * ambientColor +',
     '        kd * lambertian * diffuseColor +',
     '        ks * specular * specularColor);',
     '	} ',
     '  gl_Position = gl_ProjectionMatrix * vmPos;',
     '}'
     ].join( '\r\n' ),
     
    LightFS: 
     [
     'uniform float opacity;',
     'varying vec3 tmpcolor;',
     'void main() {',
   	 '  gl_FragColor = vec4(tmpcolor, opacity);',
     '}'        
     ].join( '\r\n' ),
     
    // Shaders with light using color array (only basic element)
	LightAndColorVS:
     [
     'uniform float ka;',
     'uniform float kd;',
     'uniform float ks;',
     'uniform float shininessVal;',
     'uniform vec3 ambientColor;',
     'uniform vec3 specularColor;',
     'uniform vec3 lights[10];',
     'uniform int numlight;',
     'varying vec3 tmpcolor;',
     'void main() {',
     '  vec4 vmPos = gl_ViewMatrix * gl_ModelMatrix * vec4(gl_Vertex.xyz, 1.0);',
     '  vec3 normal = normalize(gl_NormalMatrix * gl_Normal);',
     '  tmpcolor = vec3(0);',
     '	for(int i = 0; i < 10; i++) { ',
     '		if(i == numlight) break;',
     '		vec3 light = normalize(lights[i] - vmPos.xyz);',
     '      float lambertian = abs(dot(normal, light));',
     '      float specular = 0.0;',
     '      if(lambertian > 0.0) {',
     '        vec3 reflect = reflect(-light, normal);',
     '        vec3 vector = normalize(-vmPos.xyz);',
     '        float specAngle = max(dot(reflect, vector), 0.0);',
     '        specular = pow(specAngle, shininessVal);',
     '      }',
     '      tmpcolor += vec3(ka * ambientColor +',
     '        kd * lambertian * vec3(gl_Color) +',
     '        ks * specular * specularColor);',
     '	} ',
     '  gl_Position = gl_ProjectionMatrix * vmPos;',
     '}'
     ].join( '\r\n' ),
     
    LightAndColorFS: 
     [
     'uniform float opacity;',
     'varying vec3 tmpcolor;',
     'void main() {',
   	 '  gl_FragColor = vec4(tmpcolor, opacity);',
     '}'     
     ].join( '\r\n' ),

	//
	// Shaders for color scale
	//  Work in process!
	//    
	PaletteColorVS:
     [	
     'uniform vec3 floor;',
     'uniform vec3 ceil;',
     'uniform vec3 floorColor;',
     'uniform vec3 ceilColor;',
     'varying vec3 tmpcolor;',
     'void main() {',
     '  vec4 vmPos = gl_ViewMatrix * gl_ModelMatrix * vec4(gl_Vertex.xyz, 1.0);',
     '  vec3 normal = normalize(gl_NormalMatrix * gl_Normal);',
     '  vec3 diff = ceil - floor;',
     '  vec3 diffFloor = vmPos.xyz - floor;',
     '  vec3 diffCeil = vmPos.xyz - ceil;',
     '  float value1 = dot(diff, diffFloor);',
     '  if(value1 < 0.0) {',
     '    tmpcolor = floorColor;',
     '  } else {',
     '    float value2 = dot(diff, diffCeil);',
     '    if(value2 > 0.0) {',
     '      tmpcolor = ceilColor;',
     '    } else {',
     '      float value = length(diffFloor) / (length(diffFloor) + length(diffCeil));',
     '      tmpcolor = floorColor + value * (ceilColor - floorColor);',
     '    }',
     '  }',
     '  gl_Position = gl_ProjectionMatrix * vmPos;',
     '}'
     ].join( '\r\n' ),
     
    PaletteColorFS: 
     [
     'uniform float opacity;',
     'varying vec3 tmpcolor;',
     'void main() {',
   	 '  gl_FragColor = vec4(tmpcolor, opacity);',
     '}'      
     ].join( '\r\n' ),
          
	//
	// Shaders for texture
	//
	BasicVSwithTex :
     [	
	 'varying vec2 coord;',
	 'void main() {',
	 '  coord = gl_TexCoord.xy;',
	 '  gl_Position = gl_ProjectionMatrix * gl_ViewMatrix * gl_ModelMatrix * vec4(gl_Vertex.xyz, 1.0);',
	 '}'
     ].join( '\r\n' ),

	BasicFSwithTex:
     [	 
	 'uniform sampler2D texture;',
	 'varying vec2 coord;',
	 'void main() {',
	 '  gl_FragColor = texture2D(texture, coord);',
	 '}'	        
     ].join( '\r\n' ),
}

// Provides a convenient wrapper for WebGL shaders. A few uniforms and attributes,
// prefixed with `gl_`, are automatically added to all shader sources to make
// simple shaders easier to write.
// 
// Example usage:
// 
//     var shader = EJSS_WEBGLGRAPHICS.shader(mGL, '\
//       void main() {\
//         gl_Position = gl_ProjectionMatrix * gl_ViewMatrix * gl_ModelMatrix * gl_Vertex;\
//       }\
//     ', '\
//       uniform vec4 color;\
//       void main() {\
//         gl_FragColor = color;\
//       }\
//     ');
// 
//     shader.uniforms({
//       color: [1, 0, 0, 1]
//     }).draw(mesh);


/**
 * Compiles a shader program using the provided vertex and fragment shaders.
 * @method shader
 * @param vertexSource
 * @param fragmentSource 
 */
EJSS_WEBGLGRAPHICS.shader = function(mGL, mName, vertexSource, fragmentSource) {  
  var self = {};
  
  // Headers are prepended to the sources to provide some automatic functionality.
  var header = [
    'uniform mat3 gl_NormalMatrix;',
    'uniform mat4 gl_ModelMatrix;',
    'uniform mat4 gl_ViewMatrix;',
    'uniform mat4 gl_ProjectionMatrix;',''
    ].join('\r\n'); 
  var vertexHeader = [
    // '#extension GL_OES_standard_derivatives : enable', '',
    header, 
    'attribute vec4 gl_Vertex;',
    'attribute vec4 gl_TexCoord;',
    'attribute vec3 gl_Normal;',
    'attribute vec4 gl_Color;',''
    ].join('\r\n');
    
  var fragmentHeader = [
    // '#extension GL_OES_standard_derivatives : enable', '', 
    'precision highp float;', 
    header
    ].join('\r\n');
  
  // Attributes	 
  var attributes = {};
  // Uniforms locations
  var uniformLocations = {};

  // Check for the use of built-in matrices that require expensive matrix
  // multiplications to compute, and record these in `usedMatrices`.  
  function getUsedMatrices(vertexSource, fragmentSource) {
    var source = vertexSource + fragmentSource;
    var usedMatrices = {};
	EJSS_WEBGLGRAPHICS.Utils.regexMap(/\b(gl_[^;]*)\b;/g, header, function(groups) {
	  var name = groups[1];
	  if (source.indexOf(name) != -1) {
	    var capitalLetters = name.replace(/[a-z_]/g, '');
	    usedMatrices[capitalLetters] = mName + name;
	  }
	});
	return usedMatrices;
  }

  // The `gl_` prefix must be substituted for something else to avoid compile
  // errors, since it's a reserved prefix. This prefixes all reserved names with
  // `_`. The header is inserted after any extensions, since those must come
  // first.
  function fix(header, source) {
    var replaced = {};
    var match = /^((\s*\/\/.*\n|\s*#extension.*\n)+)[^]*$/.exec(source);
    source = match ? match[1] + header + source.substr(match[1].length) : header + source;
    EJSS_WEBGLGRAPHICS.Utils.regexMap(/\bgl_\w+\b/g, header, function(result) {
      if (!(result in replaced)) {
        source = source.replace(new RegExp('\\b' + result + '\\b', 'g'), mName + result);
        replaced[result] = true;
      }
    });
    return source;
  }
  
  // Compile and link errors are thrown as strings.
  function compileSource(type, source) {
    var shader = mGL.createShader(type);
    mGL.shaderSource(shader, source);
    mGL.compileShader(shader);
    if (!mGL.getShaderParameter(shader, mGL.COMPILE_STATUS)) {
      throw 'compile error: ' + mGL.getShaderInfoLog(shader);
    }
    return shader;
  }
   
  // 
  // Set a uniform for each property of `uniforms`. The correct `mGL.uniform*()` method is
  // inferred from the value types and from the stored uniform sampler flags.
  self.uniforms = function(uniforms) {
    mGL.useProgram(program);

    for (var name in uniforms) {
      var location = uniformLocations[name] || mGL.getUniformLocation(program, name);
      if (!location) continue;
      uniformLocations[name] = location;
      var value = uniforms[name];
      if (value instanceof Vector) {
        value = [value.x, value.y, value.z];
      } else if (value instanceof Matrix) {
        value = value.m;
      }
      if (EJSS_WEBGLGRAPHICS.Utils.isArray(value)) {
      	if(EJSS_WEBGLGRAPHICS.Utils.isArray(value[0])) {
      		// array of vector 3D (ex. lights)
      		var newvalue = [];
      		for(var j=0; j<value.length; j++) { 
      			newvalue.push(value[j][0]); 
      			newvalue.push(value[j][1]); 
      			newvalue.push(value[j][2]); 
      		}
      		mGL.uniform3fv(location, new Float32Array(newvalue)); 
      	} else {
	        switch (value.length) {
	          case 1: mGL.uniform1fv(location, new Float32Array(value)); break;
	          case 2: mGL.uniform2fv(location, new Float32Array(value)); break;
	          case 3: mGL.uniform3fv(location, new Float32Array(value)); break;
	          case 4: mGL.uniform4fv(location, new Float32Array(value)); break;
	          // Matrices are automatically transposed, since WebGL uses column-major
	          // indices instead of row-major indices.
	          case 9: mGL.uniformMatrix3fv(location, false, new Float32Array([
	            value[0], value[3], value[6],
	            value[1], value[4], value[7],
	            value[2], value[5], value[8]
	          ])); break;
	          case 16: mGL.uniformMatrix4fv(location, false, new Float32Array([
	            value[0], value[4], value[8], value[12],
	            value[1], value[5], value[9], value[13],
	            value[2], value[6], value[10], value[14],
	            value[3], value[7], value[11], value[15]
	          ])); break;
	          default: throw 'don\'t know how to load uniform "' + name + '" of length ' + value.length;
			}
        }
      } else if (EJSS_WEBGLGRAPHICS.Utils.isNumber(value)) { 
        ((name.indexOf('texture') != -1 || name.indexOf('num') != -1) ? mGL.uniform1i : mGL.uniform1f).call(mGL, location, value);
      } else {
        throw 'attempted to set uniform "' + name + '" to invalid value ' + value;
      }
    }
  }

  // 
  // Sets all uniform matrix attributes, binds all relevant buffers, and draws the
  // mesh geometry as indexed triangles or indexed lines. Set `mode` to `mGL.LINES`
  // (and either add indices to `lines` or call `computeWireframe()`) to draw the
  // mesh in wireframe.
  self.draw = function(mesh, mode) {
    self.drawBuffers(mesh.getVertexBuffers(),
      mesh.getIndexBuffers()[mode == mGL.LINES ? 'lines' : 'triangles'],
      arguments.length < 2 ? mGL.TRIANGLES : mode);
  }

  // ### .drawBuffers(vertexBuffers, indexBuffer, mode)
  // 
  // Sets all uniform matrix attributes, binds all relevant buffers, and draws the
  // indexed mesh geometry. The `vertexBuffers` argument is a map from attribute
  // names to `Buffer` objects of type `mGL.ARRAY_BUFFER`, `indexBuffer` is a `Buffer`
  // object of type `mGL.ELEMENT_ARRAY_BUFFER`, and `mode` is a WebGL primitive mode
  // like `mGL.TRIANGLES` or `mGL.LINES`. This method automatically creates and caches
  // vertex attribute pointers for attributes as needed.
  self.drawBuffers = function(vertexBuffers, indexBuffer, mode) {
    // Only construct up the built-in matrices we need for this shader.
    var used = getUsedMatrices(vertexSource, fragmentSource);
    var MM = mGL.modelMatrix;
    var VM = mGL.viewMatrix;
    var PM = mGL.projectionMatrix;
    var matrices = {};
    if (used.MM) matrices[used.MM] = MM;
    if (used.VM) matrices[used.VM] = VM;
    if (used.PM) matrices[used.PM] = PM;
    if (used.NM) {
		var m = VM.multiply(MM).inverse().m; // MVMI
      	matrices[used.NM] = [m[0], m[4], m[8], m[1], m[5], m[9], m[2], m[6], m[10]];
    }
    self.uniforms(matrices);

    // Create and enable attribute pointers as necessary.
    var length = 0;
    for (var attribute in vertexBuffers) {
      var buffer = vertexBuffers[attribute];
      var location = attributes[attribute] ||
        mGL.getAttribLocation(program, attribute.replace(/^(gl_.*)$/, mName + '$1'));
      if (location == -1 || !buffer.getBuffer()) continue;
      attributes[attribute] = location;
      mGL.bindBuffer(mGL.ARRAY_BUFFER, buffer.getBuffer());
      mGL.enableVertexAttribArray(location);
      mGL.vertexAttribPointer(location, buffer.getBuffer().spacing, mGL.FLOAT, false, 0, 0);
      length = buffer.getBuffer().length / buffer.getBuffer().spacing;
    }

    // Disable unused attribute pointers.
    for (var attribute in attributes) {
      if (!(attribute in vertexBuffers)) {
        mGL.disableVertexAttribArray(attributes[attribute]);
      }
    }
    
    // Draw the geometry.
    if (length && (!indexBuffer || indexBuffer.getBuffer())) {
      if (indexBuffer) {
        mGL.bindBuffer(mGL.ELEMENT_ARRAY_BUFFER, indexBuffer.getBuffer());
        mGL.drawElements(mode, indexBuffer.getBuffer().length, mGL.UNSIGNED_SHORT, 0);
      } else {
        mGL.drawArrays(mode, 0, length);
      }
    }
  }
 
  // 
  // Initialization 
  var program = mGL.createProgram();
  mGL.attachShader(program, compileSource(mGL.VERTEX_SHADER, fix(vertexHeader, vertexSource)));
  mGL.attachShader(program, compileSource(mGL.FRAGMENT_SHADER, fix(fragmentHeader, fragmentSource)));
  mGL.linkProgram(program);
  if (!mGL.getProgramParameter(program, mGL.LINK_STATUS)) {
    throw 'link error: ' + mGL.getProgramInfoLog(program);
  }
 
  return self;  
}


/**
 * Deployment for 2D SVG drawing.
 * @module SVGGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL custom
 */

EJSS_WEBGLGRAPHICS.spring = function(mGL, mElement) {   
	function drawSpring(loops, pointsPerLoop, radius, solenoid, thinExtremes, origp, endp, moveTo, lineTo) {
		var segments = loops * pointsPerLoop;
		var delta = 2.0 * Math.PI / pointsPerLoop;
		if (radius < 0) delta *= -1;
		var pre = pointsPerLoop / 2;
		// normalize sizes        	
		var u1 = EJSS_TOOLS.Mathematics.normalTo([endp.x,endp.y,endp.z]);
		var u2 = EJSS_TOOLS.Mathematics.normalize(EJSS_TOOLS.Mathematics.crossProduct([endp.x,endp.y,endp.z], u1));
		
	    for (var i=0; i<=segments; i++) {
	      var k;
	      if (thinExtremes) {	// extremes
	        if (i < pre) k = 0;
	        else if (i < pointsPerLoop) k = i - pre;
	        else if (i > (segments - pre)) k = 0;
	        else if (i > (segments - pointsPerLoop)) k = segments - i - pre;
	        else k = pre;
	      }
	      else k = pre;
	      var angle = Math.PI/2 + i*delta;
	      var cos = Math.cos(angle), sin = Math.sin(angle);
	      var xx = origp.x + i*endp.x/segments + k*radius*(cos*u1[0] + sin*u2[0])/pre;
	      var yy = origp.y + i*endp.y/segments + k*radius*(cos*u1[1] + sin*u2[1])/pre;
	      var zz = origp.z + i*endp.z/segments + k*radius*(cos*u1[2] + sin*u2[2])/pre;
	      if (solenoid != 0.0)  {
	        var cte = k*Math.cos(i*2*Math.PI/pointsPerLoop)/pre;
	        xx += solenoid*cte*endp.x;
	        yy += solenoid*cte*endp.y;
	        zz += solenoid*cte*endp.z;
	      }
	      if (i==0)
	      	moveTo(xx,yy,zz); 
	      else 
	      	lineTo(xx,yy,zz);
	    }
	}

    var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName());

  	var endpoint = new Vector(mElement.getSize());
  	
  	// after it is translated
  	// var origpoint = new Vector(mElement.getPosition());
    var origpoint = new Vector([0,0,0]);
    
	// set attributes
	var index = 0;
	drawSpring(mElement.getLoops(), mElement.getPointsPerLoop(), 
			mElement.getRadius(), mElement.getSolenoid(), mElement.getThinExtremes(), origpoint, endpoint,
			function(xx,yy,zz) { mesh.vertices.push([xx, yy, zz]); index++;}, 
			function(xx,yy,zz) { mesh.vertices.push([xx, yy, zz]); mesh.lines.push([index-1,index]); index++});   
	
	// compute normals
	mesh.computeNormals();
	   		
	mesh.compile();
	  	  	  
	mGL.addElement(mElement.getName() + ".mesh", mesh);

    return mesh;
    
}/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL custom
 */

EJSS_WEBGLGRAPHICS.surface = function(mGL, mElement) {   
    
    createSurfaceBuffers = function(xpoints, ypoints, data3D, mesh){
        for (var i = 0; i < xpoints - 1; i++) {
            for (var j = 0; j < ypoints - 1; j++) {
                // Create surface vertices.
                var rawP1 = data3D[j + (i * ypoints)];
                var rawP2 = data3D[j + (i * ypoints) + ypoints];
                var rawP3 = data3D[j + (i * ypoints) + ypoints + 1];
                var rawP4 = data3D[j + (i * ypoints) + 1];
                
                mesh.vertices.push(rawP1);
                mesh.vertices.push(rawP2);
                mesh.vertices.push(rawP3);
                mesh.vertices.push(rawP4);                                
            }
        }
                
        var numQuads = ((xpoints - 1) * (ypoints - 1)) / 2;        
        for (var i = 0; i < (numQuads * 8); i += 4) {
            mesh.triangles.push([i, i+1, i+2]);
            mesh.triangles.push([i, i+2, i+3]);
            
            mesh.lines.push([i,i+1]);             
            mesh.lines.push([i,i+3]);             
            mesh.lines.push([i+2,i+3]);             
            mesh.lines.push([i+2,i+1]);             
            
            mesh.coords.push([0,0]);
            mesh.coords.push([0,1]);
            mesh.coords.push([1,0]);
            mesh.coords.push([1,1]);
        }        
    };

	  var data = mElement.getData();
	  if(data || data.length > 0) {
		  var numRows = data.length;
		  var numCols = data[0].length;

		  // prepare points
	      var data3ds = new Array();
	      var index = 0;
	      for (var i = 0; i < numRows; i++) {
	        for (var j = 0; j < numCols; j++) {
	            data3ds[index] = data[i][j];
	            index++; 
	        }
	      }
		  
		  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName());
		  
		  createSurfaceBuffers(numRows, numCols, data3ds, mesh);
	
	  	  // compute normals
	  	  mesh.computeNormals();
		   		
		  mesh.compile();
		  	  	  
		  mGL.addElement(mElement.getName() + ".mesh", mesh);

	  }	  	  
    
      return mesh;
}

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL text
 */
EJSS_WEBGLGRAPHICS.text = function(mGL, mElement) {
	  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL, mElement.getName());

	  var font = mElement.getFont();
	  var backGroundColor = mElement.getBackground();
	  var text = mElement.getText();
	  var color = font.getFillColor();	  
	  
	  // build canvas with text
	  var canvas = document.createElement("canvas");		 
	  var context = canvas.getContext("2d");
	  context = canvas.getContext("2d");

	  // fit height
	  var height = font.getFontSize();
	  if(mElement.getFitText())
	  	height = Math.max(canvas.height,height);

	  // font 
 	  var fondtxt = "";
 	  fondtxt += (font.getFontStyle() != 'none')? font.getFontStyle() + ' ':'';
 	  fondtxt += (font.getFontWeight() != 'none')? font.getFontWeight() + ' ':'';
 	  fondtxt += height + 'px ';
 	  fondtxt += font.getFontFamily();
	  context.font = fondtxt;
		 
	  if(backGroundColor) {
		context.fillStyle = backGroundColor;
		context.fillRect(0, 0, canvas.width, canvas.height);
	  }
		 
	  context.textAlign = "center";
	  context.textBaseline = "middle";
	  context.fillStyle = color;
	  context.fillText(text, canvas.width / 2, canvas.height / 2, canvas.width);
		 
      // add texture 
	  var texture = EJSS_WEBGLGRAPHICS.Texture.fromImage(mGL,canvas);
	  mGL.addElement(mElement.getName() + ".texture", texture);
  		  
      // create mesh
	  var detail = 1;     // tesselation level 	
	  for (var y = 0; y <= detail; y++) {
	    var t = y / detail;
	    for (var x = 0; x <= detail; x++) {
	      var s = x / detail;
	      mesh.vertices.push([0 , 2 * t - 1, 2 * s - 1]);
	      mesh.coords.push([t, s]);	      
	      if (x < detail && y < detail) {
	        var i = x + y * (detail + 1);
	        mesh.triangles.push([i, i + 1, i + detail + 1]);
	        mesh.triangles.push([i + detail + 1, i + 1, i + detail + 2]);
	      }
	    }
	  }
	  
	  // compute normals	  	
	  mesh.computeNormals();
	  
	  // compute wire
	  mesh.computeWireframe();
	  	  
	  mesh.compile();
				 	  		  	
	  mGL.addElement(mElement.getName() + ".mesh", mesh);
        
      return mesh;
}

// Provides a simple wrapper around WebGL textures that supports render-to-texture.

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

EJSS_WEBGLGRAPHICS.Texture = {
// ### GL.Texture.fromImage(image[, options])
// 
// Return a new image created from `image`, an `<img>` tag.
fromImage: function(mGL, image, options) {
  options = options || {};
  var texture = EJSS_WEBGLGRAPHICS.texture(mGL, image.width, image.height, options);
  try {
    mGL.texImage2D(mGL.TEXTURE_2D, 0, texture.getFormat(), texture.getFormat(), texture.getType(), image);
  } catch (e) {
    if (location.protocol == 'file:') {
    	// for Chrome --allow-file-access-from-files
      throw 'image not loaded for security reasons (serve this page over "http://" instead)';
    } else {
      throw 'image not loaded for security reasons (image must originate from the same ' +
        'domain as this page or use Cross-Origin Resource Sharing)';
    }
  }
  if (options.minFilter && options.minFilter != mGL.NEAREST && options.minFilter != mGL.LINEAR) {
    mGL.generateMipmap(mGL.TEXTURE_2D);
  }
  return texture;
},

// ### GL.Texture.fromURL(url[, options])
// 
// Returns a checkerboard texture that will switch to the correct texture when
// it loads.
fromURL: function(mGL, url, options, callback) {
  checkerboardCanvas = (function() {
    var c = document.createElement('canvas').getContext('2d');
    c.canvas.width = c.canvas.height = 128;
    for (var y = 0; y < c.canvas.height; y += 16) {
      for (var x = 0; x < c.canvas.width; x += 16) {
        c.fillStyle = (x ^ y) & 16 ? '#FFF' : '#DDD';
        c.fillRect(x, y, 16, 16);
      }
    }
    return c.canvas;
  })();
  var texture = EJSS_WEBGLGRAPHICS.Texture.fromImage(mGL, checkerboardCanvas, options);
  var image = new Image();
  image.onload = function() {
  	EJSS_WEBGLGRAPHICS.Texture.fromImage(mGL, image, options).swapWith(texture);
  	callback();
  };
  image.src = url;
  return texture;
}

}

// ### new GL.Texture(width, height[, options])
//
// The arguments `width` and `height` give the size of the texture in texels.
// WebGL texture dimensions must be powers of two unless `filter` is set to
// either `mGL.NEAREST` or `mGL.LINEAR` and `wrap` is set to `mGL.CLAMP_TO_EDGE`
// (which they are by default).
//
// Texture parameters can be passed in via the `options` argument.
// Example usage:
// 
//     var t = new GL.Texture(256, 256, {
//       // Defaults to mGL.LINEAR, set both at once with "filter"
//       magFilter: mGL.NEAREST,
//       minFilter: mGL.LINEAR,
// 
//       // Defaults to mGL.CLAMP_TO_EDGE, set both at once with "wrap"
//       wrapS: mGL.REPEAT,
//       wrapT: mGL.REPEAT,
// 
//       format: mGL.RGB, // Defaults to mGL.RGBA
//       type: mGL.FLOAT // Defaults to mGL.UNSIGNED_BYTE
//     });
EJSS_WEBGLGRAPHICS.texture = function(mGL, width, height, options) {
  var self = {};
  options = options || {};

  var framebuffer;
  var renderbuffer;

  var id = mGL.createTexture();
  var format = options.format || mGL.RGBA;
  var type = options.type || mGL.UNSIGNED_BYTE;


  self.getId = function() {
  	return id;
  }
  
  self.setId = function(v) {
  	id = v;
  }
  
  self.getFormat = function() {
  	return format;
  }
  
  self.setFormat = function(v) {
  	format = v;
  }
  
  self.getType = function() {
  	return type;
  }
  
  self.setType = function(v) {
  	type = v;
  }

  self.getWidth = function() {
  	return width;
  }

  self.setWidth = function(v) {
  	width = v;
  }

  self.getHeight = function() {
    return height;	
  }
  
  self.setHeight = function(v) {
  	height = v;
  }

  // ### .bind([unit])
  // 
  // Bind this texture to the given texture unit (0-7, defaults to 0).
  self.bind = function(unit) {
    mGL.activeTexture(mGL.TEXTURE0 + (unit || 0));
    mGL.bindTexture(mGL.TEXTURE_2D, id);
  },

  // ### .unbind([unit])
  // 
  // Clear the given texture unit (0-7, defaults to 0).
  self.unbind = function(unit) {
    mGL.activeTexture(mGL.TEXTURE0 + (unit || 0));
    mGL.bindTexture(mGL.TEXTURE_2D, null);
  },

  // ### .drawTo(callback)
  // 
  // Render all draw calls in `callback` to this texture. This method sets up
  // a framebuffer with this texture as the color attachment and a renderbuffer
  // as the depth attachment. It also temporarily changes the viewport to the
  // size of the texture.
  // 
  // Example usage:
  // 
  //     texture.drawTo(function() {
  //       mGL.clearColor(1, 0, 0, 1);
  //       mGL.clear(mGL.COLOR_BUFFER_BIT);
  //     });
  self.drawTo = function(callback) {
    var v = mGL.getParameter(mGL.VIEWPORT);
    framebuffer = framebuffer || mGL.createFramebuffer();
    renderbuffer = renderbuffer || mGL.createRenderbuffer();
    mGL.bindFramebuffer(mGL.FRAMEBUFFER, framebuffer);
    mGL.bindRenderbuffer(mGL.RENDERBUFFER, renderbuffer);
    if (width != renderbuffer.width || height != renderbuffer.height) {
      renderbuffer.width = width;
      renderbuffer.height = height;
      mGL.renderbufferStorage(mGL.RENDERBUFFER, mGL.DEPTH_COMPONENT16, width, height);
    }
    mGL.framebufferTexture2D(mGL.FRAMEBUFFER, mGL.COLOR_ATTACHMENT0, mGL.TEXTURE_2D, id, 0);
    mGL.framebufferRenderbuffer(mGL.FRAMEBUFFER, mGL.DEPTH_ATTACHMENT, mGL.RENDERBUFFER, renderbuffer);
    mGL.viewport(0, 0, width, height);

    callback();

    mGL.bindFramebuffer(mGL.FRAMEBUFFER, null);
    mGL.bindRenderbuffer(mGL.RENDERBUFFER, null);
    mGL.viewport(v[0], v[1], v[2], v[3]);
  },

  // ### .swapWith(other)
  // 
  // Switch this texture with `other`, useful for the ping-pong rendering
  // technique used in multi-stage rendering.
  self.swapWith = function(other) {
    var temp;
    temp = other.getId(); other.setId(id); id = temp;
    temp = other.getWidth(); other.setWidth(width); width = temp;
    temp = other.getHeight(); other.setHeight(height); height = temp;
  }
  
  mGL.bindTexture(mGL.TEXTURE_2D, id);
  mGL.pixelStorei(mGL.UNPACK_FLIP_Y_WEBGL, 1);
  // mGL.pixelStorei(mGL.UNPACK_PREMULTIPLY_ALPHA_WEBGL, 1);
  mGL.texParameteri(mGL.TEXTURE_2D, mGL.TEXTURE_MAG_FILTER, options.filter || options.magFilter || mGL.LINEAR);
  mGL.texParameteri(mGL.TEXTURE_2D, mGL.TEXTURE_MIN_FILTER, options.filter || options.minFilter || mGL.LINEAR);
  mGL.texParameteri(mGL.TEXTURE_2D, mGL.TEXTURE_WRAP_S, options.wrap || options.wrapS || mGL.CLAMP_TO_EDGE);
  mGL.texParameteri(mGL.TEXTURE_2D, mGL.TEXTURE_WRAP_T, options.wrap || options.wrapT || mGL.CLAMP_TO_EDGE);
  mGL.texImage2D(mGL.TEXTURE_2D, 0, format, width, height, 0, format, type, null);

  return self;  
};

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};

/**
 * @param mGL Element where draw
 * @param mElement Element to draw
 * @returns A WebGL trail
 */

EJSS_WEBGLGRAPHICS.trail = function(mGL, mElement) {   
  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName(), 
	{coords:false, normals:true, triangles:true, lines:false, colors:false});
	  
  var resolutionU = mElement.getResolutionU();
  var radius = mElement.getLineWidth() / 2;

  var points = mElement.getPoints();
  for(var pos=0; pos<points.length-1; pos++) {
  	  var origpoint = new Vector(points[pos]);
  	  var endpoint = new Vector(points[pos+1]);
  	  var line = endpoint.subtract(origpoint);
      
	  // generate vertex for body
	  var vertices3D = [];	  
	  for (var x = 0; x < resolutionU; x++ ) {
		var u = x / (resolutionU - 1);
		var xpos = radius * Math.sin( u * Math.PI * 2 );
		var ypos = radius * Math.cos( u * Math.PI * 2 );

		vertices3D.push(new Vector([xpos, ypos, 0])); // top vertex
		vertices3D.push(new Vector([xpos, ypos, 0])); // bottom vertex
	  }
	  
	  // rotate and translate top and bottom
	  var cross = line.cross(new Vector(0,0,1));
	  if(cross.length()==0) cross.y = 1; // if endoint in axis Z, get axis Y to rotate 
	  // console.log("cross:" + cross.x + " " + cross.y + " " + cross.z + " ");	  
	  var angle = line.angle(new Vector(0,0,1));
	  // console.log("angle:" + angle);
	  var result = new Matrix();
	  Matrix.rotate(-angle,cross.x,cross.y,cross.z,0,0,0,result);
	  for(var i=0; i<vertices3D.length; i++) {
	  	var vector = result.transformPoint(vertices3D[i]);
	  	if (i % 2 == 0) {
	  		// add endpoint to translate	  	
	  		mesh.vertices.push(vector.add(endpoint).toArray());	  		
	  	} else {
	  		// add origpoint to translate
	  		mesh.vertices.push(vector.add(origpoint).toArray());
	  	}
	  }
	  
	  // create surface triangles
	  for (var x = 0; x < resolutionU - 1; x++ ) {
	    var first = 2 * (x + pos * resolutionU);		    
	    var fourth = 2 * (x  + pos * resolutionU) + 2;
	    var second = 2 * (x  + pos * resolutionU) + 1;
	    var third = 2 * (x  + pos * resolutionU) + 3;
			
        mesh.triangles.push([third, second, first]);      
        mesh.triangles.push([fourth, third, first]);
	  }		
  }		

  // last top cap
  mesh.vertices.push(new Vector(points[points.length-1]).toArray());		
  for (var i = 0; i < resolutionU - 1; i++) {
  	var first = mesh.vertices.length - 1;
  	var second = 2*i+2 + (2 * (points.length - 2) * resolutionU);
  	var third = 2*i + (2 * (points.length - 2) * resolutionU);
  	mesh.triangles.push([first, second, third]);
  }        

  // first bottom cap
  mesh.vertices.push(new Vector(points[0]).toArray());
  for (var i = 0; i < resolutionU - 1; i++) {
  	var first = 2*i+1;
  	var second = 2*i+3;
  	var third = mesh.vertices.length - 1;
   	mesh.triangles.push([first, second, third]);
  }        

  // nodes cap
  for(var pos=1; pos<points.length-1; pos++) {
	for (var x = 0; x < resolutionU - 1; x++ ) {
		var first = 2 * (x + (pos - 1) * resolutionU);		    
		var fourth = 2 * (x + (pos - 1) * resolutionU) + 2;
		var second = 2 * (x + pos * resolutionU) + 1;
		var third = 2 * (x + pos * resolutionU) + 3;			
        mesh.triangles.push([third, second, first]);      
        mesh.triangles.push([fourth, third, first]);
	}		    		
  }
   
  mesh.computeNormals();	
	  
  mesh.compile();	
	
  mGL.addElement(mElement.getName() + ".mesh", mesh);
        
  return mesh;
  
  
/* Previously it was possible to manage lineWidth and then the segment was easy ...
  var mesh = EJSS_WEBGLGRAPHICS.mesh(mGL,mElement.getName(), 
  	{coords:false, normals:false, triangles:false, lines:true, colors:false});
  	  
  var points = mElement.getPoints();
  var vertices = [];
  var lines = [];
  var index = 0;
  for(var i=0; i<points.length; i++) {
  	vertices[i] = [points[i][0],points[i][1],points[i][2]];
  	if(i!=0 && points[i][3] == 1) { // 0 is NOT CONNECTION
  		lines[index++] = [i-1, i];
  	}
  }
  
  // set values
  mesh.setVertices(vertices);	  	  
  if (lines.length > 0) mesh.setLines(lines);
  
  mesh.compile();
  
  mGL.addElement(mElement.getName() + ".mesh", mesh);
   
  return mesh;
*/   
}

/**
 * Deployment for 3D SVG drawing.
 * @module WebGLGraphics 
 */

var EJSS_WEBGLGRAPHICS = EJSS_WEBGLGRAPHICS || {};
EJSS_WEBGLGRAPHICS.Utils = {};

EJSS_WEBGLGRAPHICS.Utils.regexMap = function(regex, text, callback) {
  while ((result = regex.exec(text)) != null) {
    callback(result);
  }
}

EJSS_WEBGLGRAPHICS.Utils.isArray = function(obj) {
  var str = Object.prototype.toString.call(obj);
  return str == '[object Array]' || str == '[object Float32Array]';
}

EJSS_WEBGLGRAPHICS.Utils.isNumber = function(obj) {
  var str = Object.prototype.toString.call(obj);
  return str == '[object Number]' || str == '[object Boolean]';
}
// Provides a simple 3D vector class. Vector operations can be done using member
// functions, which return new vectors, or static functions, which reuse
// existing vectors to avoid generating garbage.
function Vector(x, y, z) {
  if(typeof x == 'object') {
  	this.x = x[0] || 0;
  	this.y = x[1] || 0;
  	this.z = x[2] || 0;  	
  } else {
  	this.x = x || 0;
  	this.y = y || 0;
  	this.z = z || 0;
  }
}

// ### Instance Methods
// The methods `add()`, `subtract()`, `multiply()`, and `divide()` can all
// take either a vector or a number as an argument.
Vector.prototype = {
  setx: function(x) {
  	this.x = x;
  },
  sety: function(y) {
  	this.y = y;
  },
  setz: function(z) {
  	this.z = z;
  },
  getx: function() {
  	return this.x;
  },
  gety: function() {
  	return this.y;
  },
  getz: function() {
  	return this.z;
  },
  rotate: function() {
  	return new Vector(this.z, this.x, this.y);
  },
  negative: function() {
    return new Vector(-this.x, -this.y, -this.z);
  },
  add: function(v) {
    if (v instanceof Vector) return new Vector(this.x + v.x, this.y + v.y, this.z + v.z);
    else return new Vector(this.x + v, this.y + v, this.z + v);
  },
  subtract: function(v) {
    if (v instanceof Vector) return new Vector(this.x - v.x, this.y - v.y, this.z - v.z);
    else return new Vector(this.x - v, this.y - v, this.z - v);
  },
  multiply: function(v) {
    if (v instanceof Vector) return new Vector(this.x * v.x, this.y * v.y, this.z * v.z);
    else return new Vector(this.x * v, this.y * v, this.z * v);
  },
  divide: function(v) {
    if (v instanceof Vector) return new Vector(this.x / v.x, this.y / v.y, this.z / v.z);
    else return new Vector(this.x / v, this.y / v, this.z / v);
  },
  equals: function(v) {
    return this.x == v.x && this.y == v.y && this.z == v.z;
  },
  dot: function(v) {
    return this.x * v.x + this.y * v.y + this.z * v.z;
  },
  cross: function(v) {
    return new Vector(
      this.y * v.z - this.z * v.y,
      this.z * v.x - this.x * v.z,
      this.x * v.y - this.y * v.x
    );
  },
  length: function() {
    return Math.sqrt(this.dot(this));
  },
  unit: function() {
    return this.divide(this.length());
  },
  min: function() {
    return Math.min(Math.min(this.x, this.y), this.z);
  },
  max: function() {
    return Math.max(Math.max(this.x, this.y), this.z);
  },
  angle: function(v) {
  	return Math.acos(this.dot(v)/(this.length()*v.length()));
  },
  toAngles: function() {
    return {
      theta: Math.atan2(this.z, this.x),
      phi: Math.asin(this.y / this.length())
    };
  },
  toArray: function(n) {
    return [this.x, this.y, this.z].slice(0, n || 3);
  },
  clone: function() {
    return new Vector(this.x, this.y, this.z);
  },
  init: function(x, y, z) {
    this.x = x; this.y = y; this.z = z;
    return this;
  }
};

// ### Static Methods
// `Vector.randomDirection()` returns a vector with a length of 1 and a
// statistically uniform direction. `Vector.lerp()` performs linear
// interpolation between two vectors.
Vector.negative = function(a, b) {
  b.x = -a.x; b.y = -a.y; b.z = -a.z;
  return b;
};
Vector.add = function(a, b, c) {
  if (b instanceof Vector) { c.x = a.x + b.x; c.y = a.y + b.y; c.z = a.z + b.z; }
  else { c.x = a.x + b; c.y = a.y + b; c.z = a.z + b; }
  return c;
};
Vector.subtract = function(a, b, c) {
  if (b instanceof Vector) { c.x = a.x - b.x; c.y = a.y - b.y; c.z = a.z - b.z; }
  else { c.x = a.x - b; c.y = a.y - b; c.z = a.z - b; }
  return c;
};
Vector.multiply = function(a, b, c) {
  if (b instanceof Vector) { c.x = a.x * b.x; c.y = a.y * b.y; c.z = a.z * b.z; }
  else { c.x = a.x * b; c.y = a.y * b; c.z = a.z * b; }
  return c;
};
Vector.divide = function(a, b, c) {
  if (b instanceof Vector) { c.x = a.x / b.x; c.y = a.y / b.y; c.z = a.z / b.z; }
  else { c.x = a.x / b; c.y = a.y / b; c.z = a.z / b; }
  return c;
};
Vector.cross = function(a, b, c) {
  c.x = a.y * b.z - a.z * b.y;
  c.y = a.z * b.x - a.x * b.z;
  c.z = a.x * b.y - a.y * b.x;
  return c;
};
Vector.unit = function(a, b) {
  var length = a.length();
  b.x = a.x / length;
  b.y = a.y / length;
  b.z = a.z / length;
  return b;
};
Vector.fromAngles = function(theta, phi) {
  return new Vector(Math.cos(theta) * Math.cos(phi), Math.sin(phi), Math.sin(theta) * Math.cos(phi));
};
Vector.randomDirection = function() {
  return Vector.fromAngles(Math.random() * Math.PI * 2, Math.asin(Math.random() * 2 - 1));
};
Vector.min = function(a, b) {
  return new Vector(Math.min(a.x, b.x), Math.min(a.y, b.y), Math.min(a.z, b.z));
};
Vector.max = function(a, b) {
  return new Vector(Math.max(a.x, b.x), Math.max(a.y, b.y), Math.max(a.z, b.z));
};
Vector.lerp = function(a, b, fraction) {
  return b.subtract(a).multiply(fraction).add(a);
};
Vector.fromArray = function(a) {
  return new Vector(a[0], a[1], a[2]);
};
/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Deployment for 3D drawing.
 * @module 3Dgraphics 
 */

var EJSS_GRAPHICS = EJSS_GRAPHICS || {};

/**
 * WebGLGraphics class
 * @class WebGLGraphics 
 * @constructor  
 */
EJSS_GRAPHICS.WebGLGraphics = {
    
  /**
   * Return document box
   * @return box  
   */
  getOffsetRect: function(graphics) {
  	return EJSS_GRAPHICS.GraphicsUtils.getOffsetRect(graphics);
  }
};

/**
 * Constructor for Element
 * @param mName Identifier Canvas element in HTML document
 * @returns A Canvas graphics
 */  
EJSS_GRAPHICS.webGLGraphics = function(mName) {
  var self = EJSS_INTERFACE.webGLCanvas(mName);	// reference returned     
  var canvas = self.getDOMElement();

  // get context
  var mGL = self.getContext();
  
  // blending vars
  var mDepths = {};
  var mmBlendingElements = [];

  var mTransformationOrder = 0; // 0: first trans/scale and then transformation, 1: first transformation and then trans/scale 

  self.supportsWebGL = function() {
    return mGL;
  };
  
  /**
   * Return canvas for event handle
   * @return Canvas element 
   */
  self.getEventContext = function() {  	
    return canvas;
  };
  
  /**
   * Return canvas document box
   * @return width integer 
   */
  self.getBox = function() {
  	var box = EJSS_GRAPHICS.WebGLGraphics.getOffsetRect(self)
  	canvas.width = box.width;
  	canvas.height = box.height; 
  	box.width -= 1;
  	box.height -= 1;
  	return box;
  };
    
  /**
   * Return aspect ratio of canvas 
   * @return double 
   */
  self.getAspect = function() {
  	var box = EJSS_GRAPHICS.WebGLGraphics.getOffsetRect(self)
	var height = Math.max(1, box.height);  // prevent divide by 0

  	return box.width / height;
  }

  /**
   * Set transformation order 
   * @return order 
   */
  self.setTransformationOrder = function(order) {
  	mTransformationOrder = order;
  }
    
  /**
   * Get transformation order 
   * @return order 
   */
  self.getTransformationOrder = function() {
  	return mTransformationOrder;
  }
    
  /**
   * Remove a shape
   * @param name
   */
  self.remove = function(name) {	
    
  };

  /**
   * Reset the canvas
   */
  self.reset = function(ort,dim,fov,near,far) {	  
  	if(!mGL) return;

    addMatrixStack();

	// viewport
    var box = self.getBox();
	var height = Math.max(1, box.height);  // prevent divide by 0
	mGL.viewport(0, 0, box.width, height);

	// projection matrix
    mGL.matrixMode(mGL.PROJECTION);
    mGL.loadIdentity();
	if(!ort) 
		mGL.perspective(fov, box.width / height, near, far);
	else {
		mGL.ortho(dim.left, dim.right, dim.bottom, dim.top, -near, far);								
	}
	
	// view matrix
    mGL.matrixMode(mGL.VIEW);     
    
    // rendering of both sides of each face (front and back)
  	mGL.disable(mGL.CULL_FACE);  
	mGL.enable(mGL.POLYGON_OFFSET_FILL);
	// mGL.polygonOffset(0.5, 0.5);
	
    // without blending for default	
	mGL.enable(mGL.DEPTH_TEST);
	mGL.depthFunc(mGL.LESS);
	mGL.depthMask(true);

    mGL.clear(mGL.COLOR_BUFFER_BIT | mGL.DEPTH_BUFFER_BIT);	
  };

  self.putCamera = function(ort,camera) {
  	if(!mGL) return;
  	
    mGL.clear(mGL.COLOR_BUFFER_BIT | mGL.DEPTH_BUFFER_BIT);

	// ort size
	if(ort) {
    	mGL.matrixMode(mGL.PROJECTION);
    	if (camera.delta != 0) {
			mGL.scale(camera.delta, camera.delta, camera.delta);		    		
	    	camera.delta = 0;    		
    	}
	}
	    	
  	// camera position
    mGL.matrixMode(mGL.VIEW);     	
    mGL.loadIdentity();
    mGL.lookAt(camera.location.x, camera.location.y, camera.location.z,
    			camera.focus.x, camera.focus.y, camera.focus.z,
    			camera.upvector.x, camera.upvector.y, camera.upvector.z); // up Vector
    mGL.rotate(EJSS_TOOLS.Mathematics.radians(camera.tilt), 1, 0, 0, 
    	camera.focus.x, camera.focus.y, camera.focus.z);    
    mGL.rotate(EJSS_TOOLS.Mathematics.radians(camera.altitude), 0, 1, 0, 
    	camera.focus.x, camera.focus.y, camera.focus.z);    
    mGL.rotate(EJSS_TOOLS.Mathematics.radians(camera.azimuth), 0, 0, 1, 
    	camera.focus.x, camera.focus.y, camera.focus.z);    
  }
	 
  // Ordered list of blending elements
  function addToAlphaList(element) {
	// element depth
	var v = new Vector(element.getPosition());
	var pv = mGL.viewMatrix.transformVector(v);
	var depth = pv.z;

	mDepths[element.getName()] = depth;
		
	// Binary search
	var less, more, itteration = 1, inserted = false, index = Math.floor(mBlendingElements.length/2);
	while(!inserted) {
		less = (index === 0 || mDepths[mBlendingElements[index-1].getName()] <= depth);
		more = (index >= mBlendingElements.length || mDepths[mBlendingElements[index].getName()] >= depth);
		if(less && more) {
			mBlendingElements.splice(index, 0, element);
			inserted = true;
		} else {
			itteration++;
			var step = Math.ceil(mBlendingElements.length/Math.pow(2,itteration));
			if(!less) {
				index = Math.max(0, index-step);
			} else {
				index = Math.min(mBlendingElements.length, index+step);
			}
		}
	}
  };

  /**
   * Draw Elements
   * @param elements array of Elements and Element Sets
   * @param force whether force draw elements
   */
  self.draw = function(elements,force) {  	  
  	if(!mGL) return;  		
  	
  	// globals (they could be local vars)
  	mDepths = {};
  	mBlendingElements = [];
  	
	var noBlendingElements = [];  		
  	var allElements = elements.slice();  	

	while (allElements.length) {
      var element = allElements.shift();
      if(element.getElements) {	// whether element set
      	var lasts = element.getLastElements();
  		for(var j=0, m=lasts.length; j<m; j++) { // remove last removed elements
			mGL.removeElement(lasts[j].getName() + ".mesh");
  		}    
  		// update array of elements used in iteration
  		allElements = element.getElements().concat(allElements);
      } else { // else one element
  		// it is neccesary to order elements to blending (http://delphic.me.uk/webglalpha.html)       	
		if(element.getStyle().getTransparency() > 0.0) {
			addToAlphaList(element);
		} else {
			noBlendingElements.push(element);
		}
      }
    }    
   
    // drawing elements without blending
	for(i = 0, l = noBlendingElements.length; i < l; i++) {
		var element = noBlendingElements[i];
		self.drawElement(element,force);
	}    
   
    // drawing elements with blending
	mGL.depthMask(false);
    mGL.enable(mGL.BLEND);
    mGL.blendFunc(mGL.SRC_ALPHA, mGL.ONE_MINUS_SRC_ALPHA);

    // draw ordered elements by depth
	for(i = 0, l = mBlendingElements.length; i < l; i++) {
		var element = mBlendingElements[i];
		self.drawElement(element,force);
		//console.log("Drawing element " + element.getName() + " - " + mDepths[element.getName()]);
	}    
	
	// disable blending
	mGL.disable(mGL.BLEND);
	mGL.depthMask(true);   
  };

  function applyTranformation(gtr) {
	  if (gtr && gtr.length > 0) {
			if (EJSS_WEBGLGRAPHICS.Utils.isArray(gtr[0])) { // array of rotations
				for (var i=0; i<gtr.length; i++) {
					if (gtr[i].length == 7) { 
						// rotation 
						mGL.rotate(gtr[i][0],gtr[i][1],gtr[i][2],gtr[i][3],
							gtr[i][4],gtr[i][5],gtr[i][6]); // angle,x,y,z,cx,cy,cz								
					} else {
						// transformation matrix
						mGL.applyMatrix(gtr[i]);
					}
				}
			} else { // only one rotation
				if (gtr.length == 7) { 
					// rotation 
					mGL.rotate(gtr[0],gtr[1],gtr[2],gtr[3],
						gtr[4],gtr[5],gtr[6]); // angle,x,y,z,cx,cy,cz								
				} else {
					// transformation matrix
					mGL.applyMatrix(gtr);
				}
			}
	   }  	
  }

  /**
   * Draw Element
   * @param element
   */
  self.drawElement = function (element, force) {
    // console.log("Drawing element "+element.getName());
    var build = null;
  	if (element.isGroupVisible()) {
        var build = mGL.getElement(element.getName() + ".mesh");
        if(!build || element.isMeshChanged() || element.isAlwaysUpdated()) {
          // if (build) mGL.removeElement(element.getName() + ".mesh");  		
      	  switch (element.getClass()) {
			case "ElementArrow": 	build = EJSS_WEBGLGRAPHICS.arrow(mGL,element);	break;
			case "ElementSegment": 	build = EJSS_WEBGLGRAPHICS.segment(mGL,element);	break;
			case "ElementSphere": 		build = EJSS_WEBGLGRAPHICS.ellipsoid(mGL,element);	break;
			case "ElementEllipsoid": 	build = EJSS_WEBGLGRAPHICS.ellipsoid(mGL,element);	break;						
			case "ElementBox": 		build = EJSS_WEBGLGRAPHICS.box(mGL,element);	break;
			case "ElementCylinder":	build = EJSS_WEBGLGRAPHICS.cylinder(mGL,element);	break;
			case "ElementCone":		build = EJSS_WEBGLGRAPHICS.cylinder(mGL,element);	break;
			case "ElementTetrahedron":		build = EJSS_WEBGLGRAPHICS.cylinder(mGL,element);	break;
			case "ElementDisk":		build = EJSS_WEBGLGRAPHICS.cylinder(mGL,element);	break;
			case "ElementBasic": 	build = EJSS_WEBGLGRAPHICS.basic(mGL,element);	break;
			case "ElementPlane": 	build = EJSS_WEBGLGRAPHICS.plane(mGL,element);	break;
			case "ElementSurface": 	build = EJSS_WEBGLGRAPHICS.surface(mGL,element);	break;
			case "ElementAnalyticSurface": 	build = EJSS_WEBGLGRAPHICS.analyticSurface(mGL,element);	break;
			case "ElementText": 	build = EJSS_WEBGLGRAPHICS.text(mGL,element);	break;
			case "ElementTrail": 	build = EJSS_WEBGLGRAPHICS.trail(mGL,element);	break;
			case "ElementAnalyticCurve":	build = EJSS_WEBGLGRAPHICS.analyticCurve(mGL,element);	break;
			case "ElementSpring":	build = EJSS_WEBGLGRAPHICS.spring(mGL,element);	break;
		  }
		}
		// if (build && (element.isMeshChanged() || element.isProjChanged() || force)) {
		if (build) {
    		mGL.matrixMode(mGL.MODEL);
    		mGL.loadIdentity();

			var groups = [];
			var el = element;
		    while (el.getGroup()) {
		    	el = el.getGroup();
   		        groups.unshift(el);
   		    }
			
			for (var g=0; g<groups.length; g++) {
			  el = groups[g];
			  
			  // group transformation
		      var gtr = el.getFullTransformation();
			  if (mTransformationOrder != 0) applyTranformation(gtr); 

		      // scale and translate
  	  		  mGL.translate(			
				el.getX()*element.getPanel().getSizeX(),
  	  			el.getY()*element.getPanel().getSizeY(),
  	  			el.getZ()*element.getPanel().getSizeZ());  	  		  	
			  mGL.scale(el.getSizeX(), el.getSizeY(), el.getSizeZ());
		      
			  if (mTransformationOrder == 0) applyTranformation(gtr); 
		    }		
			
			// individual transformation
			var tr = element.getFullTransformation(); // only support rotation
			if (mTransformationOrder != 0) applyTranformation(tr); 

		    // scale and translate
  	  		mGL.translate(element.getX()*element.getPanel().getSizeX(),
  	  			element.getY()*element.getPanel().getSizeY(),
  	  			element.getZ()*element.getPanel().getSizeZ());			

			if (mTransformationOrder == 0) applyTranformation(tr); 

			// adjust sizes
			if(element.getClass() == "ElementArrow" || element.getClass() == "ElementSegment" || element.getClass() == "ElementSpring" ) { // size is used to create the arrow
				mGL.scale(element.getPanel().getSizeX(),
	  	  			element.getPanel().getSizeY(),
	  	  			element.getPanel().getSizeZ());				
			} else if(element.getClass() == "ElementTrail" || element.getClass() == "ElementSurface") { // creados con su tamaño real
				mGL.scale(element.getPanel().getSizeX()*element.getSizeX(),
	  	  			element.getPanel().getSizeY()*element.getSizeY(),
	  	  			element.getPanel().getSizeZ()*element.getSizeZ());								
			} else { // creados con el doble de su tamaño
				mGL.scale(element.getPanel().getSizeX()*element.getSizeX()/2,
	  	  			element.getPanel().getSizeY()*element.getSizeY()/2,
	  	  			element.getPanel().getSizeZ()*element.getSizeZ()/2);				
			}
			
			// draw mesh
			self.drawMesh(mGL,element,build);

			// after transformations
    		mGL.matrixMode(mGL.VIEW);
		}
  	}  	  	
  	return build;
  };
 
  /**
   * Draw panel
   * @param panel
   */
  self.drawPanel = function(panel) {
 	if(!mGL) return;
 	
 	var style = panel.getStyle(); 	// style element	    	   	   

	// set background
    if(!panel.isImageUrl()) {	
		var color = style.getFillColor();
    	mGL.clearColor(color[0], color[1], color[2], (typeof color[3]!=="undefined") ? color[3] : 1.0);
    } else {
    	mGL.clearColor(0, 0, 0, 1.0);
    }
        
    return self;

  };
    
  /**
   * Draw mesh
   * @param mesh
   */
  self.drawMesh = function(mGL,element,mesh) {
  	  var opacity = (1.0 - element.getStyle().getTransparency() / 255);

  	  // get shader and texture
  	  var textureUrl = element.getTextureUrl();
	  if (element.getClass() == "ElementText" || 
	  		(textureUrl && textureUrl.length > 0)) {
		
 	  	 // Shader for texture 
		 var shader = mGL.getElement(mesh.getId() + ".shader_tex");
		 var texture = mGL.getElement(mesh.getId() + ".texture");	 
		 shader = shader || EJSS_WEBGLGRAPHICS.shader(mGL, mesh.getId() + "_tex",
		 	EJSS_WEBGLGRAPHICS.Shader["BasicVSwithTex"], EJSS_WEBGLGRAPHICS.Shader["BasicFSwithTex"]); 
	
	     if(texture) {
		  	texture.bind();
		  	shader.uniforms({ texture: 0 });
		  	shader.draw(mesh);
		  	texture.unbind();   
		  
		  	mGL.addElement(mesh.getId() + ".texture", texture);	
	     } else {
	        var tx = EJSS_WEBGLGRAPHICS.Texture.fromURL(mGL, textureUrl, {}, 
	      	function(tx) {
		  		element.getController().invokeAction("OnLoadTexture");	
		  		element.getController().reportInteractions();	    
	      	});

	    	tx.bind();
	    	shader.uniforms({ texture: 0 });
	    	shader.draw(mesh);
	    	tx.unbind();   	
			mGL.addElement(mesh.getId() + ".texture", tx);	      
	     }	
		
	     mGL.addElement(mesh.getId() + ".shader_tex", shader);
	  } else {		
		 // Shader for wire
		 var drawLines = element.getStyle().getDrawLines();
	  	 if (drawLines) {
	  		var shader = mGL.getElement(mesh.getId() + ".shader_lines");
	    	shader = shader || EJSS_WEBGLGRAPHICS.shader(mGL, mesh.getId() + "_lines", 
		 		EJSS_WEBGLGRAPHICS.Shader["BasicVS"], EJSS_WEBGLGRAPHICS.Shader["BasicFS"]);
		 		shader.uniforms({ color: element.getStyle().getLineColor(), opacity: opacity });	  	 		
	  	  	shader.draw(mesh, mGL.LINES);
	  	  	mGL.addElement(mesh.getId() + ".shader_lines", shader);
	    }            	
	  	// Shader for fill
		var drawFill = element.getStyle().getDrawFill();
	  	if (drawFill) {
		    var shader = mGL.getElement(mesh.getId() + ".shader_fill");
		
			var drawColors = (typeof element.getColors != 'undefined' && element.getColors().length > 0);	
			var drawLights = (element.getGroupPanel().getLights().length > 0);
			var hasPalette = (typeof element.getStyle().getPaletteFloor() != 'undefined');
			
			if(hasPalette) {
		    	shader = shader || EJSS_WEBGLGRAPHICS.shader(mGL, mesh.getId() + "_fill", 
			 		EJSS_WEBGLGRAPHICS.Shader["PaletteColorVS"], EJSS_WEBGLGRAPHICS.Shader["PaletteColorFS"]);
			 		shader.uniforms({ 
			 			floor: element.getStyle().getPaletteFloor(),
			 			ceil: element.getStyle().getPaletteCeil(),
			 			floorColor: element.getStyle().getPaletteFloorColor(),
			 			ceilColor: element.getStyle().getPaletteCeilColor()
			 		}); 					 					
			} else {
				if(!drawColors) { // draw one color
					 if(!drawLights) {
				    	shader = shader || EJSS_WEBGLGRAPHICS.shader(mGL, mesh.getId() + "_fill", 
					 		EJSS_WEBGLGRAPHICS.Shader["BasicVS"], EJSS_WEBGLGRAPHICS.Shader["BasicFS"]);
					 		shader.uniforms({ color: element.getStyle().getFillColor() }); 					 	
					 } else {
				    	shader = shader || EJSS_WEBGLGRAPHICS.shader(mGL, mesh.getId() + "_fill", 
					 		EJSS_WEBGLGRAPHICS.Shader["LightVS"], EJSS_WEBGLGRAPHICS.Shader["LightFS"]);
					 		shader.uniforms({ 
					 			ambientColor: element.getStyle().getAmbientColor(),
					 			diffuseColor: element.getStyle().getFillColor(),
					 			specularColor: element.getStyle().getSpecularColor(),
					 			ka: element.getStyle().getAmbientReflection(),
					 			kd: element.getStyle().getColorReflection(),
					 			ks: element.getStyle().getSpecularReflection(),
					 			shininessVal: element.getStyle().getShininessVal()
					 		}); 					 	
					 }
				} else { // draw with color array (only basic element)
					 if(!drawLights) {
				    	shader = shader || EJSS_WEBGLGRAPHICS.shader(mGL, mesh.getId() + "_fill", 
					 		EJSS_WEBGLGRAPHICS.Shader["ColorVS"], EJSS_WEBGLGRAPHICS.Shader["ColorFS"]);					 	
					 } else {
				    	shader = shader || EJSS_WEBGLGRAPHICS.shader(mGL, mesh.getId() + "_fill", 
					 		EJSS_WEBGLGRAPHICS.Shader["LightAndColorVS"], EJSS_WEBGLGRAPHICS.Shader["LightAndColorFS"]); 
					 		shader.uniforms({ 
					 			ambientColor: element.getStyle().getAmbientColor(),
					 			specularColor: element.getStyle().getSpecularColor(),
					 			ka: element.getStyle().getAmbientReflection(),
					 			kd: element.getStyle().getColorReflection(),
					 			ks: element.getStyle().getSpecularReflection(),
					 			shininessVal: element.getStyle().getShininessVal()
					 		}); 					 	
					 }					
				}				
			}			

			var lights = element.getGroupPanel().getLights();
			var numlight = lights.length;
		    shader.uniforms({ numlight: numlight, light: lights, opacity: opacity });
	  	    shader.draw(mesh, mGL.TRIANGLES);
		    mGL.addElement(mesh.getId() + ".shader_fill", shader);
		}
	  }  	
  }

// A value to bitwise-or with new enums to make them distinguishable from the
// standard WebGL enums.
var ENUM = 0x12340000;

// Implement the OpenGL modelview and projection matrix stacks, along with some
// other useful GLU matrix functions.
function addMatrixStack() {
  mGL.MODEL = ENUM | 1;
  mGL.VIEW = ENUM | 2;
  mGL.PROJECTION = ENUM | 3;
  var tempMatrix = new Matrix();
  var resultMatrix = new Matrix();
  mGL.modelMatrix = new Matrix();
  mGL.viewMatrix = new Matrix();
  mGL.projectionMatrix = new Matrix();
  var modelStack = [];
  var viewStack = [];
  var projectionStack = [];
  var matrix, stack;
  mGL.elements = {};	// shaders and mesh defined
  
  mGL.addElement = function(id, ele) {
  	// add or update  	
  	mGL.elements[id] = ele;
  }  

  mGL.getElement = function(id) {
  	return mGL.elements[id];
  }  

  mGL.removeElement = function(id) {
  	delete mGL.elements[id];
  }  

  mGL.matrixMode = function(mode) {
    switch (mode) {
      case mGL.MODEL:
        matrix = 'modelMatrix';
        stack = modelStack;
        break;
      case mGL.VIEW:
        matrix = 'viewMatrix';
        stack = viewStack;
        break;
      case mGL.PROJECTION:
        matrix = 'projectionMatrix';
        stack = projectionStack;
        break;
      default:
        throw 'invalid matrix mode ' + mode;
    }
  };
  mGL.loadIdentity = function() {
    Matrix.identity(mGL[matrix]);
  };
  mGL.loadMatrix = function(m) {
    var from = m.m, to = mGL[matrix].m;
    for (var i = 0; i < 16; i++) {
      to[i] = from[i];
    }
  };
  mGL.multMatrix = function(m) {
    mGL.loadMatrix(Matrix.multiply(mGL[matrix], m, resultMatrix));
  };
  mGL.perspective = function(fov, aspect, near, far) {
    mGL.multMatrix(Matrix.perspective(fov, aspect, near, far, tempMatrix));
  };
  mGL.frustum = function(l, r, b, t, n, f) {
    mGL.multMatrix(Matrix.frustum(l, r, b, t, n, f, tempMatrix));
  };
  mGL.ortho = function(l, r, b, t, n, f) {
    mGL.multMatrix(Matrix.ortho(l, r, b, t, n, f, tempMatrix));
  };
  mGL.scale = function(x, y, z) {
    mGL.multMatrix(Matrix.scale(x, y, z, tempMatrix));
  };
  mGL.translate = function(x, y, z) {
    mGL.multMatrix(Matrix.translate(x, y, z, tempMatrix));
  };
  mGL.applyMatrix = function(m) {
    mGL.multMatrix(new Matrix(m));
  };
  mGL.rotate = function(a, x, y, z, cx, cy, cz) {
  	cx = cx || 0;
  	cy = cy || 0;
  	cz = cz || 0;
    mGL.multMatrix(Matrix.rotate(a, x, y, z, cx, cy, cz, tempMatrix));
  };
  mGL.lookAt = function(ex, ey, ez, cx, cy, cz, ux, uy, uz) {
    mGL.multMatrix(Matrix.lookAt(ex, ey, ez, cx, cy, cz, ux, uy, uz, tempMatrix));
  };
  mGL.pushMatrix = function() {
    stack.push(Array.prototype.slice.call(mGL[matrix].m));
  };
  mGL.popMatrix = function() {
    var m = stack.pop();
    mGL[matrix].m = hasFloat32Array ? new Float32Array(m) : m;
  };
  mGL.matrixMode(mGL.MODEL);
}

 return self;           
}


/*
 * Copyright (C) 2014 Francisco Esquembre and Felix J. Garcia
 * This code is part of the Easy Javascript Simulations authoring and simulation tool
 * 
 * This code is Open Source and is provided "as is".
 */

/**
 * Framework for 3D drawing.
 * @module 3Ddrawing 
 */

var EJSS_DRAWING3D = EJSS_DRAWING3D || {};

/**
 * Plane
 * @class Plane 
 * @constructor  
 */
EJSS_DRAWING3D.Plane = {

    // ----------------------------------------------------
    // Static methods
    // ----------------------------------------------------

  	/**
   	* Copies one element into another
   	*/
  	copyTo : function(source, dest) {
      	EJSS_DRAWING3D.Element.copyTo(source,dest); // super class copy
  	
		dest.setDirectionA(source.getDirectionA());
		dest.setDirectionB(source.getDirectionB());
		dest.setSizeA(source.getSizeA());
		dest.setSizeB(source.getSizeB());
  	},


	/**
	 * static registerProperties method
	 */
	registerProperties : function(element, controller) {
		EJSS_DRAWING3D.Element.registerProperties(element, controller);
		// super class

		controller.registerProperty("DirectionA", element.setDirectionA);
		controller.registerProperty("DirectionB", element.setDirectionB);

		controller.registerProperty("SizeA", element.setSizeA);
		controller.registerProperty("SizeB", element.setSizeB);
		
	}

};

/**
 * Creates a 3D Plane
 * @method plane
 */
EJSS_DRAWING3D.plane = function (name) {
  var self = EJSS_DRAWING3D.element(name);

